/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <limits.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"

#include "imdef.h"
#include "imcaret.h"
#include "imcommit.h"
#include "imconv.h"
#include "imeventhandler.h"
#include "iminput.h"
#include "imlookup.h"
#include "impreedit.h"
#include "imstatus.h"
#include "imtext.h"
#include "phonogram_conv/include/phonogram_conv.h"


static const char ANTHY_DIR[] =  "anthyLE";


Bool
init_im(void)
{
    char buff[PATH_MAX];
    snprintf(buff, sizeof(buff), "%s/%s", IIIMLEDIR, ANTHY_DIR);
    if (im_input_init(buff) &&
	im_conv_init()) {
	return True;
    }

    term_im();
    return False;
}

void
term_im(void)
{
    im_conv_term();
    im_input_term();
}

Bool
create_desktop_data(
    iml_desktop_t *desktop
)
{
    PrivateDesktopData *pdd = malloc(sizeof(PrivateDesktopData));
    if (pdd) {
	SET_DESKTOP_DATA(desktop, pdd);
	return True;
    }

    return False;
}

void
free_desktop_data(
    iml_desktop_t *desktop
)
{
    PrivateDesktopData *pdd = DESKTOP_DATA(desktop);
    if (pdd) {
	SET_DESKTOP_DATA(desktop, NULL);
	free(pdd);
    }
}

Bool
create_session_data(
    iml_session_t *session
)
{
    PrivateSessionData *psd = calloc(1, sizeof(PrivateSessionData));
    if (psd) {
	SET_SESSION_DATA(session, psd);
	return True;
    }

    return False;
}

void
free_session_data(
    iml_session_t *session
)
{
    PrivateSessionData* psd = SESSION_DATA(session);
    if (psd) {
	clear_status(session);

	if (psd->im_input) {
	    im_input_free(psd->im_input);
	}

	if (psd->im_conv) {
	    im_conv_free(psd->im_conv);
	}

	free(psd);
	SET_SESSION_DATA(session, NULL);
    }
}

void
reset_session(
    iml_session_t *session
)
{
    PrivateSessionData* psd = SESSION_DATA(session);
    iml_inst *inst_list = NULL;

    if (psd->im_state != IM_STATE_OFF) {
	change_im_state(session, IM_STATE_EDIT_EMPTY, &inst_list, False);
	ASSERT(im_input_empty_p(psd->im_input));
/* 	if (!im_input_empty_p(psd->im_input)) { */
/* 	    im_input_clear(psd->im_input); */
/* 	    make_preedit_erase_inst(session, &inst_list); */
/* 	} */

	if (inst_list) {
	    iml_methods_t *mtbl = METHODS_TBL(session);
	    mtbl->iml_execute(session, &inst_list);
	}
    }
}

void
set_conversion_on(
    iml_session_t *session
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->im_state == IM_STATE_OFF) {
	iml_inst *inst_list = NULL;
	change_im_state(session, IM_STATE_EDIT_EMPTY, &inst_list, True);
	if (inst_list) {
	    iml_methods_t *mtbl = METHODS_TBL(session);
	    mtbl->iml_execute(session, &inst_list);
	}
    }
}

void
set_conversion_off(
    iml_session_t *session
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->im_state != IM_STATE_OFF) {
	iml_inst *inst_list = NULL;
	change_im_state(session, IM_STATE_OFF, &inst_list, True);
	if (inst_list) {
	    iml_methods_t *mtbl = METHODS_TBL(session);
	    mtbl->iml_execute(session, &inst_list);
	}
    }
}

void key_list_event_handler(iml_session_t *session, IMKeyListEvent *key_list_event)
{
    iml_inst *inst_list = NULL;
    IMKeyEventStruct *key_event = key_list_event->keylist;
    KeyEventHandler handler;
    int i;

    for (i=0; i<key_list_event->n_key; ++i, ++key_event) {
	handler = lookup_key_event_handler(session, key_event);
	if (!(*handler)(session, key_event, &inst_list)) {
	    break;
	}
    } 

    if (inst_list) {
	iml_methods_t *mtbl = METHODS_TBL(session);
	mtbl->iml_execute(session, &inst_list);
    }
}

Bool
change_im_state(
    iml_session_t *session,
    enum IM_STATE new_state,
    iml_inst **inst_list,
    Bool redraw_p
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->im_state != new_state) {
	iml_methods_t *mtbl = METHODS_TBL(session);
	iml_inst *inst;
	Bool need_preedit_redraw_p = False;
	Bool need_status_update_p = False;

	switch (psd->im_state) {
	  case IM_STATE_OFF:
	   if (new_state != IM_STATE_EDIT_EMPTY) {
	       ASSERT(0); /* cannot happen */
	       return False;
	   }

	   if (psd->im_input == NULL) {
	       psd->im_input = im_input_create();
	       if (psd->im_input == NULL) {
		   return False;
	       }
	   }

	   inst = mtbl->iml_make_start_conversion_inst(session);
	   mtbl->iml_link_inst_tail(inst_list, inst);

	   need_status_update_p = True;
	   break;

	  case IM_STATE_EDIT_EMPTY:
	   switch (new_state) {
	     case IM_STATE_OFF:
	      make_preedit_done_inst(session, inst_list);

	      inst = mtbl->iml_make_end_conversion_inst(session);
	      mtbl->iml_link_inst_tail(inst_list, inst);

	      need_status_update_p = True;
	      break;

	     case IM_STATE_EDITING:
	      break;

	     default:
	      ASSERT(0);
	      return False;
	   }
	   break;

	  case IM_STATE_EDITING:
	   switch (new_state) {
	     case IM_STATE_OFF:
	      return change_im_state(session, IM_STATE_EDIT_EMPTY, inst_list, False) &&
		  change_im_state(session, IM_STATE_OFF, inst_list, redraw_p);

	     case IM_STATE_EDIT_EMPTY:
	      im_input_clear(psd->im_input);
	      make_preedit_erase_inst(session, inst_list);
	      break;

	     case IM_STATE_LETTER_TYPE_CONVERSION:
	      im_input_reinit(psd->im_input);
	      need_preedit_redraw_p = True;
	      break;

	     case IM_STATE_CONVERSION:
	     case IM_STATE_PREDICTION:
	      {
		  int len;
		  UTFCHAR *str;

		  if (im_input_empty_p(psd->im_input)) {
		      return False;
		  }

		  if (psd->im_conv == NULL) {
		      psd->im_conv = im_conv_create();
		      if (psd->im_conv == NULL) {
			  return False;
		      }
		  }

		  im_input_break(psd->im_input);
		  str = im_input_get_string(psd->im_input, session, &len);

		  if (!im_conv_set_string(
			  psd->im_conv,
			  str,
			  len,
			  (new_state == IM_STATE_CONVERSION) ? IM_CONV_MODE_NORMAL : IM_CONV_MODE_PREDICT)) {
		      return False;
		  }

		  need_preedit_redraw_p = True;
	      }
	      break;

	     default:
	      ASSERT(0); /* cannot happen */
	      return False;
	   }
	   break;

	  case IM_STATE_LETTER_TYPE_CONVERSION:
	   switch (new_state) {
	     case IM_STATE_OFF:
	     case IM_STATE_EDIT_EMPTY:
	     case IM_STATE_CONVERSION:
	     case IM_STATE_CHANGE_SEG:
	     case IM_STATE_PREDICTION:
	      return change_im_state(session, IM_STATE_EDITING, inst_list, False) &&
		  change_im_state(session, new_state, inst_list, redraw_p);

	     case IM_STATE_EDITING:
	      need_preedit_redraw_p = True;
	      break;

	     default:
	      ASSERT(0); /* cannot happen */
	      return False;
	   }
	   break;

	  case IM_STATE_CONVERSION:
	   switch (new_state) {
	     case IM_STATE_OFF:
	     case IM_STATE_EDIT_EMPTY:
	     case IM_STATE_LETTER_TYPE_CONVERSION:
	     case IM_STATE_PREDICTION:
	      return change_im_state(session, IM_STATE_EDITING, inst_list, False) &&
		  change_im_state(session, new_state, inst_list, redraw_p);

	     case IM_STATE_EDITING:
	      end_lookup_choice(session, inst_list);
	      im_conv_reset(psd->im_conv);
	      im_input_reinit(psd->im_input);
	      need_preedit_redraw_p = True;
	      break;

	     case IM_STATE_CHANGE_SEG:
	      end_lookup_choice(session, inst_list);
	      need_preedit_redraw_p = im_conv_cur_select_unconverted(psd->im_conv);
	      break;

	     default:
	      ASSERT(0); /* cannot happen */
	      return False;
	   }
	   break;

	  case IM_STATE_CHANGE_SEG:
	   switch (new_state) {
	     case IM_STATE_OFF:
	     case IM_STATE_EDIT_EMPTY:
	     case IM_STATE_EDITING:
	     case IM_STATE_LETTER_TYPE_CONVERSION:
	     case IM_STATE_PREDICTION:
	      return change_im_state(session, IM_STATE_CONVERSION, inst_list, False) &&
		  change_im_state(session, new_state, inst_list, redraw_p);

	     case IM_STATE_CONVERSION:
	      need_preedit_redraw_p = im_conv_cur_select_cand(psd->im_conv, 0);
	      break;

	     default:
	      ASSERT(0); /* cannot happen */
	      return False;
	   }
	   break;

	  case IM_STATE_PREDICTION:
	   switch (new_state) {
	     case IM_STATE_OFF:
	     case IM_STATE_EDIT_EMPTY:
	     case IM_STATE_LETTER_TYPE_CONVERSION:
	     case IM_STATE_CONVERSION:
	      return change_im_state(session, IM_STATE_EDITING, inst_list, False) &&
		  change_im_state(session, new_state, inst_list, redraw_p);

	     case IM_STATE_EDITING:
	      end_lookup_choice(session, inst_list);
	      im_conv_reset(psd->im_conv);
	      im_input_reinit(psd->im_input);
	      need_preedit_redraw_p = True;
	      break;

	     case IM_STATE_CHANGE_SEG:
	     default:
	      ASSERT(0); /* cannot happen */
	      return False;
	   }
	   break;

	  default:
	   ASSERT(0); /* cannot happen */
	   return False;
	}

	psd->im_state = new_state;

	if (redraw_p && need_preedit_redraw_p) {
	    make_preedit_draw_inst(session, inst_list);
	}
	if (need_status_update_p) {
	    make_status_draw_inst(session, inst_list);
	}
    }
    return True;
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
