/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <stdlib.h>

#include "imcaret.h"
#include "imconv.h"
#include "imeventhandler.h"
#include "iminput.h"
#include "imlookup.h"
#include "impreedit.h"


Bool
handle_caret_move_begin(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
       if (im_input_cur_move_to_head(psd->im_input)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_input_make_preedit_text(psd->im_input, session, IMUnderline),
	       im_input_cur_pos(psd->im_input),
	       inst_list
	   );
       } else {
	   make_caret_move_inst(session, im_input_cur_pos(psd->im_input), inst_list);
       }
       return True;

      case IM_STATE_CHANGE_SEG:
       if (!change_im_state(session, IM_STATE_CONVERSION, inst_list, False)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_CONVERSION:
       end_lookup_choice(session, inst_list);

       if (im_conv_cur_move_to_head(psd->im_conv)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }
       return True;

      default:
       ASSERT(0); /* cannot happen */
       /* not break, but fall-through. */
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_PREDICTION:
       break;
    }

    return False;
}

Bool
handle_caret_move_end(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
       if (im_input_cur_move_to_tail(psd->im_input)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_input_make_preedit_text(psd->im_input, session, IMUnderline),
	       im_input_cur_pos(psd->im_input),
	       inst_list
	   );
       } else {
	   make_caret_move_inst(session, im_input_cur_pos(psd->im_input), inst_list);
       }
       return True;

      case IM_STATE_CHANGE_SEG:
       if (!change_im_state(session, IM_STATE_CONVERSION, inst_list, False)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_CONVERSION:
       end_lookup_choice(session, inst_list);

       if (im_conv_cur_move_to_tail(psd->im_conv)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }
       return True;

      default:
       ASSERT(0); /* cannot happen */
       /* not break, but fall-through. */
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_PREDICTION:
       break;
    }

    return False;
}

Bool handle_caret_move_left(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
       if (im_input_cur_move_prev(psd->im_input)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_input_make_preedit_text(psd->im_input, session, IMUnderline),
	       im_input_cur_pos(psd->im_input),
	       inst_list
	   );
       } else {
	   make_caret_move_inst(session, im_input_cur_pos(psd->im_input), inst_list);
       }
       return True;

      case IM_STATE_CHANGE_SEG:
       if (!change_im_state(session, IM_STATE_CONVERSION, inst_list, False)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_CONVERSION:
       end_lookup_choice(session, inst_list);

       if (im_conv_cur_move_prev(psd->im_conv)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }
       return True;

      default:
       ASSERT(0); /* cannot happen */
       /* not break, but fall-through. */
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_PREDICTION:
       break;
    }

    return False;
}

Bool
handle_caret_move_right(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
       if (im_input_cur_move_next(psd->im_input)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_input_make_preedit_text(psd->im_input, session, IMUnderline),
	       im_input_cur_pos(psd->im_input),
	       inst_list
	   );
       } else {
	   make_caret_move_inst(session, im_input_cur_pos(psd->im_input), inst_list);
       }
       return True;

      case IM_STATE_CHANGE_SEG:
       if (!change_im_state(session, IM_STATE_CONVERSION, inst_list, False)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_CONVERSION:
       end_lookup_choice(session, inst_list);

       if (im_conv_cur_move_next(psd->im_conv)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }
       return True;

      default:
       ASSERT(0); /* cannot happen */
       /* not break, but fall-through. */
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_PREDICTION:
       break;
    }

    return False;
}

Bool
handle_expand_current_segment(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_CONVERSION:
       if (!change_im_state(session, IM_STATE_CHANGE_SEG, inst_list, True)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_CHANGE_SEG:
       if (im_conv_cur_resize(psd->im_conv, +1)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }
       return True;

      default:
       ASSERT(0); /* cannot happen */
       /* not break, but fall-through. */
      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_PREDICTION:
       break;
    }
    return False;
}

Bool
handle_shrink_current_segment(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_CONVERSION:
       if (!change_im_state(session, IM_STATE_CHANGE_SEG, inst_list, True)) {
	   break;
       }
       /* not break, but fall-through. */
      case IM_STATE_CHANGE_SEG:
       if (im_conv_cur_resize(psd->im_conv, -1)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }
       return True;

      default:
       ASSERT(0); /* cannot happen */
       /* not break, but fall-through. */
      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_PREDICTION:
       break;
    }
    return False;
}

void
make_caret_move_inst(
    iml_session_t *session,
    int caret_pos,
    iml_inst **inst_list
)
{
    iml_methods_t *mtbl = METHODS_TBL(session);
    iml_inst *inst = mtbl->iml_make_preedit_caret_inst(session, caret_pos);
    mtbl->iml_link_inst_tail(inst_list, inst);
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
