/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#undef NDEBUG  // get assert in release mode

#include <cassert>
#include <iostream>

#include "../sh4lt/infotree/json-serializer.hpp"
#include "../sh4lt/shtype/shtype.hpp"

using namespace sh4lt;

template <typename T>
void check_value(const ShType& type, const std::string& key, const T& value) {
  auto val = type.get_prop(key);
  assert(!val.empty());
  assert(val.as<T>() == value);
}

void check_value(const ShType& type, const std::string& key, const char* value) {
  auto val = type.get_prop(key);
  assert(!val.empty());
  assert(val.as<std::string>() == value);
}

auto main() -> int {
  {  // check building a Type and consistency when serializing/deserializing it
    auto type = ShType("video/x-raw", "Camera 1", "Room");
    type.set_prop("format", "I420");
    type.set_prop("height", 4096);
    type.set_prop("interlace-mode", "progressive");
    type.set_prop("label", "unexpected \\= comma\\,");
    type.set_prop("multiview-mode", "mono");
    type.set_prop("width", 4096);
    type.set_prop("is_something", true);
    type.set_prop("pi", 3.14);
    type.set_custom_prop("framerate", "fraction", "30/1");
    type.set_custom_prop("pixel-aspect-ratio", "fraction", "1/1");
    check_value(type, "format", "I420");
    check_value(type, "framerate", "30/1");
    check_value(type, "height", 4096);
    check_value(type, "interlace-mode", "progressive");
    check_value(type, "label", "unexpected \\= comma\\,");
    check_value(type, "multiview-mode", "mono");
    check_value(type, "pixel-aspect-ratio", "1/1");
    check_value(type, "width", 4096);
    check_value(type, "is_something", true);
    check_value(type, "pi", 3.14);

    // Check serialization and deserialization
    auto type_str = ShType::serialize(type);
    assert(!type_str.empty());
    assert(infotree::json::is_valid(type_str));
    auto type2 = ShType::deserialize(type_str);
    assert(type2.media() == "video/x-raw");
    check_value(type2, "format", "I420");
    check_value(type2, "framerate", "30/1");
    check_value(type2, "height", 4096);
    check_value(type2, "interlace-mode", "progressive");
    check_value(type2, "label", "unexpected \\= comma\\,");
    check_value(type2, "multiview-mode", "mono");
    check_value(type2, "pixel-aspect-ratio", "1/1");
    check_value(type2, "width", 4096);
    // check re-serialization produce the same string
    assert(type_str == ShType::serialize(type2));
    std::cout << type_str << '\n';
  }

  { // empty serialization
    assert(ShType::serialize(ShType()).empty());
  }
  
  {  // check deserialization error (malformed JSON)
    auto err_shtype = ShType::deserialize("{");
    assert(!err_shtype.msgs().empty());
  }

  { // missing "media" key
    auto err_shtype = ShType::deserialize("{}");
    assert(!err_shtype.msgs().empty());
  }

  return 0;
}

