/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#undef NDEBUG  // get assert in release mode

#include <array>
#include <cassert>
#include <future>
#include <iostream>
#include <thread>

#include "../sh4lt/follower.hpp"
#include "../sh4lt/logger/console.hpp"
#include "../sh4lt/writer.hpp"

using namespace sh4lt;

// a struct with contiguous data storage.
using Frame = struct frame_t {
  size_t count{0};
  std::array<int, 3> data{{3, 1, 4}};
  // no vector ! vector.data is contiguous, but not vector
};

void writer(logger::Logger::ptr logger) {
  auto shtype = ShType::deserialize(R"(
{
  "label": "check sh4lt",
  "group": "testing",
  "media": "application/x-check-sh4lt",
  "property": {
    "int": 3,
    "bool": true,
    "string": "Bonjour-Hi",
    "aspect-ratio": "16/9"
  },
  "type": {
     "aspect-ratio": "fraction"
   }
}
)");

  Writer w(shtype, sizeof(Frame), logger);
  assert(w);
  std::this_thread::sleep_for(std::chrono::milliseconds(50));
  auto i = 10;
  Frame frame;
  while (0 != i--) {
    assert(w.copy_to_shm(&frame, sizeof(Frame), -1, -1));
    frame.count++;
  }
}

auto main() -> int {
  using namespace sh4lt;
  auto logger = std::make_shared<logger::Console>();
  auto server_interactions = 0;
  auto num_successive_write = 5;
  {
    Follower follower(
        ShType::get_path("check sh4lt", "testing"),
        [](void* data, size_t size, const sh4lt_time_info_t*) {
          auto frame = static_cast<Frame*>(data);
          std::cout << "(copy) new data for client " << frame->count << " (size " << size << ")"
                    << std::endl;
        },
        [&](const ShType& shtype) {
          assert(shtype.media() == "application/x-check-sh4lt");
          assert(shtype.label() == "check sh4lt");
          assert(shtype.group() == "testing");
          assert(shtype.get_prop("int").as<int>() == 3);
          assert(shtype.get_prop("bool").as<bool>() == true);
          assert(shtype.get_prop("string").as<std::string>() == "Bonjour-Hi");
          assert(shtype.get_prop("aspect-ratio").as<std::string>() == "16/9");
          assert(shtype.get_custom_type("aspect-ratio") == "fraction");
          ++server_interactions;
        },
        [&]() { ++server_interactions; },
        logger);

    auto i = num_successive_write;
    while (-1 != --i) {
      auto writer_handle = std::async(std::launch::async, writer, logger);
      writer_handle.get();
      // std::this_thread::sleep_for(std::chrono::milliseconds(50));
    }
  }

  if (server_interactions == 2 * num_successive_write) return 0;
  return 1;  // fail
}

