/*******************************************************************************
* Copyright 2023 Intel Corporation
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#include <cstdint>
#include <vector>

namespace dnnl {
namespace impl {
namespace gpu {
namespace jit {

// clang-format off
const std::vector<uint64_t> &get_conv_lookup_table_data() {
    static std::vector<uint64_t> data = {
        0x00000000000000fb, 0x0000000100000005, 0x0000000800000001, 0x0000000100000008, 0x0013000008000001, 0x3168693863690000, 0x6b38686f38636f30, 0x0000103068703368,
        0x0300070000000100, 0x0400000000000000, 0x0000080102000000, 0x0a00000003020800, 0x08030d0000001000, 0x0500000000000000, 0x0100000001000000, 0x0b0000000b000000,
        0x0100000001000000, 0x0000130000080000, 0x3031686938636900, 0x686b38686f38636f, 0x0000000830687033, 0x0003000700000001, 0x0004000000000000, 0x0000000801020000,
        0x000a000000030208, 0x0008030d00000008, 0x0005000000000000, 0x0001000000010000, 0x0008000000080000, 0x0001000000010000, 0x0000001300000800, 0x6f30316869326369,
        0x33686b38686f3863, 0x0100000010306870, 0x0000030007000000, 0x0000040000000000, 0x0800000002010200, 0x10000a0000000302, 0x000008030d000000, 0x0000050000000000,
        0x0000010000000100, 0x00000b0000000b00, 0x0000010000000100, 0x6900000013000008, 0x636f303168693263, 0x7033686b38686f38, 0x0001000000083068, 0x0000000300070000,
        0x0000000400000000, 0x0208000000020102, 0x0008000a00000003, 0x00000008030d0000, 0x0000000500000000, 0x0000000200000001, 0x0000000800000001, 0x0800000100000008,
        0x6369000000130000, 0x38636f3031686932, 0x687033686b38686f, 0x0000020000001030, 0x0800000003000700, 0x0000000000000301, 0x1002020000000300, 0x0000100009000000,
        0x0000000008010a00, 0x0100000005000000, 0x0100000002000000, 0x0b0000000b000000, 0x0008000001000000, 0x3263690000001300, 0x6f38636f30316869, 0x30687033686b3868,
        0x0000000200000008, 0x0108000000030007, 0x0000000100000003, 0x0003000000020005, 0x0000000802020000, 0x010a000000100009, 0x0000000000000008, 0x0000000300000004,
        0x0000000700000001, 0x0000000100000007, 0x001a000000010001, 0x3268693363690000, 0x686f3436636f3432, 0x687337686b323131, 0x0000000833687032, 0x0007000700000002,
        0x0000000201080000, 0x0002000a00000002, 0x00000004010d0000, 0x0004010200000004, 0x0000000402080000, 0x030d00000010000a, 0x0000000100000020, 0x0000000300000004,
        0x0000000400000001, 0x0000000100000004, 0x001a000000010001, 0x3268693363690000, 0x686f3436636f3432, 0x687337686b323131, 0x0000000833687032, 0x0007000700000001,
        0x000a000000020000, 0x0004010d00000002, 0x0102000000040000, 0x0008020800000004, 0x00000010000a0000, 0x000100000020030d, 0x0003000000040000, 0x0007000000030000,
        0x0007000000010000, 0x0000000800010000, 0x313563690000001c, 0x636f393932776932, 0x6b3337776f323135, 0x3077703477733877, 0x0000000100000008, 0x0002000000080009,
        0x0000000801020000, 0x000300000004000a, 0x0000001001020000, 0x020d00000040000a, 0x0000000300000010, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007,
        0x0019000000010001, 0x6930343663690000, 0x383231636f363168, 0x33686b3631686f30, 0x0300000010316870, 0x0000050002000000, 0x0800000003010700, 0x0000010000000302,
        0x0300000004000200, 0x0000200102000000, 0x0d00000020000a00, 0x0000000000001002, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1900000020010100,
        0x3231356369000000, 0x3031636f33316869, 0x686b3331686f3432, 0x0000000831687033, 0x0020000200000003, 0x0000000301070000, 0x0001000000030208, 0x00000008000a0000,
        0x0010020200000003, 0x0000002000090000, 0x000200000020010a, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000002001010000, 0x3135636900000019,
        0x31636f3331686932, 0x6b3331686f343230, 0x0000083168703368, 0x1000020000000300, 0x0000030107000000, 0x0200000003020800, 0x000008000a000000, 0x0300000002010d00,
        0x0000200202000000, 0x0a00000020000900, 0x0000030000001001, 0x0000030000000400, 0x0000070000000300, 0x0000070000000100, 0x1d00000010010100, 0x3830316369000000,
        0x6f30303531776938, 0x3531776f32313563, 0x30777031776b3030, 0x0000000100000008, 0x0002000001f4000d, 0x0000000801020000, 0x000300000008000a, 0x0000002002020000,
        0x010a000000100009, 0x0000000300000020, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x0015000000010001, 0x6936353263690000, 0x686f3231636f3568,
        0x1031687033686b35, 0x0200000003000000, 0x0301070000000200, 0x0000030208000000, 0x0800020000000100, 0x0200000003000000, 0x10000a0000001001, 0x000008020d000000,
        0x0000050000000000, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000016000000, 0x6f37316869323363, 0x6b3731686f323363, 0x0000103168703368,
        0x0200020000000300, 0x0000030107000000, 0x0200000003020800, 0x000002000a000000, 0x0300000004010d00, 0x0000100102000000, 0x0d00000010000a00, 0x0000030000000802,
        0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1700000020000100, 0x6369323367000000, 0x34636f3635686934, 0x7033686b3635686f, 0x0001000000083168,
        0x0000000300070000, 0x0008000d00000001, 0x0202000000040000, 0x0004030800000004, 0x0000002000090000, 0x000200000008010a, 0x0003000000040000, 0x0004000000010000,
        0x0001000000040000, 0x0000002000010000, 0x6932336700000017, 0x636f363568693463, 0x33686b3635686f34, 0x0100000008316870, 0x0000030007000000, 0x08000d0000000100,
        0x0200000004000000, 0x0403080000000802, 0x0000200009000000, 0x0200000008010a00, 0x0300000004000000, 0x0400000001000000, 0x0100000004000000, 0x0000800101000000,
        0x3263690000001b00, 0x636f363568693635, 0x6b3832686f323135, 0x3068703268733168, 0x0000000100000008, 0x0001000000080002, 0x00000008000a0000, 0x0020020200000003,
        0x0000002000090000, 0x000200000020010a, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x323163690000001c, 0x636f363168693038,
        0x6b38686f30383231, 0x3168703268733368, 0x0000000300000010, 0x0107000000280002, 0x0003020800000003, 0x000a000000010000, 0x0000000300000008, 0x000a000000200102,
        0x0008020d00000010, 0x0004000000030000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000001600000080, 0x3768693231356369, 0x37686f323135636f,
        0x000831687033686b, 0x0002000000030000, 0x0003010700000010, 0x0000000302080000, 0x0004000900000002, 0x00000004010a0000, 0x0020020200000003, 0x0000002000090000,
        0x000300000020010a, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x3032636900000019, 0x636f393477693834, 0x6b3934776f323135,
        0x0000083077703177, 0x4000020000000100, 0x0a00000002000000, 0x04010d0000000400, 0x0200000003000000, 0x2000090000002002, 0x000020010a000000, 0x0000040000000300,
        0x0000030000000300, 0x0000010000000700, 0x0300010000000700, 0x6900000026000000, 0x3031686936353263, 0x3532636f36317769, 0x31776f3031686f36, 0x7035776b35686b36,
        0x0000083277703268, 0x0a000c0000000100, 0x0200000002000000, 0x04000a0000000801, 0x0200000003000000, 0x40000a0000001001, 0x000010020d000000, 0x0000040000000300,
        0x0000010000000300, 0x0000070000000700, 0x2000010000000100, 0x6900000017000000, 0x3231356869323363, 0x323135686f33636f, 0x000831687033686b, 0x0007000000020000,
        0x0003010800000003, 0x000d000000010000, 0x0000000300000008, 0x0009000000200202, 0x0008010a00000010, 0x0004000000020000, 0x0001000000030000, 0x0004000000040000,
        0x0001000000010000, 0x0000001700000020, 0x3135686932336369, 0x3135686f33636f32, 0x0831687033686b32, 0x0700000002000000, 0x0301080000000300, 0x0d00000001000000,
        0x0000030000000800, 0x0900000020020200, 0x08010a0000001000, 0x0500000002000000, 0x0300000003000000, 0x0100000007000000, 0x0100000007000000, 0x00001c0000001000,
        0x7769363532636900, 0x3231636f35323635, 0x6b35323635776f38, 0x0000103077703177, 0x7d000d0000000100, 0x0200000002000000, 0x02000a0000000801, 0x0200000003000000,
        0x1000090000001002, 0x000040010a000000, 0x0000040000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001a000000, 0x6f34323268693363,
        0x3635686f38323163, 0x687034687334686b, 0x0000010000000830, 0x0100000004000700, 0x000002000a000000, 0x0401020000000400, 0x0000040208000000, 0x0d00000020000a00,
        0x0000000000000803, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x3800000001000100, 0x6932336369000000, 0x3232686934323264, 0x636f303631776934,
        0x6f343232646f3233, 0x3631776f34323268, 0x6b33686b33646b30, 0x3168703164703377, 0x0300000010317770, 0x0000030006000000, 0x0800000003010700, 0x0000010000000302,
        0x0300000002000200, 0x0000100102000000, 0x0d00000020000a00, 0x0000000000002002, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1b00000010000100,
        0x3635326369000000, 0x3135636f35376869, 0x31686b3833686f32, 0x0010306870326873, 0x0002000000010000, 0x0000000200000008, 0x010d00000004000a, 0x0000000300000002,
        0x0009000000200202, 0x0040010a00000010, 0x0005000000030000, 0x0003000000010000, 0x0001000000080000, 0x0001000000080000, 0x0000001300000800, 0x6f30316869326369,
        0x33686b38686f3863, 0x0200000010306870, 0x000004000c000000, 0x0000000008010d00, 0x0200000004000000, 0x0303080000000202, 0x0000100009000000, 0x0000000010010a00,
        0x0100000005000000, 0x0b00000003000000, 0x0b00000001000000, 0x0008000001000000, 0x3263690000001300, 0x6f38636f30316869, 0x30687033686b3868, 0x0000000200000008,
        0x010d00000004000c, 0x0000000000000008, 0x0002020200000004, 0x0000000303080000, 0x010a000000080009, 0x0000000000000008, 0x0000000300000005, 0x0000000700000001,
        0x0000000100000007, 0x0018000000010001, 0x3036353263690000, 0x383231636f386869, 0x7033686b38686f30, 0x0003000000103168, 0x0000005000020000, 0x0208000000030107,
        0x0000000100000003, 0x000300000002000a, 0x0000002001020000, 0x020d00000010000a, 0x0000000300000008, 0x0000000300000005, 0x0000000100000002, 0x0000000700000007,
        0x001b000000100001, 0x6938323163690000, 0x383231636f353768, 0x7333686b3833686f, 0x0000103168703268, 0x0301070000000300, 0x0000030208000000, 0x0000000004000a00,
        0x0200000003000000, 0x1000090000004002, 0x000020010a000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001b000000,
        0x3532776930343663, 0x6f30383231636f36, 0x7031776b36353277, 0x0001000000103077, 0x0000001400020000, 0x0002000a00000002, 0x00000008010d0000, 0x0020010200000003,
        0x00000020000a0000, 0x000300000010020d, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x3135636900000019, 0x32636f3934776932,
        0x6b3934776f383430, 0x0000083077703177, 0x1000020000000100, 0x0900000002000000, 0x08010a0000000400, 0x0200000003000000, 0x2000090000002002, 0x000020010a000000,
        0x0000050000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001d000000, 0x3177693038323163, 0x303436636f343230, 0x776b34323031776f,
        0x0000001030777031, 0x0028000200000001, 0x000a000000020000, 0x0004010d00000004, 0x0102000000030000, 0x0020000a00000020, 0x00000020020d0000, 0x0000000400000003,
        0x0000000100000003, 0x0000000400000004, 0x0080010100000001, 0x63690000001a0000, 0x6f34316869323135, 0x6b37686f32313563, 0x3168703268733368, 0x0000000300000008,
        0x0107000000100002, 0x0003020800000003, 0x0009000000020000, 0x0008010a00000004, 0x0202000000030000, 0x0020000900000020, 0x00000020010a0000, 0x0000000400000003,
        0x0000000100000003, 0x0000000400000004, 0x0080010100000001, 0x63690000001b0000, 0x6f36356869383231, 0x3832686f38323163, 0x687032687333686b, 0x0000030000000831,
        0x0700000004000200, 0x0302080000000301, 0x0a00000002000000, 0x08010d0000000400, 0x0200000003000000, 0x2000090000002002, 0x000020010a000000, 0x0000040000000300,
        0x0000010000000300, 0x0000040000000400, 0x8001010000000100, 0x690000001c000000, 0x3168693432303163, 0x6f38343032636f34, 0x32687331686b3768, 0x0100000008306870,
        0x0000200002000000, 0x0400090000000200, 0x000008010a000000, 0x2002020000000300, 0x0000200009000000, 0x0300000020010a00, 0x0300000005000000, 0x0700000001000000,
        0x0100000007000000, 0x0000200001000000, 0x3363690000001700, 0x636f323135686932, 0x686b323135686f33, 0x0000001031687033, 0x0003000700000002, 0x0000000301080000,
        0x0008000d00000001, 0x0202000000030000, 0x0010000900000020, 0x00000010010a0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001,
        0x6369000000380000, 0x6934323264693233, 0x3631776934323268, 0x32646f3233636f30, 0x6f343232686f3432, 0x6b33646b30363177, 0x31647033776b3368, 0x0008317770316870,
        0x0006000000030000, 0x0003010700000003, 0x0000000302080000, 0x0002000a00000002, 0x00000002010d0000, 0x0020010200000003, 0x00000010000a0000, 0x000300000014020d,
        0x0003000000050000, 0x000a000000030000, 0x000a000000010000, 0x0000002000010000, 0x6936636900000014, 0x646f31636f363264, 0x30647032646b3532, 0x0000000200000010,
        0x010d000000020009, 0x0000000000000019, 0x0008020200000004, 0x0000000203080000, 0x010a000000080009, 0x0000000000000010, 0x0000000300000005, 0x0000000700000001,
        0x0000000100000007, 0x001a000000010001, 0x3038323163690000, 0x3231636f32336869, 0x686b3233686f3038, 0x0000001031687033, 0x0005000200000003, 0x0000000301070000,
        0x0001000000030208, 0x0000000800020000, 0x0020010200000003, 0x00000020000a0000, 0x000000000020020d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000,
        0x0000001001010000, 0x3233636900000032, 0x3333686937316469, 0x3233636f33337769, 0x3333686f3731646f, 0x6b33646b3333776f, 0x31647033776b3368, 0x0010317770316870,
        0x0002000000040000, 0x0003010600000002, 0x0000000302070000, 0x0001000000030308, 0x00000002000d0000, 0x0010020200000003, 0x0000001000090000, 0x000300000020010a,
        0x0001000000050000, 0x0001000000020000, 0x0008000000080000, 0x0000080000010000, 0x6938636900000013, 0x686f38636f303168, 0x1030687033686b38, 0x0700000002000000,
        0x0301080000000300, 0x0500000001000000, 0x0000030000000200, 0x0900000010020200, 0x08010a0000001000, 0x0500000000000000, 0x0200000001000000, 0x0b00000001000000,
        0x010000000b000000, 0x0000130000080000, 0x3031686938636900, 0x686b38686f38636f, 0x0000000830687033, 0x0003000700000002, 0x0000000301080000, 0x0002000500000001,
        0x0202000000030000, 0x0010000900000008, 0x00000008010a0000, 0x0000000400000000, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001, 0x6369000000310000,
        0x3031686934323031, 0x636f313731776930, 0x303031686f323135, 0x33686b313731776f, 0x776431686433776b, 0x0832777032687031, 0x0200000003000000, 0x0301070000002000,
        0x0000030208000000, 0x08000a0000000100, 0x0200000003000000, 0x40000a0000002001, 0x000010020d000000, 0x0000040000000300, 0x0000010000000300, 0x0000040000000400,
        0x0100010000000100, 0x6900000018000000, 0x3431686936353263, 0x31686f363532636f, 0x0831687033686b34, 0x0700000002000000, 0x0301080000000300, 0x0200000001000000,
        0x0000030000000800, 0x0a00000020010200, 0x08020d0000002000, 0x0500000000000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x0000170000002000,
        0x6938636932336700, 0x686f38636f383268, 0x31687033686b3832, 0x0000000200000010, 0x0108000000030007, 0x0000000100000003, 0x000300000004000d, 0x0000002002020000,
        0x010a000000200009, 0x0000000300000010, 0x0000000300000004, 0x0000000400000001, 0x0000000100000004, 0x0018000000800101, 0x6938323163690000, 0x383231636f383268,
        0x7033686b3832686f, 0x0003000000083168, 0x0000000400020000, 0x0208000000030107, 0x0000000200000003, 0x010d00000004000a, 0x0000000300000004, 0x0009000000200202,
        0x0020010a00000020, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001c00000008, 0x3432326869336369, 0x31686f383637636f,
        0x3168733631686b34, 0x0000001030687036, 0x0010000700000002, 0x0000000401080000, 0x0008000a00000001, 0x0102000000040000, 0x0004020800000004, 0x00000040000a0000,
        0x00030000000e030d, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x353263690000001c, 0x6f36333133776936, 0x3133776f38323163,
        0x30777031776b3633, 0x0000000100000008, 0x0002000000080002, 0x00000008000a0000, 0x000300000004010d, 0x0000002002020000, 0x010a000000100009, 0x0000000300000010,
        0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x002c000000200101, 0x3868693163690000, 0x3536303031776930, 0x3038686f31636f30, 0x303536303031776f,
        0x703332776b31686b, 0x0008313177703068, 0x0008000000010000, 0x0000000100000003, 0x000400000008000d, 0x0000000202020000, 0x0009000000080308, 0x0008010a00000020,
        0x0004000000010000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000002c00000020, 0x6930386869316369, 0x6f30353630303177, 0x776f3038686f3163,
        0x686b303536303031, 0x3068703332776b31, 0x0000000831317770, 0x0003000800000001, 0x000d000000010000, 0x0000000400000008, 0x0308000000040202, 0x0020000900000008,
        0x00000008010a0000, 0x0000000500000001, 0x0000000300000003, 0x0000000100000007, 0x0020000100000007, 0x6369000000180000, 0x6f39316869323135, 0x3931686f36383463,
        0x001031687033686b, 0x000d000000010000, 0x0000000000000013, 0x0020020200000003, 0x0000001000090000, 0x000300000040010a, 0x0003000000050000, 0x0007000000010000,
        0x0001000000070000, 0x0000000100010000, 0x323363690000001c, 0x6f34323031776930, 0x3031776f30343663, 0x30777031776b3432, 0x0000000100000010, 0x00020000000a0002,
        0x00000004000a0000, 0x000300000008010d, 0x0000002001020000, 0x020d00000020000a, 0x0000000300000010, 0x0000000300000004, 0x0000000400000001, 0x0000000100000004,
        0x001a000000800101, 0x6938323163690000, 0x3135636f34383777, 0x776b343837776f32, 0x0000000830777031, 0x0004000200000001, 0x000a000000020000, 0x0004010d00000008,
        0x0202000000030000, 0x0020000900000020, 0x00000020010a0000, 0x0000000500000003, 0x0000000200000003, 0x0000000700000001, 0x0010000100000007, 0x6369000000180000,
        0x6f39316869323135, 0x3931686f36383463, 0x001031687033686b, 0x0107000000030000, 0x0003020800000003, 0x00000010000a0000, 0x0002000200000002, 0x0000000801050000,
        0x0040020200000003, 0x0000001000090000, 0x000300000020010a, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000500010000, 0x3135636900000018,
        0x35636f3431686932, 0x686b3431686f3231, 0x0000000831687033, 0x0004000200000003, 0x0000000301070000, 0x0001000000030208, 0x0000000400020000, 0x0020010200000003,
        0x00000010000a0000, 0x00000000000e020d, 0x0003000000050000, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x3231636900000017, 0x33636f3431686938,
        0x33686b3431686f32, 0x0200000010316870, 0x0000030007000000, 0x0100000003010800, 0x0000040002000000, 0x2001020000000300, 0x000010000a000000, 0x0000000008020d00,
        0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3663690000001b00, 0x636f323368693034, 0x6b3631686f303436, 0x3168703268733368,
        0x0000000300000010, 0x0107000000140002, 0x0003020800000003, 0x000a000000020000, 0x0002010d00000008, 0x0102000000030000, 0x0010000a00000020, 0x00000008020d0000,
        0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0080010100000001, 0x63690000001a0000, 0x3633313377693436, 0x3133776f3436636f, 0x30777031776b3633,
        0x0000000100000008, 0x0000000000020002, 0x0202000000030000, 0x0010000900000020, 0x00000040010a0000, 0x0000000500000000, 0x0000000100000003, 0x0000000700000007,
        0x0001000100000001, 0x63690000001c0000, 0x3532776930323931, 0x6f30383231636f36, 0x7031776b36353277, 0x0001000000103077, 0x0000003c00020000, 0x0004000a00000002,
        0x00000004010d0000, 0x0020010200000003, 0x00000020000a0000, 0x000300000010020d, 0x0003000000050000, 0x0001000000020000, 0x000a0000000a0000, 0x0000002000010000,
        0x6931636900000015, 0x6f3233636f353264, 0x647035646b353264, 0x0000040000001032, 0x0700000005010600, 0x0503080000000502, 0x000002000a000000, 0x0800050000000100,
        0x0200000003000000, 0x2000090000001002, 0x000010010a000000, 0x0000050000000200, 0x0000030000000300, 0x0000010000000700, 0x1000010000000700, 0x690000001b000000,
        0x3537686938323163, 0x33686f383231636f, 0x7032687333686b38, 0x0001000000103168, 0x00000013000c0000, 0x0004010200000002, 0x00000002000a0000, 0x0020020200000003,
        0x0000001000090000, 0x000300000040010a, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x313563690000001a, 0x636f343837776932,
        0x343837776f363532, 0x000830777031776b, 0x0002000000010000, 0x0000000100000010, 0x000300000008000a, 0x0000002002020000, 0x010a000000200009, 0x0000000200000020,
        0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x000b000000010001, 0x3834303263690000, 0x000010383231636f, 0x1000020000000100, 0x0200000001000000,
        0x0000030000000800, 0x0a00000010010200, 0x08020d0000001000, 0x0400000000000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x00001a0000008001,
        0x7769323135636900, 0x383231636f343837, 0x31776b343837776f, 0x0100000008307770, 0x0000100002000000, 0x08000a0000000100, 0x0200000003000000, 0x2000090000002002,
        0x000010010a000000, 0x0000050000000200, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001c000000, 0x3277693038323163, 0x30383231636f3635,
        0x31776b363532776f, 0x0100000010307770, 0x0000280002000000, 0x02000a0000000200, 0x000008010d000000, 0x2001020000000300, 0x000040000a000000, 0x0300000008020d00,
        0x0300000005000000, 0x0100000002000000, 0x0700000007000000, 0x0000100001000000, 0x3563690000001d00, 0x3434343177693231, 0x776f34323031636f, 0x7031776b34343431,
        0x0001000000103077, 0x00000020000a0000, 0x0002000200000002, 0x0000000801050000, 0x0040020200000003, 0x0000001000090000, 0x000300000020010a, 0x0003000000050000,
        0x000a000000030000, 0x000a000000010000, 0x0000002001010000, 0x6938636900000014, 0x646f38636f353264, 0x31647035646b3332, 0x0000000100000010, 0x000000000017000c,
        0x0202000000030000, 0x0010000900000008, 0x00000010010a0000, 0x0000000500000002, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x6369000000180000,
        0x6f38686930383231, 0x38686f3038323163, 0x001031687033686b, 0x0002000000030000, 0x0003010700000028, 0x0000000302080000, 0x0002000a00000001, 0x0102000000030000,
        0x0010000a00000020, 0x00000008020d0000, 0x0000000500000003, 0x0000000100000001, 0x0000000700000007, 0x0001000100000001, 0x6369000000140000, 0x32636f3635686932,
        0x7033686b3635686f, 0x0001000000103168, 0x0000000300070000, 0x0008000d00000001, 0x0102000000040000, 0x0003020800000002, 0x00000010000a0000, 0x000000000007030d,
        0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x693363690000001b, 0x32636f3432303168, 0x686b363532686f34, 0x0833687034687337,
        0x0700000001000000, 0x0000020000000700, 0x0d00000002000a00, 0x0000040000000801, 0x0800000004010200, 0x10000a0000000802, 0x000020030d000000, 0x0000040000000100,
        0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001a000000, 0x3135686938323163, 0x686f383231636f32, 0x687033686b323135, 0x0000030000000831,
        0x0700000004000200, 0x0302080000000301, 0x0a00000002000000, 0x08010d0000000400, 0x0200000003000000, 0x20000a0000002001, 0x000010020d000000, 0x0000050000000300,
        0x0000010000000300, 0x0000070000000700, 0x1000010000000100, 0x690000001b000000, 0x3537686938323163, 0x33686f383231636f, 0x7032687333686b38, 0x0003000000103168,
        0x0000000400020000, 0x0208000000030107, 0x0000000200000003, 0x010d00000002000a, 0x0000000300000004, 0x0009000000200202, 0x0040010a00000010, 0x0004000000030000,
        0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000001b00000080, 0x3377693635326369, 0x6f3436636f363331, 0x31776b3633313377, 0x0100000008307770,
        0x0000080002000000, 0x08000a0000000200, 0x000002010d000000, 0x2002020000000300, 0x0000100009000000, 0x0300000008010a00, 0x0300000004000000, 0x0700000001000000,
        0x0100000007000000, 0x0000180001000000, 0x3233670000001700, 0x6f36356869386369, 0x686b3635686f3863, 0x0000000831687033, 0x0003000700000001, 0x000d000000010000,
        0x0000000400000008, 0x0308000000080202, 0x000c000900000004, 0x00000008010a0000, 0x0000000400000002, 0x0000000100000003, 0x0000000700000007, 0x0005000100000001,
        0x6369000000180000, 0x6f36356869383231, 0x3635686f38323163, 0x000831687033686b, 0x0002000000030000, 0x0003010700000002, 0x0000000302080000, 0x0002000200000001,
        0x0102000000030000, 0x0020000a00000020, 0x0000000e020d0000, 0x0000000400000000, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001, 0x63690000001a0000,
        0x636f303033686933, 0x6b303531686f3233, 0x3168703268733368, 0x0000000100000008, 0x0002000000030007, 0x00000002000a0000, 0x000400000002010d, 0x0000000401020000,
        0x000a000000080208, 0x0019030d00000010, 0x0004000000030000, 0x0001000000010000, 0x0004000000040000, 0x0101000000010000, 0x0000001c00000020, 0x6931636936353267,
        0x31636f3639303477, 0x776b36393034776f, 0x0000000830777031, 0x0000000000000000, 0x0020010100000002, 0x0000001000090000, 0x0000000500000000, 0x0000000100000003,
        0x0000000700000007, 0x0010000100000001, 0x6369000000180000, 0x6f38336869363532, 0x3833686f36353263, 0x001031687033686b, 0x0002000000030000, 0x0003010700000008,
        0x0000000302080000, 0x0002000a00000002, 0x00000008010d0000, 0x0020020200000003, 0x0000001000090000, 0x000300000040010a, 0x0003000000050000, 0x000a000000030000,
        0x000a000000010000, 0x0000002001010000, 0x6938636900000014, 0x646f38636f353264, 0x32647035646b3532, 0x0000000200000010, 0x010d00000019000c, 0x0000000000000005,
        0x0008020200000003, 0x0000001000090000, 0x000200000010010a, 0x0003000000050000, 0x0004000000010000, 0x0001000000040000, 0x0000002001010000, 0x693463690000002a,
        0x3177693038303168, 0x686f31636f303239, 0x3931776f30383031, 0x33776b33686b3032, 0x0010317770316870, 0x0007000000010000, 0x0000000000000003, 0x0004020200000004,
        0x0000000803080000, 0x010a000000200009, 0x0000000300000010, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x002a000000200101, 0x3168693463690000,
        0x3239317769303830, 0x3031686f31636f30, 0x30323931776f3038, 0x687033776b33686b, 0x0000001031777031, 0x0003000700000001, 0x000d000000010000, 0x0000000400000008,
        0x0308000000040202, 0x0020000900000004, 0x00000010010a0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x63690000001a0000,
        0x3436776930383231, 0x776f30383231636f, 0x30777031776b3436, 0x0000000100000010, 0x0002000000280002, 0x00000002000a0000, 0x000300000002010d, 0x0000002001020000,
        0x020d00000010000a, 0x0000000300000010, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x0018000000010001, 0x6930323363690000, 0x303233636f343668,
        0x7033686b3436686f, 0x0003000000103168, 0x0000000a00020000, 0x0208000000030107, 0x0000000200000003, 0x010d00000002000a, 0x0000000300000004, 0x000a000000200102,
        0x0010020d00000040, 0x0005000000030000, 0x0001000000030000, 0x0004000000040000, 0x0001000000010000, 0x0000001b00000080, 0x3133776934366369, 0x6f363532636f3633,
        0x31776b3633313377, 0x0000000010307770, 0x0a00000001000000, 0x0000030000000400, 0x0a00000040010200, 0x20020d0000004000, 0x0400000003000000, 0x0100000003000000,
        0x0400000004000000, 0x0100000001000000, 0x0000160000000100, 0x3168693631636900, 0x31686f3631636f36, 0x0830687033686b34, 0x0700000002000000, 0x0201080000000300,
        0x0d00000001000000, 0x0000040000000200, 0x0800000010010200, 0x08000a0000000202, 0x000008030d000000, 0x0000040000000300, 0x0000010000000300, 0x0000040000000400,
        0x0100010000000100, 0x6900000015000000, 0x636f323368693363, 0x686b3033686f3031, 0x0000000830687033, 0x0003000700000002, 0x0000000301080000, 0x0004000d00000001,
        0x0102000000030000, 0x0008000a00000020, 0x00000008020d0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001, 0x6369000000180000,
        0x636f383231686933, 0x33686b3436686f38, 0x0008306870326873, 0x0007000000010000, 0x0000000100000003, 0x000400000002000d, 0x0000000401020000, 0x000a000000080208,
        0x0008030d00000008, 0x0005000000030000, 0x0001000000030000, 0x0004000000040000, 0x0001000000010000, 0x0000001a00000080, 0x3777693832316369, 0x6f323135636f3438,
        0x7031776b34383777, 0x0001000000103077, 0x0000000200020000, 0x0004000a00000001, 0x0102000000030000, 0x0040000a00000040, 0x0000001c020d0000, 0x0000000500000003,
        0x0000000300000001, 0x0000000100000008, 0x0020000100000008, 0x6369000000300000, 0x3268693632646936, 0x36636f3732776938, 0x3632686f3632646f, 0x6b33646b3632776f,
        0x31647038776b3368, 0x0010337770306870, 0x000c000000020000, 0x001a010d00000002, 0x0002000000010000, 0x0000000300000002, 0x0009000000080208, 0x0010010a00000010,
        0x0005000000000000, 0x0002000000030000, 0x0007000000010000, 0x0001000000070000, 0x0000001800000010, 0x6869343230316369, 0x686f3631636f3833, 0x31687033686b3833,
        0x0000000200000010, 0x0108000000030007, 0x0000000200000003, 0x0105000000020002, 0x0000000300000002, 0x0009000000400202, 0x0010010a00000010, 0x0004000000030000,
        0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000002a00000020, 0x3830316869346369, 0x6f30323931776930, 0x30383031686f3163, 0x686b30323931776f,
        0x7031687033776b33, 0x0001000000083177, 0x0000000300070000, 0x0002000d00000001, 0x0202000000040000, 0x0008030800000004, 0x0000002000090000, 0x000200000008010a,
        0x0003000000050000, 0x0004000000010000, 0x0001000000040000, 0x0000008000010000, 0x313563690000001a, 0x35636f3431686932, 0x33686b37686f3231, 0x0010316870326873,
        0x0002000000030000, 0x0003010700000008, 0x0000000302080000, 0x0004000a00000002, 0x00000008010d0000, 0x0040020200000003, 0x0000002000090000, 0x000300000040010a,
        0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x353263690000001b, 0x32636f3832686936, 0x686b3431686f3635, 0x0831687032687333,
        0x0200000003000000, 0x0301070000000800, 0x0000030208000000, 0x0400090000000200, 0x000008010a000000, 0x2002020000000300, 0x0000200009000000, 0x0300000020010a00,
        0x0300000004000000, 0x0400000001000000, 0x0100000004000000, 0x0000800101000000, 0x3163690000001b00, 0x3639317769343230, 0x31776f323135636f, 0x30777031776b3639,
        0x0000000100000008, 0x0002000000200002, 0x00000008000a0000, 0x000300000004010d, 0x0000002002020000, 0x010a000000200009, 0x0000000300000020, 0x0000000300000005,
        0x0000000700000001, 0x0000000100000007, 0x0018000000010001, 0x6936393863690000, 0x383231636f343677, 0x7031776b3436776f, 0x0001000000103077, 0x0000000700020000,
        0x0008000200000001, 0x0102000000030000, 0x0010000a00000010, 0x00000008020d0000, 0x0000000400000000, 0x0000000300000003, 0x0000000100000007, 0x0008000100000007,
        0x63690000001b0000, 0x6f33377769323135, 0x3533776f32313563, 0x777032777334776b, 0x0000010000000830, 0x0200000008000900, 0x0000080102000000, 0x0300000008000a00,
        0x0000200102000000, 0x0d00000020000a00, 0x0000030000001002, 0x0000030000000400, 0x0000040000000100, 0x0000010000000400, 0x1400000001000100, 0x6930326369000000,
        0x686f3035636f3668, 0x0830687033686b34, 0x0700000002000000, 0x0301080000000300, 0x0300000000000000, 0x0000200102000000, 0x0d00000008000a00, 0x0000000000000402,
        0x0000030000000500, 0x0000040000000100, 0x0000010000000400, 0x1900000080000100, 0x3231356369000000, 0x3032636f39347769, 0x776b3934776f3834, 0x0000001030777031,
        0x0008000200000001, 0x0009000000020000, 0x0004010a00000008, 0x0202000000030000, 0x0010000900000040, 0x00000040010a0000, 0x0000000400000003, 0x0000000100000003,
        0x0000000700000007, 0x0006000100000001, 0x6369000000260000, 0x6938326869323135, 0x323135636f313277, 0x3132776f3832686f, 0x687033776b33686b, 0x0000000831777031,
        0x0008000200000003, 0x0000000301070000, 0x0001000000030208, 0x0000000200020000, 0x0020010200000003, 0x00000010000a0000, 0x000000000015020d, 0x0003000000050000,
        0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x353263690000001a, 0x636f363168693036, 0x3631686f30383231, 0x001031687033686b, 0x0002000000030000,
        0x000301070000000a, 0x0000000302080000, 0x0008000200000001, 0x0102000000030000, 0x0020000a00000020, 0x00000010020d0000, 0x0000000500000000, 0x0000000300000001,
        0x0000000100000008, 0x0800000100000008, 0x6369000000130000, 0x38636f3031686938, 0x687033686b38686f, 0x0000020000001030, 0x0d00000008000c00, 0x0000000000000401,
        0x0802020000000300, 0x0000100009000000, 0x0000000010010a00, 0x0100000005000000, 0x0b00000003000000, 0x0b00000001000000, 0x0008000001000000, 0x3863690000001300,
        0x6f38636f30316869, 0x30687033686b3868, 0x0000000200000008, 0x010d00000008000c, 0x0000000000000002, 0x0008020200000003, 0x0000000800090000, 0x000000000008010a,
        0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000002001010000, 0x3135636900000019, 0x31636f3331686932, 0x6b3331686f343230, 0x0000103168703368,
        0x2000020000000300, 0x0000030107000000, 0x0200000003020800, 0x000002000a000000, 0x0300000008010d00, 0x0000100202000000, 0x0a00000020000900, 0x0000030000002001,
        0x0000030000000500, 0x0000040000000100, 0x0000010000000400, 0x1800000080000100, 0x3832316369000000, 0x3231636f38326869, 0x33686b3832686f38, 0x0300000010316870,
        0x0000020002000000, 0x0800000003010700, 0x0000020000000302, 0x0d00000002000a00, 0x0000030000000401, 0x0900000040020200, 0x40010a0000002000, 0x0400000003000000,
        0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x0000160000002001, 0x3168693031636900, 0x31686f3032636f35, 0x0830687033686b33, 0x0700000002000000,
        0x0201080000000300, 0x0a00000002000000, 0x04010d0000000400, 0x0200000004000000, 0x0203080000001002, 0x0000200009000000, 0x0300000008010a00, 0x0100000005000000,
        0x0400000001000000, 0x0100000004000000, 0x0000010001000000, 0x3163690000002c00, 0x3031776930386869, 0x6f31636f30353630, 0x303031776f303868, 0x776b31686b303536,
        0x3177703068703332, 0x0000000000001031, 0x04000d0000000100, 0x0800000003000000, 0x10000a0000001701, 0x000010020d000000, 0x0000050000000000, 0x0000010000000100,
        0x0000070000000700, 0x0100010000000100, 0x690000002c000000, 0x7769303868693163, 0x636f303536303031, 0x31776f3038686f31, 0x31686b3035363030, 0x703068703332776b,
        0x0000000010313177, 0x0300000000000000, 0x0000170108000000, 0x0d00000010000a00, 0x0000000000001002, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700,
        0x1600000001000100, 0x6869346369000000, 0x6f303233636f3436, 0x687033686b343668, 0x0000010000001031, 0x0000000003000700, 0x0200000004000000, 0x0402080000000401,
        0x000040000a000000, 0x0300000010030d00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3163690000001900, 0x6f32336869303832,
        0x3233686f30343663, 0x001031687033686b, 0x0002000000030000, 0x0003010700000028, 0x0000000302080000, 0x0002000a00000002, 0x00000004010d0000, 0x0020010200000003,
        0x00000040000a0000, 0x000300000008020d, 0x0003000000050000, 0x0001000000020000, 0x0007000000070000, 0x0000001000010000, 0x3031636900000019, 0x636f383368693432,
        0x6b3833686f343233, 0x0000103168703368, 0x0301070000000300, 0x0000030208000000, 0x020000000b000a00, 0x0000040002000000, 0x0300000008010500, 0x0000400202000000,
        0x0a00000010000900, 0x0000030000002001, 0x0000030000000400, 0x0000070000000300, 0x0000070000000100, 0x1a00000001000100, 0x6869336369000000, 0x383231636f343232,
        0x7334686b3635686f, 0x0000083068703468, 0x08000c0000000100, 0x0a00000001000000, 0x0000040000000200, 0x0800000004010200, 0x08000a0000000402, 0x000010030d000000,
        0x0000050000000000, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000017000000, 0x3934776936393863, 0x3934776f3233636f, 0x001030777031776b,
        0x0002000000010000, 0x0000000100000007, 0x0003000000080002, 0x0000001001020000, 0x020d00000010000a, 0x0000000000000008, 0x0000000300000005, 0x0000000700000003,
        0x0000000700000001, 0x0018000000100001, 0x3432303163690000, 0x3631636f38336869, 0x7033686b3833686f, 0x0001000000103168, 0x00000026000d0000, 0x0000000300000000,
        0x0009000000200202, 0x0010010a00000010, 0x0005000000030000, 0x0003000000030000, 0x0001000000070000, 0x0001000000070000, 0x0000001700000018, 0x6869386369323367,
        0x35686f38636f3635, 0x1031687033686b36, 0x0c00000002000000, 0x0e010d0000000200, 0x0300000000000000, 0x0000080202000000, 0x0a00000020000900, 0x0000000000001001,
        0x0000030000000500, 0x0000010000000200, 0x0000070000000700, 0x1c00000010000100, 0x3635326369000000, 0x636f353236357769, 0x323635776f383231, 0x1030777031776b35,
        0x0a00000001000000, 0x0000010000000400, 0x0300000004000200, 0x0000400202000000, 0x0a00000010000900, 0x0000030000002001, 0x0000030000000500, 0x0000040000000100,
        0x0000010000000400, 0x1600000020010100, 0x3434316369000000, 0x343431636f326869, 0x687033686b32686f, 0x0000030000001031, 0x0700000005000200, 0x0302080000000301,
        0x0900000002000000, 0x02010a0000000200, 0x0200000003000000, 0x1000090000002002, 0x000010010a000000, 0x0000040000000300, 0x0000010000000300, 0x0000070000000700,
        0x0100010000000100, 0x6900000016000000, 0x3031686932313563, 0x6b3031686f36636f, 0x0000083168703368, 0x0400020000000300, 0x0000030107000000, 0x0100000003020800,
        0x0000080002000000, 0x1001020000000300, 0x000008000a000000, 0x0000000008020d00, 0x0300000005000000, 0x0400000001000000, 0x0100000004000000, 0x0000010001000000,
        0x3163690000001700, 0x636f333168693832, 0x686b3331686f3233, 0x0000001031687033, 0x0003000700000002, 0x0000000301080000, 0x0004000200000001, 0x0102000000030000,
        0x0010000a00000020, 0x00000008020d0000, 0x0000000500000000, 0x0000000300000001, 0x0000000100000008, 0x0800000100000008, 0x6369000000140000, 0x31636f3031686938,
        0x7033686b38686f30, 0x0002000000103068, 0x00000004000c0000, 0x000000000008010d, 0x0202000000030000, 0x0010000900000008, 0x00000010010a0000, 0x0000000500000000,
        0x0000000300000001, 0x000000010000000b, 0x080000010000000b, 0x6369000000140000, 0x31636f3031686938, 0x7033686b38686f30, 0x0002000000083068, 0x00000002000c0000,
        0x000000000008010d, 0x0202000000030000, 0x0008000900000008, 0x00000010010a0000, 0x0000000500000000, 0x0000000100000003, 0x0000000400000004, 0x0020010100000001,
        0x6369000000170000, 0x636f363134686933, 0x6b363134686f3631, 0x0000103168703368, 0x0300070000000100, 0x0400000000000000, 0x0000040202000000, 0x0900000008030800,
        0x10010a0000002000, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x0000170000002001, 0x3134686933636900, 0x34686f3631636f36,
        0x31687033686b3631, 0x0000000100000010, 0x0001000000030007, 0x00000008000d0000, 0x0004020200000004, 0x0000000403080000, 0x010a000000200009, 0x0000000300000010,
        0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x001d000000010001, 0x3032393163690000, 0x636f343230317769, 0x323031776f303436, 0x1030777031776b34,
        0x0200000001000000, 0x0000020000003c00, 0x0d00000004000a00, 0x0000030000000401, 0x0a00000020010200, 0x20020d0000002000, 0x0400000003000000, 0x0100000001000000,
        0x0400000004000000, 0x0100000001000000, 0x00001c0000002001, 0x3163693432326700, 0x636f363331337769, 0x6b36333133776f31, 0x0000083077703177, 0x0000000000000000,
        0x2001010000000200, 0x0000100009000000, 0x0000050000000000, 0x0000010000000100, 0x0000040000000400, 0x0100010000000100, 0x6700000030000000, 0x6869316369383832,
        0x3635327769383231, 0x383231686f31636f, 0x35686b363532776f, 0x776433686435776b, 0x1038777038687033, 0x0700000001000000, 0x0000000000000500, 0x1000010000000300,
        0x0000050108000000, 0x0000000010020d00, 0x0300000004000000, 0x0400000001000000, 0x0100000004000000, 0x0000200001000000, 0x3233670000001700, 0x6f38326869386369,
        0x686b3832686f3863, 0x0000000831687033, 0x0003000700000001, 0x000d000000010000, 0x0000000400000008, 0x0308000000080202, 0x0020000900000004, 0x00000008010a0000,
        0x0000000500000002, 0x0000000100000001, 0x0000000800000008, 0x0400010100000001, 0x3367000000170000, 0x3635686934636932, 0x6b3635686f34636f, 0x0000103168703368,
        0x0300070000000100, 0x0d00000001000000, 0x0000040000000800, 0x0800000004020200, 0x1000090000000303, 0x000010010a000000, 0x0000050000000000, 0x0000010000000300,
        0x0000040000000400, 0x0100010000000100, 0x6900000017000000, 0x3233686938323163, 0x3233686f3436636f, 0x001031687033686b, 0x0007000000020000, 0x0003010800000003,
        0x0002000000010000, 0x0000000300000004, 0x000a000000200102, 0x0008020d00000010, 0x0004000000000000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000,
        0x0000001800000080, 0x3168693635326369, 0x686f363532636f34, 0x31687033686b3431, 0x0000000300000008, 0x0107000000080002, 0x0003020800000003, 0x0009000000020000,
        0x0004010a00000004, 0x0202000000030000, 0x0020000900000020, 0x00000020010a0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001,
        0x6369000000180000, 0x6f38323164693631, 0x383231646f363163, 0x001031647033646b, 0x0006000000030000, 0x0003010700000003, 0x0000000302080000, 0x0004000d00000001,
        0x0102000000030000, 0x0010000a00000010, 0x00000020020d0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001, 0x6369000000170000,
        0x6f32336869383231, 0x6b3233686f343663, 0x0000083168703368, 0x0300070000000200, 0x0000030108000000, 0x0400020000000100, 0x0200000003000000, 0x10000a0000002001,
        0x000008020d000000, 0x0000040000000000, 0x0000020000000300, 0x0000070000000100, 0x1000010000000700, 0x690000001f000000, 0x3177693838303163, 0x3135636f30303035,
        0x3030303531776f32, 0x000830777031776b, 0x000a000000010000, 0x0000000200000010, 0x0105000000040002, 0x0000000300000008, 0x0009000000200202, 0x0020010a00000010,
        0x0005000000030000, 0x0003000000030000, 0x0001000000070000, 0x0001000000070000, 0x0000001a00000018, 0x7769383430326369, 0x38343032636f3934, 0x7031776b3934776f,
        0x0001000000103077, 0x00000031000d0000, 0x0004010200000002, 0x00000004000a0000, 0x0020020200000003, 0x0000001000090000, 0x000300000040010a, 0x0001000000050000,
        0x0008000000030000, 0x0008000000010000, 0x0000006400010000, 0x693163690000001e, 0x636f323335393277, 0x35303935776f3031, 0x703577733031776b, 0x0001000000103077,
        0x00000040000d0000, 0x0000000300000000, 0x0009000000080208, 0x0010010a0000000a, 0x0004000000000000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000,
        0x0000001b00000080, 0x3133776934366369, 0x6f363532636f3633, 0x31776b3633313377, 0x0100000008307770, 0x0000020002000000, 0x08000d0000000100, 0x0200000003000000,
        0x2000090000002002, 0x000020010a000000, 0x0000040000000200, 0x0000010000000300, 0x0000040000000400, 0x0100010000000100, 0x6900000017000000, 0x3233326869323363,
        0x343232686f33636f, 0x000830687039686b, 0x0007000000020000, 0x0009010800000009, 0x000d000000010000, 0x0000000300000002, 0x000a000000200102, 0x001c020d00000008,
        0x0004000000010000, 0x0001000000030000, 0x0004000000040000, 0x0001000000010000, 0x0000001700000020, 0x3332686932336369, 0x3232686f33636f32, 0x0830687039686b34,
        0x0700000002000000, 0x0901080000000900, 0x0d00000001000000, 0x0000030000000400, 0x0900000020020200, 0x08010a0000002000, 0x0400000001000000, 0x0300000001000000,
        0x0100000008000000, 0x0100000008000000, 0x0000130000080000, 0x3031686932636900, 0x686b38686f38636f, 0x0000000830687033, 0x0008000c00000001, 0x0003000000000000,
        0x0000000802020000, 0x010a000000100009, 0x0000000000000008, 0x0000000300000005, 0x0000000400000001, 0x0000000100000004, 0x0019000000200001, 0x3163693233670000,
        0x31636f3431686936, 0x33686b3431686f36, 0x0200000010316870, 0x0000030007000000, 0x0100000003010800, 0x000004000d000000, 0x2002020000000300, 0x0000200009000000,
        0x0300000010010a00, 0x0100000004000000, 0x0400000001000000, 0x0100000004000000, 0x0000010001000000, 0x3233670000001700, 0x6f38326869326369, 0x686b3832686f3263,
        0x0000000831687033, 0x0003000700000001, 0x000d000000010000, 0x0000000400000002, 0x0208000000020102, 0x0008000a00000003, 0x0000000e030d0000, 0x0000000400000000,
        0x0000000300000001, 0x0000000100000008, 0x0001000100000008, 0x63690000001a0000, 0x636f343232686933, 0x6b3635686f383231, 0x3068703468733468, 0x0000000100000010,
        0x00000000001c000c, 0x0102000000030000, 0x0020000a00000003, 0x00000008020d0000, 0x0000000400000000, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001,
        0x6369000000140000, 0x33636f3968693233, 0x7033686b37686f32, 0x0002000000083068, 0x0000000300070000, 0x0001000000030108, 0x0000000200020000, 0x0010010200000003,
        0x00000010000a0000, 0x000000000007020d, 0x0003000000050000, 0x000a000000030000, 0x000a000000010000, 0x0000004001010000, 0x343663690000001c, 0x636f323230317769,
        0x303031776f383231, 0x3077703631776b37, 0x0000000200000010, 0x010d000000020009, 0x0000000200000013, 0x000a000000020102, 0x0000000300000002, 0x0009000000200202,
        0x0040010a00000008, 0x0004000000020000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000001a00000080, 0x3432326869336369, 0x3131686f3436636f,
        0x7032687337686b32, 0x0001000000083368, 0x0000000700070000, 0x0002000a00000002, 0x00000008010d0000, 0x0004020200000004, 0x0000000803080000, 0x010a000000200009,
        0x0000000300000020, 0x0000000300000005, 0x0000000700000003, 0x0000000700000001, 0x0019000000100001, 0x3432303163690000, 0x3233636f38336869, 0x33686b3833686f34,
        0x0200000010316870, 0x000002000c000000, 0x0100000026010d00, 0x000004000a000000, 0x2002020000000300, 0x0000100009000000, 0x0300000020010a00, 0x0300000005000000,
        0x0700000003000000, 0x0700000001000000, 0x0000100001000000, 0x3163690000001f00, 0x3030353177693832, 0x6f34323031636f30, 0x776b303030353177, 0x0000001030777031,
        0x00fa000d00000001, 0x0102000000020000, 0x0008000a00000004, 0x0202000000030000, 0x0010000900000020, 0x00000020010a0000, 0x0000000500000003, 0x0000000100000001,
        0x0000000800000008, 0x0064000100000001, 0x63690000001e0000, 0x3233353932776931, 0x3935776f3031636f, 0x77733031776b3530, 0x0000001030777035, 0x0000000000000000,
        0x000a020800000003, 0x0000000a00090000, 0x000000000010010a, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x3436636900000018,
        0x36636f3436686930, 0x686b3436686f3034, 0x0000001031687033, 0x0014000200000003, 0x0000000301070000, 0x0002000000030208, 0x00000002000a0000, 0x000300000004010d,
        0x0000002001020000, 0x020d00000020000a, 0x0000000300000010, 0x0000000300000005, 0x0000000400000001, 0x0000000100000004, 0x0017000000200001, 0x6869323363690000,
        0x686f33636f323332, 0x687039686b343232, 0x0000020000001030, 0x0800000009000700, 0x0000010000000901, 0x0300000008000d00, 0x0000200202000000, 0x0a00000020000900,
        0x0000030000001001, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1700000020000100, 0x6932336369000000, 0x6f33636f32333268, 0x7039686b34323268,
        0x0002000000103068, 0x0000000900070000, 0x0001000000090108, 0x00000008000d0000, 0x0020020200000003, 0x0000002000090000, 0x000300000010010a, 0x0003000000040000,
        0x0007000000030000, 0x0007000000010000, 0x0000000400010000, 0x3532636900000016, 0x3135636f38646936, 0x7033646b38646f32, 0x0001000000083164, 0x00000008000c0000,
        0x0002010200000002, 0x00000008000a0000, 0x0020010200000003, 0x00000008000a0000, 0x000300000010020d, 0x0003000000050000, 0x0007000000030000, 0x0007000000010000,
        0x0000001000010000, 0x693363690000001a, 0x3436636f30303368, 0x37686b303531686f, 0x0010336870326873, 0x000c000000020000, 0x000a010d00000002, 0x0004000000000000,
        0x0000000402020000, 0x0009000000080308, 0x0040010a00000010, 0x0004000000030000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000001c00000080,
        0x3268693231356369, 0x6f34323031636f38, 0x687331686b343168, 0x0000000830687032, 0x0010000200000001, 0x0009000000020000, 0x0008010a00000004, 0x0202000000030000,
        0x0020000900000020, 0x00000020010a0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0080010100000001, 0x63690000001b0000, 0x3639317769363532,
        0x776f34323031636f, 0x777031776b363931, 0x0000010000000830, 0x0200000008000200, 0x000008000a000000, 0x0300000004010d00, 0x0000200202000000, 0x0a00000020000900,
        0x0000030000002001, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1c00000001000100, 0x3034366369000000, 0x636f363930347769, 0x393034776f303233,
        0x1030777031776b36, 0x0200000001000000, 0x0000010000001400, 0x0300000002000a00, 0x0000200102000000, 0x0d00000020000a00, 0x0000030000002002, 0x0000010000000400,
        0x0000080000000300, 0x0000080000000100, 0x1e00000064000100, 0x7769316369000000, 0x31636f3233353932, 0x6b35303935776f30, 0x7770357773303177, 0x0000010000001030,
        0x0000000005000d00, 0x0900000002000000, 0x10010a0000001000, 0x0400000000000000, 0x0100000001000000, 0x0400000004000000, 0x0100000001000000, 0x00001b0000000100,
        0x6932636932336700, 0x6f32636f32313168, 0x687333686b363568, 0x0000000831687032, 0x0003000700000001, 0x000d000000010000, 0x0000000400000002, 0x0208000000020102,
        0x0008000a00000003, 0x0000000e030d0000, 0x0000000400000000, 0x0000000200000003, 0x0000000700000001, 0x0002000100000007, 0x63690000001a0000, 0x3030316869363532,
        0x31686f363532636f, 0x31687033686b3030, 0x0000000300000008, 0x0208000000030107, 0x0008000a00000003, 0x0002000000020000, 0x0004010500000008, 0x0102000000030000,
        0x0019020500000010, 0x00000020000a0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0010000100000001, 0x6369000000190000, 0x3833686934323031,
        0x33686f343233636f, 0x1031687033686b38, 0x0200000003000000, 0x0301070000002000, 0x0000030208000000, 0x02000a0000000200, 0x000008010d000000, 0x2002020000000300,
        0x0000100009000000, 0x0300000040010a00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000100001000000, 0x3263690000001b00, 0x636f383368693635,
        0x6b3931686f323135, 0x3168703268733368, 0x0000000300000010, 0x0107000000080002, 0x0003020800000003, 0x000a000000020000, 0x0004010d00000004, 0x0202000000030000,
        0x0010000900000020, 0x00000020010a0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x63690000001a0000, 0x3436776930363532,
        0x776f30383231636f, 0x30777031776b3436, 0x0000000100000010, 0x0001000000140002, 0x0000000400020000, 0x0020010200000003, 0x00000020000a0000, 0x000000000010020d,
        0x0001000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x6932636900000014, 0x686f32636f363568, 0x31687033686b3635, 0x0000000100000008,
        0x0000000000030007, 0x0102000000040000, 0x0003020800000002, 0x00000008000a0000, 0x000000000007030d, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000,
        0x0000000100010000, 0x693363690000001a, 0x3436636f30303368, 0x37686b303531686f, 0x0008336870326873, 0x0007000000020000, 0x0002010800000007, 0x000a000000010000,
        0x0000000400000002, 0x0208000000040102, 0x0020000a00000004, 0x0000000f030d0000, 0x0000000500000000, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001,
        0x63690000001a0000, 0x3631686930383231, 0x686f30383231636f, 0x31687033686b3631, 0x0000000300000010, 0x0107000000050002, 0x0003020800000003, 0x0002000000010000,
        0x0000000300000008, 0x000a000000200102, 0x0010020d00000020, 0x0005000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001a00000002,
        0x6869303832316369, 0x30383231636f3631, 0x7033686b3631686f, 0x0003000000103168, 0x0000005000020000, 0x0208000000030107, 0x0000000100000003, 0x000300000002000d,
        0x0000001001020000, 0x020d00000020000a, 0x0000000300000008, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x001c000000010001, 0x6930343663690000,
        0x36636f3432303177, 0x34323031776f3034, 0x001030777031776b, 0x0002000000010000, 0x0000000200000014, 0x010d00000004000a, 0x0000000300000008, 0x000a000000200102,
        0x0008020d00000040, 0x0004000000030000, 0x0001000000030000, 0x0007000000070000, 0x0101000000010000, 0x0000001900000020, 0x6869303832316369, 0x6f363435636f3031,
        0x687033686b303168, 0x0000030000000831, 0x0700000050000200, 0x0302080000000301, 0x0a00000002000000, 0x02010d0000000800, 0x0200000003000000, 0x2000090000001002,
        0x000010010a000000, 0x0000050000000300, 0x0000010000000300, 0x0000040000000400, 0x0100010000000100, 0x6900000015000000, 0x6f37686938323163, 0x686b37686f323363,
        0x0000001031687033, 0x0003000700000002, 0x0000000301080000, 0x0004000200000001, 0x0102000000030000, 0x0010000a00000020, 0x00000008020d0000, 0x0000000500000000,
        0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x6369000000150000, 0x636f376869383231, 0x33686b37686f3233, 0x0200000010316870, 0x0000030007000000,
        0x0100000003010800, 0x0000080002000000, 0x1001020000000300, 0x000010000a000000, 0x0000000008020d00, 0x0300000005000000, 0x0700000003000000, 0x0700000001000000,
        0x0000040001000000, 0x3863690000002800, 0x3177693034366869, 0x686f38636f343230, 0x323031776f303436, 0x7033776b33686b34, 0x0000103177703168, 0x20000c0000000100,
        0x0400000000000000, 0x0000080102000000, 0x0a00000003020800, 0x10030d0000001000, 0x0500000003000000, 0x0300000003000000, 0x010000000a000000, 0x010000000a000000,
        0x0000190000004001, 0x7769363532636900, 0x6f363532636f3039, 0x703233776b393577, 0x0002000000103077, 0x0000000200090000, 0x00010000003b010d, 0x0000000800020000,
        0x0020020200000003, 0x0000000800090000, 0x000200000040010a, 0x0003000000050000, 0x0007000000030000, 0x0007000000010000, 0x0000001000010000, 0x3532636900000018,
        0x32636f3833686936, 0x686b3833686f3635, 0x0000001031687033, 0x0026000c00000002, 0x00000002010d0000, 0x0008010200000002, 0x00000002000a0000, 0x0020020200000003,
        0x0000001000090000, 0x000300000040010a, 0x0003000000050000, 0x0001000000020000, 0x0007000000070000, 0x0000001000010000, 0x3233636900000032, 0x3333686937316469,
        0x3233636f33337769, 0x3333686f3731646f, 0x6b33646b3333776f, 0x31647033776b3368, 0x0010317770316870, 0x0006000000030000, 0x0003010700000003, 0x0000000302080000,
        0x0000000300000000, 0x0009000000200202, 0x0020010a00000010, 0x0005000000030000, 0x0003000000030000, 0x0001000000070000, 0x0001000000070000, 0x0000001600000010,
        0x3568693635326369, 0x35686f363834636f, 0x001031687033686b, 0x000c000000020000, 0x0005010d00000005, 0x000a000000010000, 0x0000000300000004, 0x0009000000200202,
        0x0020010a00000010, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000003800000001, 0x3232646934366369, 0x7769343232686934,
        0x6f3233636f303631, 0x3232686f34323264, 0x646b303631776f34, 0x7033776b33686b33, 0x3177703168703164, 0x0000000400000010, 0x0106000000040002, 0x0003020700000003,
        0x0000000303080000, 0x0000000300000000, 0x000a000000100102, 0x0020020d00000020, 0x0005000000030000, 0x0001000000030000, 0x0004000000040000, 0x0001000000010000,
        0x0000003800000001, 0x3232646934366369, 0x7769343232686934, 0x6f3233636f303631, 0x3232686f34323264, 0x646b303631776f34, 0x7033776b33686b33, 0x3177703168703164,
        0x0000000300000010, 0x0107000000030006, 0x0003020800000003, 0x0002000000010000, 0x0000000300000002, 0x000a000000200102, 0x0020020d00000020, 0x0005000000000000,
        0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x3368693036396369, 0x686f303436636f32, 0x31687033686b3233, 0x0000000300000010,
        0x01070000001e0002, 0x0003020800000003, 0x000a000000020000, 0x0004010d00000002, 0x0102000000030000, 0x0020000a00000020, 0x00000008020d0000, 0x0000000400000003,
        0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x6369000000150000, 0x636f356869323135, 0x33686b35686f3432, 0x0300000008316870, 0x0000040002000000,
        0x0800000003010700, 0x0000010000000302, 0x0300000008000200, 0x0000100102000000, 0x0d00000008000a00, 0x0000000000000802, 0x0000030000000400, 0x0000070000000100,
        0x0000010000000700, 0x1a00000002000100, 0x3832316369000000, 0x31636f3631686930, 0x6b3631686f303832, 0x0000083168703368, 0x5000020000000300, 0x0000030107000000,
        0x0100000003020800, 0x000002000a000000, 0x1001020000000300, 0x000020000a000000, 0x0200000010020d00, 0x0100000005000000, 0x0800000003000000, 0x0800000001000000,
        0x0000200001000000, 0x3863690000001400, 0x6f38636f35326469, 0x647035646b353264, 0x0000010000001032, 0x0000000019000d00, 0x0200000003000000, 0x1000090000000802,
        0x000010010a000000, 0x0000040000000000, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000018000000, 0x3231356869323363, 0x3135686f3233636f,
        0x0831687033686b32, 0x0200000003000000, 0x0301070000000200, 0x0000030208000000, 0x02000a0000000200, 0x000004010d000000, 0x1001020000000300, 0x000010000a000000,
        0x0300000020020d00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3163690000000a00, 0x083834636f323531, 0x0200000001000000,
        0x0000010000000900, 0x0300000008000200, 0x0000100102000000, 0x0d00000008000a00, 0x0000000000000802, 0x0000030000000500, 0x00000a0000000300, 0x00000a0000000100,
        0x1a00000001000100, 0x6869336369000000, 0x383231636f343232, 0x7334686b3635686f, 0x0000103068703468, 0x07000c0000000100, 0x0a00000001000000, 0x0000040000000800,
        0x0800000004010200, 0x10000a0000000402, 0x000008030d000000, 0x0000050000000200, 0x0000030000000300, 0x0000010000000700, 0x0100010000000700, 0x690000001a000000,
        0x6f34323268693363, 0x3635686f38323163, 0x687034687334686b, 0x0000000000001030, 0x02000a0000000100, 0x0200000004000000, 0x0402080000000801, 0x000010000a000000,
        0x0300000010030d00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3263690000001c00, 0x3635327769303635, 0x776f30383231636f,
        0x777031776b363532, 0x0000010000001030, 0x0200000050000200, 0x000002000a000000, 0x0300000008010d00, 0x0000200102000000, 0x0d00000020000a00, 0x0000030000001002,
        0x0000030000000400, 0x0000040000000100, 0x0000010000000400, 0x1800000001000100, 0x6936316369000000, 0x3233636f38303268, 0x33686b383032686f, 0x0200000008316870,
        0x0000030007000000, 0x0100000003010800, 0x000002000a000000, 0x2001020000000300, 0x000010000a000000, 0x0300000010020d00, 0x0300000004000000, 0x0400000001000000,
        0x0100000004000000, 0x0000200101000000, 0x3363690000001800, 0x636f363268693438, 0x6b3632686f363532, 0x0000083168703368, 0x0c00020000000300, 0x0000030107000000,
        0x0200000003020800, 0x000004000a000000, 0x0300000004010d00, 0x0000200202000000, 0x0a00000020000900, 0x0000030000002001, 0x0000030000000400, 0x0000040000000100,
        0x0000010000000400, 0x1b00000080010100, 0x3230316369000000, 0x636f363931776934, 0x363931776f363532, 0x000830777031776b, 0x0002000000010000, 0x0000000200000020,
        0x010d00000008000a, 0x0000000300000004, 0x0009000000200202, 0x0020010a00000020, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000,
        0x0000001900000001, 0x6869303239316369, 0x6f303436636f3233, 0x687033686b323368, 0x0000030000001031, 0x070000003c000200, 0x0302080000000301, 0x0a00000002000000,
        0x02010d0000000400, 0x0200000003000000, 0x20000a0000002001, 0x000010020d000000, 0x0000040000000300, 0x0000010000000300, 0x0000040000000400, 0x0100010000000100,
        0x6900000017000000, 0x6f38303668693363, 0x383036686f323363, 0x000831687033686b, 0x0007000000010000, 0x0000000100000003, 0x000400000002000d, 0x0000000401020000,
        0x000a000000080208, 0x0020030d00000010, 0x0004000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001a00000001, 0x3168693635326369,
        0x6f363532636f3035, 0x7033686b30353168, 0x0003000000083168, 0x0000001000020000, 0x0208000000030107, 0x0000000100000003, 0x000300000004000a, 0x0000001001020000,
        0x020d00000040000a, 0x0000000300000010, 0x0000000300000005, 0x0000000400000001, 0x0000000100000004, 0x0039000000200101, 0x3864693363690000, 0x3277693432326869,
        0x646f3631636f3432, 0x776f323131686f38, 0x686b31646b323131, 0x7332687333776b33, 0x3168703064703277, 0x0100000010317770, 0x0000030007000000, 0x0000040000000000,
        0x0800000004020200, 0x2000090000000803, 0x000010010a000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001c000000,
        0x3177693432303163, 0x38343032636f3639, 0x31776b363931776f, 0x0100000010307770, 0x0000400002000000, 0x04000a0000000200, 0x000004010d000000, 0x1001020000000300,
        0x000020000a000000, 0x0300000007020d00, 0x0100000004000000, 0x0700000001000000, 0x0100000007000000, 0x0000800001000000, 0x3431670000001c00, 0x3531686931636934,
        0x3537686f31636f31, 0x687032687333686b, 0x0000010000000830, 0x0000000003000700, 0x0100000003000000, 0x0302080000004001, 0x0000080009000000, 0x0000050000000000,
        0x0000010000000300, 0x0000040000000400, 0x2001010000000100, 0x690000002c000000, 0x7769303868693163, 0x636f303536303031, 0x31776f3038686f31, 0x31686b3035363030,
        0x703068703332776b, 0x0100000010313177, 0x0000030008000000, 0x04000d0000000100, 0x0200000004000000, 0x0803080000000402, 0x0000200009000000, 0x0300000010010a00,
        0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000200101000000, 0x3163690000002c00, 0x3031776930386869, 0x6f31636f30353630, 0x303031776f303868,
        0x776b31686b303536, 0x3177703068703332, 0x0000010000001031, 0x0100000003000800, 0x000008000d000000, 0x0202020000000400, 0x0000080308000000, 0x0a00000020000900,
        0x0000030000001001, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1800000001000100, 0x3036396369000000, 0x3233636f34366869, 0x33686b3436686f30,
        0x0300000010316870, 0x00001e0002000000, 0x0800000003010700, 0x0000020000000302, 0x0d00000002000a00, 0x0000030000000401, 0x0a00000020010200, 0x10020d0000004000,
        0x0500000003000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x0000180000000100, 0x6869383231636900, 0x6f383231636f3631, 0x687033686b363168,
        0x0000020000001031, 0x0800000003000700, 0x0000010000000301, 0x0300000004000200, 0x0000200102000000, 0x0d00000010000a00, 0x0000000000000802, 0x0000030000000400,
        0x0000040000000100, 0x0000010000000400, 0x1700000001000100, 0x6869336369000000, 0x6f3233636f323332, 0x7039686b34323268, 0x0002000000083068, 0x0000000900070000,
        0x0002000000020108, 0x00000002000a0000, 0x000400000004010d, 0x0000000401020000, 0x000a000000080208, 0x001c030d00000010, 0x0004000000010000, 0x0001000000010000,
        0x0007000000070000, 0x0001000000010000, 0x0000001900000080, 0x6869316369303467, 0x686f31636f303531, 0x687033686b303531, 0x0000010000000831, 0x0100000003000700,
        0x000008000d000000, 0x1001010000000300, 0x0000030208000000, 0x0100000008000900, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000, 0x000c800101000000,
        0x3463690000001900, 0x636f343677693834, 0x3436776f30383032, 0x000830777031776b, 0x0002000000010000, 0x000000020000001c, 0x010d00000002000a, 0x0000000300000008,
        0x0009000000100202, 0x0020010a00000020, 0x0004000000030000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000001900000c80, 0x3677693834346369,
        0x6f30383032636f34, 0x777031776b343677, 0x0000010000000830, 0x020000000e000200, 0x000004000a000000, 0x0300000008010d00, 0x0000200202000000, 0x0a00000020000900,
        0x0000030000002001, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1c00000001000100, 0x3032336369000000, 0x636f363930347769, 0x393034776f303233,
        0x1030777031776b36, 0x0200000001000000, 0x0000020000000a00, 0x0d00000002000a00, 0x0000030000000801, 0x0a00000020010200, 0x20020d0000004000, 0x0500000003000000,
        0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001b0000000100, 0x6869303233636900, 0x6f303233636f3436, 0x687333686b323368, 0x0000001031687032,
        0x000a000200000003, 0x0000000301070000, 0x0002000000030208, 0x00000004000a0000, 0x000300000002010d, 0x0000002001020000, 0x020d00000010000a, 0x0000000300000010,
        0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x000a000000010001, 0x3235313163690000, 0x000000103834636f, 0x0009000200000001, 0x0002000000010000,
        0x0000000300000008, 0x000a000000100102, 0x0008020d00000010, 0x0004000000000000, 0x0001000000030000, 0x0004000000040000, 0x0001000000010000, 0x0000001600000001,
        0x3531686930316369, 0x3331686f3032636f, 0x000830687033686b, 0x0007000000020000, 0x0002010800000003, 0x000d000000010000, 0x0000000400000002, 0x0208000000100102,
        0x0008000a00000002, 0x00000008030d0000, 0x0000000500000003, 0x0000000300000003, 0x000000010000000a, 0x002001010000000a, 0x6369000000300000, 0x3268693632646936,
        0x36636f3732776938, 0x3632686f3632646f, 0x6b33646b3632776f, 0x31647038776b3368, 0x0010337770306870, 0x000c000000020000, 0x001a010d0000000d, 0x0003000000000000,
        0x0000000802080000, 0x010a000000100009, 0x0000000200000010, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x0015000000010001, 0x6869343663690000,
        0x33686f31636f3233, 0x1031687033686b32, 0x0200000003000000, 0x0301070000000200, 0x0000030208000000, 0x04000d0000000100, 0x0200000003000000, 0x10000a0000002001,
        0x000008020d000000, 0x0000040000000300, 0x0000010000000300, 0x0000070000000700, 0x0200010000000100, 0x690000001a000000, 0x3168693036353263, 0x6f30383231636f36,
        0x687033686b363168, 0x0000030000000831, 0x07000000a0000200, 0x0302080000000301, 0x0a00000001000000, 0x0000030000000400, 0x0a00000010010200, 0x10020d0000002000,
        0x0500000002000000, 0x0200000003000000, 0x0700000001000000, 0x0100000007000000, 0x00001c0000000800, 0x7769323135636900, 0x323135636f393932, 0x7338776b3337776f,
        0x0000103077703477, 0x0801080000000200, 0x000020000a000000, 0x0800020000000200, 0x0000040105000000, 0x4001020000000300, 0x0000100205000000, 0x0300000010000a00,
        0x0300000005000000, 0x0100000002000000, 0x0700000007000000, 0x0000100001000000, 0x3263690000001d00, 0x3434343177693635, 0x776f34323031636f, 0x7031776b34343431,
        0x0001000000103077, 0x00000020000a0000, 0x0002000200000002, 0x0000000401050000, 0x0040020200000003, 0x0000001000090000, 0x000300000020010a, 0x0003000000050000,
        0x0001000000020000, 0x0007000000070000, 0x0000001000010000, 0x3532636900000018, 0x32636f3833686936, 0x686b3833686f3635, 0x0000001031687033, 0x0003010700000003,
        0x0000000302080000, 0x000200000008000a, 0x0000000200020000, 0x0003000000080105, 0x0000004002020000, 0x010a000000100009, 0x0000000300000020, 0x0000000300000005,
        0x0000000700000001, 0x0000000100000007, 0x0017000000200001, 0x3463693233670000, 0x6f34636f36356869, 0x687033686b363568, 0x0000020000001031, 0x0800000003000700,
        0x0000010000000301, 0x0300000008000d00, 0x0000100202000000, 0x0a00000020000900, 0x0000000000001001, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700,
        0x1900000001000100, 0x3635326369000000, 0x33636f3938327769, 0x776b393832776f32, 0x0000001030777031, 0x0002000200000001, 0x0002000000010000, 0x0000000300000008,
        0x000a000000100102, 0x0008020d00000010, 0x0005000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000000a00000001, 0x636f343230316369,
        0x0001000000103436, 0x0000000800020000, 0x0008000200000001, 0x0102000000030000, 0x0010000a00000010, 0x00000008020d0000, 0x0000000400000000, 0x0000000100000003,
        0x0000000400000004, 0x0080010100000001, 0x6369000000160000, 0x636f363568693436, 0x686b3635686f3436, 0x0000000831687033, 0x0002000200000003, 0x0000000301070000,
        0x0002000000030208, 0x00000002000a0000, 0x000300000008010d, 0x0000002002020000, 0x010a000000200009, 0x0000000300000020, 0x0000000100000005, 0x0000000700000001,
        0x0000000100000007, 0x0015000000010001, 0x3163693436670000, 0x686f31636f386869, 0x1031687033686b38, 0x0700000001000000, 0x0000000000000300, 0x1000010000000300,
        0x0000030108000000, 0x0000000008020d00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3263690000001400, 0x36636f3568693635,
        0x687033686b35686f, 0x0000030000000831, 0x0700000002000200, 0x0302080000000301, 0x0200000001000000, 0x0000030000000800, 0x0a00000010010200, 0x08020d0000000800,
        0x0500000000000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x0000150000000100, 0x6869383231636900, 0x36686f3233636f36, 0x001031687033686b,
        0x0007000000020000, 0x0003010800000003, 0x0002000000010000, 0x0000000300000004, 0x000a000000200102, 0x0008020d00000010, 0x0005000000000000, 0x0001000000010000,
        0x0008000000080000, 0x0001000000010000, 0x0000001400000800, 0x6f30316869386369, 0x686b38686f303163, 0x0000001030687033, 0x0003000700000001, 0x0004000000000000,
        0x0000000801020000, 0x000a000000030208, 0x0008030d00000010, 0x0005000000000000, 0x0001000000010000, 0x000b0000000b0000, 0x0001000000010000, 0x0000001400000800,
        0x6f30316869386369, 0x686b38686f303163, 0x0000001030687033, 0x0003000700000001, 0x0004000000000000, 0x0000000801020000, 0x000a000000030208, 0x0008030d00000010,
        0x0004000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x3368693832316369, 0x686f383231636f32, 0x31687033686b3233,
        0x0000000200000008, 0x0108000000030007, 0x0000000100000003, 0x0003000000080002, 0x0000001001020000, 0x020d00000020000a, 0x0000000000000010, 0x0000000300000005,
        0x0000000700000001, 0x0000000100000007, 0x001a000000200101, 0x3168693363690000, 0x686f33636f343230, 0x687337686b323135, 0x0000001033687032, 0x0007000700000002,
        0x0000000201080000, 0x0004000d00000001, 0x0202000000040000, 0x0004030800000004, 0x0000002000090000, 0x000300000010010a, 0x0003000000050000, 0x0007000000010000,
        0x0001000000070000, 0x0000002000010000, 0x3436636900000017, 0x33636f3635326869, 0x39686b363532686f, 0x0300000010346870, 0x0000020002000000, 0x0800000009010700,
        0x0000010000000902, 0x0300000008000d00, 0x0000200202000000, 0x0a00000020000900, 0x0000030000001001, 0x0000030000000400, 0x0000040000000100, 0x0000010000000400,
        0x1800000001000100, 0x3436346369000000, 0x3631636f34366869, 0x33686b3436686f30, 0x0300000008316870, 0x00000f0002000000, 0x0800000003010700, 0x0000020000000302,
        0x0d00000004000a00, 0x0000030000000201, 0x0a00000020010200, 0x20020d0000001000, 0x0500000003000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000,
        0x0000180000000100, 0x6869383231636900, 0x6f383231636f3835, 0x687033686b363568, 0x0000030000001030, 0x0700000004000200, 0x0302080000000301, 0x0a00000002000000,
        0x08010d0000000200, 0x0200000003000000, 0x20000a0000002001, 0x000007020d000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100,
        0x6900000018000000, 0x3835686938323163, 0x35686f383231636f, 0x1030687033686b36, 0x0200000003000000, 0x0301070000000800, 0x0000030208000000, 0x04000a0000000200,
        0x000004010d000000, 0x1001020000000300, 0x000010000a000000, 0x030000000e020d00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000,
        0x3363690000001800, 0x636f323368693032, 0x6b3233686f303436, 0x0000103168703368, 0x0a00020000000300, 0x0000030107000000, 0x0200000003020800, 0x000002000a000000,
        0x0300000004010d00, 0x0000200102000000, 0x0d00000020000a00, 0x0000030000000802, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1c00000008000100,
        0x6869336369000000, 0x383637636f343232, 0x3631686b3431686f, 0x0830687036316873, 0x0700000002000000, 0x0401080000001000, 0x0a00000001000000, 0x0000040000000200,
        0x0800000004010200, 0x40000a0000000402, 0x00000e030d000000, 0x0000040000000000, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000019000000,
        0x3168693038323163, 0x6f30383231636f30, 0x30687033686b3868, 0x0000000300000008, 0x0107000000140002, 0x0003020800000003, 0x0002000000010000, 0x0000000300000004,
        0x000a000000100102, 0x0008020d00000020, 0x0004000000000000, 0x0001000000010000, 0x0007000000070000, 0x0001000000010000, 0x0000002c00000001, 0x6930386869316369,
        0x6f30353630303177, 0x776f3038686f3163, 0x686b303536303031, 0x3068703332776b31, 0x0000000831317770, 0x0003000800000001, 0x000d000000010000, 0x0000000300000008,
        0x000a000000080108, 0x0010020d00000008, 0x0004000000010000, 0x0001000000010000, 0x0004000000040000, 0x0001000000010000, 0x0000002c00000001, 0x6930386869316369,
        0x6f30353630303177, 0x776f3038686f3163, 0x686b303536303031, 0x3068703332776b31, 0x0000000831317770, 0x0000000100000000, 0x000300000002000d, 0x0000001701080000,
        0x020d00000008000a, 0x000000000000000f, 0x0000000100000005, 0x0000000800000003, 0x0000000800000001, 0x0014000000200001, 0x3264693863690000, 0x3332646f38636f35,
        0x001031647035646b, 0x000d000000010000, 0x0000000000000017, 0x0008020200000003, 0x0000001000090000, 0x000000000010010a, 0x0001000000050000, 0x0001000000020000,
        0x0008000000080000, 0x0000080000010000, 0x6938636900000014, 0x6f3031636f303168, 0x30687033686b3868, 0x0000000200000010, 0x0108000000030007, 0x0000000000000003,
        0x0010020200000003, 0x0000001000090000, 0x000000000010010a, 0x0001000000050000, 0x0001000000020000, 0x000b0000000b0000, 0x0000080000010000, 0x6938636900000014,
        0x6f3031636f303168, 0x30687033686b3868, 0x0000000200000008, 0x0108000000030007, 0x0000000100000003, 0x000300000002000a, 0x0000000802020000, 0x010a000000100009,
        0x0000000000000008, 0x0000000300000005, 0x0000000700000003, 0x0000000700000001, 0x0016000000100001, 0x6869343663690000, 0x686f3436636f3537, 0x31687033686b3537,
        0x0000000200000010, 0x010d00000005000c, 0x0000000100000005, 0x0003000000040002, 0x0000001002020000, 0x010a000000100009, 0x0000000300000040, 0x0000000300000005,
        0x0000000700000001, 0x0000000100000007, 0x001a000000010001, 0x3032393163690000, 0x3231636f36316869, 0x686b3631686f3038, 0x0000001031687033, 0x0008000200000003,
        0x0000000301070000, 0x0001000000030208, 0x0000000800020000, 0x0020010200000003, 0x00000020000a0000, 0x000000000010020d, 0x0003000000050000, 0x0007000000010000,
        0x0001000000070000, 0x0000000100010000, 0x3233636900000016, 0x34636f3436686930, 0x7033686b3436686f, 0x0003000000103168, 0x0000000a00020000, 0x0208000000030107,
        0x0000000100000003, 0x000300000002000d, 0x0000002001020000, 0x020d00000010000a, 0x0000000300000008, 0x0000000300000005, 0x0000000100000002, 0x0000000700000007,
        0x001b000000100001, 0x6936353263690000, 0x323135636f353768, 0x7331686b3833686f, 0x0000103068703268, 0x10000a0000000100, 0x0300000000000000, 0x0000400202000000,
        0x0a00000010000900, 0x0000030000002001, 0x0000030000000500, 0x0000010000000200, 0x0000070000000700, 0x3200000010010100, 0x6932336369000000, 0x6933336869373164,
        0x6f3233636f333377, 0x6f3333686f373164, 0x686b33646b333377, 0x7031647033776b33, 0x0000103177703168, 0x0301060000000400, 0x0000030207000000, 0x0a00000003030800,
        0x0000010000000200, 0x0300000002000500, 0x0000200202000000, 0x0a00000010000900, 0x0000030000001001, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700,
        0x1c00000001000100, 0x3036396369000000, 0x636f363930347769, 0x393034776f303233, 0x1030777031776b36, 0x0200000001000000, 0x0000020000001e00, 0x0d00000004000a00,
        0x0000030000000401, 0x0a00000020010200, 0x20020d0000002000, 0x0500000003000000, 0x0300000003000000, 0x010000000a000000, 0x010000000a000000, 0x00001b0000004001,
        0x7769383231636900, 0x363532636f323532, 0x31776b373332776f, 0x0000001030777036, 0x0002000900000002, 0x0000004f010d0000, 0x0004010200000002, 0x00000002000a0000,
        0x0020020200000003, 0x0000000800090000, 0x000200000040010a, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x3436636900000018,
        0x33636f3436686930, 0x686b3436686f3032, 0x0000001031687033, 0x0014000200000003, 0x0000000301070000, 0x0002000000030208, 0x00000002000a0000, 0x000300000004010d,
        0x0000002001020000, 0x020d00000040000a, 0x0000000300000010, 0x0000000100000005, 0x0000000700000001, 0x0000000100000007, 0x0017000000010001, 0x3163693436670000,
        0x6f31636f36316869, 0x687033686b363168, 0x0000010000001031, 0x0100000003000700, 0x000002000d000000, 0x1000010000000300, 0x0000030108000000, 0x0000000008020d00,
        0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3663690000001800, 0x636f323368693034, 0x6b3233686f303436, 0x0000103168703368,
        0x1400020000000300, 0x0000030107000000, 0x0100000003020800, 0x000008000a000000, 0x2001020000000300, 0x000010000a000000, 0x0300000020020d00, 0x0300000005000000,
        0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3963690000001c00, 0x3432303177693036, 0x31776f303436636f, 0x777031776b343230, 0x0000010000001030,
        0x020000001e000200, 0x000004000a000000, 0x0300000004010d00, 0x0000200102000000, 0x0d00000020000a00, 0x0000030000002002, 0x0000010000000400, 0x0000070000000100,
        0x0000010000000700, 0x1c00000080000100, 0x6934343167000000, 0x6f31393168693163, 0x686b3539686f3163, 0x0830687032687333, 0x0700000001000000, 0x0000000000000300,
        0x4001010000000300, 0x0000030208000000, 0x0000000008000900, 0x0300000005000000, 0x0700000003000000, 0x0700000001000000, 0x0000100001000000, 0x3563690000001800,
        0x636f303168693231, 0x6b3031686f363834, 0x0000103168703368, 0x0a000c0000000200, 0x00000a010d000000, 0x0000030000000000, 0x0900000020020200, 0x40010a0000001000,
        0x0000000003000000
    };
    return data;
};
// clang-format on

} // namespace jit
} // namespace gpu
} // namespace impl
} // namespace dnnl
