use mdbook::errors::Result as MdbookResult;
use pulldown_cmark::{CodeBlockKind::*, Event, Options, Parser, Tag};

use crate::{
    book_config::OnFailure,
    parse::parse_admonition,
    types::{Overrides, RenderTextMode},
};

pub(crate) fn preprocess(
    content: &str,
    on_failure: OnFailure,
    overrides: &Overrides,
    render_text_mode: RenderTextMode,
) -> MdbookResult<String> {
    let mut id_counter = Default::default();
    let mut opts = Options::empty();
    opts.insert(Options::ENABLE_TABLES);
    opts.insert(Options::ENABLE_FOOTNOTES);
    opts.insert(Options::ENABLE_STRIKETHROUGH);
    opts.insert(Options::ENABLE_TASKLISTS);

    let mut admonish_blocks = vec![];

    let events = Parser::new_ext(content, opts);

    for (event, span) in events.into_offset_iter() {
        if let Event::Start(Tag::CodeBlock(Fenced(info_string))) = event.clone() {
            let span_content = &content[span.start..span.end];
            const INDENT_SCAN_MAX: usize = 1024;
            let indent = indent_of(content, span.start, INDENT_SCAN_MAX);

            let admonition = match parse_admonition(
                info_string.as_ref(),
                overrides,
                span_content,
                on_failure,
                indent,
            ) {
                Some(admonition) => admonition,
                None => continue,
            };

            let admonition = admonition?;

            // Once we've identitified admonition blocks, handle them differently
            // depending on our render mode
            let new_content = match render_text_mode {
                RenderTextMode::Html => admonition.html(&mut id_counter),
                RenderTextMode::Strip => admonition.strip(),
            };

            admonish_blocks.push((span, new_content));
        }
    }

    let mut content = content.to_string();
    for (span, block) in admonish_blocks.iter().rev() {
        let pre_content = &content[..span.start];
        let post_content = &content[span.end..];
        content = format!("{}{}{}", pre_content, block, post_content);
    }

    Ok(content)
}

/// Returns the indent of the given position.
///
/// Defined as the number of characters between the given `position` (where
/// position is a valid char boundary byte-index in `content`),
/// and the previous newline character `\n`.
///
/// `max` is the maximum number of characters to scan before assuming there is
/// no indent (will return zero if exceeded).
///
/// ## Panics
///
/// Will panic if `position` is not a valid utf-8 char boundary index of `content`.
fn indent_of(content: &str, position: usize, max: usize) -> usize {
    // Scan for a line start before this span.
    content[..position]
        .chars()
        .rev()
        // For safety, only scan up to a fixed limit of the text
        .take(max)
        .position(|c| c == '\n')
        // If we can't find a newline, assume no indent
        .unwrap_or_default()
}

#[cfg(test)]
mod test {
    use pretty_assertions::assert_eq;

    use crate::types::AdmonitionDefaults;

    use super::*;

    #[test]
    fn indent_of_samples() {
        for (content, position, max, expected) in [
            // Empty case
            ("", 0, 10, 0),
            ("no newline", 4, 10, 0),
            // Newline at position 5, difference from 8 is 3
            ("with\nnewline", 8, 10, 3),
            // If no newline in safety limit, return 0
            ("with\nnewline", 8, 2, 0),
            // Safety limit is characters, not bytes
            // Regression test for FIXME LINK
            (
                "例えばこれは",
                // Position is generated by mdbook internals, should be valid char limit
                // This mimics the second character starting the span
                "例".len(),
                // Any arbitrary safetly limit should be valid
                1,
                // Should not panic
                0,
            ),
            (
                "例え\n  れは",
                // Position is generated by mdbook internals, should be valid char limit
                // This mimics the second character starting the span
                "例え\n  ".len(),
                // Any arbitrary safetly limit should be valid
                4,
                // Should not panic
                2,
            ),
        ] {
            let actual = indent_of(content, position, max);
            assert_eq!(actual, expected);
        }
    }

    fn prep(content: &str) -> String {
        preprocess(
            content,
            OnFailure::Continue,
            &Overrides::default(),
            RenderTextMode::Html,
        )
        .unwrap()
    }

    #[test]
    fn adds_admonish() {
        let content = r#"# Chapter
```admonish
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-note" class="admonition admonish-note" role="note" aria-labelledby="admonition-note-title">
<div class="admonition-title">
<div id="admonition-note-title">

Note

</div>
<a class="admonition-anchor-link" href="#admonition-note"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn adds_admonish_longer_code_fence() {
        let content = r#"# Chapter
````admonish
```json
{}
```
````
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-note" class="admonition admonish-note" role="note" aria-labelledby="admonition-note-title">
<div class="admonition-title">
<div id="admonition-note-title">

Note

</div>
<a class="admonition-anchor-link" href="#admonition-note"></a>
</div>
<div>

```json
{}
```

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn adds_admonish_directive() {
        let content = r#"# Chapter
```admonish warning
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-warning" class="admonition admonish-warning" role="note" aria-labelledby="admonition-warning-title">
<div class="admonition-title">
<div id="admonition-warning-title">

Warning

</div>
<a class="admonition-anchor-link" href="#admonition-warning"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn adds_admonish_directive_alternate() {
        let content = r#"# Chapter
```admonish caution
A warning with alternate title.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-caution" class="admonition admonish-warning" role="note" aria-labelledby="admonition-caution-title">
<div class="admonition-title">
<div id="admonition-caution-title">

Caution

</div>
<a class="admonition-anchor-link" href="#admonition-caution"></a>
</div>
<div>

A warning with alternate title.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn adds_admonish_directive_title() {
        let content = r#"# Chapter
```admonish warning "Read **this**!"
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-read-this" class="admonition admonish-warning" role="note" aria-labelledby="admonition-read-this-title">
<div class="admonition-title">
<div id="admonition-read-this-title">

Read **this**!

</div>
<a class="admonition-anchor-link" href="#admonition-read-this"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn leaves_tables_untouched() {
        // Regression test.
        // Previously we forgot to enable the same markdwon extensions as mdbook itself.

        let content = r#"# Heading
| Head 1 | Head 2 |
|--------|--------|
| Row 1  | Row 2  |
"#;

        let expected = r#"# Heading
| Head 1 | Head 2 |
|--------|--------|
| Row 1  | Row 2  |
"#;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn leaves_html_untouched() {
        // Regression test.
        // Don't remove important newlines for syntax nested inside HTML

        let content = r#"# Heading
<del>
*foo*
</del>
"#;

        let expected = r#"# Heading
<del>
*foo*
</del>
"#;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn html_in_list() {
        // Regression test.
        // Don't remove important newlines for syntax nested inside HTML

        let content = r#"# Heading
1. paragraph 1
   ```
   code 1
   ```
2. paragraph 2
"#;

        let expected = r#"# Heading
1. paragraph 1
   ```
   code 1
   ```
2. paragraph 2
"#;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn info_string_that_changes_length_when_parsed() {
        let content = r#"
```admonish note "And \\"<i>in</i>\\" the title"
With <b>html</b> styling.
```
hello
"#;

        let expected = r##"

<div id="admonition-and-in-the-title" class="admonition admonish-note" role="note" aria-labelledby="admonition-and-in-the-title-title">
<div class="admonition-title">
<div id="admonition-and-in-the-title-title">

And "<i>in</i>" the title

</div>
<a class="admonition-anchor-link" href="#admonition-and-in-the-title"></a>
</div>
<div>

With <b>html</b> styling.

</div>
</div>
hello
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn info_string_ending_in_symbol() {
        let content = r#"
```admonish warning "Trademark™"
Should be respected
```
hello
"#;

        let expected = r##"

<div id="admonition-trademark" class="admonition admonish-warning" role="note" aria-labelledby="admonition-trademark-title">
<div class="admonition-title">
<div id="admonition-trademark-title">

Trademark™

</div>
<a class="admonition-anchor-link" href="#admonition-trademark"></a>
</div>
<div>

Should be respected

</div>
</div>
hello
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn block_with_additional_classname() {
        let content = r#"
```admonish tip.my-style.other-style
Will have bonus classnames
```
"#;

        let expected = r##"

<div id="admonition-tip" class="admonition admonish-tip my-style other-style" role="note" aria-labelledby="admonition-tip-title">
<div class="admonition-title">
<div id="admonition-tip-title">

Tip

</div>
<a class="admonition-anchor-link" href="#admonition-tip"></a>
</div>
<div>

Will have bonus classnames

</div>
</div>
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn block_with_additional_classname_and_title() {
        let content = r#"
```admonish tip.my-style.other-style "Developers don't want you to know this one weird tip!"
Will have bonus classnames
```
"#;

        let expected = r##"

<div id="admonition-developers-dont-want-you-to-know-this-one-weird-tip" class="admonition admonish-tip my-style other-style" role="note" aria-labelledby="admonition-developers-dont-want-you-to-know-this-one-weird-tip-title">
<div class="admonition-title">
<div id="admonition-developers-dont-want-you-to-know-this-one-weird-tip-title">

Developers don't want you to know this one weird tip!

</div>
<a class="admonition-anchor-link" href="#admonition-developers-dont-want-you-to-know-this-one-weird-tip"></a>
</div>
<div>

Will have bonus classnames

</div>
</div>
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn block_with_empty_additional_classnames_title_content() {
        let content = r#"
```admonish .... ""
```
"#;

        let expected = r#"

<div id="admonition-default" class="admonition admonish-note" role="note">
<div>



</div>
</div>
"#;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn unique_ids_same_title() {
        let content = r#"
```admonish note "My Note"
Content zero.
```

```admonish note "My Note"
Content one.
```
"#;

        let expected = r##"

<div id="admonition-my-note" class="admonition admonish-note" role="note" aria-labelledby="admonition-my-note-title">
<div class="admonition-title">
<div id="admonition-my-note-title">

My Note

</div>
<a class="admonition-anchor-link" href="#admonition-my-note"></a>
</div>
<div>

Content zero.

</div>
</div>


<div id="admonition-my-note-1" class="admonition admonish-note" role="note" aria-labelledby="admonition-my-note-1-title">
<div class="admonition-title">
<div id="admonition-my-note-1-title">

My Note

</div>
<a class="admonition-anchor-link" href="#admonition-my-note-1"></a>
</div>
<div>

Content one.

</div>
</div>
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn v2_config_works() {
        let content = r#"
```admonish tip class="my other-style" title="Article Heading"
Bonus content!
```
"#;

        let expected = r##"

<div id="admonition-article-heading" class="admonition admonish-tip my other-style" role="note" aria-labelledby="admonition-article-heading-title">
<div class="admonition-title">
<div id="admonition-article-heading-title">

Article Heading

</div>
<a class="admonition-anchor-link" href="#admonition-article-heading"></a>
</div>
<div>

Bonus content!

</div>
</div>
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn continue_on_error_output() {
        let content = r#"
```admonish title="
Bonus content!
```
"#;

        let expected = r##"

<div id="admonition-error-rendering-admonishment" class="admonition admonish-bug" role="note" aria-labelledby="admonition-error-rendering-admonishment-title">
<div class="admonition-title">
<div id="admonition-error-rendering-admonishment-title">

Error rendering admonishment

</div>
<a class="admonition-anchor-link" href="#admonition-error-rendering-admonishment"></a>
</div>
<div>

Failed with:

```log
'title="' is not a valid directive or TOML key-value pair.

TOML parsing error: TOML parse error at line 1, column 21
  |
1 | config = { title=" }
  |                     ^
invalid basic string

```

Original markdown input:

````markdown
```admonish title="
Bonus content!
```
````


</div>
</div>
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn bail_on_error_output() {
        let content = r#"
```admonish title="
Bonus content!
```
"#;
        assert_eq!(
            preprocess(
                content,
                OnFailure::Bail,
                &Overrides::default(),
                RenderTextMode::Html
            )
            .unwrap_err()
            .to_string(),
            r#"Error processing admonition, bailing:
```admonish title="
Bonus content!
```"#
                .to_owned()
        )
    }

    #[test]
    fn test_renderer_strip_explicit() {
        let content = r#"
````admonish title="Title"
```rust
let x = 10;
x = 20;
```
````
"#;
        assert_eq!(
            preprocess(
                content,
                OnFailure::Bail,
                &Overrides::default(),
                RenderTextMode::Strip
            )
            .unwrap(),
            r#"

```rust
let x = 10;
x = 20;
```

"#
            .to_owned()
        )
    }

    #[test]
    fn block_collapsible() {
        let content = r#"
```admonish collapsible=true
Hidden
```
"#;

        let expected = r##"

<details id="admonition-note" class="admonition admonish-note" role="note" aria-labelledby="admonition-note-title">
<summary class="admonition-title">
<div id="admonition-note-title">

Note

</div>
<a class="admonition-anchor-link" href="#admonition-note"></a>
</summary>
<div>

Hidden

</div>
</details>
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn default_toml_title() {
        let content = r#"# Chapter
```admonish
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-admonish" class="admonition admonish-note" role="note" aria-labelledby="admonition-admonish-title">
<div class="admonition-title">
<div id="admonition-admonish-title">

Admonish

</div>
<a class="admonition-anchor-link" href="#admonition-admonish"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        let preprocess_result = preprocess(
            content,
            OnFailure::Continue,
            &Overrides {
                book: AdmonitionDefaults {
                    title: Some("Admonish".to_owned()),
                    css_id_prefix: None,
                    collapsible: false,
                },
                ..Default::default()
            },
            RenderTextMode::Html,
        )
        .unwrap();
        assert_eq!(expected, preprocess_result);
    }

    #[test]
    fn empty_explicit_title_with_default() {
        let content = r#"# Chapter
```admonish title=""
A simple admonition.
```
Text
"#;

        let expected = r#"# Chapter

<div id="admonition-default" class="admonition admonish-note" role="note">
<div>

A simple admonition.

</div>
</div>
Text
"#;

        let preprocess_result = preprocess(
            content,
            OnFailure::Continue,
            &Overrides {
                book: AdmonitionDefaults {
                    title: Some("Admonish".to_owned()),
                    css_id_prefix: None,
                    collapsible: false,
                },
                ..Default::default()
            },
            RenderTextMode::Html,
        )
        .unwrap();
        assert_eq!(expected, preprocess_result);
    }

    #[test]
    fn empty_explicit_title() {
        let content = r#"# Chapter
```admonish title=""
A simple admonition.
```
Text
"#;

        let expected = r#"# Chapter

<div id="admonition-default" class="admonition admonish-note" role="note">
<div>

A simple admonition.

</div>
</div>
Text
"#;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn standard_custom_id() {
        let content = r#"# Chapter
```admonish check id="yay-custom-id"
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="yay-custom-id" class="admonition admonish-success" role="note" aria-labelledby="yay-custom-id-title">
<div class="admonition-title">
<div id="yay-custom-id-title">

Check

</div>
<a class="admonition-anchor-link" href="#yay-custom-id"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn no_custom_id_default_prefix() {
        let content = r#"# Chapter
```admonish check
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-check" class="admonition admonish-success" role="note" aria-labelledby="admonition-check-title">
<div class="admonition-title">
<div id="admonition-check-title">

Check

</div>
<a class="admonition-anchor-link" href="#admonition-check"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn no_custom_id_default_prefix_custom_title() {
        let content = r#"# Chapter
```admonish check title="Check Mark"
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-check-mark" class="admonition admonish-success" role="note" aria-labelledby="admonition-check-mark-title">
<div class="admonition-title">
<div id="admonition-check-mark-title">

Check Mark

</div>
<a class="admonition-anchor-link" href="#admonition-check-mark"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn title_and_content_with_html() {
        // Note that we use toml literal (single quoted) strings here
        // and the fact we have an equals sign in the value does not cause
        // us to break (because we're using v3 syntax, not v2)
        let content = r#"# Chapter
```admonish success title='Check <span class="emphasis">Mark</span>'
A <span class="emphasis">simple</span> admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="admonition-check-mark" class="admonition admonish-success" role="note" aria-labelledby="admonition-check-mark-title">
<div class="admonition-title">
<div id="admonition-check-mark-title">

Check <span class="emphasis">Mark</span>

</div>
<a class="admonition-anchor-link" href="#admonition-check-mark"></a>
</div>
<div>

A <span class="emphasis">simple</span> admonition.

</div>
</div>
Text
"##;

        assert_eq!(expected, prep(content));
    }

    #[test]
    fn empty_default_id_prefix() {
        let content = r#"# Chapter
```admonish info
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="info" class="admonition admonish-info" role="note" aria-labelledby="info-title">
<div class="admonition-title">
<div id="info-title">

Info

</div>
<a class="admonition-anchor-link" href="#info"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        let preprocess_result = preprocess(
            content,
            OnFailure::Continue,
            &Overrides {
                book: AdmonitionDefaults {
                    title: Some("Info".to_owned()),
                    css_id_prefix: Some("".to_owned()),
                    collapsible: false,
                },
                ..Default::default()
            },
            RenderTextMode::Html,
        )
        .unwrap();
        assert_eq!(expected, preprocess_result);
    }

    #[test]
    fn custom_id_prefix_custom_title() {
        let content = r#"# Chapter
```admonish info title="My Title"
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="prefix-my-title" class="admonition admonish-info" role="note" aria-labelledby="prefix-my-title-title">
<div class="admonition-title">
<div id="prefix-my-title-title">

My Title

</div>
<a class="admonition-anchor-link" href="#prefix-my-title"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        let preprocess_result = preprocess(
            content,
            OnFailure::Continue,
            &Overrides {
                book: AdmonitionDefaults {
                    title: Some("Info".to_owned()),
                    css_id_prefix: Some("prefix-".to_owned()),
                    collapsible: false,
                },
                ..Default::default()
            },
            RenderTextMode::Html,
        )
        .unwrap();
        assert_eq!(expected, preprocess_result);
    }

    #[test]
    fn custom_id_custom_title() {
        let content = r#"# Chapter
```admonish info title="My Title" id="my-section-id"
A simple admonition.
```
Text
"#;

        let expected = r##"# Chapter

<div id="my-section-id" class="admonition admonish-info" role="note" aria-labelledby="my-section-id-title">
<div class="admonition-title">
<div id="my-section-id-title">

My Title

</div>
<a class="admonition-anchor-link" href="#my-section-id"></a>
</div>
<div>

A simple admonition.

</div>
</div>
Text
"##;

        let preprocess_result = preprocess(
            content,
            OnFailure::Continue,
            &Overrides {
                book: AdmonitionDefaults {
                    title: Some("Info".to_owned()),
                    css_id_prefix: Some("ignored-prefix-".to_owned()),
                    collapsible: false,
                },
                ..Default::default()
            },
            RenderTextMode::Html,
        )
        .unwrap();
        assert_eq!(expected, preprocess_result);
    }

    #[test]
    fn list_embed() {
        let content = r#"# Chapter

1. Thing one

   ```sh
   Thing one
   ```

1. Thing two

   ```admonish
   Thing two
   ```

1. Thing three

   ```sh
   Thing three
   ```
"#;

        let expected = r##"# Chapter

1. Thing one

   ```sh
   Thing one
   ```

1. Thing two

   
   <div id="admonition-note" class="admonition admonish-note" role="note" aria-labelledby="admonition-note-title">
   <div class="admonition-title">
   <div id="admonition-note-title">
   
   Note
   
   </div>
   <a class="admonition-anchor-link" href="#admonition-note"></a>
   </div>
   <div>

   Thing two

   </div>
   </div>

1. Thing three

   ```sh
   Thing three
   ```
"##;

        assert_eq!(expected, prep(content));
    }

    // Regression test for https://github.com/tommilligan/mdbook-admonish/issues/223
    #[test]
    fn nested_list_should_not_render_indented_code_block() {
        let content = r#"# Chapter

- Level one

  ```admonish
  Thing one
  line two
  ```

  - Level two

    ```admonish
    Thing two
    line two
    ```

    - Level three

      ```admonish
      Thing three
      line two
      ```
"#;

        let expected = r##"# Chapter

- Level one

  
  <div id="admonition-note" class="admonition admonish-note" role="note" aria-labelledby="admonition-note-title">
  <div class="admonition-title">
  <div id="admonition-note-title">
  
  Note
  
  </div>
  <a class="admonition-anchor-link" href="#admonition-note"></a>
  </div>
  <div>

  Thing one
  line two

  </div>
  </div>

  - Level two

    
    <div id="admonition-note-1" class="admonition admonish-note" role="note" aria-labelledby="admonition-note-1-title">
    <div class="admonition-title">
    <div id="admonition-note-1-title">
    
    Note
    
    </div>
    <a class="admonition-anchor-link" href="#admonition-note-1"></a>
    </div>
    <div>

    Thing two
    line two

    </div>
    </div>

    - Level three

      
      <div id="admonition-note-2" class="admonition admonish-note" role="note" aria-labelledby="admonition-note-2-title">
      <div class="admonition-title">
      <div id="admonition-note-2-title">
      
      Note
      
      </div>
      <a class="admonition-anchor-link" href="#admonition-note-2"></a>
      </div>
      <div>

      Thing three
      line two

      </div>
      </div>
"##;

        assert_eq!(expected, prep(content));
    }
}
