//===-- SPIRVInstPrinter.h - Output SPIR-V MCInsts as ASM -------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This class prints a SPIR-V MCInst to a .s file.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIB_TARGET_SPIRV_INSTPRINTER_SPIRVINSTPRINTER_H
#define LLVM_LIB_TARGET_SPIRV_INSTPRINTER_SPIRVINSTPRINTER_H

#include "llvm/MC/MCInstPrinter.h"

namespace llvm {
class SPIRVInstPrinter : public MCInstPrinter {
private:
  void recordOpExtInstImport(const MCInst *MI);

public:
  using MCInstPrinter::MCInstPrinter;

  void printInst(const MCInst *MI, uint64_t Address, StringRef Annot,
                 const MCSubtargetInfo &STI, raw_ostream &OS) override;
  void printOperand(const MCInst *MI, unsigned OpNo, raw_ostream &O,
                    const char *Modifier = nullptr);

  void printStringImm(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printOpDecorate(const MCInst *MI, raw_ostream &O);
  void printOpExtInst(const MCInst *MI, raw_ostream &O);
  void printRemainingVariableOps(const MCInst *MI, unsigned StartIndex,
                                 raw_ostream &O, bool SkipFirstSpace = false,
                                 bool SkipImmediates = false);
  void printOpConstantVarOps(const MCInst *MI, unsigned StartIndex,
                             raw_ostream &O);

  void printExtInst(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  // SPIR-V enumerations printing.
  void printCapability(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printSourceLanguage(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printExecutionModel(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printAddressingModel(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printMemoryModel(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printExecutionMode(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printStorageClass(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printDim(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printSamplerAddressingMode(const MCInst *MI, unsigned OpNo,
                                  raw_ostream &O);
  void printSamplerFilterMode(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printImageFormat(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printImageChannelOrder(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printImageChannelDataType(const MCInst *MI, unsigned OpNo,
                                 raw_ostream &O);
  void printImageOperand(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printFPFastMathMode(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printFPRoundingMode(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printLinkageType(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printAccessQualifier(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printFunctionParameterAttribute(const MCInst *MI, unsigned OpNo,
                                       raw_ostream &O);

  void printDecoration(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printBuiltIn(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printSelectionControl(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printLoopControl(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printFunctionControl(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printMemorySemantics(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printMemoryOperand(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printScope(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printGroupOperation(const MCInst *MI, unsigned OpNo, raw_ostream &O);

  void printKernelEnqueueFlags(const MCInst *MI, unsigned OpNo, raw_ostream &O);
  void printKernelProfilingInfo(const MCInst *MI, unsigned OpNo,
                                raw_ostream &O);
  // Autogenerated by tblgen.
  std::pair<const char *, uint64_t> getMnemonic(const MCInst *MI) override;
  void printInstruction(const MCInst *MI, uint64_t Address, raw_ostream &O);
  static const char *getRegisterName(unsigned RegNo);
};
} // namespace llvm

#endif // LLVM_LIB_TARGET_SPIRV_INSTPRINTER_SPIRVINSTPRINTER_H
