#![feature(prelude_import)]
#![no_std]
#![doc(html_root_url = "https://docs.rs/serde_derive/1.0.137")]
#![allow(unknown_lints, bare_trait_objects)]
#![allow(
    clippy::branches_sharing_code,
    clippy::cognitive_complexity,
    clippy::collapsible_match,
    clippy::enum_variant_names,
    clippy::manual_map,
    clippy::match_like_matches_macro,
    clippy::needless_pass_by_value,
    clippy::too_many_arguments,
    clippy::trivially_copy_pass_by_ref,
    clippy::used_underscore_binding,
    clippy::wildcard_in_or_patterns,
    clippy::unnested_or_patterns,
)]
#![allow(
    clippy::cast_possible_truncation,
    clippy::checked_conversions,
    clippy::doc_markdown,
    clippy::enum_glob_use,
    clippy::indexing_slicing,
    clippy::items_after_statements,
    clippy::let_underscore_drop,
    clippy::manual_assert,
    clippy::map_err_ignore,
    clippy::match_same_arms,
    clippy::match_wildcard_for_single_variants,
    clippy::module_name_repetitions,
    clippy::must_use_candidate,
    clippy::option_if_let_else,
    clippy::similar_names,
    clippy::single_match_else,
    clippy::struct_excessive_bools,
    clippy::too_many_lines,
    clippy::unseparated_literal_suffix,
    clippy::unused_self,
    clippy::use_self,
    clippy::wildcard_imports
)]
#[prelude_import]
use ::std::prelude::rust_2015::*;
#[macro_use]
extern crate std;
#[macro_use]
extern crate quote;
#[macro_use]
extern crate syn;
extern crate proc_macro;
extern crate proc_macro2;
mod internals {
    pub mod ast {
        use internals::attr;
        use internals::check;
        use internals::{Ctxt, Derive};
        use syn;
        use syn::punctuated::Punctuated;
        pub struct Container<'a> {
            pub ident: syn::Ident,
            pub attrs: attr::Container,
            pub data: Data<'a>,
            pub generics: &'a syn::Generics,
            pub original: &'a syn::DeriveInput,
        }
        pub enum Data<'a> {
            Enum(Vec<Variant<'a>>),
            Struct(Style, Vec<Field<'a>>),
        }
        pub struct Variant<'a> {
            pub ident: syn::Ident,
            pub attrs: attr::Variant,
            pub style: Style,
            pub fields: Vec<Field<'a>>,
            pub original: &'a syn::Variant,
        }
        pub struct Field<'a> {
            pub member: syn::Member,
            pub attrs: attr::Field,
            pub ty: &'a syn::Type,
            pub original: &'a syn::Field,
        }
        pub enum Style {
            Struct,
            Tuple,
            Newtype,
            Unit,
        }
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::marker::Copy for Style {}
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::clone::Clone for Style {
            #[inline]
            fn clone(&self) -> Style {
                { *self }
            }
        }
        impl<'a> Container<'a> {
            pub fn from_ast(
                cx: &Ctxt,
                item: &'a syn::DeriveInput,
                derive: Derive,
            ) -> Option<Container<'a>> {
                let mut attrs = attr::Container::from_ast(cx, item);
                let mut data = match &item.data {
                    syn::Data::Enum(data) => {
                        Data::Enum(enum_from_ast(cx, &data.variants, attrs.default()))
                    }
                    syn::Data::Struct(data) => {
                        let (style, fields) = struct_from_ast(
                            cx,
                            &data.fields,
                            None,
                            attrs.default(),
                        );
                        Data::Struct(style, fields)
                    }
                    syn::Data::Union(_) => {
                        cx.error_spanned_by(
                            item,
                            "Serde does not support derive for unions",
                        );
                        return None;
                    }
                };
                let mut has_flatten = false;
                match &mut data {
                    Data::Enum(variants) => {
                        for variant in variants {
                            variant.attrs.rename_by_rules(attrs.rename_all_rules());
                            for field in &mut variant.fields {
                                if field.attrs.flatten() {
                                    has_flatten = true;
                                }
                                field
                                    .attrs
                                    .rename_by_rules(variant.attrs.rename_all_rules());
                            }
                        }
                    }
                    Data::Struct(_, fields) => {
                        for field in fields {
                            if field.attrs.flatten() {
                                has_flatten = true;
                            }
                            field.attrs.rename_by_rules(attrs.rename_all_rules());
                        }
                    }
                }
                if has_flatten {
                    attrs.mark_has_flatten();
                }
                let mut item = Container {
                    ident: item.ident.clone(),
                    attrs,
                    data,
                    generics: &item.generics,
                    original: item,
                };
                check::check(cx, &mut item, derive);
                Some(item)
            }
        }
        impl<'a> Data<'a> {
            pub fn all_fields(&'a self) -> Box<dyn Iterator<Item = &'a Field<'a>> + 'a> {
                match self {
                    Data::Enum(variants) => {
                        Box::new(
                            variants.iter().flat_map(|variant| variant.fields.iter()),
                        )
                    }
                    Data::Struct(_, fields) => Box::new(fields.iter()),
                }
            }
            pub fn has_getter(&self) -> bool {
                self.all_fields().any(|f| f.attrs.getter().is_some())
            }
        }
        fn enum_from_ast<'a>(
            cx: &Ctxt,
            variants: &'a Punctuated<syn::Variant, ::syn::token::Comma>,
            container_default: &attr::Default,
        ) -> Vec<Variant<'a>> {
            variants
                .iter()
                .map(|variant| {
                    let attrs = attr::Variant::from_ast(cx, variant);
                    let (style, fields) = struct_from_ast(
                        cx,
                        &variant.fields,
                        Some(&attrs),
                        container_default,
                    );
                    Variant {
                        ident: variant.ident.clone(),
                        attrs,
                        style,
                        fields,
                        original: variant,
                    }
                })
                .collect()
        }
        fn struct_from_ast<'a>(
            cx: &Ctxt,
            fields: &'a syn::Fields,
            attrs: Option<&attr::Variant>,
            container_default: &attr::Default,
        ) -> (Style, Vec<Field<'a>>) {
            match fields {
                syn::Fields::Named(fields) => {
                    (
                        Style::Struct,
                        fields_from_ast(cx, &fields.named, attrs, container_default),
                    )
                }
                syn::Fields::Unnamed(fields) if fields.unnamed.len() == 1 => {
                    (
                        Style::Newtype,
                        fields_from_ast(cx, &fields.unnamed, attrs, container_default),
                    )
                }
                syn::Fields::Unnamed(fields) => {
                    (
                        Style::Tuple,
                        fields_from_ast(cx, &fields.unnamed, attrs, container_default),
                    )
                }
                syn::Fields::Unit => (Style::Unit, Vec::new()),
            }
        }
        fn fields_from_ast<'a>(
            cx: &Ctxt,
            fields: &'a Punctuated<syn::Field, ::syn::token::Comma>,
            attrs: Option<&attr::Variant>,
            container_default: &attr::Default,
        ) -> Vec<Field<'a>> {
            fields
                .iter()
                .enumerate()
                .map(|(i, field)| Field {
                    member: match &field.ident {
                        Some(ident) => syn::Member::Named(ident.clone()),
                        None => syn::Member::Unnamed(i.into()),
                    },
                    attrs: attr::Field::from_ast(cx, i, field, attrs, container_default),
                    ty: &field.ty,
                    original: field,
                })
                .collect()
        }
    }
    pub mod attr {
        use internals::respan::respan;
        use internals::symbol::*;
        use internals::{ungroup, Ctxt};
        use proc_macro2::{Spacing, Span, TokenStream, TokenTree};
        use quote::ToTokens;
        use std::borrow::Cow;
        use std::collections::BTreeSet;
        use syn;
        use syn::parse::{self, Parse, ParseStream};
        use syn::punctuated::Punctuated;
        use syn::Ident;
        use syn::Meta::{List, NameValue, Path};
        use syn::NestedMeta::{Lit, Meta};
        pub use internals::case::RenameRule;
        struct Attr<'c, T> {
            cx: &'c Ctxt,
            name: Symbol,
            tokens: TokenStream,
            value: Option<T>,
        }
        impl<'c, T> Attr<'c, T> {
            fn none(cx: &'c Ctxt, name: Symbol) -> Self {
                Attr {
                    cx,
                    name,
                    tokens: TokenStream::new(),
                    value: None,
                }
            }
            fn set<A: ToTokens>(&mut self, obj: A, value: T) {
                let tokens = obj.into_token_stream();
                if self.value.is_some() {
                    self.cx
                        .error_spanned_by(
                            tokens,
                            {
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1(
                                        &["duplicate serde attribute `", "`"],
                                        &[::core::fmt::ArgumentV1::new_display(&self.name)],
                                    ),
                                );
                                res
                            },
                        );
                } else {
                    self.tokens = tokens;
                    self.value = Some(value);
                }
            }
            fn set_opt<A: ToTokens>(&mut self, obj: A, value: Option<T>) {
                if let Some(value) = value {
                    self.set(obj, value);
                }
            }
            fn set_if_none(&mut self, value: T) {
                if self.value.is_none() {
                    self.value = Some(value);
                }
            }
            fn get(self) -> Option<T> {
                self.value
            }
            fn get_with_tokens(self) -> Option<(TokenStream, T)> {
                match self.value {
                    Some(v) => Some((self.tokens, v)),
                    None => None,
                }
            }
        }
        struct BoolAttr<'c>(Attr<'c, ()>);
        impl<'c> BoolAttr<'c> {
            fn none(cx: &'c Ctxt, name: Symbol) -> Self {
                BoolAttr(Attr::none(cx, name))
            }
            fn set_true<A: ToTokens>(&mut self, obj: A) {
                self.0.set(obj, ());
            }
            fn get(&self) -> bool {
                self.0.value.is_some()
            }
        }
        struct VecAttr<'c, T> {
            cx: &'c Ctxt,
            name: Symbol,
            first_dup_tokens: TokenStream,
            values: Vec<T>,
        }
        impl<'c, T> VecAttr<'c, T> {
            fn none(cx: &'c Ctxt, name: Symbol) -> Self {
                VecAttr {
                    cx,
                    name,
                    first_dup_tokens: TokenStream::new(),
                    values: Vec::new(),
                }
            }
            fn insert<A: ToTokens>(&mut self, obj: A, value: T) {
                if self.values.len() == 1 {
                    self.first_dup_tokens = obj.into_token_stream();
                }
                self.values.push(value);
            }
            fn at_most_one(mut self) -> Result<Option<T>, ()> {
                if self.values.len() > 1 {
                    let dup_token = self.first_dup_tokens;
                    self.cx
                        .error_spanned_by(
                            dup_token,
                            {
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1(
                                        &["duplicate serde attribute `", "`"],
                                        &[::core::fmt::ArgumentV1::new_display(&self.name)],
                                    ),
                                );
                                res
                            },
                        );
                    Err(())
                } else {
                    Ok(self.values.pop())
                }
            }
            fn get(self) -> Vec<T> {
                self.values
            }
        }
        pub struct Name {
            serialize: String,
            serialize_renamed: bool,
            deserialize: String,
            deserialize_renamed: bool,
            deserialize_aliases: Vec<String>,
        }
        #[allow(deprecated)]
        fn unraw(ident: &Ident) -> String {
            ident.to_string().trim_left_matches("r#").to_owned()
        }
        impl Name {
            fn from_attrs(
                source_name: String,
                ser_name: Attr<String>,
                de_name: Attr<String>,
                de_aliases: Option<VecAttr<String>>,
            ) -> Name {
                let deserialize_aliases = match de_aliases {
                    Some(de_aliases) => {
                        let mut alias_list = BTreeSet::new();
                        for alias_name in de_aliases.get() {
                            alias_list.insert(alias_name);
                        }
                        alias_list.into_iter().collect()
                    }
                    None => Vec::new(),
                };
                let ser_name = ser_name.get();
                let ser_renamed = ser_name.is_some();
                let de_name = de_name.get();
                let de_renamed = de_name.is_some();
                Name {
                    serialize: ser_name.unwrap_or_else(|| source_name.clone()),
                    serialize_renamed: ser_renamed,
                    deserialize: de_name.unwrap_or(source_name),
                    deserialize_renamed: de_renamed,
                    deserialize_aliases,
                }
            }
            pub fn serialize_name(&self) -> String {
                self.serialize.clone()
            }
            pub fn deserialize_name(&self) -> String {
                self.deserialize.clone()
            }
            fn deserialize_aliases(&self) -> Vec<String> {
                let mut aliases = self.deserialize_aliases.clone();
                let main_name = self.deserialize_name();
                if !aliases.contains(&main_name) {
                    aliases.push(main_name);
                }
                aliases
            }
        }
        pub struct RenameAllRules {
            serialize: RenameRule,
            deserialize: RenameRule,
        }
        pub struct Container {
            name: Name,
            transparent: bool,
            deny_unknown_fields: bool,
            default: Default,
            rename_all_rules: RenameAllRules,
            ser_bound: Option<Vec<syn::WherePredicate>>,
            de_bound: Option<Vec<syn::WherePredicate>>,
            tag: TagType,
            type_from: Option<syn::Type>,
            type_try_from: Option<syn::Type>,
            type_into: Option<syn::Type>,
            remote: Option<syn::Path>,
            identifier: Identifier,
            has_flatten: bool,
            serde_path: Option<syn::Path>,
            is_packed: bool,
            expecting: Option<String>,
        }
        pub enum TagType {
            External,
            Internal { tag: String },
            Adjacent { tag: String, content: String },
            None,
        }
        pub enum Identifier {
            No,
            Field,
            Variant,
        }
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::marker::Copy for Identifier {}
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::clone::Clone for Identifier {
            #[inline]
            fn clone(&self) -> Identifier {
                { *self }
            }
        }
        impl Identifier {}
        impl Container {
            pub fn from_ast(cx: &Ctxt, item: &syn::DeriveInput) -> Self {
                let mut ser_name = Attr::none(cx, RENAME);
                let mut de_name = Attr::none(cx, RENAME);
                let mut transparent = BoolAttr::none(cx, TRANSPARENT);
                let mut deny_unknown_fields = BoolAttr::none(cx, DENY_UNKNOWN_FIELDS);
                let mut default = Attr::none(cx, DEFAULT);
                let mut rename_all_ser_rule = Attr::none(cx, RENAME_ALL);
                let mut rename_all_de_rule = Attr::none(cx, RENAME_ALL);
                let mut ser_bound = Attr::none(cx, BOUND);
                let mut de_bound = Attr::none(cx, BOUND);
                let mut untagged = BoolAttr::none(cx, UNTAGGED);
                let mut internal_tag = Attr::none(cx, TAG);
                let mut content = Attr::none(cx, CONTENT);
                let mut type_from = Attr::none(cx, FROM);
                let mut type_try_from = Attr::none(cx, TRY_FROM);
                let mut type_into = Attr::none(cx, INTO);
                let mut remote = Attr::none(cx, REMOTE);
                let mut field_identifier = BoolAttr::none(cx, FIELD_IDENTIFIER);
                let mut variant_identifier = BoolAttr::none(cx, VARIANT_IDENTIFIER);
                let mut serde_path = Attr::none(cx, CRATE);
                let mut expecting = Attr::none(cx, EXPECTING);
                for meta_item in item
                    .attrs
                    .iter()
                    .flat_map(|attr| get_serde_meta_items(cx, attr))
                    .flatten()
                {
                    match &meta_item {
                        Meta(NameValue(m)) if m.path == RENAME => {
                            if let Ok(s) = get_lit_str(cx, RENAME, &m.lit) {
                                ser_name.set(&m.path, s.value());
                                de_name.set(&m.path, s.value());
                            }
                        }
                        Meta(List(m)) if m.path == RENAME => {
                            if let Ok((ser, de)) = get_renames(cx, &m.nested) {
                                ser_name.set_opt(&m.path, ser.map(syn::LitStr::value));
                                de_name.set_opt(&m.path, de.map(syn::LitStr::value));
                            }
                        }
                        Meta(NameValue(m)) if m.path == RENAME_ALL => {
                            if let Ok(s) = get_lit_str(cx, RENAME_ALL, &m.lit) {
                                match RenameRule::from_str(&s.value()) {
                                    Ok(rename_rule) => {
                                        rename_all_ser_rule.set(&m.path, rename_rule);
                                        rename_all_de_rule.set(&m.path, rename_rule);
                                    }
                                    Err(err) => cx.error_spanned_by(s, err),
                                }
                            }
                        }
                        Meta(List(m)) if m.path == RENAME_ALL => {
                            if let Ok((ser, de)) = get_renames(cx, &m.nested) {
                                if let Some(ser) = ser {
                                    match RenameRule::from_str(&ser.value()) {
                                        Ok(rename_rule) => {
                                            rename_all_ser_rule.set(&m.path, rename_rule)
                                        }
                                        Err(err) => cx.error_spanned_by(ser, err),
                                    }
                                }
                                if let Some(de) = de {
                                    match RenameRule::from_str(&de.value()) {
                                        Ok(rename_rule) => {
                                            rename_all_de_rule.set(&m.path, rename_rule)
                                        }
                                        Err(err) => cx.error_spanned_by(de, err),
                                    }
                                }
                            }
                        }
                        Meta(Path(word)) if word == TRANSPARENT => {
                            transparent.set_true(word);
                        }
                        Meta(Path(word)) if word == DENY_UNKNOWN_FIELDS => {
                            deny_unknown_fields.set_true(word);
                        }
                        Meta(Path(word)) if word == DEFAULT => {
                            match &item.data {
                                syn::Data::Struct(syn::DataStruct { fields, .. }) => {
                                    match fields {
                                        syn::Fields::Named(_) => {
                                            default.set(word, Default::Default);
                                        }
                                        syn::Fields::Unnamed(_) | syn::Fields::Unit => {
                                            cx.error_spanned_by(
                                                fields,
                                                "#[serde(default)] can only be used on structs with named fields",
                                            )
                                        }
                                    }
                                }
                                syn::Data::Enum(syn::DataEnum { enum_token, .. }) => {
                                    cx.error_spanned_by(
                                        enum_token,
                                        "#[serde(default)] can only be used on structs with named fields",
                                    )
                                }
                                syn::Data::Union(syn::DataUnion { union_token, .. }) => {
                                    cx.error_spanned_by(
                                        union_token,
                                        "#[serde(default)] can only be used on structs with named fields",
                                    )
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == DEFAULT => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                DEFAULT,
                                &m.lit,
                            ) {
                                match &item.data {
                                    syn::Data::Struct(syn::DataStruct { fields, .. }) => {
                                        match fields {
                                            syn::Fields::Named(_) => {
                                                default.set(&m.path, Default::Path(path));
                                            }
                                            syn::Fields::Unnamed(_) | syn::Fields::Unit => {
                                                cx.error_spanned_by(
                                                    fields,
                                                    "#[serde(default = \"...\")] can only be used on structs with named fields",
                                                )
                                            }
                                        }
                                    }
                                    syn::Data::Enum(syn::DataEnum { enum_token, .. }) => {
                                        cx.error_spanned_by(
                                            enum_token,
                                            "#[serde(default = \"...\")] can only be used on structs with named fields",
                                        )
                                    }
                                    syn::Data::Union(syn::DataUnion { union_token, .. }) => {
                                        cx.error_spanned_by(
                                            union_token,
                                            "#[serde(default = \"...\")] can only be used on structs with named fields",
                                        )
                                    }
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == BOUND => {
                            if let Ok(where_predicates) = parse_lit_into_where(
                                cx,
                                BOUND,
                                BOUND,
                                &m.lit,
                            ) {
                                ser_bound.set(&m.path, where_predicates.clone());
                                de_bound.set(&m.path, where_predicates);
                            }
                        }
                        Meta(List(m)) if m.path == BOUND => {
                            if let Ok((ser, de)) = get_where_predicates(cx, &m.nested) {
                                ser_bound.set_opt(&m.path, ser);
                                de_bound.set_opt(&m.path, de);
                            }
                        }
                        Meta(Path(word)) if word == UNTAGGED => {
                            match item.data {
                                syn::Data::Enum(_) => {
                                    untagged.set_true(word);
                                }
                                syn::Data::Struct(syn::DataStruct { struct_token, .. }) => {
                                    cx.error_spanned_by(
                                        struct_token,
                                        "#[serde(untagged)] can only be used on enums",
                                    );
                                }
                                syn::Data::Union(syn::DataUnion { union_token, .. }) => {
                                    cx.error_spanned_by(
                                        union_token,
                                        "#[serde(untagged)] can only be used on enums",
                                    );
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == TAG => {
                            if let Ok(s) = get_lit_str(cx, TAG, &m.lit) {
                                match &item.data {
                                    syn::Data::Enum(_) => {
                                        internal_tag.set(&m.path, s.value());
                                    }
                                    syn::Data::Struct(syn::DataStruct { fields, .. }) => {
                                        match fields {
                                            syn::Fields::Named(_) => {
                                                internal_tag.set(&m.path, s.value());
                                            }
                                            syn::Fields::Unnamed(_) | syn::Fields::Unit => {
                                                cx.error_spanned_by(
                                                    fields,
                                                    "#[serde(tag = \"...\")] can only be used on enums and structs with named fields",
                                                );
                                            }
                                        }
                                    }
                                    syn::Data::Union(syn::DataUnion { union_token, .. }) => {
                                        cx.error_spanned_by(
                                            union_token,
                                            "#[serde(tag = \"...\")] can only be used on enums and structs with named fields",
                                        );
                                    }
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == CONTENT => {
                            if let Ok(s) = get_lit_str(cx, CONTENT, &m.lit) {
                                match &item.data {
                                    syn::Data::Enum(_) => {
                                        content.set(&m.path, s.value());
                                    }
                                    syn::Data::Struct(syn::DataStruct { struct_token, .. }) => {
                                        cx.error_spanned_by(
                                            struct_token,
                                            "#[serde(content = \"...\")] can only be used on enums",
                                        );
                                    }
                                    syn::Data::Union(syn::DataUnion { union_token, .. }) => {
                                        cx.error_spanned_by(
                                            union_token,
                                            "#[serde(content = \"...\")] can only be used on enums",
                                        );
                                    }
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == FROM => {
                            if let Ok(from_ty) = parse_lit_into_ty(cx, FROM, &m.lit) {
                                type_from.set_opt(&m.path, Some(from_ty));
                            }
                        }
                        Meta(NameValue(m)) if m.path == TRY_FROM => {
                            if let Ok(try_from_ty) = parse_lit_into_ty(
                                cx,
                                TRY_FROM,
                                &m.lit,
                            ) {
                                type_try_from.set_opt(&m.path, Some(try_from_ty));
                            }
                        }
                        Meta(NameValue(m)) if m.path == INTO => {
                            if let Ok(into_ty) = parse_lit_into_ty(cx, INTO, &m.lit) {
                                type_into.set_opt(&m.path, Some(into_ty));
                            }
                        }
                        Meta(NameValue(m)) if m.path == REMOTE => {
                            if let Ok(path) = parse_lit_into_path(cx, REMOTE, &m.lit) {
                                if is_primitive_path(&path, "Self") {
                                    remote.set(&m.path, item.ident.clone().into());
                                } else {
                                    remote.set(&m.path, path);
                                }
                            }
                        }
                        Meta(Path(word)) if word == FIELD_IDENTIFIER => {
                            field_identifier.set_true(word);
                        }
                        Meta(Path(word)) if word == VARIANT_IDENTIFIER => {
                            variant_identifier.set_true(word);
                        }
                        Meta(NameValue(m)) if m.path == CRATE => {
                            if let Ok(path) = parse_lit_into_path(cx, CRATE, &m.lit) {
                                serde_path.set(&m.path, path);
                            }
                        }
                        Meta(NameValue(m)) if m.path == EXPECTING => {
                            if let Ok(s) = get_lit_str(cx, EXPECTING, &m.lit) {
                                expecting.set(&m.path, s.value());
                            }
                        }
                        Meta(meta_item) => {
                            let path = meta_item
                                .path()
                                .into_token_stream()
                                .to_string()
                                .replace(' ', "");
                            cx.error_spanned_by(
                                meta_item.path(),
                                {
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &["unknown serde container attribute `", "`"],
                                            &[::core::fmt::ArgumentV1::new_display(&path)],
                                        ),
                                    );
                                    res
                                },
                            );
                        }
                        Lit(lit) => {
                            cx.error_spanned_by(
                                lit,
                                "unexpected literal in serde container attribute",
                            );
                        }
                    }
                }
                let mut is_packed = false;
                for attr in &item.attrs {
                    if attr.path.is_ident("repr") {
                        let _ = attr
                            .parse_args_with(|input: ParseStream| {
                                while let Some(token) = input.parse()? {
                                    if let TokenTree::Ident(ident) = token {
                                        is_packed |= ident == "packed";
                                    }
                                }
                                Ok(())
                            });
                    }
                }
                Container {
                    name: Name::from_attrs(unraw(&item.ident), ser_name, de_name, None),
                    transparent: transparent.get(),
                    deny_unknown_fields: deny_unknown_fields.get(),
                    default: default.get().unwrap_or(Default::None),
                    rename_all_rules: RenameAllRules {
                        serialize: rename_all_ser_rule.get().unwrap_or(RenameRule::None),
                        deserialize: rename_all_de_rule.get().unwrap_or(RenameRule::None),
                    },
                    ser_bound: ser_bound.get(),
                    de_bound: de_bound.get(),
                    tag: decide_tag(cx, item, untagged, internal_tag, content),
                    type_from: type_from.get(),
                    type_try_from: type_try_from.get(),
                    type_into: type_into.get(),
                    remote: remote.get(),
                    identifier: decide_identifier(
                        cx,
                        item,
                        field_identifier,
                        variant_identifier,
                    ),
                    has_flatten: false,
                    serde_path: serde_path.get(),
                    is_packed,
                    expecting: expecting.get(),
                }
            }
            pub fn name(&self) -> &Name {
                &self.name
            }
            pub fn rename_all_rules(&self) -> &RenameAllRules {
                &self.rename_all_rules
            }
            pub fn transparent(&self) -> bool {
                self.transparent
            }
            pub fn deny_unknown_fields(&self) -> bool {
                self.deny_unknown_fields
            }
            pub fn default(&self) -> &Default {
                &self.default
            }
            pub fn ser_bound(&self) -> Option<&[syn::WherePredicate]> {
                self.ser_bound.as_ref().map(|vec| &vec[..])
            }
            pub fn de_bound(&self) -> Option<&[syn::WherePredicate]> {
                self.de_bound.as_ref().map(|vec| &vec[..])
            }
            pub fn tag(&self) -> &TagType {
                &self.tag
            }
            pub fn type_from(&self) -> Option<&syn::Type> {
                self.type_from.as_ref()
            }
            pub fn type_try_from(&self) -> Option<&syn::Type> {
                self.type_try_from.as_ref()
            }
            pub fn type_into(&self) -> Option<&syn::Type> {
                self.type_into.as_ref()
            }
            pub fn remote(&self) -> Option<&syn::Path> {
                self.remote.as_ref()
            }
            pub fn is_packed(&self) -> bool {
                self.is_packed
            }
            pub fn identifier(&self) -> Identifier {
                self.identifier
            }
            pub fn has_flatten(&self) -> bool {
                self.has_flatten
            }
            pub fn mark_has_flatten(&mut self) {
                self.has_flatten = true;
            }
            pub fn custom_serde_path(&self) -> Option<&syn::Path> {
                self.serde_path.as_ref()
            }
            pub fn serde_path(&self) -> Cow<syn::Path> {
                self.custom_serde_path()
                    .map_or_else(
                        || Cow::Owned(
                            ::syn::parse_quote::parse({
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                _s
                            }),
                        ),
                        Cow::Borrowed,
                    )
            }
            pub fn expecting(&self) -> Option<&str> {
                self.expecting.as_ref().map(String::as_ref)
            }
        }
        fn decide_tag(
            cx: &Ctxt,
            item: &syn::DeriveInput,
            untagged: BoolAttr,
            internal_tag: Attr<String>,
            content: Attr<String>,
        ) -> TagType {
            match (
                untagged.0.get_with_tokens(),
                internal_tag.get_with_tokens(),
                content.get_with_tokens(),
            ) {
                (None, None, None) => TagType::External,
                (Some(_), None, None) => TagType::None,
                (None, Some((_, tag)), None) => {
                    if let syn::Data::Enum(data) = &item.data {
                        for variant in &data.variants {
                            match &variant.fields {
                                syn::Fields::Named(_) | syn::Fields::Unit => {}
                                syn::Fields::Unnamed(fields) => {
                                    if fields.unnamed.len() != 1 {
                                        cx.error_spanned_by(
                                            variant,
                                            "#[serde(tag = \"...\")] cannot be used with tuple variants",
                                        );
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    TagType::Internal { tag }
                }
                (Some((untagged_tokens, _)), Some((tag_tokens, _)), None) => {
                    cx.error_spanned_by(
                        untagged_tokens,
                        "enum cannot be both untagged and internally tagged",
                    );
                    cx.error_spanned_by(
                        tag_tokens,
                        "enum cannot be both untagged and internally tagged",
                    );
                    TagType::External
                }
                (None, None, Some((content_tokens, _))) => {
                    cx.error_spanned_by(
                        content_tokens,
                        "#[serde(tag = \"...\", content = \"...\")] must be used together",
                    );
                    TagType::External
                }
                (Some((untagged_tokens, _)), None, Some((content_tokens, _))) => {
                    cx.error_spanned_by(
                        untagged_tokens,
                        "untagged enum cannot have #[serde(content = \"...\")]",
                    );
                    cx.error_spanned_by(
                        content_tokens,
                        "untagged enum cannot have #[serde(content = \"...\")]",
                    );
                    TagType::External
                }
                (None, Some((_, tag)), Some((_, content))) => {
                    TagType::Adjacent { tag, content }
                }
                (
                    Some((untagged_tokens, _)),
                    Some((tag_tokens, _)),
                    Some((content_tokens, _)),
                ) => {
                    cx.error_spanned_by(
                        untagged_tokens,
                        "untagged enum cannot have #[serde(tag = \"...\", content = \"...\")]",
                    );
                    cx.error_spanned_by(
                        tag_tokens,
                        "untagged enum cannot have #[serde(tag = \"...\", content = \"...\")]",
                    );
                    cx.error_spanned_by(
                        content_tokens,
                        "untagged enum cannot have #[serde(tag = \"...\", content = \"...\")]",
                    );
                    TagType::External
                }
            }
        }
        fn decide_identifier(
            cx: &Ctxt,
            item: &syn::DeriveInput,
            field_identifier: BoolAttr,
            variant_identifier: BoolAttr,
        ) -> Identifier {
            match (
                &item.data,
                field_identifier.0.get_with_tokens(),
                variant_identifier.0.get_with_tokens(),
            ) {
                (_, None, None) => Identifier::No,
                (
                    _,
                    Some((field_identifier_tokens, _)),
                    Some((variant_identifier_tokens, _)),
                ) => {
                    cx.error_spanned_by(
                        field_identifier_tokens,
                        "#[serde(field_identifier)] and #[serde(variant_identifier)] cannot both be set",
                    );
                    cx.error_spanned_by(
                        variant_identifier_tokens,
                        "#[serde(field_identifier)] and #[serde(variant_identifier)] cannot both be set",
                    );
                    Identifier::No
                }
                (syn::Data::Enum(_), Some(_), None) => Identifier::Field,
                (syn::Data::Enum(_), None, Some(_)) => Identifier::Variant,
                (
                    syn::Data::Struct(syn::DataStruct { struct_token, .. }),
                    Some(_),
                    None,
                ) => {
                    cx.error_spanned_by(
                        struct_token,
                        "#[serde(field_identifier)] can only be used on an enum",
                    );
                    Identifier::No
                }
                (
                    syn::Data::Union(syn::DataUnion { union_token, .. }),
                    Some(_),
                    None,
                ) => {
                    cx.error_spanned_by(
                        union_token,
                        "#[serde(field_identifier)] can only be used on an enum",
                    );
                    Identifier::No
                }
                (
                    syn::Data::Struct(syn::DataStruct { struct_token, .. }),
                    None,
                    Some(_),
                ) => {
                    cx.error_spanned_by(
                        struct_token,
                        "#[serde(variant_identifier)] can only be used on an enum",
                    );
                    Identifier::No
                }
                (
                    syn::Data::Union(syn::DataUnion { union_token, .. }),
                    None,
                    Some(_),
                ) => {
                    cx.error_spanned_by(
                        union_token,
                        "#[serde(variant_identifier)] can only be used on an enum",
                    );
                    Identifier::No
                }
            }
        }
        pub struct Variant {
            name: Name,
            rename_all_rules: RenameAllRules,
            ser_bound: Option<Vec<syn::WherePredicate>>,
            de_bound: Option<Vec<syn::WherePredicate>>,
            skip_deserializing: bool,
            skip_serializing: bool,
            other: bool,
            serialize_with: Option<syn::ExprPath>,
            deserialize_with: Option<syn::ExprPath>,
            borrow: Option<syn::Meta>,
        }
        impl Variant {
            pub fn from_ast(cx: &Ctxt, variant: &syn::Variant) -> Self {
                let mut ser_name = Attr::none(cx, RENAME);
                let mut de_name = Attr::none(cx, RENAME);
                let mut de_aliases = VecAttr::none(cx, RENAME);
                let mut skip_deserializing = BoolAttr::none(cx, SKIP_DESERIALIZING);
                let mut skip_serializing = BoolAttr::none(cx, SKIP_SERIALIZING);
                let mut rename_all_ser_rule = Attr::none(cx, RENAME_ALL);
                let mut rename_all_de_rule = Attr::none(cx, RENAME_ALL);
                let mut ser_bound = Attr::none(cx, BOUND);
                let mut de_bound = Attr::none(cx, BOUND);
                let mut other = BoolAttr::none(cx, OTHER);
                let mut serialize_with = Attr::none(cx, SERIALIZE_WITH);
                let mut deserialize_with = Attr::none(cx, DESERIALIZE_WITH);
                let mut borrow = Attr::none(cx, BORROW);
                for meta_item in variant
                    .attrs
                    .iter()
                    .flat_map(|attr| get_serde_meta_items(cx, attr))
                    .flatten()
                {
                    match &meta_item {
                        Meta(NameValue(m)) if m.path == RENAME => {
                            if let Ok(s) = get_lit_str(cx, RENAME, &m.lit) {
                                ser_name.set(&m.path, s.value());
                                de_name.set_if_none(s.value());
                                de_aliases.insert(&m.path, s.value());
                            }
                        }
                        Meta(List(m)) if m.path == RENAME => {
                            if let Ok((ser, de)) = get_multiple_renames(cx, &m.nested) {
                                ser_name.set_opt(&m.path, ser.map(syn::LitStr::value));
                                for de_value in de {
                                    de_name.set_if_none(de_value.value());
                                    de_aliases.insert(&m.path, de_value.value());
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == ALIAS => {
                            if let Ok(s) = get_lit_str(cx, ALIAS, &m.lit) {
                                de_aliases.insert(&m.path, s.value());
                            }
                        }
                        Meta(NameValue(m)) if m.path == RENAME_ALL => {
                            if let Ok(s) = get_lit_str(cx, RENAME_ALL, &m.lit) {
                                match RenameRule::from_str(&s.value()) {
                                    Ok(rename_rule) => {
                                        rename_all_ser_rule.set(&m.path, rename_rule);
                                        rename_all_de_rule.set(&m.path, rename_rule);
                                    }
                                    Err(err) => cx.error_spanned_by(s, err),
                                }
                            }
                        }
                        Meta(List(m)) if m.path == RENAME_ALL => {
                            if let Ok((ser, de)) = get_renames(cx, &m.nested) {
                                if let Some(ser) = ser {
                                    match RenameRule::from_str(&ser.value()) {
                                        Ok(rename_rule) => {
                                            rename_all_ser_rule.set(&m.path, rename_rule)
                                        }
                                        Err(err) => cx.error_spanned_by(ser, err),
                                    }
                                }
                                if let Some(de) = de {
                                    match RenameRule::from_str(&de.value()) {
                                        Ok(rename_rule) => {
                                            rename_all_de_rule.set(&m.path, rename_rule)
                                        }
                                        Err(err) => cx.error_spanned_by(de, err),
                                    }
                                }
                            }
                        }
                        Meta(Path(word)) if word == SKIP => {
                            skip_serializing.set_true(word);
                            skip_deserializing.set_true(word);
                        }
                        Meta(Path(word)) if word == SKIP_DESERIALIZING => {
                            skip_deserializing.set_true(word);
                        }
                        Meta(Path(word)) if word == SKIP_SERIALIZING => {
                            skip_serializing.set_true(word);
                        }
                        Meta(Path(word)) if word == OTHER => {
                            other.set_true(word);
                        }
                        Meta(NameValue(m)) if m.path == BOUND => {
                            if let Ok(where_predicates) = parse_lit_into_where(
                                cx,
                                BOUND,
                                BOUND,
                                &m.lit,
                            ) {
                                ser_bound.set(&m.path, where_predicates.clone());
                                de_bound.set(&m.path, where_predicates);
                            }
                        }
                        Meta(List(m)) if m.path == BOUND => {
                            if let Ok((ser, de)) = get_where_predicates(cx, &m.nested) {
                                ser_bound.set_opt(&m.path, ser);
                                de_bound.set_opt(&m.path, de);
                            }
                        }
                        Meta(NameValue(m)) if m.path == WITH => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                WITH,
                                &m.lit,
                            ) {
                                let mut ser_path = path.clone();
                                ser_path
                                    .path
                                    .segments
                                    .push(Ident::new("serialize", Span::call_site()).into());
                                serialize_with.set(&m.path, ser_path);
                                let mut de_path = path;
                                de_path
                                    .path
                                    .segments
                                    .push(Ident::new("deserialize", Span::call_site()).into());
                                deserialize_with.set(&m.path, de_path);
                            }
                        }
                        Meta(NameValue(m)) if m.path == SERIALIZE_WITH => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                SERIALIZE_WITH,
                                &m.lit,
                            ) {
                                serialize_with.set(&m.path, path);
                            }
                        }
                        Meta(NameValue(m)) if m.path == DESERIALIZE_WITH => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                DESERIALIZE_WITH,
                                &m.lit,
                            ) {
                                deserialize_with.set(&m.path, path);
                            }
                        }
                        Meta(m) if m.path() == BORROW => {
                            match &variant.fields {
                                syn::Fields::Unnamed(
                                    fields,
                                ) if fields.unnamed.len() == 1 => {
                                    borrow.set(m.path(), m.clone());
                                }
                                _ => {
                                    cx.error_spanned_by(
                                        variant,
                                        "#[serde(borrow)] may only be used on newtype variants",
                                    );
                                }
                            }
                        }
                        Meta(meta_item) => {
                            let path = meta_item
                                .path()
                                .into_token_stream()
                                .to_string()
                                .replace(' ', "");
                            cx.error_spanned_by(
                                meta_item.path(),
                                {
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &["unknown serde variant attribute `", "`"],
                                            &[::core::fmt::ArgumentV1::new_display(&path)],
                                        ),
                                    );
                                    res
                                },
                            );
                        }
                        Lit(lit) => {
                            cx.error_spanned_by(
                                lit,
                                "unexpected literal in serde variant attribute",
                            );
                        }
                    }
                }
                Variant {
                    name: Name::from_attrs(
                        unraw(&variant.ident),
                        ser_name,
                        de_name,
                        Some(de_aliases),
                    ),
                    rename_all_rules: RenameAllRules {
                        serialize: rename_all_ser_rule.get().unwrap_or(RenameRule::None),
                        deserialize: rename_all_de_rule.get().unwrap_or(RenameRule::None),
                    },
                    ser_bound: ser_bound.get(),
                    de_bound: de_bound.get(),
                    skip_deserializing: skip_deserializing.get(),
                    skip_serializing: skip_serializing.get(),
                    other: other.get(),
                    serialize_with: serialize_with.get(),
                    deserialize_with: deserialize_with.get(),
                    borrow: borrow.get(),
                }
            }
            pub fn name(&self) -> &Name {
                &self.name
            }
            pub fn aliases(&self) -> Vec<String> {
                self.name.deserialize_aliases()
            }
            pub fn rename_by_rules(&mut self, rules: &RenameAllRules) {
                if !self.name.serialize_renamed {
                    self.name.serialize = rules
                        .serialize
                        .apply_to_variant(&self.name.serialize);
                }
                if !self.name.deserialize_renamed {
                    self.name.deserialize = rules
                        .deserialize
                        .apply_to_variant(&self.name.deserialize);
                }
            }
            pub fn rename_all_rules(&self) -> &RenameAllRules {
                &self.rename_all_rules
            }
            pub fn ser_bound(&self) -> Option<&[syn::WherePredicate]> {
                self.ser_bound.as_ref().map(|vec| &vec[..])
            }
            pub fn de_bound(&self) -> Option<&[syn::WherePredicate]> {
                self.de_bound.as_ref().map(|vec| &vec[..])
            }
            pub fn skip_deserializing(&self) -> bool {
                self.skip_deserializing
            }
            pub fn skip_serializing(&self) -> bool {
                self.skip_serializing
            }
            pub fn other(&self) -> bool {
                self.other
            }
            pub fn serialize_with(&self) -> Option<&syn::ExprPath> {
                self.serialize_with.as_ref()
            }
            pub fn deserialize_with(&self) -> Option<&syn::ExprPath> {
                self.deserialize_with.as_ref()
            }
        }
        pub struct Field {
            name: Name,
            skip_serializing: bool,
            skip_deserializing: bool,
            skip_serializing_if: Option<syn::ExprPath>,
            default: Default,
            serialize_with: Option<syn::ExprPath>,
            deserialize_with: Option<syn::ExprPath>,
            ser_bound: Option<Vec<syn::WherePredicate>>,
            de_bound: Option<Vec<syn::WherePredicate>>,
            borrowed_lifetimes: BTreeSet<syn::Lifetime>,
            getter: Option<syn::ExprPath>,
            flatten: bool,
            transparent: bool,
        }
        pub enum Default {
            None,
            Default,
            Path(syn::ExprPath),
        }
        impl Default {
            pub fn is_none(&self) -> bool {
                match self {
                    Default::None => true,
                    Default::Default | Default::Path(_) => false,
                }
            }
        }
        impl Field {
            pub fn from_ast(
                cx: &Ctxt,
                index: usize,
                field: &syn::Field,
                attrs: Option<&Variant>,
                container_default: &Default,
            ) -> Self {
                let mut ser_name = Attr::none(cx, RENAME);
                let mut de_name = Attr::none(cx, RENAME);
                let mut de_aliases = VecAttr::none(cx, RENAME);
                let mut skip_serializing = BoolAttr::none(cx, SKIP_SERIALIZING);
                let mut skip_deserializing = BoolAttr::none(cx, SKIP_DESERIALIZING);
                let mut skip_serializing_if = Attr::none(cx, SKIP_SERIALIZING_IF);
                let mut default = Attr::none(cx, DEFAULT);
                let mut serialize_with = Attr::none(cx, SERIALIZE_WITH);
                let mut deserialize_with = Attr::none(cx, DESERIALIZE_WITH);
                let mut ser_bound = Attr::none(cx, BOUND);
                let mut de_bound = Attr::none(cx, BOUND);
                let mut borrowed_lifetimes = Attr::none(cx, BORROW);
                let mut getter = Attr::none(cx, GETTER);
                let mut flatten = BoolAttr::none(cx, FLATTEN);
                let ident = match &field.ident {
                    Some(ident) => unraw(ident),
                    None => index.to_string(),
                };
                let variant_borrow = attrs
                    .and_then(|variant| variant.borrow.as_ref())
                    .map(|borrow| Meta(borrow.clone()));
                for meta_item in field
                    .attrs
                    .iter()
                    .flat_map(|attr| get_serde_meta_items(cx, attr))
                    .flatten()
                    .chain(variant_borrow)
                {
                    match &meta_item {
                        Meta(NameValue(m)) if m.path == RENAME => {
                            if let Ok(s) = get_lit_str(cx, RENAME, &m.lit) {
                                ser_name.set(&m.path, s.value());
                                de_name.set_if_none(s.value());
                                de_aliases.insert(&m.path, s.value());
                            }
                        }
                        Meta(List(m)) if m.path == RENAME => {
                            if let Ok((ser, de)) = get_multiple_renames(cx, &m.nested) {
                                ser_name.set_opt(&m.path, ser.map(syn::LitStr::value));
                                for de_value in de {
                                    de_name.set_if_none(de_value.value());
                                    de_aliases.insert(&m.path, de_value.value());
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == ALIAS => {
                            if let Ok(s) = get_lit_str(cx, ALIAS, &m.lit) {
                                de_aliases.insert(&m.path, s.value());
                            }
                        }
                        Meta(Path(word)) if word == DEFAULT => {
                            default.set(word, Default::Default);
                        }
                        Meta(NameValue(m)) if m.path == DEFAULT => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                DEFAULT,
                                &m.lit,
                            ) {
                                default.set(&m.path, Default::Path(path));
                            }
                        }
                        Meta(Path(word)) if word == SKIP_SERIALIZING => {
                            skip_serializing.set_true(word);
                        }
                        Meta(Path(word)) if word == SKIP_DESERIALIZING => {
                            skip_deserializing.set_true(word);
                        }
                        Meta(Path(word)) if word == SKIP => {
                            skip_serializing.set_true(word);
                            skip_deserializing.set_true(word);
                        }
                        Meta(NameValue(m)) if m.path == SKIP_SERIALIZING_IF => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                SKIP_SERIALIZING_IF,
                                &m.lit,
                            ) {
                                skip_serializing_if.set(&m.path, path);
                            }
                        }
                        Meta(NameValue(m)) if m.path == SERIALIZE_WITH => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                SERIALIZE_WITH,
                                &m.lit,
                            ) {
                                serialize_with.set(&m.path, path);
                            }
                        }
                        Meta(NameValue(m)) if m.path == DESERIALIZE_WITH => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                DESERIALIZE_WITH,
                                &m.lit,
                            ) {
                                deserialize_with.set(&m.path, path);
                            }
                        }
                        Meta(NameValue(m)) if m.path == WITH => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                WITH,
                                &m.lit,
                            ) {
                                let mut ser_path = path.clone();
                                ser_path
                                    .path
                                    .segments
                                    .push(Ident::new("serialize", Span::call_site()).into());
                                serialize_with.set(&m.path, ser_path);
                                let mut de_path = path;
                                de_path
                                    .path
                                    .segments
                                    .push(Ident::new("deserialize", Span::call_site()).into());
                                deserialize_with.set(&m.path, de_path);
                            }
                        }
                        Meta(NameValue(m)) if m.path == BOUND => {
                            if let Ok(where_predicates) = parse_lit_into_where(
                                cx,
                                BOUND,
                                BOUND,
                                &m.lit,
                            ) {
                                ser_bound.set(&m.path, where_predicates.clone());
                                de_bound.set(&m.path, where_predicates);
                            }
                        }
                        Meta(List(m)) if m.path == BOUND => {
                            if let Ok((ser, de)) = get_where_predicates(cx, &m.nested) {
                                ser_bound.set_opt(&m.path, ser);
                                de_bound.set_opt(&m.path, de);
                            }
                        }
                        Meta(Path(word)) if word == BORROW => {
                            if let Ok(borrowable) = borrowable_lifetimes(
                                cx,
                                &ident,
                                field,
                            ) {
                                borrowed_lifetimes.set(word, borrowable);
                            }
                        }
                        Meta(NameValue(m)) if m.path == BORROW => {
                            if let Ok(lifetimes) = parse_lit_into_lifetimes(
                                cx,
                                BORROW,
                                &m.lit,
                            ) {
                                if let Ok(borrowable) = borrowable_lifetimes(
                                    cx,
                                    &ident,
                                    field,
                                ) {
                                    for lifetime in &lifetimes {
                                        if !borrowable.contains(lifetime) {
                                            cx.error_spanned_by(
                                                field,
                                                {
                                                    let res = ::alloc::fmt::format(
                                                        ::core::fmt::Arguments::new_v1(
                                                            &["field `", "` does not have lifetime "],
                                                            &[
                                                                ::core::fmt::ArgumentV1::new_display(&ident),
                                                                ::core::fmt::ArgumentV1::new_display(&lifetime),
                                                            ],
                                                        ),
                                                    );
                                                    res
                                                },
                                            );
                                        }
                                    }
                                    borrowed_lifetimes.set(&m.path, lifetimes);
                                }
                            }
                        }
                        Meta(NameValue(m)) if m.path == GETTER => {
                            if let Ok(path) = parse_lit_into_expr_path(
                                cx,
                                GETTER,
                                &m.lit,
                            ) {
                                getter.set(&m.path, path);
                            }
                        }
                        Meta(Path(word)) if word == FLATTEN => {
                            flatten.set_true(word);
                        }
                        Meta(meta_item) => {
                            let path = meta_item
                                .path()
                                .into_token_stream()
                                .to_string()
                                .replace(' ', "");
                            cx.error_spanned_by(
                                meta_item.path(),
                                {
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &["unknown serde field attribute `", "`"],
                                            &[::core::fmt::ArgumentV1::new_display(&path)],
                                        ),
                                    );
                                    res
                                },
                            );
                        }
                        Lit(lit) => {
                            cx.error_spanned_by(
                                lit,
                                "unexpected literal in serde field attribute",
                            );
                        }
                    }
                }
                if let Default::None = *container_default {
                    if skip_deserializing.0.value.is_some() {
                        default.set_if_none(Default::Default);
                    }
                }
                let mut borrowed_lifetimes = borrowed_lifetimes
                    .get()
                    .unwrap_or_default();
                if !borrowed_lifetimes.is_empty() {
                    if is_cow(&field.ty, is_str) {
                        let mut path = syn::Path {
                            leading_colon: None,
                            segments: Punctuated::new(),
                        };
                        let span = Span::call_site();
                        path.segments.push(Ident::new("_serde", span).into());
                        path.segments.push(Ident::new("__private", span).into());
                        path.segments.push(Ident::new("de", span).into());
                        path.segments.push(Ident::new("borrow_cow_str", span).into());
                        let expr = syn::ExprPath {
                            attrs: Vec::new(),
                            qself: None,
                            path,
                        };
                        deserialize_with.set_if_none(expr);
                    } else if is_cow(&field.ty, is_slice_u8) {
                        let mut path = syn::Path {
                            leading_colon: None,
                            segments: Punctuated::new(),
                        };
                        let span = Span::call_site();
                        path.segments.push(Ident::new("_serde", span).into());
                        path.segments.push(Ident::new("__private", span).into());
                        path.segments.push(Ident::new("de", span).into());
                        path.segments.push(Ident::new("borrow_cow_bytes", span).into());
                        let expr = syn::ExprPath {
                            attrs: Vec::new(),
                            qself: None,
                            path,
                        };
                        deserialize_with.set_if_none(expr);
                    }
                } else if is_implicitly_borrowed(&field.ty) {
                    collect_lifetimes(&field.ty, &mut borrowed_lifetimes);
                }
                Field {
                    name: Name::from_attrs(ident, ser_name, de_name, Some(de_aliases)),
                    skip_serializing: skip_serializing.get(),
                    skip_deserializing: skip_deserializing.get(),
                    skip_serializing_if: skip_serializing_if.get(),
                    default: default.get().unwrap_or(Default::None),
                    serialize_with: serialize_with.get(),
                    deserialize_with: deserialize_with.get(),
                    ser_bound: ser_bound.get(),
                    de_bound: de_bound.get(),
                    borrowed_lifetimes,
                    getter: getter.get(),
                    flatten: flatten.get(),
                    transparent: false,
                }
            }
            pub fn name(&self) -> &Name {
                &self.name
            }
            pub fn aliases(&self) -> Vec<String> {
                self.name.deserialize_aliases()
            }
            pub fn rename_by_rules(&mut self, rules: &RenameAllRules) {
                if !self.name.serialize_renamed {
                    self.name.serialize = rules
                        .serialize
                        .apply_to_field(&self.name.serialize);
                }
                if !self.name.deserialize_renamed {
                    self.name.deserialize = rules
                        .deserialize
                        .apply_to_field(&self.name.deserialize);
                }
            }
            pub fn skip_serializing(&self) -> bool {
                self.skip_serializing
            }
            pub fn skip_deserializing(&self) -> bool {
                self.skip_deserializing
            }
            pub fn skip_serializing_if(&self) -> Option<&syn::ExprPath> {
                self.skip_serializing_if.as_ref()
            }
            pub fn default(&self) -> &Default {
                &self.default
            }
            pub fn serialize_with(&self) -> Option<&syn::ExprPath> {
                self.serialize_with.as_ref()
            }
            pub fn deserialize_with(&self) -> Option<&syn::ExprPath> {
                self.deserialize_with.as_ref()
            }
            pub fn ser_bound(&self) -> Option<&[syn::WherePredicate]> {
                self.ser_bound.as_ref().map(|vec| &vec[..])
            }
            pub fn de_bound(&self) -> Option<&[syn::WherePredicate]> {
                self.de_bound.as_ref().map(|vec| &vec[..])
            }
            pub fn borrowed_lifetimes(&self) -> &BTreeSet<syn::Lifetime> {
                &self.borrowed_lifetimes
            }
            pub fn getter(&self) -> Option<&syn::ExprPath> {
                self.getter.as_ref()
            }
            pub fn flatten(&self) -> bool {
                self.flatten
            }
            pub fn transparent(&self) -> bool {
                self.transparent
            }
            pub fn mark_transparent(&mut self) {
                self.transparent = true;
            }
        }
        type SerAndDe<T> = (Option<T>, Option<T>);
        fn get_ser_and_de<'a, 'b, T, F>(
            cx: &'b Ctxt,
            attr_name: Symbol,
            metas: &'a Punctuated<syn::NestedMeta, ::syn::token::Comma>,
            f: F,
        ) -> Result<(VecAttr<'b, T>, VecAttr<'b, T>), ()>
        where
            T: 'a,
            F: Fn(&Ctxt, Symbol, Symbol, &'a syn::Lit) -> Result<T, ()>,
        {
            let mut ser_meta = VecAttr::none(cx, attr_name);
            let mut de_meta = VecAttr::none(cx, attr_name);
            for meta in metas {
                match meta {
                    Meta(NameValue(meta)) if meta.path == SERIALIZE => {
                        if let Ok(v) = f(cx, attr_name, SERIALIZE, &meta.lit) {
                            ser_meta.insert(&meta.path, v);
                        }
                    }
                    Meta(NameValue(meta)) if meta.path == DESERIALIZE => {
                        if let Ok(v) = f(cx, attr_name, DESERIALIZE, &meta.lit) {
                            de_meta.insert(&meta.path, v);
                        }
                    }
                    _ => {
                        cx.error_spanned_by(
                            meta,
                            {
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1_formatted(
                                        &[
                                            "malformed ",
                                            " attribute, expected `",
                                            "(serialize = ..., deserialize = ...)`",
                                        ],
                                        &[::core::fmt::ArgumentV1::new_display(&attr_name)],
                                        &[
                                            ::core::fmt::rt::v1::Argument {
                                                position: 0usize,
                                                format: ::core::fmt::rt::v1::FormatSpec {
                                                    fill: ' ',
                                                    align: ::core::fmt::rt::v1::Alignment::Unknown,
                                                    flags: 0u32,
                                                    precision: ::core::fmt::rt::v1::Count::Implied,
                                                    width: ::core::fmt::rt::v1::Count::Implied,
                                                },
                                            },
                                            ::core::fmt::rt::v1::Argument {
                                                position: 0usize,
                                                format: ::core::fmt::rt::v1::FormatSpec {
                                                    fill: ' ',
                                                    align: ::core::fmt::rt::v1::Alignment::Unknown,
                                                    flags: 0u32,
                                                    precision: ::core::fmt::rt::v1::Count::Implied,
                                                    width: ::core::fmt::rt::v1::Count::Implied,
                                                },
                                            },
                                        ],
                                        unsafe { ::core::fmt::UnsafeArg::new() },
                                    ),
                                );
                                res
                            },
                        );
                        return Err(());
                    }
                }
            }
            Ok((ser_meta, de_meta))
        }
        fn get_renames<'a>(
            cx: &Ctxt,
            items: &'a Punctuated<syn::NestedMeta, ::syn::token::Comma>,
        ) -> Result<SerAndDe<&'a syn::LitStr>, ()> {
            let (ser, de) = get_ser_and_de(cx, RENAME, items, get_lit_str2)?;
            Ok((ser.at_most_one()?, de.at_most_one()?))
        }
        fn get_multiple_renames<'a>(
            cx: &Ctxt,
            items: &'a Punctuated<syn::NestedMeta, ::syn::token::Comma>,
        ) -> Result<(Option<&'a syn::LitStr>, Vec<&'a syn::LitStr>), ()> {
            let (ser, de) = get_ser_and_de(cx, RENAME, items, get_lit_str2)?;
            Ok((ser.at_most_one()?, de.get()))
        }
        fn get_where_predicates(
            cx: &Ctxt,
            items: &Punctuated<syn::NestedMeta, ::syn::token::Comma>,
        ) -> Result<SerAndDe<Vec<syn::WherePredicate>>, ()> {
            let (ser, de) = get_ser_and_de(cx, BOUND, items, parse_lit_into_where)?;
            Ok((ser.at_most_one()?, de.at_most_one()?))
        }
        pub fn get_serde_meta_items(
            cx: &Ctxt,
            attr: &syn::Attribute,
        ) -> Result<Vec<syn::NestedMeta>, ()> {
            if attr.path != SERDE {
                return Ok(Vec::new());
            }
            match attr.parse_meta() {
                Ok(List(meta)) => Ok(meta.nested.into_iter().collect()),
                Ok(other) => {
                    cx.error_spanned_by(other, "expected #[serde(...)]");
                    Err(())
                }
                Err(err) => {
                    cx.syn_error(err);
                    Err(())
                }
            }
        }
        fn get_lit_str<'a>(
            cx: &Ctxt,
            attr_name: Symbol,
            lit: &'a syn::Lit,
        ) -> Result<&'a syn::LitStr, ()> {
            get_lit_str2(cx, attr_name, attr_name, lit)
        }
        fn get_lit_str2<'a>(
            cx: &Ctxt,
            attr_name: Symbol,
            meta_item_name: Symbol,
            lit: &'a syn::Lit,
        ) -> Result<&'a syn::LitStr, ()> {
            if let syn::Lit::Str(lit) = lit {
                Ok(lit)
            } else {
                cx.error_spanned_by(
                    lit,
                    {
                        let res = ::alloc::fmt::format(
                            ::core::fmt::Arguments::new_v1(
                                &[
                                    "expected serde ",
                                    " attribute to be a string: `",
                                    " = \"...\"`",
                                ],
                                &[
                                    ::core::fmt::ArgumentV1::new_display(&attr_name),
                                    ::core::fmt::ArgumentV1::new_display(&meta_item_name),
                                ],
                            ),
                        );
                        res
                    },
                );
                Err(())
            }
        }
        fn parse_lit_into_path(
            cx: &Ctxt,
            attr_name: Symbol,
            lit: &syn::Lit,
        ) -> Result<syn::Path, ()> {
            let string = get_lit_str(cx, attr_name, lit)?;
            parse_lit_str(string)
                .map_err(|_| {
                    cx.error_spanned_by(
                        lit,
                        {
                            let res = ::alloc::fmt::format(
                                ::core::fmt::Arguments::new_v1(
                                    &["failed to parse path: "],
                                    &[::core::fmt::ArgumentV1::new_debug(&string.value())],
                                ),
                            );
                            res
                        },
                    );
                })
        }
        fn parse_lit_into_expr_path(
            cx: &Ctxt,
            attr_name: Symbol,
            lit: &syn::Lit,
        ) -> Result<syn::ExprPath, ()> {
            let string = get_lit_str(cx, attr_name, lit)?;
            parse_lit_str(string)
                .map_err(|_| {
                    cx.error_spanned_by(
                        lit,
                        {
                            let res = ::alloc::fmt::format(
                                ::core::fmt::Arguments::new_v1(
                                    &["failed to parse path: "],
                                    &[::core::fmt::ArgumentV1::new_debug(&string.value())],
                                ),
                            );
                            res
                        },
                    );
                })
        }
        fn parse_lit_into_where(
            cx: &Ctxt,
            attr_name: Symbol,
            meta_item_name: Symbol,
            lit: &syn::Lit,
        ) -> Result<Vec<syn::WherePredicate>, ()> {
            let string = get_lit_str2(cx, attr_name, meta_item_name, lit)?;
            if string.value().is_empty() {
                return Ok(Vec::new());
            }
            let where_string = syn::LitStr::new(
                &{
                    let res = ::alloc::fmt::format(
                        ::core::fmt::Arguments::new_v1(
                            &["where "],
                            &[::core::fmt::ArgumentV1::new_display(&string.value())],
                        ),
                    );
                    res
                },
                string.span(),
            );
            parse_lit_str::<syn::WhereClause>(&where_string)
                .map(|wh| wh.predicates.into_iter().collect())
                .map_err(|err| cx.error_spanned_by(lit, err))
        }
        fn parse_lit_into_ty(
            cx: &Ctxt,
            attr_name: Symbol,
            lit: &syn::Lit,
        ) -> Result<syn::Type, ()> {
            let string = get_lit_str(cx, attr_name, lit)?;
            parse_lit_str(string)
                .map_err(|_| {
                    cx.error_spanned_by(
                        lit,
                        {
                            let res = ::alloc::fmt::format(
                                ::core::fmt::Arguments::new_v1(
                                    &["failed to parse type: ", " = "],
                                    &[
                                        ::core::fmt::ArgumentV1::new_display(&attr_name),
                                        ::core::fmt::ArgumentV1::new_debug(&string.value()),
                                    ],
                                ),
                            );
                            res
                        },
                    );
                })
        }
        fn parse_lit_into_lifetimes(
            cx: &Ctxt,
            attr_name: Symbol,
            lit: &syn::Lit,
        ) -> Result<BTreeSet<syn::Lifetime>, ()> {
            let string = get_lit_str(cx, attr_name, lit)?;
            if string.value().is_empty() {
                cx.error_spanned_by(lit, "at least one lifetime must be borrowed");
                return Err(());
            }
            struct BorrowedLifetimes(Punctuated<syn::Lifetime, ::syn::token::Add>);
            impl Parse for BorrowedLifetimes {
                fn parse(input: ParseStream) -> parse::Result<Self> {
                    Punctuated::parse_separated_nonempty(input).map(BorrowedLifetimes)
                }
            }
            if let Ok(BorrowedLifetimes(lifetimes)) = parse_lit_str(string) {
                let mut set = BTreeSet::new();
                for lifetime in lifetimes {
                    if !set.insert(lifetime.clone()) {
                        cx.error_spanned_by(
                            lit,
                            {
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1(
                                        &["duplicate borrowed lifetime `", "`"],
                                        &[::core::fmt::ArgumentV1::new_display(&lifetime)],
                                    ),
                                );
                                res
                            },
                        );
                    }
                }
                return Ok(set);
            }
            cx.error_spanned_by(
                lit,
                {
                    let res = ::alloc::fmt::format(
                        ::core::fmt::Arguments::new_v1(
                            &["failed to parse borrowed lifetimes: "],
                            &[::core::fmt::ArgumentV1::new_debug(&string.value())],
                        ),
                    );
                    res
                },
            );
            Err(())
        }
        fn is_implicitly_borrowed(ty: &syn::Type) -> bool {
            is_implicitly_borrowed_reference(ty)
                || is_option(ty, is_implicitly_borrowed_reference)
        }
        fn is_implicitly_borrowed_reference(ty: &syn::Type) -> bool {
            is_reference(ty, is_str) || is_reference(ty, is_slice_u8)
        }
        fn is_cow(ty: &syn::Type, elem: fn(&syn::Type) -> bool) -> bool {
            let path = match ungroup(ty) {
                syn::Type::Path(ty) => &ty.path,
                _ => {
                    return false;
                }
            };
            let seg = match path.segments.last() {
                Some(seg) => seg,
                None => {
                    return false;
                }
            };
            let args = match &seg.arguments {
                syn::PathArguments::AngleBracketed(bracketed) => &bracketed.args,
                _ => {
                    return false;
                }
            };
            seg.ident == "Cow" && args.len() == 2
                && match (&args[0], &args[1]) {
                    (
                        syn::GenericArgument::Lifetime(_),
                        syn::GenericArgument::Type(arg),
                    ) => elem(arg),
                    _ => false,
                }
        }
        fn is_option(ty: &syn::Type, elem: fn(&syn::Type) -> bool) -> bool {
            let path = match ungroup(ty) {
                syn::Type::Path(ty) => &ty.path,
                _ => {
                    return false;
                }
            };
            let seg = match path.segments.last() {
                Some(seg) => seg,
                None => {
                    return false;
                }
            };
            let args = match &seg.arguments {
                syn::PathArguments::AngleBracketed(bracketed) => &bracketed.args,
                _ => {
                    return false;
                }
            };
            seg.ident == "Option" && args.len() == 1
                && match &args[0] {
                    syn::GenericArgument::Type(arg) => elem(arg),
                    _ => false,
                }
        }
        fn is_reference(ty: &syn::Type, elem: fn(&syn::Type) -> bool) -> bool {
            match ungroup(ty) {
                syn::Type::Reference(ty) => ty.mutability.is_none() && elem(&ty.elem),
                _ => false,
            }
        }
        fn is_str(ty: &syn::Type) -> bool {
            is_primitive_type(ty, "str")
        }
        fn is_slice_u8(ty: &syn::Type) -> bool {
            match ungroup(ty) {
                syn::Type::Slice(ty) => is_primitive_type(&ty.elem, "u8"),
                _ => false,
            }
        }
        fn is_primitive_type(ty: &syn::Type, primitive: &str) -> bool {
            match ungroup(ty) {
                syn::Type::Path(ty) => {
                    ty.qself.is_none() && is_primitive_path(&ty.path, primitive)
                }
                _ => false,
            }
        }
        fn is_primitive_path(path: &syn::Path, primitive: &str) -> bool {
            path.leading_colon.is_none() && path.segments.len() == 1
                && path.segments[0].ident == primitive
                && path.segments[0].arguments.is_empty()
        }
        fn borrowable_lifetimes(
            cx: &Ctxt,
            name: &str,
            field: &syn::Field,
        ) -> Result<BTreeSet<syn::Lifetime>, ()> {
            let mut lifetimes = BTreeSet::new();
            collect_lifetimes(&field.ty, &mut lifetimes);
            if lifetimes.is_empty() {
                cx.error_spanned_by(
                    field,
                    {
                        let res = ::alloc::fmt::format(
                            ::core::fmt::Arguments::new_v1(
                                &["field `", "` has no lifetimes to borrow"],
                                &[::core::fmt::ArgumentV1::new_display(&name)],
                            ),
                        );
                        res
                    },
                );
                Err(())
            } else {
                Ok(lifetimes)
            }
        }
        fn collect_lifetimes(ty: &syn::Type, out: &mut BTreeSet<syn::Lifetime>) {
            match ty {
                syn::Type::Slice(ty) => {
                    collect_lifetimes(&ty.elem, out);
                }
                syn::Type::Array(ty) => {
                    collect_lifetimes(&ty.elem, out);
                }
                syn::Type::Ptr(ty) => {
                    collect_lifetimes(&ty.elem, out);
                }
                syn::Type::Reference(ty) => {
                    out.extend(ty.lifetime.iter().cloned());
                    collect_lifetimes(&ty.elem, out);
                }
                syn::Type::Tuple(ty) => {
                    for elem in &ty.elems {
                        collect_lifetimes(elem, out);
                    }
                }
                syn::Type::Path(ty) => {
                    if let Some(qself) = &ty.qself {
                        collect_lifetimes(&qself.ty, out);
                    }
                    for seg in &ty.path.segments {
                        if let syn::PathArguments::AngleBracketed(bracketed) = &seg
                            .arguments
                        {
                            for arg in &bracketed.args {
                                match arg {
                                    syn::GenericArgument::Lifetime(lifetime) => {
                                        out.insert(lifetime.clone());
                                    }
                                    syn::GenericArgument::Type(ty) => {
                                        collect_lifetimes(ty, out);
                                    }
                                    syn::GenericArgument::Binding(binding) => {
                                        collect_lifetimes(&binding.ty, out);
                                    }
                                    syn::GenericArgument::Constraint(_)
                                    | syn::GenericArgument::Const(_) => {}
                                }
                            }
                        }
                    }
                }
                syn::Type::Paren(ty) => {
                    collect_lifetimes(&ty.elem, out);
                }
                syn::Type::Group(ty) => {
                    collect_lifetimes(&ty.elem, out);
                }
                syn::Type::Macro(ty) => {
                    collect_lifetimes_from_tokens(ty.mac.tokens.clone(), out);
                }
                syn::Type::BareFn(_)
                | syn::Type::Never(_)
                | syn::Type::TraitObject(_)
                | syn::Type::ImplTrait(_)
                | syn::Type::Infer(_)
                | syn::Type::Verbatim(_) => {}
                _ => {}
            }
        }
        fn collect_lifetimes_from_tokens(
            tokens: TokenStream,
            out: &mut BTreeSet<syn::Lifetime>,
        ) {
            let mut iter = tokens.into_iter();
            while let Some(tt) = iter.next() {
                match &tt {
                    TokenTree::Punct(
                        op,
                    ) if op.as_char() == '\'' && op.spacing() == Spacing::Joint => {
                        if let Some(TokenTree::Ident(ident)) = iter.next() {
                            out.insert(syn::Lifetime {
                                apostrophe: op.span(),
                                ident,
                            });
                        }
                    }
                    TokenTree::Group(group) => {
                        let tokens = group.stream();
                        collect_lifetimes_from_tokens(tokens, out);
                    }
                    _ => {}
                }
            }
        }
        fn parse_lit_str<T>(s: &syn::LitStr) -> parse::Result<T>
        where
            T: Parse,
        {
            let tokens = spanned_tokens(s)?;
            syn::parse2(tokens)
        }
        fn spanned_tokens(s: &syn::LitStr) -> parse::Result<TokenStream> {
            let stream = syn::parse_str(&s.value())?;
            Ok(respan(stream, s.span()))
        }
    }
    mod ctxt {
        use quote::ToTokens;
        use std::cell::RefCell;
        use std::fmt::Display;
        use std::thread;
        use syn;
        pub struct Ctxt {
            errors: RefCell<Option<Vec<syn::Error>>>,
        }
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::default::Default for Ctxt {
            #[inline]
            fn default() -> Ctxt {
                Ctxt {
                    errors: ::core::default::Default::default(),
                }
            }
        }
        impl Ctxt {
            pub fn new() -> Self {
                Ctxt {
                    errors: RefCell::new(Some(Vec::new())),
                }
            }
            pub fn error_spanned_by<A: ToTokens, T: Display>(&self, obj: A, msg: T) {
                self.errors
                    .borrow_mut()
                    .as_mut()
                    .unwrap()
                    .push(syn::Error::new_spanned(obj.into_token_stream(), msg));
            }
            pub fn syn_error(&self, err: syn::Error) {
                self.errors.borrow_mut().as_mut().unwrap().push(err);
            }
            pub fn check(self) -> Result<(), Vec<syn::Error>> {
                let errors = self.errors.borrow_mut().take().unwrap();
                match errors.len() {
                    0 => Ok(()),
                    _ => Err(errors),
                }
            }
        }
        impl Drop for Ctxt {
            fn drop(&mut self) {
                if !thread::panicking() && self.errors.borrow().is_some() {
                    { ::std::rt::begin_panic("forgot to check for errors") };
                }
            }
        }
    }
    pub use self::ctxt::Ctxt;
    mod receiver {
        use internals::respan::respan;
        use proc_macro2::Span;
        use quote::ToTokens;
        use std::mem;
        use syn::punctuated::Punctuated;
        use syn::{
            parse_quote, Data, DeriveInput, Expr, ExprPath, GenericArgument,
            GenericParam, Generics, Macro, Path, PathArguments, QSelf, ReturnType, Type,
            TypeParamBound, TypePath, WherePredicate,
        };
        pub fn replace_receiver(input: &mut DeriveInput) {
            let self_ty = {
                let ident = &input.ident;
                let ty_generics = input.generics.split_for_impl().1;
                ::syn::parse_quote::parse({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&ident, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    _s
                })
            };
            let mut visitor = ReplaceReceiver(&self_ty);
            visitor.visit_generics_mut(&mut input.generics);
            visitor.visit_data_mut(&mut input.data);
        }
        struct ReplaceReceiver<'a>(&'a TypePath);
        impl ReplaceReceiver<'_> {
            fn self_ty(&self, span: Span) -> TypePath {
                let tokens = self.0.to_token_stream();
                let respanned = respan(tokens, span);
                syn::parse2(respanned).unwrap()
            }
            fn self_to_qself(&self, qself: &mut Option<QSelf>, path: &mut Path) {
                if path.leading_colon.is_some() || path.segments[0].ident != "Self" {
                    return;
                }
                if path.segments.len() == 1 {
                    self.self_to_expr_path(path);
                    return;
                }
                let span = path.segments[0].ident.span();
                *qself = Some(QSelf {
                    lt_token: ::syn::token::Lt(span),
                    ty: Box::new(Type::Path(self.self_ty(span))),
                    position: 0,
                    as_token: None,
                    gt_token: ::syn::token::Gt(span),
                });
                path.leading_colon = Some(
                    **path.segments.pairs().next().unwrap().punct().unwrap(),
                );
                let segments = mem::replace(&mut path.segments, Punctuated::new());
                path.segments = segments.into_pairs().skip(1).collect();
            }
            fn self_to_expr_path(&self, path: &mut Path) {
                let self_ty = self.self_ty(path.segments[0].ident.span());
                let variant = mem::replace(path, self_ty.path);
                for segment in &mut path.segments {
                    if let PathArguments::AngleBracketed(bracketed) = &mut segment
                        .arguments
                    {
                        if bracketed.colon2_token.is_none() && !bracketed.args.is_empty()
                        {
                            bracketed.colon2_token = Some(
                                <::syn::token::Colon2>::default(),
                            );
                        }
                    }
                }
                if variant.segments.len() > 1 {
                    path.segments.push_punct(<::syn::token::Colon2>::default());
                    path.segments.extend(variant.segments.into_pairs().skip(1));
                }
            }
        }
        impl ReplaceReceiver<'_> {
            fn visit_type_mut(&mut self, ty: &mut Type) {
                let span = if let Type::Path(node) = ty {
                    if node.qself.is_none() && node.path.is_ident("Self") {
                        node.path.segments[0].ident.span()
                    } else {
                        self.visit_type_path_mut(node);
                        return;
                    }
                } else {
                    self.visit_type_mut_impl(ty);
                    return;
                };
                *ty = self.self_ty(span).into();
            }
            fn visit_type_path_mut(&mut self, ty: &mut TypePath) {
                if ty.qself.is_none() {
                    self.self_to_qself(&mut ty.qself, &mut ty.path);
                }
                self.visit_type_path_mut_impl(ty);
            }
            fn visit_expr_path_mut(&mut self, expr: &mut ExprPath) {
                if expr.qself.is_none() {
                    self.self_to_qself(&mut expr.qself, &mut expr.path);
                }
                self.visit_expr_path_mut_impl(expr);
            }
            fn visit_type_mut_impl(&mut self, ty: &mut Type) {
                match ty {
                    Type::Array(ty) => {
                        self.visit_type_mut(&mut ty.elem);
                        self.visit_expr_mut(&mut ty.len);
                    }
                    Type::BareFn(ty) => {
                        for arg in &mut ty.inputs {
                            self.visit_type_mut(&mut arg.ty);
                        }
                        self.visit_return_type_mut(&mut ty.output);
                    }
                    Type::Group(ty) => self.visit_type_mut(&mut ty.elem),
                    Type::ImplTrait(ty) => {
                        for bound in &mut ty.bounds {
                            self.visit_type_param_bound_mut(bound);
                        }
                    }
                    Type::Macro(ty) => self.visit_macro_mut(&mut ty.mac),
                    Type::Paren(ty) => self.visit_type_mut(&mut ty.elem),
                    Type::Path(ty) => {
                        if let Some(qself) = &mut ty.qself {
                            self.visit_type_mut(&mut qself.ty);
                        }
                        self.visit_path_mut(&mut ty.path);
                    }
                    Type::Ptr(ty) => self.visit_type_mut(&mut ty.elem),
                    Type::Reference(ty) => self.visit_type_mut(&mut ty.elem),
                    Type::Slice(ty) => self.visit_type_mut(&mut ty.elem),
                    Type::TraitObject(ty) => {
                        for bound in &mut ty.bounds {
                            self.visit_type_param_bound_mut(bound);
                        }
                    }
                    Type::Tuple(ty) => {
                        for elem in &mut ty.elems {
                            self.visit_type_mut(elem);
                        }
                    }
                    Type::Infer(_) | Type::Never(_) | Type::Verbatim(_) => {}
                    _ => {}
                }
            }
            fn visit_type_path_mut_impl(&mut self, ty: &mut TypePath) {
                if let Some(qself) = &mut ty.qself {
                    self.visit_type_mut(&mut qself.ty);
                }
                self.visit_path_mut(&mut ty.path);
            }
            fn visit_expr_path_mut_impl(&mut self, expr: &mut ExprPath) {
                if let Some(qself) = &mut expr.qself {
                    self.visit_type_mut(&mut qself.ty);
                }
                self.visit_path_mut(&mut expr.path);
            }
            fn visit_path_mut(&mut self, path: &mut Path) {
                for segment in &mut path.segments {
                    self.visit_path_arguments_mut(&mut segment.arguments);
                }
            }
            fn visit_path_arguments_mut(&mut self, arguments: &mut PathArguments) {
                match arguments {
                    PathArguments::None => {}
                    PathArguments::AngleBracketed(arguments) => {
                        for arg in &mut arguments.args {
                            match arg {
                                GenericArgument::Type(arg) => self.visit_type_mut(arg),
                                GenericArgument::Binding(arg) => {
                                    self.visit_type_mut(&mut arg.ty)
                                }
                                GenericArgument::Lifetime(_)
                                | GenericArgument::Constraint(_)
                                | GenericArgument::Const(_) => {}
                            }
                        }
                    }
                    PathArguments::Parenthesized(arguments) => {
                        for argument in &mut arguments.inputs {
                            self.visit_type_mut(argument);
                        }
                        self.visit_return_type_mut(&mut arguments.output);
                    }
                }
            }
            fn visit_return_type_mut(&mut self, return_type: &mut ReturnType) {
                match return_type {
                    ReturnType::Default => {}
                    ReturnType::Type(_, output) => self.visit_type_mut(output),
                }
            }
            fn visit_type_param_bound_mut(&mut self, bound: &mut TypeParamBound) {
                match bound {
                    TypeParamBound::Trait(bound) => self.visit_path_mut(&mut bound.path),
                    TypeParamBound::Lifetime(_) => {}
                }
            }
            fn visit_generics_mut(&mut self, generics: &mut Generics) {
                for param in &mut generics.params {
                    match param {
                        GenericParam::Type(param) => {
                            for bound in &mut param.bounds {
                                self.visit_type_param_bound_mut(bound);
                            }
                        }
                        GenericParam::Lifetime(_) | GenericParam::Const(_) => {}
                    }
                }
                if let Some(where_clause) = &mut generics.where_clause {
                    for predicate in &mut where_clause.predicates {
                        match predicate {
                            WherePredicate::Type(predicate) => {
                                self.visit_type_mut(&mut predicate.bounded_ty);
                                for bound in &mut predicate.bounds {
                                    self.visit_type_param_bound_mut(bound);
                                }
                            }
                            WherePredicate::Lifetime(_) | WherePredicate::Eq(_) => {}
                        }
                    }
                }
            }
            fn visit_data_mut(&mut self, data: &mut Data) {
                match data {
                    Data::Struct(data) => {
                        for field in &mut data.fields {
                            self.visit_type_mut(&mut field.ty);
                        }
                    }
                    Data::Enum(data) => {
                        for variant in &mut data.variants {
                            for field in &mut variant.fields {
                                self.visit_type_mut(&mut field.ty);
                            }
                        }
                    }
                    Data::Union(_) => {}
                }
            }
            fn visit_expr_mut(&mut self, expr: &mut Expr) {
                match expr {
                    Expr::Binary(expr) => {
                        self.visit_expr_mut(&mut expr.left);
                        self.visit_expr_mut(&mut expr.right);
                    }
                    Expr::Call(expr) => {
                        self.visit_expr_mut(&mut expr.func);
                        for arg in &mut expr.args {
                            self.visit_expr_mut(arg);
                        }
                    }
                    Expr::Cast(expr) => {
                        self.visit_expr_mut(&mut expr.expr);
                        self.visit_type_mut(&mut expr.ty);
                    }
                    Expr::Field(expr) => self.visit_expr_mut(&mut expr.base),
                    Expr::Index(expr) => {
                        self.visit_expr_mut(&mut expr.expr);
                        self.visit_expr_mut(&mut expr.index);
                    }
                    Expr::Paren(expr) => self.visit_expr_mut(&mut expr.expr),
                    Expr::Path(expr) => self.visit_expr_path_mut(expr),
                    Expr::Unary(expr) => self.visit_expr_mut(&mut expr.expr),
                    _ => {}
                }
            }
            fn visit_macro_mut(&mut self, _mac: &mut Macro) {}
        }
    }
    pub use self::receiver::replace_receiver;
    mod case {
        #[allow(deprecated, unused_imports)]
        use std::ascii::AsciiExt;
        use std::fmt::{self, Debug, Display};
        use self::RenameRule::*;
        pub enum RenameRule {
            None,
            LowerCase,
            UpperCase,
            PascalCase,
            CamelCase,
            SnakeCase,
            ScreamingSnakeCase,
            KebabCase,
            ScreamingKebabCase,
        }
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::marker::Copy for RenameRule {}
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::clone::Clone for RenameRule {
            #[inline]
            fn clone(&self) -> RenameRule {
                { *self }
            }
        }
        impl ::core::marker::StructuralPartialEq for RenameRule {}
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::cmp::PartialEq for RenameRule {
            #[inline]
            fn eq(&self, other: &RenameRule) -> bool {
                {
                    let __self_vi = ::core::intrinsics::discriminant_value(&*self);
                    let __arg_1_vi = ::core::intrinsics::discriminant_value(&*other);
                    if true && __self_vi == __arg_1_vi {
                        match (&*self, &*other) {
                            _ => true,
                        }
                    } else {
                        false
                    }
                }
            }
        }
        static RENAME_RULES: &[(&str, RenameRule)] = &[
            ("lowercase", LowerCase),
            ("UPPERCASE", UpperCase),
            ("PascalCase", PascalCase),
            ("camelCase", CamelCase),
            ("snake_case", SnakeCase),
            ("SCREAMING_SNAKE_CASE", ScreamingSnakeCase),
            ("kebab-case", KebabCase),
            ("SCREAMING-KEBAB-CASE", ScreamingKebabCase),
        ];
        impl RenameRule {
            pub fn from_str(rename_all_str: &str) -> Result<Self, ParseError> {
                for (name, rule) in RENAME_RULES {
                    if rename_all_str == *name {
                        return Ok(*rule);
                    }
                }
                Err(ParseError {
                    unknown: rename_all_str,
                })
            }
            pub fn apply_to_variant(&self, variant: &str) -> String {
                match *self {
                    None | PascalCase => variant.to_owned(),
                    LowerCase => variant.to_ascii_lowercase(),
                    UpperCase => variant.to_ascii_uppercase(),
                    CamelCase => variant[..1].to_ascii_lowercase() + &variant[1..],
                    SnakeCase => {
                        let mut snake = String::new();
                        for (i, ch) in variant.char_indices() {
                            if i > 0 && ch.is_uppercase() {
                                snake.push('_');
                            }
                            snake.push(ch.to_ascii_lowercase());
                        }
                        snake
                    }
                    ScreamingSnakeCase => {
                        SnakeCase.apply_to_variant(variant).to_ascii_uppercase()
                    }
                    KebabCase => SnakeCase.apply_to_variant(variant).replace('_', "-"),
                    ScreamingKebabCase => {
                        ScreamingSnakeCase.apply_to_variant(variant).replace('_', "-")
                    }
                }
            }
            pub fn apply_to_field(&self, field: &str) -> String {
                match *self {
                    None | LowerCase | SnakeCase => field.to_owned(),
                    UpperCase => field.to_ascii_uppercase(),
                    PascalCase => {
                        let mut pascal = String::new();
                        let mut capitalize = true;
                        for ch in field.chars() {
                            if ch == '_' {
                                capitalize = true;
                            } else if capitalize {
                                pascal.push(ch.to_ascii_uppercase());
                                capitalize = false;
                            } else {
                                pascal.push(ch);
                            }
                        }
                        pascal
                    }
                    CamelCase => {
                        let pascal = PascalCase.apply_to_field(field);
                        pascal[..1].to_ascii_lowercase() + &pascal[1..]
                    }
                    ScreamingSnakeCase => field.to_ascii_uppercase(),
                    KebabCase => field.replace('_', "-"),
                    ScreamingKebabCase => {
                        ScreamingSnakeCase.apply_to_field(field).replace('_', "-")
                    }
                }
            }
        }
        pub struct ParseError<'a> {
            unknown: &'a str,
        }
        impl<'a> Display for ParseError<'a> {
            fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
                f.write_str("unknown rename rule `rename_all = ")?;
                Debug::fmt(self.unknown, f)?;
                f.write_str("`, expected one of ")?;
                for (i, (name, _rule)) in RENAME_RULES.iter().enumerate() {
                    if i > 0 {
                        f.write_str(", ")?;
                    }
                    Debug::fmt(name, f)?;
                }
                Ok(())
            }
        }
    }
    mod check {
        use internals::ast::{Container, Data, Field, Style};
        use internals::attr::{Identifier, TagType};
        use internals::{ungroup, Ctxt, Derive};
        use syn::{Member, Type};
        pub fn check(cx: &Ctxt, cont: &mut Container, derive: Derive) {
            check_getter(cx, cont);
            check_flatten(cx, cont);
            check_identifier(cx, cont);
            check_variant_skip_attrs(cx, cont);
            check_internal_tag_field_name_conflict(cx, cont);
            check_adjacent_tag_conflict(cx, cont);
            check_transparent(cx, cont, derive);
            check_from_and_try_from(cx, cont);
        }
        fn check_getter(cx: &Ctxt, cont: &Container) {
            match cont.data {
                Data::Enum(_) => {
                    if cont.data.has_getter() {
                        cx.error_spanned_by(
                            cont.original,
                            "#[serde(getter = \"...\")] is not allowed in an enum",
                        );
                    }
                }
                Data::Struct(_, _) => {
                    if cont.data.has_getter() && cont.attrs.remote().is_none() {
                        cx.error_spanned_by(
                            cont.original,
                            "#[serde(getter = \"...\")] can only be used in structs that have #[serde(remote = \"...\")]",
                        );
                    }
                }
            }
        }
        fn check_flatten(cx: &Ctxt, cont: &Container) {
            match &cont.data {
                Data::Enum(variants) => {
                    for variant in variants {
                        for field in &variant.fields {
                            check_flatten_field(cx, variant.style, field);
                        }
                    }
                }
                Data::Struct(style, fields) => {
                    for field in fields {
                        check_flatten_field(cx, *style, field);
                    }
                }
            }
        }
        fn check_flatten_field(cx: &Ctxt, style: Style, field: &Field) {
            if !field.attrs.flatten() {
                return;
            }
            match style {
                Style::Tuple => {
                    cx.error_spanned_by(
                        field.original,
                        "#[serde(flatten)] cannot be used on tuple structs",
                    );
                }
                Style::Newtype => {
                    cx.error_spanned_by(
                        field.original,
                        "#[serde(flatten)] cannot be used on newtype structs",
                    );
                }
                _ => {}
            }
        }
        fn check_identifier(cx: &Ctxt, cont: &Container) {
            let variants = match &cont.data {
                Data::Enum(variants) => variants,
                Data::Struct(_, _) => {
                    return;
                }
            };
            for (i, variant) in variants.iter().enumerate() {
                match (
                    variant.style,
                    cont.attrs.identifier(),
                    variant.attrs.other(),
                    cont.attrs.tag(),
                ) {
                    (_, Identifier::Variant, true, _) => {
                        cx.error_spanned_by(
                            variant.original,
                            "#[serde(other)] may not be used on a variant identifier",
                        );
                    }
                    (_, Identifier::No, true, &TagType::None) => {
                        cx.error_spanned_by(
                            variant.original,
                            "#[serde(other)] cannot appear on untagged enum",
                        );
                    }
                    (Style::Unit, Identifier::Field, true, _)
                    | (Style::Unit, Identifier::No, true, _) => {
                        if i < variants.len() - 1 {
                            cx.error_spanned_by(
                                variant.original,
                                "#[serde(other)] must be on the last variant",
                            );
                        }
                    }
                    (_, Identifier::Field, true, _) | (_, Identifier::No, true, _) => {
                        cx.error_spanned_by(
                            variant.original,
                            "#[serde(other)] must be on a unit variant",
                        );
                    }
                    (_, Identifier::No, false, _) => {}
                    (Style::Unit, _, false, _) => {}
                    (Style::Newtype, Identifier::Field, false, _) => {
                        if i < variants.len() - 1 {
                            cx.error_spanned_by(
                                variant.original,
                                {
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &["`", "` must be the last variant"],
                                            &[::core::fmt::ArgumentV1::new_display(&variant.ident)],
                                        ),
                                    );
                                    res
                                },
                            );
                        }
                    }
                    (_, Identifier::Field, false, _) => {
                        cx.error_spanned_by(
                            variant.original,
                            "#[serde(field_identifier)] may only contain unit variants",
                        );
                    }
                    (_, Identifier::Variant, false, _) => {
                        cx.error_spanned_by(
                            variant.original,
                            "#[serde(variant_identifier)] may only contain unit variants",
                        );
                    }
                }
            }
        }
        fn check_variant_skip_attrs(cx: &Ctxt, cont: &Container) {
            let variants = match &cont.data {
                Data::Enum(variants) => variants,
                Data::Struct(_, _) => {
                    return;
                }
            };
            for variant in variants.iter() {
                if variant.attrs.serialize_with().is_some() {
                    if variant.attrs.skip_serializing() {
                        cx.error_spanned_by(
                            variant.original,
                            {
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1(
                                        &[
                                            "variant `",
                                            "` cannot have both #[serde(serialize_with)] and #[serde(skip_serializing)]",
                                        ],
                                        &[::core::fmt::ArgumentV1::new_display(&variant.ident)],
                                    ),
                                );
                                res
                            },
                        );
                    }
                    for field in &variant.fields {
                        let member = member_message(&field.member);
                        if field.attrs.skip_serializing() {
                            cx.error_spanned_by(
                                variant.original,
                                {
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &[
                                                "variant `",
                                                "` cannot have both #[serde(serialize_with)] and a field ",
                                                " marked with #[serde(skip_serializing)]",
                                            ],
                                            &[
                                                ::core::fmt::ArgumentV1::new_display(&variant.ident),
                                                ::core::fmt::ArgumentV1::new_display(&member),
                                            ],
                                        ),
                                    );
                                    res
                                },
                            );
                        }
                        if field.attrs.skip_serializing_if().is_some() {
                            cx.error_spanned_by(
                                variant.original,
                                {
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &[
                                                "variant `",
                                                "` cannot have both #[serde(serialize_with)] and a field ",
                                                " marked with #[serde(skip_serializing_if)]",
                                            ],
                                            &[
                                                ::core::fmt::ArgumentV1::new_display(&variant.ident),
                                                ::core::fmt::ArgumentV1::new_display(&member),
                                            ],
                                        ),
                                    );
                                    res
                                },
                            );
                        }
                    }
                }
                if variant.attrs.deserialize_with().is_some() {
                    if variant.attrs.skip_deserializing() {
                        cx.error_spanned_by(
                            variant.original,
                            {
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1(
                                        &[
                                            "variant `",
                                            "` cannot have both #[serde(deserialize_with)] and #[serde(skip_deserializing)]",
                                        ],
                                        &[::core::fmt::ArgumentV1::new_display(&variant.ident)],
                                    ),
                                );
                                res
                            },
                        );
                    }
                    for field in &variant.fields {
                        if field.attrs.skip_deserializing() {
                            let member = member_message(&field.member);
                            cx.error_spanned_by(
                                variant.original,
                                {
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &[
                                                "variant `",
                                                "` cannot have both #[serde(deserialize_with)] and a field ",
                                                " marked with #[serde(skip_deserializing)]",
                                            ],
                                            &[
                                                ::core::fmt::ArgumentV1::new_display(&variant.ident),
                                                ::core::fmt::ArgumentV1::new_display(&member),
                                            ],
                                        ),
                                    );
                                    res
                                },
                            );
                        }
                    }
                }
            }
        }
        fn check_internal_tag_field_name_conflict(cx: &Ctxt, cont: &Container) {
            let variants = match &cont.data {
                Data::Enum(variants) => variants,
                Data::Struct(_, _) => return,
            };
            let tag = match cont.attrs.tag() {
                TagType::Internal { tag } => tag.as_str(),
                TagType::External | TagType::Adjacent { .. } | TagType::None => return,
            };
            let diagnose_conflict = || {
                cx.error_spanned_by(
                    cont.original,
                    {
                        let res = ::alloc::fmt::format(
                            ::core::fmt::Arguments::new_v1(
                                &["variant field name `", "` conflicts with internal tag"],
                                &[::core::fmt::ArgumentV1::new_display(&tag)],
                            ),
                        );
                        res
                    },
                );
            };
            for variant in variants {
                match variant.style {
                    Style::Struct => {
                        for field in &variant.fields {
                            let check_ser = !field.attrs.skip_serializing();
                            let check_de = !field.attrs.skip_deserializing();
                            let name = field.attrs.name();
                            let ser_name = name.serialize_name();
                            if check_ser && ser_name == tag {
                                diagnose_conflict();
                                return;
                            }
                            for de_name in field.attrs.aliases() {
                                if check_de && de_name == tag {
                                    diagnose_conflict();
                                    return;
                                }
                            }
                        }
                    }
                    Style::Unit | Style::Newtype | Style::Tuple => {}
                }
            }
        }
        fn check_adjacent_tag_conflict(cx: &Ctxt, cont: &Container) {
            let (type_tag, content_tag) = match cont.attrs.tag() {
                TagType::Adjacent { tag, content } => (tag, content),
                TagType::Internal { .. } | TagType::External | TagType::None => return,
            };
            if type_tag == content_tag {
                cx.error_spanned_by(
                    cont.original,
                    {
                        let res = ::alloc::fmt::format(
                            ::core::fmt::Arguments::new_v1(
                                &[
                                    "enum tags `",
                                    "` for type and content conflict with each other",
                                ],
                                &[::core::fmt::ArgumentV1::new_display(&type_tag)],
                            ),
                        );
                        res
                    },
                );
            }
        }
        fn check_transparent(cx: &Ctxt, cont: &mut Container, derive: Derive) {
            if !cont.attrs.transparent() {
                return;
            }
            if cont.attrs.type_from().is_some() {
                cx.error_spanned_by(
                    cont.original,
                    "#[serde(transparent)] is not allowed with #[serde(from = \"...\")]",
                );
            }
            if cont.attrs.type_try_from().is_some() {
                cx.error_spanned_by(
                    cont.original,
                    "#[serde(transparent)] is not allowed with #[serde(try_from = \"...\")]",
                );
            }
            if cont.attrs.type_into().is_some() {
                cx.error_spanned_by(
                    cont.original,
                    "#[serde(transparent)] is not allowed with #[serde(into = \"...\")]",
                );
            }
            let fields = match &mut cont.data {
                Data::Enum(_) => {
                    cx.error_spanned_by(
                        cont.original,
                        "#[serde(transparent)] is not allowed on an enum",
                    );
                    return;
                }
                Data::Struct(Style::Unit, _) => {
                    cx.error_spanned_by(
                        cont.original,
                        "#[serde(transparent)] is not allowed on a unit struct",
                    );
                    return;
                }
                Data::Struct(_, fields) => fields,
            };
            let mut transparent_field = None;
            for field in fields {
                if allow_transparent(field, derive) {
                    if transparent_field.is_some() {
                        cx.error_spanned_by(
                            cont.original,
                            "#[serde(transparent)] requires struct to have at most one transparent field",
                        );
                        return;
                    }
                    transparent_field = Some(field);
                }
            }
            match transparent_field {
                Some(transparent_field) => transparent_field.attrs.mark_transparent(),
                None => {
                    match derive {
                        Derive::Serialize => {
                            cx.error_spanned_by(
                                cont.original,
                                "#[serde(transparent)] requires at least one field that is not skipped",
                            );
                        }
                        Derive::Deserialize => {
                            cx.error_spanned_by(
                                cont.original,
                                "#[serde(transparent)] requires at least one field that is neither skipped nor has a default",
                            );
                        }
                    }
                }
            }
        }
        fn member_message(member: &Member) -> String {
            match member {
                Member::Named(ident) => {
                    let res = ::alloc::fmt::format(
                        ::core::fmt::Arguments::new_v1(
                            &["`", "`"],
                            &[::core::fmt::ArgumentV1::new_display(&ident)],
                        ),
                    );
                    res
                }
                Member::Unnamed(i) => {
                    let res = ::alloc::fmt::format(
                        ::core::fmt::Arguments::new_v1(
                            &["#"],
                            &[::core::fmt::ArgumentV1::new_display(&i.index)],
                        ),
                    );
                    res
                }
            }
        }
        fn allow_transparent(field: &Field, derive: Derive) -> bool {
            if let Type::Path(ty) = ungroup(field.ty) {
                if let Some(seg) = ty.path.segments.last() {
                    if seg.ident == "PhantomData" {
                        return false;
                    }
                }
            }
            match derive {
                Derive::Serialize => !field.attrs.skip_serializing(),
                Derive::Deserialize => {
                    !field.attrs.skip_deserializing() && field.attrs.default().is_none()
                }
            }
        }
        fn check_from_and_try_from(cx: &Ctxt, cont: &mut Container) {
            if cont.attrs.type_from().is_some() && cont.attrs.type_try_from().is_some() {
                cx.error_spanned_by(
                    cont.original,
                    "#[serde(from = \"...\")] and #[serde(try_from = \"...\")] conflict with each other",
                );
            }
        }
    }
    mod respan {
        use proc_macro2::{Group, Span, TokenStream, TokenTree};
        pub(crate) fn respan(stream: TokenStream, span: Span) -> TokenStream {
            stream.into_iter().map(|token| respan_token(token, span)).collect()
        }
        fn respan_token(mut token: TokenTree, span: Span) -> TokenTree {
            if let TokenTree::Group(g) = &mut token {
                *g = Group::new(g.delimiter(), respan(g.stream(), span));
            }
            token.set_span(span);
            token
        }
    }
    mod symbol {
        use std::fmt::{self, Display};
        use syn::{Ident, Path};
        pub struct Symbol(&'static str);
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::marker::Copy for Symbol {}
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::clone::Clone for Symbol {
            #[inline]
            fn clone(&self) -> Symbol {
                {
                    let _: ::core::clone::AssertParamIsClone<&'static str>;
                    *self
                }
            }
        }
        pub const ALIAS: Symbol = Symbol("alias");
        pub const BORROW: Symbol = Symbol("borrow");
        pub const BOUND: Symbol = Symbol("bound");
        pub const CONTENT: Symbol = Symbol("content");
        pub const CRATE: Symbol = Symbol("crate");
        pub const DEFAULT: Symbol = Symbol("default");
        pub const DENY_UNKNOWN_FIELDS: Symbol = Symbol("deny_unknown_fields");
        pub const DESERIALIZE: Symbol = Symbol("deserialize");
        pub const DESERIALIZE_WITH: Symbol = Symbol("deserialize_with");
        pub const FIELD_IDENTIFIER: Symbol = Symbol("field_identifier");
        pub const FLATTEN: Symbol = Symbol("flatten");
        pub const FROM: Symbol = Symbol("from");
        pub const GETTER: Symbol = Symbol("getter");
        pub const INTO: Symbol = Symbol("into");
        pub const OTHER: Symbol = Symbol("other");
        pub const REMOTE: Symbol = Symbol("remote");
        pub const RENAME: Symbol = Symbol("rename");
        pub const RENAME_ALL: Symbol = Symbol("rename_all");
        pub const SERDE: Symbol = Symbol("serde");
        pub const SERIALIZE: Symbol = Symbol("serialize");
        pub const SERIALIZE_WITH: Symbol = Symbol("serialize_with");
        pub const SKIP: Symbol = Symbol("skip");
        pub const SKIP_DESERIALIZING: Symbol = Symbol("skip_deserializing");
        pub const SKIP_SERIALIZING: Symbol = Symbol("skip_serializing");
        pub const SKIP_SERIALIZING_IF: Symbol = Symbol("skip_serializing_if");
        pub const TAG: Symbol = Symbol("tag");
        pub const TRANSPARENT: Symbol = Symbol("transparent");
        pub const TRY_FROM: Symbol = Symbol("try_from");
        pub const UNTAGGED: Symbol = Symbol("untagged");
        pub const VARIANT_IDENTIFIER: Symbol = Symbol("variant_identifier");
        pub const WITH: Symbol = Symbol("with");
        pub const EXPECTING: Symbol = Symbol("expecting");
        impl PartialEq<Symbol> for Ident {
            fn eq(&self, word: &Symbol) -> bool {
                self == word.0
            }
        }
        impl<'a> PartialEq<Symbol> for &'a Ident {
            fn eq(&self, word: &Symbol) -> bool {
                *self == word.0
            }
        }
        impl PartialEq<Symbol> for Path {
            fn eq(&self, word: &Symbol) -> bool {
                self.is_ident(word.0)
            }
        }
        impl<'a> PartialEq<Symbol> for &'a Path {
            fn eq(&self, word: &Symbol) -> bool {
                self.is_ident(word.0)
            }
        }
        impl Display for Symbol {
            fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str(self.0)
            }
        }
    }
    use syn::Type;
    pub enum Derive {
        Serialize,
        Deserialize,
    }
    #[automatically_derived]
    #[allow(unused_qualifications)]
    impl ::core::marker::Copy for Derive {}
    #[automatically_derived]
    #[allow(unused_qualifications)]
    impl ::core::clone::Clone for Derive {
        #[inline]
        fn clone(&self) -> Derive {
            { *self }
        }
    }
    pub fn ungroup(mut ty: &Type) -> &Type {
        while let Type::Group(group) = ty {
            ty = &group.elem;
        }
        ty
    }
}
use proc_macro::TokenStream;
use syn::DeriveInput;
#[macro_use]
mod bound {
    use std::collections::HashSet;
    use syn;
    use syn::punctuated::{Pair, Punctuated};
    use internals::ast::{Container, Data};
    use internals::{attr, ungroup};
    use proc_macro2::Span;
    pub fn without_defaults(generics: &syn::Generics) -> syn::Generics {
        syn::Generics {
            params: generics
                .params
                .iter()
                .map(|param| match param {
                    syn::GenericParam::Type(param) => {
                        syn::GenericParam::Type(syn::TypeParam {
                            eq_token: None,
                            default: None,
                            ..param.clone()
                        })
                    }
                    _ => param.clone(),
                })
                .collect(),
            ..generics.clone()
        }
    }
    pub fn with_where_predicates(
        generics: &syn::Generics,
        predicates: &[syn::WherePredicate],
    ) -> syn::Generics {
        let mut generics = generics.clone();
        generics.make_where_clause().predicates.extend(predicates.iter().cloned());
        generics
    }
    pub fn with_where_predicates_from_fields(
        cont: &Container,
        generics: &syn::Generics,
        from_field: fn(&attr::Field) -> Option<&[syn::WherePredicate]>,
    ) -> syn::Generics {
        let predicates = cont
            .data
            .all_fields()
            .filter_map(|field| from_field(&field.attrs))
            .flat_map(<[syn::WherePredicate]>::to_vec);
        let mut generics = generics.clone();
        generics.make_where_clause().predicates.extend(predicates);
        generics
    }
    pub fn with_where_predicates_from_variants(
        cont: &Container,
        generics: &syn::Generics,
        from_variant: fn(&attr::Variant) -> Option<&[syn::WherePredicate]>,
    ) -> syn::Generics {
        let variants = match &cont.data {
            Data::Enum(variants) => variants,
            Data::Struct(_, _) => {
                return generics.clone();
            }
        };
        let predicates = variants
            .iter()
            .filter_map(|variant| from_variant(&variant.attrs))
            .flat_map(<[syn::WherePredicate]>::to_vec);
        let mut generics = generics.clone();
        generics.make_where_clause().predicates.extend(predicates);
        generics
    }
    pub fn with_bound(
        cont: &Container,
        generics: &syn::Generics,
        filter: fn(&attr::Field, Option<&attr::Variant>) -> bool,
        bound: &syn::Path,
    ) -> syn::Generics {
        struct FindTyParams<'ast> {
            all_type_params: HashSet<syn::Ident>,
            relevant_type_params: HashSet<syn::Ident>,
            associated_type_usage: Vec<&'ast syn::TypePath>,
        }
        impl<'ast> FindTyParams<'ast> {
            fn visit_field(&mut self, field: &'ast syn::Field) {
                if let syn::Type::Path(ty) = ungroup(&field.ty) {
                    if let Some(Pair::Punctuated(t, _)) = ty.path.segments.pairs().next()
                    {
                        if self.all_type_params.contains(&t.ident) {
                            self.associated_type_usage.push(ty);
                        }
                    }
                }
                self.visit_type(&field.ty);
            }
            fn visit_path(&mut self, path: &'ast syn::Path) {
                if let Some(seg) = path.segments.last() {
                    if seg.ident == "PhantomData" {
                        return;
                    }
                }
                if path.leading_colon.is_none() && path.segments.len() == 1 {
                    let id = &path.segments[0].ident;
                    if self.all_type_params.contains(id) {
                        self.relevant_type_params.insert(id.clone());
                    }
                }
                for segment in &path.segments {
                    self.visit_path_segment(segment);
                }
            }
            fn visit_type(&mut self, ty: &'ast syn::Type) {
                match ty {
                    syn::Type::Array(ty) => self.visit_type(&ty.elem),
                    syn::Type::BareFn(ty) => {
                        for arg in &ty.inputs {
                            self.visit_type(&arg.ty);
                        }
                        self.visit_return_type(&ty.output);
                    }
                    syn::Type::Group(ty) => self.visit_type(&ty.elem),
                    syn::Type::ImplTrait(ty) => {
                        for bound in &ty.bounds {
                            self.visit_type_param_bound(bound);
                        }
                    }
                    syn::Type::Macro(ty) => self.visit_macro(&ty.mac),
                    syn::Type::Paren(ty) => self.visit_type(&ty.elem),
                    syn::Type::Path(ty) => {
                        if let Some(qself) = &ty.qself {
                            self.visit_type(&qself.ty);
                        }
                        self.visit_path(&ty.path);
                    }
                    syn::Type::Ptr(ty) => self.visit_type(&ty.elem),
                    syn::Type::Reference(ty) => self.visit_type(&ty.elem),
                    syn::Type::Slice(ty) => self.visit_type(&ty.elem),
                    syn::Type::TraitObject(ty) => {
                        for bound in &ty.bounds {
                            self.visit_type_param_bound(bound);
                        }
                    }
                    syn::Type::Tuple(ty) => {
                        for elem in &ty.elems {
                            self.visit_type(elem);
                        }
                    }
                    syn::Type::Infer(_)
                    | syn::Type::Never(_)
                    | syn::Type::Verbatim(_) => {}
                    _ => {}
                }
            }
            fn visit_path_segment(&mut self, segment: &'ast syn::PathSegment) {
                self.visit_path_arguments(&segment.arguments);
            }
            fn visit_path_arguments(&mut self, arguments: &'ast syn::PathArguments) {
                match arguments {
                    syn::PathArguments::None => {}
                    syn::PathArguments::AngleBracketed(arguments) => {
                        for arg in &arguments.args {
                            match arg {
                                syn::GenericArgument::Type(arg) => self.visit_type(arg),
                                syn::GenericArgument::Binding(arg) => {
                                    self.visit_type(&arg.ty)
                                }
                                syn::GenericArgument::Lifetime(_)
                                | syn::GenericArgument::Constraint(_)
                                | syn::GenericArgument::Const(_) => {}
                            }
                        }
                    }
                    syn::PathArguments::Parenthesized(arguments) => {
                        for argument in &arguments.inputs {
                            self.visit_type(argument);
                        }
                        self.visit_return_type(&arguments.output);
                    }
                }
            }
            fn visit_return_type(&mut self, return_type: &'ast syn::ReturnType) {
                match return_type {
                    syn::ReturnType::Default => {}
                    syn::ReturnType::Type(_, output) => self.visit_type(output),
                }
            }
            fn visit_type_param_bound(&mut self, bound: &'ast syn::TypeParamBound) {
                match bound {
                    syn::TypeParamBound::Trait(bound) => self.visit_path(&bound.path),
                    syn::TypeParamBound::Lifetime(_) => {}
                }
            }
            fn visit_macro(&mut self, _mac: &'ast syn::Macro) {}
        }
        let all_type_params = generics
            .type_params()
            .map(|param| param.ident.clone())
            .collect();
        let mut visitor = FindTyParams {
            all_type_params,
            relevant_type_params: HashSet::new(),
            associated_type_usage: Vec::new(),
        };
        match &cont.data {
            Data::Enum(variants) => {
                for variant in variants.iter() {
                    let relevant_fields = variant
                        .fields
                        .iter()
                        .filter(|field| filter(&field.attrs, Some(&variant.attrs)));
                    for field in relevant_fields {
                        visitor.visit_field(field.original);
                    }
                }
            }
            Data::Struct(_, fields) => {
                for field in fields.iter().filter(|field| filter(&field.attrs, None)) {
                    visitor.visit_field(field.original);
                }
            }
        }
        let relevant_type_params = visitor.relevant_type_params;
        let associated_type_usage = visitor.associated_type_usage;
        let new_predicates = generics
            .type_params()
            .map(|param| param.ident.clone())
            .filter(|id| relevant_type_params.contains(id))
            .map(|id| syn::TypePath {
                qself: None,
                path: id.into(),
            })
            .chain(associated_type_usage.into_iter().cloned())
            .map(|bounded_ty| {
                syn::WherePredicate::Type(syn::PredicateType {
                    lifetimes: None,
                    bounded_ty: syn::Type::Path(bounded_ty),
                    colon_token: <::syn::token::Colon>::default(),
                    bounds: <[_]>::into_vec(
                            #[rustc_box]
                            ::alloc::boxed::Box::new([
                                syn::TypeParamBound::Trait(syn::TraitBound {
                                    paren_token: None,
                                    modifier: syn::TraitBoundModifier::None,
                                    lifetimes: None,
                                    path: bound.clone(),
                                }),
                            ]),
                        )
                        .into_iter()
                        .collect(),
                })
            });
        let mut generics = generics.clone();
        generics.make_where_clause().predicates.extend(new_predicates);
        generics
    }
    pub fn with_self_bound(
        cont: &Container,
        generics: &syn::Generics,
        bound: &syn::Path,
    ) -> syn::Generics {
        let mut generics = generics.clone();
        generics
            .make_where_clause()
            .predicates
            .push(
                syn::WherePredicate::Type(syn::PredicateType {
                    lifetimes: None,
                    bounded_ty: type_of_item(cont),
                    colon_token: <::syn::token::Colon>::default(),
                    bounds: <[_]>::into_vec(
                            #[rustc_box]
                            ::alloc::boxed::Box::new([
                                syn::TypeParamBound::Trait(syn::TraitBound {
                                    paren_token: None,
                                    modifier: syn::TraitBoundModifier::None,
                                    lifetimes: None,
                                    path: bound.clone(),
                                }),
                            ]),
                        )
                        .into_iter()
                        .collect(),
                }),
            );
        generics
    }
    pub fn with_lifetime_bound(
        generics: &syn::Generics,
        lifetime: &str,
    ) -> syn::Generics {
        let bound = syn::Lifetime::new(lifetime, Span::call_site());
        let def = syn::LifetimeDef {
            attrs: Vec::new(),
            lifetime: bound.clone(),
            colon_token: None,
            bounds: Punctuated::new(),
        };
        let params = Some(syn::GenericParam::Lifetime(def))
            .into_iter()
            .chain(
                generics
                    .params
                    .iter()
                    .cloned()
                    .map(|mut param| {
                        match &mut param {
                            syn::GenericParam::Lifetime(param) => {
                                param.bounds.push(bound.clone());
                            }
                            syn::GenericParam::Type(param) => {
                                param
                                    .bounds
                                    .push(syn::TypeParamBound::Lifetime(bound.clone()));
                            }
                            syn::GenericParam::Const(_) => {}
                        }
                        param
                    }),
            )
            .collect();
        syn::Generics {
            params,
            ..generics.clone()
        }
    }
    fn type_of_item(cont: &Container) -> syn::Type {
        syn::Type::Path(syn::TypePath {
            qself: None,
            path: syn::Path {
                leading_colon: None,
                segments: <[_]>::into_vec(
                        #[rustc_box]
                        ::alloc::boxed::Box::new([
                            syn::PathSegment {
                                ident: cont.ident.clone(),
                                arguments: syn::PathArguments::AngleBracketed(syn::AngleBracketedGenericArguments {
                                    colon2_token: None,
                                    lt_token: <::syn::token::Lt>::default(),
                                    args: cont
                                        .generics
                                        .params
                                        .iter()
                                        .map(|param| match param {
                                            syn::GenericParam::Type(param) => {
                                                syn::GenericArgument::Type(
                                                    syn::Type::Path(syn::TypePath {
                                                        qself: None,
                                                        path: param.ident.clone().into(),
                                                    }),
                                                )
                                            }
                                            syn::GenericParam::Lifetime(param) => {
                                                syn::GenericArgument::Lifetime(param.lifetime.clone())
                                            }
                                            syn::GenericParam::Const(_) => {
                                                {
                                                    ::std::rt::begin_panic(
                                                        "Serde does not support const generics yet",
                                                    )
                                                };
                                            }
                                        })
                                        .collect(),
                                    gt_token: <::syn::token::Gt>::default(),
                                }),
                            },
                        ]),
                    )
                    .into_iter()
                    .collect(),
            },
        })
    }
}
#[macro_use]
mod fragment {
    use proc_macro2::TokenStream;
    use quote::ToTokens;
    use syn::token;
    pub enum Fragment {
        Expr(TokenStream),
        Block(TokenStream),
    }
    pub struct Expr(pub Fragment);
    impl ToTokens for Expr {
        fn to_tokens(&self, out: &mut TokenStream) {
            match &self.0 {
                Fragment::Expr(expr) => expr.to_tokens(out),
                Fragment::Block(block) => {
                    token::Brace::default().surround(out, |out| block.to_tokens(out));
                }
            }
        }
    }
    pub struct Stmts(pub Fragment);
    impl ToTokens for Stmts {
        fn to_tokens(&self, out: &mut TokenStream) {
            match &self.0 {
                Fragment::Expr(expr) => expr.to_tokens(out),
                Fragment::Block(block) => block.to_tokens(out),
            }
        }
    }
    pub struct Match(pub Fragment);
    impl ToTokens for Match {
        fn to_tokens(&self, out: &mut TokenStream) {
            match &self.0 {
                Fragment::Expr(expr) => {
                    expr.to_tokens(out);
                    <::syn::token::Comma>::default().to_tokens(out);
                }
                Fragment::Block(block) => {
                    token::Brace::default().surround(out, |out| block.to_tokens(out));
                }
            }
        }
    }
    impl AsRef<TokenStream> for Fragment {
        fn as_ref(&self) -> &TokenStream {
            match self {
                Fragment::Expr(expr) => expr,
                Fragment::Block(block) => block,
            }
        }
    }
}
mod de {
    use proc_macro2::{Literal, Span, TokenStream};
    use quote::ToTokens;
    use syn::punctuated::Punctuated;
    use syn::spanned::Spanned;
    use syn::{self, Ident, Index, Member};
    use bound;
    use dummy;
    use fragment::{Expr, Fragment, Match, Stmts};
    use internals::ast::{Container, Data, Field, Style, Variant};
    use internals::{attr, replace_receiver, ungroup, Ctxt, Derive};
    use pretend;
    use std::collections::BTreeSet;
    use std::ptr;
    pub fn expand_derive_deserialize(
        input: &mut syn::DeriveInput,
    ) -> Result<TokenStream, Vec<syn::Error>> {
        replace_receiver(input);
        let ctxt = Ctxt::new();
        let cont = match Container::from_ast(&ctxt, input, Derive::Deserialize) {
            Some(cont) => cont,
            None => return Err(ctxt.check().unwrap_err()),
        };
        precondition(&ctxt, &cont);
        ctxt.check()?;
        let ident = &cont.ident;
        let params = Parameters::new(&cont);
        let (de_impl_generics, _, ty_generics, where_clause) = split_with_de_lifetime(
            &params,
        );
        let body = Stmts(deserialize_body(&cont, &params));
        let delife = params.borrowed.de_lifetime();
        let serde = cont.attrs.serde_path();
        let impl_block = if let Some(remote) = cont.attrs.remote() {
            let vis = &input.vis;
            let used = pretend::pretend_used(&cont, params.is_packed);
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "impl");
                ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&ident, &mut _s);
                ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&vis, &mut _s);
                        ::quote::__private::push_ident(&mut _s, "fn");
                        ::quote::__private::push_ident(&mut _s, "deserialize");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__deserializer");
                                ::quote::__private::push_colon(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__D");
                                _s
                            },
                        );
                        ::quote::__private::push_rarrow(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Result");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::ToTokens::to_tokens(&remote, &mut _s);
                        ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "where");
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Deserializer");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::ToTokens::to_tokens(&delife, &mut _s);
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&used, &mut _s);
                                ::quote::ToTokens::to_tokens(&body, &mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            }
        } else {
            let fn_deserialize_in_place = deserialize_in_place_body(&cont, &params);
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_pound(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "automatically_derived");
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "impl");
                ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&serde, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Deserialize");
                ::quote::__private::push_lt(&mut _s);
                ::quote::ToTokens::to_tokens(&delife, &mut _s);
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "for");
                ::quote::ToTokens::to_tokens(&ident, &mut _s);
                ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "fn");
                        ::quote::__private::push_ident(&mut _s, "deserialize");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__deserializer");
                                ::quote::__private::push_colon(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__D");
                                _s
                            },
                        );
                        ::quote::__private::push_rarrow(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Result");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "where");
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Deserializer");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::ToTokens::to_tokens(&delife, &mut _s);
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&body, &mut _s);
                                _s
                            },
                        );
                        ::quote::ToTokens::to_tokens(&fn_deserialize_in_place, &mut _s);
                        _s
                    },
                );
                _s
            }
        };
        Ok(
            dummy::wrap_in_const(
                cont.attrs.custom_serde_path(),
                "DESERIALIZE",
                ident,
                impl_block,
            ),
        )
    }
    fn precondition(cx: &Ctxt, cont: &Container) {
        precondition_sized(cx, cont);
        precondition_no_de_lifetime(cx, cont);
    }
    fn precondition_sized(cx: &Ctxt, cont: &Container) {
        if let Data::Struct(_, fields) = &cont.data {
            if let Some(last) = fields.last() {
                if let syn::Type::Slice(_) = ungroup(last.ty) {
                    cx.error_spanned_by(
                        cont.original,
                        "cannot deserialize a dynamically sized struct",
                    );
                }
            }
        }
    }
    fn precondition_no_de_lifetime(cx: &Ctxt, cont: &Container) {
        if let BorrowedLifetimes::Borrowed(_) = borrowed_lifetimes(cont) {
            for param in cont.generics.lifetimes() {
                if param.lifetime.to_string() == "'de" {
                    cx.error_spanned_by(
                        &param.lifetime,
                        "cannot deserialize when there is a lifetime parameter called 'de",
                    );
                    return;
                }
            }
        }
    }
    struct Parameters {
        local: syn::Ident,
        this: syn::Path,
        generics: syn::Generics,
        borrowed: BorrowedLifetimes,
        has_getter: bool,
        is_packed: bool,
    }
    impl Parameters {
        fn new(cont: &Container) -> Self {
            let local = cont.ident.clone();
            let this = match cont.attrs.remote() {
                Some(remote) => remote.clone(),
                None => cont.ident.clone().into(),
            };
            let borrowed = borrowed_lifetimes(cont);
            let generics = build_generics(cont, &borrowed);
            let has_getter = cont.data.has_getter();
            let is_packed = cont.attrs.is_packed();
            Parameters {
                local,
                this,
                generics,
                borrowed,
                has_getter,
                is_packed,
            }
        }
        fn type_name(&self) -> String {
            self.this.segments.last().unwrap().ident.to_string()
        }
    }
    fn build_generics(cont: &Container, borrowed: &BorrowedLifetimes) -> syn::Generics {
        let generics = bound::without_defaults(cont.generics);
        let generics = bound::with_where_predicates_from_fields(
            cont,
            &generics,
            attr::Field::de_bound,
        );
        let generics = bound::with_where_predicates_from_variants(
            cont,
            &generics,
            attr::Variant::de_bound,
        );
        match cont.attrs.de_bound() {
            Some(predicates) => bound::with_where_predicates(&generics, predicates),
            None => {
                let generics = match *cont.attrs.default() {
                    attr::Default::Default => {
                        bound::with_self_bound(
                            cont,
                            &generics,
                            &::syn::parse_quote::parse({
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__private");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Default");
                                _s
                            }),
                        )
                    }
                    attr::Default::None | attr::Default::Path(_) => generics,
                };
                let delife = borrowed.de_lifetime();
                let generics = bound::with_bound(
                    cont,
                    &generics,
                    needs_deserialize_bound,
                    &::syn::parse_quote::parse({
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Deserialize");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::ToTokens::to_tokens(&delife, &mut _s);
                        ::quote::__private::push_gt(&mut _s);
                        _s
                    }),
                );
                bound::with_bound(
                    cont,
                    &generics,
                    requires_default,
                    &::syn::parse_quote::parse({
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Default");
                        _s
                    }),
                )
            }
        }
    }
    fn needs_deserialize_bound(
        field: &attr::Field,
        variant: Option<&attr::Variant>,
    ) -> bool {
        !field.skip_deserializing() && field.deserialize_with().is_none()
            && field.de_bound().is_none()
            && variant
                .map_or(
                    true,
                    |variant| {
                        !variant.skip_deserializing()
                            && variant.deserialize_with().is_none()
                            && variant.de_bound().is_none()
                    },
                )
    }
    fn requires_default(field: &attr::Field, _variant: Option<&attr::Variant>) -> bool {
        if let attr::Default::Default = *field.default() { true } else { false }
    }
    enum BorrowedLifetimes {
        Borrowed(BTreeSet<syn::Lifetime>),
        Static,
    }
    impl BorrowedLifetimes {
        fn de_lifetime(&self) -> syn::Lifetime {
            match *self {
                BorrowedLifetimes::Borrowed(_) => {
                    syn::Lifetime::new("'de", Span::call_site())
                }
                BorrowedLifetimes::Static => {
                    syn::Lifetime::new("'static", Span::call_site())
                }
            }
        }
        fn de_lifetime_def(&self) -> Option<syn::LifetimeDef> {
            match self {
                BorrowedLifetimes::Borrowed(bounds) => {
                    Some(syn::LifetimeDef {
                        attrs: Vec::new(),
                        lifetime: syn::Lifetime::new("'de", Span::call_site()),
                        colon_token: None,
                        bounds: bounds.iter().cloned().collect(),
                    })
                }
                BorrowedLifetimes::Static => None,
            }
        }
    }
    fn borrowed_lifetimes(cont: &Container) -> BorrowedLifetimes {
        let mut lifetimes = BTreeSet::new();
        for field in cont.data.all_fields() {
            if !field.attrs.skip_deserializing() {
                lifetimes.extend(field.attrs.borrowed_lifetimes().iter().cloned());
            }
        }
        if lifetimes.iter().any(|b| b.to_string() == "'static") {
            BorrowedLifetimes::Static
        } else {
            BorrowedLifetimes::Borrowed(lifetimes)
        }
    }
    fn deserialize_body(cont: &Container, params: &Parameters) -> Fragment {
        if cont.attrs.transparent() {
            deserialize_transparent(cont, params)
        } else if let Some(type_from) = cont.attrs.type_from() {
            deserialize_from(type_from)
        } else if let Some(type_try_from) = cont.attrs.type_try_from() {
            deserialize_try_from(type_try_from)
        } else if let attr::Identifier::No = cont.attrs.identifier() {
            match &cont.data {
                Data::Enum(variants) => deserialize_enum(params, variants, &cont.attrs),
                Data::Struct(Style::Struct, fields) => {
                    deserialize_struct(
                        None,
                        params,
                        fields,
                        &cont.attrs,
                        None,
                        &Untagged::No,
                    )
                }
                Data::Struct(Style::Tuple, fields)
                | Data::Struct(Style::Newtype, fields) => {
                    deserialize_tuple(None, params, fields, &cont.attrs, None)
                }
                Data::Struct(Style::Unit, _) => {
                    deserialize_unit_struct(params, &cont.attrs)
                }
            }
        } else {
            match &cont.data {
                Data::Enum(variants) => {
                    deserialize_custom_identifier(params, variants, &cont.attrs)
                }
                Data::Struct(_, _) => {
                    ::core::panicking::unreachable_display(
                        &"checked in serde_derive_internals",
                    )
                }
            }
        }
    }
    #[cfg(not(feature = "deserialize_in_place"))]
    fn deserialize_in_place_body(
        _cont: &Container,
        _params: &Parameters,
    ) -> Option<Stmts> {
        None
    }
    fn deserialize_transparent(cont: &Container, params: &Parameters) -> Fragment {
        let fields = match &cont.data {
            Data::Struct(_, fields) => fields,
            Data::Enum(_) => {
                ::core::panicking::panic("internal error: entered unreachable code")
            }
        };
        let this = &params.this;
        let transparent_field = fields.iter().find(|f| f.attrs.transparent()).unwrap();
        let path = match transparent_field.attrs.deserialize_with() {
            Some(path) => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&path, &mut _s);
                _s
            }
            None => {
                let span = transparent_field.original.span();
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "Deserialize",
                    );
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "deserialize",
                    );
                    _s
                }
            }
        };
        let assign = fields
            .iter()
            .map(|field| {
                let member = &field.member;
                if ptr::eq(field, transparent_field) {
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&member, &mut _s);
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__transparent");
                        _s
                    }
                } else {
                    let value = match field.attrs.default() {
                        attr::Default::Default => {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Default");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "default");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                ::quote::__private::TokenStream::new(),
                            );
                            _s
                        }
                        attr::Default::Path(path) => {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&path, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                ::quote::__private::TokenStream::new(),
                            );
                            _s
                        }
                        attr::Default::None => {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                            _s
                        }
                    };
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&member, &mut _s);
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::ToTokens::to_tokens(&value, &mut _s);
                        _s
                    }
                }
            });
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Result");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "map");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&path, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_or(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__transparent");
                    ::quote::__private::push_or(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            {
                                use ::quote::__private::ext::*;
                                let mut _i = 0usize;
                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                #[allow(unused_mut)]
                                let (mut assign, i) = assign.quote_into_iter();
                                let has_iter = has_iter | i;
                                let _: ::quote::__private::HasIterator = has_iter;
                                while true {
                                    let assign = match assign.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    if _i > 0 {
                                        ::quote::__private::push_comma(&mut _s);
                                    }
                                    _i += 1;
                                    ::quote::ToTokens::to_tokens(&assign, &mut _s);
                                }
                            };
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_from(type_from: &syn::Type) -> Fragment {
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Result");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "map");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_from, &mut _s);
                    ::quote::__private::push_ident(&mut _s, "as");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserialize");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "deserialize");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "From");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "from");
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_try_from(type_try_from: &syn::Type) -> Fragment {
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Result");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "and_then");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_try_from, &mut _s);
                    ::quote::__private::push_ident(&mut _s, "as");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserialize");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "deserialize");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_or(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "v");
                    ::quote::__private::push_or(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "TryFrom");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try_from");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "v");
                            _s
                        },
                    );
                    ::quote::__private::push_dot(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "map_err");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Error");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "custom");
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_unit_struct(
        params: &Parameters,
        cattrs: &attr::Container,
    ) -> Fragment {
        let this = &params.this;
        let type_name = cattrs.name().deserialize_name();
        let expecting = {
            let res = ::alloc::fmt::format(
                ::core::fmt::Arguments::new_v1(
                    &["unit struct "],
                    &[::core::fmt::ArgumentV1::new_display(&params.type_name())],
                ),
            );
            res
        };
        let expecting = cattrs.expecting().unwrap_or(&expecting);
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_lifetime(&mut _s, "\'de");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Visitor");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_lifetime(&mut _s, "\'de");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "expecting");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__formatter");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fmt");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "write_str");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__formatter");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_pound(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Bracket,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "inline");
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "visit_unit");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__E");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__E");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__E");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Ok");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Deserializer");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "deserialize_unit_struct");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__deserializer");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__Visitor");
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_tuple(
        variant_ident: Option<&syn::Ident>,
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
        deserializer: Option<TokenStream>,
    ) -> Fragment {
        let this = &params.this;
        let (de_impl_generics, de_ty_generics, ty_generics, where_clause) = split_with_de_lifetime(
            params,
        );
        let delife = params.borrowed.de_lifetime();
        if !!cattrs.has_flatten() {
            ::core::panicking::panic("assertion failed: !cattrs.has_flatten()")
        }
        let construct = if params.has_getter {
            let local = &params.local;
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&local, &mut _s);
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                _s
            }
        };
        let is_enum = variant_ident.is_some();
        let type_path = match variant_ident {
            Some(variant_ident) => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&construct, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                _s
            }
            None => construct,
        };
        let expecting = match variant_ident {
            Some(variant_ident) => {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["tuple variant ", "::"],
                        &[
                            ::core::fmt::ArgumentV1::new_display(&params.type_name()),
                            ::core::fmt::ArgumentV1::new_display(&variant_ident),
                        ],
                    ),
                );
                res
            }
            None => {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["tuple struct "],
                        &[::core::fmt::ArgumentV1::new_display(&params.type_name())],
                    ),
                );
                res
            }
        };
        let expecting = cattrs.expecting().unwrap_or(&expecting);
        let nfields = fields.len();
        let visit_newtype_struct = if !is_enum && nfields == 1 {
            Some(deserialize_newtype_struct(&type_path, params, &fields[0]))
        } else {
            None
        };
        let visit_seq = Stmts(
            deserialize_seq(&type_path, params, fields, false, cattrs, expecting),
        );
        let visitor_expr = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            _s
        };
        let dispatch = if let Some(deserializer) = deserializer {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Deserializer");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "deserialize_tuple");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&deserializer, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&nfields, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        } else if is_enum {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "VariantAccess");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "tuple_variant");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__variant");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&nfields, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        } else if nfields == 1 {
            let type_name = cattrs.name().deserialize_name();
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Deserializer");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "deserialize_newtype_struct");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__deserializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        } else {
            let type_name = cattrs.name().deserialize_name();
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Deserializer");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "deserialize_tuple_struct");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__deserializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&nfields, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        };
        let all_skipped = fields.iter().all(|field| field.attrs.skip_deserializing());
        let visitor_var = if all_skipped {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_underscore(&mut _s);
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "mut");
                ::quote::__private::push_ident(&mut _s, "__seq");
                _s
            }
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Visitor");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "expecting");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__formatter");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fmt");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "write_str");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__formatter");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::ToTokens::to_tokens(&visit_newtype_struct, &mut _s);
                    ::quote::__private::push_pound(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Bracket,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "inline");
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "visit_seq");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&visitor_var, &mut _s);
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__A");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SeqAccess");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&visit_seq, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::ToTokens::to_tokens(&dispatch, &mut _s);
            _s
        })
    }
    fn deserialize_seq(
        type_path: &TokenStream,
        params: &Parameters,
        fields: &[Field],
        is_struct: bool,
        cattrs: &attr::Container,
        expecting: &str,
    ) -> Fragment {
        let vars = (0..fields.len()).map(field_i as fn(_) -> _);
        let deserialized_count = fields
            .iter()
            .filter(|field| !field.attrs.skip_deserializing())
            .count();
        let expecting = if deserialized_count == 1 {
            {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["", " with 1 element"],
                        &[::core::fmt::ArgumentV1::new_display(&expecting)],
                    ),
                );
                res
            }
        } else {
            {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["", " with ", " elements"],
                        &[
                            ::core::fmt::ArgumentV1::new_display(&expecting),
                            ::core::fmt::ArgumentV1::new_display(&deserialized_count),
                        ],
                    ),
                );
                res
            }
        };
        let expecting = cattrs.expecting().unwrap_or(&expecting);
        let mut index_in_seq = 0_usize;
        let let_values = vars
            .clone()
            .zip(fields)
            .map(|(var, field)| {
                if field.attrs.skip_deserializing() {
                    let default = Expr(expr_is_missing(field, cattrs));
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "let");
                        ::quote::ToTokens::to_tokens(&var, &mut _s);
                        ::quote::__private::push_eq(&mut _s);
                        ::quote::ToTokens::to_tokens(&default, &mut _s);
                        ::quote::__private::push_semi(&mut _s);
                        _s
                    }
                } else {
                    let visit = match field.attrs.deserialize_with() {
                        None => {
                            let field_ty = field.ty;
                            let span = field.original.span();
                            let func = {
                                let mut _s = ::quote::__private::TokenStream::new();
                                let _span: ::quote::__private::Span = span;
                                ::quote::__private::push_ident_spanned(
                                    &mut _s,
                                    _span,
                                    "_serde",
                                );
                                ::quote::__private::push_colon2_spanned(&mut _s, _span);
                                ::quote::__private::push_ident_spanned(
                                    &mut _s,
                                    _span,
                                    "de",
                                );
                                ::quote::__private::push_colon2_spanned(&mut _s, _span);
                                ::quote::__private::push_ident_spanned(
                                    &mut _s,
                                    _span,
                                    "SeqAccess",
                                );
                                ::quote::__private::push_colon2_spanned(&mut _s, _span);
                                ::quote::__private::push_ident_spanned(
                                    &mut _s,
                                    _span,
                                    "next_element",
                                );
                                ::quote::__private::push_colon2_spanned(&mut _s, _span);
                                ::quote::__private::push_lt_spanned(&mut _s, _span);
                                ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                                ::quote::__private::push_gt_spanned(&mut _s, _span);
                                _s
                            };
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "try");
                                ::quote::__private::push_bang(&mut _s);
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::ToTokens::to_tokens(&func, &mut _s);
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_and(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "mut");
                                                ::quote::__private::push_ident(&mut _s, "__seq");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            }
                        }
                        Some(path) => {
                            let (wrapper, wrapper_ty) = wrap_deserialize_field_with(
                                params,
                                field.ty,
                                path,
                            );
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Brace,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Option");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "map");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "try");
                                                ::quote::__private::push_bang(&mut _s);
                                                ::quote::__private::push_group(
                                                    &mut _s,
                                                    ::quote::__private::Delimiter::Parenthesis,
                                                    {
                                                        let mut _s = ::quote::__private::TokenStream::new();
                                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "de");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "SeqAccess");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "next_element");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_lt(&mut _s);
                                                        ::quote::ToTokens::to_tokens(&wrapper_ty, &mut _s);
                                                        ::quote::__private::push_gt(&mut _s);
                                                        ::quote::__private::push_group(
                                                            &mut _s,
                                                            ::quote::__private::Delimiter::Parenthesis,
                                                            {
                                                                let mut _s = ::quote::__private::TokenStream::new();
                                                                ::quote::__private::push_and(&mut _s);
                                                                ::quote::__private::push_ident(&mut _s, "mut");
                                                                ::quote::__private::push_ident(&mut _s, "__seq");
                                                                _s
                                                            },
                                                        );
                                                        _s
                                                    },
                                                );
                                                ::quote::__private::push_comma(&mut _s);
                                                ::quote::__private::push_or(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "__wrap");
                                                ::quote::__private::push_or(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "__wrap");
                                                ::quote::__private::push_dot(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            }
                        }
                    };
                    let value_if_none = match field.attrs.default() {
                        attr::Default::Default => {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Default");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "default");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                ::quote::__private::TokenStream::new(),
                            );
                            _s
                        }
                        attr::Default::Path(path) => {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&path, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                ::quote::__private::TokenStream::new(),
                            );
                            _s
                        }
                        attr::Default::None => {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "return");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Err");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "de");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Error");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "invalid_length");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::ToTokens::to_tokens(&index_in_seq, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            _s
                        }
                    };
                    let assign = {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "let");
                        ::quote::ToTokens::to_tokens(&var, &mut _s);
                        ::quote::__private::push_eq(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::ToTokens::to_tokens(&visit, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__private");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Some");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "__value");
                                        _s
                                    },
                                );
                                ::quote::__private::push_fat_arrow(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__value");
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__private");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "None");
                                ::quote::__private::push_fat_arrow(&mut _s);
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Brace,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::ToTokens::to_tokens(&value_if_none, &mut _s);
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        ::quote::__private::push_semi(&mut _s);
                        _s
                    };
                    index_in_seq += 1;
                    assign
                }
            });
        let mut result = if is_struct {
            let names = fields.iter().map(|f| &f.member);
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&type_path, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut names, i) = names.quote_into_iter();
                            let has_iter = has_iter | i;
                            #[allow(unused_mut)]
                            let (mut vars, i) = vars.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let names = match names.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                let vars = match vars.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&names, &mut _s);
                                ::quote::__private::push_colon(&mut _s);
                                ::quote::ToTokens::to_tokens(&vars, &mut _s);
                            }
                        };
                        _s
                    },
                );
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&type_path, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut vars, i) = vars.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let vars = match vars.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&vars, &mut _s);
                            }
                        };
                        _s
                    },
                );
                _s
            }
        };
        if params.has_getter {
            let this = &params.this;
            result = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Into");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_lt(&mut _s);
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "into");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&result, &mut _s);
                        _s
                    },
                );
                _s
            };
        }
        let let_default = match cattrs.default() {
            attr::Default::Default => {
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__default");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Default");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "default");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                })
            }
            attr::Default::Path(path) => {
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__default");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&path, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                })
            }
            attr::Default::None => None,
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&let_default, &mut _s);
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut let_values, i) = let_values.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let let_values = match let_values.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&let_values, &mut _s);
                }
            };
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Ok");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&result, &mut _s);
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_newtype_struct(
        type_path: &TokenStream,
        params: &Parameters,
        field: &Field,
    ) -> TokenStream {
        let delife = params.borrowed.de_lifetime();
        let field_ty = field.ty;
        let value = match field.attrs.deserialize_with() {
            None => {
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_lt_spanned(&mut _s, _span);
                    ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "as");
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "Deserialize",
                    );
                    ::quote::__private::push_gt_spanned(&mut _s, _span);
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "deserialize",
                    );
                    _s
                };
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&func, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__e");
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                }
            }
            Some(path) => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "try");
                ::quote::__private::push_bang(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&path, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__e");
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            }
        };
        let mut result = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&type_path, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__field0");
                    _s
                },
            );
            _s
        };
        if params.has_getter {
            let this = &params.this;
            result = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Into");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_lt(&mut _s);
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "into");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&result, &mut _s);
                        _s
                    },
                );
                _s
            };
        }
        {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_pound(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Bracket,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "inline");
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "fn");
            ::quote::__private::push_ident(&mut _s, "visit_newtype_struct");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "self");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__e");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__E");
                    _s
                },
            );
            ::quote::__private::push_rarrow(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Result");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Self");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Value");
            ::quote::__private::push_comma(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Error");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "where");
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_colon(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Deserializer");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_comma(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__field0");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&value, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Ok");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&result, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        }
    }
    enum Untagged {
        Yes,
        No,
    }
    fn deserialize_struct(
        variant_ident: Option<&syn::Ident>,
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
        deserializer: Option<TokenStream>,
        untagged: &Untagged,
    ) -> Fragment {
        let is_enum = variant_ident.is_some();
        let this = &params.this;
        let (de_impl_generics, de_ty_generics, ty_generics, where_clause) = split_with_de_lifetime(
            params,
        );
        let delife = params.borrowed.de_lifetime();
        let construct = if params.has_getter {
            let local = &params.local;
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&local, &mut _s);
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                _s
            }
        };
        let type_path = match variant_ident {
            Some(variant_ident) => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&construct, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                _s
            }
            None => construct,
        };
        let expecting = match variant_ident {
            Some(variant_ident) => {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["struct variant ", "::"],
                        &[
                            ::core::fmt::ArgumentV1::new_display(&params.type_name()),
                            ::core::fmt::ArgumentV1::new_display(&variant_ident),
                        ],
                    ),
                );
                res
            }
            None => {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["struct "],
                        &[::core::fmt::ArgumentV1::new_display(&params.type_name())],
                    ),
                );
                res
            }
        };
        let expecting = cattrs.expecting().unwrap_or(&expecting);
        let visit_seq = Stmts(
            deserialize_seq(&type_path, params, fields, true, cattrs, expecting),
        );
        let (field_visitor, fields_stmt, visit_map) = if cattrs.has_flatten() {
            deserialize_struct_as_map_visitor(&type_path, params, fields, cattrs)
        } else {
            deserialize_struct_as_struct_visitor(&type_path, params, fields, cattrs)
        };
        let field_visitor = Stmts(field_visitor);
        let fields_stmt = fields_stmt.map(Stmts);
        let visit_map = Stmts(visit_map);
        let visitor_expr = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            _s
        };
        let dispatch = if let Some(deserializer) = deserializer {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Deserializer");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "deserialize_any");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&deserializer, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        } else if is_enum && cattrs.has_flatten() {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "VariantAccess");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "newtype_variant_seed");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__variant");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        } else if is_enum {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "VariantAccess");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "struct_variant");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__variant");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "FIELDS");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        } else if cattrs.has_flatten() {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Deserializer");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "deserialize_map");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__deserializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        } else {
            let type_name = cattrs.name().deserialize_name();
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Deserializer");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "deserialize_struct");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__deserializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "FIELDS");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&visitor_expr, &mut _s);
                        _s
                    },
                );
                _s
            }
        };
        let all_skipped = fields.iter().all(|field| field.attrs.skip_deserializing());
        let visitor_var = if all_skipped {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_underscore(&mut _s);
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "mut");
                ::quote::__private::push_ident(&mut _s, "__seq");
                _s
            }
        };
        let visit_seq = match *untagged {
            Untagged::No if !cattrs.has_flatten() => {
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_pound(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Bracket,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "inline");
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "visit_seq");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&visitor_var, &mut _s);
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__A");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SeqAccess");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&visit_seq, &mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            _ => None,
        };
        let visitor_seed = if is_enum && cattrs.has_flatten() {
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "impl");
                ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "DeserializeSeed");
                ::quote::__private::push_lt(&mut _s);
                ::quote::ToTokens::to_tokens(&delife, &mut _s);
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "for");
                ::quote::__private::push_ident(&mut _s, "__Visitor");
                ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "type");
                        ::quote::__private::push_ident(&mut _s, "Value");
                        ::quote::__private::push_eq(&mut _s);
                        ::quote::ToTokens::to_tokens(&this, &mut _s);
                        ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                        ::quote::__private::push_semi(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "fn");
                        ::quote::__private::push_ident(&mut _s, "deserialize");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "self");
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__deserializer");
                                ::quote::__private::push_colon(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__D");
                                _s
                            },
                        );
                        ::quote::__private::push_rarrow(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Result");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Self");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Value");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "where");
                        ::quote::__private::push_ident(&mut _s, "__D");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Deserializer");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'de");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Deserializer");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "deserialize_map");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "__deserializer");
                                        ::quote::__private::push_comma(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "self");
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            })
        } else {
            None
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&field_visitor, &mut _s);
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Visitor");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "expecting");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__formatter");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fmt");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "write_str");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__formatter");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::ToTokens::to_tokens(&visit_seq, &mut _s);
                    ::quote::__private::push_pound(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Bracket,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "inline");
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "visit_map");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__map");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__A");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "MapAccess");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&visit_map, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::ToTokens::to_tokens(&visitor_seed, &mut _s);
            ::quote::ToTokens::to_tokens(&fields_stmt, &mut _s);
            ::quote::ToTokens::to_tokens(&dispatch, &mut _s);
            _s
        })
    }
    fn deserialize_enum(
        params: &Parameters,
        variants: &[Variant],
        cattrs: &attr::Container,
    ) -> Fragment {
        match cattrs.tag() {
            attr::TagType::External => {
                deserialize_externally_tagged_enum(params, variants, cattrs)
            }
            attr::TagType::Internal { tag } => {
                deserialize_internally_tagged_enum(params, variants, cattrs, tag)
            }
            attr::TagType::Adjacent { tag, content } => {
                deserialize_adjacently_tagged_enum(
                    params,
                    variants,
                    cattrs,
                    tag,
                    content,
                )
            }
            attr::TagType::None => deserialize_untagged_enum(params, variants, cattrs),
        }
    }
    fn prepare_enum_variant_enum(
        variants: &[Variant],
        cattrs: &attr::Container,
    ) -> (TokenStream, Stmts) {
        let mut deserialized_variants = variants
            .iter()
            .enumerate()
            .filter(|&(_, variant)| !variant.attrs.skip_deserializing());
        let variant_names_idents: Vec<_> = deserialized_variants
            .clone()
            .map(|(i, variant)| {
                (
                    variant.attrs.name().deserialize_name(),
                    field_i(i),
                    variant.attrs.aliases(),
                )
            })
            .collect();
        let other_idx = deserialized_variants
            .position(|(_, variant)| variant.attrs.other());
        let variants_stmt = {
            let variant_names = variant_names_idents.iter().map(|(name, _, _)| name);
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "const");
                ::quote::__private::push_ident(&mut _s, "VARIANTS");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_lifetime(&mut _s, "\'static");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_and(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'static");
                        ::quote::__private::push_ident(&mut _s, "str");
                        _s
                    },
                );
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut variant_names, i) = variant_names.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let variant_names = match variant_names.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&variant_names, &mut _s);
                            }
                        };
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                _s
            }
        };
        let variant_visitor = Stmts(
            deserialize_generated_identifier(
                &variant_names_idents,
                cattrs,
                true,
                other_idx,
            ),
        );
        (variants_stmt, variant_visitor)
    }
    fn deserialize_externally_tagged_enum(
        params: &Parameters,
        variants: &[Variant],
        cattrs: &attr::Container,
    ) -> Fragment {
        let this = &params.this;
        let (de_impl_generics, de_ty_generics, ty_generics, where_clause) = split_with_de_lifetime(
            params,
        );
        let delife = params.borrowed.de_lifetime();
        let type_name = cattrs.name().deserialize_name();
        let expecting = {
            let res = ::alloc::fmt::format(
                ::core::fmt::Arguments::new_v1(
                    &["enum "],
                    &[::core::fmt::ArgumentV1::new_display(&params.type_name())],
                ),
            );
            res
        };
        let expecting = cattrs.expecting().unwrap_or(&expecting);
        let (variants_stmt, variant_visitor) = prepare_enum_variant_enum(
            variants,
            cattrs,
        );
        let variant_arms = variants
            .iter()
            .enumerate()
            .filter(|&(_, variant)| !variant.attrs.skip_deserializing())
            .map(|(i, variant)| {
                let variant_name = field_i(i);
                let block = Match(
                    deserialize_externally_tagged_variant(params, variant, cattrs),
                );
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__Field");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__variant");
                            _s
                        },
                    );
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::ToTokens::to_tokens(&block, &mut _s);
                    _s
                }
            });
        let all_skipped = variants
            .iter()
            .all(|variant| variant.attrs.skip_deserializing());
        let match_variant = if all_skipped {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "map");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "EnumAccess");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "variant");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__Field");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__data");
                                _s
                            },
                        );
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_or(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__impossible");
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_underscore(&mut _s);
                                _s
                            },
                        );
                        ::quote::__private::push_or(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::__private::push_ident(&mut _s, "__impossible");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            ::quote::__private::TokenStream::new(),
                        );
                        _s
                    },
                );
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "match");
                ::quote::__private::push_ident(&mut _s, "try");
                ::quote::__private::push_bang(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "EnumAccess");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "variant");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__data");
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut variant_arms, i) = variant_arms.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let variant_arms = match variant_arms.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                ::quote::ToTokens::to_tokens(&variant_arms, &mut _s);
                            }
                        };
                        _s
                    },
                );
                _s
            }
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&variant_visitor, &mut _s);
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Visitor");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "expecting");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__formatter");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fmt");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "write_str");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__formatter");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "visit_enum");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__data");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__A");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "EnumAccess");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&match_variant, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::ToTokens::to_tokens(&variants_stmt, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Deserializer");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "deserialize_enum");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__deserializer");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "VARIANTS");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__Visitor");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "marker");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_lt(&mut _s);
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                            ::quote::__private::push_gt(&mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "lifetime");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                            ::quote::__private::push_comma(&mut _s);
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_internally_tagged_enum(
        params: &Parameters,
        variants: &[Variant],
        cattrs: &attr::Container,
        tag: &str,
    ) -> Fragment {
        let (variants_stmt, variant_visitor) = prepare_enum_variant_enum(
            variants,
            cattrs,
        );
        let variant_arms = variants
            .iter()
            .enumerate()
            .filter(|&(_, variant)| !variant.attrs.skip_deserializing())
            .map(|(i, variant)| {
                let variant_name = field_i(i);
                let block = Match(
                    deserialize_internally_tagged_variant(
                        params,
                        variant,
                        cattrs,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "ContentDeserializer",
                            );
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_lt(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__D");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Error");
                            ::quote::__private::push_gt(&mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "new");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__tagged");
                                    ::quote::__private::push_dot(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "content");
                                    _s
                                },
                            );
                            _s
                        },
                    ),
                );
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__Field");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::ToTokens::to_tokens(&block, &mut _s);
                    _s
                }
            });
        let expecting = {
            let res = ::alloc::fmt::format(
                ::core::fmt::Arguments::new_v1(
                    &["internally tagged enum "],
                    &[::core::fmt::ArgumentV1::new_display(&params.type_name())],
                ),
            );
            res
        };
        let expecting = cattrs.expecting().unwrap_or(&expecting);
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&variant_visitor, &mut _s);
            ::quote::ToTokens::to_tokens(&variants_stmt, &mut _s);
            ::quote::__private::push_ident(&mut _s, "let");
            ::quote::__private::push_ident(&mut _s, "__tagged");
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "deserialize_any");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "TaggedContentVisitor",
                            );
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_lt(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__Field");
                            ::quote::__private::push_gt(&mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "new");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "match");
            ::quote::__private::push_ident(&mut _s, "__tagged");
            ::quote::__private::push_dot(&mut _s);
            ::quote::__private::push_ident(&mut _s, "tag");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut variant_arms, i) = variant_arms.quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let variant_arms = match variant_arms.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::ToTokens::to_tokens(&variant_arms, &mut _s);
                        }
                    };
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_adjacently_tagged_enum(
        params: &Parameters,
        variants: &[Variant],
        cattrs: &attr::Container,
        tag: &str,
        content: &str,
    ) -> Fragment {
        let this = &params.this;
        let (de_impl_generics, de_ty_generics, ty_generics, where_clause) = split_with_de_lifetime(
            params,
        );
        let delife = params.borrowed.de_lifetime();
        let (variants_stmt, variant_visitor) = prepare_enum_variant_enum(
            variants,
            cattrs,
        );
        let variant_arms: &Vec<_> = &variants
            .iter()
            .enumerate()
            .filter(|&(_, variant)| !variant.attrs.skip_deserializing())
            .map(|(i, variant)| {
                let variant_index = field_i(i);
                let block = Match(
                    deserialize_untagged_variant(
                        params,
                        variant,
                        cattrs,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            _s
                        },
                    ),
                );
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__Field");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&variant_index, &mut _s);
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::ToTokens::to_tokens(&block, &mut _s);
                    _s
                }
            })
            .collect();
        let expecting = {
            let res = ::alloc::fmt::format(
                ::core::fmt::Arguments::new_v1(
                    &["adjacently tagged enum "],
                    &[::core::fmt::ArgumentV1::new_display(&params.type_name())],
                ),
            );
            res
        };
        let expecting = cattrs.expecting().unwrap_or(&expecting);
        let type_name = cattrs.name().deserialize_name();
        let deny_unknown_fields = cattrs.deny_unknown_fields();
        let field_visitor_ty = if deny_unknown_fields {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "TagOrContentFieldVisitor");
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "TagContentOtherFieldVisitor");
                _s
            }
        };
        let tag_or_content = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&field_visitor_ty, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "tag");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::ToTokens::to_tokens(&tag, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "content");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::ToTokens::to_tokens(&content, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            _s
        };
        let mut missing_content = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Err");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_ident(&mut _s, "as");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "missing_field");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&content, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        };
        let mut missing_content_fallthrough = ::quote::__private::TokenStream::new();
        let missing_content_arms = variants
            .iter()
            .enumerate()
            .filter(|&(_, variant)| !variant.attrs.skip_deserializing())
            .filter_map(|(i, variant)| {
                let variant_index = field_i(i);
                let variant_ident = &variant.ident;
                let arm = match variant.style {
                    Style::Unit => {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&this, &mut _s);
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                                _s
                            },
                        );
                        _s
                    }
                    Style::Newtype if variant.attrs.deserialize_with().is_none() => {
                        let span = variant.original.span();
                        let func = {
                            let mut _s = ::quote::__private::TokenStream::new();
                            let _span: ::quote::__private::Span = span;
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "_serde",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "__private",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(&mut _s, _span, "de");
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "missing_field",
                            );
                            _s
                        };
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&func, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&content, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_dot(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "map");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                                    _s
                                },
                            );
                            _s
                        }
                    }
                    _ => {
                        missing_content_fallthrough = {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_underscore(&mut _s);
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::ToTokens::to_tokens(&missing_content, &mut _s);
                            _s
                        };
                        return None;
                    }
                };
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__Field");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&variant_index, &mut _s);
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::ToTokens::to_tokens(&arm, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                })
            })
            .collect::<Vec<_>>();
        if !missing_content_arms.is_empty() {
            missing_content = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "match");
                ::quote::__private::push_ident(&mut _s, "__field");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut missing_content_arms, i) = missing_content_arms
                                .quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let missing_content_arms = match missing_content_arms.next()
                                {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                ::quote::ToTokens::to_tokens(
                                    &missing_content_arms,
                                    &mut _s,
                                );
                            }
                        };
                        ::quote::ToTokens::to_tokens(
                            &missing_content_fallthrough,
                            &mut _s,
                        );
                        _s
                    },
                );
                _s
            };
        }
        let next_key = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "MapAccess");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "next_key_seed");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__map");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&tag_or_content, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        };
        let next_relevant_key = if deny_unknown_fields {
            next_key
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "let");
                        ::quote::__private::push_ident(&mut _s, "mut");
                        ::quote::__private::push_ident(&mut _s, "__rk");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Option");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "TagOrContentField");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_eq(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "None");
                        ::quote::__private::push_semi(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "while");
                        ::quote::__private::push_ident(&mut _s, "let");
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Some");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__k");
                                _s
                            },
                        );
                        ::quote::__private::push_eq(&mut _s);
                        ::quote::ToTokens::to_tokens(&next_key, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "match");
                                ::quote::__private::push_ident(&mut _s, "__k");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Brace,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(
                                            &mut _s,
                                            "TagContentOtherField",
                                        );
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Other");
                                        ::quote::__private::push_fat_arrow(&mut _s);
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Brace,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "let");
                                                ::quote::__private::push_underscore(&mut _s);
                                                ::quote::__private::push_eq(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "try");
                                                ::quote::__private::push_bang(&mut _s);
                                                ::quote::__private::push_group(
                                                    &mut _s,
                                                    ::quote::__private::Delimiter::Parenthesis,
                                                    {
                                                        let mut _s = ::quote::__private::TokenStream::new();
                                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "de");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "MapAccess");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "next_value");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_lt(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "de");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "IgnoredAny");
                                                        ::quote::__private::push_gt(&mut _s);
                                                        ::quote::__private::push_group(
                                                            &mut _s,
                                                            ::quote::__private::Delimiter::Parenthesis,
                                                            {
                                                                let mut _s = ::quote::__private::TokenStream::new();
                                                                ::quote::__private::push_and(&mut _s);
                                                                ::quote::__private::push_ident(&mut _s, "mut");
                                                                ::quote::__private::push_ident(&mut _s, "__map");
                                                                _s
                                                            },
                                                        );
                                                        _s
                                                    },
                                                );
                                                ::quote::__private::push_semi(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "continue");
                                                ::quote::__private::push_semi(&mut _s);
                                                _s
                                            },
                                        );
                                        ::quote::__private::push_comma(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(
                                            &mut _s,
                                            "TagContentOtherField",
                                        );
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Tag");
                                        ::quote::__private::push_fat_arrow(&mut _s);
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Brace,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__rk");
                                                ::quote::__private::push_eq(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "_serde");
                                                ::quote::__private::push_colon2(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "__private");
                                                ::quote::__private::push_colon2(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "Some");
                                                ::quote::__private::push_group(
                                                    &mut _s,
                                                    ::quote::__private::Delimiter::Parenthesis,
                                                    {
                                                        let mut _s = ::quote::__private::TokenStream::new();
                                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "__private");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "de");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(
                                                            &mut _s,
                                                            "TagOrContentField",
                                                        );
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "Tag");
                                                        _s
                                                    },
                                                );
                                                ::quote::__private::push_semi(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "break");
                                                ::quote::__private::push_semi(&mut _s);
                                                _s
                                            },
                                        );
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(
                                            &mut _s,
                                            "TagContentOtherField",
                                        );
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_fat_arrow(&mut _s);
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Brace,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__rk");
                                                ::quote::__private::push_eq(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "_serde");
                                                ::quote::__private::push_colon2(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "__private");
                                                ::quote::__private::push_colon2(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "Some");
                                                ::quote::__private::push_group(
                                                    &mut _s,
                                                    ::quote::__private::Delimiter::Parenthesis,
                                                    {
                                                        let mut _s = ::quote::__private::TokenStream::new();
                                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "__private");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "de");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(
                                                            &mut _s,
                                                            "TagOrContentField",
                                                        );
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "Content");
                                                        _s
                                                    },
                                                );
                                                ::quote::__private::push_semi(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "break");
                                                ::quote::__private::push_semi(&mut _s);
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        ::quote::__private::push_ident(&mut _s, "__rk");
                        _s
                    },
                );
                _s
            }
        };
        let visit_remaining_keys = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "match");
            ::quote::ToTokens::to_tokens(&next_relevant_key, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Some");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "TagOrContentField");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Tag");
                            _s
                        },
                    );
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Err");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_lt(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__A");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Error");
                                    ::quote::__private::push_ident(&mut _s, "as");
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "de");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Error");
                                    ::quote::__private::push_gt(&mut _s);
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "duplicate_field");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Some");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "TagOrContentField");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Content");
                            _s
                        },
                    );
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Err");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_lt(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__A");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Error");
                                    ::quote::__private::push_ident(&mut _s, "as");
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "de");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Error");
                                    ::quote::__private::push_gt(&mut _s);
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "duplicate_field");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::ToTokens::to_tokens(&content, &mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "None");
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Ok");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__ret");
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            _s
        };
        let finish_content_then_tag = if variant_arms.is_empty() {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "match");
                ::quote::__private::push_ident(&mut _s, "try");
                ::quote::__private::push_bang(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "MapAccess");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "next_value");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__Field");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_and(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "mut");
                                ::quote::__private::push_ident(&mut _s, "__map");
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    ::quote::__private::TokenStream::new(),
                );
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "let");
                ::quote::__private::push_ident(&mut _s, "__ret");
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_ident(&mut _s, "try");
                ::quote::__private::push_bang(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::__private::push_ident(&mut _s, "try");
                        ::quote::__private::push_bang(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "de");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "MapAccess");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "next_value");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_and(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "mut");
                                        ::quote::__private::push_ident(&mut _s, "__map");
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut variant_arms, i) = variant_arms.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let variant_arms = match variant_arms.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        ::quote::ToTokens::to_tokens(&variant_arms, &mut _s);
                                    }
                                };
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                ::quote::ToTokens::to_tokens(&visit_remaining_keys, &mut _s);
                _s
            }
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&variant_visitor, &mut _s);
            ::quote::ToTokens::to_tokens(&variants_stmt, &mut _s);
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__Seed");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "field");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__Field");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "DeserializeSeed");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__Seed");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "deserialize");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__D");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserializer");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "match");
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_dot(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "field");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    {
                                        use ::quote::__private::ext::*;
                                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                        #[allow(unused_mut)]
                                        let (mut variant_arms, i) = variant_arms.quote_into_iter();
                                        let has_iter = has_iter | i;
                                        let _: ::quote::__private::HasIterator = has_iter;
                                        while true {
                                            let variant_arms = match variant_arms.next() {
                                                Some(_x) => ::quote::__private::RepInterp(_x),
                                                None => break,
                                            };
                                            ::quote::ToTokens::to_tokens(&variant_arms, &mut _s);
                                        }
                                    };
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Visitor");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__Visitor");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "expecting");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__formatter");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "fmt");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Formatter");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "write_str");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__formatter");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "visit_map");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__map");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__A");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "MapAccess");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "match");
                            ::quote::ToTokens::to_tokens(&next_relevant_key, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Some");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "de");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(
                                                &mut _s,
                                                "TagOrContentField",
                                            );
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Tag");
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_fat_arrow(&mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Brace,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "let");
                                            ::quote::__private::push_ident(&mut _s, "__field");
                                            ::quote::__private::push_eq(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "try");
                                            ::quote::__private::push_bang(&mut _s);
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "MapAccess");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "next_value");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_and(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "mut");
                                                            ::quote::__private::push_ident(&mut _s, "__map");
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            ::quote::__private::push_semi(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "match");
                                            ::quote::ToTokens::to_tokens(&next_relevant_key, &mut _s);
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Brace,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Some");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "de");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(
                                                                &mut _s,
                                                                "TagOrContentField",
                                                            );
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Tag");
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Brace,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Err");
                                                            ::quote::__private::push_group(
                                                                &mut _s,
                                                                ::quote::__private::Delimiter::Parenthesis,
                                                                {
                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                    ::quote::__private::push_lt(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "__A");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                                    ::quote::__private::push_ident(&mut _s, "as");
                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                                    ::quote::__private::push_gt(&mut _s);
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "duplicate_field");
                                                                    ::quote::__private::push_group(
                                                                        &mut _s,
                                                                        ::quote::__private::Delimiter::Parenthesis,
                                                                        {
                                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                                                            _s
                                                                        },
                                                                    );
                                                                    _s
                                                                },
                                                            );
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Some");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "de");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(
                                                                &mut _s,
                                                                "TagOrContentField",
                                                            );
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Content");
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Brace,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "let");
                                                            ::quote::__private::push_ident(&mut _s, "__ret");
                                                            ::quote::__private::push_eq(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "try");
                                                            ::quote::__private::push_bang(&mut _s);
                                                            ::quote::__private::push_group(
                                                                &mut _s,
                                                                ::quote::__private::Delimiter::Parenthesis,
                                                                {
                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "MapAccess");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "next_value_seed");
                                                                    ::quote::__private::push_group(
                                                                        &mut _s,
                                                                        ::quote::__private::Delimiter::Parenthesis,
                                                                        {
                                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                                            ::quote::__private::push_and(&mut _s);
                                                                            ::quote::__private::push_ident(&mut _s, "mut");
                                                                            ::quote::__private::push_ident(&mut _s, "__map");
                                                                            ::quote::__private::push_comma(&mut _s);
                                                                            ::quote::__private::push_ident(&mut _s, "__Seed");
                                                                            ::quote::__private::push_group(
                                                                                &mut _s,
                                                                                ::quote::__private::Delimiter::Brace,
                                                                                {
                                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                                    ::quote::__private::push_ident(&mut _s, "field");
                                                                                    ::quote::__private::push_colon(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "__field");
                                                                                    ::quote::__private::push_comma(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "marker");
                                                                                    ::quote::__private::push_colon(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                                                                                    ::quote::__private::push_comma(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "lifetime");
                                                                                    ::quote::__private::push_colon(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                                                                                    ::quote::__private::push_comma(&mut _s);
                                                                                    _s
                                                                                },
                                                                            );
                                                                            _s
                                                                        },
                                                                    );
                                                                    _s
                                                                },
                                                            );
                                                            ::quote::__private::push_semi(&mut _s);
                                                            ::quote::ToTokens::to_tokens(
                                                                &visit_remaining_keys,
                                                                &mut _s,
                                                            );
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "None");
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::ToTokens::to_tokens(&missing_content, &mut _s);
                                                    _s
                                                },
                                            );
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Some");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "de");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(
                                                &mut _s,
                                                "TagOrContentField",
                                            );
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Content");
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_fat_arrow(&mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Brace,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "let");
                                            ::quote::__private::push_ident(&mut _s, "__content");
                                            ::quote::__private::push_eq(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "try");
                                            ::quote::__private::push_bang(&mut _s);
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "MapAccess");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "next_value");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_lt(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Content");
                                                    ::quote::__private::push_gt(&mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_and(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "mut");
                                                            ::quote::__private::push_ident(&mut _s, "__map");
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            ::quote::__private::push_semi(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "match");
                                            ::quote::ToTokens::to_tokens(&next_relevant_key, &mut _s);
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Brace,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Some");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "de");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(
                                                                &mut _s,
                                                                "TagOrContentField",
                                                            );
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Tag");
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Brace,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "let");
                                                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                                                            ::quote::__private::push_eq(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "de");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(
                                                                &mut _s,
                                                                "ContentDeserializer",
                                                            );
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_lt(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__A");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Error");
                                                            ::quote::__private::push_gt(&mut _s);
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "new");
                                                            ::quote::__private::push_group(
                                                                &mut _s,
                                                                ::quote::__private::Delimiter::Parenthesis,
                                                                {
                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                    ::quote::__private::push_ident(&mut _s, "__content");
                                                                    _s
                                                                },
                                                            );
                                                            ::quote::__private::push_semi(&mut _s);
                                                            ::quote::ToTokens::to_tokens(
                                                                &finish_content_then_tag,
                                                                &mut _s,
                                                            );
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Some");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "de");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(
                                                                &mut _s,
                                                                "TagOrContentField",
                                                            );
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Content");
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Brace,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Err");
                                                            ::quote::__private::push_group(
                                                                &mut _s,
                                                                ::quote::__private::Delimiter::Parenthesis,
                                                                {
                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                    ::quote::__private::push_lt(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "__A");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                                    ::quote::__private::push_ident(&mut _s, "as");
                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                                    ::quote::__private::push_gt(&mut _s);
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "duplicate_field");
                                                                    ::quote::__private::push_group(
                                                                        &mut _s,
                                                                        ::quote::__private::Delimiter::Parenthesis,
                                                                        {
                                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                                            ::quote::ToTokens::to_tokens(&content, &mut _s);
                                                                            _s
                                                                        },
                                                                    );
                                                                    _s
                                                                },
                                                            );
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "None");
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Brace,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Err");
                                                            ::quote::__private::push_group(
                                                                &mut _s,
                                                                ::quote::__private::Delimiter::Parenthesis,
                                                                {
                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                    ::quote::__private::push_lt(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "__A");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                                    ::quote::__private::push_ident(&mut _s, "as");
                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                                    ::quote::__private::push_gt(&mut _s);
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "missing_field");
                                                                    ::quote::__private::push_group(
                                                                        &mut _s,
                                                                        ::quote::__private::Delimiter::Parenthesis,
                                                                        {
                                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                                                            _s
                                                                        },
                                                                    );
                                                                    _s
                                                                },
                                                            );
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "None");
                                    ::quote::__private::push_fat_arrow(&mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Brace,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Err");
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_lt(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__A");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                    ::quote::__private::push_ident(&mut _s, "as");
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                    ::quote::__private::push_gt(&mut _s);
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "missing_field");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "visit_seq");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__seq");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__A");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__A");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SeqAccess");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "match");
                            ::quote::__private::push_ident(&mut _s, "try");
                            ::quote::__private::push_bang(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "de");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "SeqAccess");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "next_element");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "mut");
                                            ::quote::__private::push_ident(&mut _s, "__seq");
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Some");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "__field");
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_fat_arrow(&mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Brace,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "match");
                                            ::quote::__private::push_ident(&mut _s, "try");
                                            ::quote::__private::push_bang(&mut _s);
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "SeqAccess");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(
                                                        &mut _s,
                                                        "next_element_seed",
                                                    );
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_and(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "mut");
                                                            ::quote::__private::push_ident(&mut _s, "__seq");
                                                            ::quote::__private::push_comma(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__Seed");
                                                            ::quote::__private::push_group(
                                                                &mut _s,
                                                                ::quote::__private::Delimiter::Brace,
                                                                {
                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                    ::quote::__private::push_ident(&mut _s, "field");
                                                                    ::quote::__private::push_colon(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "__field");
                                                                    ::quote::__private::push_comma(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "marker");
                                                                    ::quote::__private::push_colon(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                                                                    ::quote::__private::push_comma(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "lifetime");
                                                                    ::quote::__private::push_colon(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                                                                    ::quote::__private::push_comma(&mut _s);
                                                                    _s
                                                                },
                                                            );
                                                            ::quote::__private::push_comma(&mut _s);
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Brace,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Some");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "__ret");
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Ok");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "__ret");
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_comma(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "None");
                                                    ::quote::__private::push_fat_arrow(&mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Brace,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "__private");
                                                            ::quote::__private::push_colon2(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "Err");
                                                            ::quote::__private::push_group(
                                                                &mut _s,
                                                                ::quote::__private::Delimiter::Parenthesis,
                                                                {
                                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                                    ::quote::__private::push_colon2(&mut _s);
                                                                    ::quote::__private::push_ident(&mut _s, "invalid_length");
                                                                    ::quote::__private::push_group(
                                                                        &mut _s,
                                                                        ::quote::__private::Delimiter::Parenthesis,
                                                                        {
                                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                                            ::quote::__private::parse(&mut _s, "1");
                                                                            ::quote::__private::push_comma(&mut _s);
                                                                            ::quote::__private::push_and(&mut _s);
                                                                            ::quote::__private::push_ident(&mut _s, "self");
                                                                            _s
                                                                        },
                                                                    );
                                                                    _s
                                                                },
                                                            );
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "None");
                                    ::quote::__private::push_fat_arrow(&mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Brace,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Err");
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "de");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Error");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "invalid_length");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::parse(&mut _s, "0");
                                                            ::quote::__private::push_comma(&mut _s);
                                                            ::quote::__private::push_and(&mut _s);
                                                            ::quote::__private::push_ident(&mut _s, "self");
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "const");
            ::quote::__private::push_ident(&mut _s, "FIELDS");
            ::quote::__private::push_colon(&mut _s);
            ::quote::__private::push_and(&mut _s);
            ::quote::__private::push_lifetime(&mut _s, "\'static");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Bracket,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_and(&mut _s);
                    ::quote::__private::push_lifetime(&mut _s, "\'static");
                    ::quote::__private::push_ident(&mut _s, "str");
                    _s
                },
            );
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_and(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Bracket,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&tag, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::ToTokens::to_tokens(&content, &mut _s);
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Deserializer");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "deserialize_struct");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__deserializer");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "FIELDS");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__Visitor");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "marker");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_lt(&mut _s);
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                            ::quote::__private::push_gt(&mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "lifetime");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                            ::quote::__private::push_comma(&mut _s);
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_untagged_enum(
        params: &Parameters,
        variants: &[Variant],
        cattrs: &attr::Container,
    ) -> Fragment {
        let attempts = variants
            .iter()
            .filter(|variant| !variant.attrs.skip_deserializing())
            .map(|variant| {
                Expr(
                    deserialize_untagged_variant(
                        params,
                        variant,
                        cattrs,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "ContentRefDeserializer",
                            );
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_lt(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__D");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Error");
                            ::quote::__private::push_gt(&mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "new");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_and(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__content");
                                    _s
                                },
                            );
                            _s
                        },
                    ),
                )
            });
        let fallthrough_msg = {
            let res = ::alloc::fmt::format(
                ::core::fmt::Arguments::new_v1(
                    &["data did not match any variant of untagged enum "],
                    &[::core::fmt::ArgumentV1::new_display(&params.type_name())],
                ),
            );
            res
        };
        let fallthrough_msg = cattrs.expecting().unwrap_or(&fallthrough_msg);
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "let");
            ::quote::__private::push_ident(&mut _s, "__content");
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Content");
                    ::quote::__private::push_ident(&mut _s, "as");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserialize");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "deserialize");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut attempts, i) = attempts.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let attempts = match attempts.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::__private::push_ident(&mut _s, "if");
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Ok");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__ok");
                            _s
                        },
                    );
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&attempts, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "return");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Ok");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__ok");
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            _s
                        },
                    );
                }
            };
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Err");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "custom");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&fallthrough_msg, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_externally_tagged_variant(
        params: &Parameters,
        variant: &Variant,
        cattrs: &attr::Container,
    ) -> Fragment {
        if let Some(path) = variant.attrs.deserialize_with() {
            let (wrapper, wrapper_ty, unwrap_fn) = wrap_deserialize_variant_with(
                params,
                variant,
                path,
            );
            return crate::fragment::Fragment::Block({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "map");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "VariantAccess");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "newtype_variant");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::ToTokens::to_tokens(&wrapper_ty, &mut _s);
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__variant");
                                _s
                            },
                        );
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&unwrap_fn, &mut _s);
                        _s
                    },
                );
                _s
            });
        }
        let variant_ident = &variant.ident;
        match variant.style {
            Style::Unit => {
                let this = &params.this;
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "VariantAccess");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "unit_variant");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__variant");
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Ok");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Newtype => {
                deserialize_externally_tagged_newtype_variant(
                    variant_ident,
                    params,
                    &variant.fields[0],
                    cattrs,
                )
            }
            Style::Tuple => {
                deserialize_tuple(
                    Some(variant_ident),
                    params,
                    &variant.fields,
                    cattrs,
                    None,
                )
            }
            Style::Struct => {
                deserialize_struct(
                    Some(variant_ident),
                    params,
                    &variant.fields,
                    cattrs,
                    None,
                    &Untagged::No,
                )
            }
        }
    }
    fn deserialize_internally_tagged_variant(
        params: &Parameters,
        variant: &Variant,
        cattrs: &attr::Container,
        deserializer: TokenStream,
    ) -> Fragment {
        if variant.attrs.deserialize_with().is_some() {
            return deserialize_untagged_variant(params, variant, cattrs, deserializer);
        }
        let variant_ident = &variant.ident;
        match effective_style(variant) {
            Style::Unit => {
                let this = &params.this;
                let type_name = params.type_name();
                let variant_name = variant.ident.to_string();
                let default = variant
                    .fields
                    .get(0)
                    .map(|field| {
                        let default = Expr(expr_is_missing(field, cattrs));
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&default, &mut _s);
                                    _s
                                },
                            );
                            _s
                        }
                    });
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Deserializer");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "deserialize_any");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&deserializer, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "de");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(
                                        &mut _s,
                                        "InternallyTaggedUnitVisitor",
                                    );
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "new");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Ok");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                            ::quote::ToTokens::to_tokens(&default, &mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Newtype => {
                deserialize_untagged_newtype_variant(
                    variant_ident,
                    params,
                    &variant.fields[0],
                    &deserializer,
                )
            }
            Style::Struct => {
                deserialize_struct(
                    Some(variant_ident),
                    params,
                    &variant.fields,
                    cattrs,
                    Some(deserializer),
                    &Untagged::No,
                )
            }
            Style::Tuple => {
                ::core::panicking::unreachable_display(
                    &"checked in serde_derive_internals",
                )
            }
        }
    }
    fn deserialize_untagged_variant(
        params: &Parameters,
        variant: &Variant,
        cattrs: &attr::Container,
        deserializer: TokenStream,
    ) -> Fragment {
        if let Some(path) = variant.attrs.deserialize_with() {
            let unwrap_fn = unwrap_to_variant_closure(params, variant, false);
            return crate::fragment::Fragment::Block({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "map");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&path, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&deserializer, &mut _s);
                                _s
                            },
                        );
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&unwrap_fn, &mut _s);
                        _s
                    },
                );
                _s
            });
        }
        let variant_ident = &variant.ident;
        match effective_style(variant) {
            Style::Unit => {
                let this = &params.this;
                let type_name = params.type_name();
                let variant_name = variant.ident.to_string();
                let default = variant
                    .fields
                    .get(0)
                    .map(|field| {
                        let default = Expr(expr_is_missing(field, cattrs));
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&default, &mut _s);
                                    _s
                                },
                            );
                            _s
                        }
                    });
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "match");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "deserialize_any");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&deserializer, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "UntaggedUnitVisitor",
                            );
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "new");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Ok");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        ::quote::__private::TokenStream::new(),
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Ok");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                                    ::quote::ToTokens::to_tokens(&default, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Err");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__err");
                                    _s
                                },
                            );
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Err");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__err");
                                    _s
                                },
                            );
                            ::quote::__private::push_comma(&mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Newtype => {
                deserialize_untagged_newtype_variant(
                    variant_ident,
                    params,
                    &variant.fields[0],
                    &deserializer,
                )
            }
            Style::Tuple => {
                deserialize_tuple(
                    Some(variant_ident),
                    params,
                    &variant.fields,
                    cattrs,
                    Some(deserializer),
                )
            }
            Style::Struct => {
                deserialize_struct(
                    Some(variant_ident),
                    params,
                    &variant.fields,
                    cattrs,
                    Some(deserializer),
                    &Untagged::Yes,
                )
            }
        }
    }
    fn deserialize_externally_tagged_newtype_variant(
        variant_ident: &syn::Ident,
        params: &Parameters,
        field: &Field,
        cattrs: &attr::Container,
    ) -> Fragment {
        let this = &params.this;
        if field.attrs.skip_deserializing() {
            let this = &params.this;
            let default = Expr(expr_is_missing(field, cattrs));
            return crate::fragment::Fragment::Block({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "try");
                ::quote::__private::push_bang(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "VariantAccess");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "unit_variant");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__variant");
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Ok");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&this, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&default, &mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            });
        }
        match field.attrs.deserialize_with() {
            None => {
                let field_ty = field.ty;
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "de");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "VariantAccess",
                    );
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "newtype_variant",
                    );
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_lt_spanned(&mut _s, _span);
                    ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                    ::quote::__private::push_gt_spanned(&mut _s, _span);
                    _s
                };
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "map");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&func, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__variant");
                                    _s
                                },
                            );
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Some(path) => {
                let (wrapper, wrapper_ty) = wrap_deserialize_field_with(
                    params,
                    field.ty,
                    path,
                );
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "map");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "VariantAccess");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "newtype_variant");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_lt(&mut _s);
                            ::quote::ToTokens::to_tokens(&wrapper_ty, &mut _s);
                            ::quote::__private::push_gt(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__variant");
                                    _s
                                },
                            );
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_or(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__wrapper");
                            ::quote::__private::push_or(&mut _s);
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__wrapper");
                                    ::quote::__private::push_dot(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "value");
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                })
            }
        }
    }
    fn deserialize_untagged_newtype_variant(
        variant_ident: &syn::Ident,
        params: &Parameters,
        field: &Field,
        deserializer: &TokenStream,
    ) -> Fragment {
        let this = &params.this;
        let field_ty = field.ty;
        match field.attrs.deserialize_with() {
            None => {
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_lt_spanned(&mut _s, _span);
                    ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "as");
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "Deserialize",
                    );
                    ::quote::__private::push_gt_spanned(&mut _s, _span);
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "deserialize",
                    );
                    _s
                };
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "map");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&func, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&deserializer, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Some(path) => {
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_underscore(&mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&path, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&deserializer, &mut _s);
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "map");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__value");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
        }
    }
    fn deserialize_generated_identifier(
        fields: &[(String, Ident, Vec<String>)],
        cattrs: &attr::Container,
        is_variant: bool,
        other_idx: Option<usize>,
    ) -> Fragment {
        let this = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "__Field");
            _s
        };
        let field_idents: &Vec<_> = &fields.iter().map(|(_, ident, _)| ident).collect();
        let (ignore_variant, fallthrough) = if !is_variant && cattrs.has_flatten() {
            let ignore_variant = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "__other");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Content");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'de");
                        ::quote::__private::push_gt(&mut _s);
                        _s
                    },
                );
                ::quote::__private::push_comma(&mut _s);
                _s
            };
            let fallthrough = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Ok");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__Field");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__other");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__value");
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            };
            (Some(ignore_variant), Some(fallthrough))
        } else if let Some(other_idx) = other_idx {
            let ignore_variant = fields[other_idx].1.clone();
            let fallthrough = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Ok");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__Field");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::ToTokens::to_tokens(&ignore_variant, &mut _s);
                        _s
                    },
                );
                _s
            };
            (None, Some(fallthrough))
        } else if is_variant || cattrs.deny_unknown_fields() {
            (None, None)
        } else {
            let ignore_variant = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "__ignore");
                ::quote::__private::push_comma(&mut _s);
                _s
            };
            let fallthrough = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Ok");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__Field");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__ignore");
                        _s
                    },
                );
                _s
            };
            (Some(ignore_variant), Some(fallthrough))
        };
        let visitor_impl = Stmts(
            deserialize_identifier(
                &this,
                fields,
                is_variant,
                fallthrough,
                None,
                !is_variant && cattrs.has_flatten(),
                None,
            ),
        );
        let lifetime = if !is_variant && cattrs.has_flatten() {
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_lifetime(&mut _s, "\'de");
                ::quote::__private::push_gt(&mut _s);
                _s
            })
        } else {
            None
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_pound(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Bracket,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "allow");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(
                                &mut _s,
                                "non_camel_case_types",
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "enum");
            ::quote::__private::push_ident(&mut _s, "__Field");
            ::quote::ToTokens::to_tokens(&lifetime, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut field_idents, i) = field_idents.quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let field_idents = match field_idents.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::ToTokens::to_tokens(&field_idents, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                        }
                    };
                    ::quote::ToTokens::to_tokens(&ignore_variant, &mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__FieldVisitor");
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_lifetime(&mut _s, "\'de");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Visitor");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_lifetime(&mut _s, "\'de");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__FieldVisitor");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__Field");
                    ::quote::ToTokens::to_tokens(&lifetime, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::ToTokens::to_tokens(&visitor_impl, &mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_lifetime(&mut _s, "\'de");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Deserialize");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_lifetime(&mut _s, "\'de");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__Field");
            ::quote::ToTokens::to_tokens(&lifetime, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_pound(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Bracket,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "inline");
                            _s
                        },
                    );
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "deserialize");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__D");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserializer");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_lifetime(&mut _s, "\'de");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Deserializer");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "deserialize_identifier",
                            );
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__deserializer");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__FieldVisitor");
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_custom_identifier(
        params: &Parameters,
        variants: &[Variant],
        cattrs: &attr::Container,
    ) -> Fragment {
        let is_variant = match cattrs.identifier() {
            attr::Identifier::Variant => true,
            attr::Identifier::Field => false,
            attr::Identifier::No => {
                ::core::panicking::panic("internal error: entered unreachable code")
            }
        };
        let this = &params.this;
        let this = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&this, &mut _s);
            _s
        };
        let (ordinary, fallthrough, fallthrough_borrowed) = if let Some(last) = variants
            .last()
        {
            let last_ident = &last.ident;
            if last.attrs.other() {
                let ordinary = &variants[..variants.len() - 1];
                let fallthrough = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Ok");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&this, &mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::ToTokens::to_tokens(&last_ident, &mut _s);
                            _s
                        },
                    );
                    _s
                };
                (ordinary, Some(fallthrough), None)
            } else if let Style::Newtype = last.style {
                let ordinary = &variants[..variants.len() - 1];
                let fallthrough = |value| {
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Result");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "map");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Deserialize");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "deserialize");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(
                                            &mut _s,
                                            "IdentifierDeserializer",
                                        );
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "from");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::ToTokens::to_tokens(&value, &mut _s);
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::ToTokens::to_tokens(&this, &mut _s);
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::ToTokens::to_tokens(&last_ident, &mut _s);
                                _s
                            },
                        );
                        _s
                    }
                };
                (
                    ordinary,
                    Some(
                        fallthrough({
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__value");
                            _s
                        }),
                    ),
                    Some(
                        fallthrough({
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Borrowed");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__value");
                                    _s
                                },
                            );
                            _s
                        }),
                    ),
                )
            } else {
                (variants, None, None)
            }
        } else {
            (variants, None, None)
        };
        let names_idents: Vec<_> = ordinary
            .iter()
            .map(|variant| {
                (
                    variant.attrs.name().deserialize_name(),
                    variant.ident.clone(),
                    variant.attrs.aliases(),
                )
            })
            .collect();
        let names = names_idents.iter().map(|(name, _, _)| name);
        let names_const = if fallthrough.is_some() {
            None
        } else if is_variant {
            let variants = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "const");
                ::quote::__private::push_ident(&mut _s, "VARIANTS");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_lifetime(&mut _s, "\'static");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_and(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'static");
                        ::quote::__private::push_ident(&mut _s, "str");
                        _s
                    },
                );
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut names, i) = names.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let names = match names.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&names, &mut _s);
                            }
                        };
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                _s
            };
            Some(variants)
        } else {
            let fields = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "const");
                ::quote::__private::push_ident(&mut _s, "FIELDS");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_lifetime(&mut _s, "\'static");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_and(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'static");
                        ::quote::__private::push_ident(&mut _s, "str");
                        _s
                    },
                );
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut names, i) = names.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let names = match names.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&names, &mut _s);
                            }
                        };
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                _s
            };
            Some(fields)
        };
        let (de_impl_generics, de_ty_generics, ty_generics, where_clause) = split_with_de_lifetime(
            params,
        );
        let delife = params.borrowed.de_lifetime();
        let visitor_impl = Stmts(
            deserialize_identifier(
                &this,
                &names_idents,
                is_variant,
                fallthrough,
                fallthrough_borrowed,
                false,
                cattrs.expecting(),
            ),
        );
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&names_const, &mut _s);
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__FieldVisitor");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Visitor");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__FieldVisitor");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "type");
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::ToTokens::to_tokens(&visitor_impl, &mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "let");
            ::quote::__private::push_ident(&mut _s, "__visitor");
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__FieldVisitor");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "marker");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Deserializer");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "deserialize_identifier");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__deserializer");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__visitor");
                    _s
                },
            );
            _s
        })
    }
    fn deserialize_identifier(
        this: &TokenStream,
        fields: &[(String, Ident, Vec<String>)],
        is_variant: bool,
        fallthrough: Option<TokenStream>,
        fallthrough_borrowed: Option<TokenStream>,
        collect_other_fields: bool,
        expecting: Option<&str>,
    ) -> Fragment {
        let mut flat_fields = Vec::new();
        for (_, ident, aliases) in fields {
            flat_fields.extend(aliases.iter().map(|alias| (alias, ident)));
        }
        let field_strs: &Vec<_> = &flat_fields.iter().map(|(name, _)| name).collect();
        let field_bytes: &Vec<_> = &flat_fields
            .iter()
            .map(|(name, _)| Literal::byte_string(name.as_bytes()))
            .collect();
        let constructors: &Vec<_> = &flat_fields
            .iter()
            .map(|(_, ident)| {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&ident, &mut _s);
                _s
            })
            .collect();
        let main_constructors: &Vec<_> = &fields
            .iter()
            .map(|(_, ident, _)| {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&ident, &mut _s);
                _s
            })
            .collect();
        let expecting = expecting
            .unwrap_or(
                if is_variant { "variant identifier" } else { "field identifier" },
            );
        let index_expecting = if is_variant { "variant" } else { "field" };
        let bytes_to_str = if fallthrough.is_some() || collect_other_fields {
            None
        } else {
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "let");
                ::quote::__private::push_ident(&mut _s, "__value");
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "from_utf8_lossy");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__value");
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                _s
            })
        };
        let (
            value_as_str_content,
            value_as_borrowed_str_content,
            value_as_bytes_content,
            value_as_borrowed_bytes_content,
        ) = if collect_other_fields {
            (
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Content");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "String");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "ToString");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "to_string");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__value");
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }),
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Content");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Str");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__value");
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }),
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Content");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ByteBuf");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__value");
                            ::quote::__private::push_dot(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "to_vec");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                ::quote::__private::TokenStream::new(),
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }),
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "de");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Content");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Bytes");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__value");
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }),
            )
        } else {
            (None, None, None, None)
        };
        let fallthrough_arm_tokens;
        let fallthrough_arm = if let Some(fallthrough) = &fallthrough {
            fallthrough
        } else if is_variant {
            fallthrough_arm_tokens = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Err");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "unknown_variant");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__value");
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "VARIANTS");
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            };
            &fallthrough_arm_tokens
        } else {
            fallthrough_arm_tokens = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Err");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "unknown_field");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__value");
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "FIELDS");
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            };
            &fallthrough_arm_tokens
        };
        let u64_fallthrough_arm_tokens;
        let u64_fallthrough_arm = if let Some(fallthrough) = &fallthrough {
            fallthrough
        } else {
            let fallthrough_msg = {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["", " index 0 <= i < "],
                        &[
                            ::core::fmt::ArgumentV1::new_display(&index_expecting),
                            ::core::fmt::ArgumentV1::new_display(&fields.len()),
                        ],
                    ),
                );
                res
            };
            u64_fallthrough_arm_tokens = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Err");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "invalid_value");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "de");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Unexpected");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Unsigned");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "__value");
                                        _s
                                    },
                                );
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_and(&mut _s);
                                ::quote::ToTokens::to_tokens(&fallthrough_msg, &mut _s);
                                ::quote::__private::push_comma(&mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            };
            &u64_fallthrough_arm_tokens
        };
        let variant_indices = 0_u64..;
        let visit_other = if collect_other_fields {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_bool");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "bool");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Bool");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_i8");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "i8");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "I8");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_i16");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "i16");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "I16");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_i32");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "i32");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "I32");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_i64");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "i64");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "I64");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_u8");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "u8");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "U8");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_u16");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "u16");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "U16");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_u32");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "u32");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "U32");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_u64");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "u64");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "U64");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_f32");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "f32");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "F32");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_f64");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "f64");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "F64");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_char");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "char");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Char");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__value");
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_unit");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__other");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Content");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Unit");
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_u64");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "u64");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut variant_indices, i) = variant_indices
                                        .quote_into_iter();
                                    let has_iter = has_iter | i;
                                    #[allow(unused_mut)]
                                    let (mut main_constructors, i) = main_constructors
                                        .quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let variant_indices = match variant_indices.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        let main_constructors = match main_constructors.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        ::quote::ToTokens::to_tokens(&variant_indices, &mut _s);
                                        ::quote::__private::push_fat_arrow(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Ok");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::ToTokens::to_tokens(&main_constructors, &mut _s);
                                                _s
                                            },
                                        );
                                        ::quote::__private::push_comma(&mut _s);
                                    }
                                };
                                ::quote::__private::push_underscore(&mut _s);
                                ::quote::__private::push_fat_arrow(&mut _s);
                                ::quote::ToTokens::to_tokens(&u64_fallthrough_arm, &mut _s);
                                ::quote::__private::push_comma(&mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            }
        };
        let visit_borrowed = if fallthrough_borrowed.is_some() || collect_other_fields {
            let fallthrough_borrowed_arm = fallthrough_borrowed
                .as_ref()
                .unwrap_or(fallthrough_arm);
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_borrowed_str");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_and(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'de");
                        ::quote::__private::push_ident(&mut _s, "str");
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut field_strs, i) = field_strs.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    #[allow(unused_mut)]
                                    let (mut constructors, i) = constructors.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let field_strs = match field_strs.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        let constructors = match constructors.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        ::quote::ToTokens::to_tokens(&field_strs, &mut _s);
                                        ::quote::__private::push_fat_arrow(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Ok");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::ToTokens::to_tokens(&constructors, &mut _s);
                                                _s
                                            },
                                        );
                                        ::quote::__private::push_comma(&mut _s);
                                    }
                                };
                                ::quote::__private::push_underscore(&mut _s);
                                ::quote::__private::push_fat_arrow(&mut _s);
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Brace,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::ToTokens::to_tokens(
                                            &value_as_borrowed_str_content,
                                            &mut _s,
                                        );
                                        ::quote::ToTokens::to_tokens(
                                            &fallthrough_borrowed_arm,
                                            &mut _s,
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "fn");
                ::quote::__private::push_ident(&mut _s, "visit_borrowed_bytes");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "self");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::__private::push_and(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'de");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Bracket,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "u8");
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_rarrow(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Result");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Self");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Value");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "where");
                ::quote::__private::push_ident(&mut _s, "__E");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "de");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Error");
                ::quote::__private::push_comma(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::__private::push_ident(&mut _s, "__value");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut field_bytes, i) = field_bytes.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    #[allow(unused_mut)]
                                    let (mut constructors, i) = constructors.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let field_bytes = match field_bytes.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        let constructors = match constructors.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        ::quote::ToTokens::to_tokens(&field_bytes, &mut _s);
                                        ::quote::__private::push_fat_arrow(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Ok");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::ToTokens::to_tokens(&constructors, &mut _s);
                                                _s
                                            },
                                        );
                                        ::quote::__private::push_comma(&mut _s);
                                    }
                                };
                                ::quote::__private::push_underscore(&mut _s);
                                ::quote::__private::push_fat_arrow(&mut _s);
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Brace,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::ToTokens::to_tokens(&bytes_to_str, &mut _s);
                                        ::quote::ToTokens::to_tokens(
                                            &value_as_borrowed_bytes_content,
                                            &mut _s,
                                        );
                                        ::quote::ToTokens::to_tokens(
                                            &fallthrough_borrowed_arm,
                                            &mut _s,
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            })
        } else {
            None
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "fn");
            ::quote::__private::push_ident(&mut _s, "expecting");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_and(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "self");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__formatter");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "mut");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Formatter");
                    _s
                },
            );
            ::quote::__private::push_rarrow(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "fmt");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Result");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Formatter");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "write_str");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__formatter");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&expecting, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::ToTokens::to_tokens(&visit_other, &mut _s);
            ::quote::__private::push_ident(&mut _s, "fn");
            ::quote::__private::push_ident(&mut _s, "visit_str");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "self");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "str");
                    _s
                },
            );
            ::quote::__private::push_rarrow(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Result");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Self");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Value");
            ::quote::__private::push_comma(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "where");
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_colon(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Error");
            ::quote::__private::push_comma(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "match");
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            {
                                use ::quote::__private::ext::*;
                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                #[allow(unused_mut)]
                                let (mut field_strs, i) = field_strs.quote_into_iter();
                                let has_iter = has_iter | i;
                                #[allow(unused_mut)]
                                let (mut constructors, i) = constructors.quote_into_iter();
                                let has_iter = has_iter | i;
                                let _: ::quote::__private::HasIterator = has_iter;
                                while true {
                                    let field_strs = match field_strs.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    let constructors = match constructors.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    ::quote::ToTokens::to_tokens(&field_strs, &mut _s);
                                    ::quote::__private::push_fat_arrow(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Ok");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::ToTokens::to_tokens(&constructors, &mut _s);
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_comma(&mut _s);
                                }
                            };
                            ::quote::__private::push_underscore(&mut _s);
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(
                                        &value_as_str_content,
                                        &mut _s,
                                    );
                                    ::quote::ToTokens::to_tokens(&fallthrough_arm, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "fn");
            ::quote::__private::push_ident(&mut _s, "visit_bytes");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "self");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Bracket,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "u8");
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_rarrow(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Result");
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Self");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Value");
            ::quote::__private::push_comma(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "where");
            ::quote::__private::push_ident(&mut _s, "__E");
            ::quote::__private::push_colon(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "de");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Error");
            ::quote::__private::push_comma(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "match");
                    ::quote::__private::push_ident(&mut _s, "__value");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            {
                                use ::quote::__private::ext::*;
                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                #[allow(unused_mut)]
                                let (mut field_bytes, i) = field_bytes.quote_into_iter();
                                let has_iter = has_iter | i;
                                #[allow(unused_mut)]
                                let (mut constructors, i) = constructors.quote_into_iter();
                                let has_iter = has_iter | i;
                                let _: ::quote::__private::HasIterator = has_iter;
                                while true {
                                    let field_bytes = match field_bytes.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    let constructors = match constructors.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    ::quote::ToTokens::to_tokens(&field_bytes, &mut _s);
                                    ::quote::__private::push_fat_arrow(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Ok");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::ToTokens::to_tokens(&constructors, &mut _s);
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_comma(&mut _s);
                                }
                            };
                            ::quote::__private::push_underscore(&mut _s);
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&bytes_to_str, &mut _s);
                                    ::quote::ToTokens::to_tokens(
                                        &value_as_bytes_content,
                                        &mut _s,
                                    );
                                    ::quote::ToTokens::to_tokens(&fallthrough_arm, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::ToTokens::to_tokens(&visit_borrowed, &mut _s);
            _s
        })
    }
    fn deserialize_struct_as_struct_visitor(
        struct_path: &TokenStream,
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
    ) -> (Fragment, Option<Fragment>, Fragment) {
        if !!cattrs.has_flatten() {
            ::core::panicking::panic("assertion failed: !cattrs.has_flatten()")
        }
        let field_names_idents: Vec<_> = fields
            .iter()
            .enumerate()
            .filter(|&(_, field)| !field.attrs.skip_deserializing())
            .map(|(i, field)| {
                (
                    field.attrs.name().deserialize_name(),
                    field_i(i),
                    field.attrs.aliases(),
                )
            })
            .collect();
        let fields_stmt = {
            let field_names = field_names_idents.iter().map(|(name, _, _)| name);
            crate::fragment::Fragment::Block({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "const");
                ::quote::__private::push_ident(&mut _s, "FIELDS");
                ::quote::__private::push_colon(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_lifetime(&mut _s, "\'static");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_and(&mut _s);
                        ::quote::__private::push_lifetime(&mut _s, "\'static");
                        ::quote::__private::push_ident(&mut _s, "str");
                        _s
                    },
                );
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut field_names, i) = field_names.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let field_names = match field_names.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&field_names, &mut _s);
                            }
                        };
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                _s
            })
        };
        let field_visitor = deserialize_generated_identifier(
            &field_names_idents,
            cattrs,
            false,
            None,
        );
        let visit_map = deserialize_map(struct_path, params, fields, cattrs);
        (field_visitor, Some(fields_stmt), visit_map)
    }
    fn deserialize_struct_as_map_visitor(
        struct_path: &TokenStream,
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
    ) -> (Fragment, Option<Fragment>, Fragment) {
        let field_names_idents: Vec<_> = fields
            .iter()
            .enumerate()
            .filter(|&(_, field)| {
                !field.attrs.skip_deserializing() && !field.attrs.flatten()
            })
            .map(|(i, field)| {
                (
                    field.attrs.name().deserialize_name(),
                    field_i(i),
                    field.attrs.aliases(),
                )
            })
            .collect();
        let field_visitor = deserialize_generated_identifier(
            &field_names_idents,
            cattrs,
            false,
            None,
        );
        let visit_map = deserialize_map(struct_path, params, fields, cattrs);
        (field_visitor, None, visit_map)
    }
    fn deserialize_map(
        struct_path: &TokenStream,
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
    ) -> Fragment {
        let fields_names: Vec<_> = fields
            .iter()
            .enumerate()
            .map(|(i, field)| (field, field_i(i)))
            .collect();
        let let_values = fields_names
            .iter()
            .filter(|&&(field, _)| {
                !field.attrs.skip_deserializing() && !field.attrs.flatten()
            })
            .map(|(field, name)| {
                let field_ty = field.ty;
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "mut");
                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Option");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "None");
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }
            });
        let let_collect = if cattrs.has_flatten() {
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "let");
                ::quote::__private::push_ident(&mut _s, "mut");
                ::quote::__private::push_ident(&mut _s, "__collect");
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Vec");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Option");
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Content");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Content");
                        _s
                    },
                );
                ::quote::__private::push_shr(&mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "new");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    ::quote::__private::TokenStream::new(),
                );
                ::quote::__private::push_semi(&mut _s);
                _s
            })
        } else {
            None
        };
        let value_arms = fields_names
            .iter()
            .filter(|&&(field, _)| {
                !field.attrs.skip_deserializing() && !field.attrs.flatten()
            })
            .map(|(field, name)| {
                let deser_name = field.attrs.name().deserialize_name();
                let visit = match field.attrs.deserialize_with() {
                    None => {
                        let field_ty = field.ty;
                        let span = field.original.span();
                        let func = {
                            let mut _s = ::quote::__private::TokenStream::new();
                            let _span: ::quote::__private::Span = span;
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "_serde",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(&mut _s, _span, "de");
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "MapAccess",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "next_value",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_lt_spanned(&mut _s, _span);
                            ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                            ::quote::__private::push_gt_spanned(&mut _s, _span);
                            _s
                        };
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "try");
                            ::quote::__private::push_bang(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&func, &mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "mut");
                                            ::quote::__private::push_ident(&mut _s, "__map");
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        }
                    }
                    Some(path) => {
                        let (wrapper, wrapper_ty) = wrap_deserialize_field_with(
                            params,
                            field.ty,
                            path,
                        );
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                                    ::quote::__private::push_ident(&mut _s, "match");
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "de");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "MapAccess");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "next_value");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_lt(&mut _s);
                                    ::quote::ToTokens::to_tokens(&wrapper_ty, &mut _s);
                                    ::quote::__private::push_gt(&mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "mut");
                                            ::quote::__private::push_ident(&mut _s, "__map");
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Brace,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Ok");
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "__wrapper");
                                                    _s
                                                },
                                            );
                                            ::quote::__private::push_fat_arrow(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__wrapper");
                                            ::quote::__private::push_dot(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "value");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Err");
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "__err");
                                                    _s
                                                },
                                            );
                                            ::quote::__private::push_fat_arrow(&mut _s);
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Brace,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::__private::push_ident(&mut _s, "return");
                                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "__private");
                                                    ::quote::__private::push_colon2(&mut _s);
                                                    ::quote::__private::push_ident(&mut _s, "Err");
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "__err");
                                                            _s
                                                        },
                                                    );
                                                    ::quote::__private::push_semi(&mut _s);
                                                    _s
                                                },
                                            );
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        }
                    }
                };
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__Field");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "if");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Option");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "is_some");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_and(&mut _s);
                                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "return");
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Err");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_lt(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__A");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Error");
                                            ::quote::__private::push_ident(&mut _s, "as");
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "de");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "Error");
                                            ::quote::__private::push_gt(&mut _s);
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "duplicate_field");
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::ToTokens::to_tokens(&deser_name, &mut _s);
                                                    _s
                                                },
                                            );
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_semi(&mut _s);
                                    _s
                                },
                            );
                            ::quote::ToTokens::to_tokens(&name, &mut _s);
                            ::quote::__private::push_eq(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Some");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&visit, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            _s
                        },
                    );
                    _s
                }
            });
        let ignored_arm = if cattrs.has_flatten() {
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "__Field");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__other");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__name");
                        _s
                    },
                );
                ::quote::__private::push_fat_arrow(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__collect");
                        ::quote::__private::push_dot(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "push");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__private");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Some");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__name");
                                                ::quote::__private::push_comma(&mut _s);
                                                ::quote::__private::push_ident(&mut _s, "try");
                                                ::quote::__private::push_bang(&mut _s);
                                                ::quote::__private::push_group(
                                                    &mut _s,
                                                    ::quote::__private::Delimiter::Parenthesis,
                                                    {
                                                        let mut _s = ::quote::__private::TokenStream::new();
                                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "de");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "MapAccess");
                                                        ::quote::__private::push_colon2(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "next_value");
                                                        ::quote::__private::push_group(
                                                            &mut _s,
                                                            ::quote::__private::Delimiter::Parenthesis,
                                                            {
                                                                let mut _s = ::quote::__private::TokenStream::new();
                                                                ::quote::__private::push_and(&mut _s);
                                                                ::quote::__private::push_ident(&mut _s, "mut");
                                                                ::quote::__private::push_ident(&mut _s, "__map");
                                                                _s
                                                            },
                                                        );
                                                        _s
                                                    },
                                                );
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        ::quote::__private::push_semi(&mut _s);
                        _s
                    },
                );
                _s
            })
        } else if cattrs.deny_unknown_fields() {
            None
        } else {
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_underscore(&mut _s);
                ::quote::__private::push_fat_arrow(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "let");
                        ::quote::__private::push_underscore(&mut _s);
                        ::quote::__private::push_eq(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "try");
                        ::quote::__private::push_bang(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "de");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "MapAccess");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "next_value");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_lt(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "de");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "IgnoredAny");
                                ::quote::__private::push_gt(&mut _s);
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_and(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "mut");
                                        ::quote::__private::push_ident(&mut _s, "__map");
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        ::quote::__private::push_semi(&mut _s);
                        _s
                    },
                );
                _s
            })
        };
        let all_skipped = fields.iter().all(|field| field.attrs.skip_deserializing());
        let match_keys = if cattrs.deny_unknown_fields() && all_skipped {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Option");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "map");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "try");
                        ::quote::__private::push_bang(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "de");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "MapAccess");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "next_key");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_lt(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__Field");
                                ::quote::__private::push_gt(&mut _s);
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_and(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "mut");
                                        ::quote::__private::push_ident(&mut _s, "__map");
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_or(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__impossible");
                        ::quote::__private::push_or(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::__private::push_ident(&mut _s, "__impossible");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            ::quote::__private::TokenStream::new(),
                        );
                        _s
                    },
                );
                ::quote::__private::push_semi(&mut _s);
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "while");
                ::quote::__private::push_ident(&mut _s, "let");
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Some");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__key");
                        _s
                    },
                );
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_ident(&mut _s, "try");
                ::quote::__private::push_bang(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "de");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "MapAccess");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "next_key");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__Field");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_and(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "mut");
                                ::quote::__private::push_ident(&mut _s, "__map");
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "match");
                        ::quote::__private::push_ident(&mut _s, "__key");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut value_arms, i) = value_arms.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let value_arms = match value_arms.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        ::quote::ToTokens::to_tokens(&value_arms, &mut _s);
                                    }
                                };
                                ::quote::ToTokens::to_tokens(&ignored_arm, &mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            }
        };
        let extract_values = fields_names
            .iter()
            .filter(|&&(field, _)| {
                !field.attrs.skip_deserializing() && !field.attrs.flatten()
            })
            .map(|(field, name)| {
                let missing_expr = Match(expr_is_missing(field, cattrs));
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "match");
                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Some");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::ToTokens::to_tokens(&name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "None");
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::ToTokens::to_tokens(&missing_expr, &mut _s);
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }
            });
        let extract_collected = fields_names
            .iter()
            .filter(|&&(field, _)| {
                field.attrs.flatten() && !field.attrs.skip_deserializing()
            })
            .map(|(field, name)| {
                let field_ty = field.ty;
                let func = match field.attrs.deserialize_with() {
                    None => {
                        let span = field.original.span();
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            let _span: ::quote::__private::Span = span;
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "_serde",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(&mut _s, _span, "de");
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "Deserialize",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "deserialize",
                            );
                            _s
                        }
                    }
                    Some(path) => {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&path, &mut _s);
                        _s
                    }
                };
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&func, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "de");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(
                                        &mut _s,
                                        "FlatMapDeserializer",
                                    );
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "mut");
                                            ::quote::__private::push_ident(&mut _s, "__collect");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }
            });
        let collected_deny_unknown_fields = if cattrs.has_flatten()
            && cattrs.deny_unknown_fields()
        {
            Some({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "if");
                ::quote::__private::push_ident(&mut _s, "let");
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Some");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Some");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "__key");
                                        ::quote::__private::push_comma(&mut _s);
                                        ::quote::__private::push_underscore(&mut _s);
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_eq(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__collect");
                ::quote::__private::push_dot(&mut _s);
                ::quote::__private::push_ident(&mut _s, "into_iter");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    ::quote::__private::TokenStream::new(),
                );
                ::quote::__private::push_dot(&mut _s);
                ::quote::__private::push_ident(&mut _s, "filter");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Option");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "is_some");
                        _s
                    },
                );
                ::quote::__private::push_dot(&mut _s);
                ::quote::__private::push_ident(&mut _s, "next");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    ::quote::__private::TokenStream::new(),
                );
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "if");
                        ::quote::__private::push_ident(&mut _s, "let");
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Some");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__key");
                                _s
                            },
                        );
                        ::quote::__private::push_eq(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__key");
                        ::quote::__private::push_dot(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "as_str");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            ::quote::__private::TokenStream::new(),
                        );
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "return");
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__private");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Err");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Error");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "custom");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "format_args");
                                                ::quote::__private::push_bang(&mut _s);
                                                ::quote::__private::push_group(
                                                    &mut _s,
                                                    ::quote::__private::Delimiter::Parenthesis,
                                                    {
                                                        let mut _s = ::quote::__private::TokenStream::new();
                                                        ::quote::__private::parse(
                                                            &mut _s,
                                                            "\"unknown field `{}`\"",
                                                        );
                                                        ::quote::__private::push_comma(&mut _s);
                                                        ::quote::__private::push_and(&mut _s);
                                                        ::quote::__private::push_ident(&mut _s, "__key");
                                                        _s
                                                    },
                                                );
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                ::quote::__private::push_semi(&mut _s);
                                _s
                            },
                        );
                        ::quote::__private::push_ident(&mut _s, "else");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "return");
                                ::quote::__private::push_ident(&mut _s, "_serde");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__private");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "Err");
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Parenthesis,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "de");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "Error");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "custom");
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "format_args");
                                                ::quote::__private::push_bang(&mut _s);
                                                ::quote::__private::push_group(
                                                    &mut _s,
                                                    ::quote::__private::Delimiter::Parenthesis,
                                                    {
                                                        let mut _s = ::quote::__private::TokenStream::new();
                                                        ::quote::__private::parse(
                                                            &mut _s,
                                                            "\"unexpected map key\"",
                                                        );
                                                        _s
                                                    },
                                                );
                                                _s
                                            },
                                        );
                                        _s
                                    },
                                );
                                ::quote::__private::push_semi(&mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            })
        } else {
            None
        };
        let result = fields_names
            .iter()
            .map(|(field, name)| {
                let member = &field.member;
                if field.attrs.skip_deserializing() {
                    let value = Expr(expr_is_missing(field, cattrs));
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&member, &mut _s);
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::ToTokens::to_tokens(&value, &mut _s);
                        _s
                    }
                } else {
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&member, &mut _s);
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::ToTokens::to_tokens(&name, &mut _s);
                        _s
                    }
                }
            });
        let let_default = match cattrs.default() {
            attr::Default::Default => {
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__default");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Default");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "default");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                })
            }
            attr::Default::Path(path) => {
                Some({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "__default");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Value");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::ToTokens::to_tokens(&path, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                })
            }
            attr::Default::None => None,
        };
        let mut result = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&struct_path, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    {
                        use ::quote::__private::ext::*;
                        let mut _i = 0usize;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut result, i) = result.quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let result = match result.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            if _i > 0 {
                                ::quote::__private::push_comma(&mut _s);
                            }
                            _i += 1;
                            ::quote::ToTokens::to_tokens(&result, &mut _s);
                        }
                    };
                    _s
                },
            );
            _s
        };
        if params.has_getter {
            let this = &params.this;
            result = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Into");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_lt(&mut _s);
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "into");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&result, &mut _s);
                        _s
                    },
                );
                _s
            };
        }
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut let_values, i) = let_values.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let let_values = match let_values.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&let_values, &mut _s);
                }
            };
            ::quote::ToTokens::to_tokens(&let_collect, &mut _s);
            ::quote::ToTokens::to_tokens(&match_keys, &mut _s);
            ::quote::ToTokens::to_tokens(&let_default, &mut _s);
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut extract_values, i) = extract_values.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let extract_values = match extract_values.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&extract_values, &mut _s);
                }
            };
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut extract_collected, i) = extract_collected.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let extract_collected = match extract_collected.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&extract_collected, &mut _s);
                }
            };
            ::quote::ToTokens::to_tokens(&collected_deny_unknown_fields, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Ok");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&result, &mut _s);
                    _s
                },
            );
            _s
        })
    }
    fn field_i(i: usize) -> Ident {
        Ident::new(
            &{
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["__field"],
                        &[::core::fmt::ArgumentV1::new_display(&i)],
                    ),
                );
                res
            },
            Span::call_site(),
        )
    }
    fn wrap_deserialize_with(
        params: &Parameters,
        value_ty: &TokenStream,
        deserialize_with: &syn::ExprPath,
    ) -> (TokenStream, TokenStream) {
        let this = &params.this;
        let (de_impl_generics, de_ty_generics, ty_generics, where_clause) = split_with_de_lifetime(
            params,
        );
        let delife = params.borrowed.de_lifetime();
        let wrapper = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__DeserializeWith");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "value");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::ToTokens::to_tokens(&value_ty, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "phantom");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "lifetime");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&de_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Deserialize");
            ::quote::__private::push_lt(&mut _s);
            ::quote::ToTokens::to_tokens(&delife, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__DeserializeWith");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "deserialize");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__D");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Self");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__D");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Deserializer");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&delife, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Ok");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(
                                        &mut _s,
                                        "__DeserializeWith",
                                    );
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Brace,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "value");
                                            ::quote::__private::push_colon(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "try");
                                            ::quote::__private::push_bang(&mut _s);
                                            ::quote::__private::push_group(
                                                &mut _s,
                                                ::quote::__private::Delimiter::Parenthesis,
                                                {
                                                    let mut _s = ::quote::__private::TokenStream::new();
                                                    ::quote::ToTokens::to_tokens(&deserialize_with, &mut _s);
                                                    ::quote::__private::push_group(
                                                        &mut _s,
                                                        ::quote::__private::Delimiter::Parenthesis,
                                                        {
                                                            let mut _s = ::quote::__private::TokenStream::new();
                                                            ::quote::__private::push_ident(&mut _s, "__deserializer");
                                                            _s
                                                        },
                                                    );
                                                    _s
                                                },
                                            );
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "phantom");
                                            ::quote::__private::push_colon(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "lifetime");
                                            ::quote::__private::push_colon(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "_serde");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "__private");
                                            ::quote::__private::push_colon2(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "PhantomData");
                                            ::quote::__private::push_comma(&mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            _s
        };
        let wrapper_ty = {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "__DeserializeWith");
            ::quote::ToTokens::to_tokens(&de_ty_generics, &mut _s);
            _s
        };
        (wrapper, wrapper_ty)
    }
    fn wrap_deserialize_field_with(
        params: &Parameters,
        field_ty: &syn::Type,
        deserialize_with: &syn::ExprPath,
    ) -> (TokenStream, TokenStream) {
        wrap_deserialize_with(
            params,
            &{
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&field_ty, &mut _s);
                _s
            },
            deserialize_with,
        )
    }
    fn wrap_deserialize_variant_with(
        params: &Parameters,
        variant: &Variant,
        deserialize_with: &syn::ExprPath,
    ) -> (TokenStream, TokenStream, TokenStream) {
        let field_tys = variant.fields.iter().map(|field| field.ty);
        let (wrapper, wrapper_ty) = wrap_deserialize_with(
            params,
            &{
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut field_tys, i) = field_tys.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let field_tys = match field_tys.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&field_tys, &mut _s);
                            }
                        };
                        _s
                    },
                );
                _s
            },
            deserialize_with,
        );
        let unwrap_fn = unwrap_to_variant_closure(params, variant, true);
        (wrapper, wrapper_ty, unwrap_fn)
    }
    fn unwrap_to_variant_closure(
        params: &Parameters,
        variant: &Variant,
        with_wrapper: bool,
    ) -> TokenStream {
        let this = &params.this;
        let variant_ident = &variant.ident;
        let (arg, wrapper) = if with_wrapper {
            (
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__wrap");
                    _s
                },
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__wrap");
                    ::quote::__private::push_dot(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "value");
                    _s
                },
            )
        } else {
            let field_tys = variant.fields.iter().map(|field| field.ty);
            (
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__wrap");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            {
                                use ::quote::__private::ext::*;
                                let mut _i = 0usize;
                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                #[allow(unused_mut)]
                                let (mut field_tys, i) = field_tys.quote_into_iter();
                                let has_iter = has_iter | i;
                                let _: ::quote::__private::HasIterator = has_iter;
                                while true {
                                    let field_tys = match field_tys.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    if _i > 0 {
                                        ::quote::__private::push_comma(&mut _s);
                                    }
                                    _i += 1;
                                    ::quote::ToTokens::to_tokens(&field_tys, &mut _s);
                                }
                            };
                            _s
                        },
                    );
                    _s
                },
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__wrap");
                    _s
                },
            )
        };
        let field_access = (0..variant.fields.len())
            .map(|n| {
                Member::Unnamed(Index {
                    index: n as u32,
                    span: Span::call_site(),
                })
            });
        match variant.style {
            Style::Struct if variant.fields.len() == 1 => {
                let member = &variant.fields[0].member;
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_or(&mut _s);
                    ::quote::ToTokens::to_tokens(&arg, &mut _s);
                    ::quote::__private::push_or(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&member, &mut _s);
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                            _s
                        },
                    );
                    _s
                }
            }
            Style::Struct => {
                let members = variant.fields.iter().map(|field| &field.member);
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_or(&mut _s);
                    ::quote::ToTokens::to_tokens(&arg, &mut _s);
                    ::quote::__private::push_or(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            {
                                use ::quote::__private::ext::*;
                                let mut _i = 0usize;
                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                #[allow(unused_mut)]
                                let (mut members, i) = members.quote_into_iter();
                                let has_iter = has_iter | i;
                                #[allow(unused_mut)]
                                let (mut wrapper, i) = wrapper.quote_into_iter();
                                let has_iter = has_iter | i;
                                #[allow(unused_mut)]
                                let (mut field_access, i) = field_access.quote_into_iter();
                                let has_iter = has_iter | i;
                                let _: ::quote::__private::HasIterator = has_iter;
                                while true {
                                    let members = match members.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    let wrapper = match wrapper.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    let field_access = match field_access.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    if _i > 0 {
                                        ::quote::__private::push_comma(&mut _s);
                                    }
                                    _i += 1;
                                    ::quote::ToTokens::to_tokens(&members, &mut _s);
                                    ::quote::__private::push_colon(&mut _s);
                                    ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                                    ::quote::__private::push_dot(&mut _s);
                                    ::quote::ToTokens::to_tokens(&field_access, &mut _s);
                                }
                            };
                            _s
                        },
                    );
                    _s
                }
            }
            Style::Tuple => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_or(&mut _s);
                ::quote::ToTokens::to_tokens(&arg, &mut _s);
                ::quote::__private::push_or(&mut _s);
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        {
                            use ::quote::__private::ext::*;
                            let mut _i = 0usize;
                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                            #[allow(unused_mut)]
                            let (mut wrapper, i) = wrapper.quote_into_iter();
                            let has_iter = has_iter | i;
                            #[allow(unused_mut)]
                            let (mut field_access, i) = field_access.quote_into_iter();
                            let has_iter = has_iter | i;
                            let _: ::quote::__private::HasIterator = has_iter;
                            while true {
                                let wrapper = match wrapper.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                let field_access = match field_access.next() {
                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                    None => break,
                                };
                                if _i > 0 {
                                    ::quote::__private::push_comma(&mut _s);
                                }
                                _i += 1;
                                ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                                ::quote::__private::push_dot(&mut _s);
                                ::quote::ToTokens::to_tokens(&field_access, &mut _s);
                            }
                        };
                        _s
                    },
                );
                _s
            }
            Style::Newtype => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_or(&mut _s);
                ::quote::ToTokens::to_tokens(&arg, &mut _s);
                ::quote::__private::push_or(&mut _s);
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&wrapper, &mut _s);
                        _s
                    },
                );
                _s
            }
            Style::Unit => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_or(&mut _s);
                ::quote::ToTokens::to_tokens(&arg, &mut _s);
                ::quote::__private::push_or(&mut _s);
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                _s
            }
        }
    }
    fn expr_is_missing(field: &Field, cattrs: &attr::Container) -> Fragment {
        match field.attrs.default() {
            attr::Default::Default => {
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "__private");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "Default");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "default");
                    _s
                };
                return crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&func, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    _s
                });
            }
            attr::Default::Path(path) => {
                return crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&path, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        ::quote::__private::TokenStream::new(),
                    );
                    _s
                });
            }
            attr::Default::None => {}
        }
        match *cattrs.default() {
            attr::Default::Default | attr::Default::Path(_) => {
                let member = &field.member;
                return crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__default");
                    ::quote::__private::push_dot(&mut _s);
                    ::quote::ToTokens::to_tokens(&member, &mut _s);
                    _s
                });
            }
            attr::Default::None => {}
        }
        let name = field.attrs.name().deserialize_name();
        match field.attrs.deserialize_with() {
            None => {
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "__private");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "de");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "missing_field",
                    );
                    _s
                };
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&func, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                })
            }
            Some(_) => {
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "return");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Err");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_lt(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__A");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Error");
                            ::quote::__private::push_ident(&mut _s, "as");
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "de");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Error");
                            ::quote::__private::push_gt(&mut _s);
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "missing_field");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                })
            }
        }
    }
    fn effective_style(variant: &Variant) -> Style {
        match variant.style {
            Style::Newtype if variant.fields[0].attrs.skip_deserializing() => Style::Unit,
            other => other,
        }
    }
    struct DeImplGenerics<'a>(&'a Parameters);
    impl<'a> ToTokens for DeImplGenerics<'a> {
        fn to_tokens(&self, tokens: &mut TokenStream) {
            let mut generics = self.0.generics.clone();
            if let Some(de_lifetime) = self.0.borrowed.de_lifetime_def() {
                generics.params = Some(syn::GenericParam::Lifetime(de_lifetime))
                    .into_iter()
                    .chain(generics.params)
                    .collect();
            }
            let (impl_generics, _, _) = generics.split_for_impl();
            impl_generics.to_tokens(tokens);
        }
    }
    struct DeTypeGenerics<'a>(&'a Parameters);
    impl<'a> ToTokens for DeTypeGenerics<'a> {
        fn to_tokens(&self, tokens: &mut TokenStream) {
            let mut generics = self.0.generics.clone();
            if self.0.borrowed.de_lifetime_def().is_some() {
                let def = syn::LifetimeDef {
                    attrs: Vec::new(),
                    lifetime: syn::Lifetime::new("'de", Span::call_site()),
                    colon_token: None,
                    bounds: Punctuated::new(),
                };
                generics.params = Some(syn::GenericParam::Lifetime(def))
                    .into_iter()
                    .chain(generics.params)
                    .collect();
            }
            let (_, ty_generics, _) = generics.split_for_impl();
            ty_generics.to_tokens(tokens);
        }
    }
    fn split_with_de_lifetime(
        params: &Parameters,
    ) -> (DeImplGenerics, DeTypeGenerics, syn::TypeGenerics, Option<&syn::WhereClause>) {
        let de_impl_generics = DeImplGenerics(params);
        let de_ty_generics = DeTypeGenerics(params);
        let (_, ty_generics, where_clause) = params.generics.split_for_impl();
        (de_impl_generics, de_ty_generics, ty_generics, where_clause)
    }
}
mod dummy {
    use proc_macro2::{Ident, TokenStream};
    use quote::format_ident;
    use syn;
    use try;
    pub fn wrap_in_const(
        serde_path: Option<&syn::Path>,
        trait_: &str,
        ty: &Ident,
        code: TokenStream,
    ) -> TokenStream {
        let try_replacement = try::replacement();
        let dummy_const = if true {
            ::quote::__private::mk_ident(
                &{
                    let res = ::alloc::fmt::format(
                        ::core::fmt::Arguments::new_v1(&["_"], &[]),
                    );
                    res
                },
                ::std::option::Option::None,
            )
        } else {
            match ::quote::__private::IdentFragmentAdapter(&trait_) {
                arg => {
                    match ::quote::__private::IdentFragmentAdapter(&unraw(ty)) {
                        arg => {
                            ::quote::__private::mk_ident(
                                &{
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &["_IMPL_", "_FOR_"],
                                            &[
                                                ::core::fmt::ArgumentV1::new_display(&arg),
                                                ::core::fmt::ArgumentV1::new_display(&arg),
                                            ],
                                        ),
                                    );
                                    res
                                },
                                ::std::option::Option::None.or(arg.span()).or(arg.span()),
                            )
                        }
                    }
                }
            }
        };
        let use_serde = match serde_path {
            Some(path) => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "use");
                ::quote::ToTokens::to_tokens(&path, &mut _s);
                ::quote::__private::push_ident(&mut _s, "as");
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_semi(&mut _s);
                _s
            }
            None => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_pound(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "allow");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(
                                    &mut _s,
                                    "unused_extern_crates",
                                );
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "clippy");
                                ::quote::__private::push_colon2(&mut _s);
                                ::quote::__private::push_ident(
                                    &mut _s,
                                    "useless_attribute",
                                );
                                _s
                            },
                        );
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "extern");
                ::quote::__private::push_ident(&mut _s, "crate");
                ::quote::__private::push_ident(&mut _s, "serde");
                ::quote::__private::push_ident(&mut _s, "as");
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_semi(&mut _s);
                _s
            }
        };
        {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_pound(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Bracket,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "doc");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "hidden");
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_pound(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Bracket,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "allow");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(
                                &mut _s,
                                "non_upper_case_globals",
                            );
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "unused_attributes");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "unused_qualifications",
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "const");
            ::quote::ToTokens::to_tokens(&dummy_const, &mut _s);
            ::quote::__private::push_colon(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                ::quote::__private::TokenStream::new(),
            );
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&use_serde, &mut _s);
                    ::quote::ToTokens::to_tokens(&try_replacement, &mut _s);
                    ::quote::ToTokens::to_tokens(&code, &mut _s);
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            _s
        }
    }
    #[allow(deprecated)]
    fn unraw(ident: &Ident) -> String {
        ident.to_string().trim_left_matches("r#").to_owned()
    }
}
mod pretend {
    use proc_macro2::TokenStream;
    use quote::format_ident;
    use internals::ast::{Container, Data, Field, Style, Variant};
    pub fn pretend_used(cont: &Container, is_packed: bool) -> TokenStream {
        let pretend_fields = pretend_fields_used(cont, is_packed);
        let pretend_variants = pretend_variants_used(cont);
        {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&pretend_fields, &mut _s);
            ::quote::ToTokens::to_tokens(&pretend_variants, &mut _s);
            _s
        }
    }
    fn pretend_fields_used(cont: &Container, is_packed: bool) -> TokenStream {
        match &cont.data {
            Data::Enum(variants) => pretend_fields_used_enum(cont, variants),
            Data::Struct(Style::Struct, fields) => {
                if is_packed {
                    pretend_fields_used_struct_packed(cont, fields)
                } else {
                    pretend_fields_used_struct(cont, fields)
                }
            }
            Data::Struct(_, _) => ::quote::__private::TokenStream::new(),
        }
    }
    fn pretend_fields_used_struct(cont: &Container, fields: &[Field]) -> TokenStream {
        let type_ident = &cont.ident;
        let (_, ty_generics, _) = cont.generics.split_for_impl();
        let members = fields.iter().map(|field| &field.member);
        let placeholders = (0usize..)
            .map(|i| match ::quote::__private::IdentFragmentAdapter(&i) {
                arg => {
                    ::quote::__private::mk_ident(
                        &{
                            let res = ::alloc::fmt::format(
                                ::core::fmt::Arguments::new_v1(
                                    &["__v"],
                                    &[::core::fmt::ArgumentV1::new_display(&arg)],
                                ),
                            );
                            res
                        },
                        ::std::option::Option::None.or(arg.span()),
                    )
                }
            });
        {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "match");
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "None");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_and(&mut _s);
            ::quote::ToTokens::to_tokens(&type_ident, &mut _s);
            ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Some");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&type_ident, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    {
                                        use ::quote::__private::ext::*;
                                        let mut _i = 0usize;
                                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                        #[allow(unused_mut)]
                                        let (mut members, i) = members.quote_into_iter();
                                        let has_iter = has_iter | i;
                                        #[allow(unused_mut)]
                                        let (mut placeholders, i) = placeholders.quote_into_iter();
                                        let has_iter = has_iter | i;
                                        let _: ::quote::__private::HasIterator = has_iter;
                                        while true {
                                            let members = match members.next() {
                                                Some(_x) => ::quote::__private::RepInterp(_x),
                                                None => break,
                                            };
                                            let placeholders = match placeholders.next() {
                                                Some(_x) => ::quote::__private::RepInterp(_x),
                                                None => break,
                                            };
                                            if _i > 0 {
                                                ::quote::__private::push_comma(&mut _s);
                                            }
                                            _i += 1;
                                            ::quote::ToTokens::to_tokens(&members, &mut _s);
                                            ::quote::__private::push_colon(&mut _s);
                                            ::quote::ToTokens::to_tokens(&placeholders, &mut _s);
                                        }
                                    };
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        ::quote::__private::TokenStream::new(),
                    );
                    ::quote::__private::push_underscore(&mut _s);
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        ::quote::__private::TokenStream::new(),
                    );
                    _s
                },
            );
            _s
        }
    }
    fn pretend_fields_used_struct_packed(
        cont: &Container,
        fields: &[Field],
    ) -> TokenStream {
        let type_ident = &cont.ident;
        let (_, ty_generics, _) = cont.generics.split_for_impl();
        let members = fields.iter().map(|field| &field.member).collect::<Vec<_>>();
        #[cfg(ptr_addr_of)]
        {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "match");
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "None");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_lt(&mut _s);
                ::quote::__private::push_and(&mut _s);
                ::quote::ToTokens::to_tokens(&type_ident, &mut _s);
                ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                ::quote::__private::push_gt(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Some");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__v");
                                ::quote::__private::push_at(&mut _s);
                                ::quote::ToTokens::to_tokens(&type_ident, &mut _s);
                                ::quote::__private::push_group(
                                    &mut _s,
                                    ::quote::__private::Delimiter::Brace,
                                    {
                                        let mut _s = ::quote::__private::TokenStream::new();
                                        {
                                            use ::quote::__private::ext::*;
                                            let mut _i = 0usize;
                                            let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                            #[allow(unused_mut)]
                                            let (mut members, i) = members.quote_into_iter();
                                            let has_iter = has_iter | i;
                                            let _: ::quote::__private::HasIterator = has_iter;
                                            while true {
                                                let members = match members.next() {
                                                    Some(_x) => ::quote::__private::RepInterp(_x),
                                                    None => break,
                                                };
                                                if _i > 0 {
                                                    ::quote::__private::push_comma(&mut _s);
                                                }
                                                _i += 1;
                                                ::quote::ToTokens::to_tokens(&members, &mut _s);
                                                ::quote::__private::push_colon(&mut _s);
                                                ::quote::__private::push_underscore(&mut _s);
                                            }
                                        };
                                        _s
                                    },
                                );
                                _s
                            },
                        );
                        ::quote::__private::push_fat_arrow(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut members, i) = members.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let members = match members.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        ::quote::__private::push_ident(&mut _s, "let");
                                        ::quote::__private::push_underscore(&mut _s);
                                        ::quote::__private::push_eq(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "_serde");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "__private");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "ptr");
                                        ::quote::__private::push_colon2(&mut _s);
                                        ::quote::__private::push_ident(&mut _s, "addr_of");
                                        ::quote::__private::push_bang(&mut _s);
                                        ::quote::__private::push_group(
                                            &mut _s,
                                            ::quote::__private::Delimiter::Parenthesis,
                                            {
                                                let mut _s = ::quote::__private::TokenStream::new();
                                                ::quote::__private::push_ident(&mut _s, "__v");
                                                ::quote::__private::push_dot(&mut _s);
                                                ::quote::ToTokens::to_tokens(&members, &mut _s);
                                                _s
                                            },
                                        );
                                        ::quote::__private::push_semi(&mut _s);
                                    }
                                };
                                _s
                            },
                        );
                        ::quote::__private::push_underscore(&mut _s);
                        ::quote::__private::push_fat_arrow(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            ::quote::__private::TokenStream::new(),
                        );
                        _s
                    },
                );
                _s
            }
        }
    }
    fn pretend_fields_used_enum(cont: &Container, variants: &[Variant]) -> TokenStream {
        let type_ident = &cont.ident;
        let (_, ty_generics, _) = cont.generics.split_for_impl();
        let patterns = variants
            .iter()
            .filter_map(|variant| match variant.style {
                Style::Struct => {
                    let variant_ident = &variant.ident;
                    let members = variant.fields.iter().map(|field| &field.member);
                    let placeholders = (0usize..)
                        .map(|i| match ::quote::__private::IdentFragmentAdapter(&i) {
                            arg => {
                                ::quote::__private::mk_ident(
                                    &{
                                        let res = ::alloc::fmt::format(
                                            ::core::fmt::Arguments::new_v1(
                                                &["__v"],
                                                &[::core::fmt::ArgumentV1::new_display(&arg)],
                                            ),
                                        );
                                        res
                                    },
                                    ::std::option::Option::None.or(arg.span()),
                                )
                            }
                        });
                    Some({
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&type_ident, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let mut _i = 0usize;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut members, i) = members.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    #[allow(unused_mut)]
                                    let (mut placeholders, i) = placeholders.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let members = match members.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        let placeholders = match placeholders.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        if _i > 0 {
                                            ::quote::__private::push_comma(&mut _s);
                                        }
                                        _i += 1;
                                        ::quote::ToTokens::to_tokens(&members, &mut _s);
                                        ::quote::__private::push_colon(&mut _s);
                                        ::quote::ToTokens::to_tokens(&placeholders, &mut _s);
                                    }
                                };
                                _s
                            },
                        );
                        _s
                    })
                }
                _ => None,
            })
            .collect::<Vec<_>>();
        {
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "match");
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "__private");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "None");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_lt(&mut _s);
            ::quote::__private::push_and(&mut _s);
            ::quote::ToTokens::to_tokens(&type_ident, &mut _s);
            ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
            ::quote::__private::push_gt(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut patterns, i) = patterns.quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let patterns = match patterns.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Some");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&patterns, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                ::quote::__private::TokenStream::new(),
                            );
                        }
                    };
                    ::quote::__private::push_underscore(&mut _s);
                    ::quote::__private::push_fat_arrow(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        ::quote::__private::TokenStream::new(),
                    );
                    _s
                },
            );
            _s
        }
    }
    fn pretend_variants_used(cont: &Container) -> TokenStream {
        let variants = match &cont.data {
            Data::Enum(variants) => variants,
            Data::Struct(_, _) => {
                return ::quote::__private::TokenStream::new();
            }
        };
        let type_ident = &cont.ident;
        let (_, ty_generics, _) = cont.generics.split_for_impl();
        let turbofish = ty_generics.as_turbofish();
        let cases = variants
            .iter()
            .map(|variant| {
                let variant_ident = &variant.ident;
                let placeholders = &(0..variant.fields.len())
                    .map(|i| match ::quote::__private::IdentFragmentAdapter(&i) {
                        arg => {
                            ::quote::__private::mk_ident(
                                &{
                                    let res = ::alloc::fmt::format(
                                        ::core::fmt::Arguments::new_v1(
                                            &["__v"],
                                            &[::core::fmt::ArgumentV1::new_display(&arg)],
                                        ),
                                    );
                                    res
                                },
                                ::std::option::Option::None.or(arg.span()),
                            )
                        }
                    })
                    .collect::<Vec<_>>();
                let pat = match variant.style {
                    Style::Struct => {
                        let members = variant.fields.iter().map(|field| &field.member);
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    {
                                        use ::quote::__private::ext::*;
                                        let mut _i = 0usize;
                                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                        #[allow(unused_mut)]
                                        let (mut members, i) = members.quote_into_iter();
                                        let has_iter = has_iter | i;
                                        #[allow(unused_mut)]
                                        let (mut placeholders, i) = placeholders.quote_into_iter();
                                        let has_iter = has_iter | i;
                                        let _: ::quote::__private::HasIterator = has_iter;
                                        while true {
                                            let members = match members.next() {
                                                Some(_x) => ::quote::__private::RepInterp(_x),
                                                None => break,
                                            };
                                            let placeholders = match placeholders.next() {
                                                Some(_x) => ::quote::__private::RepInterp(_x),
                                                None => break,
                                            };
                                            if _i > 0 {
                                                ::quote::__private::push_comma(&mut _s);
                                            }
                                            _i += 1;
                                            ::quote::ToTokens::to_tokens(&members, &mut _s);
                                            ::quote::__private::push_colon(&mut _s);
                                            ::quote::ToTokens::to_tokens(&placeholders, &mut _s);
                                        }
                                    };
                                    _s
                                },
                            );
                            _s
                        }
                    }
                    Style::Tuple | Style::Newtype => {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let mut _i = 0usize;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut placeholders, i) = placeholders.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let placeholders = match placeholders.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        if _i > 0 {
                                            ::quote::__private::push_comma(&mut _s);
                                        }
                                        _i += 1;
                                        ::quote::ToTokens::to_tokens(&placeholders, &mut _s);
                                    }
                                };
                                _s
                            },
                        );
                        _s
                    }
                    Style::Unit => ::quote::__private::TokenStream::new(),
                };
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "match");
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "None");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Some");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            {
                                                use ::quote::__private::ext::*;
                                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                                #[allow(unused_mut)]
                                                let (mut placeholders, i) = placeholders.quote_into_iter();
                                                let has_iter = has_iter | i;
                                                let _: ::quote::__private::HasIterator = has_iter;
                                                while true {
                                                    let placeholders = match placeholders.next() {
                                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                                        None => break,
                                                    };
                                                    ::quote::ToTokens::to_tokens(&placeholders, &mut _s);
                                                    ::quote::__private::push_comma(&mut _s);
                                                }
                                            };
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "let");
                                    ::quote::__private::push_underscore(&mut _s);
                                    ::quote::__private::push_eq(&mut _s);
                                    ::quote::ToTokens::to_tokens(&type_ident, &mut _s);
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                                    ::quote::ToTokens::to_tokens(&turbofish, &mut _s);
                                    ::quote::ToTokens::to_tokens(&pat, &mut _s);
                                    ::quote::__private::push_semi(&mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_underscore(&mut _s);
                            ::quote::__private::push_fat_arrow(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                ::quote::__private::TokenStream::new(),
                            );
                            _s
                        },
                    );
                    _s
                }
            });
        {
            let mut _s = ::quote::__private::TokenStream::new();
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut cases, i) = cases.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let cases = match cases.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&cases, &mut _s);
                }
            };
            _s
        }
    }
}
mod ser {
    use proc_macro2::{Span, TokenStream};
    use syn::spanned::Spanned;
    use syn::{self, Ident, Index, Member};
    use bound;
    use dummy;
    use fragment::{Fragment, Match, Stmts};
    use internals::ast::{Container, Data, Field, Style, Variant};
    use internals::{attr, replace_receiver, Ctxt, Derive};
    use pretend;
    pub fn expand_derive_serialize(
        input: &mut syn::DeriveInput,
    ) -> Result<TokenStream, Vec<syn::Error>> {
        replace_receiver(input);
        let ctxt = Ctxt::new();
        let cont = match Container::from_ast(&ctxt, input, Derive::Serialize) {
            Some(cont) => cont,
            None => return Err(ctxt.check().unwrap_err()),
        };
        precondition(&ctxt, &cont);
        ctxt.check()?;
        let ident = &cont.ident;
        let params = Parameters::new(&cont);
        let (impl_generics, ty_generics, where_clause) = params
            .generics
            .split_for_impl();
        let body = Stmts(serialize_body(&cont, &params));
        let serde = cont.attrs.serde_path();
        let impl_block = if let Some(remote) = cont.attrs.remote() {
            let vis = &input.vis;
            let used = pretend::pretend_used(&cont, params.is_packed);
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "impl");
                ::quote::ToTokens::to_tokens(&impl_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&ident, &mut _s);
                ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&vis, &mut _s);
                        ::quote::__private::push_ident(&mut _s, "fn");
                        ::quote::__private::push_ident(&mut _s, "serialize");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_ident(&mut _s, "__self");
                                ::quote::__private::push_colon(&mut _s);
                                ::quote::__private::push_and(&mut _s);
                                ::quote::ToTokens::to_tokens(&remote, &mut _s);
                                ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__serializer");
                                ::quote::__private::push_colon(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__S");
                                _s
                            },
                        );
                        ::quote::__private::push_rarrow(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Result");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "where");
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Serializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&used, &mut _s);
                                ::quote::ToTokens::to_tokens(&body, &mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            }
        } else {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_pound(&mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Bracket,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "automatically_derived");
                        _s
                    },
                );
                ::quote::__private::push_ident(&mut _s, "impl");
                ::quote::ToTokens::to_tokens(&impl_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&serde, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Serialize");
                ::quote::__private::push_ident(&mut _s, "for");
                ::quote::ToTokens::to_tokens(&ident, &mut _s);
                ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Brace,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "fn");
                        ::quote::__private::push_ident(&mut _s, "serialize");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::push_and(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "self");
                                ::quote::__private::push_comma(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__serializer");
                                ::quote::__private::push_colon(&mut _s);
                                ::quote::__private::push_ident(&mut _s, "__S");
                                _s
                            },
                        );
                        ::quote::__private::push_rarrow(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__private");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Result");
                        ::quote::__private::push_lt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Ok");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_gt(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "where");
                        ::quote::__private::push_ident(&mut _s, "__S");
                        ::quote::__private::push_colon(&mut _s);
                        ::quote::ToTokens::to_tokens(&serde, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Serializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&body, &mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            }
        };
        Ok(
            dummy::wrap_in_const(
                cont.attrs.custom_serde_path(),
                "SERIALIZE",
                ident,
                impl_block,
            ),
        )
    }
    fn precondition(cx: &Ctxt, cont: &Container) {
        match cont.attrs.identifier() {
            attr::Identifier::No => {}
            attr::Identifier::Field => {
                cx.error_spanned_by(
                    cont.original,
                    "field identifiers cannot be serialized",
                );
            }
            attr::Identifier::Variant => {
                cx.error_spanned_by(
                    cont.original,
                    "variant identifiers cannot be serialized",
                );
            }
        }
    }
    struct Parameters {
        self_var: Ident,
        this: syn::Path,
        generics: syn::Generics,
        is_remote: bool,
        is_packed: bool,
    }
    impl Parameters {
        fn new(cont: &Container) -> Self {
            let is_remote = cont.attrs.remote().is_some();
            let self_var = if is_remote {
                Ident::new("__self", Span::call_site())
            } else {
                Ident::new("self", Span::call_site())
            };
            let this = match cont.attrs.remote() {
                Some(remote) => remote.clone(),
                None => cont.ident.clone().into(),
            };
            let is_packed = cont.attrs.is_packed();
            let generics = build_generics(cont);
            Parameters {
                self_var,
                this,
                generics,
                is_remote,
                is_packed,
            }
        }
        fn type_name(&self) -> String {
            self.this.segments.last().unwrap().ident.to_string()
        }
    }
    fn build_generics(cont: &Container) -> syn::Generics {
        let generics = bound::without_defaults(cont.generics);
        let generics = bound::with_where_predicates_from_fields(
            cont,
            &generics,
            attr::Field::ser_bound,
        );
        let generics = bound::with_where_predicates_from_variants(
            cont,
            &generics,
            attr::Variant::ser_bound,
        );
        match cont.attrs.ser_bound() {
            Some(predicates) => bound::with_where_predicates(&generics, predicates),
            None => {
                bound::with_bound(
                    cont,
                    &generics,
                    needs_serialize_bound,
                    &::syn::parse_quote::parse({
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Serialize");
                        _s
                    }),
                )
            }
        }
    }
    fn needs_serialize_bound(
        field: &attr::Field,
        variant: Option<&attr::Variant>,
    ) -> bool {
        !field.skip_serializing() && field.serialize_with().is_none()
            && field.ser_bound().is_none()
            && variant
                .map_or(
                    true,
                    |variant| {
                        !variant.skip_serializing() && variant.serialize_with().is_none()
                            && variant.ser_bound().is_none()
                    },
                )
    }
    fn serialize_body(cont: &Container, params: &Parameters) -> Fragment {
        if cont.attrs.transparent() {
            serialize_transparent(cont, params)
        } else if let Some(type_into) = cont.attrs.type_into() {
            serialize_into(params, type_into)
        } else {
            match &cont.data {
                Data::Enum(variants) => serialize_enum(params, variants, &cont.attrs),
                Data::Struct(Style::Struct, fields) => {
                    serialize_struct(params, fields, &cont.attrs)
                }
                Data::Struct(Style::Tuple, fields) => {
                    serialize_tuple_struct(params, fields, &cont.attrs)
                }
                Data::Struct(Style::Newtype, fields) => {
                    serialize_newtype_struct(params, &fields[0], &cont.attrs)
                }
                Data::Struct(Style::Unit, _) => serialize_unit_struct(&cont.attrs),
            }
        }
    }
    fn serialize_transparent(cont: &Container, params: &Parameters) -> Fragment {
        let fields = match &cont.data {
            Data::Struct(_, fields) => fields,
            Data::Enum(_) => {
                ::core::panicking::panic("internal error: entered unreachable code")
            }
        };
        let self_var = &params.self_var;
        let transparent_field = fields.iter().find(|f| f.attrs.transparent()).unwrap();
        let member = &transparent_field.member;
        let path = match transparent_field.attrs.serialize_with() {
            Some(path) => {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&path, &mut _s);
                _s
            }
            None => {
                let span = transparent_field.original.span();
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "Serialize");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "serialize");
                    _s
                }
            }
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&path, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_and(&mut _s);
                    ::quote::ToTokens::to_tokens(&self_var, &mut _s);
                    ::quote::__private::push_dot(&mut _s);
                    ::quote::ToTokens::to_tokens(&member, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__serializer");
                    _s
                },
            );
            _s
        })
    }
    fn serialize_into(params: &Parameters, type_into: &syn::Type) -> Fragment {
        let self_var = &params.self_var;
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Serialize");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "serialize");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_and(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Into");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_into, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "into");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__private");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Clone");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "clone");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&self_var, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__serializer");
                    _s
                },
            );
            _s
        })
    }
    fn serialize_unit_struct(cattrs: &attr::Container) -> Fragment {
        let type_name = cattrs.name().serialize_name();
        crate::fragment::Fragment::Expr({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Serializer");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "serialize_unit_struct");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__serializer");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                    _s
                },
            );
            _s
        })
    }
    fn serialize_newtype_struct(
        params: &Parameters,
        field: &Field,
        cattrs: &attr::Container,
    ) -> Fragment {
        let type_name = cattrs.name().serialize_name();
        let mut field_expr = get_member(
            params,
            field,
            &Member::Unnamed(Index {
                index: 0,
                span: Span::call_site(),
            }),
        );
        if let Some(path) = field.attrs.serialize_with() {
            field_expr = wrap_serialize_field_with(params, field.ty, path, &field_expr);
        }
        let span = field.original.span();
        let func = {
            let mut _s = ::quote::__private::TokenStream::new();
            let _span: ::quote::__private::Span = span;
            ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
            ::quote::__private::push_colon2_spanned(&mut _s, _span);
            ::quote::__private::push_ident_spanned(&mut _s, _span, "Serializer");
            ::quote::__private::push_colon2_spanned(&mut _s, _span);
            ::quote::__private::push_ident_spanned(
                &mut _s,
                _span,
                "serialize_newtype_struct",
            );
            _s
        };
        crate::fragment::Fragment::Expr({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::ToTokens::to_tokens(&func, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__serializer");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                    _s
                },
            );
            _s
        })
    }
    fn serialize_tuple_struct(
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
    ) -> Fragment {
        let serialize_stmts = serialize_tuple_struct_visitor(
            fields,
            params,
            false,
            &TupleTrait::SerializeTupleStruct,
        );
        let type_name = cattrs.name().serialize_name();
        let mut serialized_fields = fields
            .iter()
            .enumerate()
            .filter(|(_, field)| !field.attrs.skip_serializing())
            .peekable();
        let let_mut = mut_if(serialized_fields.peek().is_some());
        let len = serialized_fields
            .map(|(i, field)| match field.attrs.skip_serializing_if() {
                None => {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::parse(&mut _s, "1");
                    _s
                }
                Some(path) => {
                    let index = syn::Index {
                        index: i as u32,
                        span: Span::call_site(),
                    };
                    let field_expr = get_member(params, field, &Member::Unnamed(index));
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "if");
                        ::quote::ToTokens::to_tokens(&path, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                                _s
                            },
                        );
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "0");
                                _s
                            },
                        );
                        ::quote::__private::push_ident(&mut _s, "else");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "1");
                                _s
                            },
                        );
                        _s
                    }
                }
            })
            .fold(
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::parse(&mut _s, "0");
                    _s
                },
                |sum, expr| {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&sum, &mut _s);
                    ::quote::__private::push_add(&mut _s);
                    ::quote::ToTokens::to_tokens(&expr, &mut _s);
                    _s
                },
            );
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "let");
            ::quote::ToTokens::to_tokens(&let_mut, &mut _s);
            ::quote::__private::push_ident(&mut _s, "__serde_state");
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_tuple_struct");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&len, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut serialize_stmts, i) = serialize_stmts.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let serialize_stmts = match serialize_stmts.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&serialize_stmts, &mut _s);
                }
            };
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "ser");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "SerializeTupleStruct");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "end");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                    _s
                },
            );
            _s
        })
    }
    fn serialize_struct(
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
    ) -> Fragment {
        if !(fields.len() as u64 <= u64::from(u32::max_value())) {
            ::core::panicking::panic(
                "assertion failed: fields.len() as u64 <= u64::from(u32::max_value())",
            )
        }
        if cattrs.has_flatten() {
            serialize_struct_as_map(params, fields, cattrs)
        } else {
            serialize_struct_as_struct(params, fields, cattrs)
        }
    }
    fn serialize_struct_tag_field(
        cattrs: &attr::Container,
        struct_trait: &StructTrait,
    ) -> TokenStream {
        match cattrs.tag() {
            attr::TagType::Internal { tag } => {
                let type_name = cattrs.name().serialize_name();
                let func = struct_trait.serialize_field(Span::call_site());
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&func, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_and(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "mut");
                                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    _s
                }
            }
            _ => ::quote::__private::TokenStream::new(),
        }
    }
    fn serialize_struct_as_struct(
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
    ) -> Fragment {
        let serialize_fields = serialize_struct_visitor(
            fields,
            params,
            false,
            &StructTrait::SerializeStruct,
        );
        let type_name = cattrs.name().serialize_name();
        let tag_field = serialize_struct_tag_field(
            cattrs,
            &StructTrait::SerializeStruct,
        );
        let tag_field_exists = !tag_field.is_empty();
        let mut serialized_fields = fields
            .iter()
            .filter(|&field| !field.attrs.skip_serializing())
            .peekable();
        let let_mut = mut_if(serialized_fields.peek().is_some() || tag_field_exists);
        let len = serialized_fields
            .map(|field| match field.attrs.skip_serializing_if() {
                None => {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::parse(&mut _s, "1");
                    _s
                }
                Some(path) => {
                    let field_expr = get_member(params, field, &field.member);
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "if");
                        ::quote::ToTokens::to_tokens(&path, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                                _s
                            },
                        );
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "0");
                                _s
                            },
                        );
                        ::quote::__private::push_ident(&mut _s, "else");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "1");
                                _s
                            },
                        );
                        _s
                    }
                }
            })
            .fold(
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&tag_field_exists, &mut _s);
                    ::quote::__private::push_ident(&mut _s, "as");
                    ::quote::__private::push_ident(&mut _s, "usize");
                    _s
                },
                |sum, expr| {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&sum, &mut _s);
                    ::quote::__private::push_add(&mut _s);
                    ::quote::ToTokens::to_tokens(&expr, &mut _s);
                    _s
                },
            );
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "let");
            ::quote::ToTokens::to_tokens(&let_mut, &mut _s);
            ::quote::__private::push_ident(&mut _s, "__serde_state");
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_struct");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&len, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::ToTokens::to_tokens(&tag_field, &mut _s);
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut serialize_fields, i) = serialize_fields.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let serialize_fields = match serialize_fields.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&serialize_fields, &mut _s);
                }
            };
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "ser");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "SerializeStruct");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "end");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                    _s
                },
            );
            _s
        })
    }
    fn serialize_struct_as_map(
        params: &Parameters,
        fields: &[Field],
        cattrs: &attr::Container,
    ) -> Fragment {
        let serialize_fields = serialize_struct_visitor(
            fields,
            params,
            false,
            &StructTrait::SerializeMap,
        );
        let tag_field = serialize_struct_tag_field(cattrs, &StructTrait::SerializeMap);
        let tag_field_exists = !tag_field.is_empty();
        let mut serialized_fields = fields
            .iter()
            .filter(|&field| !field.attrs.skip_serializing())
            .peekable();
        let let_mut = mut_if(serialized_fields.peek().is_some() || tag_field_exists);
        let len = if cattrs.has_flatten() {
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "None");
                _s
            }
        } else {
            let len = serialized_fields
                .map(|field| match field.attrs.skip_serializing_if() {
                    None => {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::parse(&mut _s, "1");
                        _s
                    }
                    Some(path) => {
                        let field_expr = get_member(params, field, &field.member);
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "if");
                            ::quote::ToTokens::to_tokens(&path, &mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                                    _s
                                },
                            );
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::parse(&mut _s, "0");
                                    _s
                                },
                            );
                            ::quote::__private::push_ident(&mut _s, "else");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::parse(&mut _s, "1");
                                    _s
                                },
                            );
                            _s
                        }
                    }
                })
                .fold(
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&tag_field_exists, &mut _s);
                        ::quote::__private::push_ident(&mut _s, "as");
                        ::quote::__private::push_ident(&mut _s, "usize");
                        _s
                    },
                    |sum, expr| {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&sum, &mut _s);
                        ::quote::__private::push_add(&mut _s);
                        ::quote::ToTokens::to_tokens(&expr, &mut _s);
                        _s
                    },
                );
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Some");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&len, &mut _s);
                        _s
                    },
                );
                _s
            }
        };
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "let");
            ::quote::ToTokens::to_tokens(&let_mut, &mut _s);
            ::quote::__private::push_ident(&mut _s, "__serde_state");
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_map");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&len, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::ToTokens::to_tokens(&tag_field, &mut _s);
            {
                use ::quote::__private::ext::*;
                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                #[allow(unused_mut)]
                let (mut serialize_fields, i) = serialize_fields.quote_into_iter();
                let has_iter = has_iter | i;
                let _: ::quote::__private::HasIterator = has_iter;
                while true {
                    let serialize_fields = match serialize_fields.next() {
                        Some(_x) => ::quote::__private::RepInterp(_x),
                        None => break,
                    };
                    ::quote::ToTokens::to_tokens(&serialize_fields, &mut _s);
                }
            };
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "ser");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "SerializeMap");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "end");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                    _s
                },
            );
            _s
        })
    }
    fn serialize_enum(
        params: &Parameters,
        variants: &[Variant],
        cattrs: &attr::Container,
    ) -> Fragment {
        if !(variants.len() as u64 <= u64::from(u32::max_value())) {
            ::core::panicking::panic(
                "assertion failed: variants.len() as u64 <= u64::from(u32::max_value())",
            )
        }
        let self_var = &params.self_var;
        let arms: Vec<_> = variants
            .iter()
            .enumerate()
            .map(|(variant_index, variant)| {
                serialize_variant(params, variant, variant_index as u32, cattrs)
            })
            .collect();
        crate::fragment::Fragment::Expr({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "match");
            ::quote::__private::push_star(&mut _s);
            ::quote::ToTokens::to_tokens(&self_var, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut arms, i) = arms.quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let arms = match arms.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::ToTokens::to_tokens(&arms, &mut _s);
                        }
                    };
                    _s
                },
            );
            _s
        })
    }
    fn serialize_variant(
        params: &Parameters,
        variant: &Variant,
        variant_index: u32,
        cattrs: &attr::Container,
    ) -> TokenStream {
        let this = &params.this;
        let variant_ident = &variant.ident;
        if variant.attrs.skip_serializing() {
            let skipped_msg = {
                let res = ::alloc::fmt::format(
                    ::core::fmt::Arguments::new_v1(
                        &["the enum variant ", "::", " cannot be serialized"],
                        &[
                            ::core::fmt::ArgumentV1::new_display(&params.type_name()),
                            ::core::fmt::ArgumentV1::new_display(&variant_ident),
                        ],
                    ),
                );
                res
            };
            let skipped_err = {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Err");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "_serde");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "ser");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "Error");
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "custom");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&skipped_msg, &mut _s);
                                _s
                            },
                        );
                        _s
                    },
                );
                _s
            };
            let fields_pat = match variant.style {
                Style::Unit => ::quote::__private::TokenStream::new(),
                Style::Newtype | Style::Tuple => {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_dot2(&mut _s);
                            _s
                        },
                    );
                    _s
                }
                Style::Struct => {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_dot2(&mut _s);
                            _s
                        },
                    );
                    _s
                }
            };
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&this, &mut _s);
                ::quote::__private::push_colon2(&mut _s);
                ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                ::quote::ToTokens::to_tokens(&fields_pat, &mut _s);
                ::quote::__private::push_fat_arrow(&mut _s);
                ::quote::ToTokens::to_tokens(&skipped_err, &mut _s);
                ::quote::__private::push_comma(&mut _s);
                _s
            }
        } else {
            let case = match variant.style {
                Style::Unit => {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                    _s
                }
                Style::Newtype => {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "ref");
                            ::quote::__private::push_ident(&mut _s, "__field0");
                            _s
                        },
                    );
                    _s
                }
                Style::Tuple => {
                    let field_names = (0..variant.fields.len())
                        .map(|i| Ident::new(
                            &{
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1(
                                        &["__field"],
                                        &[::core::fmt::ArgumentV1::new_display(&i)],
                                    ),
                                );
                                res
                            },
                            Span::call_site(),
                        ));
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&this, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let mut _i = 0usize;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut field_names, i) = field_names.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let field_names = match field_names.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        if _i > 0 {
                                            ::quote::__private::push_comma(&mut _s);
                                        }
                                        _i += 1;
                                        ::quote::__private::push_ident(&mut _s, "ref");
                                        ::quote::ToTokens::to_tokens(&field_names, &mut _s);
                                    }
                                };
                                _s
                            },
                        );
                        _s
                    }
                }
                Style::Struct => {
                    let members = variant.fields.iter().map(|f| &f.member);
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&this, &mut _s);
                        ::quote::__private::push_colon2(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_ident, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                {
                                    use ::quote::__private::ext::*;
                                    let mut _i = 0usize;
                                    let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                    #[allow(unused_mut)]
                                    let (mut members, i) = members.quote_into_iter();
                                    let has_iter = has_iter | i;
                                    let _: ::quote::__private::HasIterator = has_iter;
                                    while true {
                                        let members = match members.next() {
                                            Some(_x) => ::quote::__private::RepInterp(_x),
                                            None => break,
                                        };
                                        if _i > 0 {
                                            ::quote::__private::push_comma(&mut _s);
                                        }
                                        _i += 1;
                                        ::quote::__private::push_ident(&mut _s, "ref");
                                        ::quote::ToTokens::to_tokens(&members, &mut _s);
                                    }
                                };
                                _s
                            },
                        );
                        _s
                    }
                }
            };
            let body = Match(
                match cattrs.tag() {
                    attr::TagType::External => {
                        serialize_externally_tagged_variant(
                            params,
                            variant,
                            variant_index,
                            cattrs,
                        )
                    }
                    attr::TagType::Internal { tag } => {
                        serialize_internally_tagged_variant(params, variant, cattrs, tag)
                    }
                    attr::TagType::Adjacent { tag, content } => {
                        serialize_adjacently_tagged_variant(
                            params,
                            variant,
                            cattrs,
                            tag,
                            content,
                        )
                    }
                    attr::TagType::None => {
                        serialize_untagged_variant(params, variant, cattrs)
                    }
                },
            );
            {
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::ToTokens::to_tokens(&case, &mut _s);
                ::quote::__private::push_fat_arrow(&mut _s);
                ::quote::ToTokens::to_tokens(&body, &mut _s);
                _s
            }
        }
    }
    fn serialize_externally_tagged_variant(
        params: &Parameters,
        variant: &Variant,
        variant_index: u32,
        cattrs: &attr::Container,
    ) -> Fragment {
        let type_name = cattrs.name().serialize_name();
        let variant_name = variant.attrs.name().serialize_name();
        if let Some(path) = variant.attrs.serialize_with() {
            let ser = wrap_serialize_variant_with(params, path, variant);
            return crate::fragment::Fragment::Expr({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Serializer");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "serialize_newtype_variant");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__serializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_index, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&ser, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        _s
                    },
                );
                _s
            });
        }
        match effective_style(variant) {
            Style::Unit => {
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_unit_variant");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_index, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Newtype => {
                let field = &variant.fields[0];
                let mut field_expr = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__field0");
                    _s
                };
                if let Some(path) = field.attrs.serialize_with() {
                    field_expr = wrap_serialize_field_with(
                        params,
                        field.ty,
                        path,
                        &field_expr,
                    );
                }
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "Serializer");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "serialize_newtype_variant",
                    );
                    _s
                };
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&func, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_index, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Tuple => {
                serialize_tuple_variant(
                    TupleVariant::ExternallyTagged {
                        type_name,
                        variant_index,
                        variant_name,
                    },
                    params,
                    &variant.fields,
                )
            }
            Style::Struct => {
                serialize_struct_variant(
                    StructVariant::ExternallyTagged {
                        variant_index,
                        variant_name,
                    },
                    params,
                    &variant.fields,
                    &type_name,
                )
            }
        }
    }
    fn serialize_internally_tagged_variant(
        params: &Parameters,
        variant: &Variant,
        cattrs: &attr::Container,
        tag: &str,
    ) -> Fragment {
        let type_name = cattrs.name().serialize_name();
        let variant_name = variant.attrs.name().serialize_name();
        let enum_ident_str = params.type_name();
        let variant_ident_str = variant.ident.to_string();
        if let Some(path) = variant.attrs.serialize_with() {
            let ser = wrap_serialize_variant_with(params, path, variant);
            return crate::fragment::Fragment::Expr({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "__private");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "ser");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "serialize_tagged_newtype");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "__serializer");
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&enum_ident_str, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_ident_str, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&tag, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::ToTokens::to_tokens(&ser, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        _s
                    },
                );
                _s
            });
        }
        match effective_style(variant) {
            Style::Unit => {
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "mut");
                    ::quote::__private::push_ident(&mut _s, "__struct");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Serializer");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "serialize_struct");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__serializer");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::__private::parse(&mut _s, "1");
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "ser");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "serialize_field");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_and(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "mut");
                                    ::quote::__private::push_ident(&mut _s, "__struct");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ser");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "end");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__struct");
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Newtype => {
                let field = &variant.fields[0];
                let mut field_expr = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__field0");
                    _s
                };
                if let Some(path) = field.attrs.serialize_with() {
                    field_expr = wrap_serialize_field_with(
                        params,
                        field.ty,
                        path,
                        &field_expr,
                    );
                }
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "__private");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "ser");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(
                        &mut _s,
                        _span,
                        "serialize_tagged_newtype",
                    );
                    _s
                };
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&func, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&enum_ident_str, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_ident_str, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Struct => {
                serialize_struct_variant(
                    StructVariant::InternallyTagged {
                        tag,
                        variant_name,
                    },
                    params,
                    &variant.fields,
                    &type_name,
                )
            }
            Style::Tuple => {
                ::core::panicking::unreachable_display(
                    &"checked in serde_derive_internals",
                )
            }
        }
    }
    fn serialize_adjacently_tagged_variant(
        params: &Parameters,
        variant: &Variant,
        cattrs: &attr::Container,
        tag: &str,
        content: &str,
    ) -> Fragment {
        let this = &params.this;
        let type_name = cattrs.name().serialize_name();
        let variant_name = variant.attrs.name().serialize_name();
        let inner = Stmts(
            if let Some(path) = variant.attrs.serialize_with() {
                let ser = wrap_serialize_variant_with(params, path, variant);
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serialize");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&ser, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            _s
                        },
                    );
                    _s
                })
            } else {
                match effective_style(variant) {
                    Style::Unit => {
                        return crate::fragment::Fragment::Block({
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "let");
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__struct");
                            ::quote::__private::push_eq(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "try");
                            ::quote::__private::push_bang(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Serializer");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "serialize_struct");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "__serializer");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::__private::parse(&mut _s, "1");
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "try");
                            ::quote::__private::push_bang(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "ser");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "serialize_field");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "mut");
                                            ::quote::__private::push_ident(&mut _s, "__struct");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "ser");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "end");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__struct");
                                    _s
                                },
                            );
                            _s
                        });
                    }
                    Style::Newtype => {
                        let field = &variant.fields[0];
                        let mut field_expr = {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__field0");
                            _s
                        };
                        if let Some(path) = field.attrs.serialize_with() {
                            field_expr = wrap_serialize_field_with(
                                params,
                                field.ty,
                                path,
                                &field_expr,
                            );
                        }
                        let span = field.original.span();
                        let func = {
                            let mut _s = ::quote::__private::TokenStream::new();
                            let _span: ::quote::__private::Span = span;
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "_serde",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "ser",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "SerializeStruct",
                            );
                            ::quote::__private::push_colon2_spanned(&mut _s, _span);
                            ::quote::__private::push_ident_spanned(
                                &mut _s,
                                _span,
                                "serialize_field",
                            );
                            _s
                        };
                        return crate::fragment::Fragment::Block({
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "let");
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__struct");
                            ::quote::__private::push_eq(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "try");
                            ::quote::__private::push_bang(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "Serializer");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "serialize_struct");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "__serializer");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::__private::parse(&mut _s, "2");
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "try");
                            ::quote::__private::push_bang(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "ser");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "serialize_field");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "mut");
                                            ::quote::__private::push_ident(&mut _s, "__struct");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "try");
                            ::quote::__private::push_bang(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::ToTokens::to_tokens(&func, &mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_and(&mut _s);
                                            ::quote::__private::push_ident(&mut _s, "mut");
                                            ::quote::__private::push_ident(&mut _s, "__struct");
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&content, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                            ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_semi(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "ser");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "end");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__struct");
                                    _s
                                },
                            );
                            _s
                        });
                    }
                    Style::Tuple => {
                        serialize_tuple_variant(
                            TupleVariant::Untagged,
                            params,
                            &variant.fields,
                        )
                    }
                    Style::Struct => {
                        serialize_struct_variant(
                            StructVariant::Untagged,
                            params,
                            &variant.fields,
                            &variant_name,
                        )
                    }
                }
            },
        );
        let fields_ty = variant.fields.iter().map(|f| &f.ty);
        let fields_ident: &Vec<_> = &match variant.style {
            Style::Unit => {
                if variant.attrs.serialize_with().is_some() {
                    ::alloc::vec::Vec::new()
                } else {
                    ::core::panicking::panic("internal error: entered unreachable code")
                }
            }
            Style::Newtype => {
                <[_]>::into_vec(
                    #[rustc_box]
                    ::alloc::boxed::Box::new([
                        Member::Named(Ident::new("__field0", Span::call_site())),
                    ]),
                )
            }
            Style::Tuple => {
                (0..variant.fields.len())
                    .map(|i| Member::Named(
                        Ident::new(
                            &{
                                let res = ::alloc::fmt::format(
                                    ::core::fmt::Arguments::new_v1(
                                        &["__field"],
                                        &[::core::fmt::ArgumentV1::new_display(&i)],
                                    ),
                                );
                                res
                            },
                            Span::call_site(),
                        ),
                    ))
                    .collect()
            }
            Style::Struct => variant.fields.iter().map(|f| f.member.clone()).collect(),
        };
        let (_, ty_generics, where_clause) = params.generics.split_for_impl();
        let wrapper_generics = if fields_ident.is_empty() {
            params.generics.clone()
        } else {
            bound::with_lifetime_bound(&params.generics, "'__a")
        };
        let (wrapper_impl_generics, wrapper_ty_generics, _) = wrapper_generics
            .split_for_impl();
        crate::fragment::Fragment::Block({
            let mut _s = ::quote::__private::TokenStream::new();
            ::quote::__private::push_ident(&mut _s, "struct");
            ::quote::__private::push_ident(&mut _s, "__AdjacentlyTagged");
            ::quote::ToTokens::to_tokens(&wrapper_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "data");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            {
                                use ::quote::__private::ext::*;
                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                #[allow(unused_mut)]
                                let (mut fields_ty, i) = fields_ty.quote_into_iter();
                                let has_iter = has_iter | i;
                                let _: ::quote::__private::HasIterator = has_iter;
                                while true {
                                    let fields_ty = match fields_ty.next() {
                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                        None => break,
                                    };
                                    ::quote::__private::push_and(&mut _s);
                                    ::quote::__private::push_lifetime(&mut _s, "\'__a");
                                    ::quote::ToTokens::to_tokens(&fields_ty, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                }
                            };
                            _s
                        },
                    );
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "phantom");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_comma(&mut _s);
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "impl");
            ::quote::ToTokens::to_tokens(&wrapper_impl_generics, &mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "Serialize");
            ::quote::__private::push_ident(&mut _s, "for");
            ::quote::__private::push_ident(&mut _s, "__AdjacentlyTagged");
            ::quote::ToTokens::to_tokens(&wrapper_ty_generics, &mut _s);
            ::quote::ToTokens::to_tokens(&where_clause, &mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Brace,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "fn");
                    ::quote::__private::push_ident(&mut _s, "serialize");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__S");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_colon(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__S");
                            _s
                        },
                    );
                    ::quote::__private::push_rarrow(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__private");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Result");
                    ::quote::__private::push_lt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__S");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Ok");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "__S");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Error");
                    ::quote::__private::push_gt(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "where");
                    ::quote::__private::push_ident(&mut _s, "__S");
                    ::quote::__private::push_colon(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serializer");
                    ::quote::__private::push_comma(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Brace,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_pound(&mut _s);
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Bracket,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "allow");
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            ::quote::__private::push_ident(&mut _s, "unused_variables");
                                            _s
                                        },
                                    );
                                    _s
                                },
                            );
                            ::quote::__private::push_ident(&mut _s, "let");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    {
                                        use ::quote::__private::ext::*;
                                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                        #[allow(unused_mut)]
                                        let (mut fields_ident, i) = fields_ident.quote_into_iter();
                                        let has_iter = has_iter | i;
                                        let _: ::quote::__private::HasIterator = has_iter;
                                        while true {
                                            let fields_ident = match fields_ident.next() {
                                                Some(_x) => ::quote::__private::RepInterp(_x),
                                                None => break,
                                            };
                                            ::quote::ToTokens::to_tokens(&fields_ident, &mut _s);
                                            ::quote::__private::push_comma(&mut _s);
                                        }
                                    };
                                    _s
                                },
                            );
                            ::quote::__private::push_eq(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "self");
                            ::quote::__private::push_dot(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "data");
                            ::quote::__private::push_semi(&mut _s);
                            ::quote::ToTokens::to_tokens(&inner, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_ident(&mut _s, "let");
            ::quote::__private::push_ident(&mut _s, "mut");
            ::quote::__private::push_ident(&mut _s, "__struct");
            ::quote::__private::push_eq(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_struct");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::parse(&mut _s, "2");
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ser");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_field");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__struct");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&tag, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "try");
            ::quote::__private::push_bang(&mut _s);
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ser");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_field");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "mut");
                            ::quote::__private::push_ident(&mut _s, "__struct");
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::ToTokens::to_tokens(&content, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_and(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "__AdjacentlyTagged",
                            );
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Brace,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "data");
                                    ::quote::__private::push_colon(&mut _s);
                                    ::quote::__private::push_group(
                                        &mut _s,
                                        ::quote::__private::Delimiter::Parenthesis,
                                        {
                                            let mut _s = ::quote::__private::TokenStream::new();
                                            {
                                                use ::quote::__private::ext::*;
                                                let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                                                #[allow(unused_mut)]
                                                let (mut fields_ident, i) = fields_ident.quote_into_iter();
                                                let has_iter = has_iter | i;
                                                let _: ::quote::__private::HasIterator = has_iter;
                                                while true {
                                                    let fields_ident = match fields_ident.next() {
                                                        Some(_x) => ::quote::__private::RepInterp(_x),
                                                        None => break,
                                                    };
                                                    ::quote::ToTokens::to_tokens(&fields_ident, &mut _s);
                                                    ::quote::__private::push_comma(&mut _s);
                                                }
                                            };
                                            _s
                                        },
                                    );
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "phantom");
                                    ::quote::__private::push_colon(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "_serde");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "__private");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "PhantomData");
                                    ::quote::__private::push_colon2(&mut _s);
                                    ::quote::__private::push_lt(&mut _s);
                                    ::quote::ToTokens::to_tokens(&this, &mut _s);
                                    ::quote::ToTokens::to_tokens(&ty_generics, &mut _s);
                                    ::quote::__private::push_gt(&mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    _s
                },
            );
            ::quote::__private::push_semi(&mut _s);
            ::quote::__private::push_ident(&mut _s, "_serde");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "ser");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "SerializeStruct");
            ::quote::__private::push_colon2(&mut _s);
            ::quote::__private::push_ident(&mut _s, "end");
            ::quote::__private::push_group(
                &mut _s,
                ::quote::__private::Delimiter::Parenthesis,
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__struct");
                    _s
                },
            );
            _s
        })
    }
    fn serialize_untagged_variant(
        params: &Parameters,
        variant: &Variant,
        cattrs: &attr::Container,
    ) -> Fragment {
        if let Some(path) = variant.attrs.serialize_with() {
            let ser = wrap_serialize_variant_with(params, path, variant);
            return crate::fragment::Fragment::Expr({
                let mut _s = ::quote::__private::TokenStream::new();
                ::quote::__private::push_ident(&mut _s, "_serde");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "Serialize");
                ::quote::__private::push_colon2(&mut _s);
                ::quote::__private::push_ident(&mut _s, "serialize");
                ::quote::__private::push_group(
                    &mut _s,
                    ::quote::__private::Delimiter::Parenthesis,
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::ToTokens::to_tokens(&ser, &mut _s);
                        ::quote::__private::push_comma(&mut _s);
                        ::quote::__private::push_ident(&mut _s, "__serializer");
                        _s
                    },
                );
                _s
            });
        }
        match effective_style(variant) {
            Style::Unit => {
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "Serializer");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "serialize_unit");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Newtype => {
                let field = &variant.fields[0];
                let mut field_expr = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "__field0");
                    _s
                };
                if let Some(path) = field.attrs.serialize_with() {
                    field_expr = wrap_serialize_field_with(
                        params,
                        field.ty,
                        path,
                        &field_expr,
                    );
                }
                let span = field.original.span();
                let func = {
                    let mut _s = ::quote::__private::TokenStream::new();
                    let _span: ::quote::__private::Span = span;
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "_serde");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "Serialize");
                    ::quote::__private::push_colon2_spanned(&mut _s, _span);
                    ::quote::__private::push_ident_spanned(&mut _s, _span, "serialize");
                    _s
                };
                crate::fragment::Fragment::Expr({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&func, &mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                            ::quote::__private::push_comma(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "__serializer");
                            _s
                        },
                    );
                    _s
                })
            }
            Style::Tuple => {
                serialize_tuple_variant(TupleVariant::Untagged, params, &variant.fields)
            }
            Style::Struct => {
                let type_name = cattrs.name().serialize_name();
                serialize_struct_variant(
                    StructVariant::Untagged,
                    params,
                    &variant.fields,
                    &type_name,
                )
            }
        }
    }
    enum TupleVariant {
        ExternallyTagged { type_name: String, variant_index: u32, variant_name: String },
        Untagged,
    }
    fn serialize_tuple_variant(
        context: TupleVariant,
        params: &Parameters,
        fields: &[Field],
    ) -> Fragment {
        let tuple_trait = match context {
            TupleVariant::ExternallyTagged { .. } => TupleTrait::SerializeTupleVariant,
            TupleVariant::Untagged => TupleTrait::SerializeTuple,
        };
        let serialize_stmts = serialize_tuple_struct_visitor(
            fields,
            params,
            true,
            &tuple_trait,
        );
        let mut serialized_fields = fields
            .iter()
            .enumerate()
            .filter(|(_, field)| !field.attrs.skip_serializing())
            .peekable();
        let let_mut = mut_if(serialized_fields.peek().is_some());
        let len = serialized_fields
            .map(|(i, field)| match field.attrs.skip_serializing_if() {
                None => {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::parse(&mut _s, "1");
                    _s
                }
                Some(path) => {
                    let field_expr = Ident::new(
                        &{
                            let res = ::alloc::fmt::format(
                                ::core::fmt::Arguments::new_v1(
                                    &["__field"],
                                    &[::core::fmt::ArgumentV1::new_display(&i)],
                                ),
                            );
                            res
                        },
                        Span::call_site(),
                    );
                    {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "if");
                        ::quote::ToTokens::to_tokens(&path, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&field_expr, &mut _s);
                                _s
                            },
                        );
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "0");
                                _s
                            },
                        );
                        ::quote::__private::push_ident(&mut _s, "else");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "1");
                                _s
                            },
                        );
                        _s
                    }
                }
            })
            .fold(
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::parse(&mut _s, "0");
                    _s
                },
                |sum, expr| {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&sum, &mut _s);
                    ::quote::__private::push_add(&mut _s);
                    ::quote::ToTokens::to_tokens(&expr, &mut _s);
                    _s
                },
            );
        match context {
            TupleVariant::ExternallyTagged { type_name, variant_index, variant_name } => {
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::ToTokens::to_tokens(&let_mut, &mut _s);
                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Serializer");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "serialize_tuple_variant",
                            );
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__serializer");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&type_name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_index, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&len, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut serialize_stmts, i) = serialize_stmts.quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let serialize_stmts = match serialize_stmts.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::ToTokens::to_tokens(&serialize_stmts, &mut _s);
                        }
                    };
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ser");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SerializeTupleVariant");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "end");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serde_state");
                            _s
                        },
                    );
                    _s
                })
            }
            TupleVariant::Untagged => {
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::ToTokens::to_tokens(&let_mut, &mut _s);
                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Serializer");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "serialize_tuple");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__serializer");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&len, &mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut serialize_stmts, i) = serialize_stmts.quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let serialize_stmts = match serialize_stmts.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::ToTokens::to_tokens(&serialize_stmts, &mut _s);
                        }
                    };
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ser");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SerializeTuple");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "end");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serde_state");
                            _s
                        },
                    );
                    _s
                })
            }
        }
    }
    enum StructVariant<'a> {
        ExternallyTagged { variant_index: u32, variant_name: String },
        InternallyTagged { tag: &'a str, variant_name: String },
        Untagged,
    }
    fn serialize_struct_variant<'a>(
        context: StructVariant<'a>,
        params: &Parameters,
        fields: &[Field],
        name: &str,
    ) -> Fragment {
        if fields.iter().any(|field| field.attrs.flatten()) {
            return serialize_struct_variant_with_flatten(context, params, fields, name);
        }
        let struct_trait = match context {
            StructVariant::ExternallyTagged { .. } => StructTrait::SerializeStructVariant,
            StructVariant::InternallyTagged { .. } | StructVariant::Untagged => {
                StructTrait::SerializeStruct
            }
        };
        let serialize_fields = serialize_struct_visitor(
            fields,
            params,
            true,
            &struct_trait,
        );
        let mut serialized_fields = fields
            .iter()
            .filter(|&field| !field.attrs.skip_serializing())
            .peekable();
        let let_mut = mut_if(serialized_fields.peek().is_some());
        let len = serialized_fields
            .map(|field| {
                let member = &field.member;
                match field.attrs.skip_serializing_if() {
                    Some(path) => {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::push_ident(&mut _s, "if");
                        ::quote::ToTokens::to_tokens(&path, &mut _s);
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Parenthesis,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::ToTokens::to_tokens(&member, &mut _s);
                                _s
                            },
                        );
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "0");
                                _s
                            },
                        );
                        ::quote::__private::push_ident(&mut _s, "else");
                        ::quote::__private::push_group(
                            &mut _s,
                            ::quote::__private::Delimiter::Brace,
                            {
                                let mut _s = ::quote::__private::TokenStream::new();
                                ::quote::__private::parse(&mut _s, "1");
                                _s
                            },
                        );
                        _s
                    }
                    None => {
                        let mut _s = ::quote::__private::TokenStream::new();
                        ::quote::__private::parse(&mut _s, "1");
                        _s
                    }
                }
            })
            .fold(
                {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::parse(&mut _s, "0");
                    _s
                },
                |sum, expr| {
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::ToTokens::to_tokens(&sum, &mut _s);
                    ::quote::__private::push_add(&mut _s);
                    ::quote::ToTokens::to_tokens(&expr, &mut _s);
                    _s
                },
            );
        match context {
            StructVariant::ExternallyTagged { variant_index, variant_name } => {
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::ToTokens::to_tokens(&let_mut, &mut _s);
                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Serializer");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(
                                &mut _s,
                                "serialize_struct_variant",
                            );
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__serializer");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_index, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&len, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut serialize_fields, i) = serialize_fields
                            .quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let serialize_fields = match serialize_fields.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::ToTokens::to_tokens(&serialize_fields, &mut _s);
                        }
                    };
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ser");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "SerializeStructVariant");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "end");
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "__serde_state");
                            _s
                        },
                    );
                    _s
                })
            }
            StructVariant::InternallyTagged { tag, variant_name } => {
                crate::fragment::Fragment::Block({
                    let mut _s = ::quote::__private::TokenStream::new();
                    ::quote::__private::push_ident(&mut _s, "let");
                    ::quote::__private::push_ident(&mut _s, "mut");
                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                    ::quote::__private::push_eq(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "Serializer");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "serialize_struct");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_ident(&mut _s, "__serializer");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&len, &mut _s);
                                    ::quote::__private::push_add(&mut _s);
                                    ::quote::__private::parse(&mut _s, "1");
                                    ::quote::__private::push_comma(&mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "try");
                    ::quote::__private::push_bang(&mut _s);
                    ::quote::__private::push_group(
                        &mut _s,
                        ::quote::__private::Delimiter::Parenthesis,
                        {
                            let mut _s = ::quote::__private::TokenStream::new();
                            ::quote::__private::push_ident(&mut _s, "_serde");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "ser");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "SerializeStruct");
                            ::quote::__private::push_colon2(&mut _s);
                            ::quote::__private::push_ident(&mut _s, "serialize_field");
                            ::quote::__private::push_group(
                                &mut _s,
                                ::quote::__private::Delimiter::Parenthesis,
                                {
                                    let mut _s = ::quote::__private::TokenStream::new();
                                    ::quote::__private::push_and(&mut _s);
                                    ::quote::__private::push_ident(&mut _s, "mut");
                                    ::quote::__private::push_ident(&mut _s, "__serde_state");
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&tag, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    ::quote::ToTokens::to_tokens(&variant_name, &mut _s);
                                    ::quote::__private::push_comma(&mut _s);
                                    _s
                                },
                            );
                            _s
                        },
                    );
                    ::quote::__private::push_semi(&mut _s);
                    {
                        use ::quote::__private::ext::*;
                        let has_iter = ::quote::__private::ThereIsNoIteratorInRepetition;
                        #[allow(unused_mut)]
                        let (mut serialize_fields, i) = serialize_fields
                            .quote_into_iter();
                        let has_iter = has_iter | i;
                        let _: ::quote::__private::HasIterator = has_iter;
                        while true {
                            let serialize_fields = match serialize_fields.next() {
                                Some(_x) => ::quote::__private::RepInterp(_x),
                                None => break,
                            };
                            ::quote::ToTokens::to_tokens(&serialize_fields, &mut _s);
                        }
                    };
                    ::quote::__private::push_ident(&mut _s, "_serde");
                    ::quote::__private::push_colon2(&mut _s);
                    ::quote::__private::push_ident(&mut _s, "ser");
                    ::quote::__private::