﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/bedrock-agent-runtime/model/FunctionDefinition.h>
#include <aws/core/utils/json/JsonSerializer.h>

#include <utility>

using namespace Aws::Utils::Json;
using namespace Aws::Utils;

namespace Aws
{
namespace BedrockAgentRuntime
{
namespace Model
{

FunctionDefinition::FunctionDefinition() : 
    m_descriptionHasBeenSet(false),
    m_nameHasBeenSet(false),
    m_parametersHasBeenSet(false),
    m_requireConfirmation(RequireConfirmation::NOT_SET),
    m_requireConfirmationHasBeenSet(false)
{
}

FunctionDefinition::FunctionDefinition(JsonView jsonValue)
  : FunctionDefinition()
{
  *this = jsonValue;
}

FunctionDefinition& FunctionDefinition::operator =(JsonView jsonValue)
{
  if(jsonValue.ValueExists("description"))
  {
    m_description = jsonValue.GetString("description");

    m_descriptionHasBeenSet = true;
  }

  if(jsonValue.ValueExists("name"))
  {
    m_name = jsonValue.GetString("name");

    m_nameHasBeenSet = true;
  }

  if(jsonValue.ValueExists("parameters"))
  {
    Aws::Map<Aws::String, JsonView> parametersJsonMap = jsonValue.GetObject("parameters").GetAllObjects();
    for(auto& parametersItem : parametersJsonMap)
    {
      m_parameters[parametersItem.first] = parametersItem.second.AsObject();
    }
    m_parametersHasBeenSet = true;
  }

  if(jsonValue.ValueExists("requireConfirmation"))
  {
    m_requireConfirmation = RequireConfirmationMapper::GetRequireConfirmationForName(jsonValue.GetString("requireConfirmation"));

    m_requireConfirmationHasBeenSet = true;
  }

  return *this;
}

JsonValue FunctionDefinition::Jsonize() const
{
  JsonValue payload;

  if(m_descriptionHasBeenSet)
  {
   payload.WithString("description", m_description);

  }

  if(m_nameHasBeenSet)
  {
   payload.WithString("name", m_name);

  }

  if(m_parametersHasBeenSet)
  {
   JsonValue parametersJsonMap;
   for(auto& parametersItem : m_parameters)
   {
     parametersJsonMap.WithObject(parametersItem.first, parametersItem.second.Jsonize());
   }
   payload.WithObject("parameters", std::move(parametersJsonMap));

  }

  if(m_requireConfirmationHasBeenSet)
  {
   payload.WithString("requireConfirmation", RequireConfirmationMapper::GetNameForRequireConfirmation(m_requireConfirmation));
  }

  return payload;
}

} // namespace Model
} // namespace BedrockAgentRuntime
} // namespace Aws
