%feature("docstring") OT::RandomWalkMetropolisHastings
"Random Walk Metropolis-Hastings method.

Refer to :ref:`bayesian_calibration`, :ref:`metropolis_hastings`.

Available constructor:
    RandomWalkMetropolisHastings(*targetDistribution, initialState, proposal, marginalIndices*)

    RandomWalkMetropolisHastings(*targetLogPDF, support, initialState, proposal, marginalIndices*)

Parameters
----------
targetDistribution : :class:`~openturns.Distribution`
    Target distribution sampled
targetLogPDF : :class:`~openturns.Function`
    Target log-density up to an additive constant
support : :class:`~openturns.Domain`
    Support of the target when defined with *targetLogPDF*
initialState : sequence of float
    Initial state of the chain
proposal : :class:`~openturns.Distribution`
    Distribution of the steps of the random walk
marginalIndices : sequence of int, optional
    Indices of the components to be updated.
    If not specified, all components are updated.
    The number of updated components must be equal to the dimension of *proposal*.

Notes
-----
The random walk Metropolis-Hastings algorithm
is a Markov Chain Monte-Carlo algorithm.
It draws candidates for the
next state of the chain as follows: denoting the current state by
:math:`\\vect{\\theta}^k`, the candidate :math:`\\vect{c}^k` for
:math:`\\vect{\\theta}^{k+1}` can be expressed as
:math:`\\vect{c}^k = \\vect{\\theta}^k +\\vect{\\delta}^k` where the
distribution of :math:`\\vect{\\delta}^k` is the provided *proposal* distribution.

See Also
--------
Gibbs, RandomVectorMetropolisHastings

Examples
--------
>>> import openturns as ot
>>> import math as m
>>> ot.RandomGenerator.SetSeed(0)

Sample from a target distribution defined through its log-PDF (defined up to some additive constant) and its support:

>>> log_density = ot.SymbolicFunction('x', 'log(2 + sin(x)^2) - (2 + cos(3*x)^3 + sin(2*x)^3) * x')
>>> support = ot.Interval([0.0], [2.0 * m.pi])
>>> proposal = ot.Normal(0.0, 2.0)
>>> initialState = [3.0]
>>> sampler = ot.RandomWalkMetropolisHastings(log_density, support, initialState, proposal)
>>> x = sampler.getSample(10)

The target distribution can also be a :class:`~openturns.Distribution`.
Since all :class:`~openturns.Distribution` objects have a :meth:`~openturns.Distribution.getSample`
method, this is only useful in Bayesian settings where we also define a likelihood.
The likelihood function penalizes the initially provided target distribution (now viewed as the *prior*)
in order to get the *posterior* distribution.
We sample from the posterior.

>>> prior = ot.ComposedDistribution([ot.Uniform(-100.0, 100.0)] * 2)
>>> proposal = ot.Normal([0.0] * 2, [0.5, 0.05], ot.IdentityMatrix(2))
>>> initialState = [0.0] * 2
>>> sampler = ot.RandomWalkMetropolisHastings(prior, initialState, proposal)
>>> conditional = ot.Bernoulli()
>>> data = ot.Sample([[53, 1], [57, 1], [58, 1], [63, 1], [66, 0], [67, 0],
... [67, 0], [67, 0], [68, 0], [69, 0], [70, 0], [70, 0], [70, 1], [70, 1],
... [72, 0], [73, 0], [75, 0], [75, 1], [76, 0], [76, 0], [78, 0], [79, 0], [81, 0]])
>>> observations = data[:, 1]
>>> covariates = data[:, 0]
>>> fun = ot.SymbolicFunction(['alpha', 'beta', 'x'], ['exp(alpha + beta * x) / (1 + exp(alpha + beta * x))'])
>>> linkFunction = ot.ParametricFunction(fun, [2], [0.0])
>>> sampler.setLikelihood(conditional, observations, linkFunction, covariates)
>>> alpha_beta = sampler.getSample(10)"

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::getProposal
"Get the proposal distribution.

Returns
-------
proposal : :class:`~openturns.Distribution`
    The distribution from which the transition kernels of the random walk
    Metropolis-Hastings algorithm is defined"

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::setProposal
"Set the proposal distribution.

Parameters
----------
proposal : :class:`~openturns.Distribution`
    The distribution from which the transition kernels of the random walk
    Metropolis-Hastings algorithm is defined"

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::getBurnIn
"Get the length of the burn-in period.

Returns
-------
burnin : int
    Length of the burn-in period: this is
    the number of iterations after which adaptation stops."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::setBurnIn
"Set the length of the burn-in period.

Parameters
----------
burnin : int
    Length of the burn-in period: this is
    the number of iterations after which adaptation stops."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::getAdaptationPeriod
"Get the adaptation step.

Returns
-------
period : positive int
    During burn-in, adaptation is performed once every *period* iterations."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::setAdaptationPeriod
"Set the adaptation period.

Parameters
----------
period : positive int
    During burn-in, adaptation is performed once every *period* iterations."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::getAdaptationExpansionFactor
"Get the adaptation expansion factor.

Returns
-------
expansionFactor : float
    Expansion factor :math:`e` of the adaptation factor."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::setAdaptationExpansionFactor
"Set the adaptation expansion factor.

Parameters
----------
expansionFactor : float, :math:`e > 1`
    Expansion factor :math:`e` of the adaptation factor."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::getAdaptationRange
"Get the expected range for the acceptance rate.
During burn-in, at the end of every adaptation period,
if the acceptance rate does not belong to this range,
the adaptation factor is multiplied
either by the expansion factor
(if the acceptance rate is larger than the *upperBound*)
or by the shrink factor
(if the acceptance rate is smaller than the *lowerBound*).

Returns
-------
range : :class:`~openturns.Interval` of dimension 1
    Range [*lowerBound*, *upperBound*] of the expected acceptance rate."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::setAdaptationRange
"Set the expected range for the acceptance rate.
During burn-in, at the end of every adaptation period,
if the acceptance rate does not belong to this range,
the adaptation factor is multiplied
either by the expansion factor
(if the acceptance rate is larger than the *upperBound*)
or by the shrink factor
(if the acceptance rate is smaller than the *lowerBound*).

Parameters
----------
range : :class:`~openturns.Interval` of dimension 1
    Range [*lowerBound*, *upperBound*] of the expected acceptance rate."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::getAdaptationShrinkFactor
"Get the adaptation shrink factor.

Returns
-------
shrinkFactor : float
    Shrink factor :math:`s` of the adaptation factor."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::setAdaptationShrinkFactor
"Set the adaptation shrink factor.

Parameters
----------
shrinkFactor : float, :math:`0 < s < 1`
    Shrink factor :math:`s` of the adaptation factor."

// ---------------------------------------------------------------------

%feature("docstring") OT::RandomWalkMetropolisHastings::getAdaptationFactor
"Get the adaptation factor.

Returns
-------
factor : float
    Current adaptation factor, for inspection."
