










package com.adacore.libadalang;

import java.util.Arrays;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Objects;

import java.lang.StringBuilder;
import java.lang.Iterable;
import java.lang.reflect.Method;

import java.math.BigInteger;

import java.io.File;
import java.nio.ByteOrder;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.strings.TruffleString;

import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.ImageInfo;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.StackValue;
import org.graalvm.nativeimage.UnmanagedMemory;
import org.graalvm.nativeimage.c.CContext;
import org.graalvm.nativeimage.c.function.CEntryPoint;
import org.graalvm.nativeimage.c.function.CEntryPointLiteral;
import org.graalvm.nativeimage.c.function.CFunction;
import org.graalvm.nativeimage.c.function.CFunctionPointer;
import org.graalvm.nativeimage.c.function.InvokeCFunctionPointer;
import org.graalvm.nativeimage.c.struct.*;
import org.graalvm.nativeimage.c.type.*;
import org.graalvm.word.PointerBase;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordFactory;


/*
====================

This is the Java bindings of the libadalang API.
You can use all functionalities of the library in a Java environment.
Those bindings call the native library using JNI and Native-Image C API.

====================
*/
public final class Libadalang {

    // ==========
    // Native entry points
    // ==========

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit requests events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitRequested(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final TextNative nameNative,
        final AnalysisUnitNative fromNative,
        final byte foundNative,
        final byte isNotFoundErrorNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            );
            final Text text = Text.wrap(nameNative)
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitRequestedCallback callback = context
                .getEventHandler()
                .getUnitRequestCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    text.getContent(),
                    AnalysisUnit.wrap(fromNative),
                    foundNative != 0,
                    isNotFoundErrorNative != 0
                );
            }
        }
    }

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit parsing events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitParsed(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final AnalysisUnitNative unitNative,
        final byte reparsedNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            )
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitParsedCallback callback = context
                .getEventHandler()
                .getUnitParsedCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    AnalysisUnit.wrap(unitNative),
                    reparsedNative != 0
                );
            }
        }
    }

    // ==========
    // Static values
    // ==========

    /** The default grammar rule to parse the inputs. */
    private static final GrammarRule DEFAULT_GRAMMAR_RULE =
        GrammarRule.COMPILATION_RULE;

    /** The os name in lower case. */
    private static final String OS =
            System.getProperty("os.name").toLowerCase();

    /** The byte order of the system. */
    private static final ByteOrder BYTE_ORDER = ByteOrder.nativeOrder();

    /** The node to convert a Java string to a truffle string */
    private static final TruffleString.FromJavaStringNode fromJavaStringNode =
        TruffleString.FromJavaStringNode.create();

    /** The node to convert a truffle string to a Java string. */
    private static final TruffleString.ToJavaStringNode toJavaStringNode =
        TruffleString.ToJavaStringNode.create();

    /** The node to convert a byte array to a truffle string. */
    private static final TruffleString.FromByteArrayNode fromByteArrayNode =
        TruffleString.FromByteArrayNode.create();

    /** The node to convert a truffle string to a byte array. */
    private static final TruffleString.CopyToByteArrayNode toByteArrayNode =
        TruffleString.CopyToByteArrayNode.create();

    /** A map to store the nodes classes from their name. */
    public static final Map<String, Class<? extends AdaNode>>
        NODE_CLASS_MAP = new HashMap<>();

    static {
        // Populate the node class map
        NODE_CLASS_MAP.put(
            "AdaNode",
            AdaNode.class
        );
        NODE_CLASS_MAP.put(
            "AbortNode",
            AbortNode.class
        );
        NODE_CLASS_MAP.put(
            "AbortAbsent",
            AbortAbsent.class
        );
        NODE_CLASS_MAP.put(
            "AbortPresent",
            AbortPresent.class
        );
        NODE_CLASS_MAP.put(
            "AbstractNode",
            AbstractNode.class
        );
        NODE_CLASS_MAP.put(
            "AbstractAbsent",
            AbstractAbsent.class
        );
        NODE_CLASS_MAP.put(
            "AbstractPresent",
            AbstractPresent.class
        );
        NODE_CLASS_MAP.put(
            "AdaList",
            AdaList.class
        );
        NODE_CLASS_MAP.put(
            "AdaNodeList",
            AdaNodeList.class
        );
        NODE_CLASS_MAP.put(
            "AbstractStateDeclList",
            AbstractStateDeclList.class
        );
        NODE_CLASS_MAP.put(
            "AlternativesList",
            AlternativesList.class
        );
        NODE_CLASS_MAP.put(
            "ConstraintList",
            ConstraintList.class
        );
        NODE_CLASS_MAP.put(
            "DeclList",
            DeclList.class
        );
        NODE_CLASS_MAP.put(
            "StmtList",
            StmtList.class
        );
        NODE_CLASS_MAP.put(
            "AspectAssocList",
            AspectAssocList.class
        );
        NODE_CLASS_MAP.put(
            "BaseAssocList",
            BaseAssocList.class
        );
        NODE_CLASS_MAP.put(
            "BasicAssocList",
            BasicAssocList.class
        );
        NODE_CLASS_MAP.put(
            "AssocList",
            AssocList.class
        );
        NODE_CLASS_MAP.put(
            "BasicDeclList",
            BasicDeclList.class
        );
        NODE_CLASS_MAP.put(
            "CaseExprAlternativeList",
            CaseExprAlternativeList.class
        );
        NODE_CLASS_MAP.put(
            "CaseStmtAlternativeList",
            CaseStmtAlternativeList.class
        );
        NODE_CLASS_MAP.put(
            "CompilationUnitList",
            CompilationUnitList.class
        );
        NODE_CLASS_MAP.put(
            "ConcatOperandList",
            ConcatOperandList.class
        );
        NODE_CLASS_MAP.put(
            "ContractCaseAssocList",
            ContractCaseAssocList.class
        );
        NODE_CLASS_MAP.put(
            "DefiningNameList",
            DefiningNameList.class
        );
        NODE_CLASS_MAP.put(
            "DiscriminantSpecList",
            DiscriminantSpecList.class
        );
        NODE_CLASS_MAP.put(
            "ElsifExprPartList",
            ElsifExprPartList.class
        );
        NODE_CLASS_MAP.put(
            "ElsifStmtPartList",
            ElsifStmtPartList.class
        );
        NODE_CLASS_MAP.put(
            "EnumLiteralDeclList",
            EnumLiteralDeclList.class
        );
        NODE_CLASS_MAP.put(
            "ExprList",
            ExprList.class
        );
        NODE_CLASS_MAP.put(
            "ExprAlternativesList",
            ExprAlternativesList.class
        );
        NODE_CLASS_MAP.put(
            "IdentifierList",
            IdentifierList.class
        );
        NODE_CLASS_MAP.put(
            "DiscriminantChoiceList",
            DiscriminantChoiceList.class
        );
        NODE_CLASS_MAP.put(
            "NameList",
            NameList.class
        );
        NODE_CLASS_MAP.put(
            "ParentList",
            ParentList.class
        );
        NODE_CLASS_MAP.put(
            "ParamSpecList",
            ParamSpecList.class
        );
        NODE_CLASS_MAP.put(
            "PragmaNodeList",
            PragmaNodeList.class
        );
        NODE_CLASS_MAP.put(
            "SelectWhenPartList",
            SelectWhenPartList.class
        );
        NODE_CLASS_MAP.put(
            "UnconstrainedArrayIndexList",
            UnconstrainedArrayIndexList.class
        );
        NODE_CLASS_MAP.put(
            "VariantList",
            VariantList.class
        );
        NODE_CLASS_MAP.put(
            "AliasedNode",
            AliasedNode.class
        );
        NODE_CLASS_MAP.put(
            "AliasedAbsent",
            AliasedAbsent.class
        );
        NODE_CLASS_MAP.put(
            "AliasedPresent",
            AliasedPresent.class
        );
        NODE_CLASS_MAP.put(
            "AllNode",
            AllNode.class
        );
        NODE_CLASS_MAP.put(
            "AllAbsent",
            AllAbsent.class
        );
        NODE_CLASS_MAP.put(
            "AllPresent",
            AllPresent.class
        );
        NODE_CLASS_MAP.put(
            "ArrayIndices",
            ArrayIndices.class
        );
        NODE_CLASS_MAP.put(
            "ConstrainedArrayIndices",
            ConstrainedArrayIndices.class
        );
        NODE_CLASS_MAP.put(
            "UnconstrainedArrayIndices",
            UnconstrainedArrayIndices.class
        );
        NODE_CLASS_MAP.put(
            "AspectAssoc",
            AspectAssoc.class
        );
        NODE_CLASS_MAP.put(
            "AspectClause",
            AspectClause.class
        );
        NODE_CLASS_MAP.put(
            "AtClause",
            AtClause.class
        );
        NODE_CLASS_MAP.put(
            "AttributeDefClause",
            AttributeDefClause.class
        );
        NODE_CLASS_MAP.put(
            "EnumRepClause",
            EnumRepClause.class
        );
        NODE_CLASS_MAP.put(
            "RecordRepClause",
            RecordRepClause.class
        );
        NODE_CLASS_MAP.put(
            "AspectSpec",
            AspectSpec.class
        );
        NODE_CLASS_MAP.put(
            "BaseAssoc",
            BaseAssoc.class
        );
        NODE_CLASS_MAP.put(
            "ContractCaseAssoc",
            ContractCaseAssoc.class
        );
        NODE_CLASS_MAP.put(
            "PragmaArgumentAssoc",
            PragmaArgumentAssoc.class
        );
        NODE_CLASS_MAP.put(
            "BaseFormalParamHolder",
            BaseFormalParamHolder.class
        );
        NODE_CLASS_MAP.put(
            "BaseSubpSpec",
            BaseSubpSpec.class
        );
        NODE_CLASS_MAP.put(
            "EntrySpec",
            EntrySpec.class
        );
        NODE_CLASS_MAP.put(
            "EnumSubpSpec",
            EnumSubpSpec.class
        );
        NODE_CLASS_MAP.put(
            "SubpSpec",
            SubpSpec.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticBinarySpec",
            SyntheticBinarySpec.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticUnarySpec",
            SyntheticUnarySpec.class
        );
        NODE_CLASS_MAP.put(
            "ComponentList",
            ComponentList.class
        );
        NODE_CLASS_MAP.put(
            "DiscriminantPart",
            DiscriminantPart.class
        );
        NODE_CLASS_MAP.put(
            "KnownDiscriminantPart",
            KnownDiscriminantPart.class
        );
        NODE_CLASS_MAP.put(
            "UnknownDiscriminantPart",
            UnknownDiscriminantPart.class
        );
        NODE_CLASS_MAP.put(
            "EntryCompletionFormalParams",
            EntryCompletionFormalParams.class
        );
        NODE_CLASS_MAP.put(
            "GenericFormalPart",
            GenericFormalPart.class
        );
        NODE_CLASS_MAP.put(
            "BaseRecordDef",
            BaseRecordDef.class
        );
        NODE_CLASS_MAP.put(
            "NullRecordDef",
            NullRecordDef.class
        );
        NODE_CLASS_MAP.put(
            "RecordDef",
            RecordDef.class
        );
        NODE_CLASS_MAP.put(
            "BasicAssoc",
            BasicAssoc.class
        );
        NODE_CLASS_MAP.put(
            "AggregateAssoc",
            AggregateAssoc.class
        );
        NODE_CLASS_MAP.put(
            "MultiDimArrayAssoc",
            MultiDimArrayAssoc.class
        );
        NODE_CLASS_MAP.put(
            "CompositeConstraintAssoc",
            CompositeConstraintAssoc.class
        );
        NODE_CLASS_MAP.put(
            "IteratedAssoc",
            IteratedAssoc.class
        );
        NODE_CLASS_MAP.put(
            "ParamAssoc",
            ParamAssoc.class
        );
        NODE_CLASS_MAP.put(
            "BasicDecl",
            BasicDecl.class
        );
        NODE_CLASS_MAP.put(
            "AbstractStateDecl",
            AbstractStateDecl.class
        );
        NODE_CLASS_MAP.put(
            "AnonymousExprDecl",
            AnonymousExprDecl.class
        );
        NODE_CLASS_MAP.put(
            "BaseFormalParamDecl",
            BaseFormalParamDecl.class
        );
        NODE_CLASS_MAP.put(
            "ComponentDecl",
            ComponentDecl.class
        );
        NODE_CLASS_MAP.put(
            "DiscriminantSpec",
            DiscriminantSpec.class
        );
        NODE_CLASS_MAP.put(
            "GenericFormal",
            GenericFormal.class
        );
        NODE_CLASS_MAP.put(
            "GenericFormalObjDecl",
            GenericFormalObjDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericFormalPackage",
            GenericFormalPackage.class
        );
        NODE_CLASS_MAP.put(
            "GenericFormalSubpDecl",
            GenericFormalSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericFormalTypeDecl",
            GenericFormalTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "ParamSpec",
            ParamSpec.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticFormalParamDecl",
            SyntheticFormalParamDecl.class
        );
        NODE_CLASS_MAP.put(
            "BasePackageDecl",
            BasePackageDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericPackageInternal",
            GenericPackageInternal.class
        );
        NODE_CLASS_MAP.put(
            "PackageDecl",
            PackageDecl.class
        );
        NODE_CLASS_MAP.put(
            "BaseTypeDecl",
            BaseTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "BaseSubtypeDecl",
            BaseSubtypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "DiscreteBaseSubtypeDecl",
            DiscreteBaseSubtypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "SubtypeDecl",
            SubtypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "ClasswideTypeDecl",
            ClasswideTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "IncompleteTypeDecl",
            IncompleteTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "IncompleteFormalTypeDecl",
            IncompleteFormalTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "IncompleteTaggedTypeDecl",
            IncompleteTaggedTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "ProtectedTypeDecl",
            ProtectedTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "TaskTypeDecl",
            TaskTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "SingleTaskTypeDecl",
            SingleTaskTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "TypeDecl",
            TypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "AnonymousTypeDecl",
            AnonymousTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "SynthAnonymousTypeDecl",
            SynthAnonymousTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "ConcreteTypeDecl",
            ConcreteTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "FormalTypeDecl",
            FormalTypeDecl.class
        );
        NODE_CLASS_MAP.put(
            "BasicSubpDecl",
            BasicSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "ClassicSubpDecl",
            ClassicSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "AbstractSubpDecl",
            AbstractSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "FormalSubpDecl",
            FormalSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "AbstractFormalSubpDecl",
            AbstractFormalSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "ConcreteFormalSubpDecl",
            ConcreteFormalSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "SubpDecl",
            SubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "EntryDecl",
            EntryDecl.class
        );
        NODE_CLASS_MAP.put(
            "EnumLiteralDecl",
            EnumLiteralDecl.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticCharEnumLit",
            SyntheticCharEnumLit.class
        );
        NODE_CLASS_MAP.put(
            "GenericSubpInternal",
            GenericSubpInternal.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticSubpDecl",
            SyntheticSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "BodyNode",
            BodyNode.class
        );
        NODE_CLASS_MAP.put(
            "AcceptStmtBody",
            AcceptStmtBody.class
        );
        NODE_CLASS_MAP.put(
            "BaseSubpBody",
            BaseSubpBody.class
        );
        NODE_CLASS_MAP.put(
            "ExprFunction",
            ExprFunction.class
        );
        NODE_CLASS_MAP.put(
            "NullSubpDecl",
            NullSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "SubpBody",
            SubpBody.class
        );
        NODE_CLASS_MAP.put(
            "SubpRenamingDecl",
            SubpRenamingDecl.class
        );
        NODE_CLASS_MAP.put(
            "BodyStub",
            BodyStub.class
        );
        NODE_CLASS_MAP.put(
            "PackageBodyStub",
            PackageBodyStub.class
        );
        NODE_CLASS_MAP.put(
            "ProtectedBodyStub",
            ProtectedBodyStub.class
        );
        NODE_CLASS_MAP.put(
            "SubpBodyStub",
            SubpBodyStub.class
        );
        NODE_CLASS_MAP.put(
            "TaskBodyStub",
            TaskBodyStub.class
        );
        NODE_CLASS_MAP.put(
            "EntryBody",
            EntryBody.class
        );
        NODE_CLASS_MAP.put(
            "PackageBody",
            PackageBody.class
        );
        NODE_CLASS_MAP.put(
            "ProtectedBody",
            ProtectedBody.class
        );
        NODE_CLASS_MAP.put(
            "TaskBody",
            TaskBody.class
        );
        NODE_CLASS_MAP.put(
            "EntryIndexSpec",
            EntryIndexSpec.class
        );
        NODE_CLASS_MAP.put(
            "ErrorDecl",
            ErrorDecl.class
        );
        NODE_CLASS_MAP.put(
            "ExceptionDecl",
            ExceptionDecl.class
        );
        NODE_CLASS_MAP.put(
            "ExceptionHandler",
            ExceptionHandler.class
        );
        NODE_CLASS_MAP.put(
            "ForLoopVarDecl",
            ForLoopVarDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericDecl",
            GenericDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericPackageDecl",
            GenericPackageDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericSubpDecl",
            GenericSubpDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericInstantiation",
            GenericInstantiation.class
        );
        NODE_CLASS_MAP.put(
            "GenericPackageInstantiation",
            GenericPackageInstantiation.class
        );
        NODE_CLASS_MAP.put(
            "GenericSubpInstantiation",
            GenericSubpInstantiation.class
        );
        NODE_CLASS_MAP.put(
            "GenericRenamingDecl",
            GenericRenamingDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericPackageRenamingDecl",
            GenericPackageRenamingDecl.class
        );
        NODE_CLASS_MAP.put(
            "GenericSubpRenamingDecl",
            GenericSubpRenamingDecl.class
        );
        NODE_CLASS_MAP.put(
            "LabelDecl",
            LabelDecl.class
        );
        NODE_CLASS_MAP.put(
            "NamedStmtDecl",
            NamedStmtDecl.class
        );
        NODE_CLASS_MAP.put(
            "NumberDecl",
            NumberDecl.class
        );
        NODE_CLASS_MAP.put(
            "ObjectDecl",
            ObjectDecl.class
        );
        NODE_CLASS_MAP.put(
            "ExtendedReturnStmtObjectDecl",
            ExtendedReturnStmtObjectDecl.class
        );
        NODE_CLASS_MAP.put(
            "NoTypeObjectRenamingDecl",
            NoTypeObjectRenamingDecl.class
        );
        NODE_CLASS_MAP.put(
            "PackageRenamingDecl",
            PackageRenamingDecl.class
        );
        NODE_CLASS_MAP.put(
            "SingleProtectedDecl",
            SingleProtectedDecl.class
        );
        NODE_CLASS_MAP.put(
            "SingleTaskDecl",
            SingleTaskDecl.class
        );
        NODE_CLASS_MAP.put(
            "CaseStmtAlternative",
            CaseStmtAlternative.class
        );
        NODE_CLASS_MAP.put(
            "CompilationUnit",
            CompilationUnit.class
        );
        NODE_CLASS_MAP.put(
            "ComponentClause",
            ComponentClause.class
        );
        NODE_CLASS_MAP.put(
            "ComponentDef",
            ComponentDef.class
        );
        NODE_CLASS_MAP.put(
            "ConstantNode",
            ConstantNode.class
        );
        NODE_CLASS_MAP.put(
            "ConstantAbsent",
            ConstantAbsent.class
        );
        NODE_CLASS_MAP.put(
            "ConstantPresent",
            ConstantPresent.class
        );
        NODE_CLASS_MAP.put(
            "Constraint",
            Constraint.class
        );
        NODE_CLASS_MAP.put(
            "CompositeConstraint",
            CompositeConstraint.class
        );
        NODE_CLASS_MAP.put(
            "DeltaConstraint",
            DeltaConstraint.class
        );
        NODE_CLASS_MAP.put(
            "DigitsConstraint",
            DigitsConstraint.class
        );
        NODE_CLASS_MAP.put(
            "RangeConstraint",
            RangeConstraint.class
        );
        NODE_CLASS_MAP.put(
            "DeclarativePart",
            DeclarativePart.class
        );
        NODE_CLASS_MAP.put(
            "PrivatePart",
            PrivatePart.class
        );
        NODE_CLASS_MAP.put(
            "PublicPart",
            PublicPart.class
        );
        NODE_CLASS_MAP.put(
            "ElsifExprPart",
            ElsifExprPart.class
        );
        NODE_CLASS_MAP.put(
            "ElsifStmtPart",
            ElsifStmtPart.class
        );
        NODE_CLASS_MAP.put(
            "Expr",
            Expr.class
        );
        NODE_CLASS_MAP.put(
            "AbstractStateDeclExpr",
            AbstractStateDeclExpr.class
        );
        NODE_CLASS_MAP.put(
            "Allocator",
            Allocator.class
        );
        NODE_CLASS_MAP.put(
            "BaseAggregate",
            BaseAggregate.class
        );
        NODE_CLASS_MAP.put(
            "Aggregate",
            Aggregate.class
        );
        NODE_CLASS_MAP.put(
            "BracketAggregate",
            BracketAggregate.class
        );
        NODE_CLASS_MAP.put(
            "DeltaAggregate",
            DeltaAggregate.class
        );
        NODE_CLASS_MAP.put(
            "BracketDeltaAggregate",
            BracketDeltaAggregate.class
        );
        NODE_CLASS_MAP.put(
            "NullRecordAggregate",
            NullRecordAggregate.class
        );
        NODE_CLASS_MAP.put(
            "BinOp",
            BinOp.class
        );
        NODE_CLASS_MAP.put(
            "RelationOp",
            RelationOp.class
        );
        NODE_CLASS_MAP.put(
            "BoxExpr",
            BoxExpr.class
        );
        NODE_CLASS_MAP.put(
            "CaseExprAlternative",
            CaseExprAlternative.class
        );
        NODE_CLASS_MAP.put(
            "ConcatOp",
            ConcatOp.class
        );
        NODE_CLASS_MAP.put(
            "ConcatOperand",
            ConcatOperand.class
        );
        NODE_CLASS_MAP.put(
            "CondExpr",
            CondExpr.class
        );
        NODE_CLASS_MAP.put(
            "CaseExpr",
            CaseExpr.class
        );
        NODE_CLASS_MAP.put(
            "IfExpr",
            IfExpr.class
        );
        NODE_CLASS_MAP.put(
            "ContractCases",
            ContractCases.class
        );
        NODE_CLASS_MAP.put(
            "DeclExpr",
            DeclExpr.class
        );
        NODE_CLASS_MAP.put(
            "MembershipExpr",
            MembershipExpr.class
        );
        NODE_CLASS_MAP.put(
            "Name",
            Name.class
        );
        NODE_CLASS_MAP.put(
            "AttributeRef",
            AttributeRef.class
        );
        NODE_CLASS_MAP.put(
            "CallExpr",
            CallExpr.class
        );
        NODE_CLASS_MAP.put(
            "DefiningName",
            DefiningName.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticDefiningName",
            SyntheticDefiningName.class
        );
        NODE_CLASS_MAP.put(
            "DiscreteSubtypeName",
            DiscreteSubtypeName.class
        );
        NODE_CLASS_MAP.put(
            "DottedName",
            DottedName.class
        );
        NODE_CLASS_MAP.put(
            "EndName",
            EndName.class
        );
        NODE_CLASS_MAP.put(
            "ExplicitDeref",
            ExplicitDeref.class
        );
        NODE_CLASS_MAP.put(
            "QualExpr",
            QualExpr.class
        );
        NODE_CLASS_MAP.put(
            "ReduceAttributeRef",
            ReduceAttributeRef.class
        );
        NODE_CLASS_MAP.put(
            "SingleTokNode",
            SingleTokNode.class
        );
        NODE_CLASS_MAP.put(
            "BaseId",
            BaseId.class
        );
        NODE_CLASS_MAP.put(
            "CharLiteral",
            CharLiteral.class
        );
        NODE_CLASS_MAP.put(
            "Identifier",
            Identifier.class
        );
        NODE_CLASS_MAP.put(
            "Op",
            Op.class
        );
        NODE_CLASS_MAP.put(
            "OpAbs",
            OpAbs.class
        );
        NODE_CLASS_MAP.put(
            "OpAnd",
            OpAnd.class
        );
        NODE_CLASS_MAP.put(
            "OpAndThen",
            OpAndThen.class
        );
        NODE_CLASS_MAP.put(
            "OpConcat",
            OpConcat.class
        );
        NODE_CLASS_MAP.put(
            "OpDiv",
            OpDiv.class
        );
        NODE_CLASS_MAP.put(
            "OpDoubleDot",
            OpDoubleDot.class
        );
        NODE_CLASS_MAP.put(
            "OpEq",
            OpEq.class
        );
        NODE_CLASS_MAP.put(
            "OpGt",
            OpGt.class
        );
        NODE_CLASS_MAP.put(
            "OpGte",
            OpGte.class
        );
        NODE_CLASS_MAP.put(
            "OpIn",
            OpIn.class
        );
        NODE_CLASS_MAP.put(
            "OpLt",
            OpLt.class
        );
        NODE_CLASS_MAP.put(
            "OpLte",
            OpLte.class
        );
        NODE_CLASS_MAP.put(
            "OpMinus",
            OpMinus.class
        );
        NODE_CLASS_MAP.put(
            "OpMod",
            OpMod.class
        );
        NODE_CLASS_MAP.put(
            "OpMult",
            OpMult.class
        );
        NODE_CLASS_MAP.put(
            "OpNeq",
            OpNeq.class
        );
        NODE_CLASS_MAP.put(
            "OpNot",
            OpNot.class
        );
        NODE_CLASS_MAP.put(
            "OpNotIn",
            OpNotIn.class
        );
        NODE_CLASS_MAP.put(
            "OpOr",
            OpOr.class
        );
        NODE_CLASS_MAP.put(
            "OpOrElse",
            OpOrElse.class
        );
        NODE_CLASS_MAP.put(
            "OpPlus",
            OpPlus.class
        );
        NODE_CLASS_MAP.put(
            "OpPow",
            OpPow.class
        );
        NODE_CLASS_MAP.put(
            "OpRem",
            OpRem.class
        );
        NODE_CLASS_MAP.put(
            "OpXor",
            OpXor.class
        );
        NODE_CLASS_MAP.put(
            "StringLiteral",
            StringLiteral.class
        );
        NODE_CLASS_MAP.put(
            "NullLiteral",
            NullLiteral.class
        );
        NODE_CLASS_MAP.put(
            "NumLiteral",
            NumLiteral.class
        );
        NODE_CLASS_MAP.put(
            "IntLiteral",
            IntLiteral.class
        );
        NODE_CLASS_MAP.put(
            "RealLiteral",
            RealLiteral.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticIdentifier",
            SyntheticIdentifier.class
        );
        NODE_CLASS_MAP.put(
            "TargetName",
            TargetName.class
        );
        NODE_CLASS_MAP.put(
            "UpdateAttributeRef",
            UpdateAttributeRef.class
        );
        NODE_CLASS_MAP.put(
            "ParenExpr",
            ParenExpr.class
        );
        NODE_CLASS_MAP.put(
            "QuantifiedExpr",
            QuantifiedExpr.class
        );
        NODE_CLASS_MAP.put(
            "RaiseExpr",
            RaiseExpr.class
        );
        NODE_CLASS_MAP.put(
            "UnOp",
            UnOp.class
        );
        NODE_CLASS_MAP.put(
            "HandledStmts",
            HandledStmts.class
        );
        NODE_CLASS_MAP.put(
            "InterfaceKind",
            InterfaceKind.class
        );
        NODE_CLASS_MAP.put(
            "InterfaceKindLimited",
            InterfaceKindLimited.class
        );
        NODE_CLASS_MAP.put(
            "InterfaceKindProtected",
            InterfaceKindProtected.class
        );
        NODE_CLASS_MAP.put(
            "InterfaceKindSynchronized",
            InterfaceKindSynchronized.class
        );
        NODE_CLASS_MAP.put(
            "InterfaceKindTask",
            InterfaceKindTask.class
        );
        NODE_CLASS_MAP.put(
            "IterType",
            IterType.class
        );
        NODE_CLASS_MAP.put(
            "IterTypeIn",
            IterTypeIn.class
        );
        NODE_CLASS_MAP.put(
            "IterTypeOf",
            IterTypeOf.class
        );
        NODE_CLASS_MAP.put(
            "LibraryItem",
            LibraryItem.class
        );
        NODE_CLASS_MAP.put(
            "LimitedNode",
            LimitedNode.class
        );
        NODE_CLASS_MAP.put(
            "LimitedAbsent",
            LimitedAbsent.class
        );
        NODE_CLASS_MAP.put(
            "LimitedPresent",
            LimitedPresent.class
        );
        NODE_CLASS_MAP.put(
            "LoopSpec",
            LoopSpec.class
        );
        NODE_CLASS_MAP.put(
            "ForLoopSpec",
            ForLoopSpec.class
        );
        NODE_CLASS_MAP.put(
            "WhileLoopSpec",
            WhileLoopSpec.class
        );
        NODE_CLASS_MAP.put(
            "Mode",
            Mode.class
        );
        NODE_CLASS_MAP.put(
            "ModeDefault",
            ModeDefault.class
        );
        NODE_CLASS_MAP.put(
            "ModeIn",
            ModeIn.class
        );
        NODE_CLASS_MAP.put(
            "ModeInOut",
            ModeInOut.class
        );
        NODE_CLASS_MAP.put(
            "ModeOut",
            ModeOut.class
        );
        NODE_CLASS_MAP.put(
            "MultiAbstractStateDecl",
            MultiAbstractStateDecl.class
        );
        NODE_CLASS_MAP.put(
            "NotNull",
            NotNull.class
        );
        NODE_CLASS_MAP.put(
            "NotNullAbsent",
            NotNullAbsent.class
        );
        NODE_CLASS_MAP.put(
            "NotNullPresent",
            NotNullPresent.class
        );
        NODE_CLASS_MAP.put(
            "NullComponentDecl",
            NullComponentDecl.class
        );
        NODE_CLASS_MAP.put(
            "OthersDesignator",
            OthersDesignator.class
        );
        NODE_CLASS_MAP.put(
            "OverridingNode",
            OverridingNode.class
        );
        NODE_CLASS_MAP.put(
            "OverridingNotOverriding",
            OverridingNotOverriding.class
        );
        NODE_CLASS_MAP.put(
            "OverridingOverriding",
            OverridingOverriding.class
        );
        NODE_CLASS_MAP.put(
            "OverridingUnspecified",
            OverridingUnspecified.class
        );
        NODE_CLASS_MAP.put(
            "Params",
            Params.class
        );
        NODE_CLASS_MAP.put(
            "ParenAbstractStateDecl",
            ParenAbstractStateDecl.class
        );
        NODE_CLASS_MAP.put(
            "PpDirective",
            PpDirective.class
        );
        NODE_CLASS_MAP.put(
            "PpElseDirective",
            PpElseDirective.class
        );
        NODE_CLASS_MAP.put(
            "PpElsifDirective",
            PpElsifDirective.class
        );
        NODE_CLASS_MAP.put(
            "PpEndIfDirective",
            PpEndIfDirective.class
        );
        NODE_CLASS_MAP.put(
            "PpIfDirective",
            PpIfDirective.class
        );
        NODE_CLASS_MAP.put(
            "PpThenKw",
            PpThenKw.class
        );
        NODE_CLASS_MAP.put(
            "PragmaNode",
            PragmaNode.class
        );
        NODE_CLASS_MAP.put(
            "PrivateNode",
            PrivateNode.class
        );
        NODE_CLASS_MAP.put(
            "PrivateAbsent",
            PrivateAbsent.class
        );
        NODE_CLASS_MAP.put(
            "PrivatePresent",
            PrivatePresent.class
        );
        NODE_CLASS_MAP.put(
            "ProtectedDef",
            ProtectedDef.class
        );
        NODE_CLASS_MAP.put(
            "ProtectedNode",
            ProtectedNode.class
        );
        NODE_CLASS_MAP.put(
            "ProtectedAbsent",
            ProtectedAbsent.class
        );
        NODE_CLASS_MAP.put(
            "ProtectedPresent",
            ProtectedPresent.class
        );
        NODE_CLASS_MAP.put(
            "Quantifier",
            Quantifier.class
        );
        NODE_CLASS_MAP.put(
            "QuantifierAll",
            QuantifierAll.class
        );
        NODE_CLASS_MAP.put(
            "QuantifierSome",
            QuantifierSome.class
        );
        NODE_CLASS_MAP.put(
            "RangeSpec",
            RangeSpec.class
        );
        NODE_CLASS_MAP.put(
            "RenamingClause",
            RenamingClause.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticRenamingClause",
            SyntheticRenamingClause.class
        );
        NODE_CLASS_MAP.put(
            "ReverseNode",
            ReverseNode.class
        );
        NODE_CLASS_MAP.put(
            "ReverseAbsent",
            ReverseAbsent.class
        );
        NODE_CLASS_MAP.put(
            "ReversePresent",
            ReversePresent.class
        );
        NODE_CLASS_MAP.put(
            "SelectWhenPart",
            SelectWhenPart.class
        );
        NODE_CLASS_MAP.put(
            "Stmt",
            Stmt.class
        );
        NODE_CLASS_MAP.put(
            "CompositeStmt",
            CompositeStmt.class
        );
        NODE_CLASS_MAP.put(
            "AcceptStmt",
            AcceptStmt.class
        );
        NODE_CLASS_MAP.put(
            "AcceptStmtWithStmts",
            AcceptStmtWithStmts.class
        );
        NODE_CLASS_MAP.put(
            "BaseLoopStmt",
            BaseLoopStmt.class
        );
        NODE_CLASS_MAP.put(
            "ForLoopStmt",
            ForLoopStmt.class
        );
        NODE_CLASS_MAP.put(
            "LoopStmt",
            LoopStmt.class
        );
        NODE_CLASS_MAP.put(
            "WhileLoopStmt",
            WhileLoopStmt.class
        );
        NODE_CLASS_MAP.put(
            "BlockStmt",
            BlockStmt.class
        );
        NODE_CLASS_MAP.put(
            "BeginBlock",
            BeginBlock.class
        );
        NODE_CLASS_MAP.put(
            "DeclBlock",
            DeclBlock.class
        );
        NODE_CLASS_MAP.put(
            "CaseStmt",
            CaseStmt.class
        );
        NODE_CLASS_MAP.put(
            "ExtendedReturnStmt",
            ExtendedReturnStmt.class
        );
        NODE_CLASS_MAP.put(
            "IfStmt",
            IfStmt.class
        );
        NODE_CLASS_MAP.put(
            "NamedStmt",
            NamedStmt.class
        );
        NODE_CLASS_MAP.put(
            "SelectStmt",
            SelectStmt.class
        );
        NODE_CLASS_MAP.put(
            "ErrorStmt",
            ErrorStmt.class
        );
        NODE_CLASS_MAP.put(
            "SimpleStmt",
            SimpleStmt.class
        );
        NODE_CLASS_MAP.put(
            "AbortStmt",
            AbortStmt.class
        );
        NODE_CLASS_MAP.put(
            "AssignStmt",
            AssignStmt.class
        );
        NODE_CLASS_MAP.put(
            "CallStmt",
            CallStmt.class
        );
        NODE_CLASS_MAP.put(
            "DelayStmt",
            DelayStmt.class
        );
        NODE_CLASS_MAP.put(
            "ExitStmt",
            ExitStmt.class
        );
        NODE_CLASS_MAP.put(
            "GotoStmt",
            GotoStmt.class
        );
        NODE_CLASS_MAP.put(
            "Label",
            Label.class
        );
        NODE_CLASS_MAP.put(
            "NullStmt",
            NullStmt.class
        );
        NODE_CLASS_MAP.put(
            "RaiseStmt",
            RaiseStmt.class
        );
        NODE_CLASS_MAP.put(
            "RequeueStmt",
            RequeueStmt.class
        );
        NODE_CLASS_MAP.put(
            "ReturnStmt",
            ReturnStmt.class
        );
        NODE_CLASS_MAP.put(
            "TerminateAlternative",
            TerminateAlternative.class
        );
        NODE_CLASS_MAP.put(
            "SubpKind",
            SubpKind.class
        );
        NODE_CLASS_MAP.put(
            "SubpKindFunction",
            SubpKindFunction.class
        );
        NODE_CLASS_MAP.put(
            "SubpKindProcedure",
            SubpKindProcedure.class
        );
        NODE_CLASS_MAP.put(
            "Subunit",
            Subunit.class
        );
        NODE_CLASS_MAP.put(
            "SynchronizedNode",
            SynchronizedNode.class
        );
        NODE_CLASS_MAP.put(
            "SynchronizedAbsent",
            SynchronizedAbsent.class
        );
        NODE_CLASS_MAP.put(
            "SynchronizedPresent",
            SynchronizedPresent.class
        );
        NODE_CLASS_MAP.put(
            "TaggedNode",
            TaggedNode.class
        );
        NODE_CLASS_MAP.put(
            "TaggedAbsent",
            TaggedAbsent.class
        );
        NODE_CLASS_MAP.put(
            "TaggedPresent",
            TaggedPresent.class
        );
        NODE_CLASS_MAP.put(
            "TaskDef",
            TaskDef.class
        );
        NODE_CLASS_MAP.put(
            "TypeAttributesRepository",
            TypeAttributesRepository.class
        );
        NODE_CLASS_MAP.put(
            "TypeDef",
            TypeDef.class
        );
        NODE_CLASS_MAP.put(
            "AccessDef",
            AccessDef.class
        );
        NODE_CLASS_MAP.put(
            "AccessToSubpDef",
            AccessToSubpDef.class
        );
        NODE_CLASS_MAP.put(
            "BaseTypeAccessDef",
            BaseTypeAccessDef.class
        );
        NODE_CLASS_MAP.put(
            "AnonymousTypeAccessDef",
            AnonymousTypeAccessDef.class
        );
        NODE_CLASS_MAP.put(
            "TypeAccessDef",
            TypeAccessDef.class
        );
        NODE_CLASS_MAP.put(
            "ArrayTypeDef",
            ArrayTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "DerivedTypeDef",
            DerivedTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "EnumTypeDef",
            EnumTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "FormalDiscreteTypeDef",
            FormalDiscreteTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "InterfaceTypeDef",
            InterfaceTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "ModIntTypeDef",
            ModIntTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "PrivateTypeDef",
            PrivateTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "RealTypeDef",
            RealTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "DecimalFixedPointDef",
            DecimalFixedPointDef.class
        );
        NODE_CLASS_MAP.put(
            "FloatingPointDef",
            FloatingPointDef.class
        );
        NODE_CLASS_MAP.put(
            "OrdinaryFixedPointDef",
            OrdinaryFixedPointDef.class
        );
        NODE_CLASS_MAP.put(
            "RecordTypeDef",
            RecordTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "SignedIntTypeDef",
            SignedIntTypeDef.class
        );
        NODE_CLASS_MAP.put(
            "TypeExpr",
            TypeExpr.class
        );
        NODE_CLASS_MAP.put(
            "AnonymousType",
            AnonymousType.class
        );
        NODE_CLASS_MAP.put(
            "EnumLitSynthTypeExpr",
            EnumLitSynthTypeExpr.class
        );
        NODE_CLASS_MAP.put(
            "SubtypeIndication",
            SubtypeIndication.class
        );
        NODE_CLASS_MAP.put(
            "ConstrainedSubtypeIndication",
            ConstrainedSubtypeIndication.class
        );
        NODE_CLASS_MAP.put(
            "DiscreteSubtypeIndication",
            DiscreteSubtypeIndication.class
        );
        NODE_CLASS_MAP.put(
            "SyntheticTypeExpr",
            SyntheticTypeExpr.class
        );
        NODE_CLASS_MAP.put(
            "UnconstrainedArrayIndex",
            UnconstrainedArrayIndex.class
        );
        NODE_CLASS_MAP.put(
            "UntilNode",
            UntilNode.class
        );
        NODE_CLASS_MAP.put(
            "UntilAbsent",
            UntilAbsent.class
        );
        NODE_CLASS_MAP.put(
            "UntilPresent",
            UntilPresent.class
        );
        NODE_CLASS_MAP.put(
            "UseClause",
            UseClause.class
        );
        NODE_CLASS_MAP.put(
            "UsePackageClause",
            UsePackageClause.class
        );
        NODE_CLASS_MAP.put(
            "UseTypeClause",
            UseTypeClause.class
        );
        NODE_CLASS_MAP.put(
            "ValueSequence",
            ValueSequence.class
        );
        NODE_CLASS_MAP.put(
            "Variant",
            Variant.class
        );
        NODE_CLASS_MAP.put(
            "VariantPart",
            VariantPart.class
        );
        NODE_CLASS_MAP.put(
            "WithClause",
            WithClause.class
        );
        NODE_CLASS_MAP.put(
            "WithPrivate",
            WithPrivate.class
        );
        NODE_CLASS_MAP.put(
            "WithPrivateAbsent",
            WithPrivateAbsent.class
        );
        NODE_CLASS_MAP.put(
            "WithPrivatePresent",
            WithPrivatePresent.class
        );
    }

    // ==========
    // Util functions
    // ==========

    /**
     * Get the string representing the memory.
     *
     * @param pointer The pointer to start displaying the memory from.
     * @param count The number of bytes to display from the pointer.
     * @return The string representing the memory as hex bytes.
     */
    private static String dumpMemory(
        final Pointer pointer,
        final long count
    ) {
        final StringBuilder res = new StringBuilder();
        for(int i = 0 ; i < count ; i++) {
            final byte toDump = pointer.readByte(i);
            res.append(String.format("%02x ", toDump));
        }
        return res.toString();
    }

    /**
     * Convert a Java string to a C string by allocating memory.
     *
     * @param jString The Java string to convert.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    @CompilerDirectives.TruffleBoundary
    private static CCharPointer toCString(
        final String jString
    ) {
        final UnsignedWord size = WordFactory.unsigned(jString.length() + 1);
        final CCharPointer res = UnmanagedMemory.calloc(size);
        if(jString.length() > 0) {
            CTypeConversion.toCString(
                jString,
                StandardCharsets.UTF_8,
                res,
                size
            );
        }

        return res;
    }

    /**
     * Convert a Java byte array in a native byte array.
     *
     * @param bytes The Java bytes.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    private static CCharPointer toCBytes(
        final byte[] bytes
    ) {
        final UnsignedWord size = WordFactory.unsigned(bytes.length);
        final CCharPointer res = UnmanagedMemory.malloc(size);
        for(int i = 0 ; i < bytes.length ; i++) {
            res.write(i, bytes[i]);
        }
        return res;
    }

    /**
     * Convert a native-image C string to a Java string.
     *
     * @param pointer The char pointer to convert to a Java string.
     * @return The Java string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String toJString(
        final CCharPointer pointer
    ) {
        return CTypeConversion.toJavaString(pointer);
    }

    /**
     * This function decode a utf 32 int array in a
     * string without calling Java charset.
     *
     * @param chars The int array to decode.
     * @return The resulting string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String decodeUTF32(
        final byte[] toDecode
    ) {
        return toJavaStringNode.execute(
            fromByteArrayNode.execute(toDecode, TruffleString.Encoding.UTF_32)
        );
    }

    /**
     * This function encode a given string to a int array
     * according to the utf 32 standard.
     *
     * @param toEncode The string to encode.
     * @return The encoded string in an int array.
     */
    @CompilerDirectives.TruffleBoundary
    private static byte[] encodeUTF32(
        final String toEncode
    ) {
        return toByteArrayNode.execute(
            fromJavaStringNode.execute(
                toEncode,
                TruffleString.Encoding.UTF_32
            ),
            TruffleString.Encoding.UTF_32
        );
    }

    /**
     * Get the string representation of the given string.
     * This function escaped needed chars and format the string.
     *
     * @param source The source string to get the representation for.
     * @return The representation of the string.
     */
    private static String stringRepresentation(
        final String source
    ) {
        return source
            .replace("\"", "\\\"")
            .replace("\n", "\\x0a");
    }

    /**
     * Convert a C Langkit exception to the LangkitException class.

     * @param
     */
    private static LangkitException wrapException(
        final LangkitExceptionNative exc
    ) {
        return new LangkitException(
            exc.get_kind(),
            toJString(exc.get_information())
        );
    }

    /**
      * Return the exception raised by the last C API call, or null if the last
      * call was successful.
      */
    private static LangkitException getLastException() {
        LangkitException result = null;

        if(ImageInfo.inImageCode()) {
            final LangkitExceptionNative exceptionNative =
                NI_LIB.ada_get_last_exception();
            if(exceptionNative.isNonNull()) {
                result = wrapException(exceptionNative);
            }
        } else {
            result = JNI_LIB.ada_get_last_exception();
        }
        return result;
    }

    /**
     * Check the last exception raised by langkit and throw it.
     *
     * @throws The last langkit exception if there is one.
     */
    private static void checkException() throws LangkitException {
        LangkitException exc = getLastException();
        if(exc != null)
            throw exc;
    }

    // ==========
    // Util interfaces
    // ==========

    /**
     * Interface to visit the AST.
     */
    public static interface BasicVisitor<T> {
        T visit(AdaNode node);
        T visit(AbortAbsent node);
        T visit(AbortPresent node);
        T visit(AbstractAbsent node);
        T visit(AbstractPresent node);
        T visit(AdaNodeList node);
        T visit(AbstractStateDeclList node);
        T visit(AlternativesList node);
        T visit(ConstraintList node);
        T visit(DeclList node);
        T visit(StmtList node);
        T visit(AspectAssocList node);
        T visit(BaseAssocList node);
        T visit(AssocList node);
        T visit(BasicDeclList node);
        T visit(CaseExprAlternativeList node);
        T visit(CaseStmtAlternativeList node);
        T visit(CompilationUnitList node);
        T visit(ConcatOperandList node);
        T visit(ContractCaseAssocList node);
        T visit(DefiningNameList node);
        T visit(DiscriminantSpecList node);
        T visit(ElsifExprPartList node);
        T visit(ElsifStmtPartList node);
        T visit(EnumLiteralDeclList node);
        T visit(ExprAlternativesList node);
        T visit(DiscriminantChoiceList node);
        T visit(NameList node);
        T visit(ParentList node);
        T visit(ParamSpecList node);
        T visit(PragmaNodeList node);
        T visit(SelectWhenPartList node);
        T visit(UnconstrainedArrayIndexList node);
        T visit(VariantList node);
        T visit(AliasedAbsent node);
        T visit(AliasedPresent node);
        T visit(AllAbsent node);
        T visit(AllPresent node);
        T visit(ConstrainedArrayIndices node);
        T visit(UnconstrainedArrayIndices node);
        T visit(AspectAssoc node);
        T visit(AtClause node);
        T visit(AttributeDefClause node);
        T visit(EnumRepClause node);
        T visit(RecordRepClause node);
        T visit(AspectSpec node);
        T visit(ContractCaseAssoc node);
        T visit(PragmaArgumentAssoc node);
        T visit(EntrySpec node);
        T visit(EnumSubpSpec node);
        T visit(SubpSpec node);
        T visit(SyntheticBinarySpec node);
        T visit(SyntheticUnarySpec node);
        T visit(ComponentList node);
        T visit(KnownDiscriminantPart node);
        T visit(UnknownDiscriminantPart node);
        T visit(EntryCompletionFormalParams node);
        T visit(GenericFormalPart node);
        T visit(NullRecordDef node);
        T visit(RecordDef node);
        T visit(AggregateAssoc node);
        T visit(MultiDimArrayAssoc node);
        T visit(CompositeConstraintAssoc node);
        T visit(IteratedAssoc node);
        T visit(ParamAssoc node);
        T visit(AbstractStateDecl node);
        T visit(AnonymousExprDecl node);
        T visit(ComponentDecl node);
        T visit(DiscriminantSpec node);
        T visit(GenericFormalObjDecl node);
        T visit(GenericFormalPackage node);
        T visit(GenericFormalSubpDecl node);
        T visit(GenericFormalTypeDecl node);
        T visit(ParamSpec node);
        T visit(SyntheticFormalParamDecl node);
        T visit(GenericPackageInternal node);
        T visit(PackageDecl node);
        T visit(DiscreteBaseSubtypeDecl node);
        T visit(SubtypeDecl node);
        T visit(ClasswideTypeDecl node);
        T visit(IncompleteTypeDecl node);
        T visit(IncompleteFormalTypeDecl node);
        T visit(IncompleteTaggedTypeDecl node);
        T visit(ProtectedTypeDecl node);
        T visit(TaskTypeDecl node);
        T visit(SingleTaskTypeDecl node);
        T visit(AnonymousTypeDecl node);
        T visit(SynthAnonymousTypeDecl node);
        T visit(ConcreteTypeDecl node);
        T visit(FormalTypeDecl node);
        T visit(AbstractSubpDecl node);
        T visit(AbstractFormalSubpDecl node);
        T visit(ConcreteFormalSubpDecl node);
        T visit(SubpDecl node);
        T visit(EntryDecl node);
        T visit(EnumLiteralDecl node);
        T visit(SyntheticCharEnumLit node);
        T visit(GenericSubpInternal node);
        T visit(SyntheticSubpDecl node);
        T visit(AcceptStmtBody node);
        T visit(ExprFunction node);
        T visit(NullSubpDecl node);
        T visit(SubpBody node);
        T visit(SubpRenamingDecl node);
        T visit(PackageBodyStub node);
        T visit(ProtectedBodyStub node);
        T visit(SubpBodyStub node);
        T visit(TaskBodyStub node);
        T visit(EntryBody node);
        T visit(PackageBody node);
        T visit(ProtectedBody node);
        T visit(TaskBody node);
        T visit(EntryIndexSpec node);
        T visit(ErrorDecl node);
        T visit(ExceptionDecl node);
        T visit(ExceptionHandler node);
        T visit(ForLoopVarDecl node);
        T visit(GenericPackageDecl node);
        T visit(GenericSubpDecl node);
        T visit(GenericPackageInstantiation node);
        T visit(GenericSubpInstantiation node);
        T visit(GenericPackageRenamingDecl node);
        T visit(GenericSubpRenamingDecl node);
        T visit(LabelDecl node);
        T visit(NamedStmtDecl node);
        T visit(NumberDecl node);
        T visit(ObjectDecl node);
        T visit(ExtendedReturnStmtObjectDecl node);
        T visit(NoTypeObjectRenamingDecl node);
        T visit(PackageRenamingDecl node);
        T visit(SingleProtectedDecl node);
        T visit(SingleTaskDecl node);
        T visit(CaseStmtAlternative node);
        T visit(CompilationUnit node);
        T visit(ComponentClause node);
        T visit(ComponentDef node);
        T visit(ConstantAbsent node);
        T visit(ConstantPresent node);
        T visit(CompositeConstraint node);
        T visit(DeltaConstraint node);
        T visit(DigitsConstraint node);
        T visit(RangeConstraint node);
        T visit(DeclarativePart node);
        T visit(PrivatePart node);
        T visit(PublicPart node);
        T visit(ElsifExprPart node);
        T visit(ElsifStmtPart node);
        T visit(AbstractStateDeclExpr node);
        T visit(Allocator node);
        T visit(Aggregate node);
        T visit(BracketAggregate node);
        T visit(DeltaAggregate node);
        T visit(BracketDeltaAggregate node);
        T visit(NullRecordAggregate node);
        T visit(BinOp node);
        T visit(RelationOp node);
        T visit(BoxExpr node);
        T visit(CaseExprAlternative node);
        T visit(ConcatOp node);
        T visit(ConcatOperand node);
        T visit(CaseExpr node);
        T visit(IfExpr node);
        T visit(ContractCases node);
        T visit(DeclExpr node);
        T visit(MembershipExpr node);
        T visit(AttributeRef node);
        T visit(CallExpr node);
        T visit(DefiningName node);
        T visit(SyntheticDefiningName node);
        T visit(DiscreteSubtypeName node);
        T visit(DottedName node);
        T visit(EndName node);
        T visit(ExplicitDeref node);
        T visit(QualExpr node);
        T visit(ReduceAttributeRef node);
        T visit(CharLiteral node);
        T visit(Identifier node);
        T visit(OpAbs node);
        T visit(OpAnd node);
        T visit(OpAndThen node);
        T visit(OpConcat node);
        T visit(OpDiv node);
        T visit(OpDoubleDot node);
        T visit(OpEq node);
        T visit(OpGt node);
        T visit(OpGte node);
        T visit(OpIn node);
        T visit(OpLt node);
        T visit(OpLte node);
        T visit(OpMinus node);
        T visit(OpMod node);
        T visit(OpMult node);
        T visit(OpNeq node);
        T visit(OpNot node);
        T visit(OpNotIn node);
        T visit(OpOr node);
        T visit(OpOrElse node);
        T visit(OpPlus node);
        T visit(OpPow node);
        T visit(OpRem node);
        T visit(OpXor node);
        T visit(StringLiteral node);
        T visit(NullLiteral node);
        T visit(IntLiteral node);
        T visit(RealLiteral node);
        T visit(SyntheticIdentifier node);
        T visit(TargetName node);
        T visit(UpdateAttributeRef node);
        T visit(ParenExpr node);
        T visit(QuantifiedExpr node);
        T visit(RaiseExpr node);
        T visit(UnOp node);
        T visit(HandledStmts node);
        T visit(InterfaceKindLimited node);
        T visit(InterfaceKindProtected node);
        T visit(InterfaceKindSynchronized node);
        T visit(InterfaceKindTask node);
        T visit(IterTypeIn node);
        T visit(IterTypeOf node);
        T visit(LibraryItem node);
        T visit(LimitedAbsent node);
        T visit(LimitedPresent node);
        T visit(ForLoopSpec node);
        T visit(WhileLoopSpec node);
        T visit(ModeDefault node);
        T visit(ModeIn node);
        T visit(ModeInOut node);
        T visit(ModeOut node);
        T visit(MultiAbstractStateDecl node);
        T visit(NotNullAbsent node);
        T visit(NotNullPresent node);
        T visit(NullComponentDecl node);
        T visit(OthersDesignator node);
        T visit(OverridingNotOverriding node);
        T visit(OverridingOverriding node);
        T visit(OverridingUnspecified node);
        T visit(Params node);
        T visit(ParenAbstractStateDecl node);
        T visit(PpElseDirective node);
        T visit(PpElsifDirective node);
        T visit(PpEndIfDirective node);
        T visit(PpIfDirective node);
        T visit(PpThenKw node);
        T visit(PragmaNode node);
        T visit(PrivateAbsent node);
        T visit(PrivatePresent node);
        T visit(ProtectedDef node);
        T visit(ProtectedAbsent node);
        T visit(ProtectedPresent node);
        T visit(QuantifierAll node);
        T visit(QuantifierSome node);
        T visit(RangeSpec node);
        T visit(RenamingClause node);
        T visit(SyntheticRenamingClause node);
        T visit(ReverseAbsent node);
        T visit(ReversePresent node);
        T visit(SelectWhenPart node);
        T visit(AcceptStmt node);
        T visit(AcceptStmtWithStmts node);
        T visit(ForLoopStmt node);
        T visit(LoopStmt node);
        T visit(WhileLoopStmt node);
        T visit(BeginBlock node);
        T visit(DeclBlock node);
        T visit(CaseStmt node);
        T visit(ExtendedReturnStmt node);
        T visit(IfStmt node);
        T visit(NamedStmt node);
        T visit(SelectStmt node);
        T visit(ErrorStmt node);
        T visit(AbortStmt node);
        T visit(AssignStmt node);
        T visit(CallStmt node);
        T visit(DelayStmt node);
        T visit(ExitStmt node);
        T visit(GotoStmt node);
        T visit(Label node);
        T visit(NullStmt node);
        T visit(RaiseStmt node);
        T visit(RequeueStmt node);
        T visit(ReturnStmt node);
        T visit(TerminateAlternative node);
        T visit(SubpKindFunction node);
        T visit(SubpKindProcedure node);
        T visit(Subunit node);
        T visit(SynchronizedAbsent node);
        T visit(SynchronizedPresent node);
        T visit(TaggedAbsent node);
        T visit(TaggedPresent node);
        T visit(TaskDef node);
        T visit(TypeAttributesRepository node);
        T visit(AccessToSubpDef node);
        T visit(AnonymousTypeAccessDef node);
        T visit(TypeAccessDef node);
        T visit(ArrayTypeDef node);
        T visit(DerivedTypeDef node);
        T visit(EnumTypeDef node);
        T visit(FormalDiscreteTypeDef node);
        T visit(InterfaceTypeDef node);
        T visit(ModIntTypeDef node);
        T visit(PrivateTypeDef node);
        T visit(DecimalFixedPointDef node);
        T visit(FloatingPointDef node);
        T visit(OrdinaryFixedPointDef node);
        T visit(RecordTypeDef node);
        T visit(SignedIntTypeDef node);
        T visit(AnonymousType node);
        T visit(EnumLitSynthTypeExpr node);
        T visit(SubtypeIndication node);
        T visit(ConstrainedSubtypeIndication node);
        T visit(DiscreteSubtypeIndication node);
        T visit(SyntheticTypeExpr node);
        T visit(UnconstrainedArrayIndex node);
        T visit(UntilAbsent node);
        T visit(UntilPresent node);
        T visit(UsePackageClause node);
        T visit(UseTypeClause node);
        T visit(ValueSequence node);
        T visit(Variant node);
        T visit(VariantPart node);
        T visit(WithClause node);
        T visit(WithPrivateAbsent node);
        T visit(WithPrivatePresent node);
    }

    /**
     * Interface to visit the AST with a parameter.
     */
    public static interface ParamVisitor<T, P> {
        T visit(AdaNode node, P param);
        T visit(AbortAbsent node, P param);
        T visit(AbortPresent node, P param);
        T visit(AbstractAbsent node, P param);
        T visit(AbstractPresent node, P param);
        T visit(AdaNodeList node, P param);
        T visit(AbstractStateDeclList node, P param);
        T visit(AlternativesList node, P param);
        T visit(ConstraintList node, P param);
        T visit(DeclList node, P param);
        T visit(StmtList node, P param);
        T visit(AspectAssocList node, P param);
        T visit(BaseAssocList node, P param);
        T visit(AssocList node, P param);
        T visit(BasicDeclList node, P param);
        T visit(CaseExprAlternativeList node, P param);
        T visit(CaseStmtAlternativeList node, P param);
        T visit(CompilationUnitList node, P param);
        T visit(ConcatOperandList node, P param);
        T visit(ContractCaseAssocList node, P param);
        T visit(DefiningNameList node, P param);
        T visit(DiscriminantSpecList node, P param);
        T visit(ElsifExprPartList node, P param);
        T visit(ElsifStmtPartList node, P param);
        T visit(EnumLiteralDeclList node, P param);
        T visit(ExprAlternativesList node, P param);
        T visit(DiscriminantChoiceList node, P param);
        T visit(NameList node, P param);
        T visit(ParentList node, P param);
        T visit(ParamSpecList node, P param);
        T visit(PragmaNodeList node, P param);
        T visit(SelectWhenPartList node, P param);
        T visit(UnconstrainedArrayIndexList node, P param);
        T visit(VariantList node, P param);
        T visit(AliasedAbsent node, P param);
        T visit(AliasedPresent node, P param);
        T visit(AllAbsent node, P param);
        T visit(AllPresent node, P param);
        T visit(ConstrainedArrayIndices node, P param);
        T visit(UnconstrainedArrayIndices node, P param);
        T visit(AspectAssoc node, P param);
        T visit(AtClause node, P param);
        T visit(AttributeDefClause node, P param);
        T visit(EnumRepClause node, P param);
        T visit(RecordRepClause node, P param);
        T visit(AspectSpec node, P param);
        T visit(ContractCaseAssoc node, P param);
        T visit(PragmaArgumentAssoc node, P param);
        T visit(EntrySpec node, P param);
        T visit(EnumSubpSpec node, P param);
        T visit(SubpSpec node, P param);
        T visit(SyntheticBinarySpec node, P param);
        T visit(SyntheticUnarySpec node, P param);
        T visit(ComponentList node, P param);
        T visit(KnownDiscriminantPart node, P param);
        T visit(UnknownDiscriminantPart node, P param);
        T visit(EntryCompletionFormalParams node, P param);
        T visit(GenericFormalPart node, P param);
        T visit(NullRecordDef node, P param);
        T visit(RecordDef node, P param);
        T visit(AggregateAssoc node, P param);
        T visit(MultiDimArrayAssoc node, P param);
        T visit(CompositeConstraintAssoc node, P param);
        T visit(IteratedAssoc node, P param);
        T visit(ParamAssoc node, P param);
        T visit(AbstractStateDecl node, P param);
        T visit(AnonymousExprDecl node, P param);
        T visit(ComponentDecl node, P param);
        T visit(DiscriminantSpec node, P param);
        T visit(GenericFormalObjDecl node, P param);
        T visit(GenericFormalPackage node, P param);
        T visit(GenericFormalSubpDecl node, P param);
        T visit(GenericFormalTypeDecl node, P param);
        T visit(ParamSpec node, P param);
        T visit(SyntheticFormalParamDecl node, P param);
        T visit(GenericPackageInternal node, P param);
        T visit(PackageDecl node, P param);
        T visit(DiscreteBaseSubtypeDecl node, P param);
        T visit(SubtypeDecl node, P param);
        T visit(ClasswideTypeDecl node, P param);
        T visit(IncompleteTypeDecl node, P param);
        T visit(IncompleteFormalTypeDecl node, P param);
        T visit(IncompleteTaggedTypeDecl node, P param);
        T visit(ProtectedTypeDecl node, P param);
        T visit(TaskTypeDecl node, P param);
        T visit(SingleTaskTypeDecl node, P param);
        T visit(AnonymousTypeDecl node, P param);
        T visit(SynthAnonymousTypeDecl node, P param);
        T visit(ConcreteTypeDecl node, P param);
        T visit(FormalTypeDecl node, P param);
        T visit(AbstractSubpDecl node, P param);
        T visit(AbstractFormalSubpDecl node, P param);
        T visit(ConcreteFormalSubpDecl node, P param);
        T visit(SubpDecl node, P param);
        T visit(EntryDecl node, P param);
        T visit(EnumLiteralDecl node, P param);
        T visit(SyntheticCharEnumLit node, P param);
        T visit(GenericSubpInternal node, P param);
        T visit(SyntheticSubpDecl node, P param);
        T visit(AcceptStmtBody node, P param);
        T visit(ExprFunction node, P param);
        T visit(NullSubpDecl node, P param);
        T visit(SubpBody node, P param);
        T visit(SubpRenamingDecl node, P param);
        T visit(PackageBodyStub node, P param);
        T visit(ProtectedBodyStub node, P param);
        T visit(SubpBodyStub node, P param);
        T visit(TaskBodyStub node, P param);
        T visit(EntryBody node, P param);
        T visit(PackageBody node, P param);
        T visit(ProtectedBody node, P param);
        T visit(TaskBody node, P param);
        T visit(EntryIndexSpec node, P param);
        T visit(ErrorDecl node, P param);
        T visit(ExceptionDecl node, P param);
        T visit(ExceptionHandler node, P param);
        T visit(ForLoopVarDecl node, P param);
        T visit(GenericPackageDecl node, P param);
        T visit(GenericSubpDecl node, P param);
        T visit(GenericPackageInstantiation node, P param);
        T visit(GenericSubpInstantiation node, P param);
        T visit(GenericPackageRenamingDecl node, P param);
        T visit(GenericSubpRenamingDecl node, P param);
        T visit(LabelDecl node, P param);
        T visit(NamedStmtDecl node, P param);
        T visit(NumberDecl node, P param);
        T visit(ObjectDecl node, P param);
        T visit(ExtendedReturnStmtObjectDecl node, P param);
        T visit(NoTypeObjectRenamingDecl node, P param);
        T visit(PackageRenamingDecl node, P param);
        T visit(SingleProtectedDecl node, P param);
        T visit(SingleTaskDecl node, P param);
        T visit(CaseStmtAlternative node, P param);
        T visit(CompilationUnit node, P param);
        T visit(ComponentClause node, P param);
        T visit(ComponentDef node, P param);
        T visit(ConstantAbsent node, P param);
        T visit(ConstantPresent node, P param);
        T visit(CompositeConstraint node, P param);
        T visit(DeltaConstraint node, P param);
        T visit(DigitsConstraint node, P param);
        T visit(RangeConstraint node, P param);
        T visit(DeclarativePart node, P param);
        T visit(PrivatePart node, P param);
        T visit(PublicPart node, P param);
        T visit(ElsifExprPart node, P param);
        T visit(ElsifStmtPart node, P param);
        T visit(AbstractStateDeclExpr node, P param);
        T visit(Allocator node, P param);
        T visit(Aggregate node, P param);
        T visit(BracketAggregate node, P param);
        T visit(DeltaAggregate node, P param);
        T visit(BracketDeltaAggregate node, P param);
        T visit(NullRecordAggregate node, P param);
        T visit(BinOp node, P param);
        T visit(RelationOp node, P param);
        T visit(BoxExpr node, P param);
        T visit(CaseExprAlternative node, P param);
        T visit(ConcatOp node, P param);
        T visit(ConcatOperand node, P param);
        T visit(CaseExpr node, P param);
        T visit(IfExpr node, P param);
        T visit(ContractCases node, P param);
        T visit(DeclExpr node, P param);
        T visit(MembershipExpr node, P param);
        T visit(AttributeRef node, P param);
        T visit(CallExpr node, P param);
        T visit(DefiningName node, P param);
        T visit(SyntheticDefiningName node, P param);
        T visit(DiscreteSubtypeName node, P param);
        T visit(DottedName node, P param);
        T visit(EndName node, P param);
        T visit(ExplicitDeref node, P param);
        T visit(QualExpr node, P param);
        T visit(ReduceAttributeRef node, P param);
        T visit(CharLiteral node, P param);
        T visit(Identifier node, P param);
        T visit(OpAbs node, P param);
        T visit(OpAnd node, P param);
        T visit(OpAndThen node, P param);
        T visit(OpConcat node, P param);
        T visit(OpDiv node, P param);
        T visit(OpDoubleDot node, P param);
        T visit(OpEq node, P param);
        T visit(OpGt node, P param);
        T visit(OpGte node, P param);
        T visit(OpIn node, P param);
        T visit(OpLt node, P param);
        T visit(OpLte node, P param);
        T visit(OpMinus node, P param);
        T visit(OpMod node, P param);
        T visit(OpMult node, P param);
        T visit(OpNeq node, P param);
        T visit(OpNot node, P param);
        T visit(OpNotIn node, P param);
        T visit(OpOr node, P param);
        T visit(OpOrElse node, P param);
        T visit(OpPlus node, P param);
        T visit(OpPow node, P param);
        T visit(OpRem node, P param);
        T visit(OpXor node, P param);
        T visit(StringLiteral node, P param);
        T visit(NullLiteral node, P param);
        T visit(IntLiteral node, P param);
        T visit(RealLiteral node, P param);
        T visit(SyntheticIdentifier node, P param);
        T visit(TargetName node, P param);
        T visit(UpdateAttributeRef node, P param);
        T visit(ParenExpr node, P param);
        T visit(QuantifiedExpr node, P param);
        T visit(RaiseExpr node, P param);
        T visit(UnOp node, P param);
        T visit(HandledStmts node, P param);
        T visit(InterfaceKindLimited node, P param);
        T visit(InterfaceKindProtected node, P param);
        T visit(InterfaceKindSynchronized node, P param);
        T visit(InterfaceKindTask node, P param);
        T visit(IterTypeIn node, P param);
        T visit(IterTypeOf node, P param);
        T visit(LibraryItem node, P param);
        T visit(LimitedAbsent node, P param);
        T visit(LimitedPresent node, P param);
        T visit(ForLoopSpec node, P param);
        T visit(WhileLoopSpec node, P param);
        T visit(ModeDefault node, P param);
        T visit(ModeIn node, P param);
        T visit(ModeInOut node, P param);
        T visit(ModeOut node, P param);
        T visit(MultiAbstractStateDecl node, P param);
        T visit(NotNullAbsent node, P param);
        T visit(NotNullPresent node, P param);
        T visit(NullComponentDecl node, P param);
        T visit(OthersDesignator node, P param);
        T visit(OverridingNotOverriding node, P param);
        T visit(OverridingOverriding node, P param);
        T visit(OverridingUnspecified node, P param);
        T visit(Params node, P param);
        T visit(ParenAbstractStateDecl node, P param);
        T visit(PpElseDirective node, P param);
        T visit(PpElsifDirective node, P param);
        T visit(PpEndIfDirective node, P param);
        T visit(PpIfDirective node, P param);
        T visit(PpThenKw node, P param);
        T visit(PragmaNode node, P param);
        T visit(PrivateAbsent node, P param);
        T visit(PrivatePresent node, P param);
        T visit(ProtectedDef node, P param);
        T visit(ProtectedAbsent node, P param);
        T visit(ProtectedPresent node, P param);
        T visit(QuantifierAll node, P param);
        T visit(QuantifierSome node, P param);
        T visit(RangeSpec node, P param);
        T visit(RenamingClause node, P param);
        T visit(SyntheticRenamingClause node, P param);
        T visit(ReverseAbsent node, P param);
        T visit(ReversePresent node, P param);
        T visit(SelectWhenPart node, P param);
        T visit(AcceptStmt node, P param);
        T visit(AcceptStmtWithStmts node, P param);
        T visit(ForLoopStmt node, P param);
        T visit(LoopStmt node, P param);
        T visit(WhileLoopStmt node, P param);
        T visit(BeginBlock node, P param);
        T visit(DeclBlock node, P param);
        T visit(CaseStmt node, P param);
        T visit(ExtendedReturnStmt node, P param);
        T visit(IfStmt node, P param);
        T visit(NamedStmt node, P param);
        T visit(SelectStmt node, P param);
        T visit(ErrorStmt node, P param);
        T visit(AbortStmt node, P param);
        T visit(AssignStmt node, P param);
        T visit(CallStmt node, P param);
        T visit(DelayStmt node, P param);
        T visit(ExitStmt node, P param);
        T visit(GotoStmt node, P param);
        T visit(Label node, P param);
        T visit(NullStmt node, P param);
        T visit(RaiseStmt node, P param);
        T visit(RequeueStmt node, P param);
        T visit(ReturnStmt node, P param);
        T visit(TerminateAlternative node, P param);
        T visit(SubpKindFunction node, P param);
        T visit(SubpKindProcedure node, P param);
        T visit(Subunit node, P param);
        T visit(SynchronizedAbsent node, P param);
        T visit(SynchronizedPresent node, P param);
        T visit(TaggedAbsent node, P param);
        T visit(TaggedPresent node, P param);
        T visit(TaskDef node, P param);
        T visit(TypeAttributesRepository node, P param);
        T visit(AccessToSubpDef node, P param);
        T visit(AnonymousTypeAccessDef node, P param);
        T visit(TypeAccessDef node, P param);
        T visit(ArrayTypeDef node, P param);
        T visit(DerivedTypeDef node, P param);
        T visit(EnumTypeDef node, P param);
        T visit(FormalDiscreteTypeDef node, P param);
        T visit(InterfaceTypeDef node, P param);
        T visit(ModIntTypeDef node, P param);
        T visit(PrivateTypeDef node, P param);
        T visit(DecimalFixedPointDef node, P param);
        T visit(FloatingPointDef node, P param);
        T visit(OrdinaryFixedPointDef node, P param);
        T visit(RecordTypeDef node, P param);
        T visit(SignedIntTypeDef node, P param);
        T visit(AnonymousType node, P param);
        T visit(EnumLitSynthTypeExpr node, P param);
        T visit(SubtypeIndication node, P param);
        T visit(ConstrainedSubtypeIndication node, P param);
        T visit(DiscreteSubtypeIndication node, P param);
        T visit(SyntheticTypeExpr node, P param);
        T visit(UnconstrainedArrayIndex node, P param);
        T visit(UntilAbsent node, P param);
        T visit(UntilPresent node, P param);
        T visit(UsePackageClause node, P param);
        T visit(UseTypeClause node, P param);
        T visit(ValueSequence node, P param);
        T visit(Variant node, P param);
        T visit(VariantPart node, P param);
        T visit(WithClause node, P param);
        T visit(WithPrivateAbsent node, P param);
        T visit(WithPrivatePresent node, P param);
    }

    // ==========
    // Util classes
    // ==========

    /**
     * This class represents a pointer and can hold NI and JNI addresses.
     */
    public static final class PointerWrapper {

        // ----- Instance attributes -----

        /** The pointer NI value. */
        private PointerBase ni;

        /** The pointer JNI value. */
        private final long jni;

        // ----- Constructors -----

        /**
         * Create a new custom pointer from a NI pointer based value.
         *
         * @param niPointer The pointer based value.
         */
        PointerWrapper(
            final PointerBase niPointer
        ) {
            this.ni = niPointer;
            this.jni = -1;
        }

        /**
         * Create a new custom pointer from a long value.
         *
         * @param jniPointer The pointer in a long value.
         */
        PointerWrapper(
            final long jniPointer
        ) {
            this.jni = jniPointer;
        }

        /**
         * Wrap the given NI pointer in the Java class.
         *
         * @param niPointer The NI pointer to wrap.
         * @return The wrapped pointer.
         */
        static PointerWrapper wrap(
            final PointerBase niPointer
        ) {
            return new PointerWrapper(niPointer);
        }

        /**
         * Get the null pointer according to the execution mode.
         *
         * @return The null custom pointer.
         */
        public static PointerWrapper nullPointer() {

            if(ImageInfo.inImageCode()) {
                return new PointerWrapper(WordFactory.nullPointer());
            } else {
                return new PointerWrapper(0L);
            }

        }

        // ----- Instance methods -----

        /**
         * Get the pointer as an NI pointer based value.
         *
         * @return The pointer based value for NI.
         */
        public <T extends PointerBase> T ni() {
            return (T) this.ni;
        }

        /**
         * Get the pointer as a long Java value.
         *
         * @return The pointer as a long value for JNI.
         */
        public long jni() {
            return this.jni;
        }

        /**
         * Get if the pointer is null.
         *
         * @return True if the pointer is null, false else.
         */
        public boolean isNull() {

            if(ImageInfo.inImageCode()) {
                return this.ni.isNull();
            } else {
                return this.jni == 0;
            }

        }

        // ----- Override methods -----

        @Override
        public String toString() {

            if(ImageInfo.inImageCode()) {
                return "PointerWrapper{"
                    + this.ni.rawValue()
                    + "}";
            } else {
                return "PointerWrapper{"
                    + this.jni
                    + "}";
            }

        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof PointerWrapper)) return false;
            final PointerWrapper other = (PointerWrapper) o;
            if(ImageInfo.inImageCode()) {
                return this.ni.equal(other.ni);
            } else {
                return this.jni == other.jni;
            }
        }

        @Override
        public int hashCode() {

            if(ImageInfo.inImageCode()) {
                return (int) this.ni.rawValue();
            } else {
                return (int) this.jni;
            }

        }

    }

    /**
     * This class represents the description of a node field.
     */
    public static final class LibadalangField {

        // ----- Instance attributes -----

        /** The Java method for the field */
        public final Method javaMethod;

        /** The parameters of the method */
        public final List<Param> params;

        // ----- Constructors -----

        /**
         * Create a new field description.
         *
         * @param method The Java method to access the field.
         * @param params The parameters of the field call.
         */
        public LibadalangField(
            final Method javaMethod,
            final List<Param> params
        ) {
            this.javaMethod = javaMethod;
            this.params = params;
        }

    }

    /**
     * This class represents a parameter description.
     */
    public static class Param {

        // ----- Instance attributes -----

        /** The type of the argument */
        public final Class<?> type;

        /** The name of the parameter */
        public final String name;

        // ----- Constructors -----

        /**
         * Create a new langkit parameter.
         *
         * @param type The type of the parameter.
         * @param name The name of the parameter.
         */
        public Param(
            final Class<?> type,
            final String name
        ) {
            this.type = type;
            this.name = name;
        }

    }

    /**
     * This class represents an parameter in a langkit function description
     * which has a default value.
     */
    public static final class ParamWithDefaultValue extends Param {

        // ----- Instance attributes -----

        /** The default value of the parameter */
        public final Object defaultValue;

        // ----- Constructors -----

        /**
         * Create a new langkit parameter.
         *
         * @param type The type of the parameter.
         * @param name The name of the parameter.
         * @param defaultValue The default value of the parameter.
         */
        public ParamWithDefaultValue(
            final Class<?> type,
            final String name,
            final Object defaultValue
        ) {
            super(type, name);
            this.defaultValue = defaultValue;
        }

    }

    // ==========
    // Language specific extensions
    // ==========

    
      


    // ==========
    // Define classes to handle project loading
    // ==========

    /**
     * Exception to represent errors during project manipulation
     */
    public static final class ProjectManagerException extends RuntimeException {
        ProjectManagerException(
            final String message
        ) {
            super(message);
        }
    }

    /**
     * Enum to represent the source file mode for a GPR project
     */
    public static enum SourceFileMode {

        // ----- Enum values -----

        DEFAULT(0),
        ROOT_PROJECT(1),
        WHOLE_PROJECT(2),
        WHOLE_PROJECT_WITH_RUNTIME(3),
        ;

        // ----- Class attributes -----

        /** Singleton that represents the none source file mode */
        public static final SourceFileMode NONE = DEFAULT;

        /** The map from int to enum values */
        private static final Map<Integer, SourceFileMode> map = new HashMap<>();

        // ----- Instance attributes -----

        /** The value of the instance */
        private final int value;

        // ----- Constructors -----

        /**
         * The private constructor
         */
        private SourceFileMode(
            final int value
        ) {
            this.value = value;
        }

        static {
            for(SourceFileMode elem : SourceFileMode.values()) {
                map.put(elem.value, elem);
            }
        }

        // ----- Enum methods -----

        /**
         * Get a source file mode from a native integer value.
         *
         * @param cValue The native value of the enum.
         * @return The Java source file mode.
         * @throws EnumException If the given native value doesn't correspond
         * to an actual enum value.
         */
        public static SourceFileMode fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get SourceFileMode from " + cValue
                );
            return (SourceFileMode) map.get(cValue);
        }

        /**
         * Get the native integer value of the enum instance.
         *
         * @return The native C value.
         */
        public int toC() {
            return this.value;
        }

    }

    /**
     * This class represents a scenario variable for a GPR project file.
     */
    public static final class ScenarioVariable {

        // ----- Class attributes -----

        /** Singleton that represents the none scenario variable. */
        public static final ScenarioVariable NONE =
            new ScenarioVariable(null, null);

        // ----- Instance attributes -----

        /** The name of the variable. */
        public final String name;

        /** The value of the variable. */
        public final String value;

        // ----- Constructors -----

        /**
         * Create a new scenario variable with its name and value.
         *
         * @param name Name of the variable.
         * @param value Value of the variable.
         */
        ScenarioVariable(
            final String name,
            final String value
        ) {
            this.name = name;
            this.value = value;
        }

        /**
         * Public method to create scenario variable.
         * This method raise a runtime exception if name or value is null.
         *
         * @param name Name of the variable.
         * @param value Value of the variable.
         */
        public static ScenarioVariable create(
            final String name,
            final String value
        ) {
            if(name == null) throw new RuntimeException("Scenario variable name cannot be null");
            if(value == null) throw new RuntimeException("Scenario variable value cannot be null");
            return new ScenarioVariable(name, value);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native scenario variable.
         *
         * @param pointer The pointer to the native scenario variable.
         * @return The wrapped scenario variable.
         */
        public static ScenarioVariable wrap(
            final Pointer pointer
        ) {
            return wrap((ScenarioVariableNative) pointer.readWord(0));
        }

        /**
         * Wrap the given native scenario variable.
         *
         * @param scenarioVariableNative The native scenario variable.
         * @return The wrapped scenario variable.
         */
        public static ScenarioVariable wrap(
            final ScenarioVariableNative scenarioVariableNative
        ) {
            final CCharPointer nameNative = scenarioVariableNative.get_name();
            final CCharPointer valueNative = scenarioVariableNative.get_value();
            return new ScenarioVariable(
                nameNative.isNull() ?
                    null :
                    toJString(nameNative),
                valueNative.isNull() ?
                    null :
                    toJString(valueNative)
            );
        }

        /**
         * Unwrap the scenario variable in the given native value.
         *
         * @param scenarioVariableNative The native value to fill.
         */
        public void unwrap(
            final ScenarioVariableNative scenarioVariableNative
        ) {
            final CCharPointer nameNative = this.name == null ?
                WordFactory.nullPointer() :
                toCString(this.name);
            final CCharPointer valueNative = this.value == null ?
                WordFactory.nullPointer() :
                toCString(this.value);
            scenarioVariableNative.set_name(nameNative);
            scenarioVariableNative.set_value(valueNative);
        }

        /**
         * Release the given native scenario variable allocated strings.
         *
         * @param scenarioVariableNative The native scenario variable to release.
         */
        public static void release(
            final ScenarioVariableNative scenarioVariableNative
        ) {
            if(scenarioVariableNative.get_name().isNonNull()) {
                UnmanagedMemory.free(scenarioVariableNative.get_name());
            }
            if(scenarioVariableNative.get_value().isNonNull()) {
                UnmanagedMemory.free(scenarioVariableNative.get_value());
            }
        }

        // ----- Override methods -----

        /** @see java.lang.Object#toString() */
        public String toString() {
            return "ScenarioVariable(" +
                this.name + " = " + this.value +
                ")";
        }

    }

    /**
     * This class is used for the GPR project loading.
     */
    public static final class ProjectManager implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none project manager. */
        public static final ProjectManager NONE =
            new ProjectManager(PointerWrapper.nullPointer());

        // ----- Instance attributes -----

        /** Reference to the native value. */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new project manager from its native reference.
         *
         * @param reference The reference to the native project manager.
         */
        ProjectManager(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        /**
         * Create a project manager for the given project file.
         *
         * @param projectFile The GPR project file to load.
         * @return The newly created project manager.
         */
        public static ProjectManager create(
            final String projectFile
        ) {
            return create(
                projectFile,
                null,
                "",
                ""
            );
        }

        /**
         * Create a project manager from a project file, target and runtime.
         *
         * @param projectFile The GPR file to load.
         * @param scenarioVariables The scenario variables for the project, it can be null.
         * @param target The target to load.
         * @param runtime The runtime to load.
         * @return The newly created project manager.
         */
        public static ProjectManager create(
            final String projectFile,
            final ScenarioVariable[] scenarioVariables,
            final String target,
            final String runtime
        ) {

            if(ImageInfo.inImageCode()) {
                // Create the scenario variable array
                final Pointer scenarioVariablesNative;
                final int scenarioVariableNativeSize = SizeOf.get(ScenarioVariableNative.class);

                if(scenarioVariables != null && scenarioVariables.length > 0) {
                    final int size = scenarioVariables.length + 1;
                    scenarioVariablesNative = UnmanagedMemory.calloc(
                        size * scenarioVariableNativeSize
                    );
                    for(int i = 0 ; i < scenarioVariables.length ; i++) {
                        final ScenarioVariableNative scenarioVariableNative = (ScenarioVariableNative)
                            scenarioVariablesNative.add(i * scenarioVariableNativeSize);
                        scenarioVariables[i].unwrap(scenarioVariableNative);
                    }
                } else {
                    scenarioVariablesNative = WordFactory.nullPointer();
                }

                // Call the native project loading function
                final CCharPointer projectFileNative = toCString(projectFile);
                final CCharPointer targetNative = toCString(target);
                final CCharPointer runtimeNative = toCString(runtime);
                final Pointer projectPointer = StackValue.get(SizeOf.get(VoidPointer.class));
                projectPointer.writeWord(0, WordFactory.nullPointer());
                final Pointer errorsPointer = StackValue.get(SizeOf.get(VoidPointer.class));
                errorsPointer.writeWord(0, WordFactory.nullPointer());
                NI_LIB.ada_gpr_project_load(
                    projectFileNative,
                    scenarioVariablesNative,
                    targetNative,
                    runtimeNative,
                    projectPointer,
                    errorsPointer
                );

                // Check the langkit exception and cast it into a project manager error
                try {
                    checkException();
                } catch (LangkitException e) {
                    throw new ProjectManagerException(e.getMessage());
                }

                // Get the result of modified values
                final ProjectManagerNative projectManagerNative = (ProjectManagerNative) projectPointer.readWord(0);
                final StringArrayNative errorArrayNative = (StringArrayNative) errorsPointer.readWord(0);

                // Free the scenario variables
                if(scenarioVariablesNative.isNonNull()) {
                    for(int i = 0 ; i < scenarioVariables.length ; i++) {
                        final ScenarioVariableNative scenarioVariableNative = (ScenarioVariableNative)
                            scenarioVariablesNative.add(i * scenarioVariableNativeSize);
                        ScenarioVariable.release(scenarioVariableNative);
                    }
                    UnmanagedMemory.free(scenarioVariablesNative);
                }

                // Free the allocated strings
                UnmanagedMemory.free(projectFileNative);
                UnmanagedMemory.free(targetNative);
                UnmanagedMemory.free(runtimeNative);

                // `errorsPointer` is not allocated if an exception was raised during project file loading
                if (errorArrayNative.isNonNull()) {
                    // Translate the error native array into a Java array
                    final String[] errors = toJStringArray(errorArrayNative);
                    for (String error : errors) {
                        System.err.println("Error during project opening: " + error);
                    }

                    // Free the error array
                    NI_LIB.ada_free_string_array(errorArrayNative);
                }

                // Check the langkit exception and cast it into a project manager error
                try {
                    checkException();
                } catch (LangkitException e) {
                    throw new ProjectManagerException(e.getMessage());
                }

                // Return the Java wrapped project manager
                return wrap(projectManagerNative);
            } else {
                final PointerWrapper reference = JNI_LIB.ada_gpr_project_load(
                    projectFile,
                    scenarioVariables,
                    target,
                    runtime
                );

                // Check the langkit exceptions
                try {
                    checkException();
                } catch (LangkitException e) {
                    throw new ProjectManagerException(e.getMessage());
                }

                // Return the project manager
                return new ProjectManager(reference);
            }

        }

        // ----- Graal C API methods -----

        /**
         * Wrap a native project manager in the Java class.
         *
         * @param pointer The pointer to the native project manager.
         * @return The newly wrapped project manager.
         */
        static ProjectManager wrap(
            final Pointer pointer
        ) {
            return wrap((ProjectManagerNative) pointer.readWord(0));
        }

        /**
         * Wrap a native project manager in the Java class.
         *
         * @param projectManagerNative The native project manager to wrap.
         * @return The newly wrapped project manager.
         */
        static ProjectManager wrap(
            final ProjectManagerNative projectManagerNative
        ) {
            return new ProjectManager(new PointerWrapper(projectManagerNative));
        }

        /**
         * Unwrap the project manager inside the given pointer.
         *
         * @param pointer The pointer to write in.
         */
        public void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Get the native value of the project manager.
         *
         * @return The native project manager.
         */
        public ProjectManagerNative unwrap() {
            return (ProjectManagerNative) this.reference.ni();
        }

        // ----- Class methods -----

        /**
         * Translate a native string array structure into a Java string
         * array.
         *
         * @param stringArrayNative The native string array structure.
         * @return The Java string array.
         */
        private static String[] toJStringArray(
            final StringArrayNative stringArrayNative
        ) {
            final String[] res = new String[stringArrayNative.get_length()];
            final CCharPointerPointer nativeFilesPointer = stringArrayNative.get_c_ptr();
            for(int i = 0 ; i < res.length ; i++) {
                final CCharPointer nativeFile = nativeFilesPointer.read(i);
                res[i] = toJString(nativeFile);
            }
            return res;
        }

        // -----  Instance methods -----

        /**
         * Create a unit provider for the given subproject.
         *
         * @param subproject The subproject for which to create a unit provider.
         * @return The unit provider for the project manager.
         */
        public UnitProvider getProvider(final String subproject) {
            final UnitProvider result;
            if(ImageInfo.inImageCode()) {
                final CCharPointer subprojectNative =
                    subproject == null ?
                    WordFactory.nullPointer() :
                    toCString(subproject);

                UnitProviderNative unitProviderNative = NI_LIB.ada_gpr_project_create_unit_provider(
                    this.reference.ni(),
                    subprojectNative
                );
                result = UnitProvider.wrap(unitProviderNative);
                if (subproject != null) {
                    UnmanagedMemory.free(subprojectNative);
                }
            } else {
                result = JNI_LIB.ada_gpr_project_create_unit_provider(
                    this,
                    subproject
                );
            }
            return result;
        }

        /**
         * Create a unit provider for root project.
         */
        public UnitProvider getProvider() {
            return this.getProvider(null);
        }

        /**
         * Create a new analysis context from a GPR project.
         *
         * The unit provider, file reader, config pragmas and default charset
         * are inferred from the designated project: see the ``getProvider``
         * method for the semantics of the ``project`` argument.
         *
         * See the ``AnalysisContext`` class constructor for the semantics of
         * the other arguments.
         *
         * .. TODO: For now, the returned ``AnalysisContext`` instance has a
         *    weak reference to the project manager: make sure the
         *    ``ProjectManager`` instance lives at least as long as the
         *    ``AnalysisContext`` one.
         */
        public AnalysisContext createContext(
            String subproject,
            EventHandler eventHandler,
            boolean withTrivia,
            int tabStop
        ) {
            if(ImageInfo.inImageCode()) {
                // Prepare C values for native calls
                final CCharPointer subproject_c =
                  subproject == null
                  ? WordFactory.nullPointer()
                  : toCString(subproject);
                EventHandlerNative eventHandler_c =
                  eventHandler == null
                  ? WordFactory.nullPointer()
                  : eventHandler.reference.ni();

                // Manually allocate a C-level analysis context so that we can
                // initialize it ourselves.
                final PointerWrapper context = new PointerWrapper(
                    NI_LIB.ada_allocate_analysis_context()
                );

                // Create the Java wrapper, so that we have one ready for
                // event handler callbacks triggered during context
                // initialization.
                AnalysisContext result =
                    AnalysisContext.fromReference(context, eventHandler, true);

                // "result" has its own ownership share: release ours
                NI_LIB.ada_context_decref(context.ni());

                // TODO: attach "this" to "result" so that the former lives at
                // least as long as the former.

                // Finally, initialize the analysis context. Note that this
                // step may raise an exception: in that case, the analysis
                // context is considered not initialized: release it.
                NI_LIB.ada_gpr_project_initialize_context(
                    this.reference.ni(),
                    context.ni(),
                    subproject_c,
                    eventHandler_c,
                    withTrivia ? 1 : 0,
                    tabStop
                );
                UnmanagedMemory.free(subproject_c);
                final LangkitExceptionNative exc_c =
                    NI_LIB.ada_get_last_exception();
                if (exc_c.isNonNull()) {
                    LangkitException exc = wrapException(exc_c);
                    NI_LIB.ada_context_decref(context.ni());
                    throw exc;
                }

                return result;
            } else {
                return JNI_LIB.ada_gpr_project_create_context(
                    this,
                    subproject,
                    eventHandler,
                    withTrivia,
                    tabStop
                );
            }
        }

        /**
         * Get the files for the given subprojects in a string array.
         *
         * @param mode The file getting mode.
         * @param subprojects The subprojects to consider.
         * @return The array that contains the project files.
         */
        public String[] getFiles(
            final SourceFileMode mode,
            final String[] subprojects
        ) {
            // Verify if the project is null
            if(this.reference.isNull())
                return new String[0];

            String[] result = null;
            LangkitException exc;

            if(ImageInfo.inImageCode()) {
                // Convert the Java array of subprojects ("subprojects"
                // argument) into the required C array
                // (subprojectCount/subprojectsNative).
                final int subprojectCount;
                final CCharPointerPointer subprojectsNative;

                if (subprojects != null && subprojects.length > 0) {
                    subprojectCount = subprojects.length;
                    subprojectsNative =
                        UnmanagedMemory.calloc(
                            subprojectCount * SizeOf.get(CCharPointerPointer.class)
                        );
                } else {
                    subprojectCount = 0;
                    subprojectsNative = WordFactory.nullPointer();
                }

                for (int i = 0; i < subprojectCount; ++i) {
                    subprojectsNative.write(i, toCString(subprojects[i]));
                }

                // Call the C API. Keep track of a potential native exception.
                StringArrayNative sourceFileArray =
                    NI_LIB.ada_gpr_project_source_files(
                        this.reference.ni(),
                        mode.toC(),
                        subprojectsNative,
                        subprojectCount
                    );
                exc = getLastException();

                // If the call was successful, create the Java array result and
                // initialize it, and free the C array result.
                if(exc == null) {
                    result = toJStringArray(sourceFileArray);
                    NI_LIB.ada_free_string_array(sourceFileArray);
                }

                // Release subprojectsNative
                for (int i = 0; i < subprojectCount; ++i) {
                    UnmanagedMemory.free(subprojectsNative.read(i));
                }
                if (subprojectCount > 0) {
                    UnmanagedMemory.free(subprojectsNative);
                }
            } else {
                result = JNI_LIB.ada_gpr_project_source_files(
                    this,
                    mode.toC(),
                    subprojects
                );
                exc = getLastException();
            }

            // If we got an exception, turn it into a ProjectManagerException
            // one.
            if(exc != null) {
                throw new ProjectManagerException(exc.getMessage());
            }

            return result;

        }

        /**
         * Get the files of the root project in a string array.
         *
         * @param mode The file getting mode.
         * @return The array that contains the project files.
         */
        public String[] getFiles(final SourceFileMode mode) {
            return this.getFiles(mode, null);
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.ada_gpr_project_free(this.reference.ni());
            } else {
                JNI_LIB.ada_gpr_project_free(this);
            }

        }

    }

    /**
     * Return a unit provider that knows which compilation units are to be
     * found in the given list of source files.
     *
     * This knowledge is built trying to parse all given input files as Ada
     * source files and listing the compilation units found there. Files that
     * cannot be parsed properly are discarded. If two compilation units are
     * found for the same unit, the first that is found in the given input
     * files is taken and the other ones are discarded.
     *
     * Source files are decoded using the given charset. If it is ``null``, the
     * default charset (ISO-8859-1) is used.
     *
     * .. TODO: Find a way to report discarded source files/compilation units.
     */
    public static UnitProvider createAutoProvider(
        final String[] sourceFiles,
        final String charset
    ) {
        if (ImageInfo.inImageCode()) {
            final CCharPointer charsetNative =
                charset == null ?
                WordFactory.nullPointer() :
                toCString(charset);

            // Allocate the C array of C strings that will contain decoded
            // source file names. Make room for one additional null pointer
            // to mark the end of the array.
            final CCharPointerPointer sourceFilesNative =
                UnmanagedMemory.calloc(
                    (sourceFiles.length + 1) * SizeOf.get(CCharPointerPointer.class)
                );

            for (int i = 0; i < sourceFiles.length; ++i) {
                sourceFilesNative.write(i, toCString(sourceFiles[i]));
            }

            // Create the auto provider
            final UnitProviderNative unitProviderNative =
                NI_LIB.ada_create_auto_provider(
                    sourceFilesNative,
                    charsetNative
                );

            // Release all temporarily allocated memory
            for (int i = 0; i < sourceFiles.length; ++i) {
                UnmanagedMemory.free(sourceFilesNative.read(i));
            }

            UnmanagedMemory.free(sourceFilesNative);

            if (charset != null) {
                UnmanagedMemory.free(charsetNative);
            }

            return UnitProvider.wrap(unitProviderNative);
        } else {
            return JNI_LIB.ada_create_auto_provider(
                sourceFiles,
                charset
            );
        }
    }



    // ==========
    // Defining the JNI bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains all native function definitions for JNI */
    public static final class JNI_LIB {

        // ----- Static initializer -----

        static {
            if(!ImageInfo.inImageCode()) {
                // Load the needed libraries
                if(OS.indexOf("win") < 0) {
                    System.loadLibrary("langkit_sigsegv_handler");
                }
                System.loadLibrary("adalang_jni");

                // Initialize the JNI library
                ada_initialize();

                // Register the library finalizer
                Runtime.getRuntime().addShutdownHook(
                    new Thread(JNI_LIB::ada_finalize)
                );
            }
        }

        // ----- Language specific functions -----

        
      


    // --- Project manager functions

    /** Function to load a GPR project */
    @CompilerDirectives.TruffleBoundary
    public static native PointerWrapper ada_gpr_project_load(
        String projectFile,
        ScenarioVariable[] scenarioVariables,
        String target,
        String runtime
    );

    /** Free the given project */
    @CompilerDirectives.TruffleBoundary
    public static native void ada_gpr_project_free(
        ProjectManager project
    );

    /** Get the unit provider for the given project */
    @CompilerDirectives.TruffleBoundary
    public static native UnitProvider ada_gpr_project_create_unit_provider(
        ProjectManager project,
        String subproject
    );

    /** Get the source file array for the given project */
    @CompilerDirectives.TruffleBoundary
    public static native String[] ada_gpr_project_source_files(
        ProjectManager project,
        int mode,
        String[] subprojects
    );

    /**
     * Create a new analysis context from a GPR project.
     *
     * The unit provider, file reader, config pragmas and default charset are
     * inferred from the designated project: see the ``getProvider`` method for
     * the semantics of the ``project`` argument.
     *
     * See the ``AnalysisContext`` class constructor for the semantics of the
     * other arguments.
     *
     * .. TODO: For now, the returned ``AnalysisContext`` instance has a weak
     *    reference to the project manager: make sure the ``ProjectManager``
     *    instance lives at least as long as the ``AnalysisContext`` one.
     */
    public static native AnalysisContext ada_gpr_project_create_context(
        ProjectManager project,
        String subproject,
        EventHandler eventHandler,
        boolean withTrivia,
        int tabStop
    );

    /** Create an auto provider reference */
    @CompilerDirectives.TruffleBoundary
    public static native UnitProvider ada_create_auto_provider(
        String[] sourceFiles,
        String charset
    );

    // --- Config pragmas

    @CompilerDirectives.TruffleBoundary
    public static native void ada_set_config_pragmas_mapping(
        AnalysisContext context,
        AnalysisUnit globalPragmas,
        AnalysisUnit[] localPragmas
    );



        // ----- Lifecycle functions ------

        /** Function to initialize the JNI library */
        public static native void ada_initialize();

        /** Function to finalize the JNI library */
        public static native void ada_finalize();

        // ----- Exception functions ------

        /** Get the last langkit exception */
        @CompilerDirectives.TruffleBoundary
        public static native LangkitException ada_get_last_exception();

        // ----- Text functions -----

        /** Create a new text from its content */
        @CompilerDirectives.TruffleBoundary
        public static native Text ada_create_text(
            byte[] utf32Content
        );

        /** Destroy the given text */
        @CompilerDirectives.TruffleBoundary
        public static native void ada_destroy_text(
            Text text
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        public static native void ada_dec_ref_file_reader(
            FileReader fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        public static native void ada_dec_ref_unit_provider(
            UnitProvider unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper ada_create_event_handler(
            EventHandler.UnitRequestedCallback unitRequestedCallback,
            EventHandler.UnitParsedCallback unitParsedCallback
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        public static native void ada_dec_ref_event_handler(
            EventHandler eventHandler
        );

        // ----- Token functions -----

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        public static native Token ada_token_next(
            Token token
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        public static native Token ada_token_previous(
            Token token
        );

        /** Get if the tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_token_is_equivalent(
            Token left,
            Token right
        );

        /** Get text between the two tokens */
        @CompilerDirectives.TruffleBoundary
        public static native Text ada_token_range_text(
            Token start,
            Token end
        );

        // ----- Analysis context functions -----

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper ada_create_analysis_context(
            String charset,
            FileReader fileReader,
            UnitProvider unitProvider,
            EventHandler eventHandler,
            boolean withTrivia,
            int tabstop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void ada_context_incref(
            long context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void ada_context_decref(
            long context
        );

        // ----- Analysis unit functions -----

        /** Get the analysis unit from a file */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        ada_get_analysis_unit_from_file(
            AnalysisContext context,
            String fileName,
            String charset,
            boolean reparse,
            int grammarRule
        );

        /** Get the analysis unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        ada_get_analysis_unit_from_buffer(
            AnalysisContext context,
            String fileName,
            String charset,
            String buffer,
            long bufferSize,
            int grammarRule
        );

        /** Get the analysis unit from the unit provider. */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        ada_get_analysis_unit_from_provider(
            AnalysisContext context,
            Text name,
            int kind,
            String charset,
            boolean reparse
        );

        /** Get the root of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_unit_root(
            AnalysisUnit unit
        );

        /** Get the file name of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_unit_filename(
            AnalysisUnit unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_unit_token_count(
            AnalysisUnit unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_unit_trivia_count(
            AnalysisUnit unit
        );

        /** Get the first token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token ada_unit_first_token(
            AnalysisUnit unit
        );

        /** Get the last token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token ada_unit_last_token(
            AnalysisUnit unit
        );

        /** Get the context of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisContext ada_unit_context(
            AnalysisUnit unit
        );

        /** Get the number of diagnostic in the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_unit_diagnostic_count(
            AnalysisUnit unit
        );

        /** Get the nth diagnostic of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Diagnostic ada_unit_diagnostic(
            AnalysisUnit unit,
            int n
        );

        // ----- Node functions -----

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_node_is_equivalent(
            Entity entity_left,
            Entity entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_node_hash(
            Entity entity
        );

        /** Get the node kind */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_node_kind(
            Entity entity
        );

        /** Get the node text */
        @CompilerDirectives.TruffleBoundary
        public static native Text ada_node_text(
            Entity entity
        );

        /** Get the node source location range */
        @CompilerDirectives.TruffleBoundary
        public static native SourceLocationRange ada_node_sloc_range(
            Entity entity
        );

        /** Get the node children count */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_node_children_count(
            Entity entity
        );

        /** Get the node nth child */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_node_child(
            Entity entity,
            int n
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_node_is_token_node(
            Entity entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit ada_node_unit(
            Entity entity
        );

        /** Get the entity image of the node */
        @CompilerDirectives.TruffleBoundary
        public static native Text ada_node_image(
            Entity entity
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of ada_ada_node_p_declarative_scope langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_declarative_scope(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_enclosing_compilation_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_enclosing_compilation_unit(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_get_uninstantiated_node langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_get_uninstantiated_node(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_complete langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native CompletionItemIterator ada_ada_node_p_complete(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_valid_keywords langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native UnboundedTextTypeArray ada_ada_node_p_valid_keywords(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_generic_instantiations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_ada_node_p_generic_instantiations(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_semantic_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_semantic_parent(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_parent_basic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_parent_basic_decl(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_filter_is_imported_by langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnitArray ada_ada_node_p_filter_is_imported_by(
            AnalysisUnitArray units,
            boolean transitive,
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_xref_entry_point langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_ada_node_p_xref_entry_point(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_resolve_names langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_ada_node_p_resolve_names(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_standard_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit ada_ada_node_p_standard_unit(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_std_entity langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_std_entity(
            Symbol sym,
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_bool_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_bool_type(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_int_type(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_universal_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_universal_int_type(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_universal_real_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_universal_real_type(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_std_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_std_char_type(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_std_wide_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_std_wide_char_type(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_std_wide_wide_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_std_wide_wide_char_type(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_top_level_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_top_level_decl(
            AnalysisUnit unit,
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_choice_match langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_ada_node_p_choice_match(
            BigInteger value,
            Entity node
        );
            

        /** Isomethod of ada_ada_node_p_gnat_xref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_p_gnat_xref(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_ada_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_parent(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_ada_node_parents(
            boolean with_self,
            Entity node
        );
            

        /** Isomethod of ada_ada_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_ada_node_children(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token ada_ada_node_token_start(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token ada_ada_node_token_end(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_ada_node_child_index(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_previous_sibling(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ada_node_next_sibling(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit ada_ada_node_unit(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_ada_node_is_ghost(
            Entity node
        );
            

        /** Isomethod of ada_ada_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_ada_node_full_sloc_image(
            Entity node
        );

        
    

            

        /** Isomethod of ada_abort_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_abort_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_abstract_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_abstract_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_assoc_list_p_zip_with_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native ParamActualArray ada_assoc_list_p_zip_with_params(
            boolean imprecise_fallback,
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_aliased_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_aliased_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_all_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_all_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_constrained_array_indices_f_list langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_constrained_array_indices_f_list(
            Entity node
        );

        
    

            

        /** Isomethod of ada_unconstrained_array_indices_f_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_unconstrained_array_indices_f_types(
            Entity node
        );

        
    

            

        /** Isomethod of ada_aspect_assoc_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_aspect_assoc_f_id(
            Entity node
        );
            

        /** Isomethod of ada_aspect_assoc_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_aspect_assoc_f_expr(
            Entity node
        );
            

        /** Isomethod of ada_aspect_assoc_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_aspect_assoc_p_is_ghost_code(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_at_clause_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_at_clause_f_name(
            Entity node
        );
            

        /** Isomethod of ada_at_clause_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_at_clause_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_attribute_def_clause_f_attribute_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_attribute_def_clause_f_attribute_expr(
            Entity node
        );
            

        /** Isomethod of ada_attribute_def_clause_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_attribute_def_clause_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_enum_rep_clause_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_enum_rep_clause_f_type_name(
            Entity node
        );
            

        /** Isomethod of ada_enum_rep_clause_f_aggregate langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_enum_rep_clause_f_aggregate(
            Entity node
        );
            

        /** Isomethod of ada_enum_rep_clause_p_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native ParamActualArray ada_enum_rep_clause_p_params(
            Entity node
        );

        
    

            

        /** Isomethod of ada_record_rep_clause_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_record_rep_clause_f_name(
            Entity node
        );
            

        /** Isomethod of ada_record_rep_clause_f_at_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_record_rep_clause_f_at_expr(
            Entity node
        );
            

        /** Isomethod of ada_record_rep_clause_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_record_rep_clause_f_components(
            Entity node
        );

        
    

            

        /** Isomethod of ada_aspect_spec_f_aspect_assocs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_aspect_spec_f_aspect_assocs(
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_assoc_p_assoc_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_assoc_p_assoc_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_contract_case_assoc_f_guard langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_contract_case_assoc_f_guard(
            Entity node
        );
            

        /** Isomethod of ada_contract_case_assoc_f_consequence langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_contract_case_assoc_f_consequence(
            Entity node
        );

        
    

            

        /** Isomethod of ada_pragma_argument_assoc_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pragma_argument_assoc_f_name(
            Entity node
        );
            

        /** Isomethod of ada_pragma_argument_assoc_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pragma_argument_assoc_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_formal_param_holder_p_abstract_formal_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_formal_param_holder_p_abstract_formal_params(
            Entity node
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_formal_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_formal_param_holder_p_formal_params(
            Entity node
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_nb_min_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_base_formal_param_holder_p_nb_min_params(
            Entity node
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_nb_max_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int ada_base_formal_param_holder_p_nb_max_params(
            Entity node
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_param_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_formal_param_holder_p_param_types(
            Entity origin,
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_subp_spec_p_returns langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_subp_spec_p_returns(
            Entity node
        );
            

        /** Isomethod of ada_base_subp_spec_p_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_subp_spec_p_params(
            Entity node
        );
            

        /** Isomethod of ada_base_subp_spec_p_primitive_subp_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_subp_spec_p_primitive_subp_types(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_base_subp_spec_p_primitive_subp_first_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_subp_spec_p_primitive_subp_first_type(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_base_subp_spec_p_primitive_subp_tagged_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_subp_spec_p_primitive_subp_tagged_type(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_base_subp_spec_p_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_subp_spec_p_return_type(
            Entity origin,
            Entity node
        );

        
    

            

        /** Isomethod of ada_entry_spec_f_entry_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_spec_f_entry_name(
            Entity node
        );
            

        /** Isomethod of ada_entry_spec_f_family_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_spec_f_family_type(
            Entity node
        );
            

        /** Isomethod of ada_entry_spec_f_entry_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_spec_f_entry_params(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_subp_spec_f_subp_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_spec_f_subp_kind(
            Entity node
        );
            

        /** Isomethod of ada_subp_spec_f_subp_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_spec_f_subp_name(
            Entity node
        );
            

        /** Isomethod of ada_subp_spec_f_subp_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_spec_f_subp_params(
            Entity node
        );
            

        /** Isomethod of ada_subp_spec_f_subp_returns langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_spec_f_subp_returns(
            Entity node
        );

        
    

            

        /** Isomethod of ada_synthetic_binary_spec_f_left_param langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_binary_spec_f_left_param(
            Entity node
        );
            

        /** Isomethod of ada_synthetic_binary_spec_f_right_param langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_binary_spec_f_right_param(
            Entity node
        );
            

        /** Isomethod of ada_synthetic_binary_spec_f_return_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_binary_spec_f_return_type_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_synthetic_unary_spec_f_right_param langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_unary_spec_f_right_param(
            Entity node
        );
            

        /** Isomethod of ada_synthetic_unary_spec_f_return_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_unary_spec_f_return_type_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_component_list_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_list_f_components(
            Entity node
        );
            

        /** Isomethod of ada_component_list_f_variant_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_list_f_variant_part(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_known_discriminant_part_f_discr_specs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_known_discriminant_part_f_discr_specs(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_entry_completion_formal_params_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_completion_formal_params_f_params(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_formal_part_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_formal_part_f_decls(
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_record_def_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_record_def_f_components(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_basic_assoc_p_get_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_basic_assoc_p_get_params(
            boolean imprecise_fallback,
            Entity node
        );

        
    

            

        /** Isomethod of ada_aggregate_assoc_f_designators langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_aggregate_assoc_f_designators(
            Entity node
        );
            

        /** Isomethod of ada_aggregate_assoc_f_r_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_aggregate_assoc_f_r_expr(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_composite_constraint_assoc_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_composite_constraint_assoc_f_ids(
            Entity node
        );
            

        /** Isomethod of ada_composite_constraint_assoc_f_constraint_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_composite_constraint_assoc_f_constraint_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_iterated_assoc_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_iterated_assoc_f_spec(
            Entity node
        );
            

        /** Isomethod of ada_iterated_assoc_f_r_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_iterated_assoc_f_r_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_param_assoc_f_designator langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_param_assoc_f_designator(
            Entity node
        );
            

        /** Isomethod of ada_param_assoc_f_r_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_param_assoc_f_r_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_basic_decl_p_is_formal langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_formal(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_doc_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DocAnnotationArray ada_basic_decl_p_doc_annotations(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_basic_decl_p_doc(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_previous_part_for_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_previous_part_for_decl(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_canonical_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_canonical_part(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_all_parts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_basic_decl_p_all_parts(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_is_static_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_static_decl(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_f_aspects langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_f_aspects(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_get_aspect_assoc langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_get_aspect_assoc(
            Symbol name,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_get_aspect_spec_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_get_aspect_spec_expr(
            Symbol name,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_get_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Aspect ada_basic_decl_p_get_aspect(
            Symbol name,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_has_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_has_aspect(
            Symbol name,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_get_pragma langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_get_pragma(
            Symbol name,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_get_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_get_representation_clause(
            Symbol name,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_get_at_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_get_at_clause(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_is_imported langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_imported(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_ghost_code(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_is_compilation_unit_root langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_compilation_unit_root(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_is_visible langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_visible(
            Entity from_node,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_base_subp_declarations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_basic_decl_p_base_subp_declarations(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_root_subp_declarations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_basic_decl_p_root_subp_declarations(
            Entity origin,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_find_all_overrides langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_basic_decl_p_find_all_overrides(
            AnalysisUnitArray units,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_defining_names langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_basic_decl_p_defining_names(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_defining_name(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_type_expression langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_type_expression(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_subp_spec_or_null langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_subp_spec_or_null(
            boolean follow_generic,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_is_subprogram langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_subprogram(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_relative_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_relative_name(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_relative_name_text langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Symbol ada_basic_decl_p_relative_name_text(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_next_part_for_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_next_part_for_decl(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_body_part_for_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_body_part_for_decl(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_most_visible_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_decl_p_most_visible_part(
            Entity origin,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_fully_qualified_name_array langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native UnboundedTextTypeArray ada_basic_decl_p_fully_qualified_name_array(
            boolean include_profile,
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_basic_decl_p_fully_qualified_name(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_canonical_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_basic_decl_p_canonical_fully_qualified_name(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_unique_identifying_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_basic_decl_p_unique_identifying_name(
            Entity node
        );
            

        /** Isomethod of ada_basic_decl_p_is_constant_object langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_decl_p_is_constant_object(
            Entity node
        );

        
    

            

        /** Isomethod of ada_abstract_state_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_abstract_state_decl_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_anonymous_expr_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_anonymous_expr_decl_f_expr(
            Entity node
        );
            

        /** Isomethod of ada_anonymous_expr_decl_p_get_formal langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_anonymous_expr_decl_p_get_formal(
            boolean imprecise_fallback,
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_formal_param_decl_p_formal_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_formal_param_decl_p_formal_type(
            Entity origin,
            Entity node
        );

        
    

            

        /** Isomethod of ada_component_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_decl_f_ids(
            Entity node
        );
            

        /** Isomethod of ada_component_decl_f_component_def langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_decl_f_component_def(
            Entity node
        );
            

        /** Isomethod of ada_component_decl_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_decl_f_default_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_discriminant_spec_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_discriminant_spec_f_ids(
            Entity node
        );
            

        /** Isomethod of ada_discriminant_spec_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_discriminant_spec_f_type_expr(
            Entity node
        );
            

        /** Isomethod of ada_discriminant_spec_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_discriminant_spec_f_default_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_formal_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_formal_f_decl(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_param_spec_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_param_spec_f_ids(
            Entity node
        );
            

        /** Isomethod of ada_param_spec_f_has_aliased langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_param_spec_f_has_aliased(
            Entity node
        );
            

        /** Isomethod of ada_param_spec_f_mode langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_param_spec_f_mode(
            Entity node
        );
            

        /** Isomethod of ada_param_spec_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_param_spec_f_type_expr(
            Entity node
        );
            

        /** Isomethod of ada_param_spec_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_param_spec_f_default_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_synthetic_formal_param_decl_f_param_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_formal_param_decl_f_param_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_package_decl_f_package_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_package_decl_f_package_name(
            Entity node
        );
            

        /** Isomethod of ada_base_package_decl_f_public_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_package_decl_f_public_part(
            Entity node
        );
            

        /** Isomethod of ada_base_package_decl_f_private_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_package_decl_f_private_part(
            Entity node
        );
            

        /** Isomethod of ada_base_package_decl_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_package_decl_f_end_name(
            Entity node
        );
            

        /** Isomethod of ada_base_package_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_package_decl_p_body_part(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_base_type_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_f_name(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_base_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_base_subtype(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_private_completion langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_private_completion(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_inherited_primitive langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_inherited_primitive(
            Entity p,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_get_record_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_get_record_representation_clause(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_get_enum_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_get_enum_representation_clause(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_record_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_record_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_array_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_array_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_find_derived_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_type_decl_p_find_derived_types(
            Entity root,
            Entity origin,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_real_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_real_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_float_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_float_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_fixed_point langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_fixed_point(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_scalar_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_scalar_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_enum_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_enum_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_access_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_access_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_specific_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_specific_type(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_char_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_classwide_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_classwide_type(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_discrete_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DiscreteRange ada_base_type_decl_p_discrete_range(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_discrete_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_discrete_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_int_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_accessed_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_accessed_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_tagged_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_tagged_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_base_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_base_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_type_decl_p_base_types(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_find_all_derived_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_type_decl_p_find_all_derived_types(
            AnalysisUnitArray units,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_comp_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_comp_type(
            boolean is_subscript,
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_index_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_index_type(
            int dim,
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_derived_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_derived_type(
            Entity other_type,
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_interface_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_interface_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_matching_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_matching_type(
            Entity expected_type,
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_canonical_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_canonical_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_previous_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_previous_part(
            boolean go_to_incomplete,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_next_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_next_part(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_full_view langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_full_view(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_definite_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_definite_subtype(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_is_private langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_type_decl_p_is_private(
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_discriminants_list langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_base_type_decl_p_discriminants_list(
            Entity stop_recurse_at,
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_root_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_type_decl_p_root_type(
            Entity origin,
            Entity node
        );
            

        /** Isomethod of ada_base_type_decl_p_shapes langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native ShapeArray ada_base_type_decl_p_shapes(
            boolean include_discriminants,
            Entity origin,
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_subtype_decl_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_subtype_decl_p_get_type(
            Entity origin,
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_subtype_decl_f_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subtype_decl_f_subtype(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_incomplete_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_incomplete_type_decl_f_discriminants(
            Entity node
        );

        
    

            

        /** Isomethod of ada_incomplete_formal_type_decl_f_is_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_incomplete_formal_type_decl_f_is_tagged(
            Entity node
        );
            

        /** Isomethod of ada_incomplete_formal_type_decl_f_default_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_incomplete_formal_type_decl_f_default_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_incomplete_tagged_type_decl_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_incomplete_tagged_type_decl_f_has_abstract(
            Entity node
        );

        
    

            

        /** Isomethod of ada_protected_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_type_decl_f_discriminants(
            Entity node
        );
            

        /** Isomethod of ada_protected_type_decl_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_type_decl_f_interfaces(
            Entity node
        );
            

        /** Isomethod of ada_protected_type_decl_f_definition langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_type_decl_f_definition(
            Entity node
        );

        
    

            

        /** Isomethod of ada_task_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_type_decl_f_discriminants(
            Entity node
        );
            

        /** Isomethod of ada_task_type_decl_f_definition langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_type_decl_f_definition(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_decl_f_discriminants(
            Entity node
        );
            

        /** Isomethod of ada_type_decl_f_type_def langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_decl_f_type_def(
            Entity node
        );
            

        /** Isomethod of ada_type_decl_p_get_primitives langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_type_decl_p_get_primitives(
            boolean only_inherited,
            boolean include_predefined_operators,
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_formal_type_decl_f_default_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_formal_type_decl_f_default_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_basic_subp_decl_p_has_spark_mode_on langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_basic_subp_decl_p_has_spark_mode_on(
            Entity node
        );
            

        /** Isomethod of ada_basic_subp_decl_p_subp_decl_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_basic_subp_decl_p_subp_decl_spec(
            Entity node
        );

        
    

            

        /** Isomethod of ada_classic_subp_decl_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_classic_subp_decl_f_overriding(
            Entity node
        );
            

        /** Isomethod of ada_classic_subp_decl_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_classic_subp_decl_f_subp_spec(
            Entity node
        );
            

        /** Isomethod of ada_classic_subp_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_classic_subp_decl_p_body_part(
            boolean imprecise_fallback,
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_formal_subp_decl_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_formal_subp_decl_f_default_expr(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_entry_decl_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_decl_f_overriding(
            Entity node
        );
            

        /** Isomethod of ada_entry_decl_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_decl_f_spec(
            Entity node
        );
            

        /** Isomethod of ada_entry_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_decl_p_body_part(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_entry_decl_p_accept_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_entry_decl_p_accept_stmts(
            Entity node
        );

        
    

            

        /** Isomethod of ada_enum_literal_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_enum_literal_decl_f_name(
            Entity node
        );
            

        /** Isomethod of ada_enum_literal_decl_p_enum_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_enum_literal_decl_p_enum_type(
            Entity node
        );
            

        /** Isomethod of ada_enum_literal_decl_p_enum_rep langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native BigInteger ada_enum_literal_decl_p_enum_rep(
            Entity node
        );

        
    

            

        /** Isomethod of ada_synthetic_char_enum_lit_p_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_char_enum_lit_p_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_subp_internal_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_internal_f_subp_spec(
            Entity node
        );

        
    

            

        /** Isomethod of ada_synthetic_subp_decl_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_subp_decl_f_spec(
            Entity node
        );

        
    

            

        /** Isomethod of ada_body_node_p_previous_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_body_node_p_previous_part(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_body_node_p_decl_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_body_node_p_decl_part(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_body_node_p_subunit_root langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_body_node_p_subunit_root(
            Entity node
        );

        
    

            

        /** Isomethod of ada_accept_stmt_body_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_accept_stmt_body_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_subp_body_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_subp_body_f_overriding(
            Entity node
        );
            

        /** Isomethod of ada_base_subp_body_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_subp_body_f_subp_spec(
            Entity node
        );
            

        /** Isomethod of ada_base_subp_body_p_is_subject_to_proof langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_subp_body_p_is_subject_to_proof(
            Entity node
        );
            

        /** Isomethod of ada_base_subp_body_p_has_spark_mode_on langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_subp_body_p_has_spark_mode_on(
            Entity node
        );

        
    

            

        /** Isomethod of ada_expr_function_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_expr_function_f_expr(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_subp_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_body_f_decls(
            Entity node
        );
            

        /** Isomethod of ada_subp_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_body_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_subp_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_body_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_subp_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_renaming_decl_f_renames(
            Entity node
        );

        
    

            

        /** Isomethod of ada_body_stub_p_syntactic_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native UnboundedTextTypeArray ada_body_stub_p_syntactic_fully_qualified_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_package_body_stub_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_body_stub_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_protected_body_stub_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_body_stub_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_subp_body_stub_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_body_stub_f_overriding(
            Entity node
        );
            

        /** Isomethod of ada_subp_body_stub_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subp_body_stub_f_subp_spec(
            Entity node
        );

        
    

            

        /** Isomethod of ada_task_body_stub_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_body_stub_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_entry_body_f_entry_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_body_f_entry_name(
            Entity node
        );
            

        /** Isomethod of ada_entry_body_f_index_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_body_f_index_spec(
            Entity node
        );
            

        /** Isomethod of ada_entry_body_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_body_f_params(
            Entity node
        );
            

        /** Isomethod of ada_entry_body_f_barrier langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_body_f_barrier(
            Entity node
        );
            

        /** Isomethod of ada_entry_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_body_f_decls(
            Entity node
        );
            

        /** Isomethod of ada_entry_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_body_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_entry_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_body_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_package_body_f_package_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_body_f_package_name(
            Entity node
        );
            

        /** Isomethod of ada_package_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_body_f_decls(
            Entity node
        );
            

        /** Isomethod of ada_package_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_body_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_package_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_body_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_protected_body_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_body_f_name(
            Entity node
        );
            

        /** Isomethod of ada_protected_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_body_f_decls(
            Entity node
        );
            

        /** Isomethod of ada_protected_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_body_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_task_body_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_body_f_name(
            Entity node
        );
            

        /** Isomethod of ada_task_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_body_f_decls(
            Entity node
        );
            

        /** Isomethod of ada_task_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_body_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_task_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_body_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_entry_index_spec_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_index_spec_f_id(
            Entity node
        );
            

        /** Isomethod of ada_entry_index_spec_f_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_entry_index_spec_f_subtype(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_exception_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_exception_decl_f_ids(
            Entity node
        );
            

        /** Isomethod of ada_exception_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_exception_decl_f_renames(
            Entity node
        );

        
    

            

        /** Isomethod of ada_exception_handler_f_exception_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_exception_handler_f_exception_name(
            Entity node
        );
            

        /** Isomethod of ada_exception_handler_f_handled_exceptions langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_exception_handler_f_handled_exceptions(
            Entity node
        );
            

        /** Isomethod of ada_exception_handler_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_exception_handler_f_stmts(
            Entity node
        );

        
    

            

        /** Isomethod of ada_for_loop_var_decl_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_for_loop_var_decl_f_id(
            Entity node
        );
            

        /** Isomethod of ada_for_loop_var_decl_f_id_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_for_loop_var_decl_f_id_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_decl_f_formal_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_decl_f_formal_part(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_package_decl_f_package_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_package_decl_f_package_decl(
            Entity node
        );
            

        /** Isomethod of ada_generic_package_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_package_decl_p_body_part(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_subp_decl_f_subp_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_decl_f_subp_decl(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_decl_p_body_part(
            boolean imprecise_fallback,
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_instantiation_p_designated_generic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_instantiation_p_designated_generic_decl(
            Entity node
        );
            

        /** Isomethod of ada_generic_instantiation_p_inst_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native ParamActualArray ada_generic_instantiation_p_inst_params(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_package_instantiation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_package_instantiation_f_name(
            Entity node
        );
            

        /** Isomethod of ada_generic_package_instantiation_f_generic_pkg_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_package_instantiation_f_generic_pkg_name(
            Entity node
        );
            

        /** Isomethod of ada_generic_package_instantiation_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_package_instantiation_f_params(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_subp_instantiation_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_instantiation_f_overriding(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_instantiation_f_kind(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_subp_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_instantiation_f_subp_name(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_generic_subp_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_instantiation_f_generic_subp_name(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_instantiation_f_params(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_instantiation_p_designated_subp langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_instantiation_p_designated_subp(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_generic_package_renaming_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_package_renaming_decl_f_name(
            Entity node
        );
            

        /** Isomethod of ada_generic_package_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_package_renaming_decl_f_renames(
            Entity node
        );

        
    

            

        /** Isomethod of ada_generic_subp_renaming_decl_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_renaming_decl_f_kind(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_renaming_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_renaming_decl_f_name(
            Entity node
        );
            

        /** Isomethod of ada_generic_subp_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_generic_subp_renaming_decl_f_renames(
            Entity node
        );

        
    

            

        /** Isomethod of ada_label_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_label_decl_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_named_stmt_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_named_stmt_decl_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_number_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_number_decl_f_ids(
            Entity node
        );
            

        /** Isomethod of ada_number_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_number_decl_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_object_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_f_ids(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_f_has_aliased langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_f_has_aliased(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_f_has_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_f_has_constant(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_f_mode langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_f_mode(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_f_type_expr(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_f_default_expr(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_f_renaming_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_f_renaming_clause(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_p_private_part_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_p_private_part_decl(
            Entity node
        );
            

        /** Isomethod of ada_object_decl_p_public_part_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_object_decl_p_public_part_decl(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_package_renaming_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_renaming_decl_f_name(
            Entity node
        );
            

        /** Isomethod of ada_package_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_renaming_decl_f_renames(
            Entity node
        );
            

        /** Isomethod of ada_package_renaming_decl_p_renamed_package langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_renaming_decl_p_renamed_package(
            Entity node
        );
            

        /** Isomethod of ada_package_renaming_decl_p_final_renamed_package langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_package_renaming_decl_p_final_renamed_package(
            Entity node
        );

        
    

            

        /** Isomethod of ada_single_protected_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_single_protected_decl_f_name(
            Entity node
        );
            

        /** Isomethod of ada_single_protected_decl_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_single_protected_decl_f_interfaces(
            Entity node
        );
            

        /** Isomethod of ada_single_protected_decl_f_definition langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_single_protected_decl_f_definition(
            Entity node
        );

        
    

            

        /** Isomethod of ada_single_task_decl_f_task_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_single_task_decl_f_task_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_case_stmt_alternative_f_choices langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_stmt_alternative_f_choices(
            Entity node
        );
            

        /** Isomethod of ada_case_stmt_alternative_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_stmt_alternative_f_stmts(
            Entity node
        );

        
    

            

        /** Isomethod of ada_compilation_unit_f_prelude langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_compilation_unit_f_prelude(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_compilation_unit_f_body(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_f_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_compilation_unit_f_pragmas(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_syntactic_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native UnboundedTextTypeArray ada_compilation_unit_p_syntactic_fully_qualified_name(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_unit_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnitKind ada_compilation_unit_p_unit_kind(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_withed_units langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_compilation_unit_p_withed_units(
            boolean include_privates,
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_imported_units langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_compilation_unit_p_imported_units(
            boolean include_privates,
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_unit_dependencies langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_compilation_unit_p_unit_dependencies(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_compilation_unit_p_decl(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_is_preelaborable langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_compilation_unit_p_is_preelaborable(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_other_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_compilation_unit_p_other_part(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_has_restriction langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_compilation_unit_p_has_restriction(
            Symbol name,
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_all_config_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_compilation_unit_p_all_config_pragmas(
            Entity node
        );
            

        /** Isomethod of ada_compilation_unit_p_config_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_compilation_unit_p_config_pragmas(
            Symbol name,
            Entity node
        );

        
    

            

        /** Isomethod of ada_component_clause_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_clause_f_id(
            Entity node
        );
            

        /** Isomethod of ada_component_clause_f_position langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_clause_f_position(
            Entity node
        );
            

        /** Isomethod of ada_component_clause_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_clause_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_component_def_f_has_aliased langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_def_f_has_aliased(
            Entity node
        );
            

        /** Isomethod of ada_component_def_f_has_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_def_f_has_constant(
            Entity node
        );
            

        /** Isomethod of ada_component_def_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_component_def_f_type_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_constant_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_constant_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_composite_constraint_f_constraints langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_composite_constraint_f_constraints(
            Entity node
        );
            

        /** Isomethod of ada_composite_constraint_p_is_index_constraint langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_composite_constraint_p_is_index_constraint(
            Entity node
        );
            

        /** Isomethod of ada_composite_constraint_p_is_discriminant_constraint langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_composite_constraint_p_is_discriminant_constraint(
            Entity node
        );

        
    

            

        /** Isomethod of ada_delta_constraint_f_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_delta_constraint_f_digits(
            Entity node
        );
            

        /** Isomethod of ada_delta_constraint_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_delta_constraint_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_digits_constraint_f_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_digits_constraint_f_digits(
            Entity node
        );
            

        /** Isomethod of ada_digits_constraint_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_digits_constraint_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_range_constraint_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_range_constraint_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_declarative_part_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_declarative_part_f_decls(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_elsif_expr_part_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_elsif_expr_part_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of ada_elsif_expr_part_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_elsif_expr_part_f_then_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_elsif_stmt_part_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_elsif_stmt_part_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of ada_elsif_stmt_part_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_elsif_stmt_part_f_stmts(
            Entity node
        );

        
    

            

        /** Isomethod of ada_expr_p_expression_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_expr_p_expression_type(
            Entity node
        );
            

        /** Isomethod of ada_expr_p_expected_expression_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_expr_p_expected_expression_type(
            Entity node
        );
            

        /** Isomethod of ada_expr_p_is_dynamically_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_expr_p_is_dynamically_tagged(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_expr_p_is_dispatching_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_expr_p_is_dispatching_call(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_expr_p_is_static_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_expr_p_is_static_expr(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_expr_p_first_corresponding_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_expr_p_first_corresponding_decl(
            Entity node
        );
            

        /** Isomethod of ada_expr_p_eval_as_int langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native BigInteger ada_expr_p_eval_as_int(
            Entity node
        );
            

        /** Isomethod of ada_expr_p_eval_as_int_in_env langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native BigInteger ada_expr_p_eval_as_int_in_env(
            SubstitutionArray env,
            Entity node
        );
            

        /** Isomethod of ada_expr_p_eval_as_string langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_expr_p_eval_as_string(
            Entity node
        );
            

        /** Isomethod of ada_expr_p_eval_as_string_in_env langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_expr_p_eval_as_string_in_env(
            SubstitutionArray env,
            Entity node
        );
            

        /** Isomethod of ada_expr_p_matching_nodes langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_expr_p_matching_nodes(
            Entity node
        );

        
    

            

        /** Isomethod of ada_abstract_state_decl_expr_f_state_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_abstract_state_decl_expr_f_state_decl(
            Entity node
        );

        
    

            

        /** Isomethod of ada_allocator_f_subpool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_allocator_f_subpool(
            Entity node
        );
            

        /** Isomethod of ada_allocator_f_type_or_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_allocator_f_type_or_expr(
            Entity node
        );
            

        /** Isomethod of ada_allocator_p_get_allocated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_allocator_p_get_allocated_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_aggregate_f_ancestor_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_aggregate_f_ancestor_expr(
            Entity node
        );
            

        /** Isomethod of ada_base_aggregate_f_assocs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_aggregate_f_assocs(
            Entity node
        );
            

        /** Isomethod of ada_base_aggregate_p_aggregate_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native ParamActualArray ada_base_aggregate_p_aggregate_params(
            Entity node
        );
            

        /** Isomethod of ada_base_aggregate_p_is_subaggregate langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_base_aggregate_p_is_subaggregate(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_bin_op_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_bin_op_f_left(
            Entity node
        );
            

        /** Isomethod of ada_bin_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_bin_op_f_op(
            Entity node
        );
            

        /** Isomethod of ada_bin_op_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_bin_op_f_right(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_case_expr_alternative_f_choices langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_expr_alternative_f_choices(
            Entity node
        );
            

        /** Isomethod of ada_case_expr_alternative_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_expr_alternative_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_concat_op_f_first_operand langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_concat_op_f_first_operand(
            Entity node
        );
            

        /** Isomethod of ada_concat_op_f_other_operands langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_concat_op_f_other_operands(
            Entity node
        );
            

        /** Isomethod of ada_concat_op_p_operands langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_concat_op_p_operands(
            Entity node
        );

        
    

            

        /** Isomethod of ada_concat_operand_f_operator langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_concat_operand_f_operator(
            Entity node
        );
            

        /** Isomethod of ada_concat_operand_f_operand langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_concat_operand_f_operand(
            Entity node
        );

        
    

            

        /** Isomethod of ada_cond_expr_p_dependent_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_cond_expr_p_dependent_exprs(
            Entity node
        );

        
    

            

        /** Isomethod of ada_case_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_expr_f_expr(
            Entity node
        );
            

        /** Isomethod of ada_case_expr_f_cases langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_expr_f_cases(
            Entity node
        );

        
    

            

        /** Isomethod of ada_if_expr_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_expr_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of ada_if_expr_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_expr_f_then_expr(
            Entity node
        );
            

        /** Isomethod of ada_if_expr_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_expr_f_alternatives(
            Entity node
        );
            

        /** Isomethod of ada_if_expr_f_else_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_expr_f_else_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_contract_cases_f_contract_cases langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_contract_cases_f_contract_cases(
            Entity node
        );

        
    

            

        /** Isomethod of ada_decl_expr_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decl_expr_f_decls(
            Entity node
        );
            

        /** Isomethod of ada_decl_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decl_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_membership_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_membership_expr_f_expr(
            Entity node
        );
            

        /** Isomethod of ada_membership_expr_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_membership_expr_f_op(
            Entity node
        );
            

        /** Isomethod of ada_membership_expr_f_membership_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_membership_expr_f_membership_exprs(
            Entity node
        );

        
    

            

        /** Isomethod of ada_name_p_enclosing_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_name_p_enclosing_defining_name(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_defining langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_defining(
            Entity node
        );
            

        /** Isomethod of ada_name_p_name_is langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_name_is(
            Symbol sym,
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_direct_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_direct_call(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_access_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_access_call(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_call(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_dot_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_dot_call(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_failsafe_referenced_def_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native RefdDef ada_name_p_failsafe_referenced_def_name(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_referenced_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_name_p_referenced_defining_name(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_all_env_elements langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_name_p_all_env_elements(
            boolean seq,
            Entity seq_from,
            Entity node
        );
            

        /** Isomethod of ada_name_p_called_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_name_p_called_subp_spec(
            Entity node
        );
            

        /** Isomethod of ada_name_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_name_p_referenced_decl(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_failsafe_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native RefdDecl ada_name_p_failsafe_referenced_decl(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_name_designated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_name_p_name_designated_type(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_static_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_static_subtype(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_name_matches langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_name_matches(
            Entity n,
            Entity node
        );
            

        /** Isomethod of ada_name_p_relative_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_name_p_relative_name(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_operator_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_operator_name(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_write_reference langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_write_reference(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_static_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_static_call(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_name_p_as_symbol_array langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native UnboundedTextTypeArray ada_name_p_as_symbol_array(
            Entity node
        );
            

        /** Isomethod of ada_name_p_canonical_text langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Symbol ada_name_p_canonical_text(
            Entity node
        );
            

        /** Isomethod of ada_name_p_is_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_name_p_is_constant(
            Entity node
        );
            

        /** Isomethod of ada_name_p_call_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native ParamActualArray ada_name_p_call_params(
            Entity node
        );

        
    

            

        /** Isomethod of ada_attribute_ref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_attribute_ref_f_prefix(
            Entity node
        );
            

        /** Isomethod of ada_attribute_ref_f_attribute langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_attribute_ref_f_attribute(
            Entity node
        );
            

        /** Isomethod of ada_attribute_ref_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_attribute_ref_f_args(
            Entity node
        );

        
    

            

        /** Isomethod of ada_call_expr_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_call_expr_f_name(
            Entity node
        );
            

        /** Isomethod of ada_call_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_call_expr_f_suffix(
            Entity node
        );
            

        /** Isomethod of ada_call_expr_p_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native CallExprKind ada_call_expr_p_kind(
            Entity node
        );
            

        /** Isomethod of ada_call_expr_p_is_array_slice langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_call_expr_p_is_array_slice(
            Entity node
        );

        
    

            

        /** Isomethod of ada_defining_name_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_f_name(
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_canonical_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_defining_name_p_canonical_fully_qualified_name(
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_unique_identifying_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_defining_name_p_unique_identifying_name(
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_fully_qualified_name_array langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native UnboundedTextTypeArray ada_defining_name_p_fully_qualified_name_array(
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_defining_name_p_fully_qualified_name(
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_basic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_basic_decl(
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_find_refs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native RefResultArray ada_defining_name_p_find_refs(
            Entity root,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_find_all_references langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native RefResultArray ada_defining_name_p_find_all_references(
            AnalysisUnitArray units,
            boolean follow_renamings,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_find_all_calls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native RefResultArray ada_defining_name_p_find_all_calls(
            AnalysisUnitArray units,
            boolean follow_renamings,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_next_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_next_part(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_previous_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_previous_part(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_canonical_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_canonical_part(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_most_visible_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_most_visible_part(
            Entity origin,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_all_parts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_defining_name_p_all_parts(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_get_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Aspect ada_defining_name_p_get_aspect(
            Symbol name,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_has_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_defining_name_p_has_aspect(
            Symbol name,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_get_pragma langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_get_pragma(
            Symbol name,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_get_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_get_representation_clause(
            Symbol name,
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_get_at_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_defining_name_p_get_at_clause(
            boolean imprecise_fallback,
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_is_imported langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_defining_name_p_is_imported(
            Entity node
        );
            

        /** Isomethod of ada_defining_name_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_defining_name_p_is_ghost_code(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_discrete_subtype_name_f_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_discrete_subtype_name_f_subtype(
            Entity node
        );

        
    

            

        /** Isomethod of ada_dotted_name_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_dotted_name_f_prefix(
            Entity node
        );
            

        /** Isomethod of ada_dotted_name_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_dotted_name_f_suffix(
            Entity node
        );

        
    

            

        /** Isomethod of ada_end_name_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_end_name_f_name(
            Entity node
        );
            

        /** Isomethod of ada_end_name_p_basic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_end_name_p_basic_decl(
            Entity node
        );

        
    

            

        /** Isomethod of ada_explicit_deref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_explicit_deref_f_prefix(
            Entity node
        );

        
    

            

        /** Isomethod of ada_qual_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_qual_expr_f_prefix(
            Entity node
        );
            

        /** Isomethod of ada_qual_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_qual_expr_f_suffix(
            Entity node
        );

        
    

            

        /** Isomethod of ada_reduce_attribute_ref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_reduce_attribute_ref_f_prefix(
            Entity node
        );
            

        /** Isomethod of ada_reduce_attribute_ref_f_attribute langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_reduce_attribute_ref_f_attribute(
            Entity node
        );
            

        /** Isomethod of ada_reduce_attribute_ref_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_reduce_attribute_ref_f_args(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_char_literal_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Char ada_char_literal_p_denoted_value(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_string_literal_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String ada_string_literal_p_denoted_value(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_int_literal_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native BigInteger ada_int_literal_p_denoted_value(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_update_attribute_ref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_update_attribute_ref_f_prefix(
            Entity node
        );
            

        /** Isomethod of ada_update_attribute_ref_f_attribute langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_update_attribute_ref_f_attribute(
            Entity node
        );
            

        /** Isomethod of ada_update_attribute_ref_f_values langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_update_attribute_ref_f_values(
            Entity node
        );

        
    

            

        /** Isomethod of ada_paren_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_paren_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_quantified_expr_f_quantifier langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_quantified_expr_f_quantifier(
            Entity node
        );
            

        /** Isomethod of ada_quantified_expr_f_loop_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_quantified_expr_f_loop_spec(
            Entity node
        );
            

        /** Isomethod of ada_quantified_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_quantified_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_raise_expr_f_exception_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_raise_expr_f_exception_name(
            Entity node
        );
            

        /** Isomethod of ada_raise_expr_f_error_message langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_raise_expr_f_error_message(
            Entity node
        );

        
    

            

        /** Isomethod of ada_un_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_un_op_f_op(
            Entity node
        );
            

        /** Isomethod of ada_un_op_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_un_op_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_handled_stmts_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_handled_stmts_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_handled_stmts_f_exceptions langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_handled_stmts_f_exceptions(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_library_item_f_has_private langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_library_item_f_has_private(
            Entity node
        );
            

        /** Isomethod of ada_library_item_f_item langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_library_item_f_item(
            Entity node
        );

        
    

            

        /** Isomethod of ada_limited_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_limited_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_for_loop_spec_f_var_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_for_loop_spec_f_var_decl(
            Entity node
        );
            

        /** Isomethod of ada_for_loop_spec_f_loop_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_for_loop_spec_f_loop_type(
            Entity node
        );
            

        /** Isomethod of ada_for_loop_spec_f_has_reverse langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_for_loop_spec_f_has_reverse(
            Entity node
        );
            

        /** Isomethod of ada_for_loop_spec_f_iter_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_for_loop_spec_f_iter_expr(
            Entity node
        );
            

        /** Isomethod of ada_for_loop_spec_f_iter_filter langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_for_loop_spec_f_iter_filter(
            Entity node
        );

        
    

            

        /** Isomethod of ada_while_loop_spec_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_while_loop_spec_f_expr(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_multi_abstract_state_decl_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_multi_abstract_state_decl_f_decls(
            Entity node
        );

        
    

            

        /** Isomethod of ada_not_null_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_not_null_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_params_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_params_f_params(
            Entity node
        );

        
    

            

        /** Isomethod of ada_paren_abstract_state_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_paren_abstract_state_decl_f_decl(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_pp_elsif_directive_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pp_elsif_directive_f_expr(
            Entity node
        );
            

        /** Isomethod of ada_pp_elsif_directive_f_then_kw langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pp_elsif_directive_f_then_kw(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_pp_if_directive_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pp_if_directive_f_expr(
            Entity node
        );
            

        /** Isomethod of ada_pp_if_directive_f_then_kw langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pp_if_directive_f_then_kw(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_pragma_node_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pragma_node_f_id(
            Entity node
        );
            

        /** Isomethod of ada_pragma_node_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_pragma_node_f_args(
            Entity node
        );
            

        /** Isomethod of ada_pragma_node_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_pragma_node_p_is_ghost_code(
            Entity node
        );
            

        /** Isomethod of ada_pragma_node_p_associated_entities langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AdaNodeArray ada_pragma_node_p_associated_entities(
            Entity node
        );

        
    

            

        /** Isomethod of ada_private_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_private_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_protected_def_f_public_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_def_f_public_part(
            Entity node
        );
            

        /** Isomethod of ada_protected_def_f_private_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_def_f_private_part(
            Entity node
        );
            

        /** Isomethod of ada_protected_def_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_protected_def_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_protected_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_protected_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_range_spec_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_range_spec_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_renaming_clause_f_renamed_object langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_renaming_clause_f_renamed_object(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_reverse_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_reverse_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_select_when_part_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_select_when_part_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of ada_select_when_part_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_select_when_part_f_stmts(
            Entity node
        );

        
    

            

        /** Isomethod of ada_stmt_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_stmt_p_is_ghost_code(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_accept_stmt_f_body_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_accept_stmt_f_body_decl(
            Entity node
        );
            

        /** Isomethod of ada_accept_stmt_f_entry_index_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_accept_stmt_f_entry_index_expr(
            Entity node
        );
            

        /** Isomethod of ada_accept_stmt_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_accept_stmt_f_params(
            Entity node
        );
            

        /** Isomethod of ada_accept_stmt_p_corresponding_entry langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_accept_stmt_p_corresponding_entry(
            Entity origin,
            Entity node
        );

        
    

            

        /** Isomethod of ada_accept_stmt_with_stmts_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_accept_stmt_with_stmts_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_accept_stmt_with_stmts_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_accept_stmt_with_stmts_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_base_loop_stmt_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_loop_stmt_f_spec(
            Entity node
        );
            

        /** Isomethod of ada_base_loop_stmt_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_loop_stmt_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_base_loop_stmt_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_base_loop_stmt_f_end_name(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_begin_block_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_begin_block_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_begin_block_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_begin_block_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_decl_block_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decl_block_f_decls(
            Entity node
        );
            

        /** Isomethod of ada_decl_block_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decl_block_f_stmts(
            Entity node
        );
            

        /** Isomethod of ada_decl_block_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decl_block_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_case_stmt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_stmt_f_expr(
            Entity node
        );
            

        /** Isomethod of ada_case_stmt_f_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_stmt_f_pragmas(
            Entity node
        );
            

        /** Isomethod of ada_case_stmt_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_case_stmt_f_alternatives(
            Entity node
        );

        
    

            

        /** Isomethod of ada_extended_return_stmt_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_extended_return_stmt_f_decl(
            Entity node
        );
            

        /** Isomethod of ada_extended_return_stmt_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_extended_return_stmt_f_stmts(
            Entity node
        );

        
    

            

        /** Isomethod of ada_if_stmt_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_stmt_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of ada_if_stmt_f_then_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_stmt_f_then_stmts(
            Entity node
        );
            

        /** Isomethod of ada_if_stmt_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_stmt_f_alternatives(
            Entity node
        );
            

        /** Isomethod of ada_if_stmt_f_else_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_if_stmt_f_else_stmts(
            Entity node
        );

        
    

            

        /** Isomethod of ada_named_stmt_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_named_stmt_f_decl(
            Entity node
        );
            

        /** Isomethod of ada_named_stmt_f_stmt langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_named_stmt_f_stmt(
            Entity node
        );

        
    

            

        /** Isomethod of ada_select_stmt_f_guards langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_select_stmt_f_guards(
            Entity node
        );
            

        /** Isomethod of ada_select_stmt_f_else_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_select_stmt_f_else_stmts(
            Entity node
        );
            

        /** Isomethod of ada_select_stmt_f_abort_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_select_stmt_f_abort_stmts(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_abort_stmt_f_names langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_abort_stmt_f_names(
            Entity node
        );

        
    

            

        /** Isomethod of ada_assign_stmt_f_dest langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_assign_stmt_f_dest(
            Entity node
        );
            

        /** Isomethod of ada_assign_stmt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_assign_stmt_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_call_stmt_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_call_stmt_f_call(
            Entity node
        );

        
    

            

        /** Isomethod of ada_delay_stmt_f_has_until langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_delay_stmt_f_has_until(
            Entity node
        );
            

        /** Isomethod of ada_delay_stmt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_delay_stmt_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_exit_stmt_f_loop_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_exit_stmt_f_loop_name(
            Entity node
        );
            

        /** Isomethod of ada_exit_stmt_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_exit_stmt_f_cond_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_goto_stmt_f_label_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_goto_stmt_f_label_name(
            Entity node
        );

        
    

            

        /** Isomethod of ada_label_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_label_f_decl(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_raise_stmt_f_exception_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_raise_stmt_f_exception_name(
            Entity node
        );
            

        /** Isomethod of ada_raise_stmt_f_error_message langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_raise_stmt_f_error_message(
            Entity node
        );

        
    

            

        /** Isomethod of ada_requeue_stmt_f_call_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_requeue_stmt_f_call_name(
            Entity node
        );
            

        /** Isomethod of ada_requeue_stmt_f_has_abort langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_requeue_stmt_f_has_abort(
            Entity node
        );

        
    

            

        /** Isomethod of ada_return_stmt_f_return_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_return_stmt_f_return_expr(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_subunit_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subunit_f_name(
            Entity node
        );
            

        /** Isomethod of ada_subunit_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subunit_f_body(
            Entity node
        );
            

        /** Isomethod of ada_subunit_p_body_root langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subunit_p_body_root(
            Entity node
        );

        
    

            

        /** Isomethod of ada_synchronized_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_synchronized_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_tagged_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_tagged_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_task_def_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_def_f_interfaces(
            Entity node
        );
            

        /** Isomethod of ada_task_def_f_public_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_def_f_public_part(
            Entity node
        );
            

        /** Isomethod of ada_task_def_f_private_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_def_f_private_part(
            Entity node
        );
            

        /** Isomethod of ada_task_def_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_task_def_f_end_name(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_access_def_f_has_not_null langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_access_def_f_has_not_null(
            Entity node
        );

        
    

            

        /** Isomethod of ada_access_to_subp_def_f_has_protected langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_access_to_subp_def_f_has_protected(
            Entity node
        );
            

        /** Isomethod of ada_access_to_subp_def_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_access_to_subp_def_f_subp_spec(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_anonymous_type_access_def_f_type_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_anonymous_type_access_def_f_type_decl(
            Entity node
        );

        
    

            

        /** Isomethod of ada_type_access_def_f_has_all langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_access_def_f_has_all(
            Entity node
        );
            

        /** Isomethod of ada_type_access_def_f_has_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_access_def_f_has_constant(
            Entity node
        );
            

        /** Isomethod of ada_type_access_def_f_subtype_indication langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_access_def_f_subtype_indication(
            Entity node
        );

        
    

            

        /** Isomethod of ada_array_type_def_f_indices langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_array_type_def_f_indices(
            Entity node
        );
            

        /** Isomethod of ada_array_type_def_f_component_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_array_type_def_f_component_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_derived_type_def_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_derived_type_def_f_has_abstract(
            Entity node
        );
            

        /** Isomethod of ada_derived_type_def_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_derived_type_def_f_has_limited(
            Entity node
        );
            

        /** Isomethod of ada_derived_type_def_f_has_synchronized langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_derived_type_def_f_has_synchronized(
            Entity node
        );
            

        /** Isomethod of ada_derived_type_def_f_subtype_indication langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_derived_type_def_f_subtype_indication(
            Entity node
        );
            

        /** Isomethod of ada_derived_type_def_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_derived_type_def_f_interfaces(
            Entity node
        );
            

        /** Isomethod of ada_derived_type_def_f_record_extension langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_derived_type_def_f_record_extension(
            Entity node
        );
            

        /** Isomethod of ada_derived_type_def_f_has_with_private langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_derived_type_def_f_has_with_private(
            Entity node
        );

        
    

            

        /** Isomethod of ada_enum_type_def_f_enum_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_enum_type_def_f_enum_literals(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_interface_type_def_f_interface_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_interface_type_def_f_interface_kind(
            Entity node
        );
            

        /** Isomethod of ada_interface_type_def_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_interface_type_def_f_interfaces(
            Entity node
        );

        
    

            

        /** Isomethod of ada_mod_int_type_def_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_mod_int_type_def_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of ada_private_type_def_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_private_type_def_f_has_abstract(
            Entity node
        );
            

        /** Isomethod of ada_private_type_def_f_has_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_private_type_def_f_has_tagged(
            Entity node
        );
            

        /** Isomethod of ada_private_type_def_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_private_type_def_f_has_limited(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_decimal_fixed_point_def_f_delta langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decimal_fixed_point_def_f_delta(
            Entity node
        );
            

        /** Isomethod of ada_decimal_fixed_point_def_f_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decimal_fixed_point_def_f_digits(
            Entity node
        );
            

        /** Isomethod of ada_decimal_fixed_point_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_decimal_fixed_point_def_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_floating_point_def_f_num_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_floating_point_def_f_num_digits(
            Entity node
        );
            

        /** Isomethod of ada_floating_point_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_floating_point_def_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_ordinary_fixed_point_def_f_delta langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ordinary_fixed_point_def_f_delta(
            Entity node
        );
            

        /** Isomethod of ada_ordinary_fixed_point_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_ordinary_fixed_point_def_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_record_type_def_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_record_type_def_f_has_abstract(
            Entity node
        );
            

        /** Isomethod of ada_record_type_def_f_has_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_record_type_def_f_has_tagged(
            Entity node
        );
            

        /** Isomethod of ada_record_type_def_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_record_type_def_f_has_limited(
            Entity node
        );
            

        /** Isomethod of ada_record_type_def_f_record_def langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_record_type_def_f_record_def(
            Entity node
        );

        
    

            

        /** Isomethod of ada_signed_int_type_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_signed_int_type_def_f_range(
            Entity node
        );

        
    

            

        /** Isomethod of ada_type_expr_p_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_expr_p_type_name(
            Entity node
        );
            

        /** Isomethod of ada_type_expr_p_designated_type_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_expr_p_designated_type_decl(
            Entity node
        );
            

        /** Isomethod of ada_type_expr_p_designated_type_decl_from langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_type_expr_p_designated_type_decl_from(
            Entity origin_node,
            Entity node
        );

        
    

            

        /** Isomethod of ada_anonymous_type_f_type_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_anonymous_type_f_type_decl(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of ada_subtype_indication_f_has_not_null langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subtype_indication_f_has_not_null(
            Entity node
        );
            

        /** Isomethod of ada_subtype_indication_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subtype_indication_f_name(
            Entity node
        );
            

        /** Isomethod of ada_subtype_indication_f_constraint langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_subtype_indication_f_constraint(
            Entity node
        );
            

        /** Isomethod of ada_subtype_indication_p_subtype_constraints langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native ParamActualArray ada_subtype_indication_p_subtype_constraints(
            Entity node
        );
            

        /** Isomethod of ada_subtype_indication_p_is_static_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_subtype_indication_p_is_static_subtype(
            boolean imprecise_fallback,
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_synthetic_type_expr_f_target_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_synthetic_type_expr_f_target_type(
            Entity node
        );

        
    

            

        /** Isomethod of ada_unconstrained_array_index_f_subtype_indication langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_unconstrained_array_index_f_subtype_indication(
            Entity node
        );

        
    

            

        /** Isomethod of ada_until_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_until_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_use_package_clause_f_packages langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_use_package_clause_f_packages(
            Entity node
        );

        
    

            

        /** Isomethod of ada_use_type_clause_f_has_all langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_use_type_clause_f_has_all(
            Entity node
        );
            

        /** Isomethod of ada_use_type_clause_f_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_use_type_clause_f_types(
            Entity node
        );

        
    

            

        /** Isomethod of ada_value_sequence_f_iter_assoc langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_value_sequence_f_iter_assoc(
            Entity node
        );

        
    

            

        /** Isomethod of ada_variant_f_choices langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_variant_f_choices(
            Entity node
        );
            

        /** Isomethod of ada_variant_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_variant_f_components(
            Entity node
        );

        
    

            

        /** Isomethod of ada_variant_part_f_discr_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_variant_part_f_discr_name(
            Entity node
        );
            

        /** Isomethod of ada_variant_part_f_variant langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_variant_part_f_variant(
            Entity node
        );

        
    

            

        /** Isomethod of ada_with_clause_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_with_clause_f_has_limited(
            Entity node
        );
            

        /** Isomethod of ada_with_clause_f_has_private langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_with_clause_f_has_private(
            Entity node
        );
            

        /** Isomethod of ada_with_clause_f_packages langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity ada_with_clause_f_packages(
            Entity node
        );

        
    

            

        /** Isomethod of ada_with_private_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean ada_with_private_p_as_bool(
            Entity node
        );

        
    


        
    



    }


    // ==========
    // Defining the Native-Image bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains the directives for the shared lib loading */
    public static final class LibDirectives implements CContext.Directives {
        @Override
        public List<String> getHeaderFiles() {
            List<String> res = new ArrayList<>();
            res.add("<libadalang.h>");
            res.add("<stdlib.h>");
            return res;
        }

        @Override
        public List<String> getLibraries() {
            List<String> res = new ArrayList<>();
            res.add("adalang");
            return res;
        }
    }

    // ===== Language specific structures =====

    
      


    // ===== Native structures =====

    /** Project manager native is just a pointer. */
    public interface ProjectManagerNative extends Pointer {}

    /** Structure for a string array. */
    @CContext(LibDirectives.class)
    @CStruct("ada_string_array_ptr_struct")
    public interface StringArrayNative extends PointerBase {
        @CField("length") public int get_length();
        @CField("length") public void set_length(int length);

        @CField("c_ptr") public CCharPointerPointer get_c_ptr();
        @CField("c_ptr") public void set_c_ptr(CCharPointerPointer c_ptr);
    }

    /** Structure for a scenrio variable. */
    @CContext(LibDirectives.class)
    @CStruct("ada_gpr_project_scenario_variable")
    public interface ScenarioVariableNative extends PointerBase {
        @CField("name") public CCharPointer get_name();
        @CField("name") public void set_name(CCharPointer name);

        @CField("value") public CCharPointer get_value();
        @CField("value") public void set_value(CCharPointer value);
    }



    // ===== Constant structures =====

    /** The structure for the langkit exceptions */
    @CContext(LibDirectives.class)
    @CStruct("ada_exception")
    public interface LangkitExceptionNative extends PointerBase {
        @CField("kind") public int get_kind();
        @CField("kind") public void set_kind(
            int kind
        );

        @CField("information") public CCharPointer get_information();
        @CField("information") public void set_information(
            CCharPointer information
        );
    }

    /** The big integers are just pointers */
    public interface BigIntegerNative extends Pointer {}

    /** The structure for the symbols */
    @CContext(LibDirectives.class)
    @CStruct("ada_symbol_type")
    public interface SymbolNative extends PointerBase {
        @CField("data") public VoidPointer get_data();
        @CField("data") public void set_data(
            VoidPointer data
        );

        @CField("bounds") public VoidPointer get_bounds();
        @CField("bounds") public void set_bounds(
            VoidPointer bounds
        );
    }

    /** The string wrappers are just pointers */
    public interface StringNative extends Pointer {}

    /** The structure for the text */
    @CContext(LibDirectives.class)
    @CStruct("ada_text")
    public interface TextNative extends PointerBase {
        @CField("chars") public CIntPointer get_chars();
        @CField("chars") public void set_chars(
            CIntPointer chars
        );

        @CField("length") public long get_length();
        @CField("length") public void set_length(
            long length
        );

        @CField("is_allocated") public int get_is_allocated();
        @CField("is_allocated") public void set_is_allocated(
            int is_allocated
        );
    }

    /** The structure for the source locations */
    @CContext(LibDirectives.class)
    @CStruct("ada_source_location")
    public interface SourceLocationNative extends PointerBase {
        @CField("line") public int get_line();
        @CField("line") public void set_line(
            int line
        );

        @CField("column") public short get_column();
        @CField("column") public void set_column(
            short column
        );
    }

    /** The structure for the source location ranges */
    @CContext(LibDirectives.class)
    @CStruct("ada_source_location_range")
    public interface SourceLocationRangeNative extends PointerBase {
        @CField("start.line") public int get_start_line();
        @CField("start.line") public void set_start_line(
            int start_line
        );

        @CField("start.column") public short get_start_column();
        @CField("start.column") public void set_start_column(
            short start_column
        );

        @CField("end.line") public int get_end_line();
        @CField("end.line") public void set_end_line(
            int end_line
        );

        @CField("end.column") public short get_end_column();
        @CField("end.column") public void set_end_column(
            short end_column
        );
    }

    /** The structure for the diagnostic */
    @CContext(LibDirectives.class)
    @CStruct("ada_diagnostic")
    public interface DiagnosticNative extends PointerBase {
        @CField("sloc_range.start.line") public int get_start_line();
        @CField("sloc_range.start.line") public void set_start_line(
            int start_line
        );

        @CField("sloc_range.start.column") public short get_start_column();
        @CField("sloc_range.start.column") public void set_start_column(
            short start_column
        );

        @CField("sloc_range.end.line") public int get_end_line();
        @CField("sloc_range.end.line") public void set_end_line(
            int end_line
        );

        @CField("sloc_range.end.column") public short get_end_column();
        @CField("sloc_range.end.column") public void set_end_column(
            short end_column
        );

        @CField("message.chars") public CIntPointer get_message_chars();
        @CField("message.chars") public void set_message_chars(
            CIntPointer chars
        );

        @CField("message.length") public long get_message_length();
        @CField("message.length") public void set_message_length(
            long length
        );

        @CField("message.is_allocated") public int get_message_is_allocated();
        @CField("message.is_allocated") public void set_message_is_allocated(
            int is_allocated
        );
    }


    /** The file reader is just a pointer */
    public interface FileReaderNative extends Pointer {}

    /** The unit provider is just a pointer */
    public interface UnitProviderNative extends Pointer {}

    /** The event handler is just a pointer */
    public interface EventHandlerNative extends Pointer {}

    /** The event handler unit requested callback type */
    public interface UnitRequestedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            TextNative name,
            AnalysisUnitNative from,
            boolean found,
            boolean is_not_found_error
        );
    }

    /** The event handler unit parsed callback type */
    public interface UnitParsedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            AnalysisUnitNative unit,
            boolean reparsed
        );
    }

    /** Anonymous structure for the token data handler */
    @RawStructure
    public interface TokenDataHandlerNative extends PointerBase {
        @RawField public long version();
    }

    /** The structure representing a token */
    @CContext(LibDirectives.class)
    @CStruct("ada_token")
    public interface TokenNative extends PointerBase {
        @CField("context") public AnalysisContextNative get_context();
        @CField("context") public void set_context(
            AnalysisContextNative context
        );

        @CField("token_data") public TokenDataHandlerNative get_data();
        @CField("token_data") public void set_data(
            TokenDataHandlerNative data
        );

        @CField("token_index") public int get_token_index();
        @CField("token_index") public void set_token_index(
            int token_index
        );

        @CField("trivia_index") public int get_trivia_index();
        @CField("trivia_index") public void set_trivia_index(
            int trivia_index
        );
    }

    /** Anonymous strucutre for analysis context */
    @RawStructure
    public interface AnalysisContextNative extends PointerBase {
        @RawField public long serial_number();
    }

    /** Anonymous strucutre for analysis unit */
    @RawStructure
    public interface AnalysisUnitNative extends PointerBase {
        @RawField public long version_number();
    }

    // ===== Generated structures =====

        
    
    

    /** The structure for the langkit ada_internal_metadata */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_metadata")
    public interface MetadataNative extends PointerBase {
        @CField("dottable_subp")
        public byte
        get_dottable_subp();

        @CField("dottable_subp")
        public void
        set_dottable_subp(
            byte val
        );

        @CFieldAddress("dottable_subp")
        public <T extends PointerBase> T address_dottable_subp();
        @CField("primitive")
        public Pointer
        get_primitive();

        @CField("primitive")
        public void
        set_primitive(
            Pointer val
        );

        @CFieldAddress("primitive")
        public <T extends PointerBase> T address_primitive();
        @CField("primitive_real_type")
        public Pointer
        get_primitive_real_type();

        @CField("primitive_real_type")
        public void
        set_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("primitive_real_type")
        public <T extends PointerBase> T address_primitive_real_type();
    }

        
    
    

    /** The structure for the langkit ada_internal_entity_info */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_entity_info")
    public interface EntityInfoNative extends PointerBase {
        @CField("md.dottable_subp")
        public byte
        get_md_dottable_subp();

        @CField("md.dottable_subp")
        public void
        set_md_dottable_subp(
            byte val
        );

        @CFieldAddress("md.dottable_subp")
        public <T extends PointerBase> T address_md_dottable_subp();
        @CField("md.primitive")
        public Pointer
        get_md_primitive();

        @CField("md.primitive")
        public void
        set_md_primitive(
            Pointer val
        );

        @CFieldAddress("md.primitive")
        public <T extends PointerBase> T address_md_primitive();
        @CField("md.primitive_real_type")
        public Pointer
        get_md_primitive_real_type();

        @CField("md.primitive_real_type")
        public void
        set_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("md.primitive_real_type")
        public <T extends PointerBase> T address_md_primitive_real_type();
        @CField("rebindings")
        public Pointer
        get_rebindings();

        @CField("rebindings")
        public void
        set_rebindings(
            Pointer val
        );

        @CFieldAddress("rebindings")
        public <T extends PointerBase> T address_rebindings();
        @CField("from_rebound")
        public byte
        get_from_rebound();

        @CField("from_rebound")
        public void
        set_from_rebound(
            byte val
        );

        @CFieldAddress("from_rebound")
        public <T extends PointerBase> T address_from_rebound();
    }

    
    

    /** The structure for the langkit ada_base_entity */
    @CContext(LibDirectives.class)
    @CStruct("ada_base_entity")
    public interface EntityNative extends PointerBase {
        @CField("node")
        public Pointer
        get_node();

        @CField("node")
        public void
        set_node(
            Pointer val
        );

        @CFieldAddress("node")
        public <T extends PointerBase> T address_node();
        @CField("info.md.dottable_subp")
        public byte
        get_info_md_dottable_subp();

        @CField("info.md.dottable_subp")
        public void
        set_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("info.md.dottable_subp")
        public <T extends PointerBase> T address_info_md_dottable_subp();
        @CField("info.md.primitive")
        public Pointer
        get_info_md_primitive();

        @CField("info.md.primitive")
        public void
        set_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("info.md.primitive")
        public <T extends PointerBase> T address_info_md_primitive();
        @CField("info.md.primitive_real_type")
        public Pointer
        get_info_md_primitive_real_type();

        @CField("info.md.primitive_real_type")
        public void
        set_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("info.md.primitive_real_type")
        public <T extends PointerBase> T address_info_md_primitive_real_type();
        @CField("info.rebindings")
        public Pointer
        get_info_rebindings();

        @CField("info.rebindings")
        public void
        set_info_rebindings(
            Pointer val
        );

        @CFieldAddress("info.rebindings")
        public <T extends PointerBase> T address_info_rebindings();
        @CField("info.from_rebound")
        public byte
        get_info_from_rebound();

        @CField("info.from_rebound")
        public void
        set_info_from_rebound(
            byte val
        );

        @CFieldAddress("info.from_rebound")
        public <T extends PointerBase> T address_info_from_rebound();
    }

        
    
    

    /** The structure for the langkit ada_internal_aspect */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_aspect")
    public interface AspectNative extends PointerBase {
        @CField("exists")
        public byte
        get_exists();

        @CField("exists")
        public void
        set_exists(
            byte val
        );

        @CFieldAddress("exists")
        public <T extends PointerBase> T address_exists();
        @CField("node.node")
        public Pointer
        get_node_node();

        @CField("node.node")
        public void
        set_node_node(
            Pointer val
        );

        @CFieldAddress("node.node")
        public <T extends PointerBase> T address_node_node();
        @CField("node.info.md.dottable_subp")
        public byte
        get_node_info_md_dottable_subp();

        @CField("node.info.md.dottable_subp")
        public void
        set_node_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("node.info.md.dottable_subp")
        public <T extends PointerBase> T address_node_info_md_dottable_subp();
        @CField("node.info.md.primitive")
        public Pointer
        get_node_info_md_primitive();

        @CField("node.info.md.primitive")
        public void
        set_node_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("node.info.md.primitive")
        public <T extends PointerBase> T address_node_info_md_primitive();
        @CField("node.info.md.primitive_real_type")
        public Pointer
        get_node_info_md_primitive_real_type();

        @CField("node.info.md.primitive_real_type")
        public void
        set_node_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("node.info.md.primitive_real_type")
        public <T extends PointerBase> T address_node_info_md_primitive_real_type();
        @CField("node.info.rebindings")
        public Pointer
        get_node_info_rebindings();

        @CField("node.info.rebindings")
        public void
        set_node_info_rebindings(
            Pointer val
        );

        @CFieldAddress("node.info.rebindings")
        public <T extends PointerBase> T address_node_info_rebindings();
        @CField("node.info.from_rebound")
        public byte
        get_node_info_from_rebound();

        @CField("node.info.from_rebound")
        public void
        set_node_info_from_rebound(
            byte val
        );

        @CFieldAddress("node.info.from_rebound")
        public <T extends PointerBase> T address_node_info_from_rebound();
        @CField("value.node")
        public Pointer
        get_value_node();

        @CField("value.node")
        public void
        set_value_node(
            Pointer val
        );

        @CFieldAddress("value.node")
        public <T extends PointerBase> T address_value_node();
        @CField("value.info.md.dottable_subp")
        public byte
        get_value_info_md_dottable_subp();

        @CField("value.info.md.dottable_subp")
        public void
        set_value_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("value.info.md.dottable_subp")
        public <T extends PointerBase> T address_value_info_md_dottable_subp();
        @CField("value.info.md.primitive")
        public Pointer
        get_value_info_md_primitive();

        @CField("value.info.md.primitive")
        public void
        set_value_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("value.info.md.primitive")
        public <T extends PointerBase> T address_value_info_md_primitive();
        @CField("value.info.md.primitive_real_type")
        public Pointer
        get_value_info_md_primitive_real_type();

        @CField("value.info.md.primitive_real_type")
        public void
        set_value_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("value.info.md.primitive_real_type")
        public <T extends PointerBase> T address_value_info_md_primitive_real_type();
        @CField("value.info.rebindings")
        public Pointer
        get_value_info_rebindings();

        @CField("value.info.rebindings")
        public void
        set_value_info_rebindings(
            Pointer val
        );

        @CFieldAddress("value.info.rebindings")
        public <T extends PointerBase> T address_value_info_rebindings();
        @CField("value.info.from_rebound")
        public byte
        get_value_info_from_rebound();

        @CField("value.info.from_rebound")
        public void
        set_value_info_from_rebound(
            byte val
        );

        @CFieldAddress("value.info.from_rebound")
        public <T extends PointerBase> T address_value_info_from_rebound();
        @CField("inherited")
        public byte
        get_inherited();

        @CField("inherited")
        public void
        set_inherited(
            byte val
        );

        @CFieldAddress("inherited")
        public <T extends PointerBase> T address_inherited();
    }

        
    
    

    /** The structure for the langkit ada_internal_completion_item */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_completion_item")
    public interface CompletionItemNative extends PointerBase {
        @CField("decl.node")
        public Pointer
        get_decl_node();

        @CField("decl.node")
        public void
        set_decl_node(
            Pointer val
        );

        @CFieldAddress("decl.node")
        public <T extends PointerBase> T address_decl_node();
        @CField("decl.info.md.dottable_subp")
        public byte
        get_decl_info_md_dottable_subp();

        @CField("decl.info.md.dottable_subp")
        public void
        set_decl_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("decl.info.md.dottable_subp")
        public <T extends PointerBase> T address_decl_info_md_dottable_subp();
        @CField("decl.info.md.primitive")
        public Pointer
        get_decl_info_md_primitive();

        @CField("decl.info.md.primitive")
        public void
        set_decl_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("decl.info.md.primitive")
        public <T extends PointerBase> T address_decl_info_md_primitive();
        @CField("decl.info.md.primitive_real_type")
        public Pointer
        get_decl_info_md_primitive_real_type();

        @CField("decl.info.md.primitive_real_type")
        public void
        set_decl_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("decl.info.md.primitive_real_type")
        public <T extends PointerBase> T address_decl_info_md_primitive_real_type();
        @CField("decl.info.rebindings")
        public Pointer
        get_decl_info_rebindings();

        @CField("decl.info.rebindings")
        public void
        set_decl_info_rebindings(
            Pointer val
        );

        @CFieldAddress("decl.info.rebindings")
        public <T extends PointerBase> T address_decl_info_rebindings();
        @CField("decl.info.from_rebound")
        public byte
        get_decl_info_from_rebound();

        @CField("decl.info.from_rebound")
        public void
        set_decl_info_from_rebound(
            byte val
        );

        @CFieldAddress("decl.info.from_rebound")
        public <T extends PointerBase> T address_decl_info_from_rebound();
        @CField("is_dot_call")
        public byte
        get_is_dot_call();

        @CField("is_dot_call")
        public void
        set_is_dot_call(
            byte val
        );

        @CFieldAddress("is_dot_call")
        public <T extends PointerBase> T address_is_dot_call();
        @CField("is_visible")
        public byte
        get_is_visible();

        @CField("is_visible")
        public void
        set_is_visible(
            byte val
        );

        @CFieldAddress("is_visible")
        public <T extends PointerBase> T address_is_visible();
        @CField("weight")
        public int
        get_weight();

        @CField("weight")
        public void
        set_weight(
            int val
        );

        @CFieldAddress("weight")
        public <T extends PointerBase> T address_weight();
    }

        
        
    
    

    /** The structure for the langkit ada_internal_discrete_range */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_discrete_range")
    public interface DiscreteRangeNative extends PointerBase {
        @CField("low_bound.node")
        public Pointer
        get_low_bound_node();

        @CField("low_bound.node")
        public void
        set_low_bound_node(
            Pointer val
        );

        @CFieldAddress("low_bound.node")
        public <T extends PointerBase> T address_low_bound_node();
        @CField("low_bound.info.md.dottable_subp")
        public byte
        get_low_bound_info_md_dottable_subp();

        @CField("low_bound.info.md.dottable_subp")
        public void
        set_low_bound_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("low_bound.info.md.dottable_subp")
        public <T extends PointerBase> T address_low_bound_info_md_dottable_subp();
        @CField("low_bound.info.md.primitive")
        public Pointer
        get_low_bound_info_md_primitive();

        @CField("low_bound.info.md.primitive")
        public void
        set_low_bound_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("low_bound.info.md.primitive")
        public <T extends PointerBase> T address_low_bound_info_md_primitive();
        @CField("low_bound.info.md.primitive_real_type")
        public Pointer
        get_low_bound_info_md_primitive_real_type();

        @CField("low_bound.info.md.primitive_real_type")
        public void
        set_low_bound_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("low_bound.info.md.primitive_real_type")
        public <T extends PointerBase> T address_low_bound_info_md_primitive_real_type();
        @CField("low_bound.info.rebindings")
        public Pointer
        get_low_bound_info_rebindings();

        @CField("low_bound.info.rebindings")
        public void
        set_low_bound_info_rebindings(
            Pointer val
        );

        @CFieldAddress("low_bound.info.rebindings")
        public <T extends PointerBase> T address_low_bound_info_rebindings();
        @CField("low_bound.info.from_rebound")
        public byte
        get_low_bound_info_from_rebound();

        @CField("low_bound.info.from_rebound")
        public void
        set_low_bound_info_from_rebound(
            byte val
        );

        @CFieldAddress("low_bound.info.from_rebound")
        public <T extends PointerBase> T address_low_bound_info_from_rebound();
        @CField("high_bound.node")
        public Pointer
        get_high_bound_node();

        @CField("high_bound.node")
        public void
        set_high_bound_node(
            Pointer val
        );

        @CFieldAddress("high_bound.node")
        public <T extends PointerBase> T address_high_bound_node();
        @CField("high_bound.info.md.dottable_subp")
        public byte
        get_high_bound_info_md_dottable_subp();

        @CField("high_bound.info.md.dottable_subp")
        public void
        set_high_bound_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("high_bound.info.md.dottable_subp")
        public <T extends PointerBase> T address_high_bound_info_md_dottable_subp();
        @CField("high_bound.info.md.primitive")
        public Pointer
        get_high_bound_info_md_primitive();

        @CField("high_bound.info.md.primitive")
        public void
        set_high_bound_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("high_bound.info.md.primitive")
        public <T extends PointerBase> T address_high_bound_info_md_primitive();
        @CField("high_bound.info.md.primitive_real_type")
        public Pointer
        get_high_bound_info_md_primitive_real_type();

        @CField("high_bound.info.md.primitive_real_type")
        public void
        set_high_bound_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("high_bound.info.md.primitive_real_type")
        public <T extends PointerBase> T address_high_bound_info_md_primitive_real_type();
        @CField("high_bound.info.rebindings")
        public Pointer
        get_high_bound_info_rebindings();

        @CField("high_bound.info.rebindings")
        public void
        set_high_bound_info_rebindings(
            Pointer val
        );

        @CFieldAddress("high_bound.info.rebindings")
        public <T extends PointerBase> T address_high_bound_info_rebindings();
        @CField("high_bound.info.from_rebound")
        public byte
        get_high_bound_info_from_rebound();

        @CField("high_bound.info.from_rebound")
        public void
        set_high_bound_info_from_rebound(
            byte val
        );

        @CFieldAddress("high_bound.info.from_rebound")
        public <T extends PointerBase> T address_high_bound_info_from_rebound();
    }

        
    
    

    /** The structure for the langkit ada_internal_discriminant_values */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_discriminant_values")
    public interface DiscriminantValuesNative extends PointerBase {
        @CField("discriminant.node")
        public Pointer
        get_discriminant_node();

        @CField("discriminant.node")
        public void
        set_discriminant_node(
            Pointer val
        );

        @CFieldAddress("discriminant.node")
        public <T extends PointerBase> T address_discriminant_node();
        @CField("discriminant.info.md.dottable_subp")
        public byte
        get_discriminant_info_md_dottable_subp();

        @CField("discriminant.info.md.dottable_subp")
        public void
        set_discriminant_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("discriminant.info.md.dottable_subp")
        public <T extends PointerBase> T address_discriminant_info_md_dottable_subp();
        @CField("discriminant.info.md.primitive")
        public Pointer
        get_discriminant_info_md_primitive();

        @CField("discriminant.info.md.primitive")
        public void
        set_discriminant_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("discriminant.info.md.primitive")
        public <T extends PointerBase> T address_discriminant_info_md_primitive();
        @CField("discriminant.info.md.primitive_real_type")
        public Pointer
        get_discriminant_info_md_primitive_real_type();

        @CField("discriminant.info.md.primitive_real_type")
        public void
        set_discriminant_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("discriminant.info.md.primitive_real_type")
        public <T extends PointerBase> T address_discriminant_info_md_primitive_real_type();
        @CField("discriminant.info.rebindings")
        public Pointer
        get_discriminant_info_rebindings();

        @CField("discriminant.info.rebindings")
        public void
        set_discriminant_info_rebindings(
            Pointer val
        );

        @CFieldAddress("discriminant.info.rebindings")
        public <T extends PointerBase> T address_discriminant_info_rebindings();
        @CField("discriminant.info.from_rebound")
        public byte
        get_discriminant_info_from_rebound();

        @CField("discriminant.info.from_rebound")
        public void
        set_discriminant_info_from_rebound(
            byte val
        );

        @CFieldAddress("discriminant.info.from_rebound")
        public <T extends PointerBase> T address_discriminant_info_from_rebound();
        @CField("values.node")
        public Pointer
        get_values_node();

        @CField("values.node")
        public void
        set_values_node(
            Pointer val
        );

        @CFieldAddress("values.node")
        public <T extends PointerBase> T address_values_node();
        @CField("values.info.md.dottable_subp")
        public byte
        get_values_info_md_dottable_subp();

        @CField("values.info.md.dottable_subp")
        public void
        set_values_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("values.info.md.dottable_subp")
        public <T extends PointerBase> T address_values_info_md_dottable_subp();
        @CField("values.info.md.primitive")
        public Pointer
        get_values_info_md_primitive();

        @CField("values.info.md.primitive")
        public void
        set_values_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("values.info.md.primitive")
        public <T extends PointerBase> T address_values_info_md_primitive();
        @CField("values.info.md.primitive_real_type")
        public Pointer
        get_values_info_md_primitive_real_type();

        @CField("values.info.md.primitive_real_type")
        public void
        set_values_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("values.info.md.primitive_real_type")
        public <T extends PointerBase> T address_values_info_md_primitive_real_type();
        @CField("values.info.rebindings")
        public Pointer
        get_values_info_rebindings();

        @CField("values.info.rebindings")
        public void
        set_values_info_rebindings(
            Pointer val
        );

        @CFieldAddress("values.info.rebindings")
        public <T extends PointerBase> T address_values_info_rebindings();
        @CField("values.info.from_rebound")
        public byte
        get_values_info_from_rebound();

        @CField("values.info.from_rebound")
        public void
        set_values_info_from_rebound(
            byte val
        );

        @CFieldAddress("values.info.from_rebound")
        public <T extends PointerBase> T address_values_info_from_rebound();
    }

        
    
    

    /** The structure for the langkit ada_internal_doc_annotation */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_doc_annotation")
    public interface DocAnnotationNative extends PointerBase {
        @CField("key")
        public StringNative
        get_key();

        @CField("key")
        public void
        set_key(
            StringNative val
        );

        @CFieldAddress("key")
        public <T extends PointerBase> T address_key();
        @CField("value")
        public StringNative
        get_value();

        @CField("value")
        public void
        set_value(
            StringNative val
        );

        @CFieldAddress("value")
        public <T extends PointerBase> T address_value();
    }

        
        
        
        
        
        
        
    
    

    /** The structure for the langkit ada_internal_param_actual */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_param_actual")
    public interface ParamActualNative extends PointerBase {
        @CField("param.node")
        public Pointer
        get_param_node();

        @CField("param.node")
        public void
        set_param_node(
            Pointer val
        );

        @CFieldAddress("param.node")
        public <T extends PointerBase> T address_param_node();
        @CField("param.info.md.dottable_subp")
        public byte
        get_param_info_md_dottable_subp();

        @CField("param.info.md.dottable_subp")
        public void
        set_param_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("param.info.md.dottable_subp")
        public <T extends PointerBase> T address_param_info_md_dottable_subp();
        @CField("param.info.md.primitive")
        public Pointer
        get_param_info_md_primitive();

        @CField("param.info.md.primitive")
        public void
        set_param_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("param.info.md.primitive")
        public <T extends PointerBase> T address_param_info_md_primitive();
        @CField("param.info.md.primitive_real_type")
        public Pointer
        get_param_info_md_primitive_real_type();

        @CField("param.info.md.primitive_real_type")
        public void
        set_param_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("param.info.md.primitive_real_type")
        public <T extends PointerBase> T address_param_info_md_primitive_real_type();
        @CField("param.info.rebindings")
        public Pointer
        get_param_info_rebindings();

        @CField("param.info.rebindings")
        public void
        set_param_info_rebindings(
            Pointer val
        );

        @CFieldAddress("param.info.rebindings")
        public <T extends PointerBase> T address_param_info_rebindings();
        @CField("param.info.from_rebound")
        public byte
        get_param_info_from_rebound();

        @CField("param.info.from_rebound")
        public void
        set_param_info_from_rebound(
            byte val
        );

        @CFieldAddress("param.info.from_rebound")
        public <T extends PointerBase> T address_param_info_from_rebound();
        @CField("actual.node")
        public Pointer
        get_actual_node();

        @CField("actual.node")
        public void
        set_actual_node(
            Pointer val
        );

        @CFieldAddress("actual.node")
        public <T extends PointerBase> T address_actual_node();
        @CField("actual.info.md.dottable_subp")
        public byte
        get_actual_info_md_dottable_subp();

        @CField("actual.info.md.dottable_subp")
        public void
        set_actual_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("actual.info.md.dottable_subp")
        public <T extends PointerBase> T address_actual_info_md_dottable_subp();
        @CField("actual.info.md.primitive")
        public Pointer
        get_actual_info_md_primitive();

        @CField("actual.info.md.primitive")
        public void
        set_actual_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("actual.info.md.primitive")
        public <T extends PointerBase> T address_actual_info_md_primitive();
        @CField("actual.info.md.primitive_real_type")
        public Pointer
        get_actual_info_md_primitive_real_type();

        @CField("actual.info.md.primitive_real_type")
        public void
        set_actual_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("actual.info.md.primitive_real_type")
        public <T extends PointerBase> T address_actual_info_md_primitive_real_type();
        @CField("actual.info.rebindings")
        public Pointer
        get_actual_info_rebindings();

        @CField("actual.info.rebindings")
        public void
        set_actual_info_rebindings(
            Pointer val
        );

        @CFieldAddress("actual.info.rebindings")
        public <T extends PointerBase> T address_actual_info_rebindings();
        @CField("actual.info.from_rebound")
        public byte
        get_actual_info_from_rebound();

        @CField("actual.info.from_rebound")
        public void
        set_actual_info_from_rebound(
            byte val
        );

        @CFieldAddress("actual.info.from_rebound")
        public <T extends PointerBase> T address_actual_info_from_rebound();
    }

        
        
        
    
    

    /** The structure for the langkit ada_internal_ref_result */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_ref_result")
    public interface RefResultNative extends PointerBase {
        @CField("ref.node")
        public Pointer
        get_ref_node();

        @CField("ref.node")
        public void
        set_ref_node(
            Pointer val
        );

        @CFieldAddress("ref.node")
        public <T extends PointerBase> T address_ref_node();
        @CField("ref.info.md.dottable_subp")
        public byte
        get_ref_info_md_dottable_subp();

        @CField("ref.info.md.dottable_subp")
        public void
        set_ref_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("ref.info.md.dottable_subp")
        public <T extends PointerBase> T address_ref_info_md_dottable_subp();
        @CField("ref.info.md.primitive")
        public Pointer
        get_ref_info_md_primitive();

        @CField("ref.info.md.primitive")
        public void
        set_ref_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("ref.info.md.primitive")
        public <T extends PointerBase> T address_ref_info_md_primitive();
        @CField("ref.info.md.primitive_real_type")
        public Pointer
        get_ref_info_md_primitive_real_type();

        @CField("ref.info.md.primitive_real_type")
        public void
        set_ref_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("ref.info.md.primitive_real_type")
        public <T extends PointerBase> T address_ref_info_md_primitive_real_type();
        @CField("ref.info.rebindings")
        public Pointer
        get_ref_info_rebindings();

        @CField("ref.info.rebindings")
        public void
        set_ref_info_rebindings(
            Pointer val
        );

        @CFieldAddress("ref.info.rebindings")
        public <T extends PointerBase> T address_ref_info_rebindings();
        @CField("ref.info.from_rebound")
        public byte
        get_ref_info_from_rebound();

        @CField("ref.info.from_rebound")
        public void
        set_ref_info_from_rebound(
            byte val
        );

        @CFieldAddress("ref.info.from_rebound")
        public <T extends PointerBase> T address_ref_info_from_rebound();
        @CField("kind")
        public int
        get_kind();

        @CField("kind")
        public void
        set_kind(
            int val
        );

        @CFieldAddress("kind")
        public <T extends PointerBase> T address_kind();
    }

        
    
    

    /** The structure for the langkit ada_internal_refd_decl */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_refd_decl")
    public interface RefdDeclNative extends PointerBase {
        @CField("decl.node")
        public Pointer
        get_decl_node();

        @CField("decl.node")
        public void
        set_decl_node(
            Pointer val
        );

        @CFieldAddress("decl.node")
        public <T extends PointerBase> T address_decl_node();
        @CField("decl.info.md.dottable_subp")
        public byte
        get_decl_info_md_dottable_subp();

        @CField("decl.info.md.dottable_subp")
        public void
        set_decl_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("decl.info.md.dottable_subp")
        public <T extends PointerBase> T address_decl_info_md_dottable_subp();
        @CField("decl.info.md.primitive")
        public Pointer
        get_decl_info_md_primitive();

        @CField("decl.info.md.primitive")
        public void
        set_decl_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("decl.info.md.primitive")
        public <T extends PointerBase> T address_decl_info_md_primitive();
        @CField("decl.info.md.primitive_real_type")
        public Pointer
        get_decl_info_md_primitive_real_type();

        @CField("decl.info.md.primitive_real_type")
        public void
        set_decl_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("decl.info.md.primitive_real_type")
        public <T extends PointerBase> T address_decl_info_md_primitive_real_type();
        @CField("decl.info.rebindings")
        public Pointer
        get_decl_info_rebindings();

        @CField("decl.info.rebindings")
        public void
        set_decl_info_rebindings(
            Pointer val
        );

        @CFieldAddress("decl.info.rebindings")
        public <T extends PointerBase> T address_decl_info_rebindings();
        @CField("decl.info.from_rebound")
        public byte
        get_decl_info_from_rebound();

        @CField("decl.info.from_rebound")
        public void
        set_decl_info_from_rebound(
            byte val
        );

        @CFieldAddress("decl.info.from_rebound")
        public <T extends PointerBase> T address_decl_info_from_rebound();
        @CField("kind")
        public int
        get_kind();

        @CField("kind")
        public void
        set_kind(
            int val
        );

        @CFieldAddress("kind")
        public <T extends PointerBase> T address_kind();
    }

        
    
    

    /** The structure for the langkit ada_internal_refd_def */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_refd_def")
    public interface RefdDefNative extends PointerBase {
        @CField("def_name.node")
        public Pointer
        get_def_name_node();

        @CField("def_name.node")
        public void
        set_def_name_node(
            Pointer val
        );

        @CFieldAddress("def_name.node")
        public <T extends PointerBase> T address_def_name_node();
        @CField("def_name.info.md.dottable_subp")
        public byte
        get_def_name_info_md_dottable_subp();

        @CField("def_name.info.md.dottable_subp")
        public void
        set_def_name_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("def_name.info.md.dottable_subp")
        public <T extends PointerBase> T address_def_name_info_md_dottable_subp();
        @CField("def_name.info.md.primitive")
        public Pointer
        get_def_name_info_md_primitive();

        @CField("def_name.info.md.primitive")
        public void
        set_def_name_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("def_name.info.md.primitive")
        public <T extends PointerBase> T address_def_name_info_md_primitive();
        @CField("def_name.info.md.primitive_real_type")
        public Pointer
        get_def_name_info_md_primitive_real_type();

        @CField("def_name.info.md.primitive_real_type")
        public void
        set_def_name_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("def_name.info.md.primitive_real_type")
        public <T extends PointerBase> T address_def_name_info_md_primitive_real_type();
        @CField("def_name.info.rebindings")
        public Pointer
        get_def_name_info_rebindings();

        @CField("def_name.info.rebindings")
        public void
        set_def_name_info_rebindings(
            Pointer val
        );

        @CFieldAddress("def_name.info.rebindings")
        public <T extends PointerBase> T address_def_name_info_rebindings();
        @CField("def_name.info.from_rebound")
        public byte
        get_def_name_info_from_rebound();

        @CField("def_name.info.from_rebound")
        public void
        set_def_name_info_from_rebound(
            byte val
        );

        @CFieldAddress("def_name.info.from_rebound")
        public <T extends PointerBase> T address_def_name_info_from_rebound();
        @CField("kind")
        public int
        get_kind();

        @CField("kind")
        public void
        set_kind(
            int val
        );

        @CFieldAddress("kind")
        public <T extends PointerBase> T address_kind();
    }

        
    
    

    /** The structure for the langkit ada_internal_shape */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_shape")
    public interface ShapeNative extends PointerBase {
        @CField("components")
        public AdaNodeArrayNative
        get_components();

        @CField("components")
        public void
        set_components(
            AdaNodeArrayNative val
        );

        @CFieldAddress("components")
        public <T extends PointerBase> T address_components();
        @CField("discriminants_values")
        public DiscriminantValuesArrayNative
        get_discriminants_values();

        @CField("discriminants_values")
        public void
        set_discriminants_values(
            DiscriminantValuesArrayNative val
        );

        @CFieldAddress("discriminants_values")
        public <T extends PointerBase> T address_discriminants_values();
    }

        
    
    

    /** The structure for the langkit ada_internal_substitution */
    @CContext(LibDirectives.class)
    @CStruct("ada_internal_substitution")
    public interface SubstitutionNative extends PointerBase {
        @CField("from_decl.node")
        public Pointer
        get_from_decl_node();

        @CField("from_decl.node")
        public void
        set_from_decl_node(
            Pointer val
        );

        @CFieldAddress("from_decl.node")
        public <T extends PointerBase> T address_from_decl_node();
        @CField("from_decl.info.md.dottable_subp")
        public byte
        get_from_decl_info_md_dottable_subp();

        @CField("from_decl.info.md.dottable_subp")
        public void
        set_from_decl_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("from_decl.info.md.dottable_subp")
        public <T extends PointerBase> T address_from_decl_info_md_dottable_subp();
        @CField("from_decl.info.md.primitive")
        public Pointer
        get_from_decl_info_md_primitive();

        @CField("from_decl.info.md.primitive")
        public void
        set_from_decl_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("from_decl.info.md.primitive")
        public <T extends PointerBase> T address_from_decl_info_md_primitive();
        @CField("from_decl.info.md.primitive_real_type")
        public Pointer
        get_from_decl_info_md_primitive_real_type();

        @CField("from_decl.info.md.primitive_real_type")
        public void
        set_from_decl_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("from_decl.info.md.primitive_real_type")
        public <T extends PointerBase> T address_from_decl_info_md_primitive_real_type();
        @CField("from_decl.info.rebindings")
        public Pointer
        get_from_decl_info_rebindings();

        @CField("from_decl.info.rebindings")
        public void
        set_from_decl_info_rebindings(
            Pointer val
        );

        @CFieldAddress("from_decl.info.rebindings")
        public <T extends PointerBase> T address_from_decl_info_rebindings();
        @CField("from_decl.info.from_rebound")
        public byte
        get_from_decl_info_from_rebound();

        @CField("from_decl.info.from_rebound")
        public void
        set_from_decl_info_from_rebound(
            byte val
        );

        @CFieldAddress("from_decl.info.from_rebound")
        public <T extends PointerBase> T address_from_decl_info_from_rebound();
        @CField("to_value")
        public BigIntegerNative
        get_to_value();

        @CField("to_value")
        public void
        set_to_value(
            BigIntegerNative val
        );

        @CFieldAddress("to_value")
        public <T extends PointerBase> T address_to_value();
        @CField("value_type.node")
        public Pointer
        get_value_type_node();

        @CField("value_type.node")
        public void
        set_value_type_node(
            Pointer val
        );

        @CFieldAddress("value_type.node")
        public <T extends PointerBase> T address_value_type_node();
        @CField("value_type.info.md.dottable_subp")
        public byte
        get_value_type_info_md_dottable_subp();

        @CField("value_type.info.md.dottable_subp")
        public void
        set_value_type_info_md_dottable_subp(
            byte val
        );

        @CFieldAddress("value_type.info.md.dottable_subp")
        public <T extends PointerBase> T address_value_type_info_md_dottable_subp();
        @CField("value_type.info.md.primitive")
        public Pointer
        get_value_type_info_md_primitive();

        @CField("value_type.info.md.primitive")
        public void
        set_value_type_info_md_primitive(
            Pointer val
        );

        @CFieldAddress("value_type.info.md.primitive")
        public <T extends PointerBase> T address_value_type_info_md_primitive();
        @CField("value_type.info.md.primitive_real_type")
        public Pointer
        get_value_type_info_md_primitive_real_type();

        @CField("value_type.info.md.primitive_real_type")
        public void
        set_value_type_info_md_primitive_real_type(
            Pointer val
        );

        @CFieldAddress("value_type.info.md.primitive_real_type")
        public <T extends PointerBase> T address_value_type_info_md_primitive_real_type();
        @CField("value_type.info.rebindings")
        public Pointer
        get_value_type_info_rebindings();

        @CField("value_type.info.rebindings")
        public void
        set_value_type_info_rebindings(
            Pointer val
        );

        @CFieldAddress("value_type.info.rebindings")
        public <T extends PointerBase> T address_value_type_info_rebindings();
        @CField("value_type.info.from_rebound")
        public byte
        get_value_type_info_from_rebound();

        @CField("value_type.info.from_rebound")
        public void
        set_value_type_info_from_rebound(
            byte val
        );

        @CFieldAddress("value_type.info.from_rebound")
        public <T extends PointerBase> T address_value_type_info_from_rebound();
    }


    // ===== Generated arrays =====

    
    

    /**
     * The native structure of the ada_discriminant_values_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_discriminant_values_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface DiscriminantValuesArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_doc_annotation_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_doc_annotation_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface DocAnnotationArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_ada_node_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_ada_node_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface AdaNodeArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_param_actual_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_param_actual_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface ParamActualArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_ref_result_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_ref_result_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface RefResultArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_shape_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_shape_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface ShapeArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_substitution_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_substitution_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface SubstitutionArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_analysis_unit_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_analysis_unit_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface AnalysisUnitArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the ada_unbounded_text_type_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "ada_unbounded_text_type_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface UnboundedTextTypeArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }


    // ===== Generated iterators =====

    
    

    /** This interface represents the langkit ada_completion_item_iterator iterator */
    public interface CompletionItemIteratorNative extends Pointer {}


    // ===== Native function definitions =====

    /** This class contains all native function definitions for NI */
    @CContext(LibDirectives.class)
    public static final class NI_LIB {

        // ----- Language specific functions -----

        
      


    // --- Project manager functions

    /** Function to load a GPR project */
    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native void ada_gpr_project_load(
        CCharPointer projectFile,
        Pointer scenarioVariable,
        CCharPointer target,
        CCharPointer runtime,
        Pointer project,
        Pointer errors
    );

    /** Free the given project */
    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native void ada_gpr_project_free(
        ProjectManagerNative project
    );

    /** Get the unit provider for the given project */
    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native UnitProviderNative ada_gpr_project_create_unit_provider(
        ProjectManagerNative project,
        CCharPointer subProject
    );

    /** Get the source file array for the given project */
    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native StringArrayNative ada_gpr_project_source_files(
        ProjectManagerNative project,
        int mode,
        CCharPointerPointer projectsData,
        int projectsLength
    );

    /**
     * Wrapper around ``Initialize_Context_From_Project`` to initialize
     * ``Context`` (an already allocated but not yet initialized analysis
     * context) from ``Self``.
     */
    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native void ada_gpr_project_initialize_context(
        ProjectManagerNative project,
        AnalysisContextNative context,
        CCharPointer subproject,
        EventHandlerNative eventHandler,
        int withTrivia,
        int tabStop
    );

    /** Free the given source file array */
    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native void ada_free_string_array(
        StringArrayNative sourceFileArray
    );

    /** Create an auto provider reference */
    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native UnitProviderNative ada_create_auto_provider(
        CCharPointerPointer sourceFiles,
        CCharPointer charset
    );


    // --- Config pragmas

    @CompilerDirectives.TruffleBoundary
    @CFunction
    public static native void ada_set_config_pragmas_mapping(
        AnalysisContextNative context,
        AnalysisUnitNative globalPragmas,
        Pointer localPragmas
    );



        // ----- Entry point literals -----

        /**
         * This entry point literal provide a pointer to the unit requested
         * callback.
         */
        public static final CEntryPointLiteral<UnitRequestedFunctionPointer>
            unitRequestedFunction = CEntryPointLiteral.create(
                Libadalang.class,
                "unitRequested",
                IsolateThread.class,
                AnalysisContextNative.class,
                TextNative.class,
                AnalysisUnitNative.class,
                byte.class,
                byte.class
            );

        /**
         * This entry point literal provide a pointer to the unit parsed
         * callback.
         */
        public static final CEntryPointLiteral<UnitParsedFunctionPointer>
            unitParsedFunction = CEntryPointLiteral.create(
                Libadalang.class,
                "unitParsed",
                IsolateThread.class,
                AnalysisContextNative.class,
                AnalysisUnitNative.class,
                byte.class
            );

        // ----- Util functions -----

        /** Util function to free langkit side allocated memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_free(
            PointerBase pointer
        );

        // ----- Exception functions -----

        /** Get the last exception raised by langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native LangkitExceptionNative
        ada_get_last_exception();

        // ----- Big integer functions -----

        /** Create a big integer from a text */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native BigIntegerNative ada_create_big_integer(
            TextNative text
        );

        /** Get the text representation of a big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_big_integer_text(
            BigIntegerNative big_integer,
            TextNative text
        );

        /** Decrease the reference counter of the big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_big_integer_decref(
            BigIntegerNative big_integer
        );

        // ----- Symbol functions -----

        /** Create a new symbol in the given context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_context_symbol(
            AnalysisContextNative context,
            TextNative text,
            SymbolNative res
        );

        /** Get the text of a given symbol */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_symbol_text(
            SymbolNative symbol,
            TextNative text
        );

        // ----- String functions -----

        /** Create a new string wrapper in langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native StringNative ada_create_string(
            CIntPointer content,
            int length
        );

        /** Decrease the reference counter of a string */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_string_dec_ref(
            StringNative string
        );

        // ----- Text functions -----

        /** Destroy a text in the memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_destroy_text(
            TextNative text
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_dec_ref_file_reader(
            FileReaderNative fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_dec_ref_unit_provider(
            UnitProviderNative unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native EventHandlerNative ada_create_event_handler(
            VoidPointer data,
            VoidPointer destroy_callback,
            UnitRequestedFunctionPointer unit_requested_func,
            UnitParsedFunctionPointer unit_parsed_func
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_dec_ref_event_handler(
            EventHandlerNative eventHandler
        );

        // ----- Token functions -----

        /**
         * Kind for this token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_token_get_kind(
            TokenNative token
        );

        /**
         * Return the source location range of the given token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_token_sloc_range(
            TokenNative token,
            SourceLocationRangeNative result
        );

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_token_next(
            TokenNative token,
            TokenNative res
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_token_previous(
            TokenNative token,
            TokenNative res
        );

        /** Get if two tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native byte ada_token_is_equivalent(
            TokenNative left,
            TokenNative right
        );

        /** Get the text in a token range */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_token_range_text(
            TokenNative start,
            TokenNative end,
            TextNative res
        );

        // ----- Analysis context functions -----

        /** Allocate a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative
        ada_allocate_analysis_context();

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void
        ada_initialize_analysis_context(
            AnalysisContextNative context,
            CCharPointer charset,
            FileReaderNative file_reader,
            UnitProviderNative unit_provider,
            EventHandlerNative event_handler,
            int with_trivia,
            int tab_stop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_context_incref(
            AnalysisContextNative context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_context_decref(
            AnalysisContextNative context
        );

        // ----- Analysis unit functions -----

        /** Get a unit from a file */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        ada_get_analysis_unit_from_file(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            int reparse,
            int rule
        );

        /** Get a unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        ada_get_analysis_unit_from_buffer(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            CCharPointer buffer,
            long buffer_size,
            int rule
        );

        /** Get a unit from the unit provider. */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        ada_get_analysis_unit_from_provider(
            AnalysisContextNative context,
            TextNative name,
            int kind,
            CCharPointer charset,
            int reparse
        );

        /** Get the root of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_unit_root(
            AnalysisUnitNative unit,
            EntityNative res
        );

        /** Get the file name for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native CCharPointer ada_unit_filename(
            AnalysisUnitNative unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_unit_token_count(
            AnalysisUnitNative unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_unit_trivia_count(
            AnalysisUnitNative unit
        );

        /** Get the first token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_unit_first_token(
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the last token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_unit_last_token (
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the context for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative ada_unit_context(
            AnalysisUnitNative unit
        );

        /** Get the diagnostic count of the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_unit_diagnostic_count(
            AnalysisUnitNative unit
        );

        /** Get the nth diagnostic for the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_unit_diagnostic(
            AnalysisUnitNative unit,
            int n,
            DiagnosticNative diagnostic
        );

        // ----- Array functions -----

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native DiscriminantValuesArrayNative ada_discriminant_values_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_discriminant_values_array_dec_ref(DiscriminantValuesArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native DocAnnotationArrayNative ada_doc_annotation_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_doc_annotation_array_dec_ref(DocAnnotationArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AdaNodeArrayNative ada_ada_node_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_ada_node_array_dec_ref(AdaNodeArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native ParamActualArrayNative ada_param_actual_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_param_actual_array_dec_ref(ParamActualArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RefResultArrayNative ada_ref_result_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_ref_result_array_dec_ref(RefResultArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native ShapeArrayNative ada_shape_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_shape_array_dec_ref(ShapeArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native SubstitutionArrayNative ada_substitution_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_substitution_array_dec_ref(SubstitutionArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitArrayNative ada_analysis_unit_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_analysis_unit_array_dec_ref(AnalysisUnitArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native UnboundedTextTypeArrayNative ada_unbounded_text_type_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_unbounded_text_type_array_dec_ref(UnboundedTextTypeArrayNative array);


        // ----- Structure functions -----

            
        
    


            
        
    


        
    


            
        
    


            
        
    


            
            
        
    


            
        
    


            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_internal_doc_annotation_dec_ref(
            DocAnnotationNative structNative
        );

            
            
            
            
            
            
            
        
    


            
            
            
        
    


            
        
    


            
        
    


            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_internal_shape_dec_ref(
            ShapeNative structNative
        );

            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_internal_substitution_dec_ref(
            SubstitutionNative structNative
        );


        // ----- Node functions -----

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_node_is_equivalent(
            EntityNative entity_left,
            EntityNative entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_node_hash(
            EntityNative entity
        );

        /** Get the type of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_node_kind(
            EntityNative entity
        );

        /** Get the text from a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_node_text(
            EntityNative entity,
            TextNative text
        );

        /** Get the source location range for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_node_sloc_range(
            EntityNative entity,
            SourceLocationRangeNative slocr
        );

        /** Get the number of children for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_node_children_count(
            EntityNative entity
        );

        /** Get the nth child for the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_node_child(
            EntityNative entity,
            int n,
            EntityNative res
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_node_is_token_node(
            EntityNative entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative ada_node_unit(
            EntityNative entity
        );

        /** Get the image of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void ada_node_image(
            EntityNative entity,
            TextNative text
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of ada_ada_node_p_declarative_scope langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_declarative_scope(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_enclosing_compilation_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_enclosing_compilation_unit(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_get_uninstantiated_node langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_get_uninstantiated_node(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_complete langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_complete(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_p_valid_keywords langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_valid_keywords(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_p_generic_instantiations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_generic_instantiations(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_p_semantic_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_semantic_parent(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_parent_basic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_parent_basic_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_filter_is_imported_by langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_filter_is_imported_by(
            EntityNative node,
            AnalysisUnitArrayNative units,
            byte transitive,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_p_xref_entry_point langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_xref_entry_point(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_ada_node_p_resolve_names langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_resolve_names(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_ada_node_p_standard_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_standard_unit(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_p_std_entity langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_std_entity(
            EntityNative node,
            SymbolNative sym,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_bool_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_bool_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_int_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_universal_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_universal_int_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_universal_real_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_universal_real_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_std_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_std_char_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_std_wide_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_std_wide_char_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_std_wide_wide_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_std_wide_wide_char_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_top_level_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_top_level_decl(
            EntityNative node,
            AnalysisUnitNative unit,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_p_choice_match langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_choice_match(
            EntityNative node,
            BigIntegerNative value,
            CCharPointer result
        );
            

        /** Isomethod of ada_ada_node_p_gnat_xref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_p_gnat_xref(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_parent(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_parents(
            EntityNative node,
            byte with_self,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_children(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_token_start(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of ada_ada_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_token_end(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of ada_ada_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_child_index(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of ada_ada_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_previous_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_next_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ada_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_unit(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_ada_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_is_ghost(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_ada_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ada_node_full_sloc_image(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_abort_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_abort_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_abstract_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_abstract_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_assoc_list_p_zip_with_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_assoc_list_p_zip_with_params(
            EntityNative node,
            byte imprecise_fallback,
            Pointer result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_aliased_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_aliased_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_all_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_all_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_constrained_array_indices_f_list langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_constrained_array_indices_f_list(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_unconstrained_array_indices_f_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_unconstrained_array_indices_f_types(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_aspect_assoc_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_aspect_assoc_f_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_aspect_assoc_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_aspect_assoc_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_aspect_assoc_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_aspect_assoc_p_is_ghost_code(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    

            

        /** Isomethod of ada_at_clause_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_at_clause_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_at_clause_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_at_clause_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_attribute_def_clause_f_attribute_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_attribute_def_clause_f_attribute_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_attribute_def_clause_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_attribute_def_clause_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_enum_rep_clause_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_enum_rep_clause_f_type_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_enum_rep_clause_f_aggregate langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_enum_rep_clause_f_aggregate(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_enum_rep_clause_p_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_enum_rep_clause_p_params(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_record_rep_clause_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_record_rep_clause_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_record_rep_clause_f_at_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_record_rep_clause_f_at_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_record_rep_clause_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_record_rep_clause_f_components(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_aspect_spec_f_aspect_assocs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_aspect_spec_f_aspect_assocs(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_assoc_p_assoc_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_assoc_p_assoc_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_contract_case_assoc_f_guard langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_contract_case_assoc_f_guard(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_contract_case_assoc_f_consequence langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_contract_case_assoc_f_consequence(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_pragma_argument_assoc_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pragma_argument_assoc_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_pragma_argument_assoc_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pragma_argument_assoc_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_formal_param_holder_p_abstract_formal_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_formal_param_holder_p_abstract_formal_params(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_formal_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_formal_param_holder_p_formal_params(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_nb_min_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_formal_param_holder_p_nb_min_params(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_nb_max_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_formal_param_holder_p_nb_max_params(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of ada_base_formal_param_holder_p_param_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_formal_param_holder_p_param_types(
            EntityNative node,
            EntityNative origin,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_base_subp_spec_p_returns langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_spec_p_returns(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_subp_spec_p_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_spec_p_params(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_base_subp_spec_p_primitive_subp_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_spec_p_primitive_subp_types(
            EntityNative node,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_base_subp_spec_p_primitive_subp_first_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_spec_p_primitive_subp_first_type(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_base_subp_spec_p_primitive_subp_tagged_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_spec_p_primitive_subp_tagged_type(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_base_subp_spec_p_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_spec_p_return_type(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_entry_spec_f_entry_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_spec_f_entry_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_spec_f_family_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_spec_f_family_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_spec_f_entry_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_spec_f_entry_params(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_subp_spec_f_subp_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_spec_f_subp_kind(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subp_spec_f_subp_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_spec_f_subp_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subp_spec_f_subp_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_spec_f_subp_params(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subp_spec_f_subp_returns langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_spec_f_subp_returns(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_synthetic_binary_spec_f_left_param langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_binary_spec_f_left_param(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_synthetic_binary_spec_f_right_param langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_binary_spec_f_right_param(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_synthetic_binary_spec_f_return_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_binary_spec_f_return_type_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_synthetic_unary_spec_f_right_param langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_unary_spec_f_right_param(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_synthetic_unary_spec_f_return_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_unary_spec_f_return_type_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_component_list_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_list_f_components(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_component_list_f_variant_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_list_f_variant_part(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_known_discriminant_part_f_discr_specs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_known_discriminant_part_f_discr_specs(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_entry_completion_formal_params_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_completion_formal_params_f_params(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_formal_part_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_formal_part_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_record_def_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_record_def_f_components(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_basic_assoc_p_get_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_assoc_p_get_params(
            EntityNative node,
            byte imprecise_fallback,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_aggregate_assoc_f_designators langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_aggregate_assoc_f_designators(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_aggregate_assoc_f_r_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_aggregate_assoc_f_r_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_composite_constraint_assoc_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_composite_constraint_assoc_f_ids(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_composite_constraint_assoc_f_constraint_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_composite_constraint_assoc_f_constraint_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_iterated_assoc_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_iterated_assoc_f_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_iterated_assoc_f_r_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_iterated_assoc_f_r_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_param_assoc_f_designator langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_param_assoc_f_designator(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_param_assoc_f_r_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_param_assoc_f_r_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_basic_decl_p_is_formal langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_formal(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_p_doc_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_doc_annotations(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_doc(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_previous_part_for_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_previous_part_for_decl(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_canonical_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_canonical_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_all_parts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_all_parts(
            EntityNative node,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_is_static_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_static_decl(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_f_aspects langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_f_aspects(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_get_aspect_assoc langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_get_aspect_assoc(
            EntityNative node,
            SymbolNative name,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_get_aspect_spec_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_get_aspect_spec_expr(
            EntityNative node,
            SymbolNative name,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_get_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_get_aspect(
            EntityNative node,
            SymbolNative name,
            byte imprecise_fallback,
            AspectNative result
        );
            

        /** Isomethod of ada_basic_decl_p_has_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_has_aspect(
            EntityNative node,
            SymbolNative name,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_p_get_pragma langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_get_pragma(
            EntityNative node,
            SymbolNative name,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_get_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_get_representation_clause(
            EntityNative node,
            SymbolNative name,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_get_at_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_get_at_clause(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_is_imported langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_imported(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_ghost_code(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_p_is_compilation_unit_root langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_compilation_unit_root(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_p_is_visible langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_visible(
            EntityNative node,
            EntityNative from_node,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_p_base_subp_declarations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_base_subp_declarations(
            EntityNative node,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_root_subp_declarations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_root_subp_declarations(
            EntityNative node,
            EntityNative origin,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_find_all_overrides langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_find_all_overrides(
            EntityNative node,
            AnalysisUnitArrayNative units,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_defining_names langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_defining_names(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_defining_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_type_expression langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_type_expression(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_subp_spec_or_null langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_subp_spec_or_null(
            EntityNative node,
            byte follow_generic,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_is_subprogram langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_subprogram(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_decl_p_relative_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_relative_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_relative_name_text langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_relative_name_text(
            EntityNative node,
            SymbolNative result
        );
            

        /** Isomethod of ada_basic_decl_p_next_part_for_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_next_part_for_decl(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_body_part_for_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_body_part_for_decl(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_most_visible_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_most_visible_part(
            EntityNative node,
            EntityNative origin,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_basic_decl_p_fully_qualified_name_array langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_fully_qualified_name_array(
            EntityNative node,
            byte include_profile,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_fully_qualified_name(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_canonical_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_canonical_fully_qualified_name(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_unique_identifying_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_unique_identifying_name(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_basic_decl_p_is_constant_object langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_decl_p_is_constant_object(
            EntityNative node,
            CCharPointer result
        );

        
    

            

        /** Isomethod of ada_abstract_state_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_abstract_state_decl_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_anonymous_expr_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_anonymous_expr_decl_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_anonymous_expr_decl_p_get_formal langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_anonymous_expr_decl_p_get_formal(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_formal_param_decl_p_formal_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_formal_param_decl_p_formal_type(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_component_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_decl_f_ids(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_component_decl_f_component_def langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_decl_f_component_def(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_component_decl_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_decl_f_default_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_discriminant_spec_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_discriminant_spec_f_ids(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_discriminant_spec_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_discriminant_spec_f_type_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_discriminant_spec_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_discriminant_spec_f_default_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_formal_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_formal_f_decl(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_param_spec_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_param_spec_f_ids(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_param_spec_f_has_aliased langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_param_spec_f_has_aliased(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_param_spec_f_mode langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_param_spec_f_mode(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_param_spec_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_param_spec_f_type_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_param_spec_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_param_spec_f_default_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_synthetic_formal_param_decl_f_param_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_formal_param_decl_f_param_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_package_decl_f_package_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_package_decl_f_package_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_package_decl_f_public_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_package_decl_f_public_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_package_decl_f_private_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_package_decl_f_private_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_package_decl_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_package_decl_f_end_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_package_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_package_decl_p_body_part(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_base_type_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_base_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_base_subtype(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_private_completion langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_private_completion(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_inherited_primitive langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_inherited_primitive(
            EntityNative node,
            EntityNative p,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_get_record_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_get_record_representation_clause(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_get_enum_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_get_enum_representation_clause(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_record_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_record_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_array_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_array_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_find_derived_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_find_derived_types(
            EntityNative node,
            EntityNative root,
            EntityNative origin,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_real_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_real_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_float_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_float_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_fixed_point langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_fixed_point(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_scalar_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_scalar_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_enum_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_enum_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_access_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_access_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_specific_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_specific_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_char_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_classwide_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_classwide_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_discrete_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_discrete_range(
            EntityNative node,
            DiscreteRangeNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_discrete_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_discrete_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_int_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_accessed_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_accessed_type(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_tagged_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_tagged_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_base_type(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_base_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_base_types(
            EntityNative node,
            EntityNative origin,
            Pointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_find_all_derived_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_find_all_derived_types(
            EntityNative node,
            AnalysisUnitArrayNative units,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_comp_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_comp_type(
            EntityNative node,
            byte is_subscript,
            EntityNative origin,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_index_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_index_type(
            EntityNative node,
            int dim,
            EntityNative origin,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_derived_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_derived_type(
            EntityNative node,
            EntityNative other_type,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_interface_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_interface_type(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_matching_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_matching_type(
            EntityNative node,
            EntityNative expected_type,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_canonical_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_canonical_type(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_previous_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_previous_part(
            EntityNative node,
            byte go_to_incomplete,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_next_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_next_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_full_view langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_full_view(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_definite_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_definite_subtype(
            EntityNative node,
            EntityNative origin,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_is_private langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_is_private(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_discriminants_list langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_discriminants_list(
            EntityNative node,
            EntityNative stop_recurse_at,
            EntityNative origin,
            Pointer result
        );
            

        /** Isomethod of ada_base_type_decl_p_root_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_root_type(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );
            

        /** Isomethod of ada_base_type_decl_p_shapes langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_type_decl_p_shapes(
            EntityNative node,
            byte include_discriminants,
            EntityNative origin,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_base_subtype_decl_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subtype_decl_p_get_type(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_subtype_decl_f_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subtype_decl_f_subtype(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_incomplete_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_incomplete_type_decl_f_discriminants(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_incomplete_formal_type_decl_f_is_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_incomplete_formal_type_decl_f_is_tagged(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_incomplete_formal_type_decl_f_default_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_incomplete_formal_type_decl_f_default_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_incomplete_tagged_type_decl_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_incomplete_tagged_type_decl_f_has_abstract(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_protected_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_type_decl_f_discriminants(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_protected_type_decl_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_type_decl_f_interfaces(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_protected_type_decl_f_definition langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_type_decl_f_definition(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_task_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_type_decl_f_discriminants(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_task_type_decl_f_definition langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_type_decl_f_definition(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_type_decl_f_discriminants langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_decl_f_discriminants(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_type_decl_f_type_def langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_decl_f_type_def(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_type_decl_p_get_primitives langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_decl_p_get_primitives(
            EntityNative node,
            byte only_inherited,
            byte include_predefined_operators,
            Pointer result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_formal_type_decl_f_default_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_formal_type_decl_f_default_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_basic_subp_decl_p_has_spark_mode_on langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_subp_decl_p_has_spark_mode_on(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_basic_subp_decl_p_subp_decl_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_basic_subp_decl_p_subp_decl_spec(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_classic_subp_decl_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_classic_subp_decl_f_overriding(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_classic_subp_decl_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_classic_subp_decl_f_subp_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_classic_subp_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_classic_subp_decl_p_body_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_formal_subp_decl_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_formal_subp_decl_f_default_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_entry_decl_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_decl_f_overriding(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_decl_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_decl_f_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_decl_p_body_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_decl_p_accept_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_decl_p_accept_stmts(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_enum_literal_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_enum_literal_decl_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_enum_literal_decl_p_enum_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_enum_literal_decl_p_enum_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_enum_literal_decl_p_enum_rep langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_enum_literal_decl_p_enum_rep(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_synthetic_char_enum_lit_p_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_char_enum_lit_p_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_subp_internal_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_internal_f_subp_spec(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_synthetic_subp_decl_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_subp_decl_f_spec(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_body_node_p_previous_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_body_node_p_previous_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_body_node_p_decl_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_body_node_p_decl_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_body_node_p_subunit_root langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_body_node_p_subunit_root(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_accept_stmt_body_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_accept_stmt_body_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_subp_body_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_body_f_overriding(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_subp_body_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_body_f_subp_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_subp_body_p_is_subject_to_proof langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_body_p_is_subject_to_proof(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_base_subp_body_p_has_spark_mode_on langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_subp_body_p_has_spark_mode_on(
            EntityNative node,
            CCharPointer result
        );

        
    

            

        /** Isomethod of ada_expr_function_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_function_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_subp_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_body_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subp_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_body_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subp_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_body_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_subp_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_renaming_decl_f_renames(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_body_stub_p_syntactic_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_body_stub_p_syntactic_fully_qualified_name(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_package_body_stub_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_body_stub_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_protected_body_stub_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_body_stub_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_subp_body_stub_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_body_stub_f_overriding(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subp_body_stub_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subp_body_stub_f_subp_spec(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_task_body_stub_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_body_stub_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_entry_body_f_entry_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_body_f_entry_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_body_f_index_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_body_f_index_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_body_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_body_f_params(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_body_f_barrier langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_body_f_barrier(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_body_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_body_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_body_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_package_body_f_package_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_body_f_package_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_package_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_body_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_package_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_body_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_package_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_body_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_protected_body_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_body_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_protected_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_body_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_protected_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_body_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_task_body_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_body_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_task_body_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_body_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_task_body_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_body_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_task_body_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_body_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_entry_index_spec_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_index_spec_f_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_entry_index_spec_f_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_entry_index_spec_f_subtype(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_exception_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_exception_decl_f_ids(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_exception_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_exception_decl_f_renames(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_exception_handler_f_exception_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_exception_handler_f_exception_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_exception_handler_f_handled_exceptions langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_exception_handler_f_handled_exceptions(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_exception_handler_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_exception_handler_f_stmts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_for_loop_var_decl_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_for_loop_var_decl_f_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_for_loop_var_decl_f_id_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_for_loop_var_decl_f_id_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_decl_f_formal_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_decl_f_formal_part(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_package_decl_f_package_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_package_decl_f_package_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_package_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_package_decl_p_body_part(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_subp_decl_f_subp_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_decl_f_subp_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_decl_p_body_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_decl_p_body_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_instantiation_p_designated_generic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_instantiation_p_designated_generic_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_instantiation_p_inst_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_instantiation_p_inst_params(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_generic_package_instantiation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_package_instantiation_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_package_instantiation_f_generic_pkg_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_package_instantiation_f_generic_pkg_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_package_instantiation_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_package_instantiation_f_params(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_subp_instantiation_f_overriding langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_instantiation_f_overriding(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_instantiation_f_kind(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_subp_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_instantiation_f_subp_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_generic_subp_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_instantiation_f_generic_subp_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_instantiation_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_instantiation_f_params(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_instantiation_p_designated_subp langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_instantiation_p_designated_subp(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_generic_package_renaming_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_package_renaming_decl_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_package_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_package_renaming_decl_f_renames(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_generic_subp_renaming_decl_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_renaming_decl_f_kind(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_renaming_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_renaming_decl_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_generic_subp_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_generic_subp_renaming_decl_f_renames(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_label_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_label_decl_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_named_stmt_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_named_stmt_decl_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_number_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_number_decl_f_ids(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_number_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_number_decl_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_object_decl_f_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_f_ids(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_f_has_aliased langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_f_has_aliased(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_f_has_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_f_has_constant(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_f_mode langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_f_mode(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_f_type_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_f_default_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_f_default_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_f_renaming_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_f_renaming_clause(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_p_private_part_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_p_private_part_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_object_decl_p_public_part_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_object_decl_p_public_part_decl(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_package_renaming_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_renaming_decl_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_package_renaming_decl_f_renames langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_renaming_decl_f_renames(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_package_renaming_decl_p_renamed_package langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_renaming_decl_p_renamed_package(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_package_renaming_decl_p_final_renamed_package langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_package_renaming_decl_p_final_renamed_package(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_single_protected_decl_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_single_protected_decl_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_single_protected_decl_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_single_protected_decl_f_interfaces(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_single_protected_decl_f_definition langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_single_protected_decl_f_definition(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_single_task_decl_f_task_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_single_task_decl_f_task_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_case_stmt_alternative_f_choices langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_stmt_alternative_f_choices(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_case_stmt_alternative_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_stmt_alternative_f_stmts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_compilation_unit_f_prelude langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_f_prelude(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_compilation_unit_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_f_body(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_compilation_unit_f_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_f_pragmas(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_compilation_unit_p_syntactic_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_syntactic_fully_qualified_name(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_unit_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_unit_kind(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_withed_units langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_withed_units(
            EntityNative node,
            byte include_privates,
            Pointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_imported_units langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_imported_units(
            EntityNative node,
            byte include_privates,
            Pointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_unit_dependencies langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_unit_dependencies(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_compilation_unit_p_is_preelaborable langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_is_preelaborable(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_other_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_other_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_compilation_unit_p_has_restriction langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_has_restriction(
            EntityNative node,
            SymbolNative name,
            CCharPointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_all_config_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_all_config_pragmas(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_compilation_unit_p_config_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_compilation_unit_p_config_pragmas(
            EntityNative node,
            SymbolNative name,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_component_clause_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_clause_f_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_component_clause_f_position langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_clause_f_position(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_component_clause_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_clause_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_component_def_f_has_aliased langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_def_f_has_aliased(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_component_def_f_has_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_def_f_has_constant(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_component_def_f_type_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_component_def_f_type_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_constant_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_constant_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_composite_constraint_f_constraints langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_composite_constraint_f_constraints(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_composite_constraint_p_is_index_constraint langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_composite_constraint_p_is_index_constraint(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_composite_constraint_p_is_discriminant_constraint langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_composite_constraint_p_is_discriminant_constraint(
            EntityNative node,
            CCharPointer result
        );

        
    

            

        /** Isomethod of ada_delta_constraint_f_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_delta_constraint_f_digits(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_delta_constraint_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_delta_constraint_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_digits_constraint_f_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_digits_constraint_f_digits(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_digits_constraint_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_digits_constraint_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_range_constraint_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_range_constraint_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_declarative_part_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_declarative_part_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_elsif_expr_part_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_elsif_expr_part_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_elsif_expr_part_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_elsif_expr_part_f_then_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_elsif_stmt_part_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_elsif_stmt_part_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_elsif_stmt_part_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_elsif_stmt_part_f_stmts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_expr_p_expression_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_expression_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_expr_p_expected_expression_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_expected_expression_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_expr_p_is_dynamically_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_is_dynamically_tagged(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_expr_p_is_dispatching_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_is_dispatching_call(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_expr_p_is_static_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_is_static_expr(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_expr_p_first_corresponding_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_first_corresponding_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_expr_p_eval_as_int langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_eval_as_int(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_expr_p_eval_as_int_in_env langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_eval_as_int_in_env(
            EntityNative node,
            SubstitutionArrayNative env,
            Pointer result
        );
            

        /** Isomethod of ada_expr_p_eval_as_string langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_eval_as_string(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_expr_p_eval_as_string_in_env langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_eval_as_string_in_env(
            EntityNative node,
            SubstitutionArrayNative env,
            Pointer result
        );
            

        /** Isomethod of ada_expr_p_matching_nodes langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_expr_p_matching_nodes(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_abstract_state_decl_expr_f_state_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_abstract_state_decl_expr_f_state_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_allocator_f_subpool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_allocator_f_subpool(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_allocator_f_type_or_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_allocator_f_type_or_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_allocator_p_get_allocated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_allocator_p_get_allocated_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_aggregate_f_ancestor_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_aggregate_f_ancestor_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_aggregate_f_assocs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_aggregate_f_assocs(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_aggregate_p_aggregate_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_aggregate_p_aggregate_params(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_base_aggregate_p_is_subaggregate langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_aggregate_p_is_subaggregate(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_bin_op_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_bin_op_f_left(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_bin_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_bin_op_f_op(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_bin_op_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_bin_op_f_right(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_case_expr_alternative_f_choices langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_expr_alternative_f_choices(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_case_expr_alternative_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_expr_alternative_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_concat_op_f_first_operand langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_concat_op_f_first_operand(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_concat_op_f_other_operands langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_concat_op_f_other_operands(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_concat_op_p_operands langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_concat_op_p_operands(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_concat_operand_f_operator langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_concat_operand_f_operator(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_concat_operand_f_operand langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_concat_operand_f_operand(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_cond_expr_p_dependent_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_cond_expr_p_dependent_exprs(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_case_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_expr_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_case_expr_f_cases langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_expr_f_cases(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_if_expr_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_expr_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_if_expr_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_expr_f_then_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_if_expr_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_expr_f_alternatives(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_if_expr_f_else_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_expr_f_else_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_contract_cases_f_contract_cases langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_contract_cases_f_contract_cases(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_decl_expr_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decl_expr_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_decl_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decl_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_membership_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_membership_expr_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_membership_expr_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_membership_expr_f_op(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_membership_expr_f_membership_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_membership_expr_f_membership_exprs(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_name_p_enclosing_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_enclosing_defining_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_name_p_is_defining langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_defining(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_name_is langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_name_is(
            EntityNative node,
            SymbolNative sym,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_is_direct_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_direct_call(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_is_access_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_access_call(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_is_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_call(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_is_dot_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_dot_call(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_failsafe_referenced_def_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_failsafe_referenced_def_name(
            EntityNative node,
            byte imprecise_fallback,
            RefdDefNative result
        );
            

        /** Isomethod of ada_name_p_referenced_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_referenced_defining_name(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_name_p_all_env_elements langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_all_env_elements(
            EntityNative node,
            byte seq,
            EntityNative seq_from,
            Pointer result
        );
            

        /** Isomethod of ada_name_p_called_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_called_subp_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_name_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_referenced_decl(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_name_p_failsafe_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_failsafe_referenced_decl(
            EntityNative node,
            byte imprecise_fallback,
            RefdDeclNative result
        );
            

        /** Isomethod of ada_name_p_name_designated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_name_designated_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_name_p_is_static_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_static_subtype(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_name_matches langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_name_matches(
            EntityNative node,
            EntityNative n,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_relative_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_relative_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_name_p_is_operator_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_operator_name(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_is_write_reference langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_write_reference(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_is_static_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_static_call(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_as_symbol_array langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_as_symbol_array(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_name_p_canonical_text langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_canonical_text(
            EntityNative node,
            SymbolNative result
        );
            

        /** Isomethod of ada_name_p_is_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_is_constant(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_name_p_call_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_name_p_call_params(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_attribute_ref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_attribute_ref_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_attribute_ref_f_attribute langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_attribute_ref_f_attribute(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_attribute_ref_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_attribute_ref_f_args(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_call_expr_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_call_expr_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_call_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_call_expr_f_suffix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_call_expr_p_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_call_expr_p_kind(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of ada_call_expr_p_is_array_slice langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_call_expr_p_is_array_slice(
            EntityNative node,
            CCharPointer result
        );

        
    

            

        /** Isomethod of ada_defining_name_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_canonical_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_canonical_fully_qualified_name(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_unique_identifying_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_unique_identifying_name(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_fully_qualified_name_array langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_fully_qualified_name_array(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_fully_qualified_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_fully_qualified_name(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_basic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_basic_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_find_refs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_find_refs(
            EntityNative node,
            EntityNative root,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_find_all_references langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_find_all_references(
            EntityNative node,
            AnalysisUnitArrayNative units,
            byte follow_renamings,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_find_all_calls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_find_all_calls(
            EntityNative node,
            AnalysisUnitArrayNative units,
            byte follow_renamings,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_next_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_next_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_previous_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_previous_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_canonical_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_canonical_part(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_most_visible_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_most_visible_part(
            EntityNative node,
            EntityNative origin,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_all_parts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_all_parts(
            EntityNative node,
            byte imprecise_fallback,
            Pointer result
        );
            

        /** Isomethod of ada_defining_name_p_get_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_get_aspect(
            EntityNative node,
            SymbolNative name,
            byte imprecise_fallback,
            AspectNative result
        );
            

        /** Isomethod of ada_defining_name_p_has_aspect langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_has_aspect(
            EntityNative node,
            SymbolNative name,
            byte imprecise_fallback,
            CCharPointer result
        );
            

        /** Isomethod of ada_defining_name_p_get_pragma langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_get_pragma(
            EntityNative node,
            SymbolNative name,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_get_representation_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_get_representation_clause(
            EntityNative node,
            SymbolNative name,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_get_at_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_get_at_clause(
            EntityNative node,
            byte imprecise_fallback,
            EntityNative result
        );
            

        /** Isomethod of ada_defining_name_p_is_imported langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_is_imported(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_defining_name_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_defining_name_p_is_ghost_code(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    

            

        /** Isomethod of ada_discrete_subtype_name_f_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_discrete_subtype_name_f_subtype(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_dotted_name_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_dotted_name_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_dotted_name_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_dotted_name_f_suffix(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_end_name_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_end_name_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_end_name_p_basic_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_end_name_p_basic_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_explicit_deref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_explicit_deref_f_prefix(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_qual_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_qual_expr_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_qual_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_qual_expr_f_suffix(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_reduce_attribute_ref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_reduce_attribute_ref_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_reduce_attribute_ref_f_attribute langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_reduce_attribute_ref_f_attribute(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_reduce_attribute_ref_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_reduce_attribute_ref_f_args(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_char_literal_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_char_literal_p_denoted_value(
            EntityNative node,
            CIntPointer result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_string_literal_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_string_literal_p_denoted_value(
            EntityNative node,
            Pointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_int_literal_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_int_literal_p_denoted_value(
            EntityNative node,
            Pointer result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_update_attribute_ref_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_update_attribute_ref_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_update_attribute_ref_f_attribute langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_update_attribute_ref_f_attribute(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_update_attribute_ref_f_values langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_update_attribute_ref_f_values(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_paren_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_paren_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_quantified_expr_f_quantifier langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_quantified_expr_f_quantifier(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_quantified_expr_f_loop_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_quantified_expr_f_loop_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_quantified_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_quantified_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_raise_expr_f_exception_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_raise_expr_f_exception_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_raise_expr_f_error_message langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_raise_expr_f_error_message(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_un_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_un_op_f_op(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_un_op_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_un_op_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_handled_stmts_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_handled_stmts_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_handled_stmts_f_exceptions langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_handled_stmts_f_exceptions(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_library_item_f_has_private langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_library_item_f_has_private(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_library_item_f_item langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_library_item_f_item(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_limited_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_limited_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_for_loop_spec_f_var_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_for_loop_spec_f_var_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_for_loop_spec_f_loop_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_for_loop_spec_f_loop_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_for_loop_spec_f_has_reverse langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_for_loop_spec_f_has_reverse(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_for_loop_spec_f_iter_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_for_loop_spec_f_iter_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_for_loop_spec_f_iter_filter langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_for_loop_spec_f_iter_filter(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_while_loop_spec_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_while_loop_spec_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_multi_abstract_state_decl_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_multi_abstract_state_decl_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_not_null_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_not_null_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_params_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_params_f_params(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_paren_abstract_state_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_paren_abstract_state_decl_f_decl(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_pp_elsif_directive_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pp_elsif_directive_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_pp_elsif_directive_f_then_kw langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pp_elsif_directive_f_then_kw(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_pp_if_directive_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pp_if_directive_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_pp_if_directive_f_then_kw langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pp_if_directive_f_then_kw(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_pragma_node_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pragma_node_f_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_pragma_node_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pragma_node_f_args(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_pragma_node_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pragma_node_p_is_ghost_code(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of ada_pragma_node_p_associated_entities langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_pragma_node_p_associated_entities(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of ada_private_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_private_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_protected_def_f_public_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_def_f_public_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_protected_def_f_private_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_def_f_private_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_protected_def_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_def_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_protected_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_protected_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_range_spec_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_range_spec_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_renaming_clause_f_renamed_object langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_renaming_clause_f_renamed_object(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_reverse_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_reverse_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_select_when_part_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_select_when_part_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_select_when_part_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_select_when_part_f_stmts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_stmt_p_is_ghost_code langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_stmt_p_is_ghost_code(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    

            

        /** Isomethod of ada_accept_stmt_f_body_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_accept_stmt_f_body_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_accept_stmt_f_entry_index_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_accept_stmt_f_entry_index_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_accept_stmt_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_accept_stmt_f_params(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_accept_stmt_p_corresponding_entry langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_accept_stmt_p_corresponding_entry(
            EntityNative node,
            EntityNative origin,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_accept_stmt_with_stmts_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_accept_stmt_with_stmts_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_accept_stmt_with_stmts_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_accept_stmt_with_stmts_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_base_loop_stmt_f_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_loop_stmt_f_spec(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_loop_stmt_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_loop_stmt_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_base_loop_stmt_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_base_loop_stmt_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_begin_block_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_begin_block_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_begin_block_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_begin_block_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_decl_block_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decl_block_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_decl_block_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decl_block_f_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_decl_block_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decl_block_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_case_stmt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_stmt_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_case_stmt_f_pragmas langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_stmt_f_pragmas(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_case_stmt_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_case_stmt_f_alternatives(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_extended_return_stmt_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_extended_return_stmt_f_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_extended_return_stmt_f_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_extended_return_stmt_f_stmts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_if_stmt_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_stmt_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_if_stmt_f_then_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_stmt_f_then_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_if_stmt_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_stmt_f_alternatives(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_if_stmt_f_else_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_if_stmt_f_else_stmts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_named_stmt_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_named_stmt_f_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_named_stmt_f_stmt langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_named_stmt_f_stmt(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_select_stmt_f_guards langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_select_stmt_f_guards(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_select_stmt_f_else_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_select_stmt_f_else_stmts(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_select_stmt_f_abort_stmts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_select_stmt_f_abort_stmts(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_abort_stmt_f_names langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_abort_stmt_f_names(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_assign_stmt_f_dest langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_assign_stmt_f_dest(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_assign_stmt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_assign_stmt_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_call_stmt_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_call_stmt_f_call(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_delay_stmt_f_has_until langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_delay_stmt_f_has_until(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_delay_stmt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_delay_stmt_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_exit_stmt_f_loop_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_exit_stmt_f_loop_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_exit_stmt_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_exit_stmt_f_cond_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_goto_stmt_f_label_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_goto_stmt_f_label_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_label_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_label_f_decl(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_raise_stmt_f_exception_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_raise_stmt_f_exception_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_raise_stmt_f_error_message langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_raise_stmt_f_error_message(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_requeue_stmt_f_call_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_requeue_stmt_f_call_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_requeue_stmt_f_has_abort langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_requeue_stmt_f_has_abort(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_return_stmt_f_return_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_return_stmt_f_return_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_subunit_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subunit_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subunit_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subunit_f_body(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subunit_p_body_root langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subunit_p_body_root(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_synchronized_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synchronized_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_tagged_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_tagged_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_task_def_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_def_f_interfaces(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_task_def_f_public_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_def_f_public_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_task_def_f_private_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_def_f_private_part(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_task_def_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_task_def_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_access_def_f_has_not_null langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_access_def_f_has_not_null(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_access_to_subp_def_f_has_protected langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_access_to_subp_def_f_has_protected(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_access_to_subp_def_f_subp_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_access_to_subp_def_f_subp_spec(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_anonymous_type_access_def_f_type_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_anonymous_type_access_def_f_type_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_type_access_def_f_has_all langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_access_def_f_has_all(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_type_access_def_f_has_constant langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_access_def_f_has_constant(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_type_access_def_f_subtype_indication langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_access_def_f_subtype_indication(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_array_type_def_f_indices langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_array_type_def_f_indices(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_array_type_def_f_component_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_array_type_def_f_component_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_derived_type_def_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_derived_type_def_f_has_abstract(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_derived_type_def_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_derived_type_def_f_has_limited(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_derived_type_def_f_has_synchronized langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_derived_type_def_f_has_synchronized(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_derived_type_def_f_subtype_indication langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_derived_type_def_f_subtype_indication(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_derived_type_def_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_derived_type_def_f_interfaces(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_derived_type_def_f_record_extension langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_derived_type_def_f_record_extension(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_derived_type_def_f_has_with_private langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_derived_type_def_f_has_with_private(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_enum_type_def_f_enum_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_enum_type_def_f_enum_literals(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_interface_type_def_f_interface_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_interface_type_def_f_interface_kind(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_interface_type_def_f_interfaces langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_interface_type_def_f_interfaces(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_mod_int_type_def_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_mod_int_type_def_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_private_type_def_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_private_type_def_f_has_abstract(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_private_type_def_f_has_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_private_type_def_f_has_tagged(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_private_type_def_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_private_type_def_f_has_limited(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_decimal_fixed_point_def_f_delta langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decimal_fixed_point_def_f_delta(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_decimal_fixed_point_def_f_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decimal_fixed_point_def_f_digits(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_decimal_fixed_point_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_decimal_fixed_point_def_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_floating_point_def_f_num_digits langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_floating_point_def_f_num_digits(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_floating_point_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_floating_point_def_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_ordinary_fixed_point_def_f_delta langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ordinary_fixed_point_def_f_delta(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_ordinary_fixed_point_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_ordinary_fixed_point_def_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_record_type_def_f_has_abstract langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_record_type_def_f_has_abstract(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_record_type_def_f_has_tagged langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_record_type_def_f_has_tagged(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_record_type_def_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_record_type_def_f_has_limited(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_record_type_def_f_record_def langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_record_type_def_f_record_def(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_signed_int_type_def_f_range langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_signed_int_type_def_f_range(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_type_expr_p_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_expr_p_type_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_type_expr_p_designated_type_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_expr_p_designated_type_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_type_expr_p_designated_type_decl_from langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_type_expr_p_designated_type_decl_from(
            EntityNative node,
            EntityNative origin_node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_anonymous_type_f_type_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_anonymous_type_f_type_decl(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of ada_subtype_indication_f_has_not_null langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subtype_indication_f_has_not_null(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subtype_indication_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subtype_indication_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subtype_indication_f_constraint langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subtype_indication_f_constraint(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_subtype_indication_p_subtype_constraints langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subtype_indication_p_subtype_constraints(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of ada_subtype_indication_p_is_static_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_subtype_indication_p_is_static_subtype(
            EntityNative node,
            byte imprecise_fallback,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of ada_synthetic_type_expr_f_target_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_synthetic_type_expr_f_target_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_unconstrained_array_index_f_subtype_indication langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_unconstrained_array_index_f_subtype_indication(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_until_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_until_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of ada_use_package_clause_f_packages langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_use_package_clause_f_packages(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_use_type_clause_f_has_all langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_use_type_clause_f_has_all(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_use_type_clause_f_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_use_type_clause_f_types(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_value_sequence_f_iter_assoc langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_value_sequence_f_iter_assoc(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_variant_f_choices langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_variant_f_choices(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_variant_f_components langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_variant_f_components(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_variant_part_f_discr_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_variant_part_f_discr_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_variant_part_f_variant langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_variant_part_f_variant(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_with_clause_f_has_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_with_clause_f_has_limited(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_with_clause_f_has_private langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_with_clause_f_has_private(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of ada_with_clause_f_packages langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_with_clause_f_packages(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of ada_with_private_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int ada_with_private_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    



    }


    // ==========
    // Exceptions
    // ==========

    /**
     * This class represents exception during symbol manipulation.
     */
    public static final class SymbolException extends RuntimeException {
        public SymbolException(
            final String symbol
        ) {
            super("Invalid symbol : '" + symbol + "'");
        }
    }

    /**
     * This class reprsents exception during enum manipulation.
     */
    public static final class EnumException extends RuntimeException {
        public EnumException(
            final String msg
        ) {
            super(msg);
        }
    }

    /**
     * This class represents an exception in the references manipulation.
     */
    public static final class ReferenceException extends RuntimeException {
        public ReferenceException(
            final String msg
        ) {
            super(msg);
        }
    }

    /**
     * This class wraps the exceptions from the langkit native library.
     */
    public static class LangkitException extends RuntimeException {

        // ----- Instance attributes -----

        /** The kind of the langkit exception. */
        public final ExceptionKind kind;

        // ----- Constructors -----

        /**
         * Create a new langkit exception.
         *
         * @param kind The kind of the exception represented by an integer
         * which will be mapped to an enum value.
         * @param message The message of the exception.
         */
        public LangkitException(
            final int kind,
            final String message
        ) {
            super(message);
            this.kind = ExceptionKind.fromC(kind);
        }

    }

    // ==========
    // Enum definitions
    // ==========

    // ===== Constants enumeration =====

    /**
     * Kind for this token.
     */
    public enum TokenKind {

        // ----- Enum values -----

        NO_TOKEN(-1, "No_Token"),
        ADA_TERMINATION(0, "Termination"),
        ADA_LEXING_FAILURE(1, "Lexing_Failure"),
        ADA_IDENTIFIER(2, "Identifier"),
        ADA_ALL(3, "All"),
        ADA_ABORT(4, "Abort"),
        ADA_ELSE(5, "Else"),
        ADA_NEW(6, "New"),
        ADA_RETURN(7, "Return"),
        ADA_ABS(8, "Abs"),
        ADA_ELSIF(9, "Elsif"),
        ADA_NOT(10, "Not"),
        ADA_REVERSE(11, "Reverse"),
        ADA_END(12, "End"),
        ADA_NULL(13, "Null"),
        ADA_ACCEPT(14, "Accept"),
        ADA_ENTRY(15, "Entry"),
        ADA_SELECT(16, "Select"),
        ADA_ACCESS(17, "Access"),
        ADA_EXCEPTION(18, "Exception"),
        ADA_OF(19, "Of"),
        ADA_SEPARATE(20, "Separate"),
        ADA_EXIT(21, "Exit"),
        ADA_OR(22, "Or"),
        ADA_OTHERS(23, "Others"),
        ADA_SUBTYPE(24, "Subtype"),
        ADA_AND(25, "And"),
        ADA_FOR(26, "For"),
        ADA_OUT(27, "Out"),
        ADA_ARRAY(28, "Array"),
        ADA_FUNCTION(29, "Function"),
        ADA_AT(30, "At"),
        ADA_GENERIC(31, "Generic"),
        ADA_PACKAGE(32, "Package"),
        ADA_TASK(33, "Task"),
        ADA_BEGIN(34, "Begin"),
        ADA_GOTO(35, "Goto"),
        ADA_PRAGMA(36, "Pragma"),
        ADA_TERMINATE(37, "Terminate"),
        ADA_BODY(38, "Body"),
        ADA_PRIVATE(39, "Private"),
        ADA_THEN(40, "Then"),
        ADA_IF(41, "If"),
        ADA_PROCEDURE(42, "Procedure"),
        ADA_TYPE(43, "Type"),
        ADA_CASE(44, "Case"),
        ADA_IN(45, "In"),
        ADA_CONSTANT(46, "Constant"),
        ADA_IS(47, "Is"),
        ADA_RAISE(48, "Raise"),
        ADA_USE(49, "Use"),
        ADA_DECLARE(50, "Declare"),
        ADA_RANGE(51, "Range"),
        ADA_DELAY(52, "Delay"),
        ADA_LIMITED(53, "Limited"),
        ADA_RECORD(54, "Record"),
        ADA_WHEN(55, "When"),
        ADA_DELTA(56, "Delta"),
        ADA_LOOP(57, "Loop"),
        ADA_REM(58, "Rem"),
        ADA_WHILE(59, "While"),
        ADA_DIGITS(60, "Digits"),
        ADA_RENAMES(61, "Renames"),
        ADA_DO(62, "Do"),
        ADA_MOD(63, "Mod"),
        ADA_XOR(64, "Xor"),
        ADA_PAR_CLOSE(65, "Par_Close"),
        ADA_PAR_OPEN(66, "Par_Open"),
        ADA_BRACK_CLOSE(67, "Brack_Close"),
        ADA_BRACK_OPEN(68, "Brack_Open"),
        ADA_SEMICOLON(69, "Semicolon"),
        ADA_COLON(70, "Colon"),
        ADA_COMMA(71, "Comma"),
        ADA_DOUBLEDOT(72, "Doubledot"),
        ADA_DOT(73, "Dot"),
        ADA_DIAMOND(74, "Diamond"),
        ADA_LTE(75, "Lte"),
        ADA_GTE(76, "Gte"),
        ADA_ARROW(77, "Arrow"),
        ADA_EQUAL(78, "Equal"),
        ADA_LT(79, "Lt"),
        ADA_GT(80, "Gt"),
        ADA_PLUS(81, "Plus"),
        ADA_MINUS(82, "Minus"),
        ADA_POWER(83, "Power"),
        ADA_MULT(84, "Mult"),
        ADA_AMP(85, "Amp"),
        ADA_NOTEQUAL(86, "Notequal"),
        ADA_DIVIDE(87, "Divide"),
        ADA_TICK(88, "Tick"),
        ADA_PIPE(89, "Pipe"),
        ADA_ASSIGN(90, "Assign"),
        ADA_LABEL_START(91, "Label_Start"),
        ADA_LABEL_END(92, "Label_End"),
        ADA_TARGET(93, "Target"),
        ADA_STRING(94, "String"),
        ADA_CHAR(95, "Char"),
        ADA_WITH(96, "With"),
        ADA_DECIMAL(97, "Decimal"),
        ADA_INTEGER(98, "Integer"),
        ADA_COMMENT(99, "Comment"),
        ADA_PREP_LINE(100, "Prep_Line"),
        ADA_WHITESPACE(101, "Whitespace"),
        ;

        // ----- Class attributes -----

        /** Singleton that represents the none token kind. */
        public static final TokenKind NONE = NO_TOKEN;

        /** The map from int to enum values. */
        private static final Map<Integer, TokenKind> map = new HashMap<>();

        // ----- Instance attributes -----

        /** The value of the enum instance. */
        private final int value;

        /** The name of the enum instance in the Langkit DSL. */
        public final String name;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(TokenKind elem : TokenKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private TokenKind(
            final int value,
            final String name
        ) {
            this.value = value;
            this.name = name;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static TokenKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get TokenKind from " + cValue
                );
            return (TokenKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    /**
     * Enumerated type describing all possible exceptions that need to be
     * handled in the C bindings.
     */
    public enum ExceptionKind {

        // ----- Enum values -----

        FILE_READ_ERROR(0),
        BAD_TYPE_ERROR(1),
        OUT_OF_BOUNDS_ERROR(2),
        INVALID_INPUT(3),
        INVALID_SYMBOL_ERROR(4),
        INVALID_UNIT_NAME_ERROR(5),
        NATIVE_EXCEPTION(6),
        PRECONDITION_FAILURE(7),
        PROPERTY_ERROR(8),
        TEMPLATE_ARGS_ERROR(9),
        TEMPLATE_FORMAT_ERROR(10),
        TEMPLATE_INSTANTIATION_ERROR(11),
        STALE_REFERENCE_ERROR(12),
        SYNTAX_ERROR(13),
        UNKNOWN_CHARSET(14),
        MALFORMED_TREE_ERROR(15),
        INVALID_PROJECT(16),
        UNSUPPORTED_VIEW_ERROR(17),
        ;

        // ----- Class attributes

        /** Singleton that represents the none expcetion kind. */
        public static final ExceptionKind NONE =
            FILE_READ_ERROR;

        /** The map from int to enum values. */
        private static final Map<Integer, ExceptionKind> map =
            new HashMap<>();

        // ----- Instance ttributes -----

        /** The value of the enum instance. */
        private final int value;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(ExceptionKind elem : ExceptionKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private ExceptionKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static ExceptionKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get ExceptionKind from " + cValue
                );
            return (ExceptionKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    // ===== Generated enums =====

    
    

    /**
     * Specify a kind of analysis unit. Specification units provide an
     * interface to the outer world while body units provide an implementation
     * for the corresponding interface.
     */
    public enum AnalysisUnitKind {

        // ----- Enum values -----

        UNIT_SPECIFICATION(0),
        UNIT_BODY(1),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final AnalysisUnitKind NONE =
            UNIT_SPECIFICATION;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, AnalysisUnitKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(AnalysisUnitKind elem : AnalysisUnitKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private AnalysisUnitKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static AnalysisUnitKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get AnalysisUnitKind from " + cValue
                );
            return (AnalysisUnitKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**

     */
    public enum LookupKind {

        // ----- Enum values -----

        RECURSIVE(0),
        FLAT(1),
        MINIMAL(2),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final LookupKind NONE =
            RECURSIVE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, LookupKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(LookupKind elem : LookupKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private LookupKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static LookupKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get LookupKind from " + cValue
                );
            return (LookupKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Discriminant for DesignatedEnv structures.
     */
    public enum DesignatedEnvKind {

        // ----- Enum values -----

        NONE_ENUM(0),
        CURRENT_ENV(1),
        NAMED_ENV(2),
        DIRECT_ENV(3),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final DesignatedEnvKind NONE =
            NONE_ENUM;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, DesignatedEnvKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(DesignatedEnvKind elem : DesignatedEnvKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private DesignatedEnvKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static DesignatedEnvKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get DesignatedEnvKind from " + cValue
                );
            return (DesignatedEnvKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Kind for the result of a cross reference operation.
     *
     * * ``no_ref`` is for no reference, it is the null value for this enum.
     *
     * * ``precise`` is when the reference result is precise.
     *
     * * ``imprecise`` is when there was an error computing the precise result,
     *   and a result was gotten in an imprecise fashion.
     *
     * * ``error`` is for unrecoverable errors (either there is no imprecise
     *   path for the request you made, or the imprecise path errored out too).
     */
    public enum RefResultKind {

        // ----- Enum values -----

        NO_REF(0),
        PRECISE(1),
        IMPRECISE(2),
        ERROR(3),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final RefResultKind NONE =
            NO_REF;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, RefResultKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(RefResultKind elem : RefResultKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private RefResultKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static RefResultKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get RefResultKind from " + cValue
                );
            return (RefResultKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Kind of CallExpr type.
     *
     * * ``call`` is when the CallExpr is a procedure or function call.
     *
     * * ``array_slice``, ``array_index`` is when the CallExpr is in fact an
     *   array slice or an array subcomponent access expression, respectively.
     *
     * * ``type_conversion`` is when the CallExpr is a type conversion.
     */
    public enum CallExprKind {

        // ----- Enum values -----

        CALL(0),
        ARRAY_SLICE(1),
        ARRAY_INDEX(2),
        TYPE_CONVERSION(3),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final CallExprKind NONE =
            CALL;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, CallExprKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(CallExprKind elem : CallExprKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private CallExprKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static CallExprKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get CallExprKind from " + cValue
                );
            return (CallExprKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Gramar rule to use for parsing.
     */
    public enum GrammarRule {

        // ----- Enum values -----

        PARENT_LIST_RULE(0),
        PROTECTED_TYPE_DECL_RULE(1),
        PROTECTED_OP_RULE(2),
        PROTECTED_EL_RULE(3),
        PROTECTED_DEF_RULE(4),
        PROTECTED_DECL_RULE(5),
        TASK_ITEM_RULE(6),
        TASK_DEF_RULE(7),
        TASK_TYPE_DECL_RULE(8),
        SUBTYPE_DECL_RULE(9),
        INTERFACE_TYPE_DEF_RULE(10),
        UNCONSTRAINED_INDEX_RULE(11),
        ARRAY_TYPE_DEF_RULE(12),
        DISCRETE_SUBTYPE_DEFINITION_RULE(13),
        CONSTRAINT_LIST_RULE(14),
        SIGNED_INT_TYPE_DEF_RULE(15),
        MOD_INT_TYPE_DEF_RULE(16),
        DERIVED_TYPE_DEF_RULE(17),
        COMPOSITE_CONSTRAINT_ASSOC_RULE(18),
        COMPOSITE_CONSTRAINT_RULE(19),
        DIGITS_CONSTRAINT_RULE(20),
        DELTA_CONSTRAINT_RULE(21),
        RANGE_CONSTRAINT_RULE(22),
        CONSTRAINT_RULE(23),
        DISCRIMINANT_SPEC_RULE(24),
        DISCR_SPEC_LIST_RULE(25),
        DISCRIMINANT_PART_RULE(26),
        ENUM_LITERAL_DECL_RULE(27),
        FORMAL_DISCRETE_TYPE_DEF_RULE(28),
        RECORD_DEF_RULE(29),
        RANGE_SPEC_RULE(30),
        REAL_TYPE_DEF_RULE(31),
        SEXPR_OR_BOX_RULE(32),
        ORDINARY_FIXED_POINT_DEF_RULE(33),
        DECIMAL_FIXED_POINT_DEF_RULE(34),
        FLOATING_POINT_DEF_RULE(35),
        RECORD_TYPE_DEF_RULE(36),
        ACCESS_DEF_RULE(37),
        ENUM_TYPE_DEF_RULE(38),
        TYPE_DEF_RULE(39),
        VARIANT_RULE(40),
        ANONYMOUS_TYPE_DECL_RULE(41),
        INCOMPLETE_TYPE_DECL_RULE(42),
        TYPE_DECL_RULE(43),
        VARIANT_PART_RULE(44),
        COMPONENT_DEF_RULE(45),
        COMPONENT_ITEM_RULE(46),
        COMPONENT_DECL_RULE(47),
        COMPONENT_LIST_RULE(48),
        GENERIC_DECL_RULE(49),
        GENERIC_FORMAL_PART_RULE(50),
        GENERIC_FORMAL_DECL_RULE(51),
        FORMAL_TYPE_DECL_RULE(52),
        FORMAL_SUBP_DECL_RULE(53),
        RENAMING_CLAUSE_RULE(54),
        GENERIC_RENAMING_DECL_RULE(55),
        GENERIC_INSTANTIATION_RULE(56),
        EXCEPTION_DECL_RULE(57),
        BASIC_DECLS_RULE(58),
        PACKAGE_RENAMING_DECL_RULE(59),
        PACKAGE_DECL_RULE(60),
        BASIC_DECL_RULE(61),
        OBJECT_DECL_RULE(62),
        SUB_OBJECT_DECL_RULE(63),
        NO_TYPE_OBJECT_RENAMING_DECL_RULE(64),
        EXT_RET_STMT_OBJECT_DECL_RULE(65),
        DEFINING_ID_LIST_RULE(66),
        NUMBER_DECL_RULE(67),
        CONTRACT_CASE_ASSOC_RULE(68),
        CONTRACT_CASES_EXPR_RULE(69),
        ABSTRACT_STATE_DECL_RULE(70),
        MULTI_ABSTRACT_STATE_DECL_RULE(71),
        ASPECT_ASSOC_RULE(72),
        ASPECT_SPEC_RULE(73),
        SINGLE_TASK_DECL_RULE(74),
        OVERRIDING_INDICATOR_RULE(75),
        ENTRY_DECL_RULE(76),
        COMPONENT_CLAUSE_RULE(77),
        ASPECT_CLAUSE_RULE(78),
        PARAM_SPEC_RULE(79),
        PARAM_SPECS_RULE(80),
        SUBP_SPEC_RULE(81),
        EXPR_FN_RULE(82),
        NULL_SUBP_DECL_RULE(83),
        ABSTRACT_SUBP_DECL_RULE(84),
        SUBP_RENAMING_DECL_RULE(85),
        SIMPLE_SUBP_DECL_RULE(86),
        SUBP_DECL_RULE(87),
        WITH_CLAUSE_RULE(88),
        CONTEXT_ITEM_RULE(89),
        USE_CLAUSE_RULE(90),
        USE_PACKAGE_CLAUSE_RULE(91),
        USE_TYPE_CLAUSE_RULE(92),
        SUBTYPE_INDICATION_RULE(93),
        DISCRETE_SUBTYPE_INDICATION_RULE(94),
        CONSTRAINED_SUBTYPE_INDICATION_RULE(95),
        TYPE_EXPR_RULE(96),
        ANONYMOUS_TYPE_RULE(97),
        MODE_RULE(98),
        PRAGMA_ARGUMENT_RULE(99),
        PRAGMA_RULE(100),
        SUBUNIT_RULE(101),
        LIBRARY_UNIT_BODY_RULE(102),
        LIBRARY_UNIT_RENAMING_DECL_RULE(103),
        LIBRARY_ITEM_RULE(104),
        COMPILATION_UNIT_RULE(105),
        COMPILATION_RULE(106),
        DECL_PART_RULE(107),
        ENTRY_BODY_RULE(108),
        PROTECTED_BODY_RULE(109),
        PROTECTED_BODY_STUB_RULE(110),
        TASK_BODY_RULE(111),
        TASK_BODY_STUB_RULE(112),
        PACKAGE_BODY_STUB_RULE(113),
        PACKAGE_BODY_RULE(114),
        TERMINATE_ALTERNATIVE_RULE(115),
        SELECT_STMT_RULE(116),
        ACCEPT_STMT_RULE(117),
        CASE_ALT_RULE(118),
        CASE_STMT_RULE(119),
        EXT_RETURN_STMT_RULE(120),
        IBLOCK_STMT_RULE(121),
        BLOCK_STMT_RULE(122),
        WHILE_LOOP_SPEC_RULE(123),
        ILOOP_STMT_RULE(124),
        LOOP_STMT_RULE(125),
        COMPOUND_STMT_RULE(126),
        ELSIF_PART_RULE(127),
        IF_STMT_RULE(128),
        RAISE_STMT_RULE(129),
        DELAY_STMT_RULE(130),
        ABORT_STMT_RULE(131),
        BODY_RULE(132),
        BODY_STUB_RULE(133),
        SUBP_BODY_STUB_RULE(134),
        RECOV_DECL_PART_RULE(135),
        SUBP_BODY_RULE(136),
        HANDLED_STMTS_RULE(137),
        EXCEPTION_HANDLER_RULE(138),
        STMTS_RULE(139),
        LABEL_RULE(140),
        STMT_RULE(141),
        CALL_STMT_RULE(142),
        SIMPLE_STMT_RULE(143),
        NULL_STMT_RULE(144),
        ASSIGNMENT_STMT_RULE(145),
        GOTO_STMT_RULE(146),
        EXIT_STMT_RULE(147),
        RETURN_STMT_RULE(148),
        REQUEUE_STMT_RULE(149),
        IDENTIFIER_RULE(150),
        CHAR_LITERAL_RULE(151),
        STRING_LITERAL_RULE(152),
        DEFINING_ID_RULE(153),
        DEC_LITERAL_RULE(154),
        INT_LITERAL_RULE(155),
        NUM_LITERAL_RULE(156),
        NULL_LITERAL_RULE(157),
        ALLOCATOR_RULE(158),
        FOR_LOOP_PARAM_SPEC_RULE(159),
        QUANTIFIED_EXPR_RULE(160),
        CASE_EXPR_RULE(161),
        CASE_EXPR_ALT_RULE(162),
        RAISE_EXPR_RULE(163),
        IF_EXPR_RULE(164),
        CONDITIONAL_EXPR_RULE(165),
        BOX_EXPR_RULE(166),
        OTHERS_DESIGNATOR_RULE(167),
        ITERATED_ASSOC_RULE(168),
        AGGREGATE_ASSOC_RULE(169),
        REGULAR_AGGREGATE_RULE(170),
        BRACKET_AGGREGATE_RULE(171),
        AGGREGATE_RULE(172),
        DIRECT_NAME_RULE(173),
        PARAM_ASSOC_RULE(174),
        CALL_SUFFIX_RULE(175),
        ATTR_SUFFIX_RULE(176),
        QUALIFIED_NAME_RULE(177),
        QUAL_NAME_INTERNAL_RULE(178),
        VALUE_SEQUENCE_RULE(179),
        NAME_RULE(180),
        DEFINING_NAME_RULE(181),
        DIRECT_NAME_OR_TARGET_NAME_RULE(182),
        TARGET_NAME_RULE(183),
        UPDATE_ATTR_AGGREGATE_RULE(184),
        UPDATE_ATTR_CONTENT_RULE(185),
        MULTIDIM_ARRAY_ASSOC_RULE(186),
        SUBTYPE_NAME_RULE(187),
        STATIC_NAME_RULE(188),
        PRIMARY_RULE(189),
        PAREN_EXPR_RULE(190),
        DECLARE_EXPR_RULE(191),
        FACTOR_RULE(192),
        TERM_RULE(193),
        UNOP_TERM_RULE(194),
        SIMPLE_EXPR_RULE(195),
        BOOLEAN_OP_RULE(196),
        DISCRETE_RANGE_RULE(197),
        CHOICE_RULE(198),
        CHOICE_LIST_RULE(199),
        REL_OP_RULE(200),
        MEMBERSHIP_CHOICE_RULE(201),
        MEMBERSHIP_CHOICE_LIST_RULE(202),
        RELATION_RULE(203),
        EXPR_RULE(204),
        PP_DIRECTIVE_RULE(205),
        PP_THEN_RULE(206),
        PP_EXPR_RULE(207),
        PP_TERM_RULE(208),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final GrammarRule NONE =
            PARENT_LIST_RULE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, GrammarRule> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(GrammarRule elem : GrammarRule.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private GrammarRule(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static GrammarRule fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get GrammarRule from " + cValue
                );
            return (GrammarRule) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }


    // ==========
    // Java wrapping classes
    // ==========

    // ===== Constant structure wrapping classes =====

    /**
     * This class wraps the langkit characters which are 32 bit wide.
     */
    public static final class Char {

        // ----- Class attributes -----

        /** Singleton that represents the none char. */
        public static final Char NONE = new Char(0);

        // ----- Instance attributes -----

        /** The value of the character. */
        public final int value;

        // ----- Constructors -----

        /**
         * Create a new character from its value. In langkit characters are
         * 32 bit wide so represented by Java integer.
         *
         * @param value The value of the character.
         */
        Char(
            final int value
        ) {
            this.value = value;
        }

        /**
         * Create a character from its integer value.
         *
         * @param value The character value.
         * @return The newly created character.
         */
        public static Char create(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Create a character from a Java character.
         *
         * @param value The source value of the character.
         * @return The newly created character.
         */
        public static Char create(
            final char value
        ) {
            return new Char((int) value);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given NI pointer in a Java class.
         *
         * @param pointer The NI pointer to wrap.
         * @return The wrapped character.
         */
        static Char wrap(
            final CIntPointer pointer
        ) {
            return wrap(pointer.read());
        }

        /**
         * Wrap an integer to a character.
         *
         * @param value The value of the character in an integer.
         * @return The newly created character.
         */
        static Char wrap(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Unwrap the character in the given int pointer.
         *
         * @param pointer The pointer to unwrap the character in.
         */
        void unwrap(
            final CIntPointer pointer
        ) {
            pointer.write(this.value);
        }

        /**
         * Unwrap the character in a Java integer.
         *
         * @return The character value in a Java integer.
         */
        int unwrap() {
            return this.value;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            final ByteBuffer buffer = ByteBuffer.allocate(4);
            buffer.order(BYTE_ORDER);
            buffer.putInt(this.value);
            return decodeUTF32(buffer.array());
        }

    }

    /**
     * Arbitrarily large integer.
     */
    static final class BigIntegerWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none big integer. */
        public static final BigInteger NONE = BigInteger.ZERO;

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer which points to a native big integer.
         *
         * @param pointer The pointer to the native big integer.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final Pointer pointer
        ) {
            return wrap((BigIntegerNative) pointer.readWord(0));
        }

        /**
         * Wrap the given native big integer in a Java big integer.
         *
         * @param bigIntegerNative The big integer native value.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final BigIntegerNative bigIntegerNative
        ) {
            final String representation = getRepresentation(bigIntegerNative);
            return new BigInteger(representation);
        }

        /**
         * Unwrap the given big integer in the given pointer as a native
         * big integer.
         *
         * @param bigInteger The big integer to unwrap.
         * @param pointer The pointer to place the big integer in.
         */
        static void unwrap(
            final BigInteger bigInteger,
            final Pointer pointer
        ) {
            pointer.writeWord(0, unwrap(bigInteger));
        }

        /**
         * Unwrap the given big integer.
         *
         * @param bigInteger The big integer to unwrap.
         * @return The native big integer newly allocated.
         */
        static BigIntegerNative unwrap(
            final BigInteger bigInteger
        ) {
            // Create the representation of the big integer
            final String representation = bigInteger.toString();
            try(final Text bigIntegerText = Text.create(representation)) {
                TextNative bigIntegerTextNative = StackValue.get(
                    TextNative.class
                );
                bigIntegerText.unwrap(bigIntegerTextNative);

                // Create the big intger by calling the native function
                return NI_LIB.ada_create_big_integer(
                    bigIntegerTextNative
                );
            }
        }

        /**
         * Release the big integer pointed by the given pointer.
         *
         * @param pointer The pointer to the big integer to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((BigIntegerNative) pointer.readWord(0));
        }

        /**
         * Release the given native big integer.
         *
         * @param bigIntegerNative The native big integer to release.
         */
        static void release(
            final BigIntegerNative bigIntegerNative
        ) {
            NI_LIB.ada_big_integer_decref(bigIntegerNative);
        }

        // ----- Class methods -----

        /**
         * Get the string representation of the given native big integer.
         *
         * @param bigIntegerNative The native big integer to get the
         * representation from.
         */
        private static String getRepresentation(
            final BigIntegerNative bigIntegerNative
        ) {
            // Allocate the stack value for the text
            final TextNative bigIntegerTextNative = StackValue.get(
                TextNative.class
            );
            Text.NONE.unwrap(bigIntegerTextNative);

            // Call the native function
            NI_LIB.ada_big_integer_text(
                bigIntegerNative,
                bigIntegerTextNative
            );

            // Wrap the text and return the result
            try(final Text bigIntegerText = Text.wrap(bigIntegerTextNative)) {
                return bigIntegerText.getContent();
            }
        }

    }

    /**
     * Reference to a symbol. Symbols are owned by analysis contexts, so they
     * must not outlive them. This type exists only in the C API, and roughly
     * wraps the corresponding Ada type (an array fat pointer).
     */
    public static final class Symbol {

        // ----- Class attributes -----

        /** Singleton that represents the none symbol. */
        public static final Symbol NONE = new Symbol("");

        // ----- Instance attributes

        /** The text of the symbol. */
        public final String text;

        // ----- Constructors -----

        /**
         * Create a new symbol from its text.
         *
         * @param text The symbol text.
         */
        Symbol(
            final String text
        ) {
            this.text = text;
        }

        /**
         * Public access to the symbol creation.
         *
         * @param text The text of the symbol.
         */
        public static Symbol create(
            final String text
        ) {
            return new Symbol(text);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native symbol.
         *
         * @param pointer The pointer to the native symbol.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final Pointer pointer
        ) {
            return wrap((SymbolNative) pointer.readWord(0));
        }

        /**
         * Wrap the given symbol native value in a Java value.
         *
         * @param symbolNative The symbol native value.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final SymbolNative symbolNative
        ) {
            // Get the symbol text
            final TextNative textNative = StackValue.get(TextNative.class);
            Text.NONE.unwrap(textNative);
            NI_LIB.ada_symbol_text(
                symbolNative,
                textNative
            );

            // Return the new symbol
            try(final Text text = Text.wrap(textNative)) {
                return new Symbol(text.getContent());
            }
        }

        /**
         * Unwrap the symbol in the given native structure.
         *
         * @param symbolNative The native structure to unwrap in.
         */
        void unwrap(
            final SymbolNative symbolNative,
            final AnalysisContext context
        ) {
            // Unwrap the symbol text
            try(Text text = Text.create(this.text)) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                text.unwrap(textNative);

                // Call the symbol creation
                final int resCode = NI_LIB.ada_context_symbol(
                    context.reference.ni(),
                    textNative,
                    symbolNative
                );

                // Check the symbol creation success
                if(resCode == 0) {
                    throw new SymbolException(this.text);
                }
            }
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.text;
        }

    }

    /**
     * Type to contain Unicode text data.
     */
    static final class StringWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none string. */
        public static final String NONE = "";

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer that points to a native string.
         *
         * @param pointer The pointer to the native string.
         * @return The Java string wrapped from the native string.
         */
        static String wrap(
            final Pointer pointer
        ) {
            return wrap((StringNative) pointer.readWord(0));
        }

        /**
         * Wrap a native string wrapper in a Java string.
         *
         * @param stringNative The native string to wrap.
         * @return The Java string created from the native one.
         */
        static String wrap(
            final StringNative stringNative
        ) {
            return getNativeStringContent(stringNative);
        }

        /**
         * Unwrap the string in the given pointer.
         *
         * @param string The string to unwrap.
         * @param pointer The pointer to place the native string in.
         */
        static void unwrap(
            final String string,
            final Pointer pointer
        ) {
            pointer.writeWord(0, unwrap(string));
        }

        /**
         * Unwrap the given string in a native one.
         *
         * @param string The string to unwrap.
         * @return The native string unwrapped.
         */
        static StringNative unwrap(
            final String string
        ) {
            return nativeStringFromContent(string);
        }

        /**
         * Release the native string at the given pointer.
         *
         * @param pointer The pointer to the native string.
         */
        static void release(
            final Pointer pointer
        ) {
            release((StringNative) pointer.readWord(0));
        }

        /**
         * Release the given native string.
         *
         * @param stringNative The native string to release.
         */
        static void release(
            final StringNative stringNative
        ) {
            NI_LIB.ada_string_dec_ref(stringNative);
        }

        // ----- Class methods -----

        /**
         * Get the content of the given native string in a Java one.
         *
         * @param stringNative The native string to get the content from.
         * @return The Java string.
         */
        private static String getNativeStringContent(
            final StringNative stringNative
        ) {
            // Prepare the working variable
            final Pointer pointer = (Pointer) stringNative;
            final int length = pointer.readInt(0);
            final byte[] contentArray = new byte[length * 4];

            // Get the content from the native string
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArray[i] = pointer.readByte(i + 8);
            }

            // Return the decoded string
            return decodeUTF32(contentArray);
        }

        /**
         * Create a native string from a Java string.
         *
         * @param string The Java string to create the native string from.
         * @return The native string.
         */
        private static StringNative nativeStringFromContent(
            final String string
        ) {
            // Encode the string in UTF-32
            final byte[] contentArray = encodeUTF32(string);
            final int length = string.length();

            // Create the native array
            final Pointer contentArrayNative = UnmanagedMemory.malloc(
                contentArray.length
            );
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArrayNative.writeByte(i, contentArray[i]);
            }

            // Create the native string
            final StringNative res = NI_LIB.ada_create_string(
                (CIntPointer) contentArrayNative,
                length
            );

            // Free the memory
            UnmanagedMemory.free(contentArrayNative);

            // Return the result
            return res;
        }

    }

    /**
     * String encoded in UTF-32 (native endianness).
     */
    public static final class Text implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none text. */
        public static final Text NONE = new Text(
            PointerWrapper.nullPointer(),
            0,
            false
        );

        // ----- Instance attributes -----

        /** The pointer to the characters. */
        private final PointerWrapper charPointer;

        /** The size of the text. */
        private final long length;

        /** If the text is allocated. */
        private final boolean isAllocated;

        /** If the text object is the owner of its buffer. */
        private final boolean isOwner;

        /** The content of the text in a Java string. */
        private String content;

        // ----- Constructors -----

        /**
         * Create a new text from its content.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                null
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param contentArray The characters of the text (for JNI)
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final byte[] contentArray
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                contentArray
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param isOwner If the Java object owns the text buffer.
         * @param contentArray The characters of the text
         * (as strings, this is only used in JNI mode).
         */
        private Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final boolean isOwner,
            final byte[] contentArray
        ) {
            this.charPointer = charPointer;
            this.length = length;
            this.isAllocated = isAllocated;
            this.isOwner = isOwner;

            if(contentArray != null) {
                this.content = decodeUTF32(contentArray);
            } else {
                this.content = null;
            }
        }

        /**
         * Create a new langkit text from its string content.
         *
         * @param content The content of the text in a Java string.
         * @return The newly created text.
         */
        public static Text create(
            final String content
        ) {
            final byte[] contentArray = encodeUTF32(content);

            if(ImageInfo.inImageCode()) {
                final PointerWrapper charPointer = new PointerWrapper(
                    toCBytes(contentArray)
                );
                return new Text(
                    charPointer,
                    (long) content.length(),
                    false,
                    true,
                    contentArray
                );
            } else {
                return JNI_LIB.ada_create_text(contentArray);
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the native text pointed by the given pointer.
         *
         * @param pointer The pointer to the native text to wrap.
         * @return The wrapped text.
         */
        static Text wrap(
            final Pointer pointer
        ) {
            return wrap((TextNative) pointer.readWord(0));
        }

        /**
         * Wrap a text native value in the Java class.
         *
         * @param textNative The text native NI value.
         * @return The newly wrapped text.
         */
        static Text wrap(
            final TextNative textNative
        ) {
            return new Text(
                new PointerWrapper(textNative.get_chars()),
                textNative.get_length(),
                textNative.get_is_allocated() != 0
            );
        }

        /**
         * Unwrap the text in the given NI pointer.
         *
         * @param textNative The NI pointer to the native text structure.
         */
        void unwrap(
            final TextNative textNative
        ) {
            textNative.set_chars(this.charPointer.ni());
            textNative.set_length(this.length);
            textNative.set_is_allocated(this.isAllocated ? 1 : 0);
        }

        // ----- Instance methods -----

        /**
         * Get the content of the text in a Java string.
         *
         * @return the content of the text.
         */
        public String getContent() {
            // Content is null only when using the Graal C API.
            if(this.content == null) {
                final byte[] contentArray = new byte[(int) this.length * 4];
                for(int i = 0 ; i < contentArray.length ; i++) {
                    contentArray[i] = (
                        (CCharPointer) this.charPointer.ni()
                    ).read(i);
                }
                this.content = decodeUTF32(contentArray);
            }

            return this.content;
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                if(this.isOwner) {
                    UnmanagedMemory.free(this.charPointer.ni());
                } else {
                    final TextNative textNative = StackValue.get(
                        TextNative.class
                    );
                    this.unwrap(textNative);
                    NI_LIB.ada_destroy_text(textNative);
                }
            } else {
                JNI_LIB.ada_destroy_text(this);
            }
            checkException();

        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getContent();
        }

    }

    /**
     * Location in a source file. Line and column numbers are one-based.
     */
    public static final class SourceLocation {

        // ----- Class attributes -----

        /** Singleton that represents the none source location. */
        public static final SourceLocation NONE = new SourceLocation(
            0,
            (short) 0
        );

        // ----- Instance attributes -----

        /** The line of the source location. */
        public final int line;

        /** The column of the source location. */
        public final short column;

        // ----- Constructors -----

        /**
         * Create a new source location from a line and a column.
         *
         * @param line The line of the source location.
         * @param column The column of the source location.
         */
        SourceLocation(
            final int line,
            final short column
        ) {
            this.line = line;
            this.column = column;
        }

        /**
         * Create a source location from its line and column.
         *
         * @param line The line.
         * @param column The column.
         * @return The newly create source location.
         */
        public static SourceLocation create(
            final int line,
            final short column
        ) {
            return new SourceLocation(
                line,
                column
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location in the Java class.
         *
         * @param pointer Pointer to the native source location.
         * @return The newly wrapper source location.
         */
        static SourceLocation wrap(
            final Pointer pointer
        ) {
            return wrap((SourceLocationNative) pointer.readWord(0));
        }

        /**
         * Wrap a source location native value in the Java class.
         *
         * @param sourceLocationNative The source location NI native value.
         * @return The newly wrapped source location.
         */
        static SourceLocation wrap(
            final SourceLocationNative sourceLocationNative
        ) {
            return new SourceLocation(
                sourceLocationNative.get_line(),
                sourceLocationNative.get_column()
            );
        }

        /**
         * Uwrap the source location in the given NI pointer.
         *
         * @param sourceLocationNative The NI pointer to the native structure
         *  to fill.
         */
        public void unwrap(
            final SourceLocationNative sourceLocationNative
        ) {
            sourceLocationNative.set_line(this.line);
            sourceLocationNative.set_column(this.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.line + ":" + this.column;
        }

    }

    /**
     * Location of a span of text in a source file.
     */
    public static final class SourceLocationRange {

        // ----- Class attributes -----

        /** Singleton that represents the none source location range. */
        public static final SourceLocationRange NONE =
            new SourceLocationRange(
                SourceLocation.NONE,
                SourceLocation.NONE
            );

        // ----- Instance attributes -----

        /** The start of the range. */
        public final SourceLocation start;

        /** The end of the range. */
        public final SourceLocation end;

        // ----- Constructors -----

        /**
         * Create a source location range from its bounds.
         *
         * @param start The start of the range.
         * @param end The end of the range.
         */
        SourceLocationRange(
            final SourceLocation start,
            final SourceLocation end
        ) {
            this.start = start;
            this.end = end;
        }

        /**
         * Create a new source location range from its bounds.
         *
         * @param start The starting bound.
         * @param end The ending bound.
         * @return The newly created source location range.
         */
        public static SourceLocationRange create(
            final SourceLocation start,
            final SourceLocation end
        ) {
            return new SourceLocationRange(
                start,
                end
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location range in the Java class.*
         *
         * @param pointer The pointer to the native source location range.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final Pointer pointer
        ) {
            return wrap((SourceLocationRangeNative) pointer.readWord(0));
        }

        /**
         * Wrap a source location range native value in the Java class.
         *
         * @param sourceLocationRangeNative The source location range NI
         * native value.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            return new SourceLocationRange(
                new SourceLocation(
                    sourceLocationRangeNative.get_start_line(),
                    sourceLocationRangeNative.get_start_column()
                ),
                new SourceLocation(
                    sourceLocationRangeNative.get_end_line(),
                    sourceLocationRangeNative.get_end_column()
                )
            );
        }

        /**
         * Uwrap the source location range in the given NI pointer.
         *
         * @param sourceLocationRangeNative The NI pointer to the native
         * structure to fill.
         */
        void unwrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            sourceLocationRangeNative.set_start_line(this.start.line);
            sourceLocationRangeNative.set_start_column(this.start.column);
            sourceLocationRangeNative.set_end_line(this.end.line);
            sourceLocationRangeNative.set_end_column(this.end.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.start.toString() + "-" + this.end.toString();
        }

    }

    /**
     * Diagnostic for an analysis unit: cannot open the source file, parsing
     * error, ...
     */
    public static final class Diagnostic {

        // ----- Class attributes -----

        /** Singleton that represents the none diagnositc. */
        public static final Diagnostic NONE = new Diagnostic(
            SourceLocationRange.NONE,
            Text.NONE
        );

        // ----- Instance attributes -----

        /** The source location range of the diagnostic. */
        public final SourceLocationRange sourceLocationRange;

        /** The message of the diagnostic. */
        public final Text message;

        // ----- Constructors -----

        /**
         * Create a diagnostic from its content.
         *
         * @param sourceLocationRange The range of the diagnostic.
         * @param message The message of the diagnostic.
         */
        Diagnostic(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            this.sourceLocationRange = sourceLocationRange;
            this.message = message;
        }

        /**
         * Create a new diagnostic from its content.
         *
         * @param sourceLocationRange The source location range concerned by
         * this diagnostic.
         * @param message The message of the diagnostic.
         * @return The newly created diagnostic
         */
        public static Diagnostic create(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            return new Diagnostic(
                sourceLocationRange,
                message
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native diagnostic.
         *
         * @param pointer The pointer to the native diagnositc.
         * @return The wrapped diagnositc.
         */
        static Diagnostic wrap(
            final Pointer pointer
        ) {
            return wrap((DiagnosticNative) pointer.readWord(0));
        }

        /**
         * Wrap a diagnostic native value in the Java class.
         *
         * @param diagnosticNative The diagnostic NI native value.
         * @return The newly wrapped diagnositc.
         */
        static Diagnostic wrap(
            final DiagnosticNative diagnosticNative
        ) {
            return new Diagnostic(
                new SourceLocationRange(
                    new SourceLocation(
                        diagnosticNative.get_start_line(),
                        diagnosticNative.get_start_column()
                    ),
                    new SourceLocation(
                        diagnosticNative.get_end_line(),
                        diagnosticNative.get_end_column()
                    )
                ),
                new Text(
                    new PointerWrapper(diagnosticNative.get_message_chars()),
                    diagnosticNative.get_message_length(),
                    diagnosticNative.get_message_is_allocated() != 0
                )
            );
        }

        /**
         * Unwrap the diagnostic in the given NI pointer.
         *
         * @param diagnosticNative The pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final DiagnosticNative diagnosticNative
        ) {
            diagnosticNative.set_start_line(
                this.sourceLocationRange.start.line
            );
            diagnosticNative.set_start_column(
                this.sourceLocationRange.start.column
            );
            diagnosticNative.set_end_line(
                this.sourceLocationRange.end.line
            );
            diagnosticNative.set_end_column(
                this.sourceLocationRange.end.column
            );
            diagnosticNative.set_message_chars(
                this.message.charPointer.ni()
            );
            diagnosticNative.set_message_length(
                this.message.length
            );
            diagnosticNative.set_message_is_allocated(
                this.message.isAllocated ? 1 : 0
            );
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.message.toString()
                + " at <"
                + this.sourceLocationRange.toString()
                + ">";
        }

    }

    /**
     * Interface to override how source files are fetched and decoded.
     */
    public static final class FileReader implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none file reader. */
        public static final FileReader NONE = new FileReader(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the file reader */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new file reader from its native reference.
         *
         * @param reference The reference to the native file reader.
         */
        FileReader(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native file reader.
         *
         * @param pointer The pointer to the native file reader.
         * @return The newly wrapped file reader.
         */
        static FileReader wrap(
            final Pointer pointer
        ) {
            return wrap((FileReaderNative) pointer.readWord(0));
        }

        /**
         * Wrap the given file reader in the Java class.
         *
         * @param fileReaderNative The native file reader to wrap.
         * @return The wrapped file reader.
         */
        static FileReader wrap(
            final FileReaderNative fileReaderNative
        ) {
            return new FileReader(
                new PointerWrapper(fileReaderNative)
            );
        }

        /**
         * Unwrap the file reader in the given pointer.
         *
         * @param pointer The pointer to unwrap the file reader in.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Get the unwrapped file reader.
         *
         * @return The unwrapped native file reader.
         */
        FileReaderNative unwrap() {
            return (FileReaderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.ada_dec_ref_file_reader(
                    this.reference.ni()
                );
            } else {
                JNI_LIB.ada_dec_ref_file_reader(this);
            }
            checkException();

        }

    }

    /*
 * Interface to fetch analysis units from a name and a unit kind.
 *
 * The unit provider mechanism provides an abstraction which assumes that to
 * any couple (unit name, unit kind) we can associate at most one source file.
 * This means that several couples can be associated to the same source file,
 * but on the other hand, only one one source file can be associated to a
 * couple.
 *
 * This is used to make the semantic analysis able to switch from one analysis
 * units to another.
 *
 * See the documentation of each unit provider for the exact semantics of the
 * unit name/kind information.
 */
    public static final class UnitProvider implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none unit provider. */
        public static final UnitProvider NONE = new UnitProvider(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the unit provider */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new unit provider with the reference to the native one.
         *
         * @param reference The reference to the native unit provider.
         */
        UnitProvider(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native unit provider.
         *
         * @param pointer The pointer to the native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final Pointer pointer
        ) {
            return wrap((UnitProviderNative) pointer.readWord(0));
        }

        /**
         * Wrap the given native unit provider.
         *
         * @param unitProviderNative The native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final UnitProviderNative unitProviderNative
        ) {
            return new UnitProvider(
                new PointerWrapper(unitProviderNative)
            );
        }

        /**
         * Unwrap the unit provider in the given native pointer.
         *
         * @param pointer The pointer to place the native unit provider in.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Get the native unit provider.
         *
         * @return The native unit provider.
         */
        UnitProviderNative unwrap() {
            return (UnitProviderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.ada_dec_ref_unit_provider(this.reference.ni());
            } else {
                JNI_LIB.ada_dec_ref_unit_provider(this);
            }
            checkException();

        }

    }

    /**
 * Interface to handle events sent by the analysis context.
 */
    public static final class EventHandler implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none event handler. */
        public static final EventHandler NONE = new EventHandler(
            PointerWrapper.nullPointer(),
            null,
            null
        );

        /** This map contains all created event handlers. */
        private static final Map<PointerWrapper, EventHandler>
            eventHandlerCache = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native event handler. */
        private final PointerWrapper reference;

        /**
         * The Java callback when an analysis unit is requested in the
         * associated context.
         */
        private final UnitRequestedCallback unitRequestedCallback;

        /**
         * The Java callback when an analysis unit is parsed in the
         * associated context.
         */
        private final UnitParsedCallback unitParsedCallback;

        // ----- Constructors -----

        /**
         * Create a new event handler from its native reference.
         *
         * @param reference The reference to the native event handler.
         * @param unitRequestedCallback The callback for unit requests.
         * @param unitParsedCallback The callback for unit parsing.
         */
        EventHandler(
            final PointerWrapper reference,
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            this.reference = reference;
            this.unitRequestedCallback = unitRequestedCallback;
            this.unitParsedCallback = unitParsedCallback;
        }

        /**
         * Create a new even handler with its callbacks. Callbacks can be null.
         *
         * @param unitRequestedCallback The callback for analysis unit requests.
         * @param unitParsedCallback The callback for analysis unit parsing.
         */
        public static EventHandler create(
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            // Prepare the reference to the native event handler
            final PointerWrapper reference;

            if(ImageInfo.inImageCode()) {
                // Get the current thread
                final IsolateThread thread = CurrentIsolate.getCurrentThread();

                // Create the native event handler
                final EventHandlerNative resNative =
                    NI_LIB.ada_create_event_handler(
                        (VoidPointer) thread,
                        WordFactory.nullPointer(),
                        NI_LIB.unitRequestedFunction.getFunctionPointer(),
                        NI_LIB.unitParsedFunction.getFunctionPointer()
                    );

                // Set the result to the created event handler
                reference = new PointerWrapper(resNative);
            } else {
                reference = JNI_LIB.ada_create_event_handler(
                    unitRequestedCallback,
                    unitParsedCallback
                );
            }

            // Return the new event handler wrapped object
            return new EventHandler(
                reference,
                unitRequestedCallback,
                unitParsedCallback
            );
        }

        /**
         * Get event handler Java object from its native pointer.
         *
         * @param reference The pointer to the native event handler.
         * @return The associated Java event handler.
         */
        static EventHandler fromReference(
            final PointerWrapper reference
        ) {
            if(eventHandlerCache.containsKey(reference)) {
                return eventHandlerCache.get(reference);
            } else {
                throw new ReferenceException(
                    "Cannot get event handler from this reference: " +
                    reference.toString()
                );
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to an event handler.
         *
         * @param pointer The pointer to the native event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final Pointer pointer
        ) {
            return wrap((EventHandlerNative) pointer.readWord(0));
        }

        /**
         * Wrap the given native event handler.
         *
         * @param eventHandlerNative The native value of the event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final EventHandlerNative eventHandlerNative
        ) {
            return fromReference(new PointerWrapper(eventHandlerNative));
        }

        /**
         * Unwrap the event handler in the given native pointer.
         *
         * @param pointer The pointer to place the native event handler.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Unwrap the event handler to a native value.
         *
         * @return The native value of the event handler.
         */
        EventHandlerNative unwrap() {
            return (EventHandlerNative) this.reference.ni();
        }

        // ----- Getters -----

        public UnitRequestedCallback getUnitRequestCallback() {
            return this.unitRequestedCallback;
        }

        public UnitParsedCallback getUnitParsedCallback() {
            return this.unitParsedCallback;
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.ada_dec_ref_event_handler(this.reference.ni());
            } else {
                JNI_LIB.ada_dec_ref_event_handler(this);
            }
            checkException();

        }

        // ----- Inner classes -----

        /**
         * Callback that will be called when a unit is requested from the
         * context ``Context``.
         *
         * ``Name`` is the name of the requested unit.
         *
         * ``From`` is the unit from which the unit was requested.
         *
         * ``Found`` indicates whether the requested unit was found or not.
         *
         * ``Is_Not_Found_Error`` indicates whether the fact that the unit was
         * not found is an error or not.
         *
         * .. warning:: The interface of this callback is probably subject to
         *    change, so should be treated as experimental.
         */
        @FunctionalInterface
        public interface UnitRequestedCallback {
            void invoke(
                AnalysisContext context,
                String name,
                AnalysisUnit from,
                boolean found,
                boolean isNotFoundError
            );
        }

        /**
         * Callback that will be called when any unit is parsed from the
         * context ``Context``.
         *
         * ``Unit`` is the resulting unit.
         *
         * ``Reparsed`` indicates whether the unit was reparsed, or whether it
         * was the first parse.
         */
        @FunctionalInterface
        public interface UnitParsedCallback {
            void invoke(
                AnalysisContext context,
                AnalysisUnit unit,
                boolean reparsed
            );
        }

    }

    /**
     * Reference to a token in an analysis unit.
     */
    public static class Token {

        // ----- Instance attributes -----

        /**
         * We only store the reference to the context to avoid ref-count
         * problems. To access the token context go throught the
         * analysis unit.
         */
        protected final PointerWrapper contextRef;

        /** The unit of the token. */
        public final AnalysisUnit unit;

        /** The pointer to the token data handler. */
        protected final PointerWrapper tokenDataHandler;

        /** The index of the token. */
        public final int tokenIndex;

        /** The trivia index. */
        public final int triviaIndex;

        /** The kind of the token. */
        public final TokenKind kind;

        /** The text of the token. */
        protected final String text;

        /** The source location range of the token. */
        public final SourceLocationRange sourceLocationRange;

        // ----- Constructors -----

        /**
         * Create a new token from its content.
         *
         * @param contextRef The context of the token.
         * @param unit The unit which owns the token.
         * @param tokenDataHandler The pointer to the token data.
         * @param tokenIndex The index of the token.
         * @param triviaIndex The trivia index of the token.
         * @param kind The kind of the token in an integer.
         * @param text The text of the token.
         * @param sourceLocationRange The location of the token.
         */
        Token(
            final PointerWrapper contextRef,
            final AnalysisUnit unit,
            final PointerWrapper tokenDataHandler,
            final int tokenIndex,
            final int triviaIndex,
            final TokenKind kind,
            final String text,
            final SourceLocationRange sourceLocationRange
        ) {
            this.contextRef = contextRef;
            this.unit = unit;
            this.tokenDataHandler = tokenDataHandler;
            this.tokenIndex = tokenIndex;
            this.triviaIndex = triviaIndex;
            this.kind = kind;
            this.text = text;
            this.sourceLocationRange = sourceLocationRange;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the pointer to a native token.
         *
         * @param pointer The pointer to the native token.
         * @param unit The analysis unit which owns the token.
         * @return The wrapped token or a none value if the token is a none
         * one.
         */
        static Token wrap(
            final Pointer pointer,
            final AnalysisUnit unit
        ) {
            return wrap(
                (TokenNative) pointer.readWord(0),
                unit
            );
        }

        /**
         * Wrap a native token value in the Java class.
         *
         * @param tokenNative The native NI token value.
         * @param unit The analysis unit that owns the token.
         * @return The wrapped token or a none value if the token data
         * handler is null.
         */
        static Token wrap(
            final TokenNative tokenNative,
            final AnalysisUnit unit
        ) {
            if(tokenNative.get_data().isNull())
                return NONE(unit);

            // Fetch the token kind, source location range and text from the
            // tokenNative reference.
            final TokenKind kind = TokenKind.fromC(
                NI_LIB.ada_token_get_kind(tokenNative)
            );

            final SourceLocationRangeNative slocRangeNative =
                StackValue.get(SourceLocationRangeNative.class);
            NI_LIB.ada_token_sloc_range(
                tokenNative,
                slocRangeNative
            );
            final SourceLocationRange slocRange =
                SourceLocationRange.wrap(slocRangeNative);

            final TextNative textNative = StackValue.get(TextNative.class);
            NI_LIB.ada_token_range_text(
                tokenNative,
                tokenNative,
                textNative
            );
            final String text = Text.wrap(textNative).getContent();
            NI_LIB.ada_destroy_text(textNative);

            // Wrap them in a high-level Token instance
            return new Token(
                new PointerWrapper(tokenNative.get_context()),
                unit,
                new PointerWrapper(tokenNative.get_data()),
                tokenNative.get_token_index(),
                tokenNative.get_trivia_index(),
                kind,
                text,
                slocRange
            );
        }

        /**
         * Unwrap the token in the given NI pointer.
         *
         * @param tokenNative The NI pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final TokenNative tokenNative
        ) {
            tokenNative.set_context(this.contextRef.ni());
            tokenNative.set_data(this.tokenDataHandler.ni());
            tokenNative.set_token_index(this.tokenIndex);
            tokenNative.set_trivia_index(this.triviaIndex);
        }

        // ----- Getters -----

        public String getText() {
            return this.text;
        }

        public boolean isTrivia() {
            return this.triviaIndex != 0;
        }

        public boolean isNone() {
            return false;
        }

        // ----- Class methods -----

        /**
         * Get a none token for the given unit.
         *
         * @param unit The unit to get a none token for.
         * @return The none token for the given unit.
         */
        public static Token NONE(
            final AnalysisUnit unit
        ) {
            return NoToken.getInstance(unit);
        }

        /**
         * Get the text from the start token to the end token.
         *
         * @param start The start token.
         * @param end The end token.
         * @return The text between the two tokens.
         */
        @CompilerDirectives.TruffleBoundary
        public static String textRange(
            final Token start,
            final Token end
        ) {

            if(ImageInfo.inImageCode()) {
                final TokenNative startNative = StackValue.get(
                    TokenNative.class
                );
                start.unwrap(startNative);

                final TokenNative endNative = StackValue.get(
                    TokenNative.class
                );
                end.unwrap(endNative);

                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                Text.NONE.unwrap(textNative);
                NI_LIB.ada_token_range_text(
                    startNative,
                    endNative,
                    textNative
                );
                try(final Text resText = Text.wrap(textNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.ada_token_range_text(
                        start,
                        end
                    )
                ) {
                    return resText.getContent();
                }
            }

        }

        // ----- Instance methods -----

        /**
         * Get the next token.
         *
         * @return The next token in the source.
         */
        public Token next() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative nextNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.ada_token_next(
                    tokenNative,
                    nextNative
                );
                return wrap(nextNative, this.unit);
            } else {
                return JNI_LIB.ada_token_next(this);
            }

        }

        /**
         * Get the previous token.
         *
         * @return The previous token in the source.
         */
        public Token previous() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative previousNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.ada_token_previous(
                    tokenNative,
                    previousNative
                );
                return wrap(previousNative, this.unit);
            } else {
                return JNI_LIB.ada_token_previous(this);
            }

        }

        /**
         * Check of the token is equivalent to the other one.
         *
         * @param other The other token to compare with.
         */
        public boolean isEquivalent(
            final Token other
        ) {

            if(ImageInfo.inImageCode()) {
                final TokenNative leftNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(leftNative);

                final TokenNative rightNative = StackValue.get(
                    TokenNative.class
                );
                other.unwrap(rightNative);

                return NI_LIB.ada_token_is_equivalent(
                    leftNative,
                    rightNative
                ) != 0;
            } else {
                return JNI_LIB.ada_token_is_equivalent(this, other);
            }

        }

        // ----- Override methods -----

        @Override
        @CompilerDirectives.TruffleBoundary
        public String toString() {
            return "<Token Kind="
                + this.kind.name
                + " Text=\""
                + stringRepresentation(this.getText())
                + "\">";
        }

        @Override
        public boolean equals(
            Object o
        ) {
            if(o == this) return true;
            if(!(o instanceof Token)) return false;
            final Token other = (Token) o;
            return other.tokenDataHandler.equals(this.tokenDataHandler) &&
                    other.tokenIndex == this.tokenIndex &&
                    other.triviaIndex == this.triviaIndex;
        }

        // ----- Inner classes -----

        /**
        * This class represents the absence of token.
        */
        private static final class NoToken extends Token {

            // ----- Class attributes -----

            /** The map of the no token instances */
            private static final Map<AnalysisUnit, NoToken> instances
                = new HashMap<>();

            // ----- Constructors -----

            /**
            * Create a new no token for the given analysis unit.
            * The constructor is private to avoid too many instances.
            *
            * @param unit The analysis unit to create the no token for.
            */
            private NoToken(
                final PointerWrapper contextRef,
                final AnalysisUnit unit
            ) {
                super(
                    contextRef,
                    unit,
                    PointerWrapper.nullPointer(),
                    0,
                    0,
                    TokenKind.fromC(-1),
                    null,
                    SourceLocationRange.NONE
                );
            }

            /**
            * Get the no token instance for the given analysis unit.
            *
            * @param unit The unit to get the no token instance for.
            * @return The no token instance.
            */
            static NoToken getInstance(
                final AnalysisUnit unit
            ) {
                if(!instances.containsKey(unit)) {
                    try(AnalysisContext context = unit.getContext()) {
                        instances.put(
                            unit,
                            new NoToken(context.reference, unit)
                        );
                    }
                }
                return instances.get(unit);
            }

            // ----- Getters -----

            @Override
            public String getText() {
                return "";
            }

            // ----- Instance methods -----

            @Override
            public Token next() {
                return this;
            }

            @Override
            public Token previous() {
                return this;
            }

            @Override
            public boolean isEquivalent(
                final Token other
            ) {
                return other instanceof NoToken;
            }

            @Override
            public boolean isNone() {
                return true;
            }

            @Override
            public void unwrap(
                final TokenNative tokenNative
            ) {
                tokenNative.set_context(this.contextRef.ni());
                tokenNative.set_data(this.tokenDataHandler.ni());
                tokenNative.set_token_index(this.tokenIndex);
                tokenNative.set_trivia_index(this.triviaIndex);
            }

            // ----- Override methods -----

            @Override
            @CompilerDirectives.TruffleBoundary
            public String toString() {
                return "<Token Kind="
                    + this.kind.name
                    + " Text=\"\">";
            }

            @Override
            public boolean equals(
                Object o
            ) {
                return o == this;
            }

        }

    }

    /**
     * This type represents a context for all source analysis. This is the
     * first type you need to create to use Libadalang. It will contain the
     * results of all analysis, and is the main holder for all the data.
     *
     * You can create several analysis contexts if you need to, which enables
     * you, for example to:
     *
     * * analyze several different projects at the same time;
     *
     * * analyze different parts of the same projects in parallel.
     *
     * In the current design, contexts always keep all of their analysis units
     * allocated. If you need to get this memory released, the only option at
     * your disposal is to destroy your analysis context instance.
     */
    public static final class AnalysisContext implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis context. */
        public static final AnalysisContext NONE = new AnalysisContext(
            PointerWrapper.nullPointer(),
            null
        );

        /** This map contains all created analysis contexts. */
        private static final Map<PointerWrapper, AnalysisContext> contextCache
            = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native analysis context. */
        private final PointerWrapper reference;

        /** The event handler associated with the context. */
        private EventHandler eventHandler;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its reference.
         *
         * @param reference The native analysis context.
         * @param eventHandler The associated event handler.
         */
        private AnalysisContext(
            final PointerWrapper reference,
            final EventHandler eventHandler
        ) {
            this.reference = reference;
            this.eventHandler = eventHandler;

            increaseRefCounter(this);
        }

        /**
         * Create a new analysis context from scratch with its configuration.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         */
        private AnalysisContext(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            // Call the function to allocate the native analysis context
            final PointerWrapper reference;
            if(ImageInfo.inImageCode()) {
                reference = new PointerWrapper(
                    NI_LIB.ada_allocate_analysis_context()
                );
            } else {
                reference = JNI_LIB.ada_create_analysis_context(
                    charset,
                    (fileReader == null ?
                        FileReader.NONE :
                        fileReader),
                    (unitProvider == null ?
                        UnitProvider.NONE :
                        unitProvider),
                    (eventHandler == null ?
                        EventHandler.NONE :
                        eventHandler),
                    withTrivia,
                    tabStop
                );
            }

            // Place the context in the cache for potention callbacks during
            // context initialization.
            this.reference = reference;
            this.eventHandler = eventHandler;
            contextCache.put(this.reference, this);

            // Perform the context initialization
            if(ImageInfo.inImageCode()) {
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                NI_LIB.ada_initialize_analysis_context(
                    (AnalysisContextNative) reference.ni(),
                    charsetNative,
                    (fileReader == null ?
                        WordFactory.nullPointer() :
                        fileReader.reference.ni()),
                    (unitProvider == null ?
                        WordFactory.nullPointer() :
                        unitProvider.reference.ni()),
                    (eventHandler == null ?
                        WordFactory.nullPointer() :
                        eventHandler.reference.ni()),
                    (withTrivia ? 1 : 0),
                    tabStop
                );
                if(charset != null) UnmanagedMemory.free(charsetNative);
            }
        }

        /**
         * Create a new analysis context with the default parameters.
         *
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create() {
            return new AnalysisContext(
                null,
                null,
                null,
                null,
                true,
                8
            );
        }

        /**
         * Create an analysis context with its parameters.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            return new AnalysisContext(
                charset,
                fileReader,
                unitProvider,
                eventHandler,
                withTrivia,
                tabStop
            );
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @param eventHandler The corresponding event handler.
         * @param setEventHandler Whether to set the result's eventHandler
         * field to eventHandler when there is already a cached
         * AnalysisContext.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference,
            final EventHandler eventHandler,
            final boolean setEventHandler
        ) {
            if(contextCache.containsKey(reference)) {
                final AnalysisContext res = contextCache.get(reference);
                increaseRefCounter(res);
                if(setEventHandler)
                    res.eventHandler = eventHandler;
                return res;
            } else {
                final AnalysisContext result =
                    new AnalysisContext(reference, eventHandler);
                contextCache.put(reference, result);
                return result;
            }
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference
        ) {
            return fromReference(reference, null, false);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a native pointer to the native analysis context in the
         * Java class.
         *
         * @param pointer The pointer to the NI analysis context
         * native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final Pointer pointer
        ) {
            return wrap((AnalysisContextNative) pointer.readWord(0));
        }

        /**
         * Wrap an analysis context native value in the Java class.
         *
         * @param analysisContextNative The NI analysis context native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final AnalysisContextNative analysisContextNative
        ) {
            return fromReference(new PointerWrapper(analysisContextNative));
        }

        /**
         * Unwrap the analysis context in the given native pointer.
         *
         * @param pointer The pointer to place the native analysis unit.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Get the native value of the analysis context.
         *
         * @return The native analysis context.
         */
        AnalysisContextNative unwrap() {
            return (AnalysisContextNative) this.reference.ni();
        }

        // ----- Getters -----

        public EventHandler getEventHandler() {
            return this.eventHandler;
        }

        // ----- Instance methods -----

        /**
         * Get an analysis unit from the given file in the current context.
         *
         * @param fileName The file to get the analysis unit from.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName
        ) {
            return this.getUnitFromFile(
                fileName,
                null,
                false,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given file in the current context
         * with additional parameters.
         *
         * @param fileName The file to get the analysis unit from.
         * @param charset The charset of the given file.
         * @param reparse If the file should be reparsed.
         * @param rule The grammar rule to parse the source with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName,
            final String charset,
            final boolean reparse,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer fileNameNative = toCString(fileName);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.ada_get_analysis_unit_from_file(
                    this.reference.ni(),
                    fileNameNative,
                    charsetNative,
                    (reparse ? 1 : 0),
                    rule.toC()
                );
                UnmanagedMemory.free(fileNameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.ada_get_analysis_unit_from_file(
                    this,
                    fileName,
                    charset,
                    reparse,
                    rule.toC()
                );
            }

        }

        /**
         * Get an analysis unit from the given buffer in the current context.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name
        ) {
            return this.getUnitFromBuffer(
                buffer,
                name,
                null,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given buffer in the current context
         * with additional parameters.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @param charset The charset of the buffer.
         * @param rule The rule to parse the buffer with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name,
            final String charset,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer bufferNative = toCString(buffer);
                final CCharPointer nameNative = toCString(name);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.ada_get_analysis_unit_from_buffer(
                    this.reference.ni(),
                    nameNative,
                    charsetNative,
                    bufferNative,
                    buffer.length(),
                    rule.toC()
                );
                UnmanagedMemory.free(bufferNative);
                UnmanagedMemory.free(nameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.ada_get_analysis_unit_from_buffer(
                    this,
                    name,
                    charset,
                    buffer,
                    buffer.length(),
                    rule.toC()
                );
            }

        }

        /**
         * Get an analysis unit from the given unit name and unit kind in the
         * current context with additional parameters.
         *
         * @param name Name of the unit.
         * @param kind Kind of the unit.
         * @param charset The charset of the buffer.
         * @param rule The rule to parse the buffer with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromProvider(
            final Text name,
            final AnalysisUnitKind kind,
            final String charset,
            final boolean reparse
        ) {
            if(ImageInfo.inImageCode()) {
                TextNative nameNative = StackValue.get(TextNative.class);
                name.unwrap(nameNative);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);
                final AnalysisUnitNative resNative =
                    NI_LIB.ada_get_analysis_unit_from_provider(
                    this.reference.ni(),
                    nameNative,
                    kind.toC(),
                    charsetNative,
                    (reparse ? 1 : 0)
                );
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.ada_get_analysis_unit_from_provider(
                    this,
                    name,
                    kind.toC(),
                    charset,
                    reparse
                );
            }
        }

        public AnalysisUnit getUnitFromProvider(
            final Text name,
            final AnalysisUnitKind kind
        ) {
            return this.getUnitFromProvider(name, kind, "", false);
        }

        /**
         * Increase the reference counter of the given context.
         *
         * @param context The context to increase the reference counter of.
         */
        private static void increaseRefCounter(
            final AnalysisContext context
        ) {
            // Increase the context reference counter of the context if not null
            if(!context.reference.isNull()) {
                if(ImageInfo.inImageCode()) {
                    NI_LIB.ada_context_incref(context.reference.ni());
                } else {
                    JNI_LIB.ada_context_incref(context.reference.jni());
                }
            }
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.ada_context_decref(this.reference.ni());
            } else {
                JNI_LIB.ada_context_decref(this.reference.jni());
            }
            checkException();

        }

        
      

        /**
         * Assign in ``Context`` configuration pragmas files to analysis units
         * as described in ``Global_Pragmas`` (configuration pragmas file that
         * applies to all analysis units, or null) and ``Local_Pragmas``
         * (mapping that associates an analysis unit to the local configuration
         * pragmas file that applies to it).
         *
         * This raises a ``Precondition_Failure`` exception if any analysis
         * unit in ``Mapping`` does not belong to ``Context`` or if an analysis
         * unit appears twice as a key in ``Mapping``.
         */
        public void setConfigPragmasMapping(
            AnalysisUnit globalPragmas,
            Map<AnalysisUnit, AnalysisUnit> localPragmas
        ) {
            // Create the flat array for local pragmas that the C API expects
            AnalysisUnit[] locals;
            int i = 0;
            int localCount;
            if (localPragmas == null) {
                localCount = 1;
                locals = new AnalysisUnit[localCount];
            } else {
                localCount = 2 * localPragmas.size() + 1;
                locals = new AnalysisUnit[localCount];
                for (Map.Entry<AnalysisUnit, AnalysisUnit> entry :
                     localPragmas.entrySet()) {
                    AnalysisUnit key = entry.getKey();
                    AnalysisUnit value = entry.getValue();

                    if (key == null)
                        throw new RuntimeException("localPragmas: null key");
                    if (value == null)
                        throw new RuntimeException("localPragmas: null value");

                    locals[i] = key;
                    locals[i + 1] = value;
                    i += 2;
                }
            }
            locals[i] = null;

            if(ImageInfo.inImageCode()) {
                final int unitSize = SizeOf.get(AnalysisUnitNative.class);
                final Pointer localPragmasNative = UnmanagedMemory.calloc(
                    localCount * unitSize
                );
                for (i = 0; i < localCount; ++i) {
                    final AnalysisUnitNative u =
                        locals[i] == null
                        ? WordFactory.nullPointer()
                        : locals[i].unwrap();
                    localPragmasNative.writeWord(unitSize * i, u);
                }

                final AnalysisUnitNative globalPragmasNative =
                    globalPragmas == null
                    ? WordFactory.nullPointer()
                    : globalPragmas.unwrap();

                NI_LIB.ada_set_config_pragmas_mapping(
                    this.reference.ni(),
                    globalPragmasNative,
                    localPragmasNative
                );
            } else {
                JNI_LIB.ada_set_config_pragmas_mapping(
                    this,
                    globalPragmas,
                    locals
                );
            }
            checkException();
        }



    }

    /**
     * This type represents the analysis of a single file.
     *
     * This type has strong-reference semantics and is ref-counted.
     * Furthermore, a reference to a unit contains an implicit reference to the
     * context that owns it. This means that keeping a reference to a unit will
     * keep the context and all the unit it contains allocated.
     */
    public static final class AnalysisUnit {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis unit. */
        public static final AnalysisUnit NONE = new AnalysisUnit(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the native analysis unit. */
        private final PointerWrapper reference;

        /** The cache for the unit root. */
        private AdaNode root;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its value.
         *
         * @param reference The native analysis unit native value in
         * a pointer wrapper.
         */
        AnalysisUnit(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native analysis unit in the Java class.
         *
         * @param pointer The pointer the native analysis unit value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final Pointer pointer
        ) {
            return wrap((AnalysisUnitNative) pointer.readWord(0));
        }

        /**
         * Wrap a NI analysis unit native value in the Java class.
         *
         * @param analysisUnitNative The NI analysis unit native value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final AnalysisUnitNative analysisUnitNative
        ) {
            return new AnalysisUnit(new PointerWrapper(analysisUnitNative));
        }

        /**
         * Unwrap the analysis unit in the given pointer.
         *
         * @param pointer The pointer to place the native analysis unit in.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Unwrap the analysis unit as a native value.
         *
         * @return The native analysis unit.
         */
        AnalysisUnitNative unwrap() {
            return (AnalysisUnitNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /**
         * Get the root node of the analysis unit.
         *
         * @return The root node.
         */
        public AdaNode getRoot() {
            if(this.root == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    NI_LIB.ada_unit_root(
                        this.reference.ni(),
                        entityNative
                    );
                    this.root = AdaNode.fromEntity(
                        Entity.wrap(entityNative)
                    );
                } else {
                    this.root = AdaNode.fromEntity(
                        JNI_LIB.ada_unit_root(this)
                    );
                }

            }
            return this.root;
        }

        /**
         * Get the analysis unit file name with its full path.
         *
         * @return The unit file name.
         */
        public String getFileName() {
            return this.getFileName(true);
        }

        /**
         * Get the unit's file name.
         *
         * @param fullPath If the method should return the
         * file full absolute path.
         * @return The file name.
         */
        @CompilerDirectives.TruffleBoundary
        public String getFileName(
            final boolean fullPath
        ) {
            final String absoluteFile;

            if(ImageInfo.inImageCode()) {
                final CCharPointer resNative = NI_LIB.ada_unit_filename(
                    this.reference.ni()
                );
                absoluteFile = toJString(resNative);
                NI_LIB.ada_free(resNative);
            } else {
                absoluteFile = JNI_LIB.ada_unit_filename(this);
            }

            if(fullPath) {
                return absoluteFile;
            } else {
                return new File(absoluteFile).getName();
            }
        }

        /**
         * Get the number of tokens in the analysis unit.
         *
         * @return The number of token.
         */
        public int getTokenCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.ada_unit_token_count(this.reference.ni());
            } else {
                return JNI_LIB.ada_unit_token_count(this);
            }

        }

        /**
         * Get the number of trivia in the analysis unit.
         *
         * @return The number of trivia.
         */
        public int getTriviaCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.ada_unit_trivia_count(
                    this.reference.ni()
                );
            } else {
                return JNI_LIB.ada_unit_trivia_count(this);
            }

        }

        /**
         * Return the first token of the analysis unit.
         *
         * @return The first token.
         */
        public Token getFirstToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.ada_unit_first_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.ada_unit_first_token(this);
            }

        }

        /**
         * Return the last token of the analysis unit.
         *
         * @return The last token.
         */
        public Token getLastToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.ada_unit_last_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.ada_unit_last_token(this);
            }

        }

        /**
         * Get the text of the analysis unit in a string.
         *
         * @return The text of the analysis unit source.
         */
        public String getText() {
            return Token.textRange(
                this.getFirstToken(),
                this.getLastToken()
            );
        }

        /**
         * Get the analysis context that owns the unit.
         *
         * @return The owning analysis context.
         */
        public AnalysisContext getContext() {

            if(ImageInfo.inImageCode()) {
                final AnalysisContextNative contextNative =
                    NI_LIB.ada_unit_context(this.reference.ni());
                return AnalysisContext.wrap(contextNative);
            } else {
                return JNI_LIB.ada_unit_context(this);
            }

        }

        /**
         * Get the list of assiated diagnositcs. Those are parsing errors.
         *
         * @return The diagnositcs of the unit.
         */
        public List<Diagnostic> getDiagnostics() {
            final int diagnosticCount;

            if(ImageInfo.inImageCode()) {
                diagnosticCount = NI_LIB.ada_unit_diagnostic_count(
                    this.reference.ni()
                );
            } else {
                diagnosticCount = JNI_LIB.ada_unit_diagnostic_count(
                    this
                );
            }

            final List<Diagnostic> res = new ArrayList<>(diagnosticCount);

            if(ImageInfo.inImageCode()) {
                final DiagnosticNative diagnosticNative = StackValue.get(
                    DiagnosticNative.class
                );
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    NI_LIB.ada_unit_diagnostic(
                        this.reference.ni(),
                        i,
                        diagnosticNative
                    );
                    res.add(Diagnostic.wrap(diagnosticNative));
                }
            } else {
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    res.add(
                        JNI_LIB.ada_unit_diagnostic(this, i)
                    );
                }
            }

            return res;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return "<AnalysisUnit \"" + this.getFileName(false) + "\">";
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof AnalysisUnit)) return false;
            final AnalysisUnit other = (AnalysisUnit) o;
            return other.reference.equals(other.reference);
        }

    }

    // ===== Generated structure wrapping classes =====

        
    
    

    /**

     */
    public static final class Metadata {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Metadata NONE = new Metadata(
            false,PointerWrapper.nullPointer(),PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        public final
        boolean
        dottableSubp;
        public final
        PointerWrapper
        primitive;
        public final
        PointerWrapper
        primitiveRealType;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Metadata(
            final boolean dottableSubp,final PointerWrapper primitive,final PointerWrapper primitiveRealType
        ) {
            this.dottableSubp = dottableSubp;
            this.primitive = primitive;
            this.primitiveRealType = primitiveRealType;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Metadata create(
            final boolean dottableSubp,final PointerWrapper primitive,final PointerWrapper primitiveRealType
        ) {
            return new Metadata(
                dottableSubp,primitive,primitiveRealType
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Metadata wrap(
            final Pointer niPointer
        ) {
            return wrap((MetadataNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Metadata wrap(
            final MetadataNative structNative
        ) {
            return new Metadata(
                structNative.get_dottable_subp() != 0,PointerWrapper.wrap(structNative.get_primitive()),PointerWrapper.wrap(structNative.get_primitive_real_type())
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final MetadataNative structNative
        ) {
            CCharPointer dottable_subpNative = structNative.address_dottable_subp();dottable_subpNative.write(this.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer primitiveNative = structNative.address_primitive();primitiveNative.writeWord(0, this.primitive.ni());
            Pointer primitive_real_typeNative = structNative.address_primitive_real_type();primitive_real_typeNative.writeWord(0, this.primitiveRealType.ni());
        }


    }

        
    
    

    /**

     */
    public static final class EntityInfo {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final EntityInfo NONE = new EntityInfo(
            Metadata.NONE,PointerWrapper.nullPointer(),false
        );

        // ----- Instance attributes -----

        public final
        Metadata
        md;
        public final
        PointerWrapper
        rebindings;
        public final
        boolean
        fromRebound;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        EntityInfo(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            this.md = md;
            this.rebindings = rebindings;
            this.fromRebound = fromRebound;
        }

        /**
         * Create a new structure with the field values.
         */
        public static EntityInfo create(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            return new EntityInfo(
                md,rebindings,fromRebound
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final Pointer niPointer
        ) {
            return wrap((EntityInfoNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final EntityInfoNative structNative
        ) {
            return new EntityInfo(
                new Metadata(structNative.get_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_md_primitive()), PointerWrapper.wrap(structNative.get_md_primitive_real_type())),PointerWrapper.wrap(structNative.get_rebindings()),structNative.get_from_rebound() != 0
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityInfoNative structNative
        ) {
            CCharPointer md_dottable_subpNative = structNative.address_md_dottable_subp();md_dottable_subpNative.write(this.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer md_primitiveNative = structNative.address_md_primitive();md_primitiveNative.writeWord(0, this.md.primitive.ni());
            Pointer md_primitive_real_typeNative = structNative.address_md_primitive_real_type();md_primitive_real_typeNative.writeWord(0, this.md.primitiveRealType.ni());
            Pointer rebindingsNative = structNative.address_rebindings();rebindingsNative.writeWord(0, this.rebindings.ni());
            CCharPointer from_reboundNative = structNative.address_from_rebound();from_reboundNative.write(this.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

    
    

    /**

     */
    public static final class Entity {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Entity NONE = new Entity(
            PointerWrapper.nullPointer(),EntityInfo.NONE
        );

        // ----- Instance attributes -----

        public final
        PointerWrapper
        node;
        public final
        EntityInfo
        info;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Entity(
            final PointerWrapper node,final EntityInfo info
        ) {
            this.node = node;
            this.info = info;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Entity create(
            final PointerWrapper node,final EntityInfo info
        ) {
            return new Entity(
                node,info
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final Pointer niPointer
        ) {
            return wrap((EntityNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final EntityNative structNative
        ) {
            return new Entity(
                PointerWrapper.wrap(structNative.get_node()),new EntityInfo(new Metadata(structNative.get_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_info_md_primitive()), PointerWrapper.wrap(structNative.get_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_info_rebindings()), structNative.get_info_from_rebound() != 0)
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityNative structNative
        ) {
            Pointer nodeNative = structNative.address_node();nodeNative.writeWord(0, this.node.ni());
            CCharPointer info_md_dottable_subpNative = structNative.address_info_md_dottable_subp();info_md_dottable_subpNative.write(this.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer info_md_primitiveNative = structNative.address_info_md_primitive();info_md_primitiveNative.writeWord(0, this.info.md.primitive.ni());
            Pointer info_md_primitive_real_typeNative = structNative.address_info_md_primitive_real_type();info_md_primitive_real_typeNative.writeWord(0, this.info.md.primitiveRealType.ni());
            Pointer info_rebindingsNative = structNative.address_info_rebindings();info_rebindingsNative.writeWord(0, this.info.rebindings.ni());
            CCharPointer info_from_reboundNative = structNative.address_info_from_rebound();info_from_reboundNative.write(this.info.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

        
    
    

    /**
     * Composite field representing the aspect of an entity (:rmlink:`13`).
     */
    public static final class Aspect {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Aspect NONE = new Aspect(
            false,Entity.NONE,Entity.NONE,false
        );

        // ----- Instance attributes -----

        public final
        boolean
        exists;
        public final
        Entity
        node;
        public final
        Entity
        value;
        public final
        boolean
        inherited;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Aspect(
            final boolean exists,final Entity node,final Entity value,final boolean inherited
        ) {
            this.exists = exists;
            this.node = node;
            this.value = value;
            this.inherited = inherited;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Aspect create(
            final boolean exists,final Entity node,final Entity value,final boolean inherited
        ) {
            return new Aspect(
                exists,node,value,inherited
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Aspect wrap(
            final Pointer niPointer
        ) {
            return wrap((AspectNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Aspect wrap(
            final AspectNative structNative
        ) {
            return new Aspect(
                structNative.get_exists() != 0,new Entity(PointerWrapper.wrap(structNative.get_node_node()), new EntityInfo(new Metadata(structNative.get_node_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_node_info_md_primitive()), PointerWrapper.wrap(structNative.get_node_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_node_info_rebindings()), structNative.get_node_info_from_rebound() != 0)),new Entity(PointerWrapper.wrap(structNative.get_value_node()), new EntityInfo(new Metadata(structNative.get_value_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_value_info_md_primitive()), PointerWrapper.wrap(structNative.get_value_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_value_info_rebindings()), structNative.get_value_info_from_rebound() != 0)),structNative.get_inherited() != 0
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final AspectNative structNative
        ) {
            CCharPointer existsNative = structNative.address_exists();existsNative.write(this.exists ? (byte) 1 : (byte) 0);
            Pointer node_nodeNative = structNative.address_node_node();node_nodeNative.writeWord(0, this.node.node.ni());
            CCharPointer node_info_md_dottable_subpNative = structNative.address_node_info_md_dottable_subp();node_info_md_dottable_subpNative.write(this.node.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer node_info_md_primitiveNative = structNative.address_node_info_md_primitive();node_info_md_primitiveNative.writeWord(0, this.node.info.md.primitive.ni());
            Pointer node_info_md_primitive_real_typeNative = structNative.address_node_info_md_primitive_real_type();node_info_md_primitive_real_typeNative.writeWord(0, this.node.info.md.primitiveRealType.ni());
            Pointer node_info_rebindingsNative = structNative.address_node_info_rebindings();node_info_rebindingsNative.writeWord(0, this.node.info.rebindings.ni());
            CCharPointer node_info_from_reboundNative = structNative.address_node_info_from_rebound();node_info_from_reboundNative.write(this.node.info.fromRebound ? (byte) 1 : (byte) 0);
            Pointer value_nodeNative = structNative.address_value_node();value_nodeNative.writeWord(0, this.value.node.ni());
            CCharPointer value_info_md_dottable_subpNative = structNative.address_value_info_md_dottable_subp();value_info_md_dottable_subpNative.write(this.value.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer value_info_md_primitiveNative = structNative.address_value_info_md_primitive();value_info_md_primitiveNative.writeWord(0, this.value.info.md.primitive.ni());
            Pointer value_info_md_primitive_real_typeNative = structNative.address_value_info_md_primitive_real_type();value_info_md_primitive_real_typeNative.writeWord(0, this.value.info.md.primitiveRealType.ni());
            Pointer value_info_rebindingsNative = structNative.address_value_info_rebindings();value_info_rebindingsNative.writeWord(0, this.value.info.rebindings.ni());
            CCharPointer value_info_from_reboundNative = structNative.address_value_info_from_rebound();value_info_from_reboundNative.write(this.value.info.fromRebound ? (byte) 1 : (byte) 0);
            CCharPointer inheritedNative = structNative.address_inherited();inheritedNative.write(this.inherited ? (byte) 1 : (byte) 0);
        }


    }

        
    
    

    /**

     */
    public static final class CompletionItem {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final CompletionItem NONE = new CompletionItem(
            Entity.NONE,false,false,0
        );

        // ----- Instance attributes -----

        public final
        Entity
        decl;
        public final
        boolean
        isDotCall;
        public final
        boolean
        isVisible;
        public final
        int
        weight;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        CompletionItem(
            final Entity decl,final boolean isDotCall,final boolean isVisible,final int weight
        ) {
            this.decl = decl;
            this.isDotCall = isDotCall;
            this.isVisible = isVisible;
            this.weight = weight;
        }

        /**
         * Create a new structure with the field values.
         */
        public static CompletionItem create(
            final Entity decl,final boolean isDotCall,final boolean isVisible,final int weight
        ) {
            return new CompletionItem(
                decl,isDotCall,isVisible,weight
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static CompletionItem wrap(
            final Pointer niPointer
        ) {
            return wrap((CompletionItemNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static CompletionItem wrap(
            final CompletionItemNative structNative
        ) {
            return new CompletionItem(
                new Entity(PointerWrapper.wrap(structNative.get_decl_node()), new EntityInfo(new Metadata(structNative.get_decl_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_decl_info_md_primitive()), PointerWrapper.wrap(structNative.get_decl_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_decl_info_rebindings()), structNative.get_decl_info_from_rebound() != 0)),structNative.get_is_dot_call() != 0,structNative.get_is_visible() != 0,structNative.get_weight()
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final CompletionItemNative structNative
        ) {
            Pointer decl_nodeNative = structNative.address_decl_node();decl_nodeNative.writeWord(0, this.decl.node.ni());
            CCharPointer decl_info_md_dottable_subpNative = structNative.address_decl_info_md_dottable_subp();decl_info_md_dottable_subpNative.write(this.decl.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer decl_info_md_primitiveNative = structNative.address_decl_info_md_primitive();decl_info_md_primitiveNative.writeWord(0, this.decl.info.md.primitive.ni());
            Pointer decl_info_md_primitive_real_typeNative = structNative.address_decl_info_md_primitive_real_type();decl_info_md_primitive_real_typeNative.writeWord(0, this.decl.info.md.primitiveRealType.ni());
            Pointer decl_info_rebindingsNative = structNative.address_decl_info_rebindings();decl_info_rebindingsNative.writeWord(0, this.decl.info.rebindings.ni());
            CCharPointer decl_info_from_reboundNative = structNative.address_decl_info_from_rebound();decl_info_from_reboundNative.write(this.decl.info.fromRebound ? (byte) 1 : (byte) 0);
            CCharPointer is_dot_callNative = structNative.address_is_dot_call();is_dot_callNative.write(this.isDotCall ? (byte) 1 : (byte) 0);
            CCharPointer is_visibleNative = structNative.address_is_visible();is_visibleNative.write(this.isVisible ? (byte) 1 : (byte) 0);
            CIntPointer weightNative = structNative.address_weight();weightNative.write(this.weight);
        }


    }

        
        
    
    

    /**
     * Represent the range of a discrete type or subtype. The bounds are not
     * evaluated, you need to call ``eval_as_int`` on them, if they're static,
     * to get their value.
     */
    public static final class DiscreteRange {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final DiscreteRange NONE = new DiscreteRange(
            Entity.NONE,Entity.NONE
        );

        // ----- Instance attributes -----

        public final
        Entity
        lowBound;
        public final
        Entity
        highBound;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        DiscreteRange(
            final Entity lowBound,final Entity highBound
        ) {
            this.lowBound = lowBound;
            this.highBound = highBound;
        }

        /**
         * Create a new structure with the field values.
         */
        public static DiscreteRange create(
            final Entity lowBound,final Entity highBound
        ) {
            return new DiscreteRange(
                lowBound,highBound
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static DiscreteRange wrap(
            final Pointer niPointer
        ) {
            return wrap((DiscreteRangeNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static DiscreteRange wrap(
            final DiscreteRangeNative structNative
        ) {
            return new DiscreteRange(
                new Entity(PointerWrapper.wrap(structNative.get_low_bound_node()), new EntityInfo(new Metadata(structNative.get_low_bound_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_low_bound_info_md_primitive()), PointerWrapper.wrap(structNative.get_low_bound_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_low_bound_info_rebindings()), structNative.get_low_bound_info_from_rebound() != 0)),new Entity(PointerWrapper.wrap(structNative.get_high_bound_node()), new EntityInfo(new Metadata(structNative.get_high_bound_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_high_bound_info_md_primitive()), PointerWrapper.wrap(structNative.get_high_bound_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_high_bound_info_rebindings()), structNative.get_high_bound_info_from_rebound() != 0))
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final DiscreteRangeNative structNative
        ) {
            Pointer low_bound_nodeNative = structNative.address_low_bound_node();low_bound_nodeNative.writeWord(0, this.lowBound.node.ni());
            CCharPointer low_bound_info_md_dottable_subpNative = structNative.address_low_bound_info_md_dottable_subp();low_bound_info_md_dottable_subpNative.write(this.lowBound.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer low_bound_info_md_primitiveNative = structNative.address_low_bound_info_md_primitive();low_bound_info_md_primitiveNative.writeWord(0, this.lowBound.info.md.primitive.ni());
            Pointer low_bound_info_md_primitive_real_typeNative = structNative.address_low_bound_info_md_primitive_real_type();low_bound_info_md_primitive_real_typeNative.writeWord(0, this.lowBound.info.md.primitiveRealType.ni());
            Pointer low_bound_info_rebindingsNative = structNative.address_low_bound_info_rebindings();low_bound_info_rebindingsNative.writeWord(0, this.lowBound.info.rebindings.ni());
            CCharPointer low_bound_info_from_reboundNative = structNative.address_low_bound_info_from_rebound();low_bound_info_from_reboundNative.write(this.lowBound.info.fromRebound ? (byte) 1 : (byte) 0);
            Pointer high_bound_nodeNative = structNative.address_high_bound_node();high_bound_nodeNative.writeWord(0, this.highBound.node.ni());
            CCharPointer high_bound_info_md_dottable_subpNative = structNative.address_high_bound_info_md_dottable_subp();high_bound_info_md_dottable_subpNative.write(this.highBound.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer high_bound_info_md_primitiveNative = structNative.address_high_bound_info_md_primitive();high_bound_info_md_primitiveNative.writeWord(0, this.highBound.info.md.primitive.ni());
            Pointer high_bound_info_md_primitive_real_typeNative = structNative.address_high_bound_info_md_primitive_real_type();high_bound_info_md_primitive_real_typeNative.writeWord(0, this.highBound.info.md.primitiveRealType.ni());
            Pointer high_bound_info_rebindingsNative = structNative.address_high_bound_info_rebindings();high_bound_info_rebindingsNative.writeWord(0, this.highBound.info.rebindings.ni());
            CCharPointer high_bound_info_from_reboundNative = structNative.address_high_bound_info_from_rebound();high_bound_info_from_reboundNative.write(this.highBound.info.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

        
    
    

    /**
     * Represent a set of values (as a list of choices) on a discriminant.
     */
    public static final class DiscriminantValues {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final DiscriminantValues NONE = new DiscriminantValues(
            Entity.NONE,Entity.NONE
        );

        // ----- Instance attributes -----

        public final
        Entity
        discriminant;
        public final
        Entity
        values;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        DiscriminantValues(
            final Entity discriminant,final Entity values
        ) {
            this.discriminant = discriminant;
            this.values = values;
        }

        /**
         * Create a new structure with the field values.
         */
        public static DiscriminantValues create(
            final Entity discriminant,final Entity values
        ) {
            return new DiscriminantValues(
                discriminant,values
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static DiscriminantValues wrap(
            final Pointer niPointer
        ) {
            return wrap((DiscriminantValuesNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static DiscriminantValues wrap(
            final DiscriminantValuesNative structNative
        ) {
            return new DiscriminantValues(
                new Entity(PointerWrapper.wrap(structNative.get_discriminant_node()), new EntityInfo(new Metadata(structNative.get_discriminant_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_discriminant_info_md_primitive()), PointerWrapper.wrap(structNative.get_discriminant_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_discriminant_info_rebindings()), structNative.get_discriminant_info_from_rebound() != 0)),new Entity(PointerWrapper.wrap(structNative.get_values_node()), new EntityInfo(new Metadata(structNative.get_values_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_values_info_md_primitive()), PointerWrapper.wrap(structNative.get_values_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_values_info_rebindings()), structNative.get_values_info_from_rebound() != 0))
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final DiscriminantValuesNative structNative
        ) {
            Pointer discriminant_nodeNative = structNative.address_discriminant_node();discriminant_nodeNative.writeWord(0, this.discriminant.node.ni());
            CCharPointer discriminant_info_md_dottable_subpNative = structNative.address_discriminant_info_md_dottable_subp();discriminant_info_md_dottable_subpNative.write(this.discriminant.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer discriminant_info_md_primitiveNative = structNative.address_discriminant_info_md_primitive();discriminant_info_md_primitiveNative.writeWord(0, this.discriminant.info.md.primitive.ni());
            Pointer discriminant_info_md_primitive_real_typeNative = structNative.address_discriminant_info_md_primitive_real_type();discriminant_info_md_primitive_real_typeNative.writeWord(0, this.discriminant.info.md.primitiveRealType.ni());
            Pointer discriminant_info_rebindingsNative = structNative.address_discriminant_info_rebindings();discriminant_info_rebindingsNative.writeWord(0, this.discriminant.info.rebindings.ni());
            CCharPointer discriminant_info_from_reboundNative = structNative.address_discriminant_info_from_rebound();discriminant_info_from_reboundNative.write(this.discriminant.info.fromRebound ? (byte) 1 : (byte) 0);
            Pointer values_nodeNative = structNative.address_values_node();values_nodeNative.writeWord(0, this.values.node.ni());
            CCharPointer values_info_md_dottable_subpNative = structNative.address_values_info_md_dottable_subp();values_info_md_dottable_subpNative.write(this.values.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer values_info_md_primitiveNative = structNative.address_values_info_md_primitive();values_info_md_primitiveNative.writeWord(0, this.values.info.md.primitive.ni());
            Pointer values_info_md_primitive_real_typeNative = structNative.address_values_info_md_primitive_real_type();values_info_md_primitive_real_typeNative.writeWord(0, this.values.info.md.primitiveRealType.ni());
            Pointer values_info_rebindingsNative = structNative.address_values_info_rebindings();values_info_rebindingsNative.writeWord(0, this.values.info.rebindings.ni());
            CCharPointer values_info_from_reboundNative = structNative.address_values_info_from_rebound();values_info_from_reboundNative.write(this.values.info.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

        
    
    

    /**
     * Documentation annotation.
     */
    public static final class DocAnnotation {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final DocAnnotation NONE = new DocAnnotation(
            StringWrapper.NONE,StringWrapper.NONE
        );

        // ----- Instance attributes -----

        public final
        String
        key;
        public final
        String
        value;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        DocAnnotation(
            final String key,final String value
        ) {
            this.key = key;
            this.value = value;
        }

        /**
         * Create a new structure with the field values.
         */
        public static DocAnnotation create(
            final String key,final String value
        ) {
            return new DocAnnotation(
                key,value
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static DocAnnotation wrap(
            final Pointer niPointer
        ) {
            return wrap((DocAnnotationNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static DocAnnotation wrap(
            final DocAnnotationNative structNative
        ) {
            return new DocAnnotation(
                StringWrapper.wrap(structNative.get_key()),StringWrapper.wrap(structNative.get_value())
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final DocAnnotationNative structNative
        ) {
            Pointer keyNative = structNative.address_key();StringWrapper.unwrap(this.key, keyNative);
            Pointer valueNative = structNative.address_value();StringWrapper.unwrap(this.value, valueNative);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(DocAnnotationNative structNative) {
            NI_LIB.ada_internal_doc_annotation_dec_ref(structNative);
        }

    }

        
        
        
        
        
        
        
    
    

    /**
     * Data structure used by zip_with_params, Name.call_params,
     * GenericInstantiation.inst_params, BaseAggregate.aggregate_params,
     * SubtypeIndication.subtype_constraints, and EnumRepClause.params
     * properties. Associates an expression (the actual) to a formal param
     * declaration (the parameter).
     */
    public static final class ParamActual {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final ParamActual NONE = new ParamActual(
            Entity.NONE,Entity.NONE
        );

        // ----- Instance attributes -----

        public final
        Entity
        param;
        public final
        Entity
        actual;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        ParamActual(
            final Entity param,final Entity actual
        ) {
            this.param = param;
            this.actual = actual;
        }

        /**
         * Create a new structure with the field values.
         */
        public static ParamActual create(
            final Entity param,final Entity actual
        ) {
            return new ParamActual(
                param,actual
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static ParamActual wrap(
            final Pointer niPointer
        ) {
            return wrap((ParamActualNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static ParamActual wrap(
            final ParamActualNative structNative
        ) {
            return new ParamActual(
                new Entity(PointerWrapper.wrap(structNative.get_param_node()), new EntityInfo(new Metadata(structNative.get_param_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_param_info_md_primitive()), PointerWrapper.wrap(structNative.get_param_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_param_info_rebindings()), structNative.get_param_info_from_rebound() != 0)),new Entity(PointerWrapper.wrap(structNative.get_actual_node()), new EntityInfo(new Metadata(structNative.get_actual_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_actual_info_md_primitive()), PointerWrapper.wrap(structNative.get_actual_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_actual_info_rebindings()), structNative.get_actual_info_from_rebound() != 0))
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final ParamActualNative structNative
        ) {
            Pointer param_nodeNative = structNative.address_param_node();param_nodeNative.writeWord(0, this.param.node.ni());
            CCharPointer param_info_md_dottable_subpNative = structNative.address_param_info_md_dottable_subp();param_info_md_dottable_subpNative.write(this.param.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer param_info_md_primitiveNative = structNative.address_param_info_md_primitive();param_info_md_primitiveNative.writeWord(0, this.param.info.md.primitive.ni());
            Pointer param_info_md_primitive_real_typeNative = structNative.address_param_info_md_primitive_real_type();param_info_md_primitive_real_typeNative.writeWord(0, this.param.info.md.primitiveRealType.ni());
            Pointer param_info_rebindingsNative = structNative.address_param_info_rebindings();param_info_rebindingsNative.writeWord(0, this.param.info.rebindings.ni());
            CCharPointer param_info_from_reboundNative = structNative.address_param_info_from_rebound();param_info_from_reboundNative.write(this.param.info.fromRebound ? (byte) 1 : (byte) 0);
            Pointer actual_nodeNative = structNative.address_actual_node();actual_nodeNative.writeWord(0, this.actual.node.ni());
            CCharPointer actual_info_md_dottable_subpNative = structNative.address_actual_info_md_dottable_subp();actual_info_md_dottable_subpNative.write(this.actual.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer actual_info_md_primitiveNative = structNative.address_actual_info_md_primitive();actual_info_md_primitiveNative.writeWord(0, this.actual.info.md.primitive.ni());
            Pointer actual_info_md_primitive_real_typeNative = structNative.address_actual_info_md_primitive_real_type();actual_info_md_primitive_real_typeNative.writeWord(0, this.actual.info.md.primitiveRealType.ni());
            Pointer actual_info_rebindingsNative = structNative.address_actual_info_rebindings();actual_info_rebindingsNative.writeWord(0, this.actual.info.rebindings.ni());
            CCharPointer actual_info_from_reboundNative = structNative.address_actual_info_from_rebound();actual_info_from_reboundNative.write(this.actual.info.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

        
        
        
    
    

    /**
     * Result for a cross reference query returning a reference.
     */
    public static final class RefResult {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final RefResult NONE = new RefResult(
            Entity.NONE,RefResultKind.NONE
        );

        // ----- Instance attributes -----

        public final
        Entity
        ref;
        public final
        RefResultKind
        kind;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        RefResult(
            final Entity ref,final RefResultKind kind
        ) {
            this.ref = ref;
            this.kind = kind;
        }

        /**
         * Create a new structure with the field values.
         */
        public static RefResult create(
            final Entity ref,final RefResultKind kind
        ) {
            return new RefResult(
                ref,kind
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefResult wrap(
            final Pointer niPointer
        ) {
            return wrap((RefResultNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefResult wrap(
            final RefResultNative structNative
        ) {
            return new RefResult(
                new Entity(PointerWrapper.wrap(structNative.get_ref_node()), new EntityInfo(new Metadata(structNative.get_ref_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_ref_info_md_primitive()), PointerWrapper.wrap(structNative.get_ref_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_ref_info_rebindings()), structNative.get_ref_info_from_rebound() != 0)),RefResultKind.fromC(structNative.get_kind())
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final RefResultNative structNative
        ) {
            Pointer ref_nodeNative = structNative.address_ref_node();ref_nodeNative.writeWord(0, this.ref.node.ni());
            CCharPointer ref_info_md_dottable_subpNative = structNative.address_ref_info_md_dottable_subp();ref_info_md_dottable_subpNative.write(this.ref.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer ref_info_md_primitiveNative = structNative.address_ref_info_md_primitive();ref_info_md_primitiveNative.writeWord(0, this.ref.info.md.primitive.ni());
            Pointer ref_info_md_primitive_real_typeNative = structNative.address_ref_info_md_primitive_real_type();ref_info_md_primitive_real_typeNative.writeWord(0, this.ref.info.md.primitiveRealType.ni());
            Pointer ref_info_rebindingsNative = structNative.address_ref_info_rebindings();ref_info_rebindingsNative.writeWord(0, this.ref.info.rebindings.ni());
            CCharPointer ref_info_from_reboundNative = structNative.address_ref_info_from_rebound();ref_info_from_reboundNative.write(this.ref.info.fromRebound ? (byte) 1 : (byte) 0);
            CIntPointer kindNative = structNative.address_kind();kindNative.write(this.kind.toC());
        }


    }

        
    
    

    /**
     * Result for a cross reference query returning a referenced decl.
     */
    public static final class RefdDecl {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final RefdDecl NONE = new RefdDecl(
            Entity.NONE,RefResultKind.NONE
        );

        // ----- Instance attributes -----

        public final
        Entity
        decl;
        public final
        RefResultKind
        kind;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        RefdDecl(
            final Entity decl,final RefResultKind kind
        ) {
            this.decl = decl;
            this.kind = kind;
        }

        /**
         * Create a new structure with the field values.
         */
        public static RefdDecl create(
            final Entity decl,final RefResultKind kind
        ) {
            return new RefdDecl(
                decl,kind
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefdDecl wrap(
            final Pointer niPointer
        ) {
            return wrap((RefdDeclNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefdDecl wrap(
            final RefdDeclNative structNative
        ) {
            return new RefdDecl(
                new Entity(PointerWrapper.wrap(structNative.get_decl_node()), new EntityInfo(new Metadata(structNative.get_decl_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_decl_info_md_primitive()), PointerWrapper.wrap(structNative.get_decl_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_decl_info_rebindings()), structNative.get_decl_info_from_rebound() != 0)),RefResultKind.fromC(structNative.get_kind())
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final RefdDeclNative structNative
        ) {
            Pointer decl_nodeNative = structNative.address_decl_node();decl_nodeNative.writeWord(0, this.decl.node.ni());
            CCharPointer decl_info_md_dottable_subpNative = structNative.address_decl_info_md_dottable_subp();decl_info_md_dottable_subpNative.write(this.decl.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer decl_info_md_primitiveNative = structNative.address_decl_info_md_primitive();decl_info_md_primitiveNative.writeWord(0, this.decl.info.md.primitive.ni());
            Pointer decl_info_md_primitive_real_typeNative = structNative.address_decl_info_md_primitive_real_type();decl_info_md_primitive_real_typeNative.writeWord(0, this.decl.info.md.primitiveRealType.ni());
            Pointer decl_info_rebindingsNative = structNative.address_decl_info_rebindings();decl_info_rebindingsNative.writeWord(0, this.decl.info.rebindings.ni());
            CCharPointer decl_info_from_reboundNative = structNative.address_decl_info_from_rebound();decl_info_from_reboundNative.write(this.decl.info.fromRebound ? (byte) 1 : (byte) 0);
            CIntPointer kindNative = structNative.address_kind();kindNative.write(this.kind.toC());
        }


    }

        
    
    

    /**
     * Result for a cross reference query returning a referenced defining name.
     */
    public static final class RefdDef {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final RefdDef NONE = new RefdDef(
            Entity.NONE,RefResultKind.NONE
        );

        // ----- Instance attributes -----

        public final
        Entity
        defName;
        public final
        RefResultKind
        kind;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        RefdDef(
            final Entity defName,final RefResultKind kind
        ) {
            this.defName = defName;
            this.kind = kind;
        }

        /**
         * Create a new structure with the field values.
         */
        public static RefdDef create(
            final Entity defName,final RefResultKind kind
        ) {
            return new RefdDef(
                defName,kind
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefdDef wrap(
            final Pointer niPointer
        ) {
            return wrap((RefdDefNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefdDef wrap(
            final RefdDefNative structNative
        ) {
            return new RefdDef(
                new Entity(PointerWrapper.wrap(structNative.get_def_name_node()), new EntityInfo(new Metadata(structNative.get_def_name_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_def_name_info_md_primitive()), PointerWrapper.wrap(structNative.get_def_name_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_def_name_info_rebindings()), structNative.get_def_name_info_from_rebound() != 0)),RefResultKind.fromC(structNative.get_kind())
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final RefdDefNative structNative
        ) {
            Pointer def_name_nodeNative = structNative.address_def_name_node();def_name_nodeNative.writeWord(0, this.defName.node.ni());
            CCharPointer def_name_info_md_dottable_subpNative = structNative.address_def_name_info_md_dottable_subp();def_name_info_md_dottable_subpNative.write(this.defName.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer def_name_info_md_primitiveNative = structNative.address_def_name_info_md_primitive();def_name_info_md_primitiveNative.writeWord(0, this.defName.info.md.primitive.ni());
            Pointer def_name_info_md_primitive_real_typeNative = structNative.address_def_name_info_md_primitive_real_type();def_name_info_md_primitive_real_typeNative.writeWord(0, this.defName.info.md.primitiveRealType.ni());
            Pointer def_name_info_rebindingsNative = structNative.address_def_name_info_rebindings();def_name_info_rebindingsNative.writeWord(0, this.defName.info.rebindings.ni());
            CCharPointer def_name_info_from_reboundNative = structNative.address_def_name_info_from_rebound();def_name_info_from_reboundNative.write(this.defName.info.fromRebound ? (byte) 1 : (byte) 0);
            CIntPointer kindNative = structNative.address_kind();kindNative.write(this.kind.toC());
        }


    }

        
    
    

    /**
     * Represent one of the shapes that a variant record can have, as a list of
     * the available components.
     */
    public static final class Shape {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Shape NONE = new Shape(
            AdaNodeArray.NONE,DiscriminantValuesArray.NONE
        );

        // ----- Instance attributes -----

        public final
        AdaNodeArray
        components;
        public final
        DiscriminantValuesArray
        discriminantsValues;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Shape(
            final AdaNodeArray components,final DiscriminantValuesArray discriminantsValues
        ) {
            this.components = components;
            this.discriminantsValues = discriminantsValues;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Shape create(
            final AdaNodeArray components,final DiscriminantValuesArray discriminantsValues
        ) {
            return new Shape(
                components,discriminantsValues
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Shape wrap(
            final Pointer niPointer
        ) {
            return wrap((ShapeNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Shape wrap(
            final ShapeNative structNative
        ) {
            return new Shape(
                AdaNodeArray.wrap(structNative.get_components()),DiscriminantValuesArray.wrap(structNative.get_discriminants_values())
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final ShapeNative structNative
        ) {
            Pointer componentsNative = structNative.address_components();this.components.unwrap(componentsNative);
            Pointer discriminants_valuesNative = structNative.address_discriminants_values();this.discriminantsValues.unwrap(discriminants_valuesNative);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(ShapeNative structNative) {
            NI_LIB.ada_internal_shape_dec_ref(structNative);
        }

    }

        
    
    

    /**
     * Represent a substitution of a BasicDecl by a given value. This can then
     * be used as part of an environment in the eval_as_*_in_env property. See
     * the declaration of those properties for more details.
     */
    public static final class Substitution {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Substitution NONE = new Substitution(
            Entity.NONE,BigIntegerWrapper.NONE,Entity.NONE
        );

        // ----- Instance attributes -----

        public final
        Entity
        fromDecl;
        public final
        BigInteger
        toValue;
        public final
        Entity
        valueType;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Substitution(
            final Entity fromDecl,final BigInteger toValue,final Entity valueType
        ) {
            this.fromDecl = fromDecl;
            this.toValue = toValue;
            this.valueType = valueType;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Substitution create(
            final Entity fromDecl,final BigInteger toValue,final Entity valueType
        ) {
            return new Substitution(
                fromDecl,toValue,valueType
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Substitution wrap(
            final Pointer niPointer
        ) {
            return wrap((SubstitutionNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Substitution wrap(
            final SubstitutionNative structNative
        ) {
            return new Substitution(
                new Entity(PointerWrapper.wrap(structNative.get_from_decl_node()), new EntityInfo(new Metadata(structNative.get_from_decl_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_from_decl_info_md_primitive()), PointerWrapper.wrap(structNative.get_from_decl_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_from_decl_info_rebindings()), structNative.get_from_decl_info_from_rebound() != 0)),BigIntegerWrapper.wrap(structNative.get_to_value()),new Entity(PointerWrapper.wrap(structNative.get_value_type_node()), new EntityInfo(new Metadata(structNative.get_value_type_info_md_dottable_subp() != 0, PointerWrapper.wrap(structNative.get_value_type_info_md_primitive()), PointerWrapper.wrap(structNative.get_value_type_info_md_primitive_real_type())), PointerWrapper.wrap(structNative.get_value_type_info_rebindings()), structNative.get_value_type_info_from_rebound() != 0))
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final SubstitutionNative structNative
        ) {
            Pointer from_decl_nodeNative = structNative.address_from_decl_node();from_decl_nodeNative.writeWord(0, this.fromDecl.node.ni());
            CCharPointer from_decl_info_md_dottable_subpNative = structNative.address_from_decl_info_md_dottable_subp();from_decl_info_md_dottable_subpNative.write(this.fromDecl.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer from_decl_info_md_primitiveNative = structNative.address_from_decl_info_md_primitive();from_decl_info_md_primitiveNative.writeWord(0, this.fromDecl.info.md.primitive.ni());
            Pointer from_decl_info_md_primitive_real_typeNative = structNative.address_from_decl_info_md_primitive_real_type();from_decl_info_md_primitive_real_typeNative.writeWord(0, this.fromDecl.info.md.primitiveRealType.ni());
            Pointer from_decl_info_rebindingsNative = structNative.address_from_decl_info_rebindings();from_decl_info_rebindingsNative.writeWord(0, this.fromDecl.info.rebindings.ni());
            CCharPointer from_decl_info_from_reboundNative = structNative.address_from_decl_info_from_rebound();from_decl_info_from_reboundNative.write(this.fromDecl.info.fromRebound ? (byte) 1 : (byte) 0);
            Pointer to_valueNative = structNative.address_to_value();BigIntegerWrapper.unwrap(this.toValue, to_valueNative);
            Pointer value_type_nodeNative = structNative.address_value_type_node();value_type_nodeNative.writeWord(0, this.valueType.node.ni());
            CCharPointer value_type_info_md_dottable_subpNative = structNative.address_value_type_info_md_dottable_subp();value_type_info_md_dottable_subpNative.write(this.valueType.info.md.dottableSubp ? (byte) 1 : (byte) 0);
            Pointer value_type_info_md_primitiveNative = structNative.address_value_type_info_md_primitive();value_type_info_md_primitiveNative.writeWord(0, this.valueType.info.md.primitive.ni());
            Pointer value_type_info_md_primitive_real_typeNative = structNative.address_value_type_info_md_primitive_real_type();value_type_info_md_primitive_real_typeNative.writeWord(0, this.valueType.info.md.primitiveRealType.ni());
            Pointer value_type_info_rebindingsNative = structNative.address_value_type_info_rebindings();value_type_info_rebindingsNative.writeWord(0, this.valueType.info.rebindings.ni());
            CCharPointer value_type_info_from_reboundNative = structNative.address_value_type_info_from_rebound();value_type_info_from_reboundNative.write(this.valueType.info.fromRebound ? (byte) 1 : (byte) 0);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(SubstitutionNative structNative) {
            NI_LIB.ada_internal_substitution_dec_ref(structNative);
        }

    }


    // ===== Generated array wrapping classes =====

    /**
     * This class is the base of all array wrapping class.
     */
    public static abstract class ArrayBase<T> implements Iterable<T> {

        // ----- Attributes -----

        /** The content of the array. */
        protected final T[] content;

        // ----- Constructors -----

        /**
         * Protected constructor.
         */
        protected ArrayBase(
            final T[] content
        ) {
            this.content = content;
        }

        // ----- Instance methods -----

        /**
         * Get the size of the array.
         *
         * @return The size of the native array.
         */
        public int size() {
            return this.content.length;
        }

        /**
         * Get the element at the given place in the array.
         *
         * @param i The index of the element to get.
         * @return The element at the given index.
         * @throws ArrayIndexOutOfBoundsException If the requested index is
         * out of bounds.
         */
        public T get(
            final int i
        ) {
            return this.content[i];
        }

        /**
         * Set the element at the given index.
         *
         * @param i The index of the element to set.
         * @param elem The element to place in the array.
         * @throws ArrayIndexOutOfBoundsException If the requested index is
         * out of bounds.
         */
        public void set(
            final int i,
            final T elem
        ) {
            this.content[i] = elem;
        }

        /** @see java.lang.Iterable#iterator() */
        @Override
        public Iterator<T> iterator() {
            return new LangkitArrayIterator<T>(this);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            final StringBuilder res = new StringBuilder("[");
            for(int i = 0 ; i < this.size() ; i++) {
                res.append(this.get(i).toString());
                if(i < this.size() - 1) res.append(", ");
            }
            res.append(']');
            return res.toString();
        }

        // ----- Inner classes -----

        /**
         * The iterator class for the langkit arrays
         */
        protected static class LangkitArrayIterator<U>
        implements Iterator<U> {

            // ----- Attributes -----

            /** The array to iterate on. */
            private final ArrayBase<U> array;

            /** The current index. */
            private int index;

            // ----- Constructors -----

            /**
             * Create a new array iterator.
             *
             * @param array The array to iterate on.
             */
            LangkitArrayIterator(
                final ArrayBase<U> array
            ) {
                this.array = array;
                this.index = 0;
            }

            // ----- Instance methods -----

            /** @see java.util.Iterator#hasNext() */
            @Override
            public boolean hasNext() {
                return this.index < this.array.size();
            }

            /** @see java.util.Iterator#next() */
            @Override
            public U next() {
                return this.array.get(this.index++);
            }

        }

    }

    
    

    /**
     * This class represents the ada_discriminant_values_array Java wrapping class
     */
    public static final class
    DiscriminantValuesArray extends ArrayBase<DiscriminantValues> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final DiscriminantValuesArray NONE = new DiscriminantValuesArray(
            new DiscriminantValues[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        DiscriminantValuesArray(
            final DiscriminantValues[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static DiscriminantValuesArray jniCreate(
            final DiscriminantValues[] jniContent
        ) {
            final DiscriminantValues[] content =
                new DiscriminantValues[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new DiscriminantValuesArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static DiscriminantValuesArray create(
            final int size
        ) {
            return new DiscriminantValuesArray(
                new DiscriminantValues[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static DiscriminantValuesArray wrap(
            final Pointer pointer
        ) {
            return wrap((DiscriminantValuesArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static DiscriminantValuesArray wrap(
            final DiscriminantValuesArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final DiscriminantValues[] content = new DiscriminantValues[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            DiscriminantValuesNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(DiscriminantValuesNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = DiscriminantValues.wrap(toRead);
            }

            // Return the new langkit array
            return new DiscriminantValuesArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final DiscriminantValuesArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        DiscriminantValuesArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final DiscriminantValuesArrayNative res = NI_LIB.ada_discriminant_values_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            DiscriminantValuesNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(DiscriminantValuesNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((DiscriminantValuesArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final DiscriminantValuesArrayNative arrayNative
        ) {
            NI_LIB.ada_discriminant_values_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private DiscriminantValues[] jniContent() {
            final DiscriminantValues[] res =
                new DiscriminantValues[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_doc_annotation_array Java wrapping class
     */
    public static final class
    DocAnnotationArray extends ArrayBase<DocAnnotation> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final DocAnnotationArray NONE = new DocAnnotationArray(
            new DocAnnotation[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        DocAnnotationArray(
            final DocAnnotation[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static DocAnnotationArray jniCreate(
            final DocAnnotation[] jniContent
        ) {
            final DocAnnotation[] content =
                new DocAnnotation[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new DocAnnotationArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static DocAnnotationArray create(
            final int size
        ) {
            return new DocAnnotationArray(
                new DocAnnotation[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static DocAnnotationArray wrap(
            final Pointer pointer
        ) {
            return wrap((DocAnnotationArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static DocAnnotationArray wrap(
            final DocAnnotationArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final DocAnnotation[] content = new DocAnnotation[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            DocAnnotationNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(DocAnnotationNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = DocAnnotation.wrap(toRead);
            }

            // Return the new langkit array
            return new DocAnnotationArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final DocAnnotationArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        DocAnnotationArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final DocAnnotationArrayNative res = NI_LIB.ada_doc_annotation_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            DocAnnotationNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(DocAnnotationNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((DocAnnotationArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final DocAnnotationArrayNative arrayNative
        ) {
            NI_LIB.ada_doc_annotation_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private DocAnnotation[] jniContent() {
            final DocAnnotation[] res =
                new DocAnnotation[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_ada_node_array Java wrapping class
     */
    public static final class
    AdaNodeArray extends ArrayBase<AdaNode> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final AdaNodeArray NONE = new AdaNodeArray(
            new AdaNode[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        AdaNodeArray(
            final AdaNode[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static AdaNodeArray jniCreate(
            final Entity[] jniContent
        ) {
            final AdaNode[] content =
                new AdaNode[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    AdaNode.fromEntity(jniContent[i]);
            }
            return new AdaNodeArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static AdaNodeArray create(
            final int size
        ) {
            return new AdaNodeArray(
                new AdaNode[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static AdaNodeArray wrap(
            final Pointer pointer
        ) {
            return wrap((AdaNodeArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static AdaNodeArray wrap(
            final AdaNodeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final AdaNode[] content = new AdaNode[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            EntityNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(EntityNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = AdaNode.fromEntity(Entity.wrap(toRead));
            }

            // Return the new langkit array
            return new AdaNodeArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final AdaNodeArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        AdaNodeArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final AdaNodeArrayNative res = NI_LIB.ada_ada_node_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            EntityNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(EntityNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].entity.unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((AdaNodeArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final AdaNodeArrayNative arrayNative
        ) {
            NI_LIB.ada_ada_node_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private Entity[] jniContent() {
            final Entity[] res =
                new Entity[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = (this.content[i] != null ? this.content[i].entity : null);
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_param_actual_array Java wrapping class
     */
    public static final class
    ParamActualArray extends ArrayBase<ParamActual> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final ParamActualArray NONE = new ParamActualArray(
            new ParamActual[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        ParamActualArray(
            final ParamActual[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static ParamActualArray jniCreate(
            final ParamActual[] jniContent
        ) {
            final ParamActual[] content =
                new ParamActual[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new ParamActualArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static ParamActualArray create(
            final int size
        ) {
            return new ParamActualArray(
                new ParamActual[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static ParamActualArray wrap(
            final Pointer pointer
        ) {
            return wrap((ParamActualArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static ParamActualArray wrap(
            final ParamActualArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final ParamActual[] content = new ParamActual[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            ParamActualNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(ParamActualNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = ParamActual.wrap(toRead);
            }

            // Return the new langkit array
            return new ParamActualArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final ParamActualArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        ParamActualArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final ParamActualArrayNative res = NI_LIB.ada_param_actual_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            ParamActualNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(ParamActualNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((ParamActualArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final ParamActualArrayNative arrayNative
        ) {
            NI_LIB.ada_param_actual_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private ParamActual[] jniContent() {
            final ParamActual[] res =
                new ParamActual[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_ref_result_array Java wrapping class
     */
    public static final class
    RefResultArray extends ArrayBase<RefResult> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final RefResultArray NONE = new RefResultArray(
            new RefResult[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        RefResultArray(
            final RefResult[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static RefResultArray jniCreate(
            final RefResult[] jniContent
        ) {
            final RefResult[] content =
                new RefResult[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new RefResultArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static RefResultArray create(
            final int size
        ) {
            return new RefResultArray(
                new RefResult[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static RefResultArray wrap(
            final Pointer pointer
        ) {
            return wrap((RefResultArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static RefResultArray wrap(
            final RefResultArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final RefResult[] content = new RefResult[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            RefResultNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(RefResultNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = RefResult.wrap(toRead);
            }

            // Return the new langkit array
            return new RefResultArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final RefResultArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        RefResultArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final RefResultArrayNative res = NI_LIB.ada_ref_result_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            RefResultNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(RefResultNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((RefResultArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final RefResultArrayNative arrayNative
        ) {
            NI_LIB.ada_ref_result_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private RefResult[] jniContent() {
            final RefResult[] res =
                new RefResult[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_shape_array Java wrapping class
     */
    public static final class
    ShapeArray extends ArrayBase<Shape> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final ShapeArray NONE = new ShapeArray(
            new Shape[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        ShapeArray(
            final Shape[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static ShapeArray jniCreate(
            final Shape[] jniContent
        ) {
            final Shape[] content =
                new Shape[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new ShapeArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static ShapeArray create(
            final int size
        ) {
            return new ShapeArray(
                new Shape[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static ShapeArray wrap(
            final Pointer pointer
        ) {
            return wrap((ShapeArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static ShapeArray wrap(
            final ShapeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final Shape[] content = new Shape[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            ShapeNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(ShapeNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = Shape.wrap(toRead);
            }

            // Return the new langkit array
            return new ShapeArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final ShapeArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        ShapeArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final ShapeArrayNative res = NI_LIB.ada_shape_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            ShapeNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(ShapeNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((ShapeArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final ShapeArrayNative arrayNative
        ) {
            NI_LIB.ada_shape_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private Shape[] jniContent() {
            final Shape[] res =
                new Shape[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_substitution_array Java wrapping class
     */
    public static final class
    SubstitutionArray extends ArrayBase<Substitution> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final SubstitutionArray NONE = new SubstitutionArray(
            new Substitution[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        SubstitutionArray(
            final Substitution[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static SubstitutionArray jniCreate(
            final Substitution[] jniContent
        ) {
            final Substitution[] content =
                new Substitution[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new SubstitutionArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static SubstitutionArray create(
            final int size
        ) {
            return new SubstitutionArray(
                new Substitution[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static SubstitutionArray wrap(
            final Pointer pointer
        ) {
            return wrap((SubstitutionArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static SubstitutionArray wrap(
            final SubstitutionArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final Substitution[] content = new Substitution[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            SubstitutionNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(SubstitutionNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = Substitution.wrap(toRead);
            }

            // Return the new langkit array
            return new SubstitutionArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final SubstitutionArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        SubstitutionArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final SubstitutionArrayNative res = NI_LIB.ada_substitution_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            SubstitutionNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(SubstitutionNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((SubstitutionArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final SubstitutionArrayNative arrayNative
        ) {
            NI_LIB.ada_substitution_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private Substitution[] jniContent() {
            final Substitution[] res =
                new Substitution[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_analysis_unit_array Java wrapping class
     */
    public static final class
    AnalysisUnitArray extends ArrayBase<AnalysisUnit> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final AnalysisUnitArray NONE = new AnalysisUnitArray(
            new AnalysisUnit[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        AnalysisUnitArray(
            final AnalysisUnit[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static AnalysisUnitArray jniCreate(
            final AnalysisUnit[] jniContent
        ) {
            final AnalysisUnit[] content =
                new AnalysisUnit[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new AnalysisUnitArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static AnalysisUnitArray create(
            final int size
        ) {
            return new AnalysisUnitArray(
                new AnalysisUnit[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static AnalysisUnitArray wrap(
            final Pointer pointer
        ) {
            return wrap((AnalysisUnitArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static AnalysisUnitArray wrap(
            final AnalysisUnitArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final AnalysisUnit[] content = new AnalysisUnit[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            Pointer toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(AnalysisUnitNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = AnalysisUnit.wrap(toRead);
            }

            // Return the new langkit array
            return new AnalysisUnitArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final AnalysisUnitArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        AnalysisUnitArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final AnalysisUnitArrayNative res = NI_LIB.ada_analysis_unit_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            Pointer toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(AnalysisUnitNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                toWrite.writeWord(0, this.content[i].unwrap());
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((AnalysisUnitArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final AnalysisUnitArrayNative arrayNative
        ) {
            NI_LIB.ada_analysis_unit_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private AnalysisUnit[] jniContent() {
            final AnalysisUnit[] res =
                new AnalysisUnit[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the ada_unbounded_text_type_array Java wrapping class
     */
    public static final class
    UnboundedTextTypeArray extends ArrayBase<Symbol> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final UnboundedTextTypeArray NONE = new UnboundedTextTypeArray(
            new Symbol[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        UnboundedTextTypeArray(
            final Symbol[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static UnboundedTextTypeArray jniCreate(
            final Symbol[] jniContent
        ) {
            final Symbol[] content =
                new Symbol[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return new UnboundedTextTypeArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static UnboundedTextTypeArray create(
            final int size
        ) {
            return new UnboundedTextTypeArray(
                new Symbol[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static UnboundedTextTypeArray wrap(
            final Pointer pointer
        ) {
            return wrap((UnboundedTextTypeArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static UnboundedTextTypeArray wrap(
            final UnboundedTextTypeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final Symbol[] content = new Symbol[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            SymbolNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(SymbolNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = Symbol.wrap(toRead);
            }

            // Return the new langkit array
            return new UnboundedTextTypeArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            , final AnalysisContext currentContext
        ) {
            // Create a new native array with the size
            final UnboundedTextTypeArrayNative resNative = this.unwrap(
                currentContext
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        UnboundedTextTypeArrayNative unwrap(
            final AnalysisContext currentContext
        ) {
            // Create a new native array with the size
            final UnboundedTextTypeArrayNative res = NI_LIB.ada_unbounded_text_type_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            SymbolNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(SymbolNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].unwrap(toWrite, currentContext);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((UnboundedTextTypeArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final UnboundedTextTypeArrayNative arrayNative
        ) {
            NI_LIB.ada_unbounded_text_type_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private Symbol[] jniContent() {
            final Symbol[] res =
                new Symbol[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = this.content[i];
            }
            return res;
        }

    }


    // ===== Generated iterator wrapping classes =====

    
    

    /**

     */
    public static final class CompletionItemIterator implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton for the iterator NONE value. */
        public static final CompletionItemIterator NONE = new CompletionItemIterator(
            PointerWrapper.nullPointer()
        );

        // ----- Attributes -----

        /** The pointer to the native iterator. */
        public final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new CompletionItemIterator from the reference to its native value.
         *
         * @param reference The reference to the native iterator value.
         */
        private CompletionItemIterator(
            PointerWrapper reference
        ) {
            this.reference = reference;
        }

        /**
         * Wrap a pointer to the iterator native value in the Java class.
         *
         * @param niPointer The pointer to the NI iterator native value.
         * @return The newly created iterator or null if the given pointer
         * is null.
         */
        static CompletionItemIterator wrap(
            Pointer niPointer
        ) {
            if(niPointer.isNull()) return null;
            else return wrap((CompletionItemIteratorNative) niPointer.readWord(0));
        }

        /**
         * Wrap the iterator native value in the Java class.
         *
         * @param iteratorNative The NI iterator native value.
         * @return The newly created iterator or null if the iterator value
         * is null.
         */
        static CompletionItemIterator wrap(
            CompletionItemIteratorNative iteratorNative
        ) {
            if(((PointerBase) iteratorNative).isNull()) return null;
            else return new CompletionItemIterator(
                new PointerWrapper(iteratorNative)
            );
        }

        // ----- Instance methods -----

        /** @see java/lang/AutoCloseable#close() */
        public void close() {
            System.err.println("HANDLE THE ITERATOR CLOSING !!!");
        }

    }


    // ===== Node classes =====

    /**
     * Data type for all nodes. Nodes are assembled to make up a tree.  See the
     * node primitives below to inspect such trees.
     *
     * Unlike for contexts and units, this type has weak-reference semantics:
     * keeping a reference to a node has no effect on the decision to keep the
     * unit that it owns allocated. This means that once all references to the
     * context and units related to a node are dropped, the context and its
     * units are deallocated and the node becomes a stale reference: most
     * operations on it will raise a ``Stale_Reference_Error``.
     *
     * Note that since reparsing an analysis unit deallocates all the nodes it
     * contains, this operation makes all reference to these nodes stale as
     * well.
     */
    public static abstract class AdaNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AdaNode";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pDeclarativeScope",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_declarative_scope",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pEnclosingCompilationUnit",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_enclosing_compilation_unit",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pGetUninstantiatedNode",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_get_uninstantiated_node",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pComplete",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_complete",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pValidKeywords",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_valid_keywords",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pGenericInstantiations",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_generic_instantiations",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pSemanticParent",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_semantic_parent",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pParentBasicDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_parent_basic_decl",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pFilterIsImportedBy",
                        new Class[]{AnalysisUnitArray.class,boolean.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new Param(
                        AnalysisUnitArray.class,
                        "units"
                    ));
                    parameters.add(new Param(
                        boolean.class,
                        "transitive"
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_filter_is_imported_by",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pXrefEntryPoint",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_xref_entry_point",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pResolveNames",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_resolve_names",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pStandardUnit",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_standard_unit",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pStdEntity",
                        new Class[]{Symbol.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new Param(
                        Symbol.class,
                        "sym"
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_std_entity",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pBoolType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_bool_type",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pIntType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_int_type",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pUniversalIntType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_universal_int_type",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pUniversalRealType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_universal_real_type",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pStdCharType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_std_char_type",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pStdWideCharType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_std_wide_char_type",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pStdWideWideCharType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_std_wide_wide_char_type",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pTopLevelDecl",
                        new Class[]{AnalysisUnit.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new Param(
                        AnalysisUnit.class,
                        "unit"
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_top_level_decl",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pChoiceMatch",
                        new Class[]{BigInteger.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new Param(
                        BigInteger.class,
                        "value"
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_choice_match",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "pGnatXref",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new ParamWithDefaultValue(
                        boolean.class,
                        "impreciseFallback",
                        false
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_gnat_xref",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "parent",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "parent",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "parents",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new ParamWithDefaultValue(
                        boolean.class,
                        "withSelf",
                        true
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "parents",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "children",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "children",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "tokenStart",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "token_start",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "tokenEnd",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "token_end",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "childIndex",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "child_index",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "previousSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "previous_sibling",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "nextSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "next_sibling",
                        new LibadalangField(method, parameters)
                    );
                }
                
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "isGhost",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "is_ghost",
                        new LibadalangField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AdaNode.class.getMethod(
                        "fullSlocImage",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "full_sloc_image",
                        new LibadalangField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AdaNode NONE = new NoneNode();

        /** The entity of the node. */
        public final Entity entity;

        /** The analysis unit that owns the node. */
        protected AnalysisUnit unit;

        /** The cache for the image of the node. */
        protected String image;

        // ----- Constructors -----

        /**
         * Create a new node with its entity.
         *
         * @param entity The node's entity.
         */
        protected AdaNode(
            final Entity entity
        ) {
            this.entity = entity;
            this.unit = null;
            this.image = null;
        }

        /**
         * Get a node from the given entity.
         *
         * @param entity The entity to get the node from.
         * @return The newly created node.
         */
        public static AdaNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AdaNode.NONE :
                dispatchNodeCreation(entity);
        }

        /**
         * Dispatch the node creation to return the valid Java object
         * according to the node kind.
         *
         * @param entity The entity to create the node from.
         * @return The wrapped node in the correct class.
         */
        protected static AdaNode dispatchNodeCreation(
            final Entity entity
        ) {
            int nodeKind = -1;

            if(ImageInfo.inImageCode()) {
                EntityNative entityNative = StackValue.get(
                    EntityNative.class
                );
                entity.unwrap(entityNative);
                nodeKind = NI_LIB.ada_node_kind(entityNative);
            } else {
                nodeKind = JNI_LIB.ada_node_kind(entity);
            }

            switch(nodeKind) {
                case 1:
                    return entity.node.isNull() ?
                        AbortAbsent.NONE :
                        new AbortAbsent(entity);
                case 2:
                    return entity.node.isNull() ?
                        AbortPresent.NONE :
                        new AbortPresent(entity);
                case 3:
                    return entity.node.isNull() ?
                        AbstractAbsent.NONE :
                        new AbstractAbsent(entity);
                case 4:
                    return entity.node.isNull() ?
                        AbstractPresent.NONE :
                        new AbstractPresent(entity);
                case 5:
                    return entity.node.isNull() ?
                        AdaNodeList.NONE :
                        new AdaNodeList(entity);
                case 6:
                    return entity.node.isNull() ?
                        AbstractStateDeclList.NONE :
                        new AbstractStateDeclList(entity);
                case 7:
                    return entity.node.isNull() ?
                        AlternativesList.NONE :
                        new AlternativesList(entity);
                case 8:
                    return entity.node.isNull() ?
                        ConstraintList.NONE :
                        new ConstraintList(entity);
                case 9:
                    return entity.node.isNull() ?
                        DeclList.NONE :
                        new DeclList(entity);
                case 10:
                    return entity.node.isNull() ?
                        StmtList.NONE :
                        new StmtList(entity);
                case 11:
                    return entity.node.isNull() ?
                        AspectAssocList.NONE :
                        new AspectAssocList(entity);
                case 12:
                    return entity.node.isNull() ?
                        BaseAssocList.NONE :
                        new BaseAssocList(entity);
                case 13:
                    return entity.node.isNull() ?
                        AssocList.NONE :
                        new AssocList(entity);
                case 14:
                    return entity.node.isNull() ?
                        BasicDeclList.NONE :
                        new BasicDeclList(entity);
                case 15:
                    return entity.node.isNull() ?
                        CaseExprAlternativeList.NONE :
                        new CaseExprAlternativeList(entity);
                case 16:
                    return entity.node.isNull() ?
                        CaseStmtAlternativeList.NONE :
                        new CaseStmtAlternativeList(entity);
                case 17:
                    return entity.node.isNull() ?
                        CompilationUnitList.NONE :
                        new CompilationUnitList(entity);
                case 18:
                    return entity.node.isNull() ?
                        ConcatOperandList.NONE :
                        new ConcatOperandList(entity);
                case 19:
                    return entity.node.isNull() ?
                        ContractCaseAssocList.NONE :
                        new ContractCaseAssocList(entity);
                case 20:
                    return entity.node.isNull() ?
                        DefiningNameList.NONE :
                        new DefiningNameList(entity);
                case 21:
                    return entity.node.isNull() ?
                        DiscriminantSpecList.NONE :
                        new DiscriminantSpecList(entity);
                case 22:
                    return entity.node.isNull() ?
                        ElsifExprPartList.NONE :
                        new ElsifExprPartList(entity);
                case 23:
                    return entity.node.isNull() ?
                        ElsifStmtPartList.NONE :
                        new ElsifStmtPartList(entity);
                case 24:
                    return entity.node.isNull() ?
                        EnumLiteralDeclList.NONE :
                        new EnumLiteralDeclList(entity);
                case 25:
                    return entity.node.isNull() ?
                        ExprAlternativesList.NONE :
                        new ExprAlternativesList(entity);
                case 26:
                    return entity.node.isNull() ?
                        DiscriminantChoiceList.NONE :
                        new DiscriminantChoiceList(entity);
                case 27:
                    return entity.node.isNull() ?
                        NameList.NONE :
                        new NameList(entity);
                case 28:
                    return entity.node.isNull() ?
                        ParentList.NONE :
                        new ParentList(entity);
                case 29:
                    return entity.node.isNull() ?
                        ParamSpecList.NONE :
                        new ParamSpecList(entity);
                case 30:
                    return entity.node.isNull() ?
                        PragmaNodeList.NONE :
                        new PragmaNodeList(entity);
                case 31:
                    return entity.node.isNull() ?
                        SelectWhenPartList.NONE :
                        new SelectWhenPartList(entity);
                case 32:
                    return entity.node.isNull() ?
                        UnconstrainedArrayIndexList.NONE :
                        new UnconstrainedArrayIndexList(entity);
                case 33:
                    return entity.node.isNull() ?
                        VariantList.NONE :
                        new VariantList(entity);
                case 34:
                    return entity.node.isNull() ?
                        AliasedAbsent.NONE :
                        new AliasedAbsent(entity);
                case 35:
                    return entity.node.isNull() ?
                        AliasedPresent.NONE :
                        new AliasedPresent(entity);
                case 36:
                    return entity.node.isNull() ?
                        AllAbsent.NONE :
                        new AllAbsent(entity);
                case 37:
                    return entity.node.isNull() ?
                        AllPresent.NONE :
                        new AllPresent(entity);
                case 38:
                    return entity.node.isNull() ?
                        ConstrainedArrayIndices.NONE :
                        new ConstrainedArrayIndices(entity);
                case 39:
                    return entity.node.isNull() ?
                        UnconstrainedArrayIndices.NONE :
                        new UnconstrainedArrayIndices(entity);
                case 40:
                    return entity.node.isNull() ?
                        AspectAssoc.NONE :
                        new AspectAssoc(entity);
                case 41:
                    return entity.node.isNull() ?
                        AtClause.NONE :
                        new AtClause(entity);
                case 42:
                    return entity.node.isNull() ?
                        AttributeDefClause.NONE :
                        new AttributeDefClause(entity);
                case 43:
                    return entity.node.isNull() ?
                        EnumRepClause.NONE :
                        new EnumRepClause(entity);
                case 44:
                    return entity.node.isNull() ?
                        RecordRepClause.NONE :
                        new RecordRepClause(entity);
                case 45:
                    return entity.node.isNull() ?
                        AspectSpec.NONE :
                        new AspectSpec(entity);
                case 46:
                    return entity.node.isNull() ?
                        ContractCaseAssoc.NONE :
                        new ContractCaseAssoc(entity);
                case 47:
                    return entity.node.isNull() ?
                        PragmaArgumentAssoc.NONE :
                        new PragmaArgumentAssoc(entity);
                case 48:
                    return entity.node.isNull() ?
                        EntrySpec.NONE :
                        new EntrySpec(entity);
                case 49:
                    return entity.node.isNull() ?
                        EnumSubpSpec.NONE :
                        new EnumSubpSpec(entity);
                case 50:
                    return entity.node.isNull() ?
                        SubpSpec.NONE :
                        new SubpSpec(entity);
                case 51:
                    return entity.node.isNull() ?
                        SyntheticBinarySpec.NONE :
                        new SyntheticBinarySpec(entity);
                case 52:
                    return entity.node.isNull() ?
                        SyntheticUnarySpec.NONE :
                        new SyntheticUnarySpec(entity);
                case 53:
                    return entity.node.isNull() ?
                        ComponentList.NONE :
                        new ComponentList(entity);
                case 54:
                    return entity.node.isNull() ?
                        KnownDiscriminantPart.NONE :
                        new KnownDiscriminantPart(entity);
                case 55:
                    return entity.node.isNull() ?
                        UnknownDiscriminantPart.NONE :
                        new UnknownDiscriminantPart(entity);
                case 56:
                    return entity.node.isNull() ?
                        EntryCompletionFormalParams.NONE :
                        new EntryCompletionFormalParams(entity);
                case 57:
                    return entity.node.isNull() ?
                        GenericFormalPart.NONE :
                        new GenericFormalPart(entity);
                case 58:
                    return entity.node.isNull() ?
                        NullRecordDef.NONE :
                        new NullRecordDef(entity);
                case 59:
                    return entity.node.isNull() ?
                        RecordDef.NONE :
                        new RecordDef(entity);
                case 60:
                    return entity.node.isNull() ?
                        AggregateAssoc.NONE :
                        new AggregateAssoc(entity);
                case 61:
                    return entity.node.isNull() ?
                        MultiDimArrayAssoc.NONE :
                        new MultiDimArrayAssoc(entity);
                case 62:
                    return entity.node.isNull() ?
                        CompositeConstraintAssoc.NONE :
                        new CompositeConstraintAssoc(entity);
                case 63:
                    return entity.node.isNull() ?
                        IteratedAssoc.NONE :
                        new IteratedAssoc(entity);
                case 64:
                    return entity.node.isNull() ?
                        ParamAssoc.NONE :
                        new ParamAssoc(entity);
                case 65:
                    return entity.node.isNull() ?
                        AbstractStateDecl.NONE :
                        new AbstractStateDecl(entity);
                case 66:
                    return entity.node.isNull() ?
                        AnonymousExprDecl.NONE :
                        new AnonymousExprDecl(entity);
                case 67:
                    return entity.node.isNull() ?
                        ComponentDecl.NONE :
                        new ComponentDecl(entity);
                case 68:
                    return entity.node.isNull() ?
                        DiscriminantSpec.NONE :
                        new DiscriminantSpec(entity);
                case 69:
                    return entity.node.isNull() ?
                        GenericFormalObjDecl.NONE :
                        new GenericFormalObjDecl(entity);
                case 70:
                    return entity.node.isNull() ?
                        GenericFormalPackage.NONE :
                        new GenericFormalPackage(entity);
                case 71:
                    return entity.node.isNull() ?
                        GenericFormalSubpDecl.NONE :
                        new GenericFormalSubpDecl(entity);
                case 72:
                    return entity.node.isNull() ?
                        GenericFormalTypeDecl.NONE :
                        new GenericFormalTypeDecl(entity);
                case 73:
                    return entity.node.isNull() ?
                        ParamSpec.NONE :
                        new ParamSpec(entity);
                case 74:
                    return entity.node.isNull() ?
                        SyntheticFormalParamDecl.NONE :
                        new SyntheticFormalParamDecl(entity);
                case 75:
                    return entity.node.isNull() ?
                        GenericPackageInternal.NONE :
                        new GenericPackageInternal(entity);
                case 76:
                    return entity.node.isNull() ?
                        PackageDecl.NONE :
                        new PackageDecl(entity);
                case 77:
                    return entity.node.isNull() ?
                        DiscreteBaseSubtypeDecl.NONE :
                        new DiscreteBaseSubtypeDecl(entity);
                case 78:
                    return entity.node.isNull() ?
                        SubtypeDecl.NONE :
                        new SubtypeDecl(entity);
                case 79:
                    return entity.node.isNull() ?
                        ClasswideTypeDecl.NONE :
                        new ClasswideTypeDecl(entity);
                case 80:
                    return entity.node.isNull() ?
                        IncompleteTypeDecl.NONE :
                        new IncompleteTypeDecl(entity);
                case 81:
                    return entity.node.isNull() ?
                        IncompleteFormalTypeDecl.NONE :
                        new IncompleteFormalTypeDecl(entity);
                case 82:
                    return entity.node.isNull() ?
                        IncompleteTaggedTypeDecl.NONE :
                        new IncompleteTaggedTypeDecl(entity);
                case 83:
                    return entity.node.isNull() ?
                        ProtectedTypeDecl.NONE :
                        new ProtectedTypeDecl(entity);
                case 84:
                    return entity.node.isNull() ?
                        TaskTypeDecl.NONE :
                        new TaskTypeDecl(entity);
                case 85:
                    return entity.node.isNull() ?
                        SingleTaskTypeDecl.NONE :
                        new SingleTaskTypeDecl(entity);
                case 86:
                    return entity.node.isNull() ?
                        AnonymousTypeDecl.NONE :
                        new AnonymousTypeDecl(entity);
                case 87:
                    return entity.node.isNull() ?
                        SynthAnonymousTypeDecl.NONE :
                        new SynthAnonymousTypeDecl(entity);
                case 88:
                    return entity.node.isNull() ?
                        ConcreteTypeDecl.NONE :
                        new ConcreteTypeDecl(entity);
                case 89:
                    return entity.node.isNull() ?
                        FormalTypeDecl.NONE :
                        new FormalTypeDecl(entity);
                case 90:
                    return entity.node.isNull() ?
                        AbstractSubpDecl.NONE :
                        new AbstractSubpDecl(entity);
                case 91:
                    return entity.node.isNull() ?
                        AbstractFormalSubpDecl.NONE :
                        new AbstractFormalSubpDecl(entity);
                case 92:
                    return entity.node.isNull() ?
                        ConcreteFormalSubpDecl.NONE :
                        new ConcreteFormalSubpDecl(entity);
                case 93:
                    return entity.node.isNull() ?
                        SubpDecl.NONE :
                        new SubpDecl(entity);
                case 94:
                    return entity.node.isNull() ?
                        EntryDecl.NONE :
                        new EntryDecl(entity);
                case 95:
                    return entity.node.isNull() ?
                        EnumLiteralDecl.NONE :
                        new EnumLiteralDecl(entity);
                case 96:
                    return entity.node.isNull() ?
                        SyntheticCharEnumLit.NONE :
                        new SyntheticCharEnumLit(entity);
                case 97:
                    return entity.node.isNull() ?
                        GenericSubpInternal.NONE :
                        new GenericSubpInternal(entity);
                case 98:
                    return entity.node.isNull() ?
                        SyntheticSubpDecl.NONE :
                        new SyntheticSubpDecl(entity);
                case 99:
                    return entity.node.isNull() ?
                        AcceptStmtBody.NONE :
                        new AcceptStmtBody(entity);
                case 100:
                    return entity.node.isNull() ?
                        ExprFunction.NONE :
                        new ExprFunction(entity);
                case 101:
                    return entity.node.isNull() ?
                        NullSubpDecl.NONE :
                        new NullSubpDecl(entity);
                case 102:
                    return entity.node.isNull() ?
                        SubpBody.NONE :
                        new SubpBody(entity);
                case 103:
                    return entity.node.isNull() ?
                        SubpRenamingDecl.NONE :
                        new SubpRenamingDecl(entity);
                case 104:
                    return entity.node.isNull() ?
                        PackageBodyStub.NONE :
                        new PackageBodyStub(entity);
                case 105:
                    return entity.node.isNull() ?
                        ProtectedBodyStub.NONE :
                        new ProtectedBodyStub(entity);
                case 106:
                    return entity.node.isNull() ?
                        SubpBodyStub.NONE :
                        new SubpBodyStub(entity);
                case 107:
                    return entity.node.isNull() ?
                        TaskBodyStub.NONE :
                        new TaskBodyStub(entity);
                case 108:
                    return entity.node.isNull() ?
                        EntryBody.NONE :
                        new EntryBody(entity);
                case 109:
                    return entity.node.isNull() ?
                        PackageBody.NONE :
                        new PackageBody(entity);
                case 110:
                    return entity.node.isNull() ?
                        ProtectedBody.NONE :
                        new ProtectedBody(entity);
                case 111:
                    return entity.node.isNull() ?
                        TaskBody.NONE :
                        new TaskBody(entity);
                case 112:
                    return entity.node.isNull() ?
                        EntryIndexSpec.NONE :
                        new EntryIndexSpec(entity);
                case 113:
                    return entity.node.isNull() ?
                        ErrorDecl.NONE :
                        new ErrorDecl(entity);
                case 114:
                    return entity.node.isNull() ?
                        ExceptionDecl.NONE :
                        new ExceptionDecl(entity);
                case 115:
                    return entity.node.isNull() ?
                        ExceptionHandler.NONE :
                        new ExceptionHandler(entity);
                case 116:
                    return entity.node.isNull() ?
                        ForLoopVarDecl.NONE :
                        new ForLoopVarDecl(entity);
                case 117:
                    return entity.node.isNull() ?
                        GenericPackageDecl.NONE :
                        new GenericPackageDecl(entity);
                case 118:
                    return entity.node.isNull() ?
                        GenericSubpDecl.NONE :
                        new GenericSubpDecl(entity);
                case 119:
                    return entity.node.isNull() ?
                        GenericPackageInstantiation.NONE :
                        new GenericPackageInstantiation(entity);
                case 120:
                    return entity.node.isNull() ?
                        GenericSubpInstantiation.NONE :
                        new GenericSubpInstantiation(entity);
                case 121:
                    return entity.node.isNull() ?
                        GenericPackageRenamingDecl.NONE :
                        new GenericPackageRenamingDecl(entity);
                case 122:
                    return entity.node.isNull() ?
                        GenericSubpRenamingDecl.NONE :
                        new GenericSubpRenamingDecl(entity);
                case 123:
                    return entity.node.isNull() ?
                        LabelDecl.NONE :
                        new LabelDecl(entity);
                case 124:
                    return entity.node.isNull() ?
                        NamedStmtDecl.NONE :
                        new NamedStmtDecl(entity);
                case 125:
                    return entity.node.isNull() ?
                        NumberDecl.NONE :
                        new NumberDecl(entity);
                case 126:
                    return entity.node.isNull() ?
                        ObjectDecl.NONE :
                        new ObjectDecl(entity);
                case 127:
                    return entity.node.isNull() ?
                        ExtendedReturnStmtObjectDecl.NONE :
                        new ExtendedReturnStmtObjectDecl(entity);
                case 128:
                    return entity.node.isNull() ?
                        NoTypeObjectRenamingDecl.NONE :
                        new NoTypeObjectRenamingDecl(entity);
                case 129:
                    return entity.node.isNull() ?
                        PackageRenamingDecl.NONE :
                        new PackageRenamingDecl(entity);
                case 130:
                    return entity.node.isNull() ?
                        SingleProtectedDecl.NONE :
                        new SingleProtectedDecl(entity);
                case 131:
                    return entity.node.isNull() ?
                        SingleTaskDecl.NONE :
                        new SingleTaskDecl(entity);
                case 132:
                    return entity.node.isNull() ?
                        CaseStmtAlternative.NONE :
                        new CaseStmtAlternative(entity);
                case 133:
                    return entity.node.isNull() ?
                        CompilationUnit.NONE :
                        new CompilationUnit(entity);
                case 134:
                    return entity.node.isNull() ?
                        ComponentClause.NONE :
                        new ComponentClause(entity);
                case 135:
                    return entity.node.isNull() ?
                        ComponentDef.NONE :
                        new ComponentDef(entity);
                case 136:
                    return entity.node.isNull() ?
                        ConstantAbsent.NONE :
                        new ConstantAbsent(entity);
                case 137:
                    return entity.node.isNull() ?
                        ConstantPresent.NONE :
                        new ConstantPresent(entity);
                case 138:
                    return entity.node.isNull() ?
                        CompositeConstraint.NONE :
                        new CompositeConstraint(entity);
                case 139:
                    return entity.node.isNull() ?
                        DeltaConstraint.NONE :
                        new DeltaConstraint(entity);
                case 140:
                    return entity.node.isNull() ?
                        DigitsConstraint.NONE :
                        new DigitsConstraint(entity);
                case 141:
                    return entity.node.isNull() ?
                        RangeConstraint.NONE :
                        new RangeConstraint(entity);
                case 142:
                    return entity.node.isNull() ?
                        DeclarativePart.NONE :
                        new DeclarativePart(entity);
                case 143:
                    return entity.node.isNull() ?
                        PrivatePart.NONE :
                        new PrivatePart(entity);
                case 144:
                    return entity.node.isNull() ?
                        PublicPart.NONE :
                        new PublicPart(entity);
                case 145:
                    return entity.node.isNull() ?
                        ElsifExprPart.NONE :
                        new ElsifExprPart(entity);
                case 146:
                    return entity.node.isNull() ?
                        ElsifStmtPart.NONE :
                        new ElsifStmtPart(entity);
                case 147:
                    return entity.node.isNull() ?
                        AbstractStateDeclExpr.NONE :
                        new AbstractStateDeclExpr(entity);
                case 148:
                    return entity.node.isNull() ?
                        Allocator.NONE :
                        new Allocator(entity);
                case 149:
                    return entity.node.isNull() ?
                        Aggregate.NONE :
                        new Aggregate(entity);
                case 150:
                    return entity.node.isNull() ?
                        BracketAggregate.NONE :
                        new BracketAggregate(entity);
                case 151:
                    return entity.node.isNull() ?
                        DeltaAggregate.NONE :
                        new DeltaAggregate(entity);
                case 152:
                    return entity.node.isNull() ?
                        BracketDeltaAggregate.NONE :
                        new BracketDeltaAggregate(entity);
                case 153:
                    return entity.node.isNull() ?
                        NullRecordAggregate.NONE :
                        new NullRecordAggregate(entity);
                case 154:
                    return entity.node.isNull() ?
                        BinOp.NONE :
                        new BinOp(entity);
                case 155:
                    return entity.node.isNull() ?
                        RelationOp.NONE :
                        new RelationOp(entity);
                case 156:
                    return entity.node.isNull() ?
                        BoxExpr.NONE :
                        new BoxExpr(entity);
                case 157:
                    return entity.node.isNull() ?
                        CaseExprAlternative.NONE :
                        new CaseExprAlternative(entity);
                case 158:
                    return entity.node.isNull() ?
                        ConcatOp.NONE :
                        new ConcatOp(entity);
                case 159:
                    return entity.node.isNull() ?
                        ConcatOperand.NONE :
                        new ConcatOperand(entity);
                case 160:
                    return entity.node.isNull() ?
                        CaseExpr.NONE :
                        new CaseExpr(entity);
                case 161:
                    return entity.node.isNull() ?
                        IfExpr.NONE :
                        new IfExpr(entity);
                case 162:
                    return entity.node.isNull() ?
                        ContractCases.NONE :
                        new ContractCases(entity);
                case 163:
                    return entity.node.isNull() ?
                        DeclExpr.NONE :
                        new DeclExpr(entity);
                case 164:
                    return entity.node.isNull() ?
                        MembershipExpr.NONE :
                        new MembershipExpr(entity);
                case 165:
                    return entity.node.isNull() ?
                        AttributeRef.NONE :
                        new AttributeRef(entity);
                case 166:
                    return entity.node.isNull() ?
                        CallExpr.NONE :
                        new CallExpr(entity);
                case 167:
                    return entity.node.isNull() ?
                        DefiningName.NONE :
                        new DefiningName(entity);
                case 168:
                    return entity.node.isNull() ?
                        SyntheticDefiningName.NONE :
                        new SyntheticDefiningName(entity);
                case 169:
                    return entity.node.isNull() ?
                        DiscreteSubtypeName.NONE :
                        new DiscreteSubtypeName(entity);
                case 170:
                    return entity.node.isNull() ?
                        DottedName.NONE :
                        new DottedName(entity);
                case 171:
                    return entity.node.isNull() ?
                        EndName.NONE :
                        new EndName(entity);
                case 172:
                    return entity.node.isNull() ?
                        ExplicitDeref.NONE :
                        new ExplicitDeref(entity);
                case 173:
                    return entity.node.isNull() ?
                        QualExpr.NONE :
                        new QualExpr(entity);
                case 174:
                    return entity.node.isNull() ?
                        ReduceAttributeRef.NONE :
                        new ReduceAttributeRef(entity);
                case 175:
                    return entity.node.isNull() ?
                        CharLiteral.NONE :
                        new CharLiteral(entity);
                case 176:
                    return entity.node.isNull() ?
                        Identifier.NONE :
                        new Identifier(entity);
                case 177:
                    return entity.node.isNull() ?
                        OpAbs.NONE :
                        new OpAbs(entity);
                case 178:
                    return entity.node.isNull() ?
                        OpAnd.NONE :
                        new OpAnd(entity);
                case 179:
                    return entity.node.isNull() ?
                        OpAndThen.NONE :
                        new OpAndThen(entity);
                case 180:
                    return entity.node.isNull() ?
                        OpConcat.NONE :
                        new OpConcat(entity);
                case 181:
                    return entity.node.isNull() ?
                        OpDiv.NONE :
                        new OpDiv(entity);
                case 182:
                    return entity.node.isNull() ?
                        OpDoubleDot.NONE :
                        new OpDoubleDot(entity);
                case 183:
                    return entity.node.isNull() ?
                        OpEq.NONE :
                        new OpEq(entity);
                case 184:
                    return entity.node.isNull() ?
                        OpGt.NONE :
                        new OpGt(entity);
                case 185:
                    return entity.node.isNull() ?
                        OpGte.NONE :
                        new OpGte(entity);
                case 186:
                    return entity.node.isNull() ?
                        OpIn.NONE :
                        new OpIn(entity);
                case 187:
                    return entity.node.isNull() ?
                        OpLt.NONE :
                        new OpLt(entity);
                case 188:
                    return entity.node.isNull() ?
                        OpLte.NONE :
                        new OpLte(entity);
                case 189:
                    return entity.node.isNull() ?
                        OpMinus.NONE :
                        new OpMinus(entity);
                case 190:
                    return entity.node.isNull() ?
                        OpMod.NONE :
                        new OpMod(entity);
                case 191:
                    return entity.node.isNull() ?
                        OpMult.NONE :
                        new OpMult(entity);
                case 192:
                    return entity.node.isNull() ?
                        OpNeq.NONE :
                        new OpNeq(entity);
                case 193:
                    return entity.node.isNull() ?
                        OpNot.NONE :
                        new OpNot(entity);
                case 194:
                    return entity.node.isNull() ?
                        OpNotIn.NONE :
                        new OpNotIn(entity);
                case 195:
                    return entity.node.isNull() ?
                        OpOr.NONE :
                        new OpOr(entity);
                case 196:
                    return entity.node.isNull() ?
                        OpOrElse.NONE :
                        new OpOrElse(entity);
                case 197:
                    return entity.node.isNull() ?
                        OpPlus.NONE :
                        new OpPlus(entity);
                case 198:
                    return entity.node.isNull() ?
                        OpPow.NONE :
                        new OpPow(entity);
                case 199:
                    return entity.node.isNull() ?
                        OpRem.NONE :
                        new OpRem(entity);
                case 200:
                    return entity.node.isNull() ?
                        OpXor.NONE :
                        new OpXor(entity);
                case 201:
                    return entity.node.isNull() ?
                        StringLiteral.NONE :
                        new StringLiteral(entity);
                case 202:
                    return entity.node.isNull() ?
                        NullLiteral.NONE :
                        new NullLiteral(entity);
                case 203:
                    return entity.node.isNull() ?
                        IntLiteral.NONE :
                        new IntLiteral(entity);
                case 204:
                    return entity.node.isNull() ?
                        RealLiteral.NONE :
                        new RealLiteral(entity);
                case 205:
                    return entity.node.isNull() ?
                        SyntheticIdentifier.NONE :
                        new SyntheticIdentifier(entity);
                case 206:
                    return entity.node.isNull() ?
                        TargetName.NONE :
                        new TargetName(entity);
                case 207:
                    return entity.node.isNull() ?
                        UpdateAttributeRef.NONE :
                        new UpdateAttributeRef(entity);
                case 208:
                    return entity.node.isNull() ?
                        ParenExpr.NONE :
                        new ParenExpr(entity);
                case 209:
                    return entity.node.isNull() ?
                        QuantifiedExpr.NONE :
                        new QuantifiedExpr(entity);
                case 210:
                    return entity.node.isNull() ?
                        RaiseExpr.NONE :
                        new RaiseExpr(entity);
                case 211:
                    return entity.node.isNull() ?
                        UnOp.NONE :
                        new UnOp(entity);
                case 212:
                    return entity.node.isNull() ?
                        HandledStmts.NONE :
                        new HandledStmts(entity);
                case 213:
                    return entity.node.isNull() ?
                        InterfaceKindLimited.NONE :
                        new InterfaceKindLimited(entity);
                case 214:
                    return entity.node.isNull() ?
                        InterfaceKindProtected.NONE :
                        new InterfaceKindProtected(entity);
                case 215:
                    return entity.node.isNull() ?
                        InterfaceKindSynchronized.NONE :
                        new InterfaceKindSynchronized(entity);
                case 216:
                    return entity.node.isNull() ?
                        InterfaceKindTask.NONE :
                        new InterfaceKindTask(entity);
                case 217:
                    return entity.node.isNull() ?
                        IterTypeIn.NONE :
                        new IterTypeIn(entity);
                case 218:
                    return entity.node.isNull() ?
                        IterTypeOf.NONE :
                        new IterTypeOf(entity);
                case 219:
                    return entity.node.isNull() ?
                        LibraryItem.NONE :
                        new LibraryItem(entity);
                case 220:
                    return entity.node.isNull() ?
                        LimitedAbsent.NONE :
                        new LimitedAbsent(entity);
                case 221:
                    return entity.node.isNull() ?
                        LimitedPresent.NONE :
                        new LimitedPresent(entity);
                case 222:
                    return entity.node.isNull() ?
                        ForLoopSpec.NONE :
                        new ForLoopSpec(entity);
                case 223:
                    return entity.node.isNull() ?
                        WhileLoopSpec.NONE :
                        new WhileLoopSpec(entity);
                case 224:
                    return entity.node.isNull() ?
                        ModeDefault.NONE :
                        new ModeDefault(entity);
                case 225:
                    return entity.node.isNull() ?
                        ModeIn.NONE :
                        new ModeIn(entity);
                case 226:
                    return entity.node.isNull() ?
                        ModeInOut.NONE :
                        new ModeInOut(entity);
                case 227:
                    return entity.node.isNull() ?
                        ModeOut.NONE :
                        new ModeOut(entity);
                case 228:
                    return entity.node.isNull() ?
                        MultiAbstractStateDecl.NONE :
                        new MultiAbstractStateDecl(entity);
                case 229:
                    return entity.node.isNull() ?
                        NotNullAbsent.NONE :
                        new NotNullAbsent(entity);
                case 230:
                    return entity.node.isNull() ?
                        NotNullPresent.NONE :
                        new NotNullPresent(entity);
                case 231:
                    return entity.node.isNull() ?
                        NullComponentDecl.NONE :
                        new NullComponentDecl(entity);
                case 232:
                    return entity.node.isNull() ?
                        OthersDesignator.NONE :
                        new OthersDesignator(entity);
                case 233:
                    return entity.node.isNull() ?
                        OverridingNotOverriding.NONE :
                        new OverridingNotOverriding(entity);
                case 234:
                    return entity.node.isNull() ?
                        OverridingOverriding.NONE :
                        new OverridingOverriding(entity);
                case 235:
                    return entity.node.isNull() ?
                        OverridingUnspecified.NONE :
                        new OverridingUnspecified(entity);
                case 236:
                    return entity.node.isNull() ?
                        Params.NONE :
                        new Params(entity);
                case 237:
                    return entity.node.isNull() ?
                        ParenAbstractStateDecl.NONE :
                        new ParenAbstractStateDecl(entity);
                case 238:
                    return entity.node.isNull() ?
                        PpElseDirective.NONE :
                        new PpElseDirective(entity);
                case 239:
                    return entity.node.isNull() ?
                        PpElsifDirective.NONE :
                        new PpElsifDirective(entity);
                case 240:
                    return entity.node.isNull() ?
                        PpEndIfDirective.NONE :
                        new PpEndIfDirective(entity);
                case 241:
                    return entity.node.isNull() ?
                        PpIfDirective.NONE :
                        new PpIfDirective(entity);
                case 242:
                    return entity.node.isNull() ?
                        PpThenKw.NONE :
                        new PpThenKw(entity);
                case 243:
                    return entity.node.isNull() ?
                        PragmaNode.NONE :
                        new PragmaNode(entity);
                case 244:
                    return entity.node.isNull() ?
                        PrivateAbsent.NONE :
                        new PrivateAbsent(entity);
                case 245:
                    return entity.node.isNull() ?
                        PrivatePresent.NONE :
                        new PrivatePresent(entity);
                case 246:
                    return entity.node.isNull() ?
                        ProtectedDef.NONE :
                        new ProtectedDef(entity);
                case 247:
                    return entity.node.isNull() ?
                        ProtectedAbsent.NONE :
                        new ProtectedAbsent(entity);
                case 248:
                    return entity.node.isNull() ?
                        ProtectedPresent.NONE :
                        new ProtectedPresent(entity);
                case 249:
                    return entity.node.isNull() ?
                        QuantifierAll.NONE :
                        new QuantifierAll(entity);
                case 250:
                    return entity.node.isNull() ?
                        QuantifierSome.NONE :
                        new QuantifierSome(entity);
                case 251:
                    return entity.node.isNull() ?
                        RangeSpec.NONE :
                        new RangeSpec(entity);
                case 252:
                    return entity.node.isNull() ?
                        RenamingClause.NONE :
                        new RenamingClause(entity);
                case 253:
                    return entity.node.isNull() ?
                        SyntheticRenamingClause.NONE :
                        new SyntheticRenamingClause(entity);
                case 254:
                    return entity.node.isNull() ?
                        ReverseAbsent.NONE :
                        new ReverseAbsent(entity);
                case 255:
                    return entity.node.isNull() ?
                        ReversePresent.NONE :
                        new ReversePresent(entity);
                case 256:
                    return entity.node.isNull() ?
                        SelectWhenPart.NONE :
                        new SelectWhenPart(entity);
                case 257:
                    return entity.node.isNull() ?
                        AcceptStmt.NONE :
                        new AcceptStmt(entity);
                case 258:
                    return entity.node.isNull() ?
                        AcceptStmtWithStmts.NONE :
                        new AcceptStmtWithStmts(entity);
                case 259:
                    return entity.node.isNull() ?
                        ForLoopStmt.NONE :
                        new ForLoopStmt(entity);
                case 260:
                    return entity.node.isNull() ?
                        LoopStmt.NONE :
                        new LoopStmt(entity);
                case 261:
                    return entity.node.isNull() ?
                        WhileLoopStmt.NONE :
                        new WhileLoopStmt(entity);
                case 262:
                    return entity.node.isNull() ?
                        BeginBlock.NONE :
                        new BeginBlock(entity);
                case 263:
                    return entity.node.isNull() ?
                        DeclBlock.NONE :
                        new DeclBlock(entity);
                case 264:
                    return entity.node.isNull() ?
                        CaseStmt.NONE :
                        new CaseStmt(entity);
                case 265:
                    return entity.node.isNull() ?
                        ExtendedReturnStmt.NONE :
                        new ExtendedReturnStmt(entity);
                case 266:
                    return entity.node.isNull() ?
                        IfStmt.NONE :
                        new IfStmt(entity);
                case 267:
                    return entity.node.isNull() ?
                        NamedStmt.NONE :
                        new NamedStmt(entity);
                case 268:
                    return entity.node.isNull() ?
                        SelectStmt.NONE :
                        new SelectStmt(entity);
                case 269:
                    return entity.node.isNull() ?
                        ErrorStmt.NONE :
                        new ErrorStmt(entity);
                case 270:
                    return entity.node.isNull() ?
                        AbortStmt.NONE :
                        new AbortStmt(entity);
                case 271:
                    return entity.node.isNull() ?
                        AssignStmt.NONE :
                        new AssignStmt(entity);
                case 272:
                    return entity.node.isNull() ?
                        CallStmt.NONE :
                        new CallStmt(entity);
                case 273:
                    return entity.node.isNull() ?
                        DelayStmt.NONE :
                        new DelayStmt(entity);
                case 274:
                    return entity.node.isNull() ?
                        ExitStmt.NONE :
                        new ExitStmt(entity);
                case 275:
                    return entity.node.isNull() ?
                        GotoStmt.NONE :
                        new GotoStmt(entity);
                case 276:
                    return entity.node.isNull() ?
                        Label.NONE :
                        new Label(entity);
                case 277:
                    return entity.node.isNull() ?
                        NullStmt.NONE :
                        new NullStmt(entity);
                case 278:
                    return entity.node.isNull() ?
                        RaiseStmt.NONE :
                        new RaiseStmt(entity);
                case 279:
                    return entity.node.isNull() ?
                        RequeueStmt.NONE :
                        new RequeueStmt(entity);
                case 280:
                    return entity.node.isNull() ?
                        ReturnStmt.NONE :
                        new ReturnStmt(entity);
                case 281:
                    return entity.node.isNull() ?
                        TerminateAlternative.NONE :
                        new TerminateAlternative(entity);
                case 282:
                    return entity.node.isNull() ?
                        SubpKindFunction.NONE :
                        new SubpKindFunction(entity);
                case 283:
                    return entity.node.isNull() ?
                        SubpKindProcedure.NONE :
                        new SubpKindProcedure(entity);
                case 284:
                    return entity.node.isNull() ?
                        Subunit.NONE :
                        new Subunit(entity);
                case 285:
                    return entity.node.isNull() ?
                        SynchronizedAbsent.NONE :
                        new SynchronizedAbsent(entity);
                case 286:
                    return entity.node.isNull() ?
                        SynchronizedPresent.NONE :
                        new SynchronizedPresent(entity);
                case 287:
                    return entity.node.isNull() ?
                        TaggedAbsent.NONE :
                        new TaggedAbsent(entity);
                case 288:
                    return entity.node.isNull() ?
                        TaggedPresent.NONE :
                        new TaggedPresent(entity);
                case 289:
                    return entity.node.isNull() ?
                        TaskDef.NONE :
                        new TaskDef(entity);
                case 290:
                    return entity.node.isNull() ?
                        TypeAttributesRepository.NONE :
                        new TypeAttributesRepository(entity);
                case 291:
                    return entity.node.isNull() ?
                        AccessToSubpDef.NONE :
                        new AccessToSubpDef(entity);
                case 292:
                    return entity.node.isNull() ?
                        AnonymousTypeAccessDef.NONE :
                        new AnonymousTypeAccessDef(entity);
                case 293:
                    return entity.node.isNull() ?
                        TypeAccessDef.NONE :
                        new TypeAccessDef(entity);
                case 294:
                    return entity.node.isNull() ?
                        ArrayTypeDef.NONE :
                        new ArrayTypeDef(entity);
                case 295:
                    return entity.node.isNull() ?
                        DerivedTypeDef.NONE :
                        new DerivedTypeDef(entity);
                case 296:
                    return entity.node.isNull() ?
                        EnumTypeDef.NONE :
                        new EnumTypeDef(entity);
                case 297:
                    return entity.node.isNull() ?
                        FormalDiscreteTypeDef.NONE :
                        new FormalDiscreteTypeDef(entity);
                case 298:
                    return entity.node.isNull() ?
                        InterfaceTypeDef.NONE :
                        new InterfaceTypeDef(entity);
                case 299:
                    return entity.node.isNull() ?
                        ModIntTypeDef.NONE :
                        new ModIntTypeDef(entity);
                case 300:
                    return entity.node.isNull() ?
                        PrivateTypeDef.NONE :
                        new PrivateTypeDef(entity);
                case 301:
                    return entity.node.isNull() ?
                        DecimalFixedPointDef.NONE :
                        new DecimalFixedPointDef(entity);
                case 302:
                    return entity.node.isNull() ?
                        FloatingPointDef.NONE :
                        new FloatingPointDef(entity);
                case 303:
                    return entity.node.isNull() ?
                        OrdinaryFixedPointDef.NONE :
                        new OrdinaryFixedPointDef(entity);
                case 304:
                    return entity.node.isNull() ?
                        RecordTypeDef.NONE :
                        new RecordTypeDef(entity);
                case 305:
                    return entity.node.isNull() ?
                        SignedIntTypeDef.NONE :
                        new SignedIntTypeDef(entity);
                case 306:
                    return entity.node.isNull() ?
                        AnonymousType.NONE :
                        new AnonymousType(entity);
                case 307:
                    return entity.node.isNull() ?
                        EnumLitSynthTypeExpr.NONE :
                        new EnumLitSynthTypeExpr(entity);
                case 308:
                    return entity.node.isNull() ?
                        SubtypeIndication.NONE :
                        new SubtypeIndication(entity);
                case 309:
                    return entity.node.isNull() ?
                        ConstrainedSubtypeIndication.NONE :
                        new ConstrainedSubtypeIndication(entity);
                case 310:
                    return entity.node.isNull() ?
                        DiscreteSubtypeIndication.NONE :
                        new DiscreteSubtypeIndication(entity);
                case 311:
                    return entity.node.isNull() ?
                        SyntheticTypeExpr.NONE :
                        new SyntheticTypeExpr(entity);
                case 312:
                    return entity.node.isNull() ?
                        UnconstrainedArrayIndex.NONE :
                        new UnconstrainedArrayIndex(entity);
                case 313:
                    return entity.node.isNull() ?
                        UntilAbsent.NONE :
                        new UntilAbsent(entity);
                case 314:
                    return entity.node.isNull() ?
                        UntilPresent.NONE :
                        new UntilPresent(entity);
                case 315:
                    return entity.node.isNull() ?
                        UsePackageClause.NONE :
                        new UsePackageClause(entity);
                case 316:
                    return entity.node.isNull() ?
                        UseTypeClause.NONE :
                        new UseTypeClause(entity);
                case 317:
                    return entity.node.isNull() ?
                        ValueSequence.NONE :
                        new ValueSequence(entity);
                case 318:
                    return entity.node.isNull() ?
                        Variant.NONE :
                        new Variant(entity);
                case 319:
                    return entity.node.isNull() ?
                        VariantPart.NONE :
                        new VariantPart(entity);
                case 320:
                    return entity.node.isNull() ?
                        WithClause.NONE :
                        new WithClause(entity);
                case 321:
                    return entity.node.isNull() ?
                        WithPrivateAbsent.NONE :
                        new WithPrivateAbsent(entity);
                case 322:
                    return entity.node.isNull() ?
                        WithPrivatePresent.NONE :
                        new WithPrivatePresent(entity);
                default:
                    throw new EnumException(
                        "Cannot find the node type from " + nodeKind
                    );
            }
        }

        // ----- Getters -----

        public String getKindName() {
            return AdaNode.kindName;
        }

        public String[] getFieldNames() {
            return AdaNode.fieldNames;
        }

        public boolean isListType() {
            return AdaNode.isListType;
        }

        public boolean isTokenNode() {

            if(ImageInfo.inImageCode()) {
                EntityNative entityNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(entityNative);
                return NI_LIB.ada_node_is_token_node(
                    entityNative
                ) != 0;
            } else {
                return JNI_LIB.ada_node_is_token_node(this.entity);
            }

        }

        public boolean isNone() {
            return this.entity.node.isNull();
        }

        // ----- Instance methods -----

        /**
         * Get the analysis unit of the node.
         *
         * @return The unit which owns the node.
         */
        public AnalysisUnit getUnit() {
            if(this.unit == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(entityNative);

                    final AnalysisUnitNative unitNative =
                        NI_LIB.ada_node_unit(entityNative);
                    this.unit = AnalysisUnit.wrap(unitNative);
                } else {
                    this.unit = JNI_LIB.ada_node_unit(this.entity);
                }

            }
            return this.unit;
        }

        /**
         * Get the descritpion of a field from its name.
         *
         * @param name The langkit field name to get the description for.
         * @return The Java description of the langkit field.
         */
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(String name) {
            return AdaNode.fieldDescriptions.get(name);
        }

        /**
         * Get the children count of the node.
         *
         * @return The children count.
         */
        public int getChildrenCount() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.ada_node_children_count(thisNative);
            } else {
                return JNI_LIB.ada_node_children_count(this.entity);
            }

        }

        /**
         * Get the child at the given position.
         *
         * @param n The index of the child to get.
         * @return The child at the given index.
         */
        public AdaNode getChild(
            final int n
        ) {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative resNative = StackValue.get(
                    EntityNative.class
                );
                NI_LIB.ada_node_child(
                    thisNative,
                    n,
                    resNative
                );

                return fromEntity(Entity.wrap(resNative));
            } else {
                return fromEntity(JNI_LIB.ada_node_child(
                    this.entity,
                    n
                ));
            }

        }

        /**
         * Get the text of the node.
         *
         * @return The text of the node.
         */
        public String getText() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final TextNative resNative = StackValue.get(TextNative.class);
                Text.NONE.unwrap(resNative);
                NI_LIB.ada_node_text(
                    thisNative,
                    resNative
                );

                try(final Text resText = Text.wrap(resNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.ada_node_text(
                        this.entity
                    );
                ) {
                    return resText.getContent();
                }
            }

        }

        /**
         * Get the image of the node.
         *
         * @return The node's image.
         */
        public String getImage() {
            if(this.image == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative thisNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(thisNative);

                    final TextNative resNative = StackValue.get(
                        TextNative.class
                    );
                    Text.NONE.unwrap(resNative);
                    NI_LIB.ada_node_image(thisNative, resNative);

                    try(final Text resText = Text.wrap(resNative)) {
                        this.image = resText.getContent();
                    }
                } else {
                    try(
                        final Text resText = JNI_LIB.ada_node_image(
                            this.entity
                        )
                    ) {
                        this.image = resText.getContent();
                    }
                }

            }

            return this.image;
        }

        /**
         * Get the source location range of the node.
         *
         * @return The source location range of the node.
         */
        public SourceLocationRange getSourceLocationRange() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final SourceLocationRangeNative resNative = StackValue.get(
                    SourceLocationRangeNative.class
                );
                NI_LIB.ada_node_sloc_range(
                    thisNative,
                    resNative
                );

                return SourceLocationRange.wrap(resNative);
            } else {
                return JNI_LIB.ada_node_sloc_range(this.entity);
            }

        }

        // ----- Dumping methods -----

        /**
         * Return the AST in a string.
         *
         * @return The string containing the representation of the AST
         * from the node.
         */
        @CompilerDirectives.TruffleBoundary
        public String dumpAST() {
            final StringBuilder builder = new StringBuilder();
            this.dumpAST(builder);
            return builder.toString();
        }

        /**
         * Dump the AST in the given string builder.
         *
         * @param builder The builder to dump the AST in.
         */
        @CompilerDirectives.TruffleBoundary
        public void dumpAST(
            final StringBuilder builder
        ) {
            this.dumpAST(builder, "");
        }

        /**
         * Dump a node field in the given string builder.
         *
         * @param builder The string builder to put the file in.
         * @param indent The current indentation string.
         * @param name The name of the field.
         * @param value The value of the field.
         */
        protected static void dumpField(
            final StringBuilder builder,
            final String indent,
            final String name,
            final AdaNode value
        ) {
            builder.append(indent)
                .append(name)
                .append(":\n");
            value.dumpAST(builder, indent + "  ");
        }

        /**
         * Dump the AST in the given string builder with the indent level.
         *
         * @param builder The builder to dump the AST in.
         * @param indent The starting indent level.
         */
        @CompilerDirectives.TruffleBoundary
        protected void dumpAST(
            final StringBuilder builder,
            String indent
        ) {
            // Prepare the working variables
            String image = this.getImage();
            image = image.substring(1, image.length());
            final int childrenCount = this.getChildrenCount();

            // Print the node
            builder.append(indent)
                .append(image);
            if(this.isTokenNode()) {
                builder.append(": ")
                    .append(this.getText());
            }
            builder.append('\n');

            // Print the field of the node
            indent = indent + "|";
            if(this.isListType()) {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final AdaNode child = this.getChild(i);
                    dumpField(builder, indent, "item_" + i, child);
                }
            } else {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final AdaNode child = this.getChild(i);
                    final String name = this.getFieldNames()[i];
                    dumpField(builder, indent, name, child);
                }
            }
        }

        // ----- Visitor methods -----

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @return The result of the visit.
         */
        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @param param The parameter of the visit.
         * @return The result of the visit.
         */
        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * Return the scope of definition of this basic declaration.
         */
        public DeclarativePart pDeclarativeScope(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_declarative_scope(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DeclarativePart res = DeclarativePart.NONE;
                if(propException == null) {
                    res = DeclarativePart.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_declarative_scope(
                    this.entity
                );

                // Wrap and return the result
                return DeclarativePart.fromEntity(res);
            }

        }

        
    

        /**
         * Return the compilation unit containing this node.
         *
         * .. note:: This returns the ``CompilationUnit`` node, which is
         *    different from the ``AnalysisUnit``. In particular, an analysis
         *    unit can contain multiple compilation units.
         */
        public CompilationUnit pEnclosingCompilationUnit(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_enclosing_compilation_unit(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                CompilationUnit res = CompilationUnit.NONE;
                if(propException == null) {
                    res = CompilationUnit.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_enclosing_compilation_unit(
                    this.entity
                );

                // Wrap and return the result
                return CompilationUnit.fromEntity(res);
            }

        }

        
    

        /**
         * Assuming this node comes from an instantiated generic declaration,
         * return its non-instantiated counterpart lying in the generic
         * declaration.
         */
        public AdaNode pGetUninstantiatedNode(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_get_uninstantiated_node(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_get_uninstantiated_node(
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return possible completions at this point in the file.
         */
        public CompletionItemIterator pComplete(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_complete(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                CompletionItemIterator res = CompletionItemIterator.NONE;
                if(propException == null) {
                    res = CompletionItemIterator.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final CompletionItemIterator res = JNI_LIB.ada_ada_node_p_complete(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the list of keywords that are valid at this point in the
         * file.
         *
         * .. note:: This is work in progress. It will return all keywords for
         *    now, without looking at the context.
         */
        public UnboundedTextTypeArray pValidKeywords(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_valid_keywords(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                UnboundedTextTypeArray res = UnboundedTextTypeArray.NONE;
                if(propException == null) {
                    res = UnboundedTextTypeArray.wrap(resNative);

                    UnboundedTextTypeArray.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final UnboundedTextTypeArray res = JNI_LIB.ada_ada_node_p_valid_keywords(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the potentially empty list of generic package/subprogram
         * instantiations that led to the creation of this entity. Outer-most
         * instantiations appear last.
         */
        public AdaNodeArray pGenericInstantiations(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_generic_instantiations(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNodeArray res = AdaNodeArray.NONE;
                if(propException == null) {
                    res = AdaNodeArray.wrap(resNative);

                    AdaNodeArray.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final AdaNodeArray res = JNI_LIB.ada_ada_node_p_generic_instantiations(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the semantic parent for this node, if applicable, null
         * otherwise.
         *
         * .. note:: A node lying outside of a library item's declaration or
         *    subunit's body does not have a parent environment, meaning that
         *    this property will return null.
         */
        public AdaNode pSemanticParent(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_semantic_parent(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_semantic_parent(
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return the parent basic decl for this node, if applicable, null
         * otherwise.
         *
         * .. note:: If the parent BasicDecl of the given node is a generic
         *    declaration, this call will return the instantiation from which
         *    the node was retrieved instead, if any.
         *
         * .. note:: When called on a subunit's body, this property will return
         *    its corresponding body stub.
         *
         * .. note:: When called on a node lying outside of a library item's
         *    declaration or subunit's body this property will return null.
         */
        public BasicDecl pParentBasicDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_parent_basic_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BasicDecl res = BasicDecl.NONE;
                if(propException == null) {
                    res = BasicDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_parent_basic_decl(
                    this.entity
                );

                // Wrap and return the result
                return BasicDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Filters out among the list of given units those that cannot refer to
         * the unit in which this node lies. If transitive is True, the whole
         * transitive closure of imports will be used to find a reference to
         * the unit of this node.
         */
        public AnalysisUnitArray pFilterIsImportedBy(
            final AnalysisUnitArray units,final boolean transitive
        ) {

            // Verify that arguments are not null
            if(units == null) throw new IllegalArgumentException(
                "Argument 'units' of type " +
                "AnalysisUnitArray cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                AnalysisUnitArrayNative unitsNative = units.unwrap();
                byte transitiveNative = (transitive ? (byte) 1 : (byte) 0);

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_filter_is_imported_by(
                    thisNative,
                    unitsNative,
                    transitiveNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AnalysisUnitArray res = AnalysisUnitArray.NONE;
                if(propException == null) {
                    res = AnalysisUnitArray.wrap(resNative);

                    AnalysisUnitArray.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters
                AnalysisUnitArray.release(
                    unitsNative
                );


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final AnalysisUnitArray res = JNI_LIB.ada_ada_node_p_filter_is_imported_by(
                    units,
                    transitive,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Designates entities that are entry point for the xref solving
         * infrastructure. If this returns true, then resolve_names can be
         * called on it.
         *
         * .. note:: For convenience, and unlike what is defined in the ARM
         *    wrt. complete contexts for name resolution, ``xref_entry_points``
         *    can be nested.
         */
        public boolean pXrefEntryPoint(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_xref_entry_point(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.ada_ada_node_p_xref_entry_point(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * This will resolve names for this node. If the operation is
         * successful, then type_var and ref_var will be bound on appropriate
         * subnodes of the statement.
         */
        public boolean pResolveNames(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_resolve_names(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.ada_ada_node_p_resolve_names(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Static method. Return the analysis unit corresponding to the
         * Standard package.
         */
        public AnalysisUnit pStandardUnit(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_standard_unit(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AnalysisUnit res = AnalysisUnit.NONE;
                if(propException == null) {
                    res = AnalysisUnit.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final AnalysisUnit res = JNI_LIB.ada_ada_node_p_standard_unit(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Static property. Return an entity from the standard package with
         * name ``sym``.
         */
        public AdaNode pStdEntity(
            final Symbol sym
        ) {

            // Verify that arguments are not null
            if(sym == null) throw new IllegalArgumentException(
                "Argument 'sym' of type " +
                "Symbol cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();

                // Get the node context
                final AnalysisContext currentContext =
                    currentUnit.getContext();

                // Unwrap the arguments
                SymbolNative symNative = StackValue.get(SymbolNative.class);sym.unwrap(symNative, currentContext);

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_std_entity(
                    thisNative,
                    symNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters

                // Close the context
                currentContext.close();

                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_std_entity(
                    sym,
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Return the standard Boolean type.
         */
        public BaseTypeDecl pBoolType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_bool_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BaseTypeDecl res = BaseTypeDecl.NONE;
                if(propException == null) {
                    res = BaseTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_bool_type(
                    this.entity
                );

                // Wrap and return the result
                return BaseTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Return the standard Integer type.
         */
        public BaseTypeDecl pIntType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_int_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BaseTypeDecl res = BaseTypeDecl.NONE;
                if(propException == null) {
                    res = BaseTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_int_type(
                    this.entity
                );

                // Wrap and return the result
                return BaseTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Return the standard Universal Integer type.
         */
        public AdaNode pUniversalIntType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_universal_int_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_universal_int_type(
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Return the standard Universal Real type.
         */
        public AdaNode pUniversalRealType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_universal_real_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_universal_real_type(
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Return the standard Character type.
         */
        public BaseTypeDecl pStdCharType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_std_char_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BaseTypeDecl res = BaseTypeDecl.NONE;
                if(propException == null) {
                    res = BaseTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_std_char_type(
                    this.entity
                );

                // Wrap and return the result
                return BaseTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Return the standard Wide_Character type.
         */
        public BaseTypeDecl pStdWideCharType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_std_wide_char_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BaseTypeDecl res = BaseTypeDecl.NONE;
                if(propException == null) {
                    res = BaseTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_std_wide_char_type(
                    this.entity
                );

                // Wrap and return the result
                return BaseTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Return the standard Wide_Wide_Character type.
         */
        public BaseTypeDecl pStdWideWideCharType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_std_wide_wide_char_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BaseTypeDecl res = BaseTypeDecl.NONE;
                if(propException == null) {
                    res = BaseTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_std_wide_wide_char_type(
                    this.entity
                );

                // Wrap and return the result
                return BaseTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Static method. Get the top-level decl in ``unit``.  This is the body
         * of a Subunit, or the item of a ``LibraryItem``.
         */
        public BasicDecl pTopLevelDecl(
            final AnalysisUnit unit
        ) {

            // Verify that arguments are not null
            if(unit == null) throw new IllegalArgumentException(
                "Argument 'unit' of type " +
                "AnalysisUnit cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                AnalysisUnitNative unitNative = unit.unwrap();

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_top_level_decl(
                    thisNative,
                    unitNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BasicDecl res = BasicDecl.NONE;
                if(propException == null) {
                    res = BasicDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_top_level_decl(
                    unit,
                    this.entity
                );

                // Wrap and return the result
                return BasicDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Assuming that self is a choice expression (such as what can appear
         * in an alternative of a case statement or in the RHS of a membership
         * expression, this property returns whether the given value satisfies
         * it.
         *
         * .. attention:: This is an experimental feature, so even if it is
         *    exposed to allow experiments, it is totally unsupported and the
         *    API and behavior are very likely to change in the future.
         */
        public boolean pChoiceMatch(
            final BigInteger value
        ) {

            // Verify that arguments are not null
            if(value == null) throw new IllegalArgumentException(
                "Argument 'value' of type " +
                "BigInteger cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                BigIntegerNative valueNative = StackValue.get(SizeOf.get(VoidPointer.class));BigIntegerWrapper.unwrap(value, valueNative);

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_p_choice_match(
                    thisNative,
                    valueNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters
                BigIntegerWrapper.release(
                    valueNative
                );


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.ada_ada_node_p_choice_match(
                    value,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return a cross reference from this name to a defining identifier,
         * trying to mimic GNAT's xrefs as much as possible.
         */
        public DefiningName pGnatXref(
            final boolean impreciseFallback
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte impreciseFallbackNative = (impreciseFallback ? (byte) 1 : (byte) 0);

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_p_gnat_xref(
                    thisNative,
                    impreciseFallbackNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DefiningName res = DefiningName.NONE;
                if(propException == null) {
                    res = DefiningName.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_p_gnat_xref(
                    impreciseFallback,
                    this.entity
                );

                // Wrap and return the result
                return DefiningName.fromEntity(res);
            }

        }

        
    

        /**
         * Return the syntactic parent for this node. Return null for the root
         * node.
         */
        public AdaNode parent(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_parent(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_parent(
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return an array that contains the lexical parents, this node
         * included iff ``with_self`` is True. Nearer parents are first in the
         * list.
         */
        public AdaNodeArray parents(
            final boolean withSelf
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte withSelfNative = (withSelf ? (byte) 1 : (byte) 0);

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_parents(
                    thisNative,
                    withSelfNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNodeArray res = AdaNodeArray.NONE;
                if(propException == null) {
                    res = AdaNodeArray.wrap(resNative);

                    AdaNodeArray.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final AdaNodeArray res = JNI_LIB.ada_ada_node_parents(
                    withSelf,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return an array that contains the direct lexical children.
         *
         * .. warning:: This constructs a whole array every-time you call it,
         *    and as such is less efficient than calling the ``Child`` built-
         *    in.
         */
        public AdaNodeArray children(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_children(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNodeArray res = AdaNodeArray.NONE;
                if(propException == null) {
                    res = AdaNodeArray.wrap(resNative);

                    AdaNodeArray.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final AdaNodeArray res = JNI_LIB.ada_ada_node_children(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the first token used to parse this node.
         */
        public Token tokenStart(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_token_start(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.ada_ada_node_token_start(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the last token used to parse this node.
         */
        public Token tokenEnd(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_token_end(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.ada_ada_node_token_end(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the 0-based index for Node in its parent's children.
         */
        public int childIndex(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CIntPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_child_index(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                int res = 0;
                if(propException == null) {
                    res = resNative.read();

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final int res = JNI_LIB.ada_ada_node_child_index(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the node's previous sibling, or null if there is no such
         * sibling.
         */
        public AdaNode previousSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_previous_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_previous_sibling(
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return the node's next sibling, or null if there is no such sibling.
         */
        public AdaNode nextSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.ada_ada_node_next_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AdaNode res = AdaNode.NONE;
                if(propException == null) {
                    res = AdaNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.ada_ada_node_next_sibling(
                    this.entity
                );

                // Wrap and return the result
                return AdaNode.fromEntity(res);
            }

        }

        
    


        
    

        /**
         * Return whether the node is a ghost.
         *
         * Unlike regular nodes, ghost nodes cover no token in the input
         * source: they are logically located instead between two tokens. Both
         * the ``token_start`` and the ``token_end`` of all ghost nodes is the
         * token right after this logical position.
         */
        public boolean isGhost(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_is_ghost(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.ada_ada_node_is_ghost(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return a string containing the filename + the sloc in GNU conformant
         * format. Useful to create diagnostics from a node.
         */
        public String fullSlocImage(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_ada_node_full_sloc_image(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.ada_ada_node_full_sloc_image(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getImage();
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof AdaNode)) return false;
            final AdaNode other = (AdaNode) o;

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative otherNative = StackValue.get(
                    EntityNative.class
                );
                other.entity.unwrap(otherNative);

                return NI_LIB.ada_node_is_equivalent(
                    thisNative,
                    otherNative
                ) != 0;
            } else {
                return JNI_LIB.ada_node_is_equivalent(
                    this.entity, other.entity
                ) != 0;
            }
        }

        @Override
        public int hashCode() {
            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.ada_node_hash(thisNative);
            } else {
                return JNI_LIB.ada_node_hash(this.entity);
            }

        }

        // ----- Inner classes -----

        /**
         * This class represents the none node without any concrete type.
         */
        private static final class NoneNode extends AdaNode {
            NoneNode() {super(Entity.NONE);}
        }

    }

    // ===== Generated AST node wrapping classes =====

    
    

    /**
     * Qualifier for the ``abort`` keyword.
     */
    public static abstract
    class AbortNode extends AdaNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AbortNode";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AbortNode.class.getMethod(
                        "pAsBool",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_as_bool",
                        new LibadalangField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AbortNode NONE =
            new AbortNodeNone();

        // ----- Constructors -----

        protected AbortNode(
            final Entity entity
        ) {
            super(entity);
        }

        public static AbortNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AbortNode.NONE :

                (AbortNode) AdaNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AbortNode.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AbortNode.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AbortNode.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AbortNode.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----

        
    

        /**
         * Return whether this is an instance of AbortPresent
         */
        public boolean pAsBool(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_abort_node_p_as_bool(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.ada_abort_node_p_as_bool(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * AbortNode.
         */
        private static final class AbortNodeNone extends AbortNode {
            AbortNodeNone() {super(Entity.NONE);}
        }

    }


    
    

    /**

     */
    public static 
    class AbortAbsent extends AbortNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AbortAbsent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AbortNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AbortAbsent NONE =
            new AbortAbsent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AbortAbsent(
            final Entity entity
        ) {
            super(entity);
        }

        public static AbortAbsent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AbortAbsent.NONE :

                new AbortAbsent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AbortAbsent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AbortAbsent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AbortAbsent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AbortAbsent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class AbortPresent extends AbortNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AbortPresent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AbortNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AbortPresent NONE =
            new AbortPresent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AbortPresent(
            final Entity entity
        ) {
            super(entity);
        }

        public static AbortPresent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AbortPresent.NONE :

                new AbortPresent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AbortPresent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AbortPresent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AbortPresent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AbortPresent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Qualifier for the ``abstract`` keyword.
     */
    public static abstract
    class AbstractNode extends AdaNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AbstractNode";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AbstractNode.class.getMethod(
                        "pAsBool",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_as_bool",
                        new LibadalangField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AbstractNode NONE =
            new AbstractNodeNone();

        // ----- Constructors -----

        protected AbstractNode(
            final Entity entity
        ) {
            super(entity);
        }

        public static AbstractNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AbstractNode.NONE :

                (AbstractNode) AdaNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AbstractNode.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AbstractNode.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AbstractNode.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AbstractNode.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----

        
    

        /**
         * Return whether this is an instance of AbstractPresent
         */
        public boolean pAsBool(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_abstract_node_p_as_bool(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.ada_abstract_node_p_as_bool(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * AbstractNode.
         */
        private static final class AbstractNodeNone extends AbstractNode {
            AbstractNodeNone() {super(Entity.NONE);}
        }

    }


    
    

    /**

     */
    public static 
    class AbstractAbsent extends AbstractNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AbstractAbsent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AbstractNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AbstractAbsent NONE =
            new AbstractAbsent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AbstractAbsent(
            final Entity entity
        ) {
            super(entity);
        }

        public static AbstractAbsent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AbstractAbsent.NONE :

                new AbstractAbsent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AbstractAbsent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AbstractAbsent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AbstractAbsent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AbstractAbsent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class AbstractPresent extends AbstractNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AbstractPresent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AbstractNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AbstractPresent NONE =
            new AbstractPresent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AbstractPresent(
            final Entity entity
        ) {
            super(entity);
        }

        public static AbstractPresent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AbstractPresent.NONE :

                new AbstractPresent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AbstractPresent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AbstractPresent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AbstractPresent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AbstractPresent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static abstract
    class AdaList extends AdaNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AdaList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AdaList NONE =
            new AdaListNone();

        // ----- Constructors -----

        protected AdaList(
            final Entity entity
        ) {
            super(entity);
        }

        public static AdaList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AdaList.NONE :

                (AdaList) AdaNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AdaList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AdaList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AdaList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AdaList.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * AdaList.
         */
        private static final class AdaListNone extends AdaList {
            AdaListNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * List of AdaNode.
     *
     * This list node can contain one of the following nodes:
     * ``AbstractStateDecl``, ``AbstractSubpDecl``, ``Allocator``,
     * ``AspectClause``, ``AttributeRef``, ``BaseAggregate``, ``BaseSubpBody``,
     * ``BinOp``, ``BodyStub``, ``CallExpr``, ``CharLiteral``,
     * ``ComponentClause``, ``ComponentDecl``, ``ConcatOp``,
     * ``ConcreteTypeDecl``, ``CondExpr``, ``DeclExpr``, ``DottedName``,
     * ``EntryBody``, ``EntryDecl``, ``ErrorDecl``, ``ExceptionDecl``,
     * ``ExceptionHandler``, ``ExplicitDeref``, ``GenericDecl``,
     * ``GenericFormal``, ``GenericInstantiation``, ``GenericRenamingDecl``,
     * ``Identifier``, ``IncompleteTypeDecl``, ``MembershipExpr``,
     * ``NullComponentDecl``, ``NullLiteral``, ``NumLiteral``, ``NumberDecl``,
     * ``ObjectDecl``, ``OthersDesignator``, ``PackageBody``, ``PackageDecl``,
     * ``PackageRenamingDecl``, ``ParenAbstractStateDecl``, ``ParenExpr``,
     * ``PragmaNode``, ``ProtectedBody``, ``ProtectedTypeDecl``, ``QualExpr``,
     * ``QuantifiedExpr``, ``RaiseExpr``, ``ReduceAttributeRef``,
     * ``SingleProtectedDecl``, ``SingleTaskDecl``, ``Stmt``,
     * ``StringLiteral``, ``SubpDecl``, ``SubtypeDecl``, ``SubtypeIndication``,
     * ``TargetName``, ``TaskBody``, ``TaskTypeDecl``, ``UnOp``,
     * ``UpdateAttributeRef``, ``UseClause``, ``WithClause``
     */
    public static 
    class AdaNodeList extends AdaList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AdaNodeList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AdaNodeList NONE =
            new AdaNodeList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AdaNodeList(
            final Entity entity
        ) {
            super(entity);
        }

        public static AdaNodeList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AdaNodeList.NONE :

                new AdaNodeList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AdaNodeList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AdaNodeList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AdaNodeList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AdaNodeList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of AbstractStateDecls.
     *
     * This list node can contain one of the following nodes:
     * ``AbstractStateDecl``, ``ParenAbstractStateDecl``
     */
    public static 
    class AbstractStateDeclList extends AdaNodeList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AbstractStateDeclList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNodeList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AbstractStateDeclList NONE =
            new AbstractStateDeclList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AbstractStateDeclList(
            final Entity entity
        ) {
            super(entity);
        }

        public static AbstractStateDeclList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AbstractStateDeclList.NONE :

                new AbstractStateDeclList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AbstractStateDeclList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AbstractStateDeclList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AbstractStateDeclList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AbstractStateDeclList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of alternatives in a ``when ...`` clause.
     *
     * This list node can contain one of the following nodes: ``Allocator``,
     * ``AttributeRef``, ``BaseAggregate``, ``BinOp``, ``CallExpr``,
     * ``CharLiteral``, ``ConcatOp``, ``CondExpr``, ``DeclExpr``,
     * ``DiscreteSubtypeIndication``, ``DottedName``, ``ExplicitDeref``,
     * ``Identifier``, ``MembershipExpr``, ``NullLiteral``, ``NumLiteral``,
     * ``OthersDesignator``, ``ParenExpr``, ``QualExpr``, ``QuantifiedExpr``,
     * ``RaiseExpr``, ``ReduceAttributeRef``, ``StringLiteral``,
     * ``TargetName``, ``UnOp``, ``UpdateAttributeRef``
     */
    public static 
    class AlternativesList extends AdaNodeList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AlternativesList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNodeList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AlternativesList NONE =
            new AlternativesList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AlternativesList(
            final Entity entity
        ) {
            super(entity);
        }

        public static AlternativesList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AlternativesList.NONE :

                new AlternativesList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AlternativesList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AlternativesList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AlternativesList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AlternativesList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of constraints.
     *
     * This list node can contain one of the following nodes: ``AttributeRef``,
     * ``BinOp``, ``CallExpr``, ``CharLiteral``, ``DottedName``,
     * ``ExplicitDeref``, ``Identifier``, ``QualExpr``, ``ReduceAttributeRef``,
     * ``StringLiteral``, ``SubtypeIndication``, ``TargetName``,
     * ``UpdateAttributeRef``
     */
    public static 
    class ConstraintList extends AdaNodeList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ConstraintList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNodeList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ConstraintList NONE =
            new ConstraintList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ConstraintList(
            final Entity entity
        ) {
            super(entity);
        }

        public static ConstraintList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ConstraintList.NONE :

                new ConstraintList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ConstraintList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ConstraintList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ConstraintList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return ConstraintList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of declarations.
     *
     * This list node can contain one of the following nodes:
     * ``AbstractSubpDecl``, ``AspectClause``, ``ComponentDecl``,
     * ``EntryDecl``, ``ExprFunction``, ``NullSubpDecl``, ``PragmaNode``,
     * ``SubpDecl``, ``SubpRenamingDecl``
     */
    public static 
    class DeclList extends AdaNodeList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "DeclList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNodeList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final DeclList NONE =
            new DeclList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected DeclList(
            final Entity entity
        ) {
            super(entity);
        }

        public static DeclList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                DeclList.NONE :

                new DeclList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return DeclList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return DeclList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return DeclList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return DeclList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of statements.
     *
     * This list node can contain one of the following nodes: ``PragmaNode``,
     * ``Stmt``
     */
    public static 
    class StmtList extends AdaNodeList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "StmtList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaNodeList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final StmtList NONE =
            new StmtList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected StmtList(
            final Entity entity
        ) {
            super(entity);
        }

        public static StmtList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                StmtList.NONE :

                new StmtList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return StmtList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return StmtList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return StmtList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return StmtList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of AspectAssoc.
     */
    public static 
    class AspectAssocList extends AdaList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AspectAssocList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AspectAssocList NONE =
            new AspectAssocList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AspectAssocList(
            final Entity entity
        ) {
            super(entity);
        }

        public static AspectAssocList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AspectAssocList.NONE :

                new AspectAssocList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AspectAssocList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AspectAssocList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AspectAssocList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AspectAssocList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of BaseAssoc.
     */
    public static 
    class BaseAssocList extends AdaList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "BaseAssocList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseAssocList NONE =
            new BaseAssocList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BaseAssocList(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseAssocList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseAssocList.NONE :

                new BaseAssocList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return BaseAssocList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return BaseAssocList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return BaseAssocList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return BaseAssocList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of BasicAssoc.
     */
    public static abstract
    class BasicAssocList extends AdaList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "BasicAssocList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            AdaList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BasicAssocList NONE =
            new BasicAssocListNone();

        // ----- Constructors -----

        protected BasicAssocList(
            final Entity entity
        ) {
            super(entity);
        }

        public static BasicAssocList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BasicAssocList.NONE :

                (BasicAssocList) AdaNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return BasicAssocList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return BasicAssocList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return BasicAssocList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return BasicAssocList.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BasicAssocList.
         */
        private static final class BasicAssocListNone extends BasicAssocList {
            BasicAssocListNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * List of associations.
     */
    public static 
    class AssocList extends BasicAssocList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AssocList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, LibadalangField>
        fieldDescriptions = new HashMap<>(
            BasicAssocList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AssocList.class.getMethod(
                        "pZipWithParams",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new ParamWithDefaultValue(
                        boolean.class,
                        "impreciseFallback",
                        false
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_zip_with_params",
                        new LibadalangField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AssocList NONE =
            new AssocList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AssocList(
            final Entity entity
        ) {
            super(entity);
        }

        public static AssocList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AssocList.NONE :

                new AssocList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AssocList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AssocList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AssocList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public LibadalangField getFieldDescription(
            final String name
        ) {
            return AssocList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * Returns an array of pairs, associating formal parameters to actual
         * expressions. The formals to match are retrieved by resolving the
         * call which this AssocList represents the actuals of.
         */
        public ParamActualArray pZipWithParams(
            final boolean impreciseFallback
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte impreciseFallbackNative = (impreciseFallback ? (byte) 1 : (byte) 0);

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.ada_assoc_list_p_zip_with_params(
                    thisNative,
                    impreciseFallbackNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ParamActualArray res = ParamActualArray.NONE;