%define OT_KarhunenLoeveAlgorithm_doc
"Base class for Karhunen-Loeve algorithms.

Parameters
----------
covModel : :class:`~openturns.CovarianceModel`
    The covariance model.
s : float, :math:`\\geq0`
    The minimal relative amplitude of the eigenvalues to consider in
    the decomposition wrt the sum of the preceding eigenvalues. The default value is 0.

Notes
-----
The Karhunen-Loeve decomposition enables to build some finite approximations of stochastic processes which are optimal with respect to the norm :math:`L^2`.

We suppose that :math:`C:\\cD \\times \\cD \\rightarrow  \\cS^+_d(\\Rset)` is a covariance function defined on :math:`\\cD \\in \\Rset^n`, continuous at :math:`\\vect{0}`. 

The class :class:`~openturns.KarhunenLoeveAlgorithm` enables to determine the solutions of the second kind Fredholm equation associated to  :math:`C`, ie to find the :math:`(\\lambda_k,  \\vect{\\varphi}_k)_{k \\geq 1}` such that: 

.. math::
    :label: fredholm

    \\int_{\\cD} \\mat{C}(\\vect{s},\\vect{t}) \\vect{\\varphi}_k(\\vect{t})\\,  d\\vect{t} = \\lambda_k  \\vect{\\varphi}_k(\\vect{s}) \\quad \\forall \\vect{s} \\in \\cD


where :math:`(\\lambda_k)_{k \\geq 1}` is a nonincreasing sequence of nonnegative values (the **eigenvalues**) and :math:`(\\vect{\\varphi}_k)_{k \\geq 1}` the   associated sequence of **eigenfunctions**, normalized by :math:`\\int_{\\cD}\\|\\vect{\\varphi}_k(\\vect{s})\\|^2\\di{\\vect{s}}=1`. They form an hilbertian basis of :math:`L^2(\\cD, \\Rset^d)`.


The Mercer theorem shows that the covariance function  :math:`C` writes:

.. math::
    :label: covFuncMercer

    \\mat{C}(\\vect{s},\\vect{t}) = \\sum_{k=1}^{+\\infty} \\lambda_k \\vect{\\varphi}_k(\\vect{s}) \\Tr{\\vect{\\varphi}}_k(\\vect{t}) \\quad \\forall (\\vect{s}, \\vect{t}) \\in \\cD \\times \\cD


The threshold :math:`s` is used in order to select the most significant eigenvalues, ie all the eigenvalues such that (the infinite sum on the right being replaced by the sum of all computed eigenvalues in numerical algorithms): 

.. math::
    :label: thresholdK

    K_s = \\min \\left\\{ k \\in \\Nset \\, | \\, \\sum_{i=1}^{k}{\\lambda_i} \\geq (1-s) \\times \\sum_{i \\geq 1}^{+\\infty} \\lambda_i \\right\\}

The number of significant eigenvalues can also be specified directly by using :func:`setNbModes`. In this case, only :math:`n` eigenvalues will be computed, and the number of selected eigenvalues will be:

.. math::
    :label: combinedK
    
    K = \\min \\left\\{ n \\, , \\, K_s \\right\\}
    
with :math:`K_s` as defined in :eq:`thresholdK`.

Thus if threshold is set to 0, the number of selected eigenvalues is set by :func:`setNbModes`.
    
To solve :eq:`fredholm`, we use the functional basis :math:`(\\theta_p)_{1 \\leq p \\leq P}` of :math:`L^2(\\cD, \\Rset)` with :math:`P` elements defined on :math:`\\cD`. We search the solutions of type:

.. math::

    \\tilde{\\vect{\\varphi}}_k(\\vect{t})=\\sum_{p=1}^{P} \\vect{\\phi}_{pk}\\theta_p(\\vect{t})

where :math:`\\vect{\\phi}_{pk} \\in \\Rset^d`. We note:

.. math::

    \\begin{align*}
        \\vect{\\Phi}_k =
        \\left(
          \\begin{array}{l}
            \\vect{\\phi}_{1k} \\\\
            \\dots \\\\
            \\vect{\\phi}_{Pk}
          \\end{array}
        \\right) \\in \\Rset^{Pd}
     \\end{align*}

and :math:`\\mat{\\vect{\\Phi}} = (\\vect{\\Phi}_1\\, |\\, \\dots \\, | \\vect{\\Phi}_K)` the matrix of the :math:`K` first modes of the Karhunen-Loeve decomposition.

The approximated Fredholm problem writes for all :math:`k \\geq 1`:

.. math::

   \\int_{\\cD} \\mat{C}(\\vect{s},\\vect{t}) \\tilde{\\vect{\\varphi}}_k(\\vect{t})\\,  d\\vect{t} = \\lambda_k   \\tilde{\\vect{\\varphi}}_k(\\vect{s}) \\quad  \\forall \\vect{s} \\in \\cD

which enables to define the **residual function** :math:`\\vect{r}: \\cD \\rightarrow \\Rset^d` defined by

.. math::
    :label: fredholmApprox

    \\vect{r}(\\vect{s}) = \\int_{\\cD} \\mat{C}(\\vect{s},\\vect{t}) \\tilde{\\vect{\\varphi}}_k(\\vect{t})\\,  d\\vect{t} - \\lambda_k  \\tilde{\\vect{\\varphi}}_k(\\vect{s})

The Fredholm problem writes:

.. math::
    :label: pbResidu

    \\vect{r}(\\vect{s}) = \\vect{0} \\quad \\forall \\vect{s} \\in \\cD

which is solved either by the **Galerkin** approach or the **collocation** approach.

The integrals in :eq:`fredholmApprox` can be evaluated with:

    - a :math:`P_1` -approach: see :class:`~openturns.KarhunenLoeveP1Algorithm`,
    - a quadrature approach: see :class:`~openturns.KarhunenLoeveQuadratureAlgorithm`,
    - a singular values decomposition approach: see :class:`~openturns.KarhunenLoeveSVDAlgorithm`.

See also
--------
KarhunenLoeveResult"
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation
OT_KarhunenLoeveAlgorithm_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_getThreshold_doc
"Accessor to the threshold used to select the most significant eigenmodes.

Returns
-------
s : float, positive
    The threshold :math:`s`. 

Notes
-----
OpenTURNS truncates the sequence :math:`(\\lambda_k,  \\vect{\\varphi}_k)_{k \\geq 1}`  at the index :math:`K` defined in :eq:`thresholdK`."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::getThreshold
OT_KarhunenLoeveAlgorithm_getThreshold_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_getCovarianceModel_doc
"Accessor to the covariance model.

Returns
-------
covModel : :class:`~openturns.CovarianceModel`
    The covariance model."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::getCovarianceModel
OT_KarhunenLoeveAlgorithm_getCovarianceModel_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_setThreshold_doc
"Accessor to the limit ratio on eigenvalues.

Parameters
----------
s : float, :math:`\\geq 0`
    The threshold :math:`s` defined in :eq:`thresholdK`."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::setThreshold
OT_KarhunenLoeveAlgorithm_setThreshold_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_setCovarianceModel_doc
"Accessor to the covariance model.

Parameters
----------
covModel : :class:`~openturns.CovarianceModel`
    The covariance model."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::setCovarianceModel
OT_KarhunenLoeveAlgorithm_setCovarianceModel_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_run_doc
"Launch the algorithm.

Notes
-----
It launches the algorithm and creates a :class:`~openturns.KarhunenLoeveResult`,
structure containing all the results."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::run
OT_KarhunenLoeveAlgorithm_run_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_getResult_doc
"Get the result structure.

Returns
-------
resKL : :class:`~openturns.KarhunenLoeveResult`
    The structure containing all the results of the Fredholm problem.

Notes
-----
The structure contains all the results of the Fredholm problem."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::getResult
OT_KarhunenLoeveAlgorithm_getResult_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_setNbModes_doc
"Accessor to the maximum number of modes to compute.

Parameters
----------
n : int
    The maximum number of modes to compute.
    The actual number of modes also depends on the threshold criterion."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::setNbModes
OT_KarhunenLoeveAlgorithm_setNbModes_doc

// ---------------------------------------------------------------------

%define OT_KarhunenLoeveAlgorithm_getNbModes_doc
"Accessor to number of modes to compute.

Returns
-------
n : int
    The maximum number of modes to compute.
    The actual number of modes also depends on the threshold criterion."
%enddef
%feature("docstring") OT::KarhunenLoeveAlgorithmImplementation::getNbModes
OT_KarhunenLoeveAlgorithm_getNbModes_doc

// ---------------------------------------------------------------------
