// Copyright (c) 2024, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#ifndef NGLOG_LOG_SEVERITY_H
#define NGLOG_LOG_SEVERITY_H

#if defined(NGLOG_USE_EXPORT)
#  include "ng-log/export.h"
#endif

#if !defined(NGLOG_EXPORT)
#  error <ng-log/log_severity.h> was not included correctly. See the documentation for how to consume the library.
#endif

namespace nglog {

// The recommended semantics of the log levels are as follows:
//
// INFO:
//   Use for state changes or other major events, or to aid debugging.
// WARNING:
//   Use for undesired but relatively expected events, which may indicate a
//   problem
// ERROR:
//   Use for undesired and unexpected events that the program can recover from.
//   All ERRORs should be actionable - it should be appropriate to file a bug
//   whenever an ERROR occurs in production.
// FATAL:
//   Use for undesired and unexpected events that the program cannot recover
//   from.

// Variables of type LogSeverity are widely taken to lie in the range
// [0, NUM_SEVERITIES-1].  Be careful to preserve this assumption if
// you ever need to change their values or add a new severity.

enum LogSeverity {
  NGLOG_INFO = 0,
  NGLOG_WARNING = 1,
  NGLOG_ERROR = 2,
  NGLOG_FATAL = 3,
#ifndef NGLOG_NO_ABBREVIATED_SEVERITIES
#  ifdef ERROR
#  error ERROR macro is defined. Define NGLOG_NO_ABBREVIATED_SEVERITIES before including logging.h. See the document for detail.
#  endif
  INFO = NGLOG_INFO,
  WARNING = NGLOG_WARNING,
  ERROR = NGLOG_ERROR,
  FATAL = NGLOG_FATAL
#endif
};

#if defined(__cpp_inline_variables)
#  if (__cpp_inline_variables >= 201606L)
#    define NGLOG_INLINE_VARIABLE inline
#  endif  // (__cpp_inline_variables >= 201606L)
#endif    // defined(__cpp_inline_variables)

#if !defined(NGLOG_INLINE_VARIABLE)
#  define NGLOG_INLINE_VARIABLE
#endif  // !defined(NGLOG_INLINE_VARIABLE)

NGLOG_INLINE_VARIABLE
constexpr int NUM_SEVERITIES = 4;

// DFATAL is FATAL in debug mode, ERROR in normal mode
#ifdef NDEBUG
#  define DFATAL_LEVEL ERROR
#else
#  define DFATAL_LEVEL FATAL
#endif

// NDEBUG usage helpers related to (RAW_)DCHECK:
//
// DEBUG_MODE is for small !NDEBUG uses like
//   if (DEBUG_MODE) foo.CheckThatFoo();
// instead of substantially more verbose
//   #ifndef NDEBUG
//     foo.CheckThatFoo();
//   #endif
//
// IF_DEBUG_MODE is for small !NDEBUG uses like
//   IF_DEBUG_MODE( string error; )
//   DCHECK(Foo(&error)) << error;
// instead of substantially more verbose
//   #ifndef NDEBUG
//     string error;
//     DCHECK(Foo(&error)) << error;
//   #endif
//
#ifdef NDEBUG
enum { DEBUG_MODE = 0 };
#  define IF_DEBUG_MODE(x)
#else
enum { DEBUG_MODE = 1 };
#  define IF_DEBUG_MODE(x) x
#endif

}  // namespace nglog

#endif  // NGLOG_LOG_SEVERITY_H
