/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#pragma once

#include <glib.h>

/**
 * SECTION: version
 * @section_id: version-macros
 * @title: Version related macros
 * @include: gandiva-glib/gandiva-glib.h
 *
 * Gandiva GLib provides macros that can be used by C pre-processor.
 * They are useful to check version related things at compile time.
 */

/**
 * GGANDIVA_VERSION_MAJOR:
 *
 * The major version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_MAJOR (19)

/**
 * GGANDIVA_VERSION_MINOR:
 *
 * The minor version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_MINOR (0)

/**
 * GGANDIVA_VERSION_MICRO:
 *
 * The micro version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_MICRO (0)

/**
 * GGANDIVA_VERSION_TAG:
 *
 * The version tag. Normally, it's an empty string. It's "SNAPSHOT"
 * for snapshot version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_TAG   ""

/**
 * GGANDIVA_VERSION_CHECK:
 * @major: A major version to check for.
 * @minor: A minor version to check for.
 * @micro: A micro version to check for.
 *
 * You can use this macro in C pre-processor.
 *
 * Returns: %TRUE if the compile time Gandiva GLib version is the
 *   same as or newer than the passed version, %FALSE otherwise.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_CHECK(major, minor, micro)     \
  (GGANDIVA_MAJOR_VERSION > (major) ||                  \
   (GGANDIVA_MAJOR_VERSION == (major) &&                \
    GGANDIVA_MINOR_VERSION > (minor)) ||                \
   (GGANDIVA_MAJOR_VERSION == (major) &&                \
    GGANDIVA_MINOR_VERSION == (minor) &&                \
    GGANDIVA_MICRO_VERSION >= (micro)))

/**
 * GGANDIVA_DISABLE_DEPRECATION_WARNINGS:
 *
 * If this macro is defined, no deprecated warnings are produced.
 *
 * You must define this macro before including the
 * gandiva-glib/gandiva-glib.h header.
 *
 * Since: 1.0.0
 */

#ifdef GGANDIVA_DISABLE_DEPRECATION_WARNINGS
#  define GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_FOR(function)
#  define GGANDIVA_UNAVAILABLE(major, minor)
#else
#  define GGANDIVA_DEPRECATED G_DEPRECATED
#  define GGANDIVA_DEPRECATED_FOR(function) G_DEPRECATED_FOR(function)
#  define GGANDIVA_UNAVAILABLE(major, minor) G_UNAVAILABLE(major, minor)
#endif

/**
 * GGANDIVA_VERSION_19_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 19.0.0
 */
#define GGANDIVA_VERSION_19_0 G_ENCODE_VERSION(19, 0)

/**
 * GGANDIVA_VERSION_18_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 18.0.0
 */
#define GGANDIVA_VERSION_18_0 G_ENCODE_VERSION(18, 0)

/**
 * GGANDIVA_VERSION_17_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 17.0.0
 */
#define GGANDIVA_VERSION_17_0 G_ENCODE_VERSION(17, 0)

/**
 * GGANDIVA_VERSION_16_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 16.0.0
 */
#define GGANDIVA_VERSION_16_0 G_ENCODE_VERSION(16, 0)

/**
 * GGANDIVA_VERSION_15_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 15.0.0
 */
#define GGANDIVA_VERSION_15_0 G_ENCODE_VERSION(15, 0)

/**
 * GGANDIVA_VERSION_14_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 14.0.0
 */
#define GGANDIVA_VERSION_14_0 G_ENCODE_VERSION(14, 0)

/**
 * GGANDIVA_VERSION_13_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 13.0.0
 */
#define GGANDIVA_VERSION_13_0 G_ENCODE_VERSION(13, 0)

/**
 * GGANDIVA_VERSION_12_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 12.0.0
 */
#define GGANDIVA_VERSION_12_0 G_ENCODE_VERSION(12, 0)

/**
 * GGANDIVA_VERSION_11_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 11.0.0
 */
#define GGANDIVA_VERSION_11_0 G_ENCODE_VERSION(11, 0)

/**
 * GGANDIVA_VERSION_10_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 10.0.0
 */
#define GGANDIVA_VERSION_10_0 G_ENCODE_VERSION(10, 0)

/**
 * GGANDIVA_VERSION_9_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 9.0.0
 */
#define GGANDIVA_VERSION_9_0 G_ENCODE_VERSION(9, 0)

/**
 * GGANDIVA_VERSION_8_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 8.0.0
 */
#define GGANDIVA_VERSION_8_0 G_ENCODE_VERSION(8, 0)

/**
 * GGANDIVA_VERSION_7_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 7.0.0
 */
#define GGANDIVA_VERSION_7_0 G_ENCODE_VERSION(7, 0)

/**
 * GGANDIVA_VERSION_6_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 6.0.0
 */
#define GGANDIVA_VERSION_6_0 G_ENCODE_VERSION(6, 0)

/**
 * GGANDIVA_VERSION_5_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 5.0.0
 */
#define GGANDIVA_VERSION_5_0 G_ENCODE_VERSION(5, 0)

/**
 * GGANDIVA_VERSION_4_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 4.0.0
 */
#define GGANDIVA_VERSION_4_0 G_ENCODE_VERSION(4, 0)

/**
 * GGANDIVA_VERSION_3_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 3.0.0
 */
#define GGANDIVA_VERSION_3_0 G_ENCODE_VERSION(3, 0)

/**
 * GGANDIVA_VERSION_2_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 2.0.0
 */
#define GGANDIVA_VERSION_2_0 G_ENCODE_VERSION(2, 0)

/**
 * GGANDIVA_VERSION_1_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_1_0 G_ENCODE_VERSION(1, 0)

/**
 * GGANDIVA_VERSION_0_17:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.17.0
 */
#define GGANDIVA_VERSION_0_17 G_ENCODE_VERSION(0, 17)

/**
 * GGANDIVA_VERSION_0_16:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.16.0
 */
#define GGANDIVA_VERSION_0_16 G_ENCODE_VERSION(0, 16)

/**
 * GGANDIVA_VERSION_0_15:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.15.0
 */
#define GGANDIVA_VERSION_0_15 G_ENCODE_VERSION(0, 15)

/**
 * GGANDIVA_VERSION_0_14:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.14.0
 */
#define GGANDIVA_VERSION_0_14 G_ENCODE_VERSION(0, 14)

/**
 * GGANDIVA_VERSION_0_13:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.13.0
 */
#define GGANDIVA_VERSION_0_13 G_ENCODE_VERSION(0, 13)

/**
 * GGANDIVA_VERSION_0_12:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.12.0
 */
#define GGANDIVA_VERSION_0_12 G_ENCODE_VERSION(0, 12)

/**
 * GGANDIVA_VERSION_0_11:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.11.0
 */
#define GGANDIVA_VERSION_0_11 G_ENCODE_VERSION(0, 11)

/**
 * GGANDIVA_VERSION_0_10:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.10.0
 */
#define GGANDIVA_VERSION_0_10 G_ENCODE_VERSION(0, 10)

/**
 * GGANDIVA_VERSION_MIN_REQUIRED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GGANDIVA_VERSION_1_0.
 *
 * If you use any functions that is defined by newer version than
 * %GGANDIVA_VERSION_MIN_REQUIRED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * gandiva-glib/gandiva-glib.h header.
 *
 * Since: 1.0.0
 */
#ifndef GGANDIVA_VERSION_MIN_REQUIRED
#  define GGANDIVA_VERSION_MIN_REQUIRED                                 \
  G_ENCODE_VERSION(GGANDIVA_VERSION_MAJOR, GGANDIVA_VERSION_MINOR)
#endif

/**
 * GGANDIVA_VERSION_MAX_ALLOWED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GGANDIVA_VERSION_1_0.
 *
 * If you use any functions that is defined by newer version than
 * %GGANDIVA_VERSION_MAX_ALLOWED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * gandiva-glib/gandiva-glib.h header.
 *
 * Since: 1.0.0
 */
#ifndef GGANDIVA_VERSION_MAX_ALLOWED
#  define GGANDIVA_VERSION_MAX_ALLOWED                                  \
  G_ENCODE_VERSION(GGANDIVA_VERSION_MAJOR, GGANDIVA_VERSION_MINOR)
#endif

#if (defined(_WIN32) || defined(__CYGWIN__)) && defined(_MSC_VER) &&   !defined(GGANDIVA_STATIC_COMPILATION)
#  define GGANDIVA_EXPORT __declspec(dllexport)
#  define GGANDIVA_IMPORT __declspec(dllimport)
#else
#  define GGANDIVA_EXPORT
#  define GGANDIVA_IMPORT
#endif

#ifdef GGANDIVA_COMPILATION
#  define GGANDIVA_API GGANDIVA_EXPORT
#else
#  define GGANDIVA_API GGANDIVA_IMPORT
#endif

#define GGANDIVA_EXTERN GGANDIVA_API extern

#define GGANDIVA_AVAILABLE_IN_ALL GGANDIVA_EXTERN

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_19_0
#  define GGANDIVA_DEPRECATED_IN_19_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_19_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_19_0
#  define GGANDIVA_DEPRECATED_IN_19_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_19_0
#  define GGANDIVA_AVAILABLE_IN_19_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(19, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_19_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_18_0
#  define GGANDIVA_DEPRECATED_IN_18_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_18_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_18_0
#  define GGANDIVA_DEPRECATED_IN_18_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_18_0
#  define GGANDIVA_AVAILABLE_IN_18_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(18, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_18_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_17_0
#  define GGANDIVA_DEPRECATED_IN_17_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_17_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_17_0
#  define GGANDIVA_DEPRECATED_IN_17_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_17_0
#  define GGANDIVA_AVAILABLE_IN_17_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(17, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_17_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_16_0
#  define GGANDIVA_DEPRECATED_IN_16_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_16_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_16_0
#  define GGANDIVA_DEPRECATED_IN_16_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_16_0
#  define GGANDIVA_AVAILABLE_IN_16_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(16, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_16_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_15_0
#  define GGANDIVA_DEPRECATED_IN_15_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_15_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_15_0
#  define GGANDIVA_DEPRECATED_IN_15_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_15_0
#  define GGANDIVA_AVAILABLE_IN_15_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(15, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_15_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_14_0
#  define GGANDIVA_DEPRECATED_IN_14_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_14_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_14_0
#  define GGANDIVA_DEPRECATED_IN_14_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_14_0
#  define GGANDIVA_AVAILABLE_IN_14_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(14, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_14_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_13_0
#  define GGANDIVA_DEPRECATED_IN_13_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_13_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_13_0
#  define GGANDIVA_DEPRECATED_IN_13_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_13_0
#  define GGANDIVA_AVAILABLE_IN_13_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(13, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_13_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_12_0
#  define GGANDIVA_DEPRECATED_IN_12_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_12_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_12_0
#  define GGANDIVA_DEPRECATED_IN_12_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_12_0
#  define GGANDIVA_AVAILABLE_IN_12_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(12, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_12_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_11_0
#  define GGANDIVA_DEPRECATED_IN_11_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_11_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_11_0
#  define GGANDIVA_DEPRECATED_IN_11_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_11_0
#  define GGANDIVA_AVAILABLE_IN_11_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(11, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_11_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_10_0
#  define GGANDIVA_DEPRECATED_IN_10_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_10_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_10_0
#  define GGANDIVA_DEPRECATED_IN_10_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_10_0
#  define GGANDIVA_AVAILABLE_IN_10_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(10, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_10_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_9_0
#  define GGANDIVA_DEPRECATED_IN_9_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_9_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_9_0
#  define GGANDIVA_DEPRECATED_IN_9_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_9_0
#  define GGANDIVA_AVAILABLE_IN_9_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(9, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_9_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_8_0
#  define GGANDIVA_DEPRECATED_IN_8_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_8_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_8_0
#  define GGANDIVA_DEPRECATED_IN_8_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_8_0
#  define GGANDIVA_AVAILABLE_IN_8_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(8, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_8_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_7_0
#  define GGANDIVA_DEPRECATED_IN_7_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_7_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_7_0
#  define GGANDIVA_DEPRECATED_IN_7_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_7_0
#  define GGANDIVA_AVAILABLE_IN_7_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(7, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_7_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_6_0
#  define GGANDIVA_DEPRECATED_IN_6_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_6_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_6_0
#  define GGANDIVA_DEPRECATED_IN_6_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_6_0
#  define GGANDIVA_AVAILABLE_IN_6_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(6, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_6_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_5_0
#  define GGANDIVA_DEPRECATED_IN_5_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_5_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_5_0
#  define GGANDIVA_DEPRECATED_IN_5_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_5_0
#  define GGANDIVA_AVAILABLE_IN_5_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(5, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_5_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_4_0
#  define GGANDIVA_DEPRECATED_IN_4_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_4_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_4_0
#  define GGANDIVA_DEPRECATED_IN_4_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_4_0
#  define GGANDIVA_AVAILABLE_IN_4_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(4, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_4_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_3_0
#  define GGANDIVA_DEPRECATED_IN_3_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_3_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_3_0
#  define GGANDIVA_DEPRECATED_IN_3_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_3_0
#  define GGANDIVA_AVAILABLE_IN_3_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(3, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_3_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_2_0
#  define GGANDIVA_DEPRECATED_IN_2_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_2_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_2_0
#  define GGANDIVA_DEPRECATED_IN_2_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_2_0
#  define GGANDIVA_AVAILABLE_IN_2_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(2, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_2_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_1_0
#  define GGANDIVA_DEPRECATED_IN_1_0               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_1_0_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_1_0
#  define GGANDIVA_DEPRECATED_IN_1_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_1_0
#  define GGANDIVA_AVAILABLE_IN_1_0 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(1, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_1_0 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_17
#  define GGANDIVA_DEPRECATED_IN_0_17               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_17_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_17
#  define GGANDIVA_DEPRECATED_IN_0_17_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_17
#  define GGANDIVA_AVAILABLE_IN_0_17 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 17)
#else
#  define GGANDIVA_AVAILABLE_IN_0_17 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_16
#  define GGANDIVA_DEPRECATED_IN_0_16               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_16_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_16
#  define GGANDIVA_DEPRECATED_IN_0_16_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_16
#  define GGANDIVA_AVAILABLE_IN_0_16 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 16)
#else
#  define GGANDIVA_AVAILABLE_IN_0_16 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_15
#  define GGANDIVA_DEPRECATED_IN_0_15               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_15_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_15
#  define GGANDIVA_DEPRECATED_IN_0_15_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_15
#  define GGANDIVA_AVAILABLE_IN_0_15 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 15)
#else
#  define GGANDIVA_AVAILABLE_IN_0_15 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_14
#  define GGANDIVA_DEPRECATED_IN_0_14               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_14_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_14
#  define GGANDIVA_DEPRECATED_IN_0_14_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_14
#  define GGANDIVA_AVAILABLE_IN_0_14 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 14)
#else
#  define GGANDIVA_AVAILABLE_IN_0_14 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_13
#  define GGANDIVA_DEPRECATED_IN_0_13               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_13_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_13
#  define GGANDIVA_DEPRECATED_IN_0_13_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_13
#  define GGANDIVA_AVAILABLE_IN_0_13 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 13)
#else
#  define GGANDIVA_AVAILABLE_IN_0_13 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_12
#  define GGANDIVA_DEPRECATED_IN_0_12               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_12_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_12
#  define GGANDIVA_DEPRECATED_IN_0_12_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_12
#  define GGANDIVA_AVAILABLE_IN_0_12 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 12)
#else
#  define GGANDIVA_AVAILABLE_IN_0_12 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_11
#  define GGANDIVA_DEPRECATED_IN_0_11               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_11_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_11
#  define GGANDIVA_DEPRECATED_IN_0_11_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_11
#  define GGANDIVA_AVAILABLE_IN_0_11 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 11)
#else
#  define GGANDIVA_AVAILABLE_IN_0_11 GGANDIVA_EXTERN
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_10
#  define GGANDIVA_DEPRECATED_IN_0_10               GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_10_FOR(function) GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_10
#  define GGANDIVA_DEPRECATED_IN_0_10_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_10
#  define GGANDIVA_AVAILABLE_IN_0_10 GGANDIVA_EXTERN GGANDIVA_UNAVAILABLE(0, 10)
#else
#  define GGANDIVA_AVAILABLE_IN_0_10 GGANDIVA_EXTERN
#endif
