%feature("docstring") OT::ProductCovarianceModel
"Univariate covariance function defined as a product.

Parameters
----------
coll : sequence of :class:`~openturns.CovarianceModel`
    Collection of covariance models :math:`(C_k)_{1 \\leq k \\leq K}` of dimension :math:`d=1`.

Notes
-----
The *product* covariance function is a covariance function with 
dimension :math:`d=1`. 
This allows one to create a higher input dimension covariance model
by combining models of smaller input dimensions. 
The input dimension of each model in the collection does not necessarily have to be equal to 1. 
It defines a covariance model from the given collection as follows.

We consider the scalar stochastic process :math:`X: \\Omega \\times\\cD \\rightarrow \\Rset`, where :math:`\\omega \\in \\Omega` is an event, :math:`\\cD` is a domain of :math:`\\Rset^n`.

The product covariance function writes:

.. math::

    C(\\vect{s}, \\vect{t}) = \\prod_{k=1}^K  C_k(\\vect{s}_k, \\vect{t}_k), \\quad \\forall (\\vect{s}, \\vect{t}) \\in \\cD

where :math:`C_k : \\cD_k \\times \\cD_k \\rightarrow  \\cS_d^+(\\Rset)`  with :math:`\\cD_k \\in \\Rset^{d_k}`, :math:`\\cD = \\cD_1 \\times \\dots \\times \\cD_K`, :math:`\\cD_k \\subset \\Rset^{n_k}` and :math:`n1 + \\dots + n_K=n`.

:math:`C` also writes:

.. math::

    C(\\vect{s}, \\vect{t}) =  \\sigma^2 \\rho\\left(\\dfrac{\\vect{s}}{\\theta}, \\dfrac{\\vect{t}}{\\theta}\\right), \\quad \\forall (\\vect{s}, \\vect{t}) \\in \\cD


where the amplitude is:

.. math::

    \\sigma= \\prod_{k=1}^K \\sigma_k

the scale is:

.. math::

    \\Tr{\\vect{\\theta}}= \\left(\\Tr{\\vect{\\theta}}_1, \\dots, \\Tr{\\vect{\\theta}}_K \\right)

and the correlation function :math:`\\rho` is:

.. math::

    \\rho \\left( \\vect{s}, \\vect{t} \\right) = \\prod_{k=1}^K \\rho_k\\left( \\vect{s}_k, \\vect{t}_k \\right)


Examples
--------
Create a product covariance function from two exponential functions, each one defined on :math:`\\Rset \\times \\Rset \\rightarrow \\cS_1^+(\\Rset)`:

>>> import openturns as ot
>>> amplitude = [1.0]
>>> scale1 = [4.0]
>>> scale2 = [4.0]
>>> cov1 = ot.ExponentialModel(scale1, amplitude)
>>> cov2 = ot.ExponentialModel(scale2, amplitude)
>>> covarianceModel = ot.ProductCovarianceModel([cov1, cov2])"

// ----------------------------------------------------------------------------

%feature("docstring") OT::ProductCovarianceModel::getCollection
"Covariance model collection accessor.

Returns
-------
coll : sequence of :class:`~openturns.CovarianceModel`"

