%feature("docstring") OT::LinearModelAnalysis
"Analyse a linear model.

Available constructors:
    LinearModelAnalysis(linearModelResult)

Parameters
----------
linearModelResult : :class:`~openturns.LinearModelResult`
    A linear model result.

See Also
--------
LinearModelResult

Notes
-----
This class relies on a linear model result structure and performs diagnostic 
of linearity. This diagnostic mainly relies on graphics and a `summary` like
function (pretty-print)

By default, on graphs, labels of the 3 most significant points are displayed.
This number can be changed by modifying the ResourceMap key
(``LinearModelAnalysis-Identifiers``).

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Normal()
>>> Xsample = distribution.getSample(30)
>>> func = ot.SymbolicFunction(['x'], ['2 * x + 1'])
>>> Ysample = func(Xsample) + ot.Normal().getSample(30)
>>> algo = ot.LinearModelAlgorithm(Ysample, Xsample)
>>> result = algo.getResult()
>>> analysis = ot.LinearModelAnalysis(result)
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getLinearModelResult
"Accessor to the linear model result.

Returns
-------
linearModelResult : :class:`~openturns.LinearModelResult`
    The  linear model result which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsTScores
"Accessor to the coefficients of linear expansion over their standard error.

Returns
-------
tScores : :class:`~openturns.Point`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsPValues
"Accessor to the coefficients of the p values.

Returns
-------
pValues : :class:`~openturns.Point`
"
// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsConfidenceInterval
"Accessor to the confidence interval of level :math:`\\alpha` for the coefficients
of the linear expansion

Returns
-------
confidenceInterval : :class:`~openturns.Interval`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getFisherScore
"Accessor to the Fisher test.

Returns
-------
fisherScore : float
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getFisherPValue
"Accessor to the Fisher p value.

Returns
-------
fisherPValue : float
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultChiSquared
"Performs Chi-Square test.
The statistical test checks the Gaussian assumption of the model (null hypothesis).

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.

Notes
-----
The Chi-Square test is a goodness of fit test which objective is to check the
normality assumption (null hypothesis) of residuals (and thus the model).

Usually, Chi-Square test applies for discrete distributions. Here we rely on
the :meth:`~openturns.FittingTest.ChiSquared` to check the normality.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultKolmogorovSmirnov
"Performs Kolmogorov test.

Performs Kolmogorov test to check Gaussian assumption of the model (null hypothesis).

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :class:`~openturns.FittingTest::Kolmogorov`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultAndersonDarling
"Performs Anderson-Darling test.
The statistical test checks the Gaussian assumption of the model (null hypothesis).

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :class:`~openturns.NormalityTest::AndersonDarling`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestCramerVonMises
"Performs Cramer-Von Mises test.

The statistical test checks the Gaussian assumption of the model (null hypothesis).

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :class:`~openturns.NormalityTest::CramerVonMisesNormal`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawModelVsFitted
"Accessor to plot of model versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------
%feature("docstring") OT::LinearModelAnalysis::drawResidualsVsFitted
"Accessor to plot of residuals versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawScaleLocation
"Accessor to a Scale-Location plot of sqrt(abs(residuals)) versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawQQplot
"Accessor to plot a Normal quantiles-quantiles plot of standardized residuals.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawCookDistance
"Accessor to plot of Cook's distances versus row labels.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawResidualsVsLeverages
"Accessor to plot of residuals versus leverages that adds bands corresponding to Cook's distances of 0.5 and 1.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawCookVsLeverages
"Accessor to plot of Cook's distances versus leverage/(1-leverage). 

Returns
-------
graph : :class:`~openturns.Graph`
"

