﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/email/SESRequest.h>
#include <aws/email/SES_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SES {
namespace Model {

/**
 * <p>Represents a request to set the position of a receipt rule in a receipt rule
 * set. You use receipt rule sets to receive email with Amazon SES. For more
 * information, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-concepts.html">Amazon
 * SES Developer Guide</a>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/email-2010-12-01/SetReceiptRulePositionRequest">AWS
 * API Reference</a></p>
 */
class SetReceiptRulePositionRequest : public SESRequest {
 public:
  AWS_SES_API SetReceiptRulePositionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SetReceiptRulePosition"; }

  AWS_SES_API Aws::String SerializePayload() const override;

 protected:
  AWS_SES_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the receipt rule set that contains the receipt rule to
   * reposition.</p>
   */
  inline const Aws::String& GetRuleSetName() const { return m_ruleSetName; }
  inline bool RuleSetNameHasBeenSet() const { return m_ruleSetNameHasBeenSet; }
  template <typename RuleSetNameT = Aws::String>
  void SetRuleSetName(RuleSetNameT&& value) {
    m_ruleSetNameHasBeenSet = true;
    m_ruleSetName = std::forward<RuleSetNameT>(value);
  }
  template <typename RuleSetNameT = Aws::String>
  SetReceiptRulePositionRequest& WithRuleSetName(RuleSetNameT&& value) {
    SetRuleSetName(std::forward<RuleSetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the receipt rule to reposition.</p>
   */
  inline const Aws::String& GetRuleName() const { return m_ruleName; }
  inline bool RuleNameHasBeenSet() const { return m_ruleNameHasBeenSet; }
  template <typename RuleNameT = Aws::String>
  void SetRuleName(RuleNameT&& value) {
    m_ruleNameHasBeenSet = true;
    m_ruleName = std::forward<RuleNameT>(value);
  }
  template <typename RuleNameT = Aws::String>
  SetReceiptRulePositionRequest& WithRuleName(RuleNameT&& value) {
    SetRuleName(std::forward<RuleNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the receipt rule after which to place the specified receipt
   * rule.</p>
   */
  inline const Aws::String& GetAfter() const { return m_after; }
  inline bool AfterHasBeenSet() const { return m_afterHasBeenSet; }
  template <typename AfterT = Aws::String>
  void SetAfter(AfterT&& value) {
    m_afterHasBeenSet = true;
    m_after = std::forward<AfterT>(value);
  }
  template <typename AfterT = Aws::String>
  SetReceiptRulePositionRequest& WithAfter(AfterT&& value) {
    SetAfter(std::forward<AfterT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_ruleSetName;

  Aws::String m_ruleName;

  Aws::String m_after;
  bool m_ruleSetNameHasBeenSet = false;
  bool m_ruleNameHasBeenSet = false;
  bool m_afterHasBeenSet = false;
};

}  // namespace Model
}  // namespace SES
}  // namespace Aws
