﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>
#include <aws/elasticbeanstalk/model/ConfigurationOptionSetting.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 * <p>A list of validation messages for a specified configuration
 * template.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/ValidateConfigurationSettingsMessage">AWS
 * API Reference</a></p>
 */
class ValidateConfigurationSettingsRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API ValidateConfigurationSettingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ValidateConfigurationSettings"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the application that the configuration template or environment
   * belongs to.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  ValidateConfigurationSettingsRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the configuration template to validate the settings against.</p>
   * <p>Condition: You cannot specify both this and an environment name.</p>
   */
  inline const Aws::String& GetTemplateName() const { return m_templateName; }
  inline bool TemplateNameHasBeenSet() const { return m_templateNameHasBeenSet; }
  template <typename TemplateNameT = Aws::String>
  void SetTemplateName(TemplateNameT&& value) {
    m_templateNameHasBeenSet = true;
    m_templateName = std::forward<TemplateNameT>(value);
  }
  template <typename TemplateNameT = Aws::String>
  ValidateConfigurationSettingsRequest& WithTemplateName(TemplateNameT&& value) {
    SetTemplateName(std::forward<TemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the environment to validate the settings against.</p>
   * <p>Condition: You cannot specify both this and a configuration template
   * name.</p>
   */
  inline const Aws::String& GetEnvironmentName() const { return m_environmentName; }
  inline bool EnvironmentNameHasBeenSet() const { return m_environmentNameHasBeenSet; }
  template <typename EnvironmentNameT = Aws::String>
  void SetEnvironmentName(EnvironmentNameT&& value) {
    m_environmentNameHasBeenSet = true;
    m_environmentName = std::forward<EnvironmentNameT>(value);
  }
  template <typename EnvironmentNameT = Aws::String>
  ValidateConfigurationSettingsRequest& WithEnvironmentName(EnvironmentNameT&& value) {
    SetEnvironmentName(std::forward<EnvironmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of the options and desired values to evaluate.</p>
   */
  inline const Aws::Vector<ConfigurationOptionSetting>& GetOptionSettings() const { return m_optionSettings; }
  inline bool OptionSettingsHasBeenSet() const { return m_optionSettingsHasBeenSet; }
  template <typename OptionSettingsT = Aws::Vector<ConfigurationOptionSetting>>
  void SetOptionSettings(OptionSettingsT&& value) {
    m_optionSettingsHasBeenSet = true;
    m_optionSettings = std::forward<OptionSettingsT>(value);
  }
  template <typename OptionSettingsT = Aws::Vector<ConfigurationOptionSetting>>
  ValidateConfigurationSettingsRequest& WithOptionSettings(OptionSettingsT&& value) {
    SetOptionSettings(std::forward<OptionSettingsT>(value));
    return *this;
  }
  template <typename OptionSettingsT = ConfigurationOptionSetting>
  ValidateConfigurationSettingsRequest& AddOptionSettings(OptionSettingsT&& value) {
    m_optionSettingsHasBeenSet = true;
    m_optionSettings.emplace_back(std::forward<OptionSettingsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  Aws::String m_templateName;

  Aws::String m_environmentName;

  Aws::Vector<ConfigurationOptionSetting> m_optionSettings;
  bool m_applicationNameHasBeenSet = false;
  bool m_templateNameHasBeenSet = false;
  bool m_environmentNameHasBeenSet = false;
  bool m_optionSettingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
