﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>
#include <aws/elasticbeanstalk/model/ConfigurationOptionSetting.h>
#include <aws/elasticbeanstalk/model/SourceConfiguration.h>
#include <aws/elasticbeanstalk/model/Tag.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 * <p>Request to create a configuration template.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/CreateConfigurationTemplateMessage">AWS
 * API Reference</a></p>
 */
class CreateConfigurationTemplateRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API CreateConfigurationTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateConfigurationTemplate"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the Elastic Beanstalk application to associate with this
   * configuration template.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  CreateConfigurationTemplateRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the configuration template.</p> <p>Constraint: This name must be
   * unique per application.</p>
   */
  inline const Aws::String& GetTemplateName() const { return m_templateName; }
  inline bool TemplateNameHasBeenSet() const { return m_templateNameHasBeenSet; }
  template <typename TemplateNameT = Aws::String>
  void SetTemplateName(TemplateNameT&& value) {
    m_templateNameHasBeenSet = true;
    m_templateName = std::forward<TemplateNameT>(value);
  }
  template <typename TemplateNameT = Aws::String>
  CreateConfigurationTemplateRequest& WithTemplateName(TemplateNameT&& value) {
    SetTemplateName(std::forward<TemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of an Elastic Beanstalk solution stack (platform version) that this
   * configuration uses. For example, <code>64bit Amazon Linux 2013.09 running Tomcat
   * 7 Java 7</code>. A solution stack specifies the operating system, runtime, and
   * application server for a configuration template. It also determines the set of
   * configuration options as well as the possible and default values. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/concepts.platforms.html">Supported
   * Platforms</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p> <p>You
   * must specify <code>SolutionStackName</code> if you don't specify
   * <code>PlatformArn</code>, <code>EnvironmentId</code>, or
   * <code>SourceConfiguration</code>.</p> <p>Use the <a
   * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html">
   * <code>ListAvailableSolutionStacks</code> </a> API to obtain a list of available
   * solution stacks.</p>
   */
  inline const Aws::String& GetSolutionStackName() const { return m_solutionStackName; }
  inline bool SolutionStackNameHasBeenSet() const { return m_solutionStackNameHasBeenSet; }
  template <typename SolutionStackNameT = Aws::String>
  void SetSolutionStackName(SolutionStackNameT&& value) {
    m_solutionStackNameHasBeenSet = true;
    m_solutionStackName = std::forward<SolutionStackNameT>(value);
  }
  template <typename SolutionStackNameT = Aws::String>
  CreateConfigurationTemplateRequest& WithSolutionStackName(SolutionStackNameT&& value) {
    SetSolutionStackName(std::forward<SolutionStackNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the custom platform. For more information,
   * see <a
   * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/custom-platforms.html">
   * Custom Platforms</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p>
   *  <p>If you specify <code>PlatformArn</code>, then don't specify
   * <code>SolutionStackName</code>.</p>
   */
  inline const Aws::String& GetPlatformArn() const { return m_platformArn; }
  inline bool PlatformArnHasBeenSet() const { return m_platformArnHasBeenSet; }
  template <typename PlatformArnT = Aws::String>
  void SetPlatformArn(PlatformArnT&& value) {
    m_platformArnHasBeenSet = true;
    m_platformArn = std::forward<PlatformArnT>(value);
  }
  template <typename PlatformArnT = Aws::String>
  CreateConfigurationTemplateRequest& WithPlatformArn(PlatformArnT&& value) {
    SetPlatformArn(std::forward<PlatformArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An Elastic Beanstalk configuration template to base this one on. If
   * specified, Elastic Beanstalk uses the configuration values from the specified
   * configuration template to create a new configuration.</p> <p>Values specified in
   * <code>OptionSettings</code> override any values obtained from the
   * <code>SourceConfiguration</code>.</p> <p>You must specify
   * <code>SourceConfiguration</code> if you don't specify <code>PlatformArn</code>,
   * <code>EnvironmentId</code>, or <code>SolutionStackName</code>.</p>
   * <p>Constraint: If both solution stack name and source configuration are
   * specified, the solution stack of the source configuration template must match
   * the specified solution stack name.</p>
   */
  inline const SourceConfiguration& GetSourceConfiguration() const { return m_sourceConfiguration; }
  inline bool SourceConfigurationHasBeenSet() const { return m_sourceConfigurationHasBeenSet; }
  template <typename SourceConfigurationT = SourceConfiguration>
  void SetSourceConfiguration(SourceConfigurationT&& value) {
    m_sourceConfigurationHasBeenSet = true;
    m_sourceConfiguration = std::forward<SourceConfigurationT>(value);
  }
  template <typename SourceConfigurationT = SourceConfiguration>
  CreateConfigurationTemplateRequest& WithSourceConfiguration(SourceConfigurationT&& value) {
    SetSourceConfiguration(std::forward<SourceConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of an environment whose settings you want to use to create the
   * configuration template. You must specify <code>EnvironmentId</code> if you don't
   * specify <code>PlatformArn</code>, <code>SolutionStackName</code>, or
   * <code>SourceConfiguration</code>.</p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  inline bool EnvironmentIdHasBeenSet() const { return m_environmentIdHasBeenSet; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  CreateConfigurationTemplateRequest& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional description for this configuration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateConfigurationTemplateRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Option values for the Elastic Beanstalk configuration, such as the instance
   * type. If specified, these values override the values obtained from the solution
   * stack or the source configuration template. For a complete list of Elastic
   * Beanstalk configuration options, see <a
   * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option
   * Values</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p>
   */
  inline const Aws::Vector<ConfigurationOptionSetting>& GetOptionSettings() const { return m_optionSettings; }
  inline bool OptionSettingsHasBeenSet() const { return m_optionSettingsHasBeenSet; }
  template <typename OptionSettingsT = Aws::Vector<ConfigurationOptionSetting>>
  void SetOptionSettings(OptionSettingsT&& value) {
    m_optionSettingsHasBeenSet = true;
    m_optionSettings = std::forward<OptionSettingsT>(value);
  }
  template <typename OptionSettingsT = Aws::Vector<ConfigurationOptionSetting>>
  CreateConfigurationTemplateRequest& WithOptionSettings(OptionSettingsT&& value) {
    SetOptionSettings(std::forward<OptionSettingsT>(value));
    return *this;
  }
  template <typename OptionSettingsT = ConfigurationOptionSetting>
  CreateConfigurationTemplateRequest& AddOptionSettings(OptionSettingsT&& value) {
    m_optionSettingsHasBeenSet = true;
    m_optionSettings.emplace_back(std::forward<OptionSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the tags applied to the configuration template.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateConfigurationTemplateRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateConfigurationTemplateRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  Aws::String m_templateName;

  Aws::String m_solutionStackName;

  Aws::String m_platformArn;

  SourceConfiguration m_sourceConfiguration;

  Aws::String m_environmentId;

  Aws::String m_description;

  Aws::Vector<ConfigurationOptionSetting> m_optionSettings;

  Aws::Vector<Tag> m_tags;
  bool m_applicationNameHasBeenSet = false;
  bool m_templateNameHasBeenSet = false;
  bool m_solutionStackNameHasBeenSet = false;
  bool m_platformArnHasBeenSet = false;
  bool m_sourceConfigurationHasBeenSet = false;
  bool m_environmentIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_optionSettingsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
