﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-sync/CognitoSyncRequest.h>
#include <aws/cognito-sync/CognitoSync_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CognitoSync {
namespace Model {

/**
 * Request for a list of datasets for an identity.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/ListDatasetsRequest">AWS
 * API Reference</a></p>
 */
class ListDatasetsRequest : public CognitoSyncRequest {
 public:
  AWS_COGNITOSYNC_API ListDatasetsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListDatasets"; }

  AWS_COGNITOSYNC_API Aws::String SerializePayload() const override;

  AWS_COGNITOSYNC_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * A name-spaced GUID (for example, us-east-1:23EC4050-6AEA-7089-A2DD-08002EXAMPLE)
   * created by Amazon Cognito. GUID generation is unique within a region.
   */
  inline const Aws::String& GetIdentityPoolId() const { return m_identityPoolId; }
  inline bool IdentityPoolIdHasBeenSet() const { return m_identityPoolIdHasBeenSet; }
  template <typename IdentityPoolIdT = Aws::String>
  void SetIdentityPoolId(IdentityPoolIdT&& value) {
    m_identityPoolIdHasBeenSet = true;
    m_identityPoolId = std::forward<IdentityPoolIdT>(value);
  }
  template <typename IdentityPoolIdT = Aws::String>
  ListDatasetsRequest& WithIdentityPoolId(IdentityPoolIdT&& value) {
    SetIdentityPoolId(std::forward<IdentityPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * A name-spaced GUID (for example, us-east-1:23EC4050-6AEA-7089-A2DD-08002EXAMPLE)
   * created by Amazon Cognito. GUID generation is unique within a region.
   */
  inline const Aws::String& GetIdentityId() const { return m_identityId; }
  inline bool IdentityIdHasBeenSet() const { return m_identityIdHasBeenSet; }
  template <typename IdentityIdT = Aws::String>
  void SetIdentityId(IdentityIdT&& value) {
    m_identityIdHasBeenSet = true;
    m_identityId = std::forward<IdentityIdT>(value);
  }
  template <typename IdentityIdT = Aws::String>
  ListDatasetsRequest& WithIdentityId(IdentityIdT&& value) {
    SetIdentityId(std::forward<IdentityIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * A pagination token for obtaining the next page of results.
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListDatasetsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The maximum number of results to be returned.
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListDatasetsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_identityPoolId;

  Aws::String m_identityId;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_identityPoolIdHasBeenSet = false;
  bool m_identityIdHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoSync
}  // namespace Aws
