# ----------------------------------------------------------------------------
#
#  Copyright (C) 2024 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


from time import sleep
from zita_jacktools.jackclient import JackClient
from zita_jacktools import jackfwplay_ext


class JackFwplay (JackClient):
    """
    JackFwplay is used together with one or more JackFwcapt instances
    to process audio files via a Jack graph in freewheel mode.
    """

    def __init__(self, nchan, client_name, server_name = None):
        """
        Create a new JackFwplay instance.

        The result should be checked using get_state().
        The optional 'server_name' allows to select between running
        Jack servers.
        """
        assert (nchan > 0)
        self._jfwplay, base = jackfwplay_ext.makecaps (self, client_name, server_name, nchan)
        super().__init__(base)


    def load_file (self, filename):
        """
        Load an audio file for playback.

        Returns zero for success, non zero otherwise.
        Also the range is set to the entire file.
        load_file (None) unloads the current file.
        """
        return jackfwplay_ext.load_file (self._jfwplay, filename)


    def fileinfo (self):
        """
        Return the lenght in samples and the channel count
        of the loaded file.
        """
        return jackfwplay_ext.fileinfo (self._jfwplay)


    def set_range (self, p0, p1):
        """
        Set the range of samples to play.
        p0 must be zero or positive to skip samples.
        p1 can be beyond the file size, adding silence.
        The files recorded by the JackFwcapt client(s)
        will have exactly p1-p0 samples.
        """
        return jackfwplay_ext.set_range (self._jfwplay, p0, p1)


    def start (self):
        """
        Start processing in freewheel mode. This will also
        trigger any connected JackFwcapt objects which have
        an open output file to start recording.
        """
        jackfwplay_ext.set_state (self._jfwplay, 1)


    def stop (self):
        """
        Stop processing. This will also make any connected
        JackFwcapt objects stop recording and close their
        output file.
        """
        return jackfwplay_ext.set_state (self._jfwplay, 0)
    
        
    def process (self, dt = 0.1):
        """
        Start, wait until the entire set range is processed,
        then stop. The currently loaded file remains active,
        so this can be repeated any number of times.
        """
        self.start ()
        sleep (0.1)
        while self.get_state () == self.PROCESS: sleep (dt);
        self.stop ()

