/*
 * Decompiled with CFR 0.152.
 */
package db.buffers;

import db.buffers.BlockStream;
import db.buffers.BufferFile;
import db.buffers.BufferFileAdapter;
import db.buffers.BufferFileBlock;
import db.buffers.ChangeMap;
import db.buffers.DataBuffer;
import db.buffers.InputBlockStream;
import db.buffers.LocalManagedBufferFile;
import db.buffers.ManagedBufferFileAdapter;
import db.buffers.OutputBlockStream;
import ghidra.util.BigEndianDataConverter;
import ghidra.util.Msg;
import ghidra.util.datastruct.IntSet;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.ClosedException;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.task.CancelledListener;
import ghidra.util.task.TaskMonitor;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.io.SyncFailedException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Random;

public class LocalBufferFile
implements BufferFile {
    static final long MAGIC_NUMBER = 3400271784588160042L;
    public static final String BUFFER_FILE_EXTENSION = ".gbf";
    public static final String PRESAVE_FILE_EXT = ".ps";
    public static final String PRESAVE_FILE_PREFIX = "tmp";
    public static final String TEMP_FILE_EXT = ".tmp";
    private static final String STRING_ENCODING = "UTF-8";
    private static final int MINIMUM_BLOCK_SIZE = 128;
    private static final Random random = new Random();
    private static final int HEADER_FORMAT_VERSION = 1;
    private static final int FILE_ID_OFFSET = 8;
    private static final int BUFFER_PREFIX_SIZE = 5;
    private static final int VER1_FIXED_HEADER_LENGTH = 32;
    private static final byte EMPTY_BUFFER = 1;
    static final int MAX_BUFFER_INDEX = 0x7FFFFFFE;
    private Hashtable<String, Integer> userParms = new Hashtable();
    private int[] freeIndexes = new int[0];
    private File file;
    private RandomAccessFile raf;
    private volatile LocalOutputBlockStream activeOutputBlockStream;
    private static InputBlockStreamFactory inputBlockStreamFactory = new DefaultInputBlockStreamFactory();
    private boolean temporary = false;
    private long fileId;
    private boolean readOnly;
    private int blockSize;
    private int bufferSize;
    private int bufferCount = 0;

    LocalBufferFile(int bufferSize, String tmpPrefix, String tmpExtension) throws IOException {
        this.bufferSize = bufferSize;
        this.blockSize = bufferSize + 5;
        this.readOnly = false;
        this.temporary = true;
        this.file = File.createTempFile(tmpPrefix, tmpExtension);
        this.raf = new RandomAccessFile(this.file, "rw");
    }

    public LocalBufferFile(File file, int bufferSize) throws IOException {
        if (file.exists()) {
            throw new DuplicateFileException("File " + file + " already exists");
        }
        this.file = file;
        this.bufferSize = bufferSize;
        this.blockSize = bufferSize + 5;
        this.readOnly = false;
        this.raf = new RandomAccessFile(file, "rw");
        this.fileId = random.nextLong();
    }

    public LocalBufferFile(File file, boolean readOnly) throws IOException {
        this.file = file;
        this.readOnly = readOnly;
        this.raf = new RandomAccessFile(file, readOnly ? "r" : "rw");
        this.readHeader();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void poke(File file, int bufferIndex, DataBuffer buf) throws IOException {
        try (LocalBufferFile bf = new LocalBufferFile(file, false);){
            bf.put(buf, bufferIndex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static DataBuffer peek(File file, int bufferIndex) throws IOException {
        try (LocalBufferFile bf = new LocalBufferFile(file, false);){
            DataBuffer buf = new DataBuffer(bf.getBufferSize());
            bf.get(buf, bufferIndex);
            DataBuffer dataBuffer = buf;
            return dataBuffer;
        }
    }

    public File getFile() {
        return this.file;
    }

    @Override
    public boolean isReadOnly() {
        return this.readOnly;
    }

    @Override
    public int getParameter(String name) throws NoSuchElementException {
        Integer obj = this.userParms.get(name);
        if (obj == null) {
            throw new NoSuchElementException(name);
        }
        return obj;
    }

    @Override
    public void setParameter(String name, int value) {
        this.userParms.put(name, value);
    }

    @Override
    public void clearParameters() {
        this.userParms.clear();
    }

    @Override
    public String[] getParameterNames() {
        ArrayList<String> list = new ArrayList<String>();
        Enumeration<String> it = this.userParms.keys();
        while (it.hasMoreElements()) {
            list.add(it.nextElement());
        }
        String[] names = new String[list.size()];
        list.toArray(names);
        return names;
    }

    @Override
    public int[] getFreeIndexes() {
        return (int[])this.freeIndexes.clone();
    }

    @Override
    public void setFreeIndexes(int[] indexes) {
        this.freeIndexes = (int[])indexes.clone();
        Arrays.sort(this.freeIndexes);
    }

    long getFileId() {
        return this.fileId;
    }

    void setFileId(long id) {
        this.fileId = id;
    }

    int getBufferCount() {
        return this.bufferCount;
    }

    void setBufferCount(int count) {
        this.bufferCount = count;
    }

    void setTemporary(boolean isTemporary) {
        this.temporary = isTemporary;
    }

    boolean renameFile(File newFile) throws IOException {
        if (this.raf != null) {
            this.raf.close();
        }
        if (this.file.renameTo(newFile)) {
            this.file = newFile;
            if (this.raf != null) {
                this.raf = new RandomAccessFile(this.file, "rw");
            }
            return true;
        }
        return false;
    }

    private int seekBufferBlock(int bufferIndex) throws IOException {
        int blockIndex = bufferIndex + 1;
        long offset = (long)blockIndex * (long)this.blockSize;
        this.raf.seek(offset);
        return blockIndex;
    }

    private void seekBlock(int blockIndex, int offsetWithinBlock) throws IOException {
        long offset = (long)blockIndex * (long)this.blockSize + (long)offsetWithinBlock;
        this.raf.seek(offset);
    }

    private void readHeader() throws IOException {
        this.seekBlock(0, 0);
        long magicNumber = this.raf.readLong();
        if (magicNumber != 3400271784588160042L) {
            throw new IOException("Unrecognized file format");
        }
        this.fileId = this.raf.readLong();
        int headerFormatVersion = this.raf.readInt();
        if (headerFormatVersion != 1) {
            throw new IOException("Unrecognized file format");
        }
        this.blockSize = this.raf.readInt();
        this.bufferSize = this.blockSize - 5;
        int firstFreeBufferIndex = this.raf.readInt();
        long len = this.raf.length();
        if (len % (long)this.blockSize != 0L) {
            throw new IOException("Corrupt file");
        }
        this.bufferCount = (int)(len / (long)this.blockSize) - 1;
        int cnt = this.raf.readInt();
        this.clearParameters();
        for (int i = 0; i < cnt; ++i) {
            int nameLen = this.raf.readInt();
            byte[] nameBytes = new byte[nameLen];
            this.raf.read(nameBytes);
            this.setParameter(new String(nameBytes, STRING_ENCODING), this.raf.readInt());
        }
        this.buildFreeIndexList(firstFreeBufferIndex);
    }

    private void writeHeader() throws IOException {
        if (this.readOnly) {
            throw new IOException("File is read-only");
        }
        int prev = -1;
        for (int index : this.freeIndexes) {
            this.putFreeBlock(index, prev);
            prev = index;
        }
        this.seekBlock(0, 0);
        this.raf.writeLong(3400271784588160042L);
        this.raf.writeLong(this.fileId);
        this.raf.writeInt(1);
        this.raf.writeInt(this.blockSize);
        this.raf.writeInt(prev);
        String[] parmNames = this.getParameterNames();
        this.raf.writeInt(parmNames.length);
        int cnt = 32;
        for (String parmName : parmNames) {
            byte[] nameBytes = parmName.getBytes(STRING_ENCODING);
            if ((cnt += 8 + nameBytes.length) > this.bufferSize) {
                throw new IOException("Buffer size too small");
            }
            this.raf.writeInt(nameBytes.length);
            this.raf.write(nameBytes);
            this.raf.writeInt(this.getParameter(parmName));
        }
    }

    private void buildFreeIndexList(int firstFreeBufferIndex) throws IOException {
        ArrayList<Integer> freeIndexList = new ArrayList<Integer>();
        int nextIndex = firstFreeBufferIndex;
        while (nextIndex >= 0) {
            freeIndexList.add(nextIndex);
            this.seekBufferBlock(nextIndex);
            byte flags = this.raf.readByte();
            if ((flags & 1) == 0) {
                throw new IOException("Corrupt file");
            }
            nextIndex = this.raf.readInt();
        }
        int[] newFreeIndexes = new int[freeIndexList.size()];
        for (int i = 0; i < newFreeIndexes.length; ++i) {
            newFreeIndexes[i] = (Integer)freeIndexList.get(i);
        }
        this.setFreeIndexes(newFreeIndexes);
    }

    void putFreeBlock(int index, int nextFreeIndex) throws IOException {
        if (index > this.bufferCount) {
            throw new EOFException("Free buffer index too large (" + index + " > " + this.bufferCount + ")");
        }
        if (index == this.bufferCount) {
            ++this.bufferCount;
        }
        this.seekBufferBlock(index);
        this.raf.writeByte(1);
        this.raf.writeInt(nextFreeIndex);
    }

    public static DataBuffer getDataBuffer(BufferFileBlock block) {
        int blockIndex = block.getIndex();
        if (blockIndex <= 0) {
            return null;
        }
        byte[] blockData = block.getData();
        byte flags = blockData[0];
        DataBuffer buf = new DataBuffer();
        if (flags == 1) {
            buf.setId(-1);
            buf.setEmpty(true);
            return buf;
        }
        int bufferId = BigEndianDataConverter.INSTANCE.getInt(blockData, 1);
        buf.setId(bufferId);
        byte[] bufData = new byte[blockData.length - 5];
        System.arraycopy(blockData, 5, bufData, 0, bufData.length);
        buf.setData(bufData);
        return buf;
    }

    public static BufferFileBlock getBufferFileBlock(DataBuffer buf, int bufferSize) {
        byte[] data = buf.data;
        boolean empty = buf.isEmpty();
        if (!empty && data.length != bufferSize) {
            throw new IllegalArgumentException("Bad buffer size");
        }
        int blockIndex = buf.getId() + 1;
        byte[] bytes = new byte[bufferSize + 5];
        bytes[0] = empty ? (byte)1 : 0;
        BigEndianDataConverter.INSTANCE.putInt(bytes, 1, buf.getId());
        if (!empty) {
            System.arraycopy(data, 0, bytes, 5, data.length);
        }
        return new BufferFileBlock(blockIndex, bytes);
    }

    @Override
    public synchronized DataBuffer get(DataBuffer buf, int index) throws IOException {
        if (index > this.bufferCount) {
            throw new EOFException("Buffer index too large (" + index + " > " + this.bufferCount + ")");
        }
        if (this.raf == null) {
            throw new ClosedException();
        }
        this.seekBufferBlock(index);
        byte flags = this.raf.readByte();
        buf.setId(this.raf.readInt());
        if ((flags & 1) != 0) {
            buf.setEmpty(true);
            buf.setId(-1);
        } else {
            buf.setEmpty(false);
            byte[] data = buf.data;
            if (data == null) {
                buf.data = data = new byte[this.bufferSize];
            } else if (data.length != this.bufferSize) {
                throw new IllegalArgumentException("Bad buffer size");
            }
            this.raf.readFully(data);
        }
        buf.setDirty(false);
        return buf;
    }

    @Override
    public synchronized void put(DataBuffer buf, int index) throws IOException {
        if (this.readOnly) {
            throw new IOException("File is read-only");
        }
        if (this.raf == null) {
            throw new ClosedException();
        }
        if (index > 0x7FFFFFFE) {
            throw new EOFException("Buffer index too large, exceeds max-int");
        }
        byte[] data = buf.data;
        boolean empty = buf.isEmpty();
        if (!empty && data.length != this.bufferSize) {
            throw new IllegalArgumentException("Bad buffer size");
        }
        this.seekBufferBlock(index);
        if (empty) {
            this.raf.writeByte(1);
            this.raf.writeInt(buf.getId());
        } else {
            this.raf.writeByte(0);
            this.raf.writeInt(buf.getId());
            this.raf.write(data, 0, this.bufferSize);
        }
        if (index >= this.bufferCount) {
            this.bufferCount = index + 1;
        }
    }

    @Override
    public int getBufferSize() {
        return this.bufferSize;
    }

    @Override
    public int getIndexCount() {
        return this.bufferCount;
    }

    void truncate(int indexCount) throws IOException {
        if (this.readOnly) {
            throw new IOException("File is read-only");
        }
        long size = (indexCount + 1) * this.blockSize;
        this.raf.setLength(size);
        this.bufferCount = indexCount;
    }

    boolean flush() throws IOException {
        if (this.raf == null || this.readOnly || this.temporary) {
            return false;
        }
        this.writeHeader();
        long len = this.raf.length();
        long d = len % (long)this.blockSize;
        if (d != 0L) {
            this.raf.setLength(len - d + (long)this.blockSize);
        }
        try {
            this.raf.getFD().sync();
        }
        catch (SyncFailedException syncFailedException) {
            // empty catch block
        }
        return true;
    }

    @Override
    public void dispose() {
        try {
            if (!this.readOnly) {
                this.delete();
            } else {
                this.close();
            }
        }
        catch (Throwable t) {
            Msg.error((Object)this, (Object)t);
        }
        finally {
            if (this.activeOutputBlockStream != null) {
                try {
                    this.activeOutputBlockStream.close();
                }
                catch (IOException iOException) {}
            }
            this.activeOutputBlockStream = null;
        }
    }

    @Override
    public synchronized boolean setReadOnly() throws IOException {
        if (!this.flush()) {
            return false;
        }
        this.raf.close();
        this.raf = new RandomAccessFile(this.file, "r");
        this.readOnly = true;
        return true;
    }

    boolean isTemporary() {
        return this.temporary;
    }

    boolean isClosed() {
        return this.raf == null;
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.raf == null) {
            return;
        }
        boolean commit = false;
        try {
            if (this.activeOutputBlockStream != null) {
                try {
                    this.activeOutputBlockStream.close();
                }
                catch (IOException iOException) {}
            } else {
                commit = this.flush();
            }
            this.raf.close();
        }
        finally {
            this.raf = null;
            if (!this.readOnly && !commit) {
                this.file.delete();
                if (this.activeOutputBlockStream != null) {
                    this.activeOutputBlockStream = null;
                    throw new IOException("active block stream was not closed properly");
                }
            }
        }
    }

    @Override
    public synchronized boolean delete() {
        if (this.raf == null || this.readOnly) {
            return false;
        }
        boolean success = false;
        try {
            try {
                this.raf.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            this.raf = null;
        }
        finally {
            success = this.file.delete();
        }
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clone(File destinationFile, TaskMonitor monitor) throws IOException, CancelledException {
        LocalBufferFile destBf = new LocalBufferFile(destinationFile, this.bufferSize);
        boolean success = false;
        try {
            LocalBufferFile.copyFile(this, destBf, null, monitor);
            destBf.setFileId(this.fileId);
            destBf.close();
            success = true;
        }
        finally {
            if (!success) {
                destBf.delete();
            }
        }
    }

    public String toString() {
        return this.file.toString();
    }

    public InputBlockStream getInputBlockStream() throws IOException {
        return inputBlockStreamFactory.createInputBlockStream(this);
    }

    public OutputBlockStream getOutputBlockStream(int blockCount) throws IOException {
        return new LocalOutputBlockStream(blockCount);
    }

    private static InputBlockStream getInputBlockStream(BufferFile bufferFile) throws IOException {
        if (bufferFile instanceof BufferFileAdapter) {
            return ((BufferFileAdapter)bufferFile).getInputBlockStream();
        }
        if (bufferFile instanceof LocalBufferFile) {
            return ((LocalBufferFile)bufferFile).getInputBlockStream();
        }
        throw new IllegalArgumentException("Unsupported buffer file implementation: " + bufferFile.getClass().getName());
    }

    private static InputBlockStream getInputBlockStream(BufferFile bufferFile, ChangeMap changeMap) throws IOException {
        if (changeMap == null) {
            return LocalBufferFile.getInputBlockStream(bufferFile);
        }
        if (bufferFile instanceof ManagedBufferFileAdapter) {
            return ((ManagedBufferFileAdapter)bufferFile).getInputBlockStream(changeMap.getData());
        }
        if (bufferFile instanceof LocalManagedBufferFile) {
            return ((LocalManagedBufferFile)bufferFile).getInputBlockStream(changeMap.getData());
        }
        throw new IllegalArgumentException("Unsupported buffer file implementation: " + bufferFile.getClass().getName());
    }

    static OutputBlockStream getOutputBlockStream(BufferFile bufferFile, int blockCount) throws IOException {
        if (bufferFile instanceof BufferFileAdapter) {
            return ((BufferFileAdapter)bufferFile).getOutputBlockStream(blockCount);
        }
        if (bufferFile instanceof LocalBufferFile) {
            return ((LocalBufferFile)bufferFile).getOutputBlockStream(blockCount);
        }
        throw new IllegalArgumentException("Unsupported buffer file implementation: " + bufferFile.getClass().getName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void copyFile(BufferFile srcFile, BufferFile destFile, ChangeMap changeMap, TaskMonitor monitor) throws IOException, CancelledException {
        int srcBlockCnt;
        boolean headerTransferRequired;
        if (destFile.isReadOnly()) {
            throw new IOException("File is read-only");
        }
        if (srcFile.getBufferSize() != destFile.getBufferSize()) {
            throw new IOException("Buffer sizes differ");
        }
        monitor = TaskMonitor.dummyIfNull((TaskMonitor)monitor);
        try (InputBlockStream in = LocalBufferFile.getInputBlockStream(srcFile, changeMap);){
            headerTransferRequired = !in.includesHeaderBlock();
            srcBlockCnt = in.getBlockCount();
            monitor.initialize((long)(srcBlockCnt + 2));
            try (OutputBlockStream out = LocalBufferFile.getOutputBlockStream(destFile, in.getBlockCount());){
                LocalBufferFile.completeBlockStreamTransfer(in, out, monitor);
            }
        }
        finally {
            monitor.checkCancelled();
        }
        if (headerTransferRequired) {
            String[] parmNames;
            destFile.clearParameters();
            for (String name : parmNames = srcFile.getParameterNames()) {
                destFile.setParameter(name, srcFile.getParameter(name));
            }
            monitor.setProgress((long)(srcBlockCnt + 1));
            destFile.setFreeIndexes(srcFile.getFreeIndexes());
        }
        monitor.setProgress((long)(srcBlockCnt + 2));
    }

    static void completeBlockStreamTransfer(InputBlockStream in, OutputBlockStream out, TaskMonitor monitor) throws CancelledException, IOException {
        int count = 0;
        try (BlockStreamCancelMonitor cancelMonitor = new BlockStreamCancelMonitor(monitor, in, out);){
            BufferFileBlock block;
            int srcBlockCnt = in.getBlockCount();
            while ((block = in.readBlock()) != null) {
                monitor.checkCancelled();
                out.writeBlock(block);
                monitor.setProgress((long)count++);
            }
            if (count != srcBlockCnt) {
                throw new IOException("unexpected block transfer count");
            }
        }
    }

    public static void cleanupOldPreSaveFiles(File dir, long beforeNow) {
        File[] oldFiles = dir.listFiles(new BufferFileFilter(null, PRESAVE_FILE_EXT));
        if (oldFiles == null) {
            return;
        }
        for (File oldFile : oldFiles) {
            if (beforeNow != 0L && oldFile.lastModified() >= beforeNow || !oldFile.delete()) continue;
            Msg.info(LocalBufferFile.class, (Object)("Removed old presave file: " + oldFile));
        }
    }

    static int getRecommendedBufferSize(int requestedBufferSize) {
        int size = requestedBufferSize + 5 & 0xFFFFFF80;
        if (size <= 0) {
            size = 128;
        }
        return size - 5;
    }

    class LocalOutputBlockStream
    implements OutputBlockStream {
        private final long orignalFileId;
        private final int blockCount;
        private boolean isClosed;
        private boolean refreshOnClose;

        public LocalOutputBlockStream(int blockCount) throws IOException {
            this.orignalFileId = LocalBufferFile.this.fileId;
            this.isClosed = false;
            if (LocalBufferFile.this.readOnly) {
                throw new IOException("Write stream only permitted on updateable buffer file");
            }
            if (LocalBufferFile.this.activeOutputBlockStream != null) {
                throw new IOException("Active block stream already exists");
            }
            this.blockCount = blockCount;
            LocalBufferFile.this.activeOutputBlockStream = this;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            LocalBufferFile localBufferFile = LocalBufferFile.this;
            synchronized (localBufferFile) {
                if (this.isClosed) {
                    return;
                }
                this.isClosed = true;
                if (LocalBufferFile.this.raf == null) {
                    throw new ClosedException();
                }
                try {
                    if (this.refreshOnClose) {
                        LocalBufferFile.this.readHeader();
                        LocalBufferFile.this.fileId = this.orignalFileId;
                        LocalBufferFile.this.seekBlock(0, 8);
                        LocalBufferFile.this.raf.writeLong(LocalBufferFile.this.fileId);
                    }
                }
                finally {
                    LocalBufferFile.this.activeOutputBlockStream = null;
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void writeBlock(BufferFileBlock block) throws IOException {
            if (LocalBufferFile.this.blockSize != block.size()) {
                throw new IOException("incompatible block size");
            }
            LocalBufferFile localBufferFile = LocalBufferFile.this;
            synchronized (localBufferFile) {
                if (LocalBufferFile.this.raf == null) {
                    throw new ClosedException();
                }
                int blockIndex = block.getIndex();
                if (blockIndex == 0) {
                    this.refreshOnClose = true;
                }
                LocalBufferFile.this.seekBlock(blockIndex, 0);
                LocalBufferFile.this.raf.write(block.getData());
                if (blockIndex > LocalBufferFile.this.bufferCount) {
                    LocalBufferFile.this.bufferCount = blockIndex;
                }
            }
        }

        @Override
        public int getBlockCount() {
            return this.blockCount;
        }

        @Override
        public int getBlockSize() {
            return LocalBufferFile.this.blockSize;
        }
    }

    public static interface InputBlockStreamFactory {
        public InputBlockStream createInputBlockStream(LocalBufferFile var1) throws IOException;
    }

    private static class BlockStreamCancelMonitor
    implements Closeable,
    CancelledListener {
        private TaskMonitor monitor;
        private BlockStream[] blockStreams;

        BlockStreamCancelMonitor(TaskMonitor monitor, BlockStream ... blockStreams) {
            this.monitor = monitor;
            this.blockStreams = blockStreams;
            monitor.addCancelledListener((CancelledListener)this);
        }

        @Override
        public void close() throws IOException {
            this.monitor.removeCancelledListener((CancelledListener)this);
        }

        public void cancelled() {
            for (BlockStream blockStream : this.blockStreams) {
                try {
                    blockStream.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }
    }

    static class BufferFileFilter
    implements FileFilter {
        private final String ext;
        private final String prefix;

        BufferFileFilter(String prefix, String ext) {
            this.prefix = prefix;
            this.ext = ext;
        }

        @Override
        public boolean accept(File file) {
            if (file.isFile()) {
                String name = file.getName();
                if (!(this.prefix != null && name.indexOf(this.prefix) != 0 || this.ext != null && !name.endsWith(this.ext))) {
                    return true;
                }
            }
            return false;
        }
    }

    private static class DefaultInputBlockStreamFactory
    implements InputBlockStreamFactory {
        private DefaultInputBlockStreamFactory() {
        }

        @Override
        public InputBlockStream createInputBlockStream(LocalBufferFile bf) throws IOException {
            if (!bf.readOnly) {
                throw new IOException("Read stream only permitted on read-only buffer file");
            }
            return bf.new LocalFileInputBlockStream();
        }
    }

    class LocalRandomInputBlockStream
    implements InputBlockStream {
        private List<Integer> bufferIndexList;
        private int nextIndex;

        LocalRandomInputBlockStream(byte[] changeMapData) throws IOException {
            if (!LocalBufferFile.this.readOnly) {
                throw new IOException("Read stream only permitted on read-only buffer file");
            }
            this.buildBufferIndexList(changeMapData);
        }

        private void buildBufferIndexList(byte[] changeMapData) {
            ChangeMap changeMap = new ChangeMap(changeMapData);
            this.bufferIndexList = new ArrayList<Integer>();
            IntSet emptySet = new IntSet(LocalBufferFile.this.getFreeIndexes());
            for (int bufferIndex = 0; bufferIndex < LocalBufferFile.this.bufferCount; ++bufferIndex) {
                if (emptySet.contains(bufferIndex) || !changeMap.hasChanged(bufferIndex) && changeMap.containsIndex(bufferIndex)) continue;
                this.bufferIndexList.add(bufferIndex);
            }
        }

        @Override
        public boolean includesHeaderBlock() {
            return false;
        }

        @Override
        public int getBlockCount() {
            return this.bufferIndexList != null ? this.bufferIndexList.size() : LocalBufferFile.this.bufferCount;
        }

        @Override
        public int getBlockSize() {
            return LocalBufferFile.this.blockSize;
        }

        @Override
        public void close() throws IOException {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public BufferFileBlock readBlock() throws IOException {
            LocalBufferFile localBufferFile = LocalBufferFile.this;
            synchronized (localBufferFile) {
                if (LocalBufferFile.this.raf == null) {
                    throw new ClosedException();
                }
                if (this.nextIndex == this.bufferIndexList.size()) {
                    return null;
                }
                int blockIndex = LocalBufferFile.this.seekBufferBlock(this.bufferIndexList.get(this.nextIndex++));
                byte[] block = new byte[LocalBufferFile.this.blockSize];
                LocalBufferFile.this.raf.readFully(block);
                return new BufferFileBlock(blockIndex, block);
            }
        }
    }

    private class LocalFileInputBlockStream
    implements InputBlockStream {
        private InputStream fin;
        private int blockCount;
        private int nextIndex = 0;

        LocalFileInputBlockStream() throws IOException {
            if (!LocalBufferFile.this.readOnly) {
                throw new IOException("Read stream only permitted on read-only buffer file");
            }
            long fileLength = LocalBufferFile.this.file.length();
            if (fileLength <= 0L || fileLength % (long)LocalBufferFile.this.blockSize != 0L) {
                throw new IOException("Corrupt file");
            }
            this.blockCount = (int)(fileLength / (long)LocalBufferFile.this.blockSize);
            this.fin = new BufferedInputStream(new FileInputStream(LocalBufferFile.this.file));
        }

        @Override
        public boolean includesHeaderBlock() {
            return true;
        }

        @Override
        public int getBlockCount() {
            return this.blockCount;
        }

        @Override
        public int getBlockSize() {
            return LocalBufferFile.this.blockSize;
        }

        @Override
        public void close() throws IOException {
            this.fin.close();
        }

        @Override
        public BufferFileBlock readBlock() throws IOException {
            int readlen;
            byte[] data = new byte[LocalBufferFile.this.blockSize];
            for (int total = 0; total < LocalBufferFile.this.blockSize; total += readlen) {
                readlen = this.fin.read(data, total, LocalBufferFile.this.blockSize - total);
                if (readlen >= 0) continue;
                if (total == 0 && this.nextIndex == this.blockCount) {
                    return null;
                }
                throw new EOFException("unexpected end of file");
            }
            return new BufferFileBlock(this.nextIndex++, data);
        }
    }

    class LocalBufferInputBlockStream
    implements InputBlockStream {
        private int nextBufferIndex = 0;
        private DataBuffer buf;

        LocalBufferInputBlockStream() throws IOException {
            if (!LocalBufferFile.this.isReadOnly()) {
                throw new IOException("Read stream only permitted on read-only buffer file");
            }
            this.buf = new DataBuffer();
        }

        @Override
        public boolean includesHeaderBlock() {
            return false;
        }

        @Override
        public int getBlockCount() {
            return LocalBufferFile.this.getBufferCount();
        }

        @Override
        public int getBlockSize() {
            return LocalBufferFile.this.blockSize;
        }

        @Override
        public void close() throws IOException {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public BufferFileBlock readBlock() throws IOException {
            LocalBufferFile localBufferFile = LocalBufferFile.this;
            synchronized (localBufferFile) {
                if (this.nextBufferIndex == LocalBufferFile.this.getBufferCount()) {
                    return null;
                }
                LocalBufferFile.this.get(this.buf, this.nextBufferIndex);
                int blockIndex = this.nextBufferIndex++ + 1;
                byte[] block = new byte[this.getBlockSize()];
                if (this.buf.isEmpty()) {
                    block[0] = 1;
                }
                BigEndianDataConverter.INSTANCE.putInt(block, 1, this.buf.getId());
                if (!this.buf.isEmpty()) {
                    byte[] data = this.buf.getData();
                    System.arraycopy(data, 0, block, 5, data.length);
                }
                return new BufferFileBlock(blockIndex, block);
            }
        }
    }
}

