/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.debug.service.emulation;

import db.Transaction;
import ghidra.app.plugin.core.debug.service.emulation.EmulatorOutOfMemoryException;
import ghidra.app.plugin.core.debug.service.modules.DebuggerStaticMappingUtils;
import ghidra.dbg.target.TargetMemoryRegion;
import ghidra.dbg.target.TargetObject;
import ghidra.dbg.target.TargetThread;
import ghidra.dbg.target.schema.TargetObjectSchema;
import ghidra.dbg.util.PathMatcher;
import ghidra.dbg.util.PathPattern;
import ghidra.dbg.util.PathPredicates;
import ghidra.dbg.util.PathUtils;
import ghidra.framework.model.DomainFile;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.AddressRange;
import ghidra.program.model.address.AddressRangeImpl;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.lang.CompilerSpec;
import ghidra.program.model.lang.LanguageNotFoundException;
import ghidra.program.model.lang.Register;
import ghidra.program.model.lang.RegisterValue;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.ProgramContext;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.util.ProgramLocation;
import ghidra.trace.database.DBTrace;
import ghidra.trace.model.DefaultTraceLocation;
import ghidra.trace.model.Lifespan;
import ghidra.trace.model.Trace;
import ghidra.trace.model.TraceLocation;
import ghidra.trace.model.memory.TraceMemoryFlag;
import ghidra.trace.model.memory.TraceMemoryManager;
import ghidra.trace.model.memory.TraceMemorySpace;
import ghidra.trace.model.memory.TraceOverlappedRegionException;
import ghidra.trace.model.modules.TraceConflictedMappingException;
import ghidra.trace.model.target.TraceObject;
import ghidra.trace.model.target.TraceObjectKeyPath;
import ghidra.trace.model.thread.TraceObjectThread;
import ghidra.trace.model.thread.TraceThread;
import ghidra.trace.model.thread.TraceThreadManager;
import ghidra.trace.model.time.TraceSnapshot;
import ghidra.util.DifferenceAddressSetView;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.DuplicateNameException;
import java.io.IOException;
import java.math.BigInteger;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ProgramEmulationUtils {
    public static final String BLOCK_NAME_STACK = "STACK";
    public static final String EMULATION_STARTED_AT = "Emulation started at ";

    private ProgramEmulationUtils() {
    }

    public static String getTraceName(Program program) {
        DomainFile df = program.getDomainFile();
        if (df != null) {
            return "Emulate " + df.getName();
        }
        return "Emulate " + program.getName();
    }

    public static String getModuleName(Program program) {
        String executablePath = program.getExecutablePath();
        if (executablePath != null) {
            return executablePath;
        }
        DomainFile df = program.getDomainFile();
        if (df != null) {
            return df.getName();
        }
        return program.getName();
    }

    public static Set<TraceMemoryFlag> getRegionFlags(MemoryBlock block) {
        EnumSet<TraceMemoryFlag> result = EnumSet.noneOf(TraceMemoryFlag.class);
        int mask = block.getPermissions();
        if ((mask & 4) != 0) {
            result.add(TraceMemoryFlag.READ);
        }
        if ((mask & 2) != 0) {
            result.add(TraceMemoryFlag.WRITE);
        }
        if ((mask & 1) != 0) {
            result.add(TraceMemoryFlag.EXECUTE);
        }
        if ((mask & 8) != 0) {
            result.add(TraceMemoryFlag.VOLATILE);
        }
        return result;
    }

    public static void loadExecutable(TraceSnapshot snapshot, Program program, List<AddressSpace> activeOverlays) {
        Trace trace = snapshot.getTrace();
        PathPattern patRegion = ProgramEmulationUtils.computePatternRegion(trace);
        HashMap<AddressSpace, DebuggerStaticMappingUtils.Extrema> extremaBySpace = new HashMap<AddressSpace, DebuggerStaticMappingUtils.Extrema>();
        Lifespan nowOn = Lifespan.nowOn((long)snapshot.getKey());
        try {
            for (MemoryBlock block : program.getMemory().getBlocks()) {
                if (!DebuggerStaticMappingUtils.isReal(block)) continue;
                AddressRangeImpl range = new AddressRangeImpl(block.getStart(), block.getEnd());
                extremaBySpace.computeIfAbsent(range.getAddressSpace(), s -> new DebuggerStaticMappingUtils.Extrema()).consider((AddressRange)range);
                String modName = ProgramEmulationUtils.getModuleName(program);
                String path = PathUtils.toString((List)patRegion.applyKeys(new String[]{block.getStart() + "-" + modName + ":" + block.getName()}).getSingletonPath());
                trace.getMemoryManager().createRegion(path, snapshot.getKey(), (AddressRange)range, ProgramEmulationUtils.getRegionFlags(block));
            }
            AddressSet identical = new AddressSet();
            for (DebuggerStaticMappingUtils.Extrema extrema : extremaBySpace.values()) {
                identical.add(extrema.getMin(), extrema.getMax());
            }
            for (MemoryBlock block : program.getMemory().getBlocks()) {
                if (!block.isOverlay() || !activeOverlays.contains(block.getStart().getAddressSpace())) continue;
                Address phys = block.getStart().getPhysicalAddress();
                DebuggerStaticMappingUtils.addMapping((TraceLocation)new DefaultTraceLocation(trace, null, nowOn, phys), new ProgramLocation(program, block.getStart()), block.getSize(), false);
                identical.delete(phys, block.getEnd().getPhysicalAddress());
            }
            for (AddressRange range : identical) {
                DebuggerStaticMappingUtils.addMapping((TraceLocation)new DefaultTraceLocation(trace, null, nowOn, range.getMinAddress()), new ProgramLocation(program, range.getMinAddress()), range.getLength(), false);
            }
        }
        catch (TraceOverlappedRegionException | TraceConflictedMappingException | DuplicateNameException e) {
            throw new AssertionError((Object)e);
        }
    }

    public static PathPattern computePattern(Trace trace, Class<? extends TargetObject> iface) {
        TargetObjectSchema root = trace.getObjectManager().getRootSchema();
        if (root == null) {
            return new PathPattern(PathUtils.parse((String)"Memory[]"));
        }
        PathMatcher matcher = root.searchFor(iface, true);
        PathPattern pattern = matcher.getSingletonPattern();
        if (pattern == null || pattern.countWildcards() != 1) {
            throw new IllegalArgumentException("Cannot find unique " + iface.getSimpleName() + " container");
        }
        return pattern;
    }

    public static PathPattern computePatternRegion(Trace trace) {
        return ProgramEmulationUtils.computePattern(trace, TargetMemoryRegion.class);
    }

    public static PathPattern computePatternThread(Trace trace) {
        return ProgramEmulationUtils.computePattern(trace, TargetThread.class);
    }

    public static TraceThread spawnThread(Trace trace, long snap) {
        String path;
        TraceThreadManager tm = trace.getThreadManager();
        PathPattern patThread = ProgramEmulationUtils.computePatternThread(trace);
        long next = tm.getAllThreads().size();
        while (!tm.getThreadsByPath(path = PathUtils.toString((List)patThread.applyKeys(new String[]{Long.toString(next)}).getSingletonPath())).isEmpty()) {
            ++next;
        }
        try {
            return tm.createThread(path, "[" + next + "]", snap);
        }
        catch (DuplicateNameException e) {
            throw new AssertionError((Object)e);
        }
    }

    public static void initializeRegisters(Trace trace, long snap, TraceThread thread, Program program, Address tracePc, Address programPc, AddressRange stack) {
        Register regPC;
        TraceMemoryManager memory = trace.getMemoryManager();
        if (thread instanceof TraceObjectThread) {
            TraceObjectThread ot = (TraceObjectThread)thread;
            TraceObject object = ot.getObject();
            PathPredicates regsMatcher = object.getRoot().getTargetSchema().searchForRegisterContainer(0, object.getCanonicalPath().getKeyList());
            if (regsMatcher.isEmpty()) {
                throw new IllegalArgumentException("Cannot create register container");
            }
            Iterator iterator = regsMatcher.getPatterns().iterator();
            if (iterator.hasNext()) {
                PathPattern regsPattern = (PathPattern)iterator.next();
                trace.getObjectManager().createObject(TraceObjectKeyPath.of((List)regsPattern.getSingletonPath()));
            }
        }
        TraceMemorySpace regSpace = memory.getMemoryRegisterSpace(thread, true);
        if (program != null) {
            ProgramContext ctx = program.getProgramContext();
            for (Register reg : Stream.of(ctx.getRegistersWithValues()).map(Register::getBaseRegister).collect(Collectors.toSet())) {
                RegisterValue rv = ctx.getRegisterValue(reg, programPc);
                if (rv == null || !rv.hasAnyValue()) continue;
                TraceMemorySpace space = reg.getAddressSpace().isRegisterSpace() ? regSpace : memory;
                space.setValue(snap, new RegisterValue(reg, BigInteger.ZERO).combineValues(rv));
            }
        }
        TraceMemorySpace spacePC = (regPC = trace.getBaseLanguage().getProgramCounter()).getAddressSpace().isRegisterSpace() ? regSpace : memory;
        spacePC.setValue(snap, new RegisterValue(regPC, NumericUtilities.unsignedLongToBigInteger((long)tracePc.getAddressableWordOffset())));
        if (stack != null) {
            CompilerSpec cSpec = trace.getBaseCompilerSpec();
            Address sp = cSpec.stackGrowsNegative() ? stack.getMaxAddress().addWrap(1L) : stack.getMinAddress();
            Register regSP = cSpec.getStackPointer();
            if (regSP != null) {
                TraceMemorySpace spaceSP = regSP.getAddressSpace().isRegisterSpace() ? regSpace : memory;
                spaceSP.setValue(snap, new RegisterValue(regSP, NumericUtilities.unsignedLongToBigInteger((long)sp.getAddressableWordOffset())));
            }
        }
    }

    public static AddressRange allocateStackCustom(Trace trace, long snap, TraceThread thread, Program program) {
        if (program == null) {
            return null;
        }
        AddressSpace space = trace.getBaseCompilerSpec().getStackBaseSpace();
        MemoryBlock stackBlock = program.getMemory().getBlock(BLOCK_NAME_STACK);
        if (stackBlock == null) {
            return null;
        }
        if (space != stackBlock.getStart().getAddressSpace().getPhysicalSpace()) {
            Msg.showError(ProgramEmulationUtils.class, null, (String)"Invalid STACK block", (Object)"The STACK block must be in the stack's base space. Ignoring.");
            return null;
        }
        AddressRangeImpl alloc = new AddressRangeImpl(stackBlock.getStart().getPhysicalAddress(), stackBlock.getEnd().getPhysicalAddress());
        if (stackBlock.isOverlay() || DebuggerStaticMappingUtils.isReal(stackBlock)) {
            return alloc;
        }
        PathPattern patRegion = ProgramEmulationUtils.computePatternRegion(trace);
        String path = PathUtils.toString((List)patRegion.applyKeys(new String[]{stackBlock.getStart() + "-STACK"}).getSingletonPath());
        TraceMemoryManager mm = trace.getMemoryManager();
        try {
            return mm.createRegion(path, snap, (AddressRange)alloc, new TraceMemoryFlag[]{TraceMemoryFlag.READ, TraceMemoryFlag.WRITE}).getRange();
        }
        catch (TraceOverlappedRegionException e) {
            Msg.showError(ProgramEmulationUtils.class, null, (String)"Stack conflict", (Object)"The STACK region %s conflicts with another: %s. You may need to initialize the stack pointer manually.".formatted(alloc, e.getConflicts().iterator().next()));
            return alloc;
        }
        catch (DuplicateNameException e) {
            Msg.showError(ProgramEmulationUtils.class, null, (String)"Stack conflict", (Object)"A region already exists with the same name: %s. You may need to initialize the stack pointer manually.".formatted(path));
            return alloc;
        }
    }

    public static AddressRange allocateStack(Trace trace, long snap, TraceThread thread, Program program, long size) {
        AddressRange custom = ProgramEmulationUtils.allocateStackCustom(trace, snap, thread, program);
        if (custom != null) {
            return custom;
        }
        AddressSpace space = trace.getBaseCompilerSpec().getStackBaseSpace();
        Address max = space.getMaxAddress();
        AddressSet eligible = max.getOffsetAsBigInteger().compareTo(BigInteger.valueOf(4096L)) < 0 ? new AddressSet(space.getMinAddress(), max) : new AddressSet(space.getAddress(4096L), max);
        TraceMemoryManager mm = trace.getMemoryManager();
        DifferenceAddressSetView left = new DifferenceAddressSetView((AddressSetView)eligible, mm.getRegionsAddressSet(snap));
        PathPattern patRegion = ProgramEmulationUtils.computePatternRegion(trace);
        try {
            for (AddressRange candidate : left) {
                if (Long.compareUnsigned(candidate.getLength(), size) < 0) continue;
                AddressRangeImpl alloc = new AddressRangeImpl(candidate.getMinAddress(), size);
                String threadName = PathUtils.isIndex((String)thread.getName()) ? PathUtils.parseIndex((String)thread.getName()) : thread.getName();
                String path = PathUtils.toString((List)patRegion.applyKeys(new String[]{alloc.getMinAddress() + "-stack " + threadName}).getSingletonPath());
                return mm.createRegion(path, snap, (AddressRange)alloc, new TraceMemoryFlag[]{TraceMemoryFlag.READ, TraceMemoryFlag.WRITE}).getRange();
            }
        }
        catch (AddressOverflowException | TraceOverlappedRegionException | DuplicateNameException e) {
            throw new AssertionError((Object)e);
        }
        throw new EmulatorOutOfMemoryException();
    }

    public static Trace launchEmulationTrace(Program program, Address pc, Object consumer) throws IOException {
        DBTrace trace = null;
        boolean success = false;
        try {
            trace = new DBTrace(ProgramEmulationUtils.getTraceName(program), program.getCompilerSpec(), consumer);
            try (Transaction tx = trace.openTransaction("Emulate");){
                TraceSnapshot initial = trace.getTimeManager().createSnapshot(EMULATION_STARTED_AT + pc);
                long snap = initial.getKey();
                List<AddressSpace> overlays = pc.getAddressSpace().isOverlaySpace() ? List.of(pc.getAddressSpace()) : List.of();
                ProgramEmulationUtils.loadExecutable(initial, program, overlays);
                ProgramEmulationUtils.doLaunchEmulationThread((Trace)trace, snap, program, pc, pc);
            }
            trace.clearUndo();
            success = true;
            tx = trace;
            return tx;
        }
        catch (LanguageNotFoundException e) {
            throw new AssertionError((Object)e);
        }
        finally {
            if (!success && trace != null) {
                trace.release(consumer);
            }
        }
    }

    public static TraceThread doLaunchEmulationThread(Trace trace, long snap, Program program, Address tracePc, Address programPc) {
        AddressRange stack;
        TraceThread thread = ProgramEmulationUtils.spawnThread(trace, snap);
        try {
            stack = ProgramEmulationUtils.allocateStack(trace, snap, thread, program, 16384L);
        }
        catch (EmulatorOutOfMemoryException e) {
            Msg.warn(ProgramEmulationUtils.class, (Object)"Cannot allocate a stack. Please initialize manually.");
            stack = null;
        }
        ProgramEmulationUtils.initializeRegisters(trace, snap, thread, program, tracePc, programPc, stack);
        return thread;
    }

    public static TraceThread launchEmulationThread(Trace trace, long snap, Program program, Address tracePc, Address programPc) {
        try (Transaction tx = trace.openTransaction("Emulate new Thread");){
            TraceThread thread;
            TraceThread traceThread = thread = ProgramEmulationUtils.doLaunchEmulationThread(trace, snap, program, tracePc, programPc);
            return traceThread;
        }
    }

    public static boolean isEmulatedProgram(Trace trace) {
        TraceSnapshot snapshot = trace.getTimeManager().getSnapshot(0L, false);
        if (snapshot == null) {
            return false;
        }
        return snapshot.getDescription().startsWith(EMULATION_STARTED_AT);
    }
}

