/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gnome;
import org.gnu.glib.GObject;
import org.gnu.glib.Type;
import org.gnu.gtk.Container;
import org.gnu.gtk.MenuBar;
import org.gnu.gtk.StatusBar;
import org.gnu.gtk.ToolBar;
import org.gnu.gtk.VBox;
import org.gnu.gtk.Widget;
import org.gnu.gtk.Window;

/**
 * The App widget is the main window of a GNOME application. It is
 * a container widget that can hold a single child widget. It also
 * includes facilities for attaching menus, toolbars, a status bar, and
 * widgets that can be docked.
 */
public class App extends Window {

	/**
	 * Instinates a new Application window with the indicated name and title.
	 * @param appname The name of the application.
	 * @param title The title of the application window.  The title can be
	 *              null, in which case the window's title will not be set.
	 */
	public App(String appname, String title) {
		setHandle(gnome_app_new(appname, title == null ? null : title));
		initializeEventHandlers();
	}
	
	public App(int handle) {
		super(handle);
	}

	protected void noopInit() {
	}

	/**
	 * Returns the dock.
	 * @return The dock.
	 */
	public Widget getDock() {
		int hndl = getDock(getHandle());
		if (0 == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	// AccelGroup is not yet implemented
	//    /**
	//     * Main accelerator group for this window (hotkeys live here).
	//     * @return The accelorator group.
	//     */
	//    public AccelGroup getAccelGroup()
	//    {
	//	return new AccelGroup(getAccelGroup(handle));
	//    }

	/**
	 * Returns the menu bar.
	 * @return The MenuBar.
	 */
	public MenuBar getMenuBar() {
		int hndl = getMenubar(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (MenuBar)obj;
		return new MenuBar(hndl);
	}

	/**
	 * Gets the name of the application.
	 * @return The name of the application.
	 */
	public String getName() {
		return getName(getHandle());
	}

	/**
	 * Gets the prefix for gnome-config, which is used to save the layout.
	 * @return The prefix for gnome-config.
	 */
	public String getPrefix() {
		return getPrefix(getHandle());
	}

	/**
	 * Gets the status bar of the application windows.
	 * @return The status bar.
	 */
	public StatusBar getStatusBar() {
		int hndl = getStatusbar(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (StatusBar)obj;
		return new StatusBar(hndl);
	}

	/**
	 * <I>From gnome-app.h</I>The vbox widget that ties them.
	 * @return The VBox.
	 */
	public VBox getVBox() {
		int hndl = getVbox(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (VBox)obj;
		return new VBox(hndl);
	}

	/**
	 * If true, the application uses gnome-config to retrieve and save the
	 * docking configuration automatically.
	 * @return True if the docking configuration is to be handled
	 &         automatically, else false.
	 */
	public boolean layoutConfig() {
		return getEnableLayoutConfig(getHandle());
	}

	/**
	 * If true, the application uses gnome-config to retrieve and save the
	 * docking configuration automatically.
	 * @param enable True if the docking configuration is to be handled
	 *               automatically, else false.
	 */
	public void layoutConfig(boolean enabled) {
		gnome_app_enable_layout_config(getHandle(), enabled);
	}

	/**
	 * Sets the content area of the application window.
	 * @param contents The widget that contains the content of the window.
	 */
	public void setContent(Widget contents) {
		gnome_app_set_contents(getHandle(), contents.getHandle());
	}

	/**
	 * Sets the menu bar of the window.
	 * @param menuBar The menu bar to be used for the window.
	 */
	public void setMenuBar(MenuBar menuBar) {
		gnome_app_set_menus(getHandle(), menuBar.getHandle());
	}
	
	/**
	 * Construct a menu bar and attach it to the specified application window.
	 * @param uiinfos An array of UIInfo objects that define the menu.
	 */
	public void createMenus(UIInfo [] uiinfos) {
		UIInfo.makeNative( uiinfos );	
		if (uiinfos.length > 0)
			gnome_app_create_menus(getHandle(), uiinfos[0].getHandle());
	}

	/**
	 * Removes <i>num</i> items from the existing app's menu structure
	 * beginning with the item described by <i>path</i>.
	 * @param path The path to first item to remove.
	 * @param num The number of items to remove.
	 */
	public void removeMenus(String path, int num) {
		gnome_app_remove_menus(getHandle(), path, num);
	}
	
	/**
	 * Inserts a menu in the existing app's menu structure right after
	 * the item described by <i>path</i>.
	 * @param path The path to the item that preceeds the insertion.
	 * @param uiinfos An array of UIInfo objects that describe the menu
	 * to be inserted.
	 */
	public void insertMenu(String path, UIInfo [] uiinfos) {
		UIInfo.makeNative( uiinfos );
		if (uiinfos.length > 0)
			gnome_app_insert_menus(getHandle(), path, uiinfos[0].getHandle() );
	}
	
	/**
	 * Sets the tool bar of the window.
	 * @param toolBar The tool bar to be used for the window.
	 */
	public void setToolBar(ToolBar toolBar) {
		gnome_app_set_toolbar(getHandle(), toolBar.getHandle());
	}

	/**
	 * Construct a toolbar and attach it to the specified application window.
	 * @param uiinfos An array of UIInfo objects that define the toolbar.
	 */
	public void createToolBar(UIInfo [] uiinfos) {
		UIInfo.makeNative(uiinfos);
		if (uiinfos.length > 0)
			gnome_app_create_toolbar(getHandle(), uiinfos[0].getHandle() );
	}

	/**
	 * Sets the status bar of the application window.
	 * @param appBar The app bar to use for the window.
	 */
	public void setStatusBar(AppBar appBar) {
		gnome_app_set_statusbar(getHandle(), appBar.getHandle());
	}

	/**
	 * Sets the status bar of the application window.
	 * @param statusBar The status bar to use for the window.
	 */
	public void setStatusBar(StatusBar statusBar) {
		gnome_app_set_statusbar(getHandle(), statusBar.getHandle());
	}

	/**
	 * Sets the status bar of the application window, but uses the given
	 * container widget rather than creating a new one.
	 * @param appBar The app bar to use for the window.
	 * @param container The container for the status bar.
	 */
	public void setStatusBar(AppBar appBar, Container container) {
		gnome_app_set_statusbar_custom(getHandle(), container.getHandle(), appBar.getHandle());
	}

	/**
	 * Sets the status bar of the application window, but uses the given
	 * container widget rather than creating a new one.
	 * @param statusBar The status bar to use for the window.
	 * @param container The container for the status bar.
	 */
	public void setStatusBar(StatusBar statusBar, Container container) {
		gnome_app_set_statusbar_custom(getHandle(), container.getHandle(), statusBar.getHandle());
	}
	
	/**
	 * Activate the menu item hints, displaying in an appbar.
	 * @param appBar The AppBar to install the hints.
	 * @param uiinfos An array of UIInfo objects that contain the menu items
	 * for which the hints will be created.
	 */
//	public void installAppBarMenuHints(AppBar appBar, UIInfo [] uiinfos) {
//		int [] hndls= new int[uiinfos.length];
//		for (int i = 0; i < uiinfos.length; i++) {
//			hndls[i] = uiinfos[i].getHandle();
//		}
//		gnome_app_install_appbar_menu_hints(appBar.getHandle(), hndls);
//	}
	
	/**
	 * Activate the menu item hints, displaying in a statusbar.
	 * @param statusBar The StatusBar to install the hints.
	 * @param uiinfos An array of UIInfo objects that contain the menu items
	 * for which the hints will be created.
	 */
//	public void installStatusBarMenuHints(StatusBar statusBar, UIInfo [] uiinfos) {
//		int [] hndls= new int[uiinfos.length];
//		for (int i = 0; i < uiinfos.length; i++) {
//			hndls[i] = uiinfos[i].getHandle();
//		}
//		gnome_app_install_statusbar_menu_hints(statusBar.getHandle(), hndls);
//	}
	
	/**
	 * Activate the menu item hings, displaying in the statusbar or appbar.
	 * @param uiinfos An array of UIInfo objects that contain the menu items
	 * for which the hints will be created.
	 */
	public void installMenuHints(UIInfo [] uiinfos) {
		if (uiinfos.length > 0)
			gnome_app_install_menu_hints(getHandle(), uiinfos[0].getHandle());
	}

	/**
	 * Display a simple message in an OK dialog or the status bar.  Requires
	 * confirmation from the user before it goes away.
	 * @param message The text to display.
	 * @return The dialog created or null.
	 * @deprecated
	 */
	public Widget message(String message) {
		int hndl = gnome_app_message(getHandle(), message);
		if (0 == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}
	
	/**
	 * Flash a message in the status bar for a few moments.  If this app doesn't
	 * have a status bar this method will do nothing.
	 * @param flash The message to flash.
	 * @deprecated
	 */
	public void flash(String flash) {
		gnome_app_flash(getHandle(), flash);
	}
	
	/**
	 * Display a not-so-important error message in an OK dialog or the status bar.
	 * @param warning The text to display.
	 * @return The dialog created or null.
	 * @deprecated
	 */
	public Widget warning(String warning) {
		int hndl = gnome_app_warning(getHandle(), warning);
		if (0 == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	/**
	 * Display an important error message in an OK dialog or the status bar.
	 * @param error The text to display.
	 * @return The dialog created or null.
	 * @deprecated
	 */
	public Widget error(String error) {
		int hndl = gnome_app_error(getHandle(), error);
		if (0 == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gnome_app_get_type());
	}

	/****************************************
	 * BEGINNING OF GENERATED CODE
	 ****************************************/
	native static final protected String getName(int cptr);
	native static final protected String getPrefix(int cptr);
	native static final protected int getDock(int cptr);
	native static final protected int getStatusbar(int cptr);
	native static final protected int getVbox(int cptr);
	native static final protected int getMenubar(int cptr);
	native static final protected int getContents(int cptr);
	native static final protected int getAccelGroup(int cptr);
	native static final protected boolean getEnableLayoutConfig(int cptr);
	native static final protected int gnome_app_get_type();
	native static final protected int gnome_app_new(String appname, String title);
	native static final protected void gnome_app_set_menus(int app, int menubar);
	native static final protected void gnome_app_set_toolbar(int app, int toolbar);
	native static final protected void gnome_app_set_statusbar(int app, int statusbar);
	native static final protected void gnome_app_set_statusbar_custom(int app, int container, int statusbar);
	native static final protected void gnome_app_set_contents(int app, int contents);
	native static final protected void gnome_app_enable_layout_config(int app, boolean enable);
	native static final protected void gnome_app_ui_configure_configurable(int uiinfo);
	native static final protected void gnome_app_create_menus(int app, int uiinfo);
	native static final protected void gnome_app_create_menus_custom(int app, int uiinfo, int uibdate);
	native static final protected void gnome_app_create_toolbar(int app, int uiinfo);
	native static final protected void gnome_app_create_toolbar_custom(int app, int uiinfo, int uibdata);
	native static final protected int gnome_app_find_menu_pos(int parent, String path, int pos);
	native static final protected void gnome_app_remove_menus(int app, String path, int items);
	native static final protected void gnome_app_remove_menu_range(int app, String path, int start, int items);
	native static final protected void gnome_app_insert_menus_custom(int app, String path, int menuinfo, int uibdata);
	native static final protected void gnome_app_insert_menus(int app, String path, int menuinfo);
	native static final protected void gnome_app_install_appbar_menu_hints(int appbar, int uiinfo);
	native static final protected void gnome_app_install_statusbar_menu_hints(int bar, int uiinfo);
	native static final protected void gnome_app_install_menu_hints(int app, int uiinfo);
	native static final protected void gnome_app_fill_menu(int menuShell, int uiInfo, int accelGroup, boolean ulineAccels, int pos);
	native static final protected void gnome_app_fill_menu_with_data(int menuShell, int uiInfo, int accelGroup, boolean ulineAccels, int pos, int userData);
	native static final protected void gnome_app_fill_menu_custom(int menuShell, int uiInfo, int uiBData, int accelGroup, boolean ulineAccels, int pos);
	native static final protected void gnome_app_fill_toolbar(int toolbar, int uiInfo, int accelGroup);
	native static final protected void gnome_app_fill_toolbar_with_data(int toolbar, int uiInfo, int accelGroup, int userData);
	native static final protected void gnome_app_fill_toolbar_custom(int toolbar, int uiInfo, int uiBData, int accelGroup);
    native static final protected int gnome_app_message (int app, String message);
    native static final protected void gnome_app_flash (int app, String flash);
    native static final protected int gnome_app_error (int app, String error);
    native static final protected int gnome_app_warning (int app, String warning);
	/****************************************
	 * END OF GENERATED CODE
	 ****************************************/
}
