# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gobject
import gtk

from gazpacho import project, util, placeholder, widgetregistry
from gazpacho.loader import tags
import gazpacho.property
from gazpacho.popup import Popup
from gazpacho.choice import enum_to_string, flags_to_string

from xml.sax.saxutils import escape
import xml.dom

def get_widget_from_gtk_widget(gtk_widget):
    return gtk_widget.get_data('GladeWidgetDataTag')

def load_widget_from_gtk_widget(gtk_widget, project, widget_tree,
                                blacklist=None):
    """  This function creates a GWidget from a GtkWidget.
    It recursively creates any children of the original gtk_widget.
    This method is mainly used when loading a .glade file with
    gazpacho.loader

    Also makes sure the name of the widget is unique in the project
    parameter and if not, change it for a valid one. Blacklist is a list
    used to avoid repeating the names of the widgets that are being loaded
    """
    if blacklist is None:
        blacklist = []

    # gtk_widget may not have a name if it is an intermediate internal children
    # For example, the label of a button does not have a name
    if gtk_widget.name is None:
        # this is not an important widget but maybe its children are
        if isinstance(gtk_widget, gtk.Container):
            for child in gtk_widget.get_children():
                load_widget_from_gtk_widget(child, project, widget_tree,
                                            blacklist)
        return

    type_name = gobject.type_name(gtk_widget)
    klass = widgetregistry.widget_registry.get_by_name(type_name)
    if klass is None:
        print _('Warning: could not get the class from widget: %s') % \
              gtk_widget
        return

    if klass.load_function is not None:
        context = project.context
        return klass.load_function(context, gtk_widget, widget_tree, blacklist)

    widget = load_gwidget(gtk_widget, klass, project, blacklist)

    load_properties(widget)
    
    widget.name = gtk_widget.name
    
    # create the children
    if isinstance(gtk_widget, gtk.Container):
        for child in gtk_widget.get_children():
            load_child(widget, child, project, widget_tree, blacklist)

    load_signals(widget, widget_tree)

    # if the widget is a toplevel we need to attach the accel groups
    # of the application
    if widget.is_toplevel():
        setup_toplevel(widget)

    return widget

def load_gwidget(gtk_widget, klass, project, blacklist):
    """Helper function of load_widget_from_gtk_widget. Create the Gazpacho
    widget and create a right name for it
    """
    internal_name = None

    # check if the widget is an internal child
    internal_info = get_internal_info(gtk_widget)
    if internal_info is not None:
        internal_name, gparent = internal_info
        gtk_widget.set_property('name', '%s-%s' % (gparent.name,internal_name))
    else:
        # check that the name is unique
        new_name = resolve_name_collisions(gtk_widget.name, klass, project,
                                           blacklist)
        if new_name != gtk_widget.name:
            gtk_widget.set_property('name', new_name)
            blacklist.append(new_name)
        
    widget = Widget(klass, project)

    widget.internal_name = internal_name
    widget.setup_widget(gtk_widget)

    return widget

def load_properties(widget):
    """Helper function for load_widget_from_gtk_widget. Create and copy
    the properties from the gtk widget.
    """
    widget._create_properties()
    
    widget.sync_properties_from_gtk()

    # Get i18n metadata for translatable string properties
    for prop in widget.klass.properties:
        if prop.is_translatable and prop.type == gobject.TYPE_STRING:
            for p in widget.properties.values():
                if p.klass == prop:
                    p.read_i18n_data(widget.gtk_widget)
                    break

def load_child(gparent, child, project, widget_tree, blacklist):
    """Helper function for load_widget_from_gtk_widget. Load one child
    of gparent only if it is not a placeholder.
    """
    if isinstance(child, placeholder.Placeholder):
        return

    gchild = load_widget_from_gtk_widget(child, project, widget_tree, blacklist)
    # be aware that some loaders change the gtk_widget in loading time so
    # gchild.gtk_widget = child is not always true (see menubar loader for and
    # example)
    if gchild:
        gchild._set_packing_properties(gparent)
        gchild.sync_packing_properties_from_gtk(gparent.gtk_widget,
                                                gchild.gtk_widget)
    
def load_signals(widget, widget_tree):
    """Helper function for load_widget_from_gtk_widget. Load the signals for
    the widget.
    """
    # set the signals
    signals = widget_tree._signals
    widget_signals = [s[1:] for s in signals if s[0] == widget.gtk_widget]
    for signal_name, signal_handler, after in widget_signals:
        handler = {'name': signal_name, 'handler': signal_handler,
                   'after': after}
        widget.add_signal_handler(handler)
    
def resolve_name_collisions(name, klass, project, blacklist=[]):
    """Resolve potential name conflicts for the widget. If the name of
    the widget is already used in the project return a new valid name"""
    new_name = name
    if project.get_widget_by_name(name) is not None or name in blacklist:
        # we need to generate a new name
        new_name = project.new_widget_name(klass.generic_name, blacklist)

    return new_name

def get_internal_info(gtk_child):
    """Get the internal name and the parent for gtk_child.

    If gtk_child is not an internal child of a widget it returns None
    """
    ancestor = gtk_child.get_parent()
    while ancestor is not None:
        gwidget = get_widget_from_gtk_widget(ancestor)
        if gwidget:
            internal_name = gwidget.name_of_internal_child(gtk_child)
            if internal_name is not None:
                return (internal_name, gwidget)
        
        ancestor = ancestor.get_parent()
    
def setup_toplevel(gwidget):
    """Add the action groups of Gazpacho to this toplevel and
    also set this toplevel transient for the main window.
    """
    for ag in gwidget.project._app.get_accel_groups():
        gwidget.gtk_widget.add_accel_group(ag)

    # make window management easier by making created windows
    # transient for the editor window
    app_window = gwidget.project._app.get_window()
    gwidget.gtk_widget.set_transient_for(app_window)
    
def replace_widget(old_widget, new_widget, parent):
    gnew_widget = get_widget_from_gtk_widget(new_widget)
    gold_widget = old_widget and get_widget_from_gtk_widget(old_widget)

    gtk_new_widget = gnew_widget and gnew_widget.gtk_widget or new_widget
    gtk_old_widget = gold_widget and gold_widget.gtk_widget or old_widget

    if parent is None:
        parent = util.get_parent(old_widget)

    if parent.klass.replace_child:
        parent.klass.replace_child(parent.project.context, gtk_old_widget,
                                   gtk_new_widget, parent.gtk_widget)
        if gnew_widget:
            gnew_widget._set_packing_properties(parent)
            gnew_widget._connect_signal_handlers(gnew_widget.gtk_widget)
    else:
        print _("Could not replace a placeholder because a replace function has not been implemented for '%s'") % parent.klass.name

        
def get_ui_widgets(gtk_widget):
    """Get a list of all the widgets inside the hierarchy that are widgets
    related to a UI Manager (e.g. Toolbars and Menubars).
    """
    result = []
    if isinstance(gtk_widget, (gtk.Toolbar, gtk.MenuBar)):
        result.append(gtk_widget)
        
    elif isinstance(gtk_widget, gtk.Container):
        for child in gtk_widget.get_children():
            result += get_ui_widgets(child)

    return result

class Widget(gobject.GObject):
    """ 
    This is essentially a wrapper around regular GtkWidgets with more data and
    functionality needed for Gazpacho.
    
    It has access to the GtkWidget it is associated with, the Project it
    belongs to and the instance of its WidgetClass.
    
    Given a GtkWidget you can ask which (Gazpacho) Widget is associated with it
    using the get_from_gtk_widget function or, if no one has been created yet you
    can create one with the create_from_gtk_widget method.
    
    A Widget have a signal dictionary where the keys are signal names and the
    values are lists of signal handler names. There are methods to
    add/remove/edit these.
    
    It store all the Properies (wrappers for GtkWidgets properties) and you can
    get any of them with get_glade_property. It knows how to handle normal
    properties and packing properties.
    """
    __gproperties__ = {
        'name' :    (str,
                     'Name',
                     'The name of the widget',
                     '',
                     gobject.PARAM_READWRITE),
        }
    
    __gsignals__ = {
        'add_signal_handler' :    (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove_signal_handler' : (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'change_signal_handler' : (gobject.SIGNAL_RUN_LAST, None,
                                   (object, object))
        }

    def __init__(self, klass, project):
        """This init method is not complete because there are several scenarios
        where widgets are created:
            
            - Normal creation in Gazpacho
            - When loading from a .glade file
            - When copying from another widget
            - Internal widgets
            
        So you should call the appropiate method after this basic initialization.
        """
        gobject.GObject.__init__(self)

        # If the widget is an internal child of another widget this is the
        # name of the internal child, otherwise is None
        # Internal children can not be deleted
        self.internal_name = None
        
        # A dict of Properties. A Property is an instance of a PropertyClass
        self.properties = {}
        
        # A dict of Properties. Note that these properties are related to
        # the container of the widget, thus they change after pasting the
        # widget to a different container. Toplevels widget do not have
        # packing properties. See also child_properties of WidgetClass
        self.packing_properties = {}
        
        # A dictionary of signals (signal, handlers) indexed by their names
        self.signals = {}
        
        # The WidgetClass this widget is an instance of
        self.klass = klass
        
        # The Project this widget belongs to
        self.project = project
        
        # The GtkWidget associated with this Widget
        self.gtk_widget = None

        self._name = None

        # we need to store the events in a separate variable because we can
        # not trust the events property of a widget since Gazpacho manipulates
        # it for its own purposes (see EventsAdaptor)
        self._events = 0

    def create_gtk_widget(self, interactive=True):
        """Second part of the init process when creating a widget in
        the usual way.
        If the 'interactive' argument is false no popups will show up to
        ask for options
        """

        gtk_widget = self.klass.library.create_widget(self.klass.type)

        if self.klass.pre_create_function:
            self.klass.pre_create_function(self.project.context, gtk_widget,
                                           interactive)
            
        self._name = self.project.new_widget_name(self.klass.generic_name)
        gtk_widget.set_property('name', self._name)

        self.setup_widget(gtk_widget)

        self._create_properties()

        self.apply_properties()
        
        if self.klass.post_create_function:
            self.klass.post_create_function(self.project.context, gtk_widget,
                                            interactive)

        if self.klass.fill_empty_function:
            self.klass.fill_empty_function(self.project.context, gtk_widget)

    def setup_widget(self, gtk_widget):
        if self.gtk_widget is not None:
            self.gtk_widget.set_data('GladeWidgetDataTag', None)
            self.gtk_widget = None

        self.gtk_widget = gtk_widget
        self.gtk_widget.set_data('GladeWidgetDataTag', self)
        self.gtk_widget.add_events(gtk.gdk.BUTTON_PRESS_MASK | \
                                   gtk.gdk.BUTTON_RELEASE_MASK | \
                                   gtk.gdk.KEY_PRESS_MASK)
        if self.gtk_widget.flags() & gtk.TOPLEVEL:
            self.gtk_widget.connect('delete_event', self._hide_on_delete)
        
        self.gtk_widget.connect('popup_menu', self._popup_menu)
        self.gtk_widget.connect('key_press_event', self._key_press)

        self._connect_signal_handlers(gtk_widget)
        
    def setup_internal_widget(self, gtk_widget, internal_name, parent_name):
        self.internal_name = internal_name
        self._name = parent_name + '-' + internal_name

        self.setup_widget(gtk_widget)
 
        self._create_properties()
        self.sync_properties_from_gtk()

        parent = self.get_parent()
        
        if parent is not None:
            self._set_packing_properties(parent)
        
    def do_get_property(self, prop):
        try:
            return getattr(self, 'get_' + prop.name)(self)
        except:
            raise AttributeError(_('Unknown property %s') % prop.name)

    def do_set_property(self, prop, value):
        try:
            getattr(self, 'set_' + prop.name)(self, value)
        except:
            raise AttributeError(_('Unknown property %s') % prop.name)

    # the properties
    def get_name(self): return self._name
    def set_name(self, value):
        self._name = value
        self.notify('name')
    name = property(get_name, set_name)

    def _create_properties(self):
        """This method should be called *after* setting the gtk_widget"""
        for prop in self.klass.properties:
            p = gazpacho.property.Property(prop, self)
            self.properties[prop.id] = p

    def sync_properties_from_gtk(self):
        """Copy the values of the gtk properties to the Gazpacho ones.
        Actually only copy readable properties for obvious reasons.
        This is usually needed when loading objects from a file
        """
        for pspec in filter(lambda p: p.flags & gobject.PARAM_READABLE,
                            gobject.list_properties(self.gtk_widget)):
            prop = self.properties.get(pspec.name)
            if prop is not None:
                prop._value = self.gtk_widget.get_property(pspec.name)

        # for custom properties we explicity reread them
        for prop in filter(lambda p: p.klass.custom, self.properties.values()):
            prop._value = prop.value

        # for properties that are not readable but we provide a custom
        # getter we reread them
        for prop in filter(lambda p: p.klass._get_function is not None,
                           self.properties.values()):
            prop._value = prop.value

    # utility getter methods
    def get_parent(self):
        if self.gtk_widget.flags() & gtk.TOPLEVEL:
            return None

        return util.get_parent(self.gtk_widget)

    def add_signal_handler(self, signal_handler):
        self.emit('add_signal_handler', signal_handler)
        
    def do_add_signal_handler(self, signal_handler):
        signals = self.list_signal_handlers(signal_handler['name'])
        if not signals:
            self.signals[signal_handler['name']] = []

        self.signals[signal_handler['name']].append(signal_handler)

    def remove_signal_handler(self, signal_handler):
        self.emit('remove_signal_handler', signal_handler)
        
    def do_remove_signal_handler(self, signal_handler):
        signals = self.list_signal_handlers(signal_handler.name)
        # trying to remove an inexistent signal?
        assert signals != []

        self.signals[signal_handler['name']].remove(signal_handler)
        
    def change_signal_handler(self, old_signal_handler, new_signal_handler):
        self.emit('change_signal_handler',
                   old_signal_handler, new_signal_handler)
        
    def do_change_signal_handler(self, old_signal_handler,
                                  new_signal_handler):
        if old_signal_handler['name'] != new_signal_handler['name']:
            return

        signals = self.list_signal_handlers(old_signal_handler['name'])
        # trying to remove an inexistent signal?
        assert signals != []

        index = signals.index(old_signal_handler)
        signals[index]['handler'] = new_signal_handler['handler']
        signals[index]['after'] = new_signal_handler['after']

    def _expose_event(self, widget, event):
        util.queue_draw_nodes(event.window)

    def _find_inside_container(self, widget, data):
        window = data['toplevel']
        coords = window.translate_coordinates(widget, data['x'], data['y'])
        if len(coords) == 2:
            x, y = coords
        else:
            return
        
        # sometimes the found widget is not mapped or visible
        # think about a widget in a notebook page which is not selected
        if (0 <= x < widget.allocation.width and
            0 <= y < widget.allocation.height and
            widget.flags() & gtk.MAPPED and
            widget.flags() & gtk.VISIBLE and
            get_widget_from_gtk_widget(widget)):
            data['found'] = widget
            
    def _find_deepest_child_at_position(self, toplevel, container,
                                        top_x, top_y):
        data = {'x': top_x, 'y': top_y, 'toplevel': toplevel, 'found': None}
        container.forall(self._find_inside_container, data)
        if data['found'] and isinstance(data['found'], gtk.Container):
            return self._find_deepest_child_at_position(toplevel,
                                                        data['found'],
                                                        top_x, top_y)
        elif data['found']:
            return get_widget_from_gtk_widget(data['found'])
        else:
            return get_widget_from_gtk_widget(container)
    
    def _retrieve_from_position(self, base, x, y):
        """ Returns the real GtkWidget in x, y of base.
        This is needed because for GtkWidgets than does not have a
        GdkWindow the click event goes right to their parent.
        """
        toplevel_widget = base.get_toplevel()
        if not (toplevel_widget.flags() & gtk.TOPLEVEL):
            return None
        top_x, top_y = base.translate_coordinates(toplevel_widget, x, y)
        return self._find_deepest_child_at_position(toplevel_widget,
                                                    toplevel_widget,
                                                    top_x, top_y)

    def _event(self, widget, event):
        """We only delegate this call to the appropiate event handler"""
        if event.type == gtk.gdk.BUTTON_PRESS:
            return self._button_press_event(widget, event)
        elif event.type == gtk.gdk.EXPOSE:
            self._expose_event(widget, event)

        return False
                
    def _button_press_event(self, widget, event):
        gwidget = self._retrieve_from_position(widget,
                                               int(event.x),
                                               int(event.y))
        widget = gwidget.gtk_widget
        # Make sure to grab focus, since we may stop default handlers
        if widget.flags() & gtk.CAN_FOCUS:
            widget.grab_focus()

        if event.button == 1:
            if event.type is not gtk.gdk.BUTTON_PRESS:
                #only single clicks allowed, thanks
                return False

            # Shift clicking circles through the widget tree by
            # choosing the parent of the currently selected widget.
            if event.state & gtk.gdk.SHIFT_MASK:
                util.circle_select(widget)
                return True

            # if it's already selected don't stop default handlers,
            # e.g toggle button
            selected = util.has_nodes(widget)
            self.project.selection_set(widget, True)
            return not selected
        elif event.button == 3:
            # first select the widget
            self.project.selection_set(widget, True)

            # then popup the menu
            app = self.project._app
            popup = Popup(app, gwidget)
            popup.pop(event)
            return True

        return False
    
    def _popup_menu(self, widget):
        return True # XXX TODO

    def _key_press(self, widget, event):
        gwidget = get_widget_from_gtk_widget(widget)

        if event.keyval in (gtk.keysyms.Delete, gtk.keysyms.KP_Delete):
            # We will delete all the selected items
            gwidget.project.delete_selection()
            return True
        
        return False

    def _hide_on_delete(self, widget, event):
        return widget.hide_on_delete()
    
    def _connect_signal_handlers(self, gtk_widget):
        # don't connect handlers for placeholders
        if isinstance(gtk_widget, placeholder.Placeholder):
            return

        # check if we've already connected an event handler
        if not gtk_widget.get_data(tags.EVENT_HANDLER_CONNECTED):
            # we are connecting to the event signal instead of the more
            # apropiated expose-event and button-press-event because some
            # widgets (ComboBox) does not allows us to connec to those
            # signals. See http://bugzilla.gnome.org/show_bug.cgi?id=171125
            # Hopefully we will be able to fix this hack if GTK+ is changed
            # in this case
            gtk_widget.connect('event', self._event)
            
#            gtk_widget.connect("expose-event", self._expose_event)
#            gtk_widget.connect("button-press-event", self._button_press_event)
            gtk_widget.set_data(tags.EVENT_HANDLER_CONNECTED, 1)
            
        # we also need to get expose events for any children
        if isinstance(gtk_widget, gtk.Container):
            gtk_widget.forall(self._connect_signal_handlers)

    def apply_properties(self, properties=None):
        """Set the state of the gtkwidget with the values of our properties.
        This method does pretty much the opposite as sync_properties_from_gtk

        If the properties parameter is not None we use that properties to
        setup the gtk_widget and also copy our properties. This is useful
        when copying a GWidget into another
        """
        self.gtk_widget.freeze_notify()

        props = properties or self.properties
        
        for prop_id, prop in props.items():
            
            if prop.klass._set_function:
                prop.value = prop._value
                prop.klass._set_function(self.project.context, self.gtk_widget,
                                         prop._value)
            # we don't support object properties here
            elif prop.klass.type != gobject.TYPE_OBJECT:
                self.gtk_widget.set_property(prop.klass.id, prop.value)

            # if we are copying the properties from another widget put that
            # value into our properties
            if properties is not None:
                gazpacho.property.copy_property(prop, self.properties[prop_id])
                
        self.gtk_widget.thaw_notify()

    def _get_apply_packing_properties(self, container):
        return [pc for pc in container.klass.child_properties \
                if container.klass.child_property_applies(container.gtk_widget,
                                                          self.gtk_widget,
                                                          pc.id)]
    
    def _create_packing_properties(self, container):
        """Create the Gazpacho Packing Properties for self based on the
        container.
        These are the properties that go in the Packing tab of the editor.
        """
        new_props = {}
        for property_class in self._get_apply_packing_properties(container):
            prop = gazpacho.property.Property(property_class, self)
            new_props[property_class.id] = prop

        return new_props

    def _set_default_packing_properties(self, container):
        """Setup self.gtk_widget with the default packing properties of the
        container.
        """
        for property_class in self._get_apply_packing_properties(container):

            v = self.klass.get_packing_default(container.klass,
                                               property_class.id)
            if not v:
                continue

            value = util.get_property_value_from_string (property_class, v)
            container.gtk_widget.child_set_property(self.gtk_widget,
                                                    property_class.id,
                                                    value)

    def _set_packing_properties(self, container):
        """When adding a child to a container this function is called to add
        the packing properties"""
        self._set_default_packing_properties(container)
        self.packing_properties = self._create_packing_properties(container)

        # update the values of the properties to the ones we get from gtk
        for prop_id, prop in self.packing_properties.items():
            prop.value = container.gtk_widget.child_get_property(self.gtk_widget,
                                                                 prop_id)

    def sync_packing_properties_from_gtk(self, container, gtk_widget):
        con_type = type(container)
        for pspec in gtk.container_class_list_child_properties(con_type):
            value = container.child_get_property(gtk_widget, pspec.name)
            self.packing_properties[pspec.name].value = value
        
    def get_glade_property(self, prop_id):
        # XXX Need to changed all 'glade' names for 'gazpacho'
        p = self.properties.get(prop_id)
        if p is not None:
            return p

        p = self.packing_properties.get(prop_id)
        if p is not None:
            return p

        print _('Could not get property %s for widget %s') % \
              (prop_id, self._name)

    def is_toplevel(self):
        return self.klass.is_toplevel()

    def _get_glade_widgets(self, gtk_widget):
        """ Return the first child (and its brothers) of gtk_widget which is a
        GladeWidget """
        if not isinstance(gtk_widget, gtk.Container):
            return []

        result = []
        for child in gtk_widget.get_children():
            gchild = get_widget_from_gtk_widget(child)
            if gchild is not None:
                result.append(child)
            elif isinstance(child, gtk.Container):
                result += self._get_glade_widgets(child)

        return result

    def get_as_xml_document(self):
        """Serialize the widget and return an xml document."""
        xml_document = xml.dom.getDOMImplementation().createDocument(None, None,
                                                                     None)
        element = xml_document.createElement(tags.XML_TAG_PROJECT)
        root = xml_document.appendChild(element)

        libglade_module = self.klass.library.libglade_module
        if libglade_module:
            requires = xml_document.createElement(tags.XML_TAG_REQUIRES)
            requires.setAttribute(tags.XML_TAG_LIB, libglade_module)
            root.appendChild(requires)

        # save the UI Manager if needed
        ui_widgets = get_ui_widgets(self.gtk_widget)
        if ui_widgets:
            ui_gwidgets = map(get_widget_from_gtk_widget, ui_widgets)
            uim_node = self.project.uim.save(xml_document, ui_gwidgets)
            root.appendChild(uim_node)
        
        root.appendChild(self.write(xml_document))
        return xml_document
    
    def write(self, document):
        """Serializes this widget into a XML node and returns this node"""

        # if we have a custom saver, use it
        if self.klass.save_function is not None:
            return self.klass.save_function(self.project.context, document,
                                            self)

        # otherwise use the default saver
        node = self.write_basic_information(document)
        
        self.write_properties(document, node)

        self.write_signals(document, node)
        
        # Children
        if isinstance(self.gtk_widget, gtk.Container):
            for child_widget in self.gtk_widget.get_children():
                gwidget = get_widget_from_gtk_widget(child_widget)
                if (isinstance(child_widget, placeholder.Placeholder)
                    or gwidget is not None):
                    child = self.write_child(child_widget, document)
                    if child is not None:
                        node.appendChild(child)
                else:
                    # sometimes there are intermediate widgets that we don't
                    # want to save, but maybe they contain children that are
                    # valid widget (we want to save)
                    for c in self._get_glade_widgets(child_widget):
                        cn = self.write_child(c, document)
                        if cn:
                            node.appendChild(cn)
                    
        return node

    def write_basic_information(self, document):
        node = document.createElement(tags.XML_TAG_WIDGET)
        node.setAttribute(tags.XML_TAG_CLASS, self.klass.name)
        node.setAttribute(tags.XML_TAG_ID, self.name)
        return node
        
    def write_properties(self, document, widget_node):
        gtk_props = [pspec.name for pspec in
                     gobject.list_properties(self.gtk_widget)]
        
        visible_node = None
        # write the properties
        for prop_id, prop in self.properties.items():
            # don't save the name property since it is saved in the id
            # attribute of the tag
            if prop_id == 'name':
                continue

            # packing properties are saved later
            if prop.klass.packing:
                continue

            # we don't store glade specific properties because libglade gets
            # confused with them
            if prop_id in gtk_props:
                child = prop.write(document)
                # the visible property of GtkWidget must be the last one
                # because otherwise props like default-width or
                # default-height doesn't work when loading the widget
                if child is not None:
                    if prop_id == 'visible':
                        visible_node = child
                    else:
                        widget_node.appendChild(child)
            # unless there is a custom save function for this property which
            # means we really want to save it
            elif prop.klass._save_function is not None:
                child = prop.write(document)
                widget_node.appendChild(child)
                
        if visible_node is not None:
            widget_node.appendChild(visible_node)

    def write_signals(self, document, widget_node):
        for signal_name, handlers in self.signals.items():
            for handler in handlers:
                child = document.createElement(tags.XML_TAG_SIGNAL)
                child.setAttribute(tags.XML_TAG_NAME, handler['name'])
                child.setAttribute(tags.XML_TAG_HANDLER, handler['handler'])
                child.setAttribute(tags.XML_TAG_AFTER,
                                   handler['after'] \
                                   and tags.TRUE or tags.FALSE)
                widget_node.appendChild(child)
                
    def write_child(self, gtk_widget, document):
        child_tag = document.createElement(tags.XML_TAG_CHILD)
        if isinstance(gtk_widget, placeholder.Placeholder):
            child = document.createElement(tags.XML_TAG_PLACEHOLDER)
            child_tag.appendChild(child)
            # we need to write the packing properties of the placeholder.
            # otherwise the container gets confused when loading its
            # children
            packing = self._write_placeholder_properties(gtk_widget, document)
            if packing is not None:
                child_tag.appendChild(packing)
            return child_tag

        child_widget = get_widget_from_gtk_widget(gtk_widget)
        if child_widget is None:
            return None

        if child_widget.internal_name is not None:
            child_tag.setAttribute(tags.XML_TAG_INTERNAL_CHILD,
                                   child_widget.internal_name)

        child = child_widget.write(document)
        child_tag.appendChild(child)

        # Append the packing properties
        if child_widget.packing_properties:
            packing = document.createElement(tags.XML_TAG_PACKING)
            child_tag.appendChild(packing)
            for prop in child_widget.packing_properties.values():
                packing_property = prop.write(document)
                if packing_property is not None:
                    packing.appendChild(packing_property)

        return child_tag

    def _write_placeholder_properties(self, placeholder, document):
        parent = placeholder.get_parent()
        # get the non default packing properties
        packing_list = []
        props = gtk.container_class_list_child_properties(parent)
        for prop in props:
            v = parent.child_get_property(placeholder, prop.name)
            if v != prop.default_value:
                packing_list.append((prop, v))
        
        if packing_list:
            packing_node = document.createElement(tags.XML_TAG_PACKING)
            for prop, value in packing_list:
                prop_node = document.createElement(tags.XML_TAG_PROPERTY)
                prop_name = prop.name.replace('-', '_')
                prop_node.setAttribute(tags.XML_TAG_NAME, prop_name)
                if prop.value_type == gobject.TYPE_ENUM:
                    v = enum_to_string(value, prop)
                elif prop.value_type == gobject.TYPE_FLAGS:
                    v = flags_to_string(value, prop)
                else:
                    v = escape(str(value))
                
                text = document.createTextNode(v)
                prop_node.appendChild(text)
                packing_node.appendChild(prop_node)
            
            return packing_node
    
    def list_signal_handlers(self, signal_name):
        result = []
        try:
            result = self.signals[signal_name]
        except KeyError:
            pass

        return result

    def set_default_gtk_packing_properties(self, gtk_widget, old_widget_klass):
        """Set the default gtk packing properties of this widget to
        the gtk_widget. Actually, only set those properties that are different
        from old_widget_klass.get_packing_default properties

        This is needed when removing a widget from a container and puting
        a placeholder in that place. For example, if the child was a Label
        with expand=False, when putting a placeholder, we need to restore
        the orginal packing properties or we won't see the placeholder.
        See bug #167464
        """
        container = self.gtk_widget
        container_type = type(container)
        container_klass = self.klass
        props = gtk.container_class_list_child_properties(container_type)
        for pspec in props:
            old_default = old_widget_klass.get_packing_default(container_klass,
                                                               pspec.name)
            if old_default is not None:
                # This means the old widget uses a different default from the
                # gtk one
                container.child_set_property(gtk_widget,
                                             pspec.name, pspec.default_value)
            # note that if the old widget uses the gtk default value for this
            # property we don't change its current value

    def name_of_internal_child(self, gtk_child):
        """Return the internal name of gtk_child, which is supposed to
        be an internal child.

        If gtk_child is not an internal child, returns None
        """
        # check if this widget does have internal children
        if self.klass.get_internal_child is None:
            return None

        possible_names = self.klass.get_internal_child(self.project.context,
                                                       self.gtk_widget)
        for internal_name in possible_names:
            internal_child = self.klass.get_internal_child(self.project.context,
                                                           self.gtk_widget,
                                                           internal_name)
            if internal_child is gtk_child:
                return internal_name

gobject.type_register(Widget)

