# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gobject
import gtk

from gazpacho import propertyclass as pc
from gazpacho.loader import tags
from gazpacho.choice import enum_to_string, flags_to_string
from gazpacho import util

from xml.sax.saxutils import escape

def copy_property(src, dst):
    """Copy the attributes of the property src into the property dst"""
    assert dst.klass == src.klass
    dst.enabled = src.enabled
    dst.loading = src.loading
    dst._value = src._value
    dst.is_translatable = src.is_translatable
    dst.i18n_comment = src.i18n_comment
    dst.has_i18n_context = src.has_i18n_context
    # don't copy the widget or changing the old property will change the new
    # one

class Property(object):
    """ A Property is an instance of a PropertyClass. There will be one
    PropertyClass for "GtkLabel->label" but one Property for each GtkLabel
    in the GladeProject. """
    
    def __init__(self, property_class, widget):
        self.klass = property_class
        self.widget = widget
        self.enabled = True
        self.loading = False
        self._value = None
        self.is_translatable = True
        self.i18n_comment = None
        self.has_i18n_context = False   

        if property_class.type in (gobject.TYPE_DOUBLE, gobject.TYPE_INT,
                                   gobject.TYPE_FLOAT):
            self.enabled = property_class.optional_default

        initial_value = None
        try:
            initial_value = widget.gtk_widget.get_property(property_class.id)
        except TypeError:
            pass

        if property_class.type in (gobject.TYPE_ENUM, gobject.TYPE_FLAGS,
                                   gobject.TYPE_BOOLEAN, gobject.TYPE_STRING,
                                   gobject.TYPE_UINT, gobject.TYPE_DOUBLE,
                                   gobject.TYPE_INT, gobject.TYPE_FLOAT):
            # give more preference to the default value defined in the
            # catalog file
            if property_class.default is not None:
                initial_value = property_class.default
            else:
                initial_value = initial_value
                
            self.value = initial_value
        else:
            self.value = initial_value

#        print 'Creating property %s of type %s with value' % (self._klass.id, pc.type2str(self._klass.type)), self._value, initial_value

    def get_value(self):
        # Use custom get function if available, fallback to the
        # regular value.
        if self.klass._get_function is not None:
            context = self.widget.project.context
            value = self.klass._get_function(context, self.widget.gtk_widget)
            if value is not None:
                return value

        return self._value

    def set_value(self, value):
        if self.klass.type in (gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE,
                               gobject.TYPE_INT) \
           and value != self.klass.default:
            self.enabled = True
                
        if self.klass.type == gobject.TYPE_BOOLEAN:
            self._value = bool(value)
        elif self.klass.type in (gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE):
            self._value = float(value or 0.0)
        elif self.klass.type == gobject.TYPE_INT:
            self._value = int(value or 0)
        elif self.klass.type == gobject.TYPE_STRING:
            self._value = value
        elif self.klass.type == gobject.TYPE_UINT:
            if self.klass.id in pc.UNICHAR_PROPERTIES:
                self._value = unicode(value and value[0] or "")
            else:
                self._value = int(value or 0)
        elif self.klass.type == gobject.TYPE_ENUM:
            self._value = int(value or 0)
        elif self.klass.type == gobject.TYPE_FLAGS:
            self._value = int(value or 0)
        elif gobject.type_is_a(self.klass.type, gobject.TYPE_OBJECT):
            self._value = value
        elif self.klass.type is None:
            print _("Error: the property %s should not be of type None") % \
                  self.klass.name
            return

    value = property(get_value, set_value)

    def _set_property(self, value):
        if self.klass.packing:
            parent = self.widget.get_parent()
            container = parent.gtk_widget
            child = self.widget.gtk_widget
            container.child_set_property(child, self.klass.id, value)
        else:
            try:
                self.widget.gtk_widget.set_property(self.klass.id, value)
            except TypeError:
                pass # XXX TODO there is a bug in pygtk:
            # >>> e.set_property('invisible-char', unicode('*'))
            # Traceback (most recent call last):
            # File "<stdin>", line 1, in ?
            # TypeError: could not convert argument to correct param type

    def set(self, value):
        if not self.enabled: return
        if self.loading: return

        self.loading = True

        # first, trigger the conversions
        self.value = value
        
        # if there is a custom set property use it
        if self.klass._set_function:
            context = self.widget.project.context
            self.klass._set_function(context, self.widget.gtk_widget,
                                     self._value)
        else:
            self._set_property(self.value)
            
        self.loading = False


    def _internal_write(self, document, value):
        if not self.enabled:
            return None

        if not self.klass.is_visible(self.widget.klass):
            return None

        node = document.createElement(tags.XML_TAG_PROPERTY)

        # We should change each '-' by '_' on the name of the property
        tmp = self.klass.id.replace('-', '_')

        # put the name="..." part on the <property ...> tag
        node.setAttribute(tags.XML_TAG_NAME, tmp)

        # Write any i18n metadata for translatable string properties
        if self.klass.is_translatable \
               and self.klass.type == gobject.TYPE_STRING:
            # Only write context and comment if translatable is
            # enabled, to mimic glade-2
            if self.is_translatable:
                node.setAttribute(tags.XML_TAG_TRANSLATABLE, tags.TRUE)
                if self.has_i18n_context:
                    node.setAttribute(tags.XML_TAG_CONTEXT, tags.TRUE)
                if self.i18n_comment is not None:
                    node.setAttribute(tags.XML_TAG_COMMENT, self.i18n_comment)

        if self.klass.original_default == self._value:
            return None

        text = document.createTextNode(value)
        node.appendChild(text)
        return node
        
    def write(self, document):
        if self._value is None:
            return None

        if self.klass._save_function:
            context = self.widget.project.context
            gtk_widget = self.widget.gtk_widget
            value = self.klass._save_function(context, gtk_widget)
            
        elif self.klass.type == gobject.TYPE_ENUM:
            value = enum_to_string(self._value, self.klass.spec)
        elif self.klass.type == gobject.TYPE_FLAGS:
            value = flags_to_string(self._value, self.klass.spec)
        elif gobject.type_is_a(self.klass.type, gtk.Adjustment.__gtype__):
            data = "%f %f %f %f %f %f" % (self._value.value, self._value.lower,
                                          self._value.upper,
                                          self._value.step_increment,
                                          self._value.page_increment,
                                          self._value.page_size)
            value = escape(data)
        elif not gobject.type_is_a(self.klass.type, gobject.TYPE_OBJECT):
            value = escape(str(self.value))
        else:
            # we don't save properties of type == object unless they
            # have a custom save_function (see above)
            return None

        return self._internal_write(document, value)
    
    def read_i18n_data(self, gtk_widget):
        comment = util.gtk_widget_get_metadata(gtk_widget,
                                               util.METADATA_I18N_COMMENT,
                                               self.klass.id)
        if comment is not None:
            self.i18n_comment = comment

        self.is_translatable = util.gtk_widget_get_metadata(gtk_widget,
                                                            util.METADATA_I18N_IS_TRANSLATABLE,
                                                            self.klass.id)

        self.has_i18n_context = util.gtk_widget_get_metadata(gtk_widget,
                                                             util.METADATA_I18N_HAS_CONTEXT,
                                                             self.klass.id)
