# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

from gazpacho.loader import tags
from gazpacho.widgetclass import WidgetClass
from gazpacho.path import catalogs_dir
from gazpacho.widgetregistry import widget_registry
from gazpacho import library

import os.path
import sys

import xml.dom.minidom

import gtk

class BadCatalogSyntax(Exception):
    def __init__(self, file, msg):
        self.file, self.msg = file, msg
    def __str__(self):
        return "Bad syntax in the catalog %s: %s" % (self.file, self.msg)
    
def error_dialog(error_msg):
    """Show an error message about problems when loading the catalogs.

    Not that the application window has not been created yet so we don't have
    a parent window for the dialog.
    """
    msg = _("<b>There were errors while loading Gazpacho</b>")
    d = gtk.MessageDialog(parent=None, flags=gtk.DIALOG_MODAL,
                          type=gtk.MESSAGE_ERROR, buttons=gtk.BUTTONS_OK)
    d.set_markup(msg)
    expander = gtk.Expander("Click here for details")
    expander.set_border_width(6)
    label = gtk.Label(error_msg)
    expander.add(label)
    d.vbox.pack_end(expander)
    d.show_all()
    d.run()
    d.destroy()
    
def load_all():
    """Return a list of existing Catalogs.

    The application widget registry is cleared and previously loaded classes
    and groups removed before rereading existing catalogs.
    The GTK+ catalog takes precedence and catalogs are ordered after
    their declared priority.
    """
    widget_registry.clean()
    catalogs = []

    catalog_error = ""
    dir_list = []
    try:
        dir_list = os.listdir(catalogs_dir)
    except OSError, e:
        if e.errno == 2:
            catalog_error = _("The catalog directory %s does not exist" % \
                              catalogs_dir)
        elif e.errno == 13:
            catalog_error = _("You do not have read permissions in the "
                              "catalog directory %s" % catalogs_dir)
        else:
            catalog_error = e.strerror

    if 'gtk+.xml' not in dir_list:
        raise SystemExit, "Unable to find GTK+ catalog which is mandatory"

    if catalog_error:
        error_dialog(catalog_error)

    else:
        # the gtk+.xml catalog is always loaded first
        del dir_list[dir_list.index('gtk+.xml')]
        dir_list.insert(0, 'gtk+.xml')
        
        problems = []
        for filename in dir_list:

            if filename[-4:] != '.xml':
                continue

            try:
                catalog = load_catalog(filename)
                catalogs.append(catalog)
            except IOError, e:
                problems.append(str(e))
            except BadCatalogSyntax, e:
                problems.append(str(e))

        if problems:
            error_dialog('\n'.join(problems))

    catalogs.sort(lambda c1, c2: cmp(c1.priority, c2.priority))
    
    return catalogs

def load_catalog(filename):
    """Return a Catalog of Widget Groups and Widget Classes in a file.
    """
    f = file(os.path.join(catalogs_dir, filename))

    dom = xml.dom.minidom.parse(f)
    
    f.close()
    root_node = dom.documentElement

    if root_node.nodeName != tags.GLADE_CATALOG:
        msg = "Root node is not glade catalog: %s" % root_node.nodeName
        raise BadCatalogSyntax(filename, msg)

    catalog = Catalog()
    catalog.title = root_node.getAttribute(tags.NAME)

    try:
        catalog.priority = int(root_node.getAttribute(tags.PRIORITY))
    except ValueError, e:
        catalog.priority = sys.maxint
        
    library_name = root_node.getAttribute(tags.LIBRARY)

    if root_node.hasAttribute(tags.LIBRARY_TYPE):
        library_type = root_node.getAttribute(tags.LIBRARY_TYPE)
    else:
        library_type = "python"
    
    path_tag = root_node.getAttribute(tags.RESOURCE_PATH)
    
    if root_node.hasAttribute(tags.RESOURCE_PATH):
        resource_path = root_node.getAttribute(tags.RESOURCE_PATH)
    else:
        resource_path = None

    if resource_path and resource_path not in sys.path:
        sys.path.append(resource_path)

    if root_node.hasAttribute(tags.LIBGLADE_MODULE):
        libglade_module = root_node.getAttribute(tags.LIBGLADE_MODULE)
    else:
        libglade_module = None
    catalog.library = library.load_library(library_name, library_type,
                                           libglade_module)

    for child in root_node.childNodes:
        if child.nodeName == tags.GLADE_WIDGET_CLASSES:
            for class_node in child.childNodes:
                if class_node.nodeName == tags.GLADE_WIDGET_CLASS:
                    rtf = class_node.getAttribute(tags.REGISTER_TYPE_FUNCTION)
                    widget_class = WidgetClass(class_node,
                                               catalog.library,
                                               resource_path,
                                               rtf)
                                               
                    if widget_class is not None:
                        catalog.widget_classes.append(widget_class)
            
        elif child.nodeName == tags.GLADE_WIDGET_GROUP:
            widget_group = WidgetGroup(child)
            if widget_group is not None:
                catalog.widget_groups.append(widget_group)

    return catalog

def load_classes(catalog, node):
    """Read child widgets contained by a widget in a given node.
    """
    #XXX: Lorenzo, is this used anywhere??????
    for child in node.childNodes:
        if child.nodeName == tags.GLADE_WIDGET_CLASS:
            widget_class = WidgetClass(child, catalog.library)
            if widget_class is not None:
                catalog.widget_classes.append(widget_class)

class WidgetGroup(list):
    """List class used to store a widget group and its child nodes.

    The name property sets/gets the group internal name.
    The title property sets/gets the group title string.
    The class list contains the widget_class strings for the group.
    """
    def __init__(self, node):
        # Do some parsing
        self._name = ''
        self._title = ''

        self.load(node)

    def get_name(self):
        return self._name

    def set_name(self, name):
        self._name = name

    def get_title(self):
        return self._title

    def set_title(self, title):
        self._title = title

    name = property(get_name, set_name)
    title = property(get_title, set_title)

    def load(self, node):
        "Get name and title of a node and append its widget_class elements"
        self._name = node.getAttribute(tags.NAME)
        self._title = node.getAttribute(tags.TITLE)

        for child in node.childNodes:
            if child.nodeName != tags.GLADE_WIDGET_CLASS_REF:
                continue

            name = child.getAttribute(tags.NAME)
            widget_class = widget_registry.get_by_name(name)
            if widget_class is not None:
                self.append (widget_class)
            else:
                print 'could not load widget class %s' % name

class Catalog:
    """Class to hold widget classes and groups.

    The title property gets/sets the catalog title string.
    The library property gets/sets the catalog library string.
    The widget_classes attribute holds a list of widget classes in the catalog.
    The widget_groups attribute holds a list of widget groups in the catalog.
    """
    def __init__(self):
        self._title = ''
        self._library = ''
        self.widget_classes = []
        self.widget_groups = []

    def get_title(self):
        return self._title

    def set_title(self, value):
        self._title = value

    title = property(get_title, set_title)

    def get_library(self):
        return self._library

    def set_library(self, library):
        self._library = library

    library = property(get_library, set_library)

    def __repr__(self):
        return 'Catalog->' + self._title
