# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gobject

from gazpacho import propertyclass as pc
from gazpacho.loader import tags
from gazpacho.l10n import _

from xml.sax.saxutils import escape

class Property(object):
    """ A Property is an instance of a PropertyClass. There will be one
    PropertyClass for "GtkLabel->label" but one Property for each GtkLabel
    in the GladeProject. """
    
    def __init__(self, property_class, widget):
        self._klass = property_class
        self._widget = widget
        self._enabled = True
        self._loading = False
        self._value = None
        self._is_translatable = True
        self._i18n_comment = None
        self._has_i18n_context = False   

        if property_class.type in (gobject.TYPE_DOUBLE, gobject.TYPE_INT,
                                   gobject.TYPE_FLOAT):
            self._enabled = property_class.optional_default

        initial_value = ''
        try:
            initial_value = widget.get_property(property_class.id)
        except TypeError:
            pass

        if property_class.type in (gobject.TYPE_ENUM, gobject.TYPE_FLAGS,
                                   gobject.TYPE_BOOLEAN, gobject.TYPE_STRING,
                                   gobject.TYPE_UINT, gobject.TYPE_DOUBLE,
                                   gobject.TYPE_INT, gobject.TYPE_FLOAT):
            initial_value = initial_value or property_class.default
            self.value = initial_value
        else:
            self.value = initial_value

#        print 'Creating property %s of type %s with value' % (self._klass.id, pc.type2str(self._klass.type)), self._value, initial_value

    # properties
    def get_klass(self): return self._klass
    klass = property(get_klass)

    def get_widget(self): return self._widget
    widget = property(get_widget)

    def get_enabled(self): return self._enabled
    def set_enabled(self, value): self._enabled = value
    enabled = property(get_enabled, set_enabled)

    def get_loading(self): return self._loading
    def set_loading(self, value): self._loading = value
    loading = property(get_loading, set_loading)

    def get_value(self):
        # Use custom get function if available, fallback to the
        # regular value.
        if self.klass._get_function is not None:
            value = self._klass._get_function(self._widget.widget)
            if value is not None:
                return value

        return self._value

    def set_value(self, value):
        if self._klass.type in (gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE,
                                gobject.TYPE_INT) \
                                and value != self._klass.default:
            self._enabled = True
                
        if self._klass.type == gobject.TYPE_BOOLEAN:
            self._value = bool(value)
        elif self._klass.type in (gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE):
            self._value = float(value or 0.0)
        elif self._klass.type == gobject.TYPE_INT:
            self._value = int(value or 0)
        elif self._klass.type == gobject.TYPE_STRING:
            self._value = value or ""
        elif self._klass.type == gobject.TYPE_UINT:
            if self._klass.id in pc.UNICHAR_PROPERTIES:
                self._value = unicode(value and value[0] or "")
            else:
                self._value = int(value or 0)
        elif self._klass.type == gobject.TYPE_ENUM:
            self._value = int(value or 0)
        elif self._klass.type == gobject.TYPE_FLAGS:
            self._value = int(value or 0)
        elif self._klass.type == gobject.TYPE_OBJECT:
            self._value = value
        elif self._klass.type is None:
            print _("Error: the property %s should not be of type None") % \
                  self._klass.name
            return

    value = property(get_value, set_value)

    def get_is_translatable(self):
        return self._is_translatable

    def set_is_translatable(self, value):
        self._is_translatable = value

    is_translatable = property(get_is_translatable, set_is_translatable)

    def get_has_i18n_context(self):
        return self._has_i18n_context

    def set_has_i18n_context(self, value):
        self._has_i18n_context = value

    has_i18n_context = property(get_has_i18n_context, set_has_i18n_context)

    def get_i18n_comment(self):
        return self._i18n_comment

    def set_i18n_comment(self, value):
        self._i18n_comment = value

    i18n_comment = property(get_i18n_comment, set_i18n_comment)

    def _set_property(self, value):
        if self._klass.packing:
            parent = self._widget.get_parent()
            container = parent.widget
            child = self._widget.widget
            container.child_set_property(child, self._klass.id, value)
        else:
            try:
                self._widget.widget.set_property(self._klass.id, value)
            except TypeError:
                pass # XXX TODO there is a bug in pygtk:
            # >>> e.set_property('invisible-char', unicode('*'))
            # Traceback (most recent call last):
            # File "<stdin>", line 1, in ?
            # TypeError: could not convert argument to correct param type

    def set(self, value):
        if not self._enabled: return
        if self._loading: return

        self._loading = True

        # first, trigger the conversions
        self.value = value
        
        # if there is a custom set property use it
        if self._klass._set_function:
            self._klass._set_function(self._widget.widget, self._value)
        else:
            self._set_property(self.value)
            
        self._loading = False


    def _internal_write(self, document, value):
        if not self.enabled:
            return None

        if not self.klass.is_visible(self.widget.klass):
            return None

        node = document.createElement(tags.XML_TAG_PROPERTY)

        # We should change each '-' by '_' on the name of the property
        tmp = self.klass.id.replace('-', '_')

        # put the name="..." part on the <property ...> tag
        node.setAttribute(tags.XML_TAG_NAME, tmp)

        # Write any i18n metadata for translatable string properties
        if self.klass.is_translatable \
               and self.klass.type == gobject.TYPE_STRING:
            # Only write context and comment if translatable is
            # enabled, to mimic glade-2
            if self.is_translatable:
                node.setAttribute(tags.XML_TAG_TRANSLATABLE, tags.TRUE)
                if self.has_i18n_context:
                    node.setAttribute(tags.XML_TAG_CONTEXT, tags.TRUE)
                if self.i18n_comment is not None:
                    node.setAttribute(tags.XML_TAG_COMMENT, self.i18n_comment)

#        if self.klass.default == self._value:
#            return None

        text = document.createTextNode(value)
        node.appendChild(text)
        return node

        
    def write(self, document):
        if self._value is not None:
            return self._internal_write(document, escape(str(self._value)))
    
    def write_in_cdata(self, document):
        if self._value is not None:
            data = '<![CDATA[%s]]>' % self._value
            return self._internal_write(document, data)
