# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

from gazpacho.util import unselect_when_clicked_on_empty_space, select_iter
from gazpacho.util import xml_create_string_prop_node
from gazpacho.loader import tags
from gazpacho.l10n import _

def get_gaction_from_gtk_action(gtk_action, parent):
    name = gtk_action.get_name()
    label = gtk_action.get_property('label')
    tooltip = gtk_action.get_property('tooltip')
    stock_id = gtk_action.get_property('stock-id') or None
    gaction = GAction(parent, name, label, tooltip, stock_id)

    # check if it has accelerator
    tmp = gtk.accel_map_lookup_entry('<Actions>/%s/%s' % (parent.name, name))
    if tmp is not None:
        key, modifier = tmp
        if key != 0:
            try:
                key_string = chr(key)
            except ValueError:
                key_string = key
            
            table = {gtk.gdk.CONTROL_MASK: '<control>',
                     gtk.gdk.MOD1_MASK: '<alt>'}

            accelerator = '%s%s' % (table[modifier], key_string)
            gaction.accelerator = accelerator

    # check if it has signal handler
    callback = gtk_action.get_data(tags.SIGNAL_HANDLER)
    if callback is not None:
        gaction.callback = callback
     
    return gaction


class GAction(object):
    """A GAction has the same information that a GtkAction but it
    is easier to work with. The menubar/toolbar editor code will
    convert this GActions to GtkActions and viceversa.
    """
    def __init__(self, parent, name='', label='', tooltip='', stock_id=None,
                 callback='', accel=''):
        # parent GActionGroup
        self.parent = parent 
        
        self.name = name
        self.label = label
        self.tooltip = tooltip
        self.stock_id = stock_id
        self.callback = callback
        self.accelerator = accel
    
    def __str__(self):
        return "%s %s %s %s %s" % (self.name, self.label, self.stock_id,
                                   self.accelerator, self.tooltip)

    def write(self, xml_doc):
        node = xml_doc.createElement(tags.XML_TAG_OBJECT)
        node.setAttribute(tags.XML_TAG_CLASS, "GtkAction")
        node.setAttribute(tags.XML_TAG_ID, self.name)
        node.appendChild(xml_create_string_prop_node(xml_doc, 'label',
                                                     self.label))
        node.appendChild(xml_create_string_prop_node(xml_doc, 'tooltip',
                                                     self.tooltip))
        if self.stock_id is not None:
            node.appendChild(xml_create_string_prop_node(xml_doc, 'stock_id',
                                                         self.stock_id))
        node.appendChild(xml_create_string_prop_node(xml_doc, 'callback',
                                                     self.callback))
        node.appendChild(xml_create_string_prop_node(xml_doc, 'accelerator',
                                                     self.accelerator))
        return node

def get_gaction_group_from_gtk_action_group(gtk_action_group):
    gag = GActionGroup(gtk_action_group.get_name())
    return gag

class GActionGroup(gobject.GObject):
    """A GActionGroup is just a list of GActions with a name.
    Associated with each GActionGroup there is a GtkActionGroup with
    is keep synchronized with the GActionGroup:
        
        - As we can't change the name of a GtkActionGroup, if the
        user change the name of a GActionGroup we have to regenerate
        the GtkActionGroup again, copying the actions from the old one
        to the new one.
        
        - When an action is added or removed, we create or destroy the
        GtkAction associated with it. If an action is only changed, we
        regenerate the GtkAction and tell the UIManager to update itself
        by removing the GtkActionGroup and then addind it again.
        Anyone knows a better way to do so?
    """
    __gsignals__ = {
        'add_action':    (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove_action': (gobject.SIGNAL_RUN_LAST, None, (object,)),
        }

    def __init__(self, name=''):
        gobject.GObject.__init__(self)
        self._name = name
        self.actions = []
        self.uimanager = None

    def create_gtk_action_group(self, uimanager):
        self.uimanager = uimanager
        gtk_action_group = gtk.ActionGroup(self.name)
        gtk_action_group.set_data('gaction_group', self)
        self.set_data('gtk_action_group', gtk_action_group)
        self.uimanager.insert_action_group(gtk_action_group, 0)
        return gtk_action_group
        
    def destroy_gtk_action_group(self):
        gtk_action_group = self.get_data('gtk_action_group')
        self.set_data('gtk_action_group', None)
        self.uimanager.remove_action_group(gtk_action_group)
    
    def get_name(self):
        return self._name
    
    def set_name(self, new_name):
        """GtkAction groups can't change their name

        That's why we need to copy the old one into a new one
        """
        old_name = self._name
        self._name = new_name
        
        if self.uimanager is not None:
            gtk_action_group = self.get_data('gtk_action_group')
            self.uimanager.remove_action_group(gtk_action_group)
            new_gtk_action_group = self.create_gtk_action_group(self.uimanager)

            for action in gtk_action_group.list_actions():
                gtk_action_group.remove_action(action)
                new_gtk_action_group.add_action(action)

    name = property(get_name, set_name)

    def append(self, action):
        self.actions.append(action)

        self._add_gtk_action(action)
        
        self.emit('add-action', action)
    
    def _add_gtk_action(self, action):
        gtk_action_group = self.get_data('gtk_action_group')
        gtk_action_group.add_actions((
            (action.name, action.stock_id, action.label, action.accelerator,
             action.tooltip, self._dumb_callback),
            ))
        
    def remove(self, action):
        self.actions.remove(action)
        self._remove_gtk_action(action.name)
        self.emit('remove-action', action)

    def _remove_gtk_action(self, name):
        gtk_action_group = self.get_data('gtk_action_group')
        gtk_action = gtk_action_group.get_action(name)
        gtk_action_group.remove_action(gtk_action)
        
    def update_action(self, action, old_name):
        self._remove_gtk_action(old_name)
        self._add_gtk_action(action)
        if self.uimanager is not None:
            # we need to remove it and then add it to the uimanager
            # so all the proxies are updated
            gtk_action_group = self.get_data('gtk_action_group')
            self.uimanager.remove_action_group(gtk_action_group)
            self.uimanager.insert_action_group(gtk_action_group, 0)

    def _dumb_callback(self, gtk_action):
        pass

    def get_action(self, action_name):
        for action in self.actions:
            if action.name == action_name:
                return action
            
    def write(self, xml_doc):
        node = xml_doc.createElement(tags.XML_TAG_OBJECT)
        node.setAttribute(tags.XML_TAG_CLASS, "GtkActionGroup")
        node.setAttribute(tags.XML_TAG_ID, self.name)
        for action in self.actions:
            child_node = xml_doc.createElement(tags.XML_TAG_CHILD)
            node.appendChild(child_node)
            action_node = action.write(xml_doc)
            child_node.appendChild(action_node)
        return node

gobject.type_register(GActionGroup)

class GActionsView(gtk.ScrolledWindow):
    """A GActionsView is basically a TreeView where the actions
    and action groups are shown.
    
    The data it shows is always a two level hierarchy tree, where
    toplevel nodes are GActionGroups and leaves are GActions.
    """
    def __init__(self):
        gtk.ScrolledWindow.__init__(self)
        self.set_shadow_type(gtk.SHADOW_IN)
        
        self.project = None

        self._model = gtk.TreeStore(object)
        self._treeview = gtk.TreeView(self._model)
        self._treeview.set_headers_visible(False)
        self._treeview.connect('row-activated', self._row_activated_cb)
        self._treeview.connect('button-press-event',
                              unselect_when_clicked_on_empty_space)
        
        column = gtk.TreeViewColumn()
        renderer1 = gtk.CellRendererPixbuf()
        column.pack_start(renderer1, expand=False)
        column.set_cell_data_func(renderer1, self._draw_action, 0)

        renderer2 = gtk.CellRendererText()
        column.pack_start(renderer2, expand=True)
        column.set_cell_data_func(renderer2, self._draw_action, 1)

        self._treeview.append_column(column)

        self.add(self._treeview)

    def set_project(self, project):
        if self.project is not None:
            self.project.disconnect(self._add_action_id)
            self.project.disconnect(self._remove_action_id)
            self.project.disconnect(self._action_name_changed_id)
            
        self.project = project
        self._model.clear()
        if self.project is not None:
            self._add_action_id = self.project.connect('add-action',
                                                       self._add_action_cb)
            self._remove_action_id = self.project.connect('remove-action',
                                                          self._remove_action_cb)
            self._action_name_changed_id = self.project.connect('action-name-changed', self._action_name_changed_cb)
            self._fill_model()
        
    def _fill_model(self):
        for gaction_group in self.project.action_groups:
            parent_iter = self._model.append(None, (gaction_group,))
            for gaction in gaction_group.actions:
                self._model.append(parent_iter, (gaction,))

    def _add_action_cb(self, project, action):
        new_iter = None
        if isinstance(action, GActionGroup):
            new_iter = self._model.append(None, (action,))
        elif isinstance(action, GAction):
            parent = action.parent
            iter = self._find_action_group(parent)
            if iter is not None:
                new_iter = self._model.append(iter, (action,))
        
        if new_iter is not None:
            select_iter(self._treeview, new_iter)
    
    def _find_action_group(self, action_group):
        model_iter = self._model.get_iter_first()
        while model_iter:
            if self._model.get_value(model_iter, 0) == action_group:
                return model_iter
            model_iter = self._model.iter_next(model_iter)

    def _find_action(self, parent_iter, gaction):
        model_iter = self._model.iter_children(parent_iter)
        while model_iter:
            if self._model.get_value(model_iter, 0) == gaction:
                return model_iter
            model_iter = self._model.iter_next(model_iter)
            
    def _remove_action_cb(self, project, gaction):
        if isinstance(gaction, GActionGroup):
            action_group_iter = self._find_action_group(gaction)
            if action_group_iter is not None:
                self._model.remove(action_group_iter)
        elif isinstance(gaction, GAction):
            parent_iter = self._find_action_group(gaction.parent)
            if parent_iter is not None:
                child_iter = self._find_action(parent_iter, gaction)
                if child_iter is not None:
                    self._model.remove(child_iter)

    def _action_name_changed_cb(self, project, gaction):
        action_iter = None
        if isinstance(gaction, GAction):
            parent_iter = self._find_action_group(gaction.parent)
            action_iter = self._find_action(parent_iter, gaction)
        elif isinstance(gaction, GActionGroup):
            action_iter = self._find_action_group(gaction)
            
        if action_iter is not None:
            path = self._model.get_path(action_iter)
            self._model.row_changed(path, action_iter)            
    
    def get_selected_action(self):
        selection = self._treeview.get_selection()
        model, model_iter = selection.get_selected()
        if model_iter is not None:
            return model.get_value(model_iter, 0)

    def get_selected_action_group(self):
        selection = self._treeview.get_selection()
        model, model_iter = selection.get_selected()
        if model_iter is not None:
            gaction = model.get_value(model_iter, 0)
            if isinstance(gaction, GActionGroup):
                return gaction
    
    def _draw_action(self, column, cell, model, iter, model_column):
        action = model.get_value(iter, 0)
        if model_column == 0:
            prop = 'stock-id'
            if isinstance(action, GActionGroup):
                data = 'gtk-execute'
            else:
                data = action.stock_id or ''
        elif model_column == 1:
            prop = 'text'
            data = action.name
            
        cell.set_property(prop, data)

    def _row_activated_cb(self, treeview, path, column):
        iter = self._model.get_iter(path)
        gaction = self._model.get_value(iter, 0)
        self.project._app._real_edit_action(gaction)

gobject.type_register(GActionsView)

class GActionDialog(gtk.Dialog):
    """This Dialog allows the creation and edition of a GAction."""
    def __init__(self, toplevel=None, action=None):
        gtk.Dialog.__init__(self,
                            title='',
                            parent=toplevel,
                            flags=gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            buttons=(gtk.STOCK_OK, gtk.RESPONSE_OK,
                                     gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))

        self.set_border_width(6)
        self.set_resizable(False)
        self.set_has_separator(False)
        self.vbox.set_spacing(6)
        
        if action is None:
            self.set_title(_('Add Action'))
        else:
            self.set_title(_('Edit Action'))

        self._create_widgets()

        self.action = action

        self.set_default_response(gtk.RESPONSE_OK)
        
        self.vbox.show_all()


    def _create_widgets(self):

        size_group = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)

        # id
        hbox = gtk.HBox(spacing=6)
        self._create_label('Id', size_group, hbox)
        self.id = gtk.Entry()
        self.id.set_activates_default(True)
        hbox.pack_start(self.id)
        self.vbox.pack_start(hbox)

        # label
        hbox = gtk.HBox(spacing=6)
        self._create_label(_('Label'), size_group, hbox)
        self.label = gtk.Entry()
        self.label.set_activates_default(True)
        hbox.pack_start(self.label)
        self.vbox.pack_start(hbox)

        # stock
        hbox = gtk.HBox(spacing=6)
        self._create_label(_('Stock'), size_group, hbox)
        self.stock_enabled = gtk.CheckButton()
        self.stock_enabled.connect('toggled', self._on_stock_check_toggled)
        hbox.pack_start(self.stock_enabled, False, False)

        stock_model = gtk.ListStore(str)
        stocks = gtk.stock_list_ids()
        stocks.sort()

        for stock in stocks:
            stock_model.append((stock,))

        self.stock = gtk.ComboBox(stock_model)
        renderer = gtk.CellRendererPixbuf()
        renderer.set_property('xalign', 0.0)
        self.stock.pack_start(renderer)
        self.stock.add_attribute(renderer, 'stock-id', 0)
        renderer = gtk.CellRendererText()
        renderer.set_property('xalign', 0.0)
        self.stock.pack_start(renderer)
        self.stock.add_attribute(renderer, 'text', 0)

        self.stock.set_wrap_width(3)
        
        self.stock.set_active(0)
        self.stock.set_sensitive(False)
        hbox.pack_start(self.stock)
        self.vbox.pack_start(hbox)

        # accelerator
        hbox = gtk.HBox(spacing=6)
        self._create_label(_('Accelerator'), size_group, hbox)
        self.accelerator = gtk.Entry()
        self.accelerator.set_activates_default(True)
        self.accelerator.set_text('Press a key combination')
        self.accelerator.connect('key-press-event',
                                        self._on_accelerator_entry_key_press)
        hbox.pack_start(self.accelerator)
        
        self.vbox.pack_start(hbox)

        # tooltip
        hbox = gtk.HBox(spacing=6)
        self._create_label(_('Tooltip'), size_group, hbox)
        self.tooltip = gtk.Entry()
        self.tooltip.set_activates_default(True)
        hbox.pack_start(self.tooltip)
        self.vbox.pack_start(hbox)

        # callback
        hbox = gtk.HBox(spacing=6)
        self._create_label(_('Callback'), size_group, hbox)
        self.callback = gtk.Entry()
        self.callback.set_activates_default(True)
        hbox.pack_start(self.callback)
        self.vbox.pack_start(hbox)
        
    def _create_label(self, text, size_group, hbox):
        label = gtk.Label(text+':')
        label.set_alignment(0.0, 0.5)
        size_group.add_widget(label)
        hbox.pack_start(label, False, False)

    def _on_stock_check_toggled(self, button):
        active = button.get_active()
        self.stock.set_sensitive(active)

    def _on_accelerator_entry_key_press(self, entry, event):
        if event.keyval == gtk.keysyms.Tab:
            return False

        if event.get_state() & gtk.gdk.CONTROL_MASK == gtk.gdk.CONTROL_MASK:
            msg = '<control>%s' % gtk.gdk.keyval_name(event.keyval)
            entry.set_text(msg)

        elif event.get_state() & gtk.gdk.MOD1_MASK == gtk.gdk.MOD1_MASK:
            msg = '<alt>%s' % gtk.gdk.keyval_name(event.keyval)
            entry.set_text(msg)

        return True

    def _clear_widgets(self):
        self.id.set_text('')
        self.label.set_text('')
        self.stock_enabled.set_active(False)
        self.stock.set_active(0)
        self.stock.set_sensitive(False)
        
        self.accelerator.set_text('')
        self.tooltip.set_text('')
        self.callback.set_text('')
        
    def _load_widgets(self):
        self.id.set_text(self._action.name)
        self.label.set_text(self._action.label)
        if self._action.stock_id is None:
            self.stock_enabled.set_active(False)
            self.stock.set_active(0)
            self.stock.set_sensitive(False)
        else:
            self.stock_enabled.set_active(True)
            model = self.stock.get_model()
            model_iter = model.get_iter_first()
            while model_iter:
                stock = model.get_value(model_iter, 0)
                if stock == self._action.stock_id:
                    self.stock.set_active_iter(model_iter)
                    break
                model_iter = model.iter_next(model_iter)
        self.accelerator.set_text(self._action.accelerator)
        self.tooltip.set_text(self._action.tooltip)
        self.callback.set_text(self._action.callback)
        
    def _fill_action(self):
        self._action.name = self.id.get_text()
        self._action.label = self.label.get_text()
        if self.stock_enabled.get_active():
            model = self.stock.get_model()
            stock = model.get_value(self.stock.get_active_iter(), 0)
            self._action.stock_id = stock
        else:
            self._action.stock_id = None
        self._action.accelerator = self.accelerator.get_text()
        self._action.tooltip = self.tooltip.get_text()
        self._action.callback = self.callback.get_text()
        
    def get_action(self):
        self._fill_action()
        return self._action
    
    def set_action(self, gaction):
        if gaction is None:
            self._clear_widgets()
            self._action = GAction(None)
        else:
            self._action = gaction
            self._load_widgets()

    action = property(get_action, set_action)
    
    def get_values(self):
        values = {}
        values['name'] = self.id.get_text()
        values['label'] = self.label.get_text()
        values['accelerator'] = self.accelerator.get_text()
        values['tooltip'] = self.tooltip.get_text()
        values['callback'] = self.callback.get_text()
        if self.stock_enabled.get_active():
            model = self.stock.get_model()
            stock = model.get_value(self.stock.get_active_iter(), 0)
            values['stock_id'] = stock
        else:
            values['stock_id'] = None

        return values
    
gobject.type_register(GActionDialog)

class GActionGroupDialog(gtk.Dialog):
    """This Dialog allows the creation and edition of a GActionGroup."""
    def __init__(self, toplevel=None, action_group=None):
        gtk.Dialog.__init__(self,
                            title='',
                            parent=toplevel,
                            flags=gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            buttons=(gtk.STOCK_OK, gtk.RESPONSE_OK,
                                     gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))

        self.set_border_width(6)
        self.set_resizable(False)
        self.set_has_separator(False)
        self.vbox.set_spacing(6)
        
        if action_group is None:
            self.set_title(_('Add Action Group'))
        else:
            self.set_title(_('Edit Action Group'))

        # id
        hbox = gtk.HBox(spacing=6)
        label = gtk.Label(_('Id')+':')
        label.set_alignment(0.0, 0.5)
        hbox.pack_start(label, False, False)
        self.id = gtk.Entry()
        self.id.set_activates_default(True)
        hbox.pack_start(self.id)
        self.vbox.pack_start(hbox)

        self.action_group = action_group

        self.set_default_response(gtk.RESPONSE_OK)
        
        self.vbox.show_all()

    def _clear_widgets(self):
        self.id.set_text('')

    def _load_widgets(self):
        self.id.set_text(self._action_group.name)
        
    def _fill_action_group(self):
        self._action_group.name = self.id.get_text()
        
    def get_action_group(self):
        self._fill_action_group()
        return self._action_group
    
    def set_action_group(self, action_group):
        if action_group is None:
            self._clear_widgets()
            self._action_group = GActionGroup(None)
        else:
            self._action_group = action_group
            self._load_widgets()
            
    action_group = property(get_action_group, set_action_group)
    
    def get_action_group_name(self):
        return self.id.get_text()

gobject.type_register(GActionGroupDialog)
