/*
 *    WMAcpiLoad - A dockapp to monitor ACPI status
 *    Copyright (C) 2002  Thomas Nemeth <tnemeth@free.fr>
 *
 *    Patch by Alan Carriou <cariou_alan@yahoo.fr> (C) 2004-2005
 *
 *    Based on work by Seiichi SATO <ssato@sh.rim.or.jp>
 *    Copyright (C) 2001,2002  Seiichi SATO <ssato@sh.rim.or.jp>
 *    and on work by Mark Staggs <me@markstaggs.net>
 *    Copyright (C) 2002  Mark Staggs <me@markstaggs.net>

 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.

 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.

 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <sys/stat.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>              /* for isspace() */

#include "lib_acpi.h"
#include "wmacpiload.h"
#include "battery.h"
#include "thermal.h"
#include "ac_adapter.h"

void
acpi_detect_devices(AcpiInfos *k)
{
    char dir[FILENAME_MAX + 1];
    int n;

    if (k) {
        if (batteries_enabled) {
            n = snprintf(dir, FILENAME_MAX + 1, "%s%s",
                         "/proc/acpi", battery_dir);
            if (n > FILENAME_MAX) {
                fprintf(stderr, "%s : file name too long, "
                        "battery detection disabled\n", PACKAGE);
                k->bat = NULL;
            } else {
                k->bat = acpi_detect(dir, &battery_init);
            }

            if (!k->bat)
                fprintf(stderr, "%s : no battery slot detected\n",
                        PACKAGE);
            else if (verbose)
                printf("Monitoring battery %s.\n", k->bat->name);
        } else {
            k->bat = NULL;
            printf("Battery detection disabled by user.\n");
        }

        n = snprintf(dir, FILENAME_MAX + 1, "%s%s", "/proc/acpi",
                     ac_adapter_dir);
        if (n > FILENAME_MAX) {
            fprintf(stderr, "%s : file name too long, "
                    "AC adapter detection disabled\n", PACKAGE);
            k->ac = NULL;
        } else {
            k->ac = acpi_detect(dir, &ac_adapter_init);
        }
        if (!k->ac)
            fprintf(stderr, "%s : no AC adapter detected\n", PACKAGE);

        n = snprintf(dir, FILENAME_MAX + 1, "%s%s", "/proc/acpi",
                     thermal_zone_dir);
        if (n > FILENAME_MAX) {
            fprintf(stderr, "%s : file name too long, "
                    "Thermal zone detection disabled\n", PACKAGE);
            k->thermal = NULL;
        } else {
            k->thermal = acpi_detect(dir, &thermal_zone_init);
        }
        if (!k->thermal)
            fprintf(stderr, "%s : no thermal zone detected\n", PACKAGE);
        else if (verbose)
            printf("Monitoring thermal zone %s.\n", k->thermal->name);

        k->bat_first = k->bat;
        k->thermal_first = k->thermal;
    }
}

/*
 * Used within acpi_detect to select "valid" device names, i.e. exclude ".",
 * ".." and, if asked, some extra device.
 */
static int
valid_name(const struct dirent *f)
{
    int valid = 1; /* default situation: valid name */

    if (f) {
        valid = (strcmp(f->d_name, ".") != 0) && (strcmp(f->d_name, "..") != 0);

        if (valid != 0 && ignored_device != NULL) {
            valid = strcmp(f->d_name, ignored_device) != 0;
            if (!valid && verbose) {
                printf("ACPI device %s ignored\n", ignored_device);
            }
        }
    }

    return valid;
}

/*
 * This function uses a generic way to detect acpi devices (idea taken from
 * the "acpi client for Linux" source code, written by Grahame Bowland).
 * The fact is that detecting battery slots or ac adapters follows almost the
 * same algorithm, but the data definitions are different... So I've chosen
 * a somehow object-oriented approach : this function is the skeleton to
 * device detection, and calls a specialized function to make all the specific
 * stuff.
 */
void *
acpi_detect(char *dir, void *(*device_init) (struct dirent *))
{
    struct dirent *fd;
    struct stat sts;
    DIR *d;
    acpi_device *first, *p, *prev;
    char device[FILENAME_MAX + 1];
    int n;

    if (!dir)
        return NULL;

    if (!device_init) {
        fprintf(stderr, "%s : Nothing to handle device type.\n", PACKAGE);
        return NULL;
    }

    d = opendir(dir);
    if (!d) {
        fprintf(stderr, "%s : Couldn't access %s directory\n", PACKAGE,
                dir);
        return NULL;
    }
    p = prev = first = NULL;

    while ((fd = readdir(d))) {
        /* excludes "." and ".." */
        if (!valid_name(fd))
            continue;

        /* checks if *fd is a directory */
        if (dir[strlen(dir) - 1] == '/')
            n = snprintf(device, FILENAME_MAX + 1, "%s%s", dir,
                         fd->d_name);
        else
            n = snprintf(device, FILENAME_MAX + 1, "%s/%s", dir,
                         fd->d_name);

        if (n > FILENAME_MAX)
            continue;
        if (stat(device, &sts) != 0)
            continue;
        if (!S_ISDIR(sts.st_mode))
            continue;

        /* malloc()s and initializes p with the info found in fd */
        p = (*device_init) (fd);
        if (!p)
            continue;

        if (first == NULL) {
            first = p;
        } else {
            /* The following line works because "next" is always the first
             * field of the structs we have put in the acpi_device union.
             */
            prev->next = p;
        }
        prev = p;
    }
    closedir(d);
    return first;
}

char *
strcat4(char *root, char *dir, char *device, char *file)
{
    size_t size;
    char *dest;

    size = strlen(root) + strlen(dir) + strlen(device) + strlen(file) + 1;
    dest = malloc(sizeof(*dest) * size);
    if (dest == NULL)
        fprintf(stderr,
                "%s : couldn't allocate memory for device '%s'\n",
                PACKAGE, device);
    else
        sprintf(dest, "%s%s%s%s", root, dir, device, file);
    return dest;
}

int
acpi_exists(void)
{
    int res;

    if (access(ACPIDEV, F_OK))
        res = 1;
    else if (access(ACPIDEV, R_OK))
        res = 2;                /* ACPI exists but we are not allowed to use it */
    else
        res = 0;

    return res;
}

void
acpi_update_status(AcpiInfos *i)
{
    if (i != NULL) {
        AC_adapter *acad;

        thermal_zone_update_status(i->thermal);

        i->AC_power = Off_line;
        for (acad = i->ac; acad; acad = acad->next) {
            ac_adapter_update_status(acad);
            /* we want to know if any of the acads is on-line */
            if (acad->status == On_line)
                i->AC_power = On_line;
        }

        battery_update_plugged(i->bat);
        battery_update_status(i->bat);
    }
}

void
acpi_cleanup(void)
{
    Battery *bat, *bat_prev;
    Thermal_Zone *tz, *tz_prev;
    AC_adapter *acad, *acad_prev;

    /* First, clean battery information */
    bat = cur_acpi_infos.bat_first;
    while (bat) {
        bat_prev = bat;
        bat = bat->next;
        battery_free(bat_prev);
    }
    cur_acpi_infos.bat = NULL;
    cur_acpi_infos.bat_first = NULL;

    /* Then, thermal zones */
    tz = cur_acpi_infos.thermal_first;
    while (tz) {
        tz_prev = tz;
        tz = tz->next;
        thermal_zone_free(tz_prev);
    }
    cur_acpi_infos.thermal = NULL;
    cur_acpi_infos.thermal_first = NULL;

    /* And, finally, AC adapters... */
    acad = cur_acpi_infos.ac;
    while (acad) {
        acad_prev = acad;
        acad = acad->next;
        ac_adapter_free(acad_prev);
    }
    cur_acpi_infos.ac = NULL;
}

int
acpi_check_alarm(AcpiInfos *i)
{
    int res = 0; /* default: we're not in an alarm situation */

    if (i != NULL) {
        if (i->bat && i->bat->percentage <= i->bat_alarm_level)
            res = 1;
 
        if (res == 0 && i->thermal && i->thermal->temp >= i->thermal_alarm_level)
            res = 1;
    }

    return res;
}

/* used only in parse_file */
#define MIN(a,b) (((a) < (b)) ? (a) : (b))

char *
parse_file(char *file_name, char *searched, char *dest, size_t dest_size)
{
    /* there won't be lines longer than 511 chars in those files, will it ? */
    /* static */ char buffer[512];
    FILE *fd;
    char *res;
    size_t n;

    fd = NULL;
    res = NULL;
    n = strlen(searched);

    if (dest != NULL && dest_size > 0
        && n > 0 && n < (sizeof buffer) - 1
        && file_name != NULL && (fd = fopen(file_name, "r")) != NULL) {
        while (fgets(buffer, sizeof buffer, fd) != NULL)
            if (strncmp(buffer, searched, n) == 0) {
                char *src;
                char *end;
                size_t len;

                /* will not go out of buffer */
                for (src = buffer + n; isspace(*src) != 0; src++);

                len = strlen(src);
                if (len > 0) {
                    for (end = &src[len - 1];
                         isspace(*end) != 0 && end > src; end--);

                    end[1] = '\0';

                    len = end - src + 1;
                }
                strncpy(dest, src, MIN(dest_size, len + 1));
                dest[dest_size - 1] = '\0';
                res = dest;
                break;
            }
        fclose(fd);
    }
    return res;
}
