//------------------------------ $Keywords ----------------------------------
// GPGee - GNU Privacy Guard Explorer Extension
// GPGeeContextMenuExtensionImpl.cpp - Explorer extension COM code
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of GPGee.
//
// GPGee is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// GPGee is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 1 $ $Revision: 7 $
/*
$History: **** V 1.0 by kfitzner ****
$History: * gpgeecontextmenuextensionimpl.cpp - 2005-05-13 11:14:15 PM - 15785 Bytes
$History: * gpgeecontextmenuextensionimpl.h - 2005-05-07 3:13:04 PM - 3805 Bytes
$History: * Initial check-in
$History: **** V 1.1 by kfitzner ****
$History: * gpgeecontextmenuextensionimpl.cpp - 2005-08-08 6:48:55 AM - 15997 Bytes
$History: * gpgeecontextmenuextensionimpl.h - 2005-08-08 6:36:16 AM - 3741 Bytes
$History: * License change - remove option for later versions of GPL
$History: **** V 1.2 by kfitzner ****
$History: * gpgeecontextmenuextensionimpl.cpp - 2005-08-31 6:58:23 PM - 16300 Bytes
$History: * gpgeecontextmenuextensionimpl.h - 2005-08-08 6:36:16 AM - 3741 Bytes
$History: * Add version checking
$History: **** V 1.3 by kfitzner ****
$History: * gpgeecontextmenuextensionimpl.cpp - 2005-12-11 3:31:03 PM - 17666 Bytes
$History: * gpgeecontextmenuextensionimpl.h - 2005-12-11 1:50:40 PM - 3785 Bytes
$History: * Add separate symetrical * PK encrypt menu entries, 
$History: * directory recursion, speed up filename processing
$History: **** V 1.4 by kfitzner ****
$History: * gpgeecontextmenuextensionimpl.cpp - 2005-12-12 11:34:49 AM - 17930 Bytes
$History: * gpgeecontextmenuextensionimpl.h - 2005-12-12 11:30:54 AM - 3785 Bytes
$History: * Fix bug where configure menu doesn't appear
$History: **** V 1.5 by kfitzner ****
$History: * gpgeecontextmenuextensionimpl.cpp - 2005-12-15 11:01:33 AM - 18221 Bytes
$History: * gpgeecontextmenuextensionimpl.h - 2005-12-12 11:30:54 AM - 3785 Bytes
$History: * Lower-case all extensions rather than use canse insensitive 
$History: * comparisons
$History: **** V 1.6 by kfitzner ****
$History: * gpgeecontextmenuextensionimpl.cpp - 2005-12-16 5:08:11 PM - 18324 Bytes
$History: * gpgeecontextmenuextensionimpl.h - 2005-12-12 11:30:54 AM - 3785 Bytes
$History: * Move directory recursing from here to the sign or 
$History: * verify form's processing loop so that right-clicking 
$History: * on a large directory doesn't take forever.
$History: **** Latest ** V 1.7 by kfitzner ** 2006-01-02 2:48:18 PM ****
$History: * Move deletion of FileNames object to each form.   
$History: * At some point, the encryption and decryption forms 
$History: * will be created from their own threads.
*/
//----------------------------  $NoKeywords ---------------------------------


//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 21 Mar 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// The main COM server implementation is here.  This code sets up the
// Windows explorer context menu.  Also, through some black magic, the
// DLL's DllRegisterServer() call ends up calling the UpdateRegistry()
// method below.
//
// The basic framework for how to make an explorer extension is convered
// in a tutorial available from Borland's developer network web site at
// http://bdn.borland.com/article/0,1410,26650,00.html
// The article is titled "Context Menu Shell Extensions Using COM" - it was
// helpful though it infers that creating a submenu of the context menu is
// only possible using owner draw menu items, which is not true (as is shown
// below)
//---------------------------------------------------------------------------

// GPGEECONTEXTMENUEXTENSIONIMPL : Implementation of TGPGeeContextMenuExtensionImpl (CoClass: GPGeeContextMenuExtension, Interface: IGPGeeContextMenuExtension)

#include <vcl.h>
#pragma hdrstop

#include "GPGeeSignEncrypt.h"
#include "GPGeeVerifyDecrypt.h"
#include "GPGeeUtility.h"
#include "GPGeeConfigure.h"
#include "TConfiguration.h"
#include "TProgramLog.h"
#include "GPGEECONTEXTMENUEXTENSIONIMPL.H"
//---------------------------------------------------------------------------
extern TConfiguration *GPGeeConfig;
//---------------------------------------------------------------------------

/////////////////////////////////////////////////////////////////////////////
// TGPGeeContextMenuExtensionImpl

//---------------------------------------------------------------------------
// SLOW, Construction ahead.
//
TGPGeeContextMenuExtensionImpl::TGPGeeContextMenuExtensionImpl() {
  __ENTERFUNCTION__;
  FileNames = NULL;
  MenuKeyBitmap = NULL;
  MenuSignEncryptBitmap = NULL;
  MenuSignBitmap = NULL;
  MenuEncryptBitmap = NULL;
  MenuVerifyDecryptBitmap = NULL;
  MenuConfigureBitmap = NULL;

  // FileNames = new TStringList;
  MenuKeyBitmap = LoadBitmap(HInstance,"Key");
  MenuSignEncryptBitmap = LoadBitmap(HInstance, "SignEncrypt");
  MenuSignBitmap = LoadBitmap(HInstance, "Sign");
  MenuEncryptBitmap = LoadBitmap(HInstance, "Encrypt");
  MenuVerifyDecryptBitmap = LoadBitmap(HInstance, "VerifyDecrypt");
  MenuConfigureBitmap = LoadBitmap(HInstance, "Configure");
  __RETURNFUNCTION__;
}  // TGPGeeContextMenuExtensionImpl::TGPGeeContextMenuExtensionImpl()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//  End Construction Zone
//
TGPGeeContextMenuExtensionImpl::~TGPGeeContextMenuExtensionImpl() {
  __ENTERFUNCTION__;
  if (FileNames)
    delete FileNames;
  if (MenuKeyBitmap)
    DeleteObject(MenuKeyBitmap);
  if (MenuSignEncryptBitmap)
    DeleteObject(MenuSignBitmap);
  if (MenuSignBitmap)
    DeleteObject(MenuSignBitmap);
  if (MenuEncryptBitmap)
    DeleteObject(MenuEncryptBitmap);
  if (MenuVerifyDecryptBitmap)
    DeleteObject(MenuVerifyDecryptBitmap);
  if (MenuConfigureBitmap)
    DeleteObject(MenuConfigureBitmap);
  __RETURNFUNCTION__;
}  // TGPGeeContextMenuExtensionImpl::~TGPGeeContextMenuExtensionImpl()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// (Un)Register the object
//
HRESULT WINAPI TGPGeeContextMenuExtensionImpl::UpdateRegistry(BOOL bRegister)
{
  HRESULT retval;

  __ENTERFUNCTION__;

  if (bRegister) {
    LOG(LOG_MESSAGE, "Registering GPGee.dll.");
    CreateRegKey("*\\shellex", "", "");
    CreateRegKey("*\\shellex\\ContextMenuHandlers", "", "");
    CreateRegKey("*\\shellex\\ContextMenuHandlers\\GPGee", "", Comobj::GUIDToString(GetObjectCLSID()));
    CreateRegKey("Directory", "", "");
    CreateRegKey("Directory\\shellex", "", "");
    CreateRegKey("Directory\\shellex\\ContextMenuHandlers", "", "");
    CreateRegKey("Directory\\shellex\\ContextMenuHandlers\\GPGee", "", Comobj::GUIDToString(GetObjectCLSID()));
  } else {
    LOG(LOG_MESSAGE, "Unregistering GPGee.dll.");
    DeleteRegKey("*\\shellex\\ContextMenuHandlers\\GPGee");
    DeleteRegKey("Directory\\shellex\\ContextMenuHandlers\\GPGee");
  }  // if (bRegister)
  TTypedComServerRegistrarT<TGPGeeContextMenuExtensionImpl>
  regObj(GetObjectCLSID(), GetProgID(), GetDescription());
  retval = regObj.UpdateRegistry(bRegister);

  __RETURNFUNCTION(retval);
}  // HRESULT WINAPI TGPGeeContextMenuExtensionImpl::UpdateRegistry()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//
// IShellExtInit
//
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Initialize method for IShellExtInit
//
// Get the filenames of all the files the user is invoking the context menu
// over and store them in a TStringList.  This is then passed to whichever
// form will be handling them (either the sign/encrypt or verify/decrypt
// form).
//
STDMETHODIMP TGPGeeContextMenuExtensionImpl::Initialize(LPCITEMIDLIST pidlFolder, LPDATAOBJECT lpdobj, HKEY hkeyProgID)
{

  HDROP hDrop;
  HRESULT hResult;
  FORMATETC fmtEtc;
  STGMEDIUM medium;
  UINT uFileCount;
  char *FileName;
  AnsiString Extension;

  __ENTERFUNCTION__;

  fmtEtc.cfFormat = CF_HDROP;
  fmtEtc.ptd      = NULL;
  fmtEtc.dwAspect = DVASPECT_CONTENT;
  fmtEtc.lindex   = -1;
  fmtEtc.tymed    = TYMED_HGLOBAL;

  medium.tymed    = TYMED_HGLOBAL;

  if ((hResult = lpdobj->GetData(&fmtEtc,&medium)) < 0)
    return E_INVALIDARG;

  if ((hDrop = (HDROP)GlobalLock(medium.hGlobal)) == NULL)
    return E_INVALIDARG;

  FileName = (char *)SysGetMem(MAX_PATH);
  uFileCount = DragQueryFile(hDrop, 0xFFFFFFFF, NULL, 0);
  LOG1(LOG_DEBUG, "%d file(s) in the processing list.", uFileCount);
  AllFilesAreGPG = true;
  if (uFileCount) {
    FileNames = new TStringList();
    // FileNames->Clear();
    for (UINT n=0; n < uFileCount; n++) {
      DragQueryFile(hDrop, n, FileName, MAX_PATH);
      FileNames->Add(FileName);
      if (AllFilesAreGPG) {
        Extension = ExtractFileExt(String(FileName)).LowerCase();
        if (Extension != ".pgp" && Extension != ".gpg" && Extension != ".asc" && Extension != ".sig")
          AllFilesAreGPG = false;
      }  // if (AllFilesAreGPG)
    }  // for (UINT n=0; n < uFileCount; n++)
    hResult = S_OK;
  } else {
    hResult = E_INVALIDARG;
  }  // if (uFileCount)

  GlobalUnlock(medium.hGlobal);
  ReleaseStgMedium(&medium);
  SysFreeMem(FileName);

  __RETURNFUNCTION(hResult);
}  // TGPGeeContextMenuExtensionImpl::Initialize()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//
// IContextMenu
//
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// GetCommandString()
// Set the help string associated with our menu entries
// FIXME: Put something here
//
STDMETHODIMP TGPGeeContextMenuExtensionImpl::GetCommandString(UINT idCmd, UINT uFlags, UINT* pwReserved, LPSTR pszName, UINT cchMax)
{
/*  HRESULT hResult;
  USES_CONVERSION;*/

  return S_OK;
}  // STDMETHODIMP TGPGeeContextMenuExtensionImpl::GetCommandString()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// QueryContextMenu()
// Add our items to the context menu
//
STDMETHODIMP TGPGeeContextMenuExtensionImpl::QueryContextMenu(HMENU hMenu, UINT indexMenu, UINT idCmdFirst, UINT idCmdLast, UINT uFlags)
{
  HMENU hPopup;
  STDMETHODIMP retval;

  __ENTERFUNCTION__;

  if (uFlags & CMF_DEFAULTONLY || !FileNames->Count)
    return MAKE_HRESULT(SEVERITY_SUCCESS, FACILITY_NULL, 0);

  hPopup = CreatePopupMenu();
  if (hPopup) {
    if (AllFilesAreGPG) {
      LOG(LOG_DEBUG, "Creating verify/decrypt popup menu.");
      InsertMenu(hPopup, GPGEE_VERIFYDECRYPT, MF_BYPOSITION|MF_STRING, idCmdFirst++, GetMessage(MSG_MENU_VERIFYDECRYPT).c_str());
      InsertMenu(hPopup, GPGEE_SEP1,          MF_SEPARATOR|MF_BYPOSITION, idCmdFirst++, NULL);
      InsertMenu(hPopup, GPGEE_DCONFIGURE,    MF_BYPOSITION|MF_STRING, idCmdFirst++, GetMessage(MSG_MENU_CONFIGURE).c_str());
      SetMenuItemBitmaps(hPopup, GPGEE_VERIFYDECRYPT, MF_BYPOSITION, MenuVerifyDecryptBitmap, MenuVerifyDecryptBitmap);
      SetMenuItemBitmaps(hPopup, GPGEE_DCONFIGURE,    MF_BYPOSITION, MenuConfigureBitmap, MenuConfigureBitmap);
      retval = MAKE_HRESULT(SEVERITY_SUCCESS, FACILITY_NULL, 3);
    } else {
      LOG(LOG_DEBUG, "Creating sign/encrypt popup menu.");
      InsertMenu(hPopup, GPGEE_SIGNENCRYPT,   MF_BYPOSITION|MF_STRING, idCmdFirst++, GetMessage(MSG_MENU_SIGNENCRYPT).c_str());
      InsertMenu(hPopup, GPGEE_SIGN,          MF_BYPOSITION|MF_STRING, idCmdFirst++, GetMessage(MSG_MENU_SIGN).c_str());
      InsertMenu(hPopup, GPGEE_ENCRYPT,       MF_BYPOSITION|MF_STRING, idCmdFirst++, GetMessage(MSG_MENU_ENCRYPT).c_str());
      InsertMenu(hPopup, GPGEE_SENCRYPT,      MF_BYPOSITION|MF_STRING, idCmdFirst++, GetMessage(MSG_MENU_SENCRYPT).c_str());
      InsertMenu(hPopup, GPGEE_SEP2,          MF_SEPARATOR|MF_BYPOSITION, idCmdFirst++, NULL);
      InsertMenu(hPopup, GPGEE_ECONFIGURE,    MF_BYPOSITION|MF_STRING, idCmdFirst++, GetMessage(MSG_MENU_CONFIGURE).c_str());
      SetMenuItemBitmaps(hPopup, GPGEE_SIGNENCRYPT,   MF_BYPOSITION, MenuSignEncryptBitmap, MenuSignEncryptBitmap);
      SetMenuItemBitmaps(hPopup, GPGEE_SIGN,          MF_BYPOSITION, MenuSignBitmap, MenuSignBitmap);
      SetMenuItemBitmaps(hPopup, GPGEE_ENCRYPT,       MF_BYPOSITION, MenuEncryptBitmap, MenuEncryptBitmap);
      SetMenuItemBitmaps(hPopup, GPGEE_SENCRYPT,      MF_BYPOSITION, MenuEncryptBitmap, MenuEncryptBitmap);
      SetMenuItemBitmaps(hPopup, GPGEE_ECONFIGURE,    MF_BYPOSITION, MenuConfigureBitmap, MenuConfigureBitmap);
      retval = MAKE_HRESULT(SEVERITY_SUCCESS, FACILITY_NULL, 6);
    }  // if (AllFilesAreGPG)
    // InsertMenu(hPopup, GPGEE_VERIFY,      MF_BYPOSITION|MF_STRING, idCmdFirst++, "Decrypt/Verify");
  }  // if (hPopup)

  InsertMenu(hMenu, indexMenu++, MF_SEPARATOR|MF_BYPOSITION, 0, NULL);
  InsertMenu(hMenu, indexMenu, MF_STRING|MF_BYPOSITION|MF_POPUP, (UINT)hPopup, "GPGee");
  SetMenuItemBitmaps(hMenu, indexMenu, MF_BYPOSITION, MenuKeyBitmap, MenuKeyBitmap);

  __RETURNFUNCTION(retval);
}  // STDMETHODIMP TGPGeeContextMenuExtensionImpl::QueryContextMenu()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// InvokeCommand
// Handler for the context menu items
//
STDMETHODIMP TGPGeeContextMenuExtensionImpl::InvokeCommand(LPCMINVOKECOMMANDINFO lpici)
{
  WORD cmdId = LOWORD(lpici->lpVerb);
  HWND handle;

  __ENTERFUNCTION__;

  if (HIWORD(lpici->lpVerb))
    return E_FAIL;

  LOG1(LOG_DEBUG, "Processing command id=%d.", cmdId);
  if (AllFilesAreGPG) {
    switch(cmdId) {
      case GPGEE_VERIFYDECRYPT: HandleVerifyDecrypt(); break;
      case GPGEE_DCONFIGURE:    HandleConfigure(); break;
      default:
        return E_FAIL;
    }  // switch(cmdId))
  } else {
    switch(cmdId) {
      case GPGEE_SIGNENCRYPT:   HandleSignEncrypt(true, true, false);  break;
      case GPGEE_SIGN:          HandleSignEncrypt(true, false, false); break;
      case GPGEE_ENCRYPT:       HandleSignEncrypt(false, true, false); break;
      case GPGEE_SENCRYPT:      HandleSignEncrypt(false, false, true); break;
      case GPGEE_ECONFIGURE:    HandleConfigure(); break;
      default:
        return E_FAIL;
    }  // switch(cmdId)
  }  // if (AllFilesAreGPG)

  __RETURNFUNCTION(S_OK);
}  // STDMETHODIMP TGPGeeContextMenuExtensionImpl::InvokeCommand()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Handle signing and encrypting - create the form and then pass the buck
//
void __fastcall TGPGeeContextMenuExtensionImpl::HandleSignEncrypt(bool bSign, bool bPKEncrypt, bool bSEncrypt)
{
  __ENTERFUNCTION__;
  try {
    CheckVersion();
    formGPGeeSignEncrypt = new TformGPGeeSignEncrypt(NULL);
    formGPGeeSignEncrypt->ShowModalWithDefaults(bSign, bPKEncrypt, bSEncrypt, FileNames);
    formGPGeeSignEncrypt->Release();
    // delete FileNames;
    FileNames = NULL;
  }  // try
  catch (Exception &e)
  {
    LOG(LOG_ERROR, "Exception " + e.ClassName() + " raised with message " + e.Message);
    Application->MessageBox(e.Message.c_str(),"GPGee Exception",MB_OK);
  }
  __RETURNFUNCTION__;
}  // __fastcall TGPGeeContextMenuExtensionImpl::HandleSignEncrypt(bool bSign, bool bEncrypt)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Handle verifying and decrypting
//
void __fastcall TGPGeeContextMenuExtensionImpl::HandleVerifyDecrypt(void)
{
  __ENTERFUNCTION__;
  try {
    CheckVersion();
    formGPGeeVerifyDecrypt = new TformGPGeeVerifyDecrypt(NULL);
    formGPGeeVerifyDecrypt->ShowModalWithDefaults(FileNames);
    formGPGeeVerifyDecrypt->Release();
    // delete FileNames;
    FileNames = NULL;
  }  // try
  catch (Exception &e)
  {
    LOG(LOG_ERROR, "Exception " + e.ClassName() + " raised with message " + e.Message);
    Application->MessageBox(e.Message.c_str(),"GPGee Exception",MB_OK);
  }
  __RETURNFUNCTION__;
}  // __fastcall TGPGeeContextMenuExtensionImpl::HandleSignEncrypt(bool bSign, bool bEncrypt)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Handle configuration of the extension
//
void __fastcall TGPGeeContextMenuExtensionImpl::HandleConfigure(void)
{
  __ENTERFUNCTION__;
  try {
    CheckVersion();
    formGPGeeConfigure = new TformGPGeeConfigure(NULL);
    formGPGeeConfigure->ShowModal();
    formGPGeeConfigure->Release();
  }  // try
  catch (Exception &e)
  {
    LOG(LOG_ERROR, "Exception " + e.ClassName() + " raised with message " + e.Message);
    Application->MessageBox(e.Message.c_str(),"GPGee Exception",MB_OK);
  }
  __RETURNFUNCTION__;
}  // void __fastcall TGPGeeContextMenuExtensionImpl::HandleConfigure(void)
//---------------------------------------------------------------------------

