//------------------------------ $Keywords ----------------------------------
// GPGee - GNU Privacy Guard Explorer Extension
// GPGeeVerifyDecrypt.cpp - Verification form code
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of GPGee.
//
// GPGee is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// GPGee is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 1 $ $Revision: 4 $
/*
$History: **** V 1.0 by kfitzner ****
$History: * gpgeeverifydecrypt.ddp - 2005-05-13 5:48:24 PM - 1892 Bytes
$History: * gpgeeverifydecrypt.cpp - 2005-05-13 11:14:16 PM - 37140 Bytes
$History: * gpgeeverifydecrypt.dfm - 2005-05-06 7:36:54 AM - 34977 Bytes
$History: * gpgeeverifydecrypt.h - 2005-05-06 7:36:32 AM - 4258 Bytes
$History: * Initial check-in
$History: **** V 1.1 by kfitzner ****
$History: * gpgeeverifydecrypt.cpp - 2005-05-17 2:16:34 AM - 35891 Bytes
$History: * gpgeeverifydecrypt.dfm - 2005-05-16 11:30:10 PM - 34977 Bytes
$History: * gpgeeverifydecrypt.h - 2005-05-16 11:30:10 PM - 4216 Bytes
$History: * gpgeeverifydecrypt.ddp - 2005-05-17 1:19:02 AM - 33 Bytes
$History: * Add persistent key cache
$History: **** V 1.2 by kfitzner ****
$History: * gpgeeverifydecrypt.cpp - 2005-07-29 2:51:36 PM - 38244 Bytes
$History: * gpgeeverifydecrypt.dfm - 2005-05-16 11:30:10 PM - 34977 Bytes
$History: * gpgeeverifydecrypt.h - 2005-07-28 9:17:40 PM - 4391 Bytes
$History: * gpgeeverifydecrypt.ddp - 2005-07-28 10:17:22 PM - 33 Bytes
$History: * Lock passphrase memory pages to prevent swapping. 
$History: * and add some limited passphrase caching.
$History: **** V 1.3 by kfitzner ****
$History: * gpgeeverifydecrypt.cpp - 2005-08-08 6:50:33 AM - 38581 Bytes
$History: * gpgeeverifydecrypt.dfm - 2005-05-16 11:30:10 PM - 34977 Bytes
$History: * gpgeeverifydecrypt.h - 2005-08-08 6:35:46 AM - 4327 Bytes
$History: * gpgeeverifydecrypt.ddp - 2005-08-08 6:35:48 AM - 33 Bytes
$History: * License change - remove option for later versions of GPL
$History: **** Latest ** V 1.4 by kfitzner ** 2005-09-05 6:00:54 AM ****
$History: * Make passphrase callback function static - also added 
$History: * ability to verify multi-signed signatures
*/
//----------------------------  $NoKeywords ---------------------------------


//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 27 Apr 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// This is the file verification and decryption form.  Both the UI code and
// operational code are here.
//
// Unlike the sign/encrypt code which performs a single operation (although
// it can do it on multiple files), this code attempts to analyze each file
// to determine the correct operation to perform.  Thus it can verify a
// detached signature, an attached one, pk decrypt, symmetrical decrypt, etc.
// The key to this is determining what operation to perform.  This is
// dependent on what type of gpg file each file is.  The determination of the
// type is actually done externally by a function in the GPGeeUtility.cpp
// unit.  It gives us either GPG_SIGNATURE or GPG_MESSAGE.  The former is a
// detached signature - it is fed to gpgme_op_file_verify().  The latter can
// be many different things - an encrypted file (symmetrically or pk), or a
// standalone signed file (binary or clearsigned).  While it would be nice
// to further categorize the type (so we don't end up just telling the user
// it's a "message"), it isn't actually necesary since GnuPG will handle any
// of those with the --decrypt switch.  That means that the
// gpgme_op_file_decrypt() function can handle them.
//
// For this form we set up a passphrase callback function.  This is what lets
// us handle a standalone signed-file and an encrypted file the same way.  We
// don't care which of the two it is - if it's an ecnrypted file that needs
// a passphrase, the callback function will get called and we can ask for the
// passphrase at that time.  There is no password caching done, which is
// something we might want to think about in the future.  My preference is
// that we get some code in GPGME that will allow us to figure out exactly
// what sort of file any particular gpg file is.  This way we wouldn't need
// to do much password caching at all.  We could sort the files by their type
// and do all the operations that need a password for a given key at the same
// time - so only that one passphrase would need caching.  It would also
// allow us to proactively ask for a passphrase rather than needing to do it
// on request from the callback.  This would eliminate some issues with the
// callback (see the noted for the callback function).
//
// 25 July 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// Adding in some passphrase caching.  I've resisted this because I think
// the "right way" is not to cache but to sort (see above), and because
// storing passphrases safely is tricky.  But, until the needed functionality
// is in GPGME, no reasong that GPGee users should be stuck.
//
// I'm going to try and make sure all the memory used to cache a passphrase
// is locked so that Windows can't page it out to the swap file.  This should
// make caching relatively safe.
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "TConfiguration.h"
#include "GPGeeExceptions.h"
#include "GPGeePassPhrase.h"
#include "TProgramLog.h"
#include "GPGeeVerifyDecrypt.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TformGPGeeVerifyDecrypt *formGPGeeVerifyDecrypt;
extern TConfiguration *GPGeeConfig;
extern TStringList *PassphraseCache;
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Form constructor
//
__fastcall TformGPGeeVerifyDecrypt::TformGPGeeVerifyDecrypt(TComponent* Owner): TForm(Owner)
{
  gpgme_error_t err;
  AnsiString sConfigString;

  __ENTERFUNCTION__;

  PassphraseCache = NULL;

  ctxGPG = NULL;
  cacheSecret = NULL;
  cachePublic = NULL;

  bCancel = false;

  try {
    // Initialize the passphrase cache
    PassphraseCache = new TStringList();

    // Initialize the GPGME library main and keycache contexts
    LOG(LOG_DEBUG, "Allocating the GPGME main context.");
    err = gpgme_new(&ctxGPG);
    if (err)
      throw EGPGMEError(FormMessage(MSG_ERROR_GPGME_LIB_INIT, gpgme_strerror(err)));

    // Set GPGME's settings to match GPGee's configuration
    LOG(LOG_DEBUG, "Setting the GPGME configuration.");
    gpgme_control(ctxGPG, GPGME_CTRL_FORCETRUST, GPGeeConfig->Values["Force Trust"]?1:0);
    sConfigString = GPGeeConfig->Values["GPG Program"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_GPGPROGRAM, sConfigString.c_str());
    sConfigString = GPGeeConfig->Values["Options File"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_OPTFILE, sConfigString.c_str());
    sConfigString = GPGeeConfig->Values["Keyring"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_PUBRING, sConfigString.c_str());
    sConfigString = GPGeeConfig->Values["Secret Keyring"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_SECRING, sConfigString.c_str());

    // Cache the secret keys - this cache is used when a passphrase for one of these keys is needed - then we get all the
    // info about the key from the cache to present it to the user.  The public keys aren't cached here because they are
    // so seldomly needed.
    LOG(LOG_DEBUG, "Initializing the secret key cache.");
    cacheSecret = GetKeyCache(true);

    // Set the passphrase callback function.  This can't be a method because of C++ limitations (bah), so we supply a
    // pointer to this form to the callback so that a class method can be invoked from the callback.
    LOG(LOG_DEBUG, "Setting the passphrase callback function.");
    gpgme_set_passphrase_cb(ctxGPG, PassphraseCallback, this);

  }  // try
  catch(Exception &e) {
    LOG(LOG_ERROR, "Exception " + e.ClassName() + " raised with message " + e.Message);
    Cleanup();
    __LEAVEFUNCTION__;
    throw;
  }
  __RETURNFUNCTION__;
}  // __fastcall TformGPGeeVerifyDecrypt::TformGPGeeVerifyDecrypt(TComponent* Owner): TForm(Owner)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Death and destruction
//
__fastcall TformGPGeeVerifyDecrypt::~TformGPGeeVerifyDecrypt()
{
  __ENTERFUNCTION__;
  Cleanup();
  __RETURNFUNCTION__;
}  // __fastcall TformGPGeeVerifyDecrypt::~TformGPGeeVerifyDecrypt()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Cleanup in preparation for the form to be deleted
//
void __fastcall TformGPGeeVerifyDecrypt::Cleanup(void)
{
  __ENTERFUNCTION__;

  // Clean up the lvFileList component - free all the data structures attached to each TListItem
  for (int n=0; n<lvFileList->Items->Count; n++) {
    if (lvFileList->Items->Item[n]->Data) {
      delete (VerifyDecryptInfo_t *)lvFileList->Items->Item[n]->Data;
      lvFileList->Items->Item[n]->Data = NULL;
    }  // if (lvFileList->Items->Item[n]->Data)
  }  // for (int n=0; n<lvFileList->Items->Count; n++)

  // Release any GPGME contexts that were created.
  if (ctxGPG) {
    gpgme_release(ctxGPG);
    ctxGPG = NULL;
  }  // if (ctxGPG)

  CleanupPassphraseCache(PassphraseCache);

  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::Cleanup(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnShow event handler for the form.  The form has been fully streamed in
// by the time this is called, so we can do all the form's visual setup here.
//
void __fastcall TformGPGeeVerifyDecrypt::FormShow(TObject *Sender)
{
  TListItem *FileItem;

  __ENTERFUNCTION__;

  // Sort the filenames, clear the lvFileList TListView and then add each file to the list, creating a
  // VerifyDecryptInfo_t structure for each one.  The structure holds the actual information and operations
  // results for each file.
  FileNames->Sort();
  lvFileList->Clear();
  for (int n=0; n<FileNames->Count; n++) {
    VerifyDecryptInfo_t *ItemInfo = new VerifyDecryptInfo_t;

    FileItem = lvFileList->Items->Add();
    FileItem->Caption = MinimizeName(FileNames->Strings[n],lvFileList->Font,lvFileList->Column[0]->Width-20);
    FileItem->Data = ItemInfo;

    ItemInfo->FileName = FileNames->Strings[n];
    ItemInfo->FileType = IdentifyFile(ItemInfo->FileName);
    if (ItemInfo->FileType == GPG_UNKNOWN) {
      ItemInfo->ResultString = GetMessage(MSG_ERROR_GPGEE_IDENTIFY);
      FileItem->StateIndex = GPGEE_STATE_INVALID;
      FileItem->SubItems->Add(GetMessage(MSG_GPGEE_TYPE_UNKNOWN));
      LOG1(LOG_DEBUG, "Added \"%s\" as an invalid file to the processing list.", ItemInfo->FileName); 
    } else {
      AnsiString sFileType;
      if (ItemInfo->FileType == GPG_MESSAGE)
        sFileType = GetMessage(MSG_GPGEE_TYPE_MESSAGE);
      else
        sFileType = GetMessage(MSG_GPGEE_TYPE_SIGNATURE);
      FileItem->StateIndex = GPGEE_STATE_PENDING;
      FileItem->SubItems->Add(sFileType);
      LOG2(LOG_DEBUG, "Added file \"%s\" to the processing list as type '%s'.", ItemInfo->FileName, sFileType.c_str()); 
    }  // if (ItemInfo->FileType == GPG_UNKNOWN)
  }  // for (int n=0; n<FileNames->Count; n++)
  lvFileList->SetFocus();
  LOG1(LOG_MESSAGE, "Starting processing timer with %d millisecond delay.", StartTimer->Interval);
  StartTimer->Enabled = true;

  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeVerifyDecrypt::FormShow(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Set the default modes and show the form.  Not much done here for this form
// except copy in the list of filenames we'll be working on.
//
int __fastcall TformGPGeeVerifyDecrypt::ShowModalWithDefaults(TStringList *FileNames)
{
  int retval;

  __ENTERFUNCTION__;
  this->FileNames = FileNames;
  LOG(LOG_DEBUG, "Showing the verify/decryption form.");
  retval = ShowModal();
  __RETURNFUNCTION(retval);
}  // int __fastcall TformGPGeeSignEncrypt::ShowModalWithDefaults(bool bSign, bool bEncrypt)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// StartTimer OnTimer event - this is used to allow the form time to get
// painted before we start into the operations.  This is set to 250ms in the
// form designer.
//
void __fastcall TformGPGeeVerifyDecrypt::StartTimerTimer(TObject *Sender)
{
  __ENTERFUNCTION__;
  LOG(LOG_MESSAGE, "Processing timer fired - disabling timer and starting file processing.");
  StartTimer->Enabled = false;
  VerifyDecryptFiles();
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeVerifyDecrypt::StartTimerTimer(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// The main work of verification/decryption is done here.
// The theory is simple - iterate through each entry in the lvFileList
// TListView compoment, figure out what type of file it is, perform the
// proper operation on it, and store the results for the user to check.
//
void __fastcall TformGPGeeVerifyDecrypt::VerifyDecryptFiles(void)
{
  gpgme_error_t err;
  gpgme_sig_t sig = NULL;
  bool Success;
  bool Detached = false;

  __ENTERFUNCTION__;

  try {
    try {
      // We want to make sure the form is drawn.  It should be by the time this method is called, but just in case
      // the system is really bogged, we do a ProcessMessages() right off the bat.  We also do one each time something
      // visual changes so that the form acts as its own progress bar, so to speak.  The user will be able to see as
      // each file is processed.
      Application->ProcessMessages();
      for (nOpNumber = 0; nOpNumber<lvFileList->Items->Count; nOpNumber++) {
        TListItem *Item = lvFileList->Items->Item[nOpNumber];
        VerifyDecryptInfo_t *ItemInfo = (VerifyDecryptInfo_t *)Item->Data;
        // This is the filename of the output file - simply the source filename without the .gpg/.sig/.asc extension
        AnsiString DataFileName = ItemInfo->FileName.SubString(0,ItemInfo->FileName.Length()-4);

        Item->Selected = true;                                      // Select each item as it's about to be processed
        if (Item->StateIndex == GPGEE_STATE_INVALID) {              // Go on to the next file if this file is already marked as invalid
          LOG1(LOG_DEBUG, "Skipping invalid file \"%s\".", ItemInfo->FileName.c_str());
          continue;
        }  // if (Item->StateIndex == GPGEE_STATE_INVALID)
        Item->StateIndex = GPGEE_STATE_INPROGRESS;                  // Otherwise mark the file as being in progress
        Application->ProcessMessages();                             // Make sure the user sees the in-progress icon
        if (bCancel) {                                              // If the cancel flag is set then all subsequent files
          Item->StateIndex = GPGEE_STATE_INVALID;                   // are marked as invalid.
          LOG1(LOG_DEBUG, "Cancelling processing of file \"%s\".", ItemInfo->FileName.c_str());
          ItemInfo->ResultString = GetMessage(MSG_ERROR_GPGEE_CANCELED);
          continue;
        }  // if (Cancel)
        switch (ItemInfo->FileType) {
          case GPG_MESSAGE: {                                       // If this is "message" then we will decrypt it
            gpgme_op_flags_t OpFlags;                               // Result flags for the operation
            char KeyID[17];

            sCurrentOpFileName = ItemInfo->FileName;                // Store the filename in case this file is going to be symmetrically decrypted so the the passphrase dialog can show the filename
            if (FileExists(DataFileName)) {                         // Figure out if we would be overwriting a file and then check if we're allowed to
              AnsiString FileOverwrite = GPGeeConfig->Values["File Overwrite"];
              if (FileOverwrite.AnsiCompareIC("never") == 0 || FileOverwrite.AnsiCompareIC("ask") == 0 &&
                  Application->MessageBox(FormMessage(MSG_QUERY_GPGEE_OVERWRITE_MSG,DataFileName.c_str()).c_str(), GetMessage(MSG_QUERY_GPGEE_OVERWRITE_CAPTION).c_str(), MB_YESNO) == IDNO) {
                Item->StateIndex = GPGEE_STATE_INVALID;
                Application->ProcessMessages();
                LOG1(LOG_DEBUG, "Cannot process file \"%s\" as overwriting is forbidden by user.", ItemInfo->FileName.c_str());
                ItemInfo->ResultString = GetMessage(MSG_ERROR_GPGEE_FILE_EXISTS);
                continue;
              }  // if (can't overwrite)
            }  // if (FileExists(DataFileName))
            // Now actually "decrypt" the file.
            LOG1(LOG_MESSAGE, "Performing decryption of file \"%s\".", ItemInfo->FileName.c_str());
            err = gpgme_op_file_decrypt(ctxGPG, ("\"" + ItemInfo->FileName + "\"").c_str(), ("\"" + DataFileName + "\"").c_str());
            gpgme_decrypt_get_status(ctxGPG, KeyID, &OpFlags);
            if (err == GPGME_No_Error) {                            // If decrypting was successfil...
              ItemInfo->ResultString = FormMessage(MSG_RESULT_GPGEE_DECRYPT_SUCCESS,DataFileName.c_str());
              gpgme_decrypt_get_sig_ctx(ctxGPG, &sig);              // Check to see if the file we decrypted was signed
              // If the file was signed then we base the success of the operation on whether the signature matches.
              // Otherwise, if the file wasn't signed, then the operation was a success
              if (sig) {
                LOG(LOG_MESSAGE, "Decryption succeeded - file is signed, processing signature to evaluate overall success.");
                Item->StateIndex = InterpretSignatures(sig, ItemInfo->ResultString);
              } else {
                LOG(LOG_MESSAGE, "Decryption succeeded - file is unsigned.");
                Item->StateIndex = GPGEE_STATE_SUCCESS;
              }  // if (sig)
            } else {                                                // If decrypting wasn't successful...
              LOG1(LOG_WARNING, "Decryption failed: %s", gpgme_strerror(err));
              ItemInfo->ResultString = String(gpgme_strerror(err));
              Item->StateIndex = GPGEE_STATE_FAILURE;
            }  // if (err == GPGME_No_Error)
            Application->ProcessMessages();                         // Make sure the user sees the state icon change
            // We check the operation's result flags to provide more information to the user.
            if (OpFlags & GPGME_OPFLAG_NOSECKEY) {
              LOG(LOG_DEBUG, "No secret key to process file file.");
              ItemInfo->ResultString += " " + FormMessage(MSG_RESULT_GPGEE_NO_SECRET_KEY,KeyID);
            }  // if (OpFlags & GPGME_OPFLAG_NOSECKEY)
            if (OpFlags & GPGME_OPFLAG_BADARMOR) {
              LOG(LOG_DEBUG, "File has bad armor.");
              ItemInfo->ResultString += " " + GetMessage(MSG_RESULT_GPGEE_BAD_ARMOR);
            }  // if (OpFlags & GPGME_OPFLAG_BADARMOR)
            if (OpFlags & GPGME_OPFLAG_BADMDC) {
              LOG(LOG_DEBUG, "File failed modification detection check.");
              ItemInfo->ResultString += " " + GetMessage(MSG_RESULT_GPGEE_BAD_MDC);
            }  // if (OpFlags & GPGME_OPFLAG_BADMDC)
            break;
          }  // case GPG_MESSAGE:
          case GPG_SIGNATURE:                                       // If this is a detached signature then we verify it
            // If this is a detached signature then DataFileName isn't the output file but the name of the file that
            // was signed.  In this case the file must exist before the signature can be checked.
            if (!FileExists(DataFileName)) {
              Item->StateIndex = GPGEE_STATE_INVALID;
              Application->ProcessMessages();
              LOG1(LOG_MESSAGE, "Could not find data file that signature \"%s\" was for - processing aborted.", ItemInfo->FileName.c_str());
              ItemInfo->ResultString = FormMessage(MSG_RESULT_GPGEE_NO_VERIFY_FILE,DataFileName.c_str());
              break;
            }  // if (!FileExists(DataFileName))
            Detached = true;
            // No break here
          case GPG_CLEARSIGNED:                                 // Clearsigned file
            // Verify the signature and then parse the results
            LOG1(LOG_MESSAGE, "Verifying file \"%s\".", ItemInfo->FileName.c_str());
            err = gpgme_op_file_verify(ctxGPG, Detached?GPGME_SIG_MODE_DETACH:GPGME_SIG_MODE_CLEAR, &sig, ("\"" + ItemInfo->FileName + "\"").c_str(), ("\"" + DataFileName + "\"").c_str());
            LOG1(LOG_MESSAGE, "Verification returned \"%s\" - processing signature to evaluate overall success.", gpgme_strerror(err));
            Item->StateIndex = InterpretSignatures(sig, ItemInfo->ResultString, err);
            break;
        }  // switch (ItemInfo->FileType)
        if (sig) {
          // If our above operation ended up referencing a signature result context then release it and set the pointer
          // to NULL so we can tell for the next file if a signature result is referenced
          gpgme_sig_release(sig);
          sig = NULL;
        }  // if (sig)
      }  // for (int n=0; n<lvFileList->Items->Count; n++)
      Success = true;
    }  // inner try
    catch (Exception &e) {
      Success = false;
      LOG(LOG_ERROR, "Exception " + e.ClassName() + " raised with message \"" + e.Message + "\".");
      throw;
    }  // catch
  }  // outer try
  __finally {
    lvFileList->Selected = lvFileList->Items->Item[0];                // Select the first item in the list when we're done.
    if (Success)
      btnOkCancel->Caption = GetMessage(MSG_CAPTION_GPGEE_BUTTON_OK); // The Ok/Cancel button becomes the Ok button when all is done successfully
    btnOkCancel->Tag = true;
    __RETURNFUNCTION__;
  }  // __finally
}  // void __fastcall TformGPGeeVerifyDecrypt::VerifyDecryptFiles(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Wrapper function for InterpretSignatureResult too enable supporting
// multiple signatures
//
int  __fastcall TformGPGeeVerifyDecrypt::InterpretSignatures(gpgme_sig_t sig, AnsiString &ResultString, gpgme_error_t err)
{
  AnsiString MultiResults;
  int OverallState = GPGEE_STATE_SUCCESS;
  int ThisState;

  __ENTERFUNCTION__;
  gpgme_sig_t CurrentSig = sig;
  for (int idx=0; CurrentSig != NULL; idx++, CurrentSig = (gpgme_sig_t)gpgme_sig_get_ulong_attr(sig, idx, GPGME_ATTR_OPAQUE)) {
    //if (!MultiResults.IsEmpty())
    //  MultiResults += "\r\n";
    ThisState = InterpretSignatureResult(CurrentSig, MultiResults, err);
    switch (OverallState) {
      case GPGEE_STATE_SUCCESS:
        OverallState = ThisState;
        break;
      case GPGEE_STATE_PARTIAL:
        if (ThisState == GPGEE_STATE_FAILURE)
          OverallState == GPGEE_STATE_FAILURE;
        break;
      case GPGEE_STATE_FAILURE:
        break;
    }  // switch (PreviosState)
  }  // for (all signatures)
  ResultString = MultiResults;
  __RETURNFUNCTION(OverallState);
}  // int __fastcall TformGPGeeVerifyDecrypt::InterpretSignatures(gpgme_sit_t sig, AnsiString &ResultString, gogme_error_t err)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Look at the result of checking a signature and return the appropriate
// state code.  The state code returned is the same state code used to
// represent the icon state for the listview component.  In additaion,
// ResultString is appended with a text description of the signature result.
//
int __fastcall TformGPGeeVerifyDecrypt::InterpretSignatureResult(gpgme_sig_t sig, AnsiString &ResultString, gpgme_error_t err)
{
  AnsiString ResultStringPrefix, ResultStringBody, ResultStringPostfix;
  AnsiString SigKeyId;
  AnsiString SigUserId;
  gpgme_validity_t trust;
  int retval;

  __ENTERFUNCTION__;

  if (!ResultString.IsEmpty())
    ResultString += "\r\n";                                     // Add a CRLF if the result string already has some stuff in it

  SigKeyId = String(gpgme_sig_get_string_attr(sig, GPGME_ATTR_KEYID)).SubString(9,8);
  SigUserId = String(gpgme_sig_get_string_attr(sig, GPGME_ATTR_USERID));
  // The basis for whether we consider the signature good or not is the signature's validity attribute.
  switch (gpgme_sig_get_ulong_attr(sig, 0, GPGME_ATTR_VALIDITY)) {
    case GPGME_SIG_STAT_NONE:                                   // No signature??
      ResultStringBody = FormMessage(MSG_RESULT_GPGEE_BODYNONE,gpgme_strerror(err));
      retval = GPGEE_STATE_FAILURE;
      break;
    case GPGME_SIG_STAT_GOOD:                                   // Good signature
      ResultStringPrefix = GetMessage(MSG_RESULT_GPGEE_PREFIXGOOD);
      retval = GPGEE_STATE_SUCCESS;
      break;
    case GPGME_SIG_STAT_BAD:                                    // Bad signature
      ResultStringBody = FormMessage(MSG_RESULT_GPGEE_BODYBAD, SigUserId.c_str(), SigKeyId.c_str());
      retval = GPGEE_STATE_FAILURE;
      break;
    case GPGME_SIG_STAT_NOKEY:                                  // Don't have key to check signature with
      ResultStringBody = FormMessage(MSG_RESULT_GPGEE_BODYNOKEY,SigKeyId.c_str());
      retval = GPGEE_STATE_FAILURE;
      break;
    case GPGME_SIG_STAT_NOSIG:                                  // No signature??
      ResultStringBody = GetMessage(MSG_RESULT_GPGEE_BODYNOSIG);
      retval = GPGEE_STATE_FAILURE;
      break;
    case GPGME_SIG_STAT_ERROR:                                  // Signature error of some sort?
      ResultStringBody = FormMessage(MSG_RESULT_GPGEE_BODYERROR,gpgme_strerror(err));
      retval = GPGEE_STATE_FAILURE;
      break;
    case GPGME_SIG_STAT_E_GOOD: {                               // Good signature with expired key - check dates on signature and key expiry
      gpgme_key_t SigningKey;
      TDateTime KeyExpiryDate, SignatureCreationDate;

      ResultStringPrefix = GetMessage(MSG_RESULT_GPGEE_PREFIXEGOOD);
      if (!cachePublic)
        cachePublic = GetKeyCache(false);
      // Here we check the expiry date of the signing key against the date the signature was created.  If the signature
      // was created before the key expired then we give a GPGEE_STATE_SUCCESS result (green checkmark).  If the signature
      // was created after the key expired (or if for some reason we run into a problem checking the dates) we give a
      // GPGEE_STATE_PARTIAL (yellow checkmark) result and warn the user.
      err = gpgme_keycache_find_key(cachePublic, SigKeyId.c_str(), false, &SigningKey);
      if (err == GPGME_No_Error) {
        // We don't convert dates from GMT to local time here.  The dates are left in GMT so there isn't any confusion
        // if the user needs to check with the document's signer about the expiry date on his/her key.  Any dates that
        // are presented to the user in GMT, though, are marked as such so that the user knows this.
        SignatureCreationDate = GPGToDateTime(gpgme_sig_get_ulong_attr(sig, 0, GPGME_ATTR_CREATED), false);
        KeyExpiryDate = GPGToDateTime(gpgme_key_get_ulong_attr(SigningKey, GPGME_ATTR_EXPIRES, NULL, 0), false);
        if (SignatureCreationDate > KeyExpiryDate) {
          LOG(LOG_DEBUG, "Expired-key signature made after key expired.");
          ResultStringPostfix = FormMessage(MSG_RESULT_GPGEE_POSTEXPIREDBAD,DateTimeToStr(KeyExpiryDate).c_str());
          retval = GPGEE_STATE_PARTIAL;
        } else {
          LOG(LOG_DEBUG, "Expired-key signature made before key expired.");
          ResultStringPostfix = GetMessage(MSG_RESULT_GPGEE_POSTEXPIREDOK);
          retval = GPGEE_STATE_SUCCESS;
        }  // if (SignatureCreationDate > KeyExpiryDate)
      } else {
        LOG(LOG_DEBUG, "Could not determine if expired-key signature predates key expiry.");
        ResultStringPostfix = GetMessage(MSG_RESULT_GPGEE_POSTEXPIREDUNK);
        retval = GPGEE_STATE_PARTIAL;
      }  // if (err == GPGME_No_Error)
      break;
    }  // case GPGME_SIG_STAT_E_GOOD:
    case GPGME_SIG_STAT_R_GOOD:                                 // Good signature with revoked key
      ResultStringPrefix = GetMessage(MSG_RESULT_GPGEE_PREFIXRGOOD);
      retval = GPGEE_STATE_PARTIAL;
      ResultStringPostfix = GetMessage(MSG_RESULT_GPGEE_POSTREVOKED);
      break;
  }  // switch (gpgme_sig_get_ulong_attr(sig, 0, GPGME_ATTR_VALIDITY))
  // We add information about the signature to some of the results.
  if (ResultStringBody.IsEmpty())
    ResultStringBody = " " + FormMessage(MSG_RESULT_GPGEE_BODYSTANDARD, DigestToString((gpgme_md_t)gpgme_sig_get_ulong_attr(sig, 0, GPGME_ATTR_MD)), SigUserId.c_str(), KeyTypeToString((gpgme_pk_cipher_t)gpgme_sig_get_ulong_attr(sig, 0, GPGME_ATTR_ALGO)).c_str(), SigKeyId.c_str(), DateTimeToStr(GPGToDateTime(gpgme_sig_get_ulong_attr(sig, 0, GPGME_ATTR_CREATED))).c_str());
  if (!ResultStringPostfix.IsEmpty())
    ResultStringPostfix = "\r\n" + ResultStringPostfix;
  trust = (gpgme_validity_t)gpgme_sig_get_ulong_attr(sig, 0, GPGME_ATTR_OTRUST);
  if (trust == GPGME_TRUST_DONTKNOW || trust == GPGME_TRUST_NEVER || trust == GPGME_TRUST_MARGINAL || trust == GPGME_TRUST_UNDEFINED) {
    ResultStringPostfix += "\r\n" + FormMessage(MSG_RESULT_GPGEE_POSTDODGYTRUST, TrustToString(trust).c_str());
    if (retval == GPGEE_STATE_SUCCESS && trust != GPGME_TRUST_MARGINAL)
      retval = GPGEE_STATE_PARTIAL;
  }  // if (dodgy validity)
  ResultString += ResultStringPrefix + ResultStringBody + ResultStringPostfix;
  LOG(LOG_DEBUG, ResultStringPrefix + ResultStringBody);
  __RETURNFUNCTION(retval);
}  // int __fastcall TformGPGeeVerifyDecrypt::InterpretSignatureResult(gpgme_sig_t sig, AnsiString &ResultString, gpgme_error_t err)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnChange event handler for the listview component.  Display the extended
// information about a file in the memo field when the user selects it.
//
void __fastcall TformGPGeeVerifyDecrypt::lvFileListChange(TObject *Sender, TListItem *Item, TItemChange Change)
{
  VerifyDecryptInfo_t *Info;

  __ENTERFUNCTION__;
  Info = (VerifyDecryptInfo_t *)Item->Data;
  if (Change == ctState && Info)
    richResults->Lines->Text = Info->ResultString;
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeVerifyDecrypt::lvFileListChange(TObject *Sender, TListItem *Item, TItemChange Change)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// User pressed the ok/cancel button - If we are currently performing
// operations then we set the cancel flag.  This will cause the rest of the
// operations to abort and leave the dialog up so the user can see the result
// of any completed operations.  If operations are completed when this is
// called, then the button has been changed to the "Ok" button and we simply
// close the dialog.
//
void __fastcall TformGPGeeVerifyDecrypt::btnOkCancelClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  if (btnOkCancel->Tag)
    ModalResult = mrOk;
  else {
    LOG(LOG_MESSAGE, "User cancel - setting cancel flag.");
    bCancel = true;
  }
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeVerifyDecrypt::btnOkCancelClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// User pressed the help button - pull up the correct help page
//
void __fastcall TformGPGeeVerifyDecrypt::btnHelpClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  Application->HelpContext(3);
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeVerifyDecrypt::btnHelpClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Make the buttons line up when the form changes size
//
void __fastcall TformGPGeeVerifyDecrypt::FormResize(TObject *Sender)
{
  __ENTERFUNCTION__;
  btnOkCancel->Left = Width / 2 - btnOkCancel->Width / 2;
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeVerifyDecrypt::FormResize(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// This is where the real work of getting a passphrase is done.  It's easier
// to do this from inside the class, so the callback passes control here.
// When GPGME asks for a passphrase, it puts some information in a text
// string as to what sort of passphrase it is asking for.  It appears to
// always be three lines of text.  The second line contains a "User ID hint"
// which is the Key Id of the key for which the password is needed to unlock.
// If that Key Id is present it is assumed this is being called to unlock a
// secret key, and we get all the info on that key to present to the user.
// If that hint is absent then it is assumed this is being called to get a
// symmetrical-encryption passphrase.  OpenPGP smart cards are handled a
// little differently - I don't have one so I can't check on this yet.  It's
// likely that this won't work with them right now.
//
const char * __fastcall TformGPGeeVerifyDecrypt::GetPassphrase(const char *desc)
{
  static AnsiString Passphrase = "";
  static int nPreviousOpNumber = -1;
  gpgme_key_t Key;
  AnsiString KeyId, KeyName, KeyType, KeySize, KeyCreationDate;
  int nCachePos;

  __ENTERFUNCTION__;

  if (desc) {
    TStringList *desclines = new TStringList;
    formGPGeePassPhrase = new TformGPGeePassPhrase(NULL);

    // Drop the passphrase "description" text into a TStringList to parse it out into the individual lines
    desclines->Text = desc;
    LOG1(LOG_DEBUG, "Passphrase callback activated: User ID hint=\"%s\".", desclines->Strings[1].c_str());
    // If there is no User ID hint then we assume that we are being invoked to get a symmetrical-encryption passphrase
    if (desclines->Strings[1] == "[User ID hint missing]") {
      if (formGPGeePassPhrase->GetPassphrase(FormMessage(MSG_QUERY_GPGEE_SYMPASSPHRASE,sCurrentOpFileName.c_str())) == mrOk)
        Passphrase = formGPGeePassPhrase->edtPassphrase->Text;
      else
        Passphrase = "";
    // Otherwise, check line 2 of the description for the key id of the secret key we are getting the passphrase for and
    // present the user with the info about that key
    } else if (!desclines->Strings[1].IsEmpty()) {
      KeyId = desclines->Strings[1].SubString(9,8);
      nCachePos = PassphraseCache->IndexOfName(KeyId);
      if (nCachePos != -1 && nOpNumber != nPreviousOpNumber) {
        nPreviousOpNumber = nOpNumber;
        Passphrase = PassphraseCache->Values[KeyId];
        __RETURNFUNCTION(Passphrase.c_str());
      }  // if (nCachePos != -1 && nOpNumber != nPreviousOpNumber)
      gpgme_keycache_rewind(cacheSecret);
      if (gpgme_keycache_find_key(cacheSecret,KeyId.c_str(), false, &Key) == GPGME_No_Error) {
        KeyName = String(gpgme_key_get_string_attr(Key, GPGME_ATTR_NAME, NULL, 0)) + " <" + String(gpgme_key_get_string_attr(Key, GPGME_ATTR_EMAIL, NULL, 0)) + ">";
        KeyType = String(gpgme_key_get_string_attr(Key, GPGME_ATTR_ALGO, NULL, 0));
        KeySize = String(gpgme_key_get_ulong_attr(Key, GPGME_ATTR_LEN, NULL, 0));
        KeyCreationDate = DateToStr(GPGToDateTime(gpgme_key_get_ulong_attr(Key, GPGME_ATTR_CREATED, NULL, 0)));
        if (formGPGeePassPhrase->GetPassphrase(KeyName, KeyId, KeyType, KeySize, KeyCreationDate) == mrOk)
          Passphrase = formGPGeePassPhrase->edtPassphrase->Text;
        else
          Passphrase = "";
        //  zzz ADD CACHE CODE
      }  // if (gpgme_keycache_find_key(cacheSecret,KeyId.c_str(), false, &Key) == GPGME_No_Error)
      else
        LOG(LOG_WARNING, "Could not find secret key in keycache referenced by passphrase callback.");
    }  // if (desclines->Strings[1] == "[User ID hint missing]")
    delete formGPGeePassPhrase;
    nPreviousOpNumber = nOpNumber;
    PassphraseCache->Add(KeyId + "=" + Passphrase);
    VirtualLock((void *)(PassphraseCache->Strings[PassphraseCache->Count-1].data()),Passphrase.Length());
    __RETURNFUNCTION(Passphrase.c_str());
  // If we're called with an empty desc then we are being asked to clean up after a previous call (the passphrase is
  // no longer needed).  If our passphrase variable isn't empty then overwrite it before setting it to null length.
  } else if (!Passphrase.IsEmpty()) {
    // Make sure that when the passphrase is no longer needed that it is erased
    Passphrase.StringOfChar('*', Passphrase.Length());
    Passphrase.StringOfChar('*', 128);
    Passphrase = "";
  }  // if (desc)
  __RETURNFUNCTION(NULL);
}  // const char * __fastcall TformGPGeeVerifyDecrypt::GetPassphrase(const char *desc)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// The actual passphrase callback - just dumps control to the Passphrase
// method of the form class.
//
static const char * PassphraseCallback(void *hook, const char *desc, void *dummy)
{
  const char *retval;

  __ENTERFUNCTION__;
  TformGPGeeVerifyDecrypt *formGPGeeVerifyDecrypt = (TformGPGeeVerifyDecrypt *)hook;
  retval = formGPGeeVerifyDecrypt->GetPassphrase(desc);
  __RETURNFUNCTION(retval);
}
//---------------------------------------------------------------------------




