"use strict";
// *****************************************************************************
// Copyright (C) 2023 Ericsson and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArrayUtils = void 0;
var ArrayUtils;
(function (ArrayUtils) {
    ArrayUtils.TailImpl = {
        tail() {
            return this[this.length - 1];
        },
    };
    ArrayUtils.HeadAndChildrenImpl = {
        head() {
            return this[0];
        },
        children() {
            return Object.assign(this.slice(1), ArrayUtils.TailImpl);
        }
    };
    function asTail(array) {
        return Object.assign(array, ArrayUtils.TailImpl);
    }
    ArrayUtils.asTail = asTail;
    function asHeadAndTail(array) {
        return Object.assign(array, ArrayUtils.HeadAndChildrenImpl, ArrayUtils.TailImpl);
    }
    ArrayUtils.asHeadAndTail = asHeadAndTail;
    let Sort;
    (function (Sort) {
        Sort[Sort["LeftBeforeRight"] = -1] = "LeftBeforeRight";
        Sort[Sort["RightBeforeLeft"] = 1] = "RightBeforeLeft";
        Sort[Sort["Equal"] = 0] = "Equal";
    })(Sort = ArrayUtils.Sort || (ArrayUtils.Sort = {}));
    // Copied from https://github.com/microsoft/vscode/blob/9c29becfad5f68270b9b23efeafb147722c5feba/src/vs/base/common/arrays.ts
    /**
     * Performs a binary search algorithm over a sorted collection. Useful for cases
     * when we need to perform a binary search over something that isn't actually an
     * array, and converting data to an array would defeat the use of binary search
     * in the first place.
     *
     * @param length The collection length.
     * @param compareToKey A function that takes an index of an element in the
     *   collection and returns zero if the value at this index is equal to the
     *   search key, a negative number if the value precedes the search key in the
     *   sorting order, or a positive number if the search key precedes the value.
     * @return A non-negative index of an element, if found. If not found, the
     *   result is -(n+1) (or ~n, using bitwise notation), where n is the index
     *   where the key should be inserted to maintain the sorting order.
     */
    function binarySearch2(length, compareToKey) {
        let low = 0;
        let high = length - 1;
        while (low <= high) {
            const mid = ((low + high) / 2) | 0;
            const comp = compareToKey(mid);
            if (comp < 0) {
                low = mid + 1;
            }
            else if (comp > 0) {
                high = mid - 1;
            }
            else {
                return mid;
            }
        }
        return -(low + 1);
    }
    ArrayUtils.binarySearch2 = binarySearch2;
    function partition(array, filter) {
        const pass = [];
        const fail = [];
        array.forEach((e, idx, arr) => (filter(e, idx, arr) ? pass : fail).push(e));
        return [pass, fail];
    }
    ArrayUtils.partition = partition;
    /**
     * @returns New array with all falsy values removed. The original array IS NOT modified.
     */
    function coalesce(array) {
        return array.filter(e => !!e);
    }
    ArrayUtils.coalesce = coalesce;
})(ArrayUtils = exports.ArrayUtils || (exports.ArrayUtils = {}));
//# sourceMappingURL=array-utils.js.map