/**
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.patternlanguage.emf.jvmmodel;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrerUtil;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.JavadocInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.util.EMFJvmTypesBuilder;
import org.eclipse.viatra.query.patternlanguage.emf.util.EMFPatternLanguageGeneratorConfig;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.validation.IssueCodes;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.matchers.tuple.Tuple;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmConstructor;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmVisibility;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.xbase.jvmmodel.JvmAnnotationReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * {@link ViatraQueryMatcher} implementation inferrer.
 * 
 * @author Mark Czotter
 * @noreference
 */
@SuppressWarnings("all")
public class PatternMatcherClassInferrer {
  @Inject
  @Extension
  private EMFJvmTypesBuilder _eMFJvmTypesBuilder;
  
  @Inject
  @Extension
  private EMFPatternLanguageJvmModelInferrerUtil _eMFPatternLanguageJvmModelInferrerUtil;
  
  @Inject
  @Extension
  private JavadocInferrer _javadocInferrer;
  
  @Extension
  private JvmTypeReferenceBuilder builder;
  
  @Extension
  private JvmAnnotationReferenceBuilder annBuilder;
  
  @Inject
  private IErrorFeedback feedback;
  
  public Boolean inferMatcherClassElements(final JvmGenericType matcherClass, final Pattern pattern, final JvmDeclaredType specificationClass, final JvmDeclaredType matchClass, final JvmTypeReferenceBuilder builder, final JvmAnnotationReferenceBuilder annBuilder, final EMFPatternLanguageGeneratorConfig config) {
    boolean _xtrycatchfinallyexpression = false;
    try {
      boolean _xblockexpression = false;
      {
        final boolean generateMatchProcessors = config.isGenerateMatchProcessors();
        this.builder = builder;
        this.annBuilder = annBuilder;
        this._eMFJvmTypesBuilder.setDocumentation(matcherClass, this._javadocInferrer.javadocMatcherClass(pattern).toString());
        this.inferStaticMethods(matcherClass, pattern);
        this.inferFields(matcherClass, pattern);
        this.inferConstructors(matcherClass, pattern);
        this.inferMethods(matcherClass, pattern, matchClass, generateMatchProcessors);
        EList<JvmMember> _members = matcherClass.getMembers();
        final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
          it.setVisibility(JvmVisibility.PUBLIC);
          it.setStatic(true);
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocQuerySpecificationMethod(pattern).toString());
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return ");
              JvmTypeReference _typeRef = PatternMatcherClassInferrer.this.builder.typeRef(specificationClass);
              _builder.append(_typeRef);
              _builder.append(".instance();");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(pattern, "querySpecification", this.builder.typeRef(IQuerySpecification.class, this.builder.typeRef(matcherClass)), _function);
        _xblockexpression = this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
      }
      _xtrycatchfinallyexpression = _xblockexpression;
    } catch (final Throwable _t) {
      if (_t instanceof IllegalStateException) {
        final IllegalStateException ex = (IllegalStateException)_t;
        this.feedback.reportError(pattern, ex.getMessage(), IssueCodes.OTHER_ISSUE, Severity.ERROR, 
          IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return Boolean.valueOf(_xtrycatchfinallyexpression);
  }
  
  /**
   * Infers fields for Matcher class based on the input 'pattern'.
   */
  public boolean inferFields(final JvmDeclaredType matcherClass, final Pattern pattern) {
    boolean _xblockexpression = false;
    {
      EList<Variable> _parameters = pattern.getParameters();
      for (final Variable variable : _parameters) {
        EList<JvmMember> _members = matcherClass.getMembers();
        final Procedure1<JvmField> _function = (JvmField it) -> {
          it.setStatic(true);
          it.setFinal(true);
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              int _indexOf = pattern.getParameters().indexOf(variable);
              _builder.append(_indexOf);
            }
          };
          this._eMFJvmTypesBuilder.setInitializer(it, _client);
        };
        JvmField _field = this._eMFJvmTypesBuilder.toField(pattern, this._eMFPatternLanguageJvmModelInferrerUtil.positionConstant(variable), this.builder.typeRef(int.class), _function);
        this._eMFJvmTypesBuilder.<JvmField>operator_add(_members, _field);
      }
      EList<JvmMember> _members_1 = matcherClass.getMembers();
      final Procedure1<JvmField> _function_1 = (JvmField it) -> {
        it.setStatic(true);
        it.setFinal(true);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append(ViatraQueryLoggingUtil.class);
            _builder.append(".getLogger(");
            _builder.append(matcherClass);
            _builder.append(".class)");
            _builder.newLineIfNotEmpty();
          }
        };
        this._eMFJvmTypesBuilder.setInitializer(it, _client);
      };
      JvmField _field_1 = this._eMFJvmTypesBuilder.toField(pattern, "LOGGER", this.builder.typeRef(Logger.class), _function_1);
      _xblockexpression = this._eMFJvmTypesBuilder.<JvmField>operator_add(_members_1, _field_1);
    }
    return _xblockexpression;
  }
  
  /**
   * Infers static methods for Matcher class based on the input 'pattern'.
   * NOTE: queryDefinition() will be inferred later, in EMFPatternLanguageJvmModelInferrer
   */
  public boolean inferStaticMethods(final JvmGenericType matcherClass, final Pattern pattern) {
    boolean _xblockexpression = false;
    {
      EList<JvmMember> _members = matcherClass.getMembers();
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        it.setStatic(true);
        it.setVisibility(JvmVisibility.PUBLIC);
        this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocMatcherStaticOnEngine(pattern).toString());
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "engine", this.builder.typeRef(ViatraQueryEngine.class));
        this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("// check if matcher already exists");
            _builder.newLine();
            String _simpleName = matcherClass.getSimpleName();
            _builder.append(_simpleName);
            _builder.append(" matcher = engine.getExistingMatcher(querySpecification());");
            _builder.newLineIfNotEmpty();
            _builder.append("if (matcher == null) {");
            _builder.newLine();
            _builder.append("    ");
            _builder.append("matcher = (");
            String _simpleName_1 = matcherClass.getSimpleName();
            _builder.append(_simpleName_1, "    ");
            _builder.append(")engine.getMatcher(querySpecification());");
            _builder.newLineIfNotEmpty();
            _builder.append("}");
            _builder.newLine();
            _builder.append("return matcher;");
            _builder.newLine();
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(pattern, "on", this.builder.typeRef(matcherClass), _function);
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
      EList<JvmMember> _members_1 = matcherClass.getMembers();
      final Procedure1<JvmOperation> _function_1 = (JvmOperation it) -> {
        it.setStatic(true);
        it.setVisibility(JvmVisibility.PUBLIC);
        this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocMatcherStaticCreate(pattern).toString());
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return new ");
            String _simpleName = matcherClass.getSimpleName();
            _builder.append(_simpleName);
            _builder.append("();");
            _builder.newLineIfNotEmpty();
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method_1 = this._eMFJvmTypesBuilder.toMethod(pattern, "create", this.builder.typeRef(matcherClass), _function_1);
      _xblockexpression = this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_1, _method_1);
    }
    return _xblockexpression;
  }
  
  /**
   * Infers constructors for Matcher class based on the input 'pattern'.
   */
  public boolean inferConstructors(final JvmDeclaredType matcherClass, final Pattern pattern) {
    EList<JvmMember> _members = matcherClass.getMembers();
    final Procedure1<JvmConstructor> _function = (JvmConstructor it) -> {
      it.setVisibility(JvmVisibility.PRIVATE);
      this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocMatcherConstructorEngine(pattern).toString());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("super(querySpecification());");
        }
      };
      this._eMFJvmTypesBuilder.setBody(it, _client);
    };
    JvmConstructor _constructor = this._eMFJvmTypesBuilder.toConstructor(pattern, _function);
    return this._eMFJvmTypesBuilder.<JvmConstructor>operator_add(_members, _constructor);
  }
  
  /**
   * Infers methods for Matcher class based on the input 'pattern'.
   */
  public boolean inferMethods(final JvmDeclaredType type, final Pattern pattern, final JvmType matchClass, final boolean generateMatchProcessor) {
    boolean _xblockexpression = false;
    {
      this.builder = this.builder;
      boolean _isEmpty = pattern.getParameters().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        EList<JvmMember> _members = type.getMembers();
        final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocGetAllMatchesMethod(pattern).toString());
          EList<Variable> _parameters = pattern.getParameters();
          for (final Variable parameter : _parameters) {
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
          }
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return rawStreamAllMatches(new Object[]{");
              {
                EList<Variable> _parameters = pattern.getParameters();
                boolean _hasElements = false;
                for(final Variable p : _parameters) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(", ", "");
                  }
                  String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                  _builder.append(_parameterName);
                }
              }
              _builder.append("}).collect(");
              _builder.append(Collectors.class);
              _builder.append(".toSet());");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(pattern, "getAllMatches", this.builder.typeRef(Collection.class, this.builder.typeRef(matchClass)), _function);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
        EList<JvmMember> _members_1 = type.getMembers();
        final Procedure1<JvmOperation> _function_1 = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocStreamAllMatchesMethod(pattern).toString());
          EList<Variable> _parameters = pattern.getParameters();
          for (final Variable parameter : _parameters) {
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
          }
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return rawStreamAllMatches(new Object[]{");
              {
                EList<Variable> _parameters = pattern.getParameters();
                boolean _hasElements = false;
                for(final Variable p : _parameters) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(", ", "");
                  }
                  String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                  _builder.append(_parameterName);
                }
              }
              _builder.append("});");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method_1 = this._eMFJvmTypesBuilder.toMethod(pattern, "streamAllMatches", this.builder.typeRef(Stream.class, this.builder.typeRef(matchClass)), _function_1);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_1, _method_1);
        EList<JvmMember> _members_2 = type.getMembers();
        final Procedure1<JvmOperation> _function_2 = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocGetOneArbitraryMatchMethod(pattern).toString());
          EList<Variable> _parameters = pattern.getParameters();
          for (final Variable parameter : _parameters) {
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
          }
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return rawGetOneArbitraryMatch(new Object[]{");
              {
                EList<Variable> _parameters = pattern.getParameters();
                boolean _hasElements = false;
                for(final Variable p : _parameters) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(", ", "");
                  }
                  String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                  _builder.append(_parameterName);
                }
              }
              _builder.append("});");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method_2 = this._eMFJvmTypesBuilder.toMethod(pattern, "getOneArbitraryMatch", this.builder.typeRef(Optional.class, this.builder.typeRef(matchClass)), _function_2);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_2, _method_2);
        EList<JvmMember> _members_3 = type.getMembers();
        final Procedure1<JvmOperation> _function_3 = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocHasMatchMethod(pattern).toString());
          EList<Variable> _parameters = pattern.getParameters();
          for (final Variable parameter : _parameters) {
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
          }
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return rawHasMatch(new Object[]{");
              {
                EList<Variable> _parameters = pattern.getParameters();
                boolean _hasElements = false;
                for(final Variable p : _parameters) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(", ", "");
                  }
                  String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                  _builder.append(_parameterName);
                }
              }
              _builder.append("});");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method_3 = this._eMFJvmTypesBuilder.toMethod(pattern, "hasMatch", this.builder.typeRef(boolean.class), _function_3);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_3, _method_3);
        EList<JvmMember> _members_4 = type.getMembers();
        final Procedure1<JvmOperation> _function_4 = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocCountMatchesMethod(pattern).toString());
          EList<Variable> _parameters = pattern.getParameters();
          for (final Variable parameter : _parameters) {
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
          }
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return rawCountMatches(new Object[]{");
              {
                EList<Variable> _parameters = pattern.getParameters();
                boolean _hasElements = false;
                for(final Variable p : _parameters) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(", ", "");
                  }
                  String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                  _builder.append(_parameterName);
                }
              }
              _builder.append("});");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method_4 = this._eMFJvmTypesBuilder.toMethod(pattern, "countMatches", this.builder.typeRef(int.class), _function_4);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_4, _method_4);
        if (generateMatchProcessor) {
          EList<JvmMember> _members_5 = type.getMembers();
          final Procedure1<JvmOperation> _function_5 = (JvmOperation it) -> {
            it.setReturnType(this.builder.typeRef(Void.TYPE));
            this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocForEachMatchMethod(pattern).toString());
            EList<Variable> _parameters = pattern.getParameters();
            for (final Variable parameter : _parameters) {
              EList<JvmFormalParameter> _parameters_1 = it.getParameters();
              JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
              this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
            }
            EList<JvmFormalParameter> _parameters_2 = it.getParameters();
            JvmFormalParameter _parameter_1 = this._eMFJvmTypesBuilder.toParameter(pattern, "processor", this.builder.typeRef(Consumer.class, this.builder.wildcardSuper(this.builder.typeRef(matchClass))));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter_1);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("rawForEachMatch(new Object[]{");
                {
                  EList<Variable> _parameters = pattern.getParameters();
                  boolean _hasElements = false;
                  for(final Variable p : _parameters) {
                    if (!_hasElements) {
                      _hasElements = true;
                    } else {
                      _builder.appendImmediate(", ", "");
                    }
                    String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                    _builder.append(_parameterName);
                  }
                }
                _builder.append("}, processor);");
                _builder.newLineIfNotEmpty();
              }
            };
            this._eMFJvmTypesBuilder.setBody(it, _client);
          };
          JvmOperation _method_5 = this._eMFJvmTypesBuilder.toMethod(pattern, "forEachMatch", null, _function_5);
          this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_5, _method_5);
        }
        EList<JvmMember> _members_6 = type.getMembers();
        final Procedure1<JvmOperation> _function_6 = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocForOneArbitraryMatchMethod(pattern).toString());
          EList<Variable> _parameters = pattern.getParameters();
          for (final Variable parameter : _parameters) {
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
          }
          EList<JvmFormalParameter> _parameters_2 = it.getParameters();
          JvmFormalParameter _parameter_1 = this._eMFJvmTypesBuilder.toParameter(pattern, "processor", this.builder.typeRef(Consumer.class, this.builder.wildcardSuper(this.builder.typeRef(matchClass))));
          this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter_1);
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return rawForOneArbitraryMatch(new Object[]{");
              {
                EList<Variable> _parameters = pattern.getParameters();
                boolean _hasElements = false;
                for(final Variable p : _parameters) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(", ", "");
                  }
                  String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                  _builder.append(_parameterName);
                }
              }
              _builder.append("}, processor);");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method_6 = this._eMFJvmTypesBuilder.toMethod(pattern, "forOneArbitraryMatch", this.builder.typeRef(boolean.class), _function_6);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_6, _method_6);
        EList<JvmMember> _members_7 = type.getMembers();
        final Procedure1<JvmOperation> _function_7 = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocNewMatchMethod(pattern).toString());
          EList<Variable> _parameters = pattern.getParameters();
          for (final Variable parameter : _parameters) {
            EList<JvmFormalParameter> _parameters_1 = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
          }
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return ");
              JvmTypeReference _typeRef = PatternMatcherClassInferrer.this.builder.typeRef(matchClass);
              _builder.append(_typeRef);
              _builder.append(".newMatch(");
              {
                EList<Variable> _parameters = pattern.getParameters();
                boolean _hasElements = false;
                for(final Variable p : _parameters) {
                  if (!_hasElements) {
                    _hasElements = true;
                  } else {
                    _builder.appendImmediate(", ", "");
                  }
                  String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                  _builder.append(_parameterName);
                }
              }
              _builder.append(");");
              _builder.newLineIfNotEmpty();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method_7 = this._eMFJvmTypesBuilder.toMethod(pattern, "newMatch", this.builder.typeRef(matchClass), _function_7);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_7, _method_7);
        EList<Variable> _parameters = pattern.getParameters();
        for (final Variable variable : _parameters) {
          {
            final JvmTypeReference typeOfVariable = this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(variable);
            EList<JvmMember> _members_8 = type.getMembers();
            String _name = variable.getName();
            String _plus = ("rawStreamAllValuesOf" + _name);
            final Procedure1<JvmOperation> _function_8 = (JvmOperation it) -> {
              this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocGetAllValuesOfMethod(variable).toString());
              EList<JvmFormalParameter> _parameters_1 = it.getParameters();
              JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(variable, "parameters", this._eMFJvmTypesBuilder.addArrayTypeDimension(this.builder.typeRef(Object.class)));
              this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
              it.setVisibility(JvmVisibility.PROTECTED);
              StringConcatenationClient _client = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append("return rawStreamAllValues(");
                  String _positionConstant = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.positionConstant(variable);
                  _builder.append(_positionConstant);
                  _builder.append(", parameters).map(");
                  _builder.append(typeOfVariable);
                  _builder.append(".class::cast);");
                  _builder.newLineIfNotEmpty();
                }
              };
              this._eMFJvmTypesBuilder.setBody(it, _client);
            };
            JvmOperation _method_8 = this._eMFJvmTypesBuilder.toMethod(variable, _plus, this.builder.typeRef(Stream.class, typeOfVariable), _function_8);
            this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_8, _method_8);
            EList<JvmMember> _members_9 = type.getMembers();
            String _name_1 = variable.getName();
            String _plus_1 = ("getAllValuesOf" + _name_1);
            final Procedure1<JvmOperation> _function_9 = (JvmOperation it) -> {
              this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocGetAllValuesOfMethod(variable).toString());
              StringConcatenationClient _client = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append("return rawStreamAllValuesOf");
                  String _name = variable.getName();
                  _builder.append(_name);
                  _builder.append("(emptyArray()).collect(");
                  _builder.append(Collectors.class);
                  _builder.append(".toSet());");
                  _builder.newLineIfNotEmpty();
                }
              };
              this._eMFJvmTypesBuilder.setBody(it, _client);
            };
            JvmOperation _method_9 = this._eMFJvmTypesBuilder.toMethod(pattern, _plus_1, this.builder.typeRef(Set.class, typeOfVariable), _function_9);
            this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_9, _method_9);
            EList<JvmMember> _members_10 = type.getMembers();
            String _name_2 = variable.getName();
            String _plus_2 = ("streamAllValuesOf" + _name_2);
            final Procedure1<JvmOperation> _function_10 = (JvmOperation it) -> {
              this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocGetAllValuesOfMethod(variable).toString());
              StringConcatenationClient _client = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append("return rawStreamAllValuesOf");
                  String _name = variable.getName();
                  _builder.append(_name);
                  _builder.append("(emptyArray());");
                  _builder.newLineIfNotEmpty();
                }
              };
              this._eMFJvmTypesBuilder.setBody(it, _client);
            };
            JvmOperation _method_10 = this._eMFJvmTypesBuilder.toMethod(pattern, _plus_2, this.builder.typeRef(Stream.class, typeOfVariable), _function_10);
            this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_10, _method_10);
            int _size = pattern.getParameters().size();
            boolean _greaterThan = (_size > 1);
            if (_greaterThan) {
              EList<JvmMember> _members_11 = type.getMembers();
              String _name_3 = variable.getName();
              String _plus_3 = ("streamAllValuesOf" + _name_3);
              final Procedure1<JvmOperation> _function_11 = (JvmOperation it) -> {
                this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocStreamAllValuesOfMethod(variable).toString());
                EList<JvmFormalParameter> _parameters_1 = it.getParameters();
                JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "partialMatch", this.builder.typeRef(matchClass));
                this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("return rawStreamAllValuesOf");
                    String _name = variable.getName();
                    _builder.append(_name);
                    _builder.append("(partialMatch.toArray());");
                    _builder.newLineIfNotEmpty();
                  }
                };
                this._eMFJvmTypesBuilder.setBody(it, _client);
              };
              JvmOperation _method_11 = this._eMFJvmTypesBuilder.toMethod(variable, _plus_3, this.builder.typeRef(Stream.class, typeOfVariable), _function_11);
              this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_11, _method_11);
              EList<JvmMember> _members_12 = type.getMembers();
              String _name_4 = variable.getName();
              String _plus_4 = ("streamAllValuesOf" + _name_4);
              final Procedure1<JvmOperation> _function_12 = (JvmOperation it) -> {
                this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocStreamAllValuesOfMethod(variable).toString());
                EList<Variable> _parameters_1 = pattern.getParameters();
                for (final Variable parameter : _parameters_1) {
                  boolean _notEquals = (!Objects.equal(parameter, variable));
                  if (_notEquals) {
                    EList<JvmFormalParameter> _parameters_2 = it.getParameters();
                    JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
                    this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter);
                  }
                }
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("return rawStreamAllValuesOf");
                    String _name = variable.getName();
                    _builder.append(_name);
                    _builder.append("(new Object[]{");
                    {
                      EList<Variable> _parameters = pattern.getParameters();
                      boolean _hasElements = false;
                      for(final Variable p : _parameters) {
                        if (!_hasElements) {
                          _hasElements = true;
                        } else {
                          _builder.appendImmediate(", ", "");
                        }
                        String _xifexpression = null;
                        String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                        String _parameterName_1 = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(variable);
                        boolean _equals = Objects.equal(_parameterName, _parameterName_1);
                        if (_equals) {
                          _xifexpression = "null";
                        } else {
                          _xifexpression = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                        }
                        _builder.append(_xifexpression);
                      }
                    }
                    _builder.append("});");
                    _builder.newLineIfNotEmpty();
                  }
                };
                this._eMFJvmTypesBuilder.setBody(it, _client);
              };
              JvmOperation _method_12 = this._eMFJvmTypesBuilder.toMethod(variable, _plus_4, this.builder.typeRef(Stream.class, typeOfVariable), _function_12);
              this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_12, _method_12);
              EList<JvmMember> _members_13 = type.getMembers();
              String _name_5 = variable.getName();
              String _plus_5 = ("getAllValuesOf" + _name_5);
              final Procedure1<JvmOperation> _function_13 = (JvmOperation it) -> {
                this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocGetAllValuesOfMethod(variable).toString());
                EList<JvmFormalParameter> _parameters_1 = it.getParameters();
                JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "partialMatch", this.builder.typeRef(matchClass));
                this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter);
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("return rawStreamAllValuesOf");
                    String _name = variable.getName();
                    _builder.append(_name);
                    _builder.append("(partialMatch.toArray()).collect(");
                    _builder.append(Collectors.class);
                    _builder.append(".toSet());");
                    _builder.newLineIfNotEmpty();
                  }
                };
                this._eMFJvmTypesBuilder.setBody(it, _client);
              };
              JvmOperation _method_13 = this._eMFJvmTypesBuilder.toMethod(variable, _plus_5, this.builder.typeRef(Set.class, typeOfVariable), _function_13);
              this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_13, _method_13);
              EList<JvmMember> _members_14 = type.getMembers();
              String _name_6 = variable.getName();
              String _plus_6 = ("getAllValuesOf" + _name_6);
              final Procedure1<JvmOperation> _function_14 = (JvmOperation it) -> {
                this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocGetAllValuesOfMethod(variable).toString());
                EList<Variable> _parameters_1 = pattern.getParameters();
                for (final Variable parameter : _parameters_1) {
                  boolean _notEquals = (!Objects.equal(parameter, variable));
                  if (_notEquals) {
                    EList<JvmFormalParameter> _parameters_2 = it.getParameters();
                    JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(parameter, this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(parameter), this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(parameter));
                    this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter);
                  }
                }
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("return rawStreamAllValuesOf");
                    String _name = variable.getName();
                    _builder.append(_name);
                    _builder.append("(new Object[]{");
                    {
                      EList<Variable> _parameters = pattern.getParameters();
                      boolean _hasElements = false;
                      for(final Variable p : _parameters) {
                        if (!_hasElements) {
                          _hasElements = true;
                        } else {
                          _builder.appendImmediate(", ", "");
                        }
                        String _xifexpression = null;
                        String _parameterName = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                        String _parameterName_1 = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(variable);
                        boolean _equals = Objects.equal(_parameterName, _parameterName_1);
                        if (_equals) {
                          _xifexpression = "null";
                        } else {
                          _xifexpression = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
                        }
                        _builder.append(_xifexpression);
                      }
                    }
                    _builder.append("}).collect(");
                    _builder.append(Collectors.class);
                    _builder.append(".toSet());");
                    _builder.newLineIfNotEmpty();
                  }
                };
                this._eMFJvmTypesBuilder.setBody(it, _client);
              };
              JvmOperation _method_14 = this._eMFJvmTypesBuilder.toMethod(variable, _plus_6, this.builder.typeRef(Set.class, typeOfVariable), _function_14);
              this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_14, _method_14);
            }
          }
        }
      } else {
        EList<JvmMember> _members_8 = type.getMembers();
        final Procedure1<JvmOperation> _function_8 = (JvmOperation it) -> {
          this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocHasMatchMethodNoParameter(pattern).toString());
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("return rawHasMatch(new Object[]{});");
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        };
        JvmOperation _method_8 = this._eMFJvmTypesBuilder.toMethod(pattern, "hasMatch", this.builder.typeRef(boolean.class), _function_8);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_8, _method_8);
      }
      _xblockexpression = this.inferMatcherClassToMatchMethods(type, pattern, matchClass);
    }
    return _xblockexpression;
  }
  
  /**
   * Infers tupleToMatch, arrayToMatch methods for Matcher class based on the input 'pattern'.
   */
  public boolean inferMatcherClassToMatchMethods(final JvmDeclaredType matcherClass, final Pattern pattern, final JvmType matchClass) {
    boolean _xblockexpression = false;
    {
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
        this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        it.setVisibility(JvmVisibility.PROTECTED);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "t", this.builder.typeRef(Tuple.class));
        this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
      };
      final JvmOperation tupleToMatchMethod = this._eMFJvmTypesBuilder.toMethod(pattern, "tupleToMatch", this.builder.typeRef(matchClass), _function);
      final Procedure1<JvmOperation> _function_1 = (JvmOperation it) -> {
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
        this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        it.setVisibility(JvmVisibility.PROTECTED);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "match", this._eMFJvmTypesBuilder.addArrayTypeDimension(this.builder.typeRef(Object.class)));
        this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
      };
      final JvmOperation arrayToMatchMethod = this._eMFJvmTypesBuilder.toMethod(pattern, "arrayToMatch", this.builder.typeRef(matchClass), _function_1);
      final Procedure1<JvmOperation> _function_2 = (JvmOperation it) -> {
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
        this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        it.setVisibility(JvmVisibility.PROTECTED);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "match", this._eMFJvmTypesBuilder.addArrayTypeDimension(this.builder.typeRef(Object.class)));
        this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
      };
      final JvmOperation arrayToMatchMutableMethod = this._eMFJvmTypesBuilder.toMethod(pattern, "arrayToMatchMutable", this.builder.typeRef(matchClass), _function_2);
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          StringConcatenationClient _inferTupleToMatchMethodBody = PatternMatcherClassInferrer.this.inferTupleToMatchMethodBody(pattern, matchClass);
          _builder.append(_inferTupleToMatchMethodBody);
        }
      };
      this._eMFJvmTypesBuilder.setBody(tupleToMatchMethod, _client);
      StringConcatenationClient _client_1 = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          StringConcatenationClient _inferArrayToMatchMethodBody = PatternMatcherClassInferrer.this.inferArrayToMatchMethodBody(pattern, matchClass);
          _builder.append(_inferArrayToMatchMethodBody);
        }
      };
      this._eMFJvmTypesBuilder.setBody(arrayToMatchMethod, _client_1);
      StringConcatenationClient _client_2 = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          StringConcatenationClient _inferArrayToMatchMutableMethodBody = PatternMatcherClassInferrer.this.inferArrayToMatchMutableMethodBody(pattern, matchClass);
          _builder.append(_inferArrayToMatchMutableMethodBody);
        }
      };
      this._eMFJvmTypesBuilder.setBody(arrayToMatchMutableMethod, _client_2);
      EList<JvmMember> _members = matcherClass.getMembers();
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members, tupleToMatchMethod);
      EList<JvmMember> _members_1 = matcherClass.getMembers();
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_1, arrayToMatchMethod);
      EList<JvmMember> _members_2 = matcherClass.getMembers();
      _xblockexpression = this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_2, arrayToMatchMutableMethod);
    }
    return _xblockexpression;
  }
  
  /**
   * Infers the tupleToMatch method body.
   */
  public StringConcatenationClient inferTupleToMatchMethodBody(final Pattern pattern, final JvmType matchClass) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("try {");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("return ");
        _builder.append(matchClass, "    ");
        _builder.append(".newMatch(");
        {
          EList<Variable> _parameters = pattern.getParameters();
          boolean _hasElements = false;
          for(final Variable p : _parameters) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "    ");
            }
            _builder.append("(");
            JvmType _type = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(p).getType();
            _builder.append(_type, "    ");
            _builder.append(") t.get(");
            String _positionConstant = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.positionConstant(p);
            _builder.append(_positionConstant, "    ");
            _builder.append(")");
          }
        }
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        _builder.append("} catch(ClassCastException e) {");
        _builder.newLine();
        _builder.append("    ");
        CharSequence _inferErrorLogging = PatternMatcherClassInferrer.this.inferErrorLogging("Element(s) in tuple not properly typed!", "e");
        _builder.append(_inferErrorLogging, "    ");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        _builder.append("return null;");
        _builder.newLine();
        _builder.append("}");
        _builder.newLine();
      }
    };
    return _client;
  }
  
  /**
   * Infers the arrayToMatch method body.
   */
  public StringConcatenationClient inferArrayToMatchMethodBody(final Pattern pattern, final JvmType matchClass) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("try {");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("return ");
        _builder.append(matchClass, "    ");
        _builder.append(".newMatch(");
        {
          EList<Variable> _parameters = pattern.getParameters();
          boolean _hasElements = false;
          for(final Variable p : _parameters) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "    ");
            }
            _builder.append("(");
            JvmType _type = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(p).getType();
            _builder.append(_type, "    ");
            _builder.append(") match[");
            String _positionConstant = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.positionConstant(p);
            _builder.append(_positionConstant, "    ");
            _builder.append("]");
          }
        }
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        _builder.append("} catch(ClassCastException e) {");
        _builder.newLine();
        _builder.append("    ");
        CharSequence _inferErrorLogging = PatternMatcherClassInferrer.this.inferErrorLogging("Element(s) in array not properly typed!", "e");
        _builder.append(_inferErrorLogging, "    ");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        _builder.append("return null;");
        _builder.newLine();
        _builder.append("}");
        _builder.newLine();
      }
    };
    return _client;
  }
  
  /**
   * Infers the arrayToMatch method body.
   */
  public StringConcatenationClient inferArrayToMatchMutableMethodBody(final Pattern pattern, final JvmType matchClass) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("try {");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("return ");
        _builder.append(matchClass, "    ");
        _builder.append(".newMutableMatch(");
        {
          EList<Variable> _parameters = pattern.getParameters();
          boolean _hasElements = false;
          for(final Variable p : _parameters) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "    ");
            }
            _builder.append("(");
            JvmType _type = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.calculateType(p).getType();
            _builder.append(_type, "    ");
            _builder.append(") match[");
            String _positionConstant = PatternMatcherClassInferrer.this._eMFPatternLanguageJvmModelInferrerUtil.positionConstant(p);
            _builder.append(_positionConstant, "    ");
            _builder.append("]");
          }
        }
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        _builder.append("} catch(ClassCastException e) {");
        _builder.newLine();
        _builder.append("    ");
        CharSequence _inferErrorLogging = PatternMatcherClassInferrer.this.inferErrorLogging("Element(s) in array not properly typed!", "e");
        _builder.append(_inferErrorLogging, "    ");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        _builder.append("return null;");
        _builder.newLine();
        _builder.append("}");
        _builder.newLine();
      }
    };
    return _client;
  }
  
  /**
   * Infers the appropriate logging based on the parameters.
   */
  public CharSequence inferErrorLogging(final String message, final String exceptionName) {
    CharSequence _xifexpression = null;
    if ((exceptionName == null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("LOGGER.error(\"");
      _builder.append(message);
      _builder.append("\");");
      _xifexpression = _builder;
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("LOGGER.error(\"");
      _builder_1.append(message);
      _builder_1.append("\",");
      _builder_1.append(exceptionName);
      _builder_1.append(");");
      _xifexpression = _builder_1;
    }
    return _xifexpression;
  }
}
