"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
const jsonSchemaService_1 = require("./services/jsonSchemaService");
const documentSymbols_1 = require("./services/documentSymbols");
const yamlCompletion_1 = require("./services/yamlCompletion");
const yamlHover_1 = require("./services/yamlHover");
const yamlValidation_1 = require("./services/yamlValidation");
const yamlFormatter_1 = require("./services/yamlFormatter");
const vscode_json_languageservice_1 = require("vscode-json-languageservice");
function getLanguageService(schemaRequestService, workspaceContext, contributions, promiseConstructor) {
    const promise = promiseConstructor || Promise;
    const jsonLanguageService = vscode_json_languageservice_1.getLanguageService({
        schemaRequestService,
        workspaceContext
    });
    const schemaService = new jsonSchemaService_1.JSONSchemaService(schemaRequestService, workspaceContext);
    const completer = new yamlCompletion_1.YAMLCompletion(schemaService, contributions, promise);
    const hover = new yamlHover_1.YAMLHover(promise, jsonLanguageService);
    const yamlDocumentSymbols = new documentSymbols_1.YAMLDocumentSymbols(jsonLanguageService);
    const yamlValidation = new yamlValidation_1.YAMLValidation(promise, jsonLanguageService);
    const formatter = new yamlFormatter_1.YAMLFormatter();
    return {
        configure: settings => {
            jsonLanguageService.configure(settings);
            schemaService.clearExternalSchemas();
            if (settings.schemas) {
                settings.schemas.forEach(settings => {
                    schemaService.registerExternalSchema(settings.uri, settings.fileMatch, settings.schema);
                });
            }
            yamlValidation.configure(settings);
            hover.configure(settings);
            const customTagsSetting = settings && settings['customTags'] ? settings['customTags'] : [];
            completer.configure(settings, customTagsSetting);
            formatter.configure(settings);
        },
        registerCustomSchemaProvider: (schemaProvider) => {
            schemaService.registerCustomSchemaProvider(schemaProvider);
        },
        doComplete: completer.doComplete.bind(completer),
        doResolve: completer.doResolve.bind(completer),
        doValidation: yamlValidation.doValidation.bind(yamlValidation),
        doHover: hover.doHover.bind(hover),
        findDocumentSymbols: yamlDocumentSymbols.findDocumentSymbols.bind(yamlDocumentSymbols),
        findDocumentSymbols2: yamlDocumentSymbols.findHierarchicalDocumentSymbols.bind(yamlDocumentSymbols),
        resetSchema: (uri) => {
            jsonLanguageService.resetSchema(uri);
            return schemaService.onResourceChange(uri);
        },
        doFormat: formatter.format.bind(formatter)
    };
}
exports.getLanguageService = getLanguageService;
//# sourceMappingURL=yamlLanguageService.js.map