/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
'use strict';
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const yamlParser07_1 = require("../parser/yamlParser07");
class YAMLValidation {
    constructor(promiseConstructor, jsonLanguageService) {
        this.MATCHES_MULTIPLE = 'Matches multiple schemas when only one must validate.';
        this.promise = promiseConstructor;
        this.validationEnabled = true;
        this.jsonLanguageService = jsonLanguageService;
    }
    configure(settings) {
        if (settings) {
            this.validationEnabled = settings.validate;
            this.customTags = settings.customTags;
        }
    }
    doValidation(textDocument, isKubernetes = false) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.validationEnabled) {
                return this.promise.resolve([]);
            }
            const yamlDocument = yamlParser07_1.parse(textDocument.getText(), this.customTags);
            const validationResult = [];
            for (const currentYAMLDoc of yamlDocument.documents) {
                currentYAMLDoc.isKubernetes = isKubernetes;
                const validation = yield this.jsonLanguageService.doValidation(textDocument, currentYAMLDoc);
                const syd = currentYAMLDoc;
                if (syd.errors.length > 0) {
                    validationResult.push(...syd.errors);
                }
                validationResult.push(...validation);
            }
            const foundSignatures = new Set();
            const duplicateMessagesRemoved = [];
            for (const err of validationResult) {
                /**
                 * A patch ontop of the validation that removes the
                 * 'Matches many schemas' error for kubernetes
                 * for a better user experience.
                 */
                if (isKubernetes && err.message === this.MATCHES_MULTIPLE) {
                    continue;
                }
                const errSig = err.range.start.line + ' ' + err.range.start.character + ' ' + err.message;
                if (!foundSignatures.has(errSig)) {
                    duplicateMessagesRemoved.push(err);
                    foundSignatures.add(errSig);
                }
            }
            return duplicateMessagesRemoved;
        });
    }
}
exports.YAMLValidation = YAMLValidation;
//# sourceMappingURL=yamlValidation.js.map