/*******************************************************************************
 * Copyright (c) 2004-2008 Gabor Bergmann and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 *******************************************************************************/

package org.eclipse.viatra.query.runtime.rete.index;

import java.util.Collection;
import java.util.Map;

import org.eclipse.viatra.query.runtime.matchers.tuple.Tuple;
import org.eclipse.viatra.query.runtime.matchers.tuple.TupleMask;
import org.eclipse.viatra.query.runtime.rete.network.Direction;
import org.eclipse.viatra.query.runtime.rete.network.ReteContainer;
import org.eclipse.viatra.query.runtime.rete.network.communication.Timestamp;

/**
 * @author Gabor Bergmann
 * 
 */
public class JoinNode extends DualInputNode {

    public JoinNode(final ReteContainer reteContainer, final TupleMask complementerSecondaryMask) {
        super(reteContainer, complementerSecondaryMask);
        this.logic = createLogic();
    }

    @Override
    public Tuple calibrate(final Tuple primary, final Tuple secondary) {
        return unify(primary, secondary);
    }

    private final NetworkStructureChangeSensitiveLogic TIMELESS = new NetworkStructureChangeSensitiveLogic() {

        @Override
        public void pullIntoWithTimestamp(final Map<Tuple, Timestamp> collector, final boolean flush) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void pullInto(final Collection<Tuple> collector, final boolean flush) {
            if (primarySlot == null || secondarySlot == null) {
                return;
            }

            if (flush) {
                reteContainer.flushUpdates();
            }

            for (final Tuple signature : primarySlot.getSignatures()) {
                // primaries can not be null due to the contract of IterableIndex.getSignatures()
                final Collection<Tuple> primaries = primarySlot.get(signature);
                final Collection<Tuple> opposites = secondarySlot.get(signature);
                if (opposites != null) {
                    for (final Tuple primary : primaries) {
                        for (final Tuple opposite : opposites) {
                            collector.add(unify(primary, opposite));
                        }
                    }
                }
            }
        }

        @Override
        public void notifyUpdate(final Side side, final Direction direction, final Tuple updateElement,
                final Tuple signature, final boolean change, final Timestamp timestamp) {
            // in the default case, all timestamps must be zero
            assert timestamp == Timestamp.ZERO;

            final Collection<Tuple> opposites = retrieveOpposites(side, signature);

            if (!coincidence) {
                if (opposites != null) {
                    for (final Tuple opposite : opposites) {
                        propagateUpdate(direction, unify(side, updateElement, opposite), timestamp);
                    }
                }
            } else {
                // compensate for coincidence of slots - this is the case when an Indexer is joined with itself
                if (opposites != null) {
                    for (final Tuple opposite : opposites) {
                        if (opposite.equals(updateElement)) {
                            // handle self-joins of a single tuple separately
                            continue;
                        }
                        propagateUpdate(direction, unify(opposite, updateElement), timestamp);
                        propagateUpdate(direction, unify(updateElement, opposite), timestamp);
                    }
                }

                // handle self-joins here
                propagateUpdate(direction, unify(updateElement, updateElement), timestamp);
            }
        }
    };

    private final NetworkStructureChangeSensitiveLogic TIMELY = new NetworkStructureChangeSensitiveLogic() {

        @Override
        public void pullIntoWithTimestamp(final Map<Tuple, Timestamp> collector, final boolean flush) {
            if (primarySlot == null || secondarySlot == null) {
                return;
            }

            if (flush) {
                reteContainer.flushUpdates();
            }

            for (final Tuple signature : primarySlot.getSignatures()) {
                // primaries can not be null due to the contract of IterableIndex.getSignatures()
                final Map<Tuple, Timestamp> primaries = getWithTimestamp(signature, primarySlot);
                final Map<Tuple, Timestamp> opposites = getWithTimestamp(signature, secondarySlot);
                if (opposites != null) {
                    for (final Tuple primary : primaries.keySet()) {
                        final Timestamp primaryTimestamp = primaries.get(primary);
                        for (final Tuple opposite : opposites.keySet()) {
                            collector.put(unify(primary, opposite), primaryTimestamp.max(opposites.get(opposite)));
                        }
                    }
                }
            }
        }

        @Override
        public void pullInto(final Collection<Tuple> collector, final boolean flush) {
            JoinNode.this.TIMELESS.pullInto(collector, flush);
        }

        @Override
        public void notifyUpdate(final Side side, final Direction direction, final Tuple updateElement,
                final Tuple signature, final boolean change, final Timestamp timestamp) {
            final Indexer oppositeIndexer = getSlot(side.opposite());
            final Map<Tuple, Timestamp> opposites = getWithTimestamp(signature, oppositeIndexer);

            if (!coincidence) {
                if (opposites != null) {
                    for (final Tuple opposite : opposites.keySet()) {
                        propagateUpdate(direction, unify(side, updateElement, opposite),
                                timestamp.max(opposites.get(opposite)));
                    }
                }
            } else {
                // compensate for coincidence of slots - this is the case when an Indexer is joined with itself
                if (opposites != null) {
                    for (final Tuple opposite : opposites.keySet()) {
                        if (opposite.equals(updateElement)) {
                            // handle self-joins of a single tuple separately
                            continue;
                        }
                        final Timestamp oppositeTimestamp = opposites.get(opposite);
                        propagateUpdate(direction, unify(opposite, updateElement),
                                timestamp.max(oppositeTimestamp));
                        propagateUpdate(direction, unify(updateElement, opposite),
                                timestamp.max(oppositeTimestamp));
                    }
                }

                // handle self-join here
                propagateUpdate(direction, unify(updateElement, updateElement), timestamp);
            }
        }
    };

    @Override
    protected NetworkStructureChangeSensitiveLogic createTimelessLogic() {
        return this.TIMELESS;
    }

    @Override
    protected NetworkStructureChangeSensitiveLogic createTimelyLogic() {
        return this.TIMELY;
    }

}
