/**
 * Copyright (c) 2010-2016, Tamas Szabo, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 *   Tamas Szabo, Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.emf.util;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.viatra.query.patternlanguage.emf.helper.JavaTypesHelper;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.vql.AggregatedValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ValueReference;
import org.eclipse.viatra.query.patternlanguage.emf.vql.VariableReference;
import org.eclipse.viatra.query.runtime.matchers.psystem.aggregations.AggregatorType;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmAnnotationValue;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeAnnotationValue;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * @author Tamas Szabo, Zoltan Ujhelyi
 * @since 2.0
 */
@SuppressWarnings("all")
public class AggregatorUtil {
  private final static String PARAMETER_TYPES_NAME = "parameterTypes";
  
  private final static String RETURN_TYPES_NAME = "returnTypes";
  
  private static List<JvmType> getAggregatorType(final JvmDeclaredType aggregatorType, final String typeString) {
    List<JvmType> _xblockexpression = null;
    {
      final Function1<JvmAnnotationReference, Boolean> _function = (JvmAnnotationReference it) -> {
        String _qualifiedName = it.getAnnotation().getQualifiedName();
        String _name = AggregatorType.class.getName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, _name));
      };
      final JvmAnnotationReference annotationType = IterableExtensions.<JvmAnnotationReference>findFirst(aggregatorType.getAnnotations(), _function);
      EList<JvmAnnotationValue> _explicitValues = null;
      if (annotationType!=null) {
        _explicitValues=annotationType.getExplicitValues();
      }
      JvmAnnotationValue _findFirst = null;
      if (_explicitValues!=null) {
        final Function1<JvmAnnotationValue, Boolean> _function_1 = (JvmAnnotationValue it) -> {
          String _valueName = it.getValueName();
          return Boolean.valueOf(Objects.equal(_valueName, typeString));
        };
        _findFirst=IterableExtensions.<JvmAnnotationValue>findFirst(_explicitValues, _function_1);
      }
      final JvmAnnotationValue annotationValue = _findFirst;
      List<JvmType> _xifexpression = null;
      if ((annotationValue instanceof JvmTypeAnnotationValue)) {
        final Function1<JvmTypeReference, JvmType> _function_2 = (JvmTypeReference it) -> {
          return it.getType();
        };
        _xifexpression = ListExtensions.<JvmTypeReference, JvmType>map(((JvmTypeAnnotationValue)annotationValue).getValues(), _function_2);
      } else {
        _xifexpression = Collections.<JvmType>emptyList();
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public static List<JvmType> getReturnTypes(final JvmDeclaredType aggregatorType) {
    return AggregatorUtil.getAggregatorType(aggregatorType, AggregatorUtil.RETURN_TYPES_NAME);
  }
  
  public static List<JvmType> getParameterTypes(final JvmDeclaredType aggregatorType) {
    return AggregatorUtil.getAggregatorType(aggregatorType, AggregatorUtil.PARAMETER_TYPES_NAME);
  }
  
  /**
   * An aggregator expression may only have aggregated value as parameters if the corresponding {@link AggregatorType} annotation
   * does not define a single Void parameter. However, in that case, it _must_ have an aggregate parameter.
   */
  public static boolean mustHaveAggregatorVariables(final AggregatedValue value) {
    boolean _xblockexpression = false;
    {
      final List<JvmType> types = AggregatorUtil.getParameterTypes(value.getAggregator());
      _xblockexpression = ((types != null) && ((types.size() > 1) || ((types.size() == 1) && (!JavaTypesHelper.is(types.get(0), Void.class)))));
    }
    return _xblockexpression;
  }
  
  public static int getAggregateVariableIndex(final AggregatedValue value) {
    int index = 0;
    List<ValueReference> _callParameters = PatternLanguageHelper.getCallParameters(value.getCall());
    for (final ValueReference param : _callParameters) {
      {
        if (((param instanceof VariableReference) && ((VariableReference) param).isAggregator())) {
          return index;
        }
        index++;
      }
    }
    return (-1);
  }
  
  private final static Function1<VariableReference, Boolean> aggregator = ((Function1<VariableReference, Boolean>) (VariableReference v) -> {
    return Boolean.valueOf(v.isAggregator());
  });
  
  /**
   * Returns the aggregate variable the aggregator should work with. Given in a well-formed AggregatedValue only a
   * single aggregate variable should be present, this should be unique.
   */
  public static VariableReference getAggregatorVariable(final AggregatedValue value) {
    return IterableExtensions.<VariableReference>findFirst(Iterables.<VariableReference>filter(PatternLanguageHelper.getCallParameters(value.getCall()), VariableReference.class), AggregatorUtil.aggregator);
  }
  
  /**
   * Returns all aggregate variables of the AggregatedValue. If the AggregatedValue has more aggregate variables,
   * it represents an error in the specification.
   */
  public static List<VariableReference> getAllAggregatorVariables(final AggregatedValue value) {
    return ImmutableList.<VariableReference>copyOf(IterableExtensions.<VariableReference>filter(Iterables.<VariableReference>filter(PatternLanguageHelper.getCallParameters(value.getCall()), VariableReference.class), AggregatorUtil.aggregator));
  }
}
