## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 4,
  dpi = 96,
  message = FALSE,
  warning = FALSE,
  fig.alt = "Figure generated by tikatuwq package"
)

## ----load-package-------------------------------------------------------------
library(tikatuwq)
data("wq_demo", package = "tikatuwq")

## ----iqa-weights--------------------------------------------------------------
# Default weights
default_weights <- c(
  od = 0.17,
  coliformes = 0.15,
  dbo = 0.10,
  nt_total = 0.10,
  p_total = 0.10,
  turbidez = 0.08,
  tds = 0.08,
  pH = 0.12,
  temperatura = 0.10
)

# Check sum equals 1
sum(default_weights)

## ----compute-iqa--------------------------------------------------------------
# Compute IQA with default settings
df_iqa <- iqa(wq_demo, na_rm = TRUE)

# View results
cols_show <- intersect(c("ponto", "IQA", "IQA_status"), names(df_iqa))
head(df_iqa[, cols_show, drop = FALSE])

# Distribution
hist(df_iqa$IQA, breaks = 10, main = "IQA Distribution", xlab = "IQA")

## ----iqa-missing--------------------------------------------------------------
# Example with missing parameters
df_missing <- wq_demo
df_missing$tds <- NULL  # Remove one parameter

df_iqa_missing <- iqa(df_missing, na_rm = TRUE)
head(df_iqa_missing$IQA)

## ----iqa-custom-weights-------------------------------------------------------
# Custom weights (must sum to 1)
custom_weights <- c(
  od = 0.20,
  coliformes = 0.20,
  dbo = 0.10,
  nt_total = 0.10,
  p_total = 0.10,
  turbidez = 0.10,
  tds = 0.10,
  pH = 0.05,
  temperatura = 0.05
)

df_iqa_custom <- iqa(wq_demo, pesos = custom_weights, na_rm = TRUE)
cols_show2 <- intersect(c("IQA", "IQA_status"), names(df_iqa_custom))
head(df_iqa_custom[, cols_show2, drop = FALSE])

## ----iqa-classification-------------------------------------------------------
# Classification function
classify_iqa(c(15, 40, 65, 80, 95))

# English labels
classify_iqa(c(15, 40, 65, 80, 95), locale = "en")

# Distribution in demo data
table(df_iqa$IQA_status)

## ----iet-carlson, eval=FALSE--------------------------------------------------
# # Example dataset with required parameters
# # df_lake <- data.frame(
# #   ponto = c("L1", "L2"),
# #   secchi = c(2.5, 1.0),  # meters
# #   clorofila = c(5, 20),  # ug/L
# #   p_total = c(0.02, 0.10)  # mg/L (converted to ug/L internally)
# # )
# #
# # iet_carlson(df_lake, .keep_ids = TRUE)

## ----iet-lamparelli, eval=FALSE-----------------------------------------------
# # iet_lamparelli(df_lake, .keep_ids = TRUE)

## ----trend-single-------------------------------------------------------------
# Add temporal structure to demo data
df_temporal <- wq_demo
df_temporal$data <- as.Date("2025-01-01") + seq_len(nrow(df_temporal)) - 1

# Compute trend for turbidity
trend_result <- trend_param(df_temporal, param = "turbidez")

print(trend_result)

## ----plot-trend---------------------------------------------------------------
library(ggplot2)

# Plot with trend line
p_trend <- plot_trend(df_temporal, param = "turbidez", method = "theilsen")
print(p_trend)

# With LOESS smoothing
p_loess <- plot_trend(df_temporal, param = "turbidez", method = "loess")
print(p_loess)

## ----trend-multi--------------------------------------------------------------
# Trends for multiple parameters
params <- c("turbidez", "od", "dbo")
trends_multi <- param_trend_multi(df_temporal, parametros = params)

print(trends_multi)

## ----param-summary------------------------------------------------------------
# Summary for one parameter
summary_turb <- param_summary(df_temporal, parametro = "turbidez")
print(summary_turb)

# Multi-parameter summary
summary_multi <- param_summary_multi(df_temporal, parametros = c("turbidez", "od", "dbo"))
print(summary_multi)

## ----param-plot---------------------------------------------------------------
# Single parameter plot
p1 <- param_plot(df_temporal, parametro = "turbidez")
print(p1)

# Multi-parameter plot
p2 <- param_plot_multi(df_temporal, parametros = c("turbidez", "od", "dbo"))
print(p2)

## ----theil-sen-details, eval=FALSE--------------------------------------------
# # Theil-Sen computes median of all pairwise slopes
# # For data with outliers, it is more reliable than OLS
# # Used by default in trend_param() and plot_trend()

## ----spearman-details, eval=FALSE---------------------------------------------
# # Spearman correlation tests for monotonic relationship
# # Does not assume linearity
# # p-value indicates significance of trend

