## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  warning = FALSE, message = FALSE
)

## ----out.width = "850px", dpi = 200, echo=FALSE-------------------------------
knitr::include_graphics("flowchart.png")

## ----example1a----------------------------------------------------------------
# Load the fireexposuR library
library(fireexposuR)

# Load the terra library for reading/writing spatial data
library(terra)

# read example hazard data
hazard_file_path <- "extdata/hazard.tif"
hazard <- terra::rast(system.file(hazard_file_path, package = "fireexposuR"))

# read example non-burnable data
nb_file_path <- "extdata/nb.tif"
no_burn <- terra::rast(system.file(nb_file_path, package = "fireexposuR"))


## ----example1b----------------------------------------------------------------
# compute long-range ember exposure using the long-range hazard raster
exposure <- fire_exp(hazard, t_dist = 500, no_burn = no_burn)


# compute long-range ember exposure without removing non-burnable cells
exposure_all_cells <- fire_exp(hazard, t_dist = 500)

## ----example1c, fig.width=6, fig.height=5.5,  fig.show="hold", out.width="50%"----
# visualize in a map
fire_exp_map(exposure, title = "Long-range exposure (non-burnable removed)")

fire_exp_map(exposure_all_cells, title = "Long-range exposure (all cells)")

## ----example1d----------------------------------------------------------------
# visualize in a summary table
fire_exp_summary(exposure, classify = "landscape")

fire_exp_summary(exposure_all_cells, classify = "landscape")


## ----example2, fig.width=6, fig.height=5.5------------------------------------
# read example area of interest
polygon_path <- system.file("extdata", "polygon.shp", package ="fireexposuR")
aoi <- terra::vect(polygon_path)

# assess directional exposure for a single point
transects <- fire_exp_dir(exposure, aoi)

# visualize the transects in a map
fire_exp_dir_map(transects, value = aoi)


## ----example3a----------------------------------------------------------------
# Load the fireexposuR library
library(fireexposuR)

# Load the terra library for reading/writing spatial data
library(terra)

# read raster of landcover types
fuel_file_path <- "extdata/fuel.tif"
fuel <- terra::rast(system.file(fuel_file_path, package = "fireexposuR"))

## ----example3b, echo = FALSE--------------------------------------------------
terra::plot(fuel, type = "classes", levels = c("Conifer Forest",
                                               "Mixedwood Forest",
                                               "Shrubland",
                                               "Grassland",
                                               "Wetland",
                                               "Barren Land",
                                               "Urban and Built Up",
                                               "Open Water"),
            col = c("darkgreen",
                    "green3",
                    "brown",
                    "yellow3",
                    "lightblue1",
                    "grey",
                    "grey40",
                    "blue"))

## ----example3c----------------------------------------------------------------
# create a reclassification matrix based on the raster values
hazard_matrix_long <- matrix(c(1, 1,  # Conifer Forest
                               6, 1,  # Mixedwood Forest
                               8, 0,  # Shrubland
                               10, 0, # Grassland
                               14, 0, # Wetland
                               16, 0, # Barren Land
                               17, 0, # Urban and Built Up
                               18, 0  # Open Water
                               ), ncol = 2, byrow = TRUE)

# use the long-range matrix to reclassify the cells in the fuel grid raster
hazard_long <- terra::classify(fuel, hazard_matrix_long)


# create a reclassification matrix based on the raster values
hazard_matrix_short <- matrix(c(1, 1,  # Conifer Forest
                                6, 1,  # Mixedwood Forest
                                8, 1,  # Shrubland
                                10, 1, # Grassland
                                14, 0, # Wetland
                                16, 0, # Barren Land
                                17, 0, # Urban and Built Up
                                18, 0  # Open Water
                                ), ncol = 2, byrow = TRUE)

# use the short-range matrix to reclassify the cells in the fuel grid raster
hazard_short <- terra::classify(fuel, hazard_matrix_short)

## ----example3d, echo = FALSE, fig.show="hold", out.width="50%", fig.height=6----
terra::plot(hazard_long, main = "Long-range hazard")

terra::plot(hazard_short, main = "short-range hazard")

set.seed(0)


## ----example3e----------------------------------------------------------------
# compute long-range ember exposure using the long-range hazard layer
exposure_long <- fire_exp(hazard_long, t_dist = 500)

# compute short-range ember exposure using the short-range hazard layer
exposure_short <- fire_exp(hazard_short, t_dist = 100)

## ----example3f, fig.show="hold", out.width="50%", fig.height=6----------------
# read example area of interest
polygon_path <- system.file("extdata", "polygon.shp", package ="fireexposuR")
aoi <- terra::vect(polygon_path)


fire_exp_map(exposure_long, aoi = aoi, 
             classify = "local", 
             title = "Long-range exposure")

fire_exp_map(exposure_short, aoi = aoi, 
             classify = "local", 
             title = "short-range exposure")


## ----example3g, fig.show="hold", out.width="50%", fig.height=6----------------
# randomly generate points to represent values within the area of interest
values <- terra::spatSample(aoi, 20)

exposure_short_values <- fire_exp_extract(exposure_short, values)

exposure_long_values <- fire_exp_extract(exposure_long, values)

fire_exp_extract_map(exposure_short_values,
             title = "Short-range exposure to values")

fire_exp_extract_map(exposure_long_values,
             title = "Long-range exposure to values")


## ----example4a, eval = FALSE--------------------------------------------------
# # Load the fireexposuR library
# library(fireexposuR)
# 
# # Load the terra library for reading/writing spatial data
# library(terra)
# 
# # PLEASE NOTE: The following code is to demonstrate an example workflow. The
# # referenced input data does not exist.
# 
# # read long-range ember hazard raster
# hazard <- rast("hazard.tif")
# 
# # read non-burnable landscape raster
# no_burn <- rast("no_burn.tif")
# 
# # compute long-range ember exposure
# exposure <- fire_exp(hazard, t_dist = 500, no_burn = no_burn)
# 
# # read the fire perimeters with terra
# fires <- vect("fires.shp")
# 
# # generate validation table, and then plot the results
# output <- fire_exp_validate(exposure, fires)
# fire_exp_validate_plot(output)

## ----example4b, eval = FALSE--------------------------------------------------
# # Load the fireexposuR library
# library(fireexposuR)
# 
# # Load the terra library for reading/writing spatial data
# library(terra)
# 
# # PLEASE NOTE: The following code is to demonstrate an example workflow. The
# # referenced input data does not exist.
# 
# # read long-range ember hazard raster
# hazard <- rast("hazard.tif")
# 
# # read non-burnable landscape raster
# no_burn <- rast("no_burn.tif")
# 
# # compute long-range ember exposure for default transmission distance
# exposure_a <- fire_exp(hazard, no_burn = no_burn)
# 
# # compute long-range ember exposure for a custom transmission distance
# exposure_b <- fire_exp(hazard, t_dist = 800, no_burn = no_burn)
# 
# # read the fire perimeters with terra
# fires <- vect("fires.shp")
# 
# # validation tables for both options
# output_a <- fire_exp_validate(exposure_a, fires)
# output_b <- fire_exp_validate(exposure_b, fires)

## ----example4c, eval = FALSE--------------------------------------------------
# # Load the fireexposuR library
# library(fireexposuR)
# 
# # Load the terra library for reading/writing spatial data
# library(terra)
# 
# # PLEASE NOTE: The following code is to demonstrate an example workflow. The
# # referenced input data does not exist.
# 
# # read long-range ember hazard raster: option A
# hazard_a <- rast("hazard_a.tif")
# 
# # read non-burnable landscape raster: option A
# no_burn_a <- rast("no_burn_a.tif")
# 
# # read long-range ember hazard raster: option B
# hazard_b <- rast("hazard_b.tif")
# 
# # read non-burnable landscape raster: option B
# no_burn_b <- rast("no_burn_b.tif")
# 
# # compute long-range ember exposure for option A
# exposure_a <- fire_exp(hazard_a, no_burn = no_burn_a)
# 
# # compute long-range ember exposure for option A
# exposure_b <- fire_exp(hazard_b, no_burn = no_burn_b)
# 
# # read the fire perimeters with terra
# fires <- vect("fires.shp")
# 
# # validation tables for both options
# output_a <- fire_exp_validate(exposure_a, fires)
# output_b <- fire_exp_validate(exposure_b, fires)

