﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-sync/CognitoSyncRequest.h>
#include <aws/cognito-sync/CognitoSync_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CognitoSync {
namespace Model {

/**
 * <p>A request to configure Cognito Events"</p>"<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/SetCognitoEventsRequest">AWS
 * API Reference</a></p>
 */
class SetCognitoEventsRequest : public CognitoSyncRequest {
 public:
  AWS_COGNITOSYNC_API SetCognitoEventsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SetCognitoEvents"; }

  AWS_COGNITOSYNC_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Cognito Identity Pool to use when configuring Cognito Events</p>
   */
  inline const Aws::String& GetIdentityPoolId() const { return m_identityPoolId; }
  inline bool IdentityPoolIdHasBeenSet() const { return m_identityPoolIdHasBeenSet; }
  template <typename IdentityPoolIdT = Aws::String>
  void SetIdentityPoolId(IdentityPoolIdT&& value) {
    m_identityPoolIdHasBeenSet = true;
    m_identityPoolId = std::forward<IdentityPoolIdT>(value);
  }
  template <typename IdentityPoolIdT = Aws::String>
  SetCognitoEventsRequest& WithIdentityPoolId(IdentityPoolIdT&& value) {
    SetIdentityPoolId(std::forward<IdentityPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The events to configure</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetEvents() const { return m_events; }
  inline bool EventsHasBeenSet() const { return m_eventsHasBeenSet; }
  template <typename EventsT = Aws::Map<Aws::String, Aws::String>>
  void SetEvents(EventsT&& value) {
    m_eventsHasBeenSet = true;
    m_events = std::forward<EventsT>(value);
  }
  template <typename EventsT = Aws::Map<Aws::String, Aws::String>>
  SetCognitoEventsRequest& WithEvents(EventsT&& value) {
    SetEvents(std::forward<EventsT>(value));
    return *this;
  }
  template <typename EventsKeyT = Aws::String, typename EventsValueT = Aws::String>
  SetCognitoEventsRequest& AddEvents(EventsKeyT&& key, EventsValueT&& value) {
    m_eventsHasBeenSet = true;
    m_events.emplace(std::forward<EventsKeyT>(key), std::forward<EventsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_identityPoolId;
  bool m_identityPoolIdHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_events;
  bool m_eventsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoSync
}  // namespace Aws
