/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.provider;

import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.rocketmq.tieredstore.MessageStoreConfig;
import org.apache.rocketmq.tieredstore.MessageStoreExecutor;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.common.FileSegmentType;
import org.apache.rocketmq.tieredstore.exception.TieredStoreErrorCode;
import org.apache.rocketmq.tieredstore.exception.TieredStoreException;
import org.apache.rocketmq.tieredstore.provider.FileSegmentProvider;
import org.apache.rocketmq.tieredstore.stream.FileSegmentInputStream;
import org.apache.rocketmq.tieredstore.stream.FileSegmentInputStreamFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class FileSegment
implements Comparable<FileSegment>,
FileSegmentProvider {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    protected static final Long GET_FILE_SIZE_ERROR = -1L;
    protected final long baseOffset;
    protected final String filePath;
    protected final FileSegmentType fileType;
    protected final MessageStoreConfig storeConfig;
    protected final long maxSize;
    protected final MessageStoreExecutor executor;
    protected final ReentrantLock fileLock = new ReentrantLock();
    protected final Semaphore commitLock = new Semaphore(1);
    protected volatile boolean closed = false;
    protected volatile long minTimestamp = Long.MAX_VALUE;
    protected volatile long maxTimestamp = Long.MAX_VALUE;
    protected volatile long commitPosition = 0L;
    protected volatile long appendPosition = 0L;
    protected volatile List<ByteBuffer> bufferList = new ArrayList<ByteBuffer>();
    protected volatile FileSegmentInputStream fileSegmentInputStream;
    protected volatile CompletableFuture<Boolean> flightCommitRequest;

    public FileSegment(MessageStoreConfig storeConfig, FileSegmentType fileType, String filePath, long baseOffset, MessageStoreExecutor executor) {
        this.storeConfig = storeConfig;
        this.fileType = fileType;
        this.filePath = filePath;
        this.baseOffset = baseOffset;
        this.executor = executor;
        this.maxSize = this.getMaxSizeByFileType();
    }

    @Override
    public int compareTo(FileSegment o) {
        return Long.compare(this.baseOffset, o.baseOffset);
    }

    public long getBaseOffset() {
        return this.baseOffset;
    }

    public void initPosition(long pos) {
        this.fileLock.lock();
        try {
            this.commitPosition = pos;
            this.appendPosition = pos;
        }
        finally {
            this.fileLock.unlock();
        }
    }

    public long getCommitPosition() {
        return this.commitPosition;
    }

    public long getAppendPosition() {
        return this.appendPosition;
    }

    public long getCommitOffset() {
        return this.baseOffset + this.commitPosition;
    }

    public long getAppendOffset() {
        return this.baseOffset + this.appendPosition;
    }

    public FileSegmentType getFileType() {
        return this.fileType;
    }

    public long getMaxSizeByFileType() {
        switch (this.fileType) {
            case COMMIT_LOG: {
                return this.storeConfig.getTieredStoreCommitLogMaxSize();
            }
            case CONSUME_QUEUE: {
                return this.storeConfig.getTieredStoreConsumeQueueMaxSize();
            }
        }
        return Long.MAX_VALUE;
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    public long getMinTimestamp() {
        return this.minTimestamp;
    }

    public void setMinTimestamp(long minTimestamp) {
        this.minTimestamp = minTimestamp;
    }

    public long getMaxTimestamp() {
        return this.maxTimestamp;
    }

    public void setMaxTimestamp(long maxTimestamp) {
        this.maxTimestamp = maxTimestamp;
    }

    public boolean isClosed() {
        return this.closed;
    }

    public void close() {
        this.fileLock.lock();
        try {
            this.closed = true;
        }
        finally {
            this.fileLock.unlock();
        }
    }

    protected List<ByteBuffer> borrowBuffer() {
        List<ByteBuffer> temp;
        this.fileLock.lock();
        try {
            temp = this.bufferList;
            this.bufferList = new ArrayList<ByteBuffer>();
        }
        finally {
            this.fileLock.unlock();
        }
        return temp;
    }

    protected void updateTimestamp(long timestamp) {
        this.fileLock.lock();
        try {
            if (this.maxTimestamp == Long.MAX_VALUE && this.minTimestamp == Long.MAX_VALUE) {
                this.maxTimestamp = timestamp;
                this.minTimestamp = timestamp;
                return;
            }
            this.maxTimestamp = Math.max(this.maxTimestamp, timestamp);
            this.minTimestamp = Math.min(this.minTimestamp, timestamp);
        }
        finally {
            this.fileLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AppendResult append(ByteBuffer buffer, long timestamp) {
        this.fileLock.lock();
        try {
            if (this.closed) {
                AppendResult appendResult = AppendResult.FILE_CLOSED;
                return appendResult;
            }
            if (this.appendPosition + (long)buffer.remaining() > this.maxSize) {
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.bufferList.size() >= this.storeConfig.getTieredStoreMaxGroupCommitCount()) {
                AppendResult appendResult = AppendResult.BUFFER_FULL;
                return appendResult;
            }
            this.appendPosition += (long)buffer.remaining();
            this.bufferList.add(buffer);
            this.updateTimestamp(timestamp);
        }
        finally {
            this.fileLock.unlock();
        }
        return AppendResult.SUCCESS;
    }

    public boolean needCommit() {
        return this.appendPosition > this.commitPosition;
    }

    public CompletableFuture<Boolean> commitAsync() {
        int bufferSize;
        if (this.closed) {
            return CompletableFuture.completedFuture(false);
        }
        if (!this.needCommit()) {
            return CompletableFuture.completedFuture(true);
        }
        if (this.commitLock.drainPermits() <= 0) {
            return CompletableFuture.completedFuture(false);
        }
        if (this.fileSegmentInputStream != null) {
            long fileSize = this.getSize();
            if (fileSize == GET_FILE_SIZE_ERROR) {
                log.error("FileSegment correct position error, fileName={}, commit={}, append={}, buffer={}", new Object[]{this.getPath(), this.commitPosition, this.appendPosition, this.fileSegmentInputStream.getContentLength()});
                this.releaseCommitLock();
                return CompletableFuture.completedFuture(false);
            }
            if (this.correctPosition(fileSize)) {
                this.fileSegmentInputStream = null;
            }
        }
        if (this.fileSegmentInputStream != null) {
            this.fileSegmentInputStream.rewind();
            bufferSize = this.fileSegmentInputStream.available();
        } else {
            List<ByteBuffer> bufferList = this.borrowBuffer();
            bufferSize = bufferList.stream().mapToInt(Buffer::remaining).sum();
            if (bufferSize == 0) {
                this.releaseCommitLock();
                return CompletableFuture.completedFuture(true);
            }
            this.fileSegmentInputStream = FileSegmentInputStreamFactory.build(this.fileType, this.getCommitOffset(), bufferList, null, bufferSize);
        }
        boolean append = this.fileType != FileSegmentType.INDEX;
        this.flightCommitRequest = ((CompletableFuture)((CompletableFuture)this.commit0(this.fileSegmentInputStream, this.commitPosition, bufferSize, append).thenApply(result -> {
            if (result.booleanValue()) {
                this.commitPosition += (long)bufferSize;
                this.fileSegmentInputStream = null;
                return true;
            }
            this.fileSegmentInputStream.rewind();
            return false;
        })).exceptionally(this::handleCommitException)).whenComplete((result, e) -> this.releaseCommitLock());
        return this.flightCommitRequest;
    }

    private boolean handleCommitException(Throwable e) {
        log.warn("FileSegment commit exception, filePath={}", (Object)this.filePath, (Object)e);
        Throwable rootCause = e.getCause() != null ? e.getCause() : e;
        long fileSize = rootCause instanceof TieredStoreException ? ((TieredStoreException)rootCause).getPosition() : this.getSize();
        long expectPosition = this.commitPosition + (long)this.fileSegmentInputStream.getContentLength();
        if (fileSize == GET_FILE_SIZE_ERROR) {
            log.error("Get file size error after commit, FileName: {}, Commit: {}, Content: {}, Expect: {}, Append: {}", new Object[]{this.getPath(), this.commitPosition, this.fileSegmentInputStream.getContentLength(), expectPosition, this.appendPosition});
            return false;
        }
        if (this.correctPosition(fileSize)) {
            this.fileSegmentInputStream = null;
            return true;
        }
        this.fileSegmentInputStream.rewind();
        return false;
    }

    private void releaseCommitLock() {
        if (this.commitLock.availablePermits() == 0) {
            this.commitLock.release();
        }
    }

    private boolean correctPosition(long fileSize) {
        long expectPosition = this.commitPosition + (long)this.fileSegmentInputStream.getContentLength();
        this.commitPosition = fileSize;
        return expectPosition == fileSize;
    }

    public ByteBuffer read(long position, int length) {
        return this.readAsync(position, length).join();
    }

    public CompletableFuture<ByteBuffer> readAsync(long position, int length) {
        CompletableFuture<ByteBuffer> future = new CompletableFuture<ByteBuffer>();
        if (position < 0L || position >= this.commitPosition) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, String.format("FileSegment read position illegal, filePath=%s, fileType=%s, position=%d, length=%d, commit=%d", new Object[]{this.filePath, this.fileType, position, length, this.commitPosition})));
            return future;
        }
        if (length <= 0) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, String.format("FileSegment read length illegal, filePath=%s, fileType=%s, position=%d, length=%d, commit=%d", new Object[]{this.filePath, this.fileType, position, length, this.commitPosition})));
            return future;
        }
        int readableBytes = (int)(this.commitPosition - position);
        if (readableBytes < length) {
            length = readableBytes;
            log.debug("FileSegment expect request position is greater than commit position, file: {}, request position: {}, commit position: {}, change length from {} to {}", new Object[]{this.getPath(), position, this.commitPosition, length, readableBytes});
        }
        return this.read0(position, length);
    }
}

