/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.api.objs;

import javax.annotation.Nullable;

import com.google.common.annotations.Beta;

/** A record of a highlight e.g. attached to an {@link EntityAdjunct}.
 * <p>
 *  Time of 0 or negative indicates the highlight is ongoing. Task ID is optional. */
@Beta
public class HighlightTuple {

    private String description;
    private long time;
    private String taskId;

    @SuppressWarnings("unused") //required for JSON de-serialisation
    private HighlightTuple(){}

    public HighlightTuple(String description, long time, @Nullable String taskId) {
        this.description = description;
        this.time = time;
        this.taskId = taskId;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public long getTime() {
        return time;
    }

    public void setTime(long time) {
        this.time = time;
    }

    @Nullable public String getTaskId() {
        return taskId;
    }

    public void setTaskId(String taskId) {
        this.taskId = taskId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        HighlightTuple that = (HighlightTuple) o;

        if (time != that.time) return false;
        if (description != null ? !description.equals(that.description) : that.description != null) return false;
        return taskId != null ? taskId.equals(that.taskId) : that.taskId == null;
    }

    @Override
    public int hashCode() {
        int result = description != null ? description.hashCode() : 0;
        result = 31 * result + (int) (time ^ (time >>> 32));
        result = 31 * result + (taskId != null ? taskId.hashCode() : 0);
        return result;
    }
    
    @Override
    public String toString() {
        return "highlight["+description+(time<=0 ? "" : "; time="+time)+(taskId==null ? "" : "; task="+taskId)+"]";
    }
}
