/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.core.mgmt.entitlement;

import java.util.Map;
import java.util.Set;

import org.apache.brooklyn.api.mgmt.entitlement.EntitlementClass;
import org.apache.brooklyn.api.mgmt.entitlement.EntitlementContext;
import org.apache.brooklyn.api.mgmt.entitlement.EntitlementManager;
import org.apache.brooklyn.config.ConfigKey;
import org.apache.brooklyn.core.config.ConfigKeys;
import org.apache.brooklyn.core.config.ConfigPredicates;
import org.apache.brooklyn.core.internal.BrooklynProperties;
import org.apache.brooklyn.util.collections.MutableMap;
import org.apache.brooklyn.util.text.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Preconditions;

public class PerUserEntitlementManager implements EntitlementManager {

    private static final Logger log = LoggerFactory.getLogger(PerUserEntitlementManager.class);
    
    public final static String PER_USER_ENTITLEMENTS_CONFIG_PREFIX = Entitlements.ENTITLEMENTS_CONFIG_PREFIX+".perUser";
    
    public final static ConfigKey<String> DEFAULT_MANAGER = ConfigKeys.newStringConfigKey(PER_USER_ENTITLEMENTS_CONFIG_PREFIX+
        ".default", "Default entitlements manager for users without further specification", "minimal");
    
    protected final EntitlementManager defaultManager;
    protected final Map<String,EntitlementManager> perUserManagers = MutableMap.of();

    private final static ThreadLocal<Boolean> ACTIVE = new ThreadLocal<Boolean>();
    
    private static EntitlementManager load(BrooklynProperties properties, String type) {
        if (Boolean.TRUE.equals(ACTIVE.get())) {
            // prevent infinite loop
            throw new IllegalStateException("Cannot set "+PerUserEntitlementManager.class.getName()+" within config for itself");
        }
        try {
            ACTIVE.set(true);
            return Entitlements.load(null, properties, type);
        } finally {
            ACTIVE.remove();
        }
    }
    
    public PerUserEntitlementManager(BrooklynProperties properties) {
        this(load(properties, properties.getConfig(DEFAULT_MANAGER)));
        
        Set<ConfigKey<?>> users = properties.findKeysPresent(ConfigPredicates.nameStartsWith(PER_USER_ENTITLEMENTS_CONFIG_PREFIX+"."));
        for (ConfigKey<?> key: users) {
            if (key.getName().equals(DEFAULT_MANAGER.getName())) continue;
            String user = Strings.removeFromStart(key.getName(), PER_USER_ENTITLEMENTS_CONFIG_PREFIX+".");
            addUser(user, load(properties, Strings.toString(properties.getConfig(key))));
        }
        
        log.info(getClass().getSimpleName()+" created with "+perUserManagers.size()+" user"+Strings.s(perUserManagers)+" and "
            + "default "+defaultManager+" (users: "+perUserManagers+")");
    }
    
    public PerUserEntitlementManager(EntitlementManager defaultManager) {
        this.defaultManager = Preconditions.checkNotNull(defaultManager);
    }

    public void addUser(String user, EntitlementManager managerForThisUser) {
        perUserManagers.put(Preconditions.checkNotNull(user, "user"), Preconditions.checkNotNull(managerForThisUser, "managerForThisUser"));
    }

    @Override
    public <T> boolean isEntitled(EntitlementContext context, EntitlementClass<T> entitlementClass, T entitlementClassArgument) {
        EntitlementManager entitlementInEffect;
        if (context==null || context.user()==null) {
            // no user means it is running as an internal process, always has root
            entitlementInEffect = Entitlements.root(); 
        } else {
            entitlementInEffect = perUserManagers.get(context.user());
            if (entitlementInEffect==null) entitlementInEffect = defaultManager;
        }
        return entitlementInEffect.isEntitled(context, entitlementClass, entitlementClassArgument);
    }
    
}
