#---------------------------------------------------------------------------
#
#   libtunepimp -- The MusicBrainz tagging library.  
#                  Let a thousand taggers bloom!
#   
#   Copyright (C) Robert Kaye 2003
#   
#   This file is part of libtunepimp.
#
#   libtunepimp is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   libtunepimp is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with libtunepimp; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   $Id: tunepimp.py,v 1.24 2005/07/01 07:25:03 robert Exp $
#
#---------------------------------------------------------------------------
import os, types, sys
from ctypes import *
import track

# TODO
#   Unhandled case - handle result_t casting cases   

# TPCallBackEnum values used for GetNotification
eFileAdded = 0;
eFileChanged = 1;
eFileRemoved = 2;
eWriteTagsComplete = 3;

# TPError returned by WriteTags and GetResults
eOk                     = 0
eTooManyTRMs            = 1
eNoUserInfo             = 2
eLookupError            = 3
eSubmitError            = 4
eInvalidIndex           = 5
eInvalidObject          = 6

# TPFileStatus
eMetadataRead           = 0    # pending metadata read
ePending                = 1    # pending trm calculation
eUnrecognized           = 2    # unrecognized
eRecognized             = 3    # Recognized and previously saved
eTRMLookup              = 4    # trm done, pending trm lookup
eTRMCollision           = 5    # trm done, pending trm lookup
eFileLookup             = 6    # trm done, no matches, pending file lookup
eUserSelection          = 7    # file lookup done, needs user selection
eVerified               = 8    # User verified, about to write changes to disk 
eSaved                  = 9    # File was saved
eDeleted                = 10   # to be deleted, waiting for refcount == 0
eError                  = 11   # Error

# TPResultType
eNone                   = 0
eArtistList             = 1
eAlbumList              = 2
eTrackList              = 3
eMatchedTrack           = 4

# TPAlbumType
eAlbumType_Album        = 0
eAlbumType_Single       = 1
eAlbumType_EP           = 2
eAlbumType_Compilation  = 3
eAlbumType_Soundtrack   = 4
eAlbumType_Spokenword   = 5
eAlbumType_Interview    = 6
eAlbumType_Audiobook    = 7
eAlbumType_Live         = 8
eAlbumType_Remix        = 9
eAlbumType_Other        = 10;
eAlbumType_Error        = 11;

# TPAlbumStatus
eAlbumStatus_Official   = 0
eAlbumStatus_Promotion  = 1
eAlbumStatus_Bootleg    = 2

# TPID3Encoding
eLatin1                 = 0
eUTF8                   = 1
eUTF16                  = 2
eEncodingError          = 3

# TPThreadPriorityEnum
eIdle                   = 0
eLowest                 = 1
eLow                    = 2
eNormal                 = 3
eHigh                   = 4
eHigher                 = 5
eTimeCritical           = 6

# Thread identifiers
tpThreadNone            = 0x0000
tpThreadLookupTRM       = 0x0001
tpThreadLookupFile      = 0x0002
tpThreadWrite           = 0x0004
tpThreadRead            = 0x0008
tpThreadAnalyzer        = 0x0010
tpThreadAll             = 0xFFFF

__version__ = "0.4.x"

# TODO: Hook up proper version number fetching from the makefiles

class TunePimpError(Exception):
    pass
Error = TunePimpError

def toUTF8(text):
    '''Thus convenience function converts common types to strings with UTF-8 encoidng'''

    if text.__class__.__name__ == 'int' or text.__class__.__name__ == 'float':
        return str(text).decode("utf-8", 'replace')

    if text.__class__.__name__ == 'unicode' or text.__class__.__name__ == 'str':
        return text.encode("utf-8", 'replace')

    assert 0, "Don't know how to convert " + text.__class__.__name__ + " to UTF-8"

def findLibrary():
    
    if sys.platform == "darwin":
        lib = "libtunepimp.3.dylib"
    else:
        if sys.platform == "linux2":
           lib = "libtunepimp.so.3"
        else:
           raise TunePimpError, "Unknown platform: " + sys.platform 

    pathList = [ ]
    ldLibPath = os.environ.get('LD_LIBRARY_PATH')
    if ldLibPath:
        pathList.extend(ldLibPath.split(':'))

    pathList.extend([ '/usr/lib', '/usr/local/lib' ])
    for path in pathList: 
        newPath = os.path.join(path, lib)
        if os.access(newPath, os.F_OK):
           return newPath

    raise TunePimpError, "Cannot find TunePimp shared library: " + lib

tplib = None
if sys.platform == "win32":
    tplib = cdll.LoadLibrary("libtunepimp.dll")
    tplib.tp_WSAInit.argtypes = [c_int]
    tplib.tp_WSAStop.argtypes = [c_int]
else:
    tplib = cdll.LoadLibrary(findLibrary())

class tunepimp(object):
    '''This is the main tunepimp class. For details on how to use this class, 
       please look up the main tunepimp documentation'''

    BUFLEN = 255

    def __init__(self, appName, appVersion, threads=tpThreadAll, pluginDir = None):
        self.tplib = tplib
        tplib.tp_NewWithArgs.argtypes = [c_char_p, c_char_p, c_int, c_char_p]
        self.tp = tplib.tp_NewWithArgs(toUTF8(appName), toUTF8(appVersion), threads, pluginDir)
        # need to hold ref for __del__ to work
        
        if sys.platform == "win32":
            tplib.tp_WSAInit(self.tp)

    tplib.tp_Delete.argtypes = [c_int]
    def __del__(self):
        if sys.platform == "win32":
            self.tplib.tp_WSAStop(self.tp)
            
        self.tplib.tp_Delete(self.tp)
        self.tplib = None

    def GetVersion(self):
        major = c_int()
        minor = c_int()
        rev = c_int()
        tplib.tp_GetVersion(self.tp, byref(major), byref(minor), byref(rev))
        return (major.value, minor.value, rev.value,)

    tplib.tp_SetUserInfo.argtypes = [c_int, c_char_p, c_char_p]
    def setUserInfo(self, user, passwd):
        tplib.tp_SetUserInfo(self.tp, toUTF8(user), toUTF8(passwd))

    def getUserInfo(self):
        user = c_buffer(self.BUFLEN)
        userLen = c_int(self.BUFLEN)
        passwd = c_buffer(self.BUFLEN)
        passwdLen = c_int(self.BUFLEN)
        tplib.tp_GetUserInfo(self.tp, user, userLen, passwd, passwdLen)
        return (unicode(user.value, "UTF-8", 'replace'), unicode(passwd.value, "UTF-8", 'replace'))

    tplib.tp_SetServer.argtypes = [c_int, c_char_p, c_int]
    def setServer(self, server, port):
        tplib.tp_SetServer(self.tp, toUTF8(server), port)

    def getServer(self):
        server = c_buffer(self.BUFLEN)
        port = c_short()
        len = c_int(self.BUFLEN)
        tplib.tp_GetServer(self.tp, server, len, byref(port))
        return (unicode(server.value, "UTF-8", 'replace'), port.value)

    tplib.tp_SetProxy.argtypes = [c_int, c_char_p, c_short]
    def setProxy(self, server, port):
        tplib.tp_SetProxy(self.tp, toUTF8(server), port)

    def getProxy(self):
        proxy = c_buffer(self.BUFLEN)
        port = c_short()
        len = c_int(self.BUFLEN)
        tplib.tp_GetProxy(self.tp, proxy, len, byref(port))
        return (unicode(proxy.value, "UTF-8", 'replace'), port.value) 

    tplib.tp_GetNumSupportedExtensions.argtypes = [c_int]
    def getNumSupportedExtensions(self):
        return tplib.tp_GetNumSupportedExtensions(self.tp)

    TP_EXTENSION_LEN = 32
    def getSupportedExtensions(self):
        num = self.getNumSupportedExtensions()
        extClass = c_char * self.TP_EXTENSION_LEN
        arrayClass = extClass * num
        extension = arrayClass()
        tplib.tp_GetSupportedExtensions(self.tp, extension)
        ret = []
        for ext in extension:
           ret.append(unicode(ext.value, "UTF-8", 'replace'))

        return ret

    tplib.tp_SetAnalyzerPriority.argtypes = [c_int, c_int]
    def setAnalyzerPriority(self, priority):
        tplib.tp_SetAnalyzerPriority(self.tp, priority)

    tplib.tp_GetAnalyzerPriority.argtypes = [c_int]
    def getAnalyzerPriority(self):
        return tplib.tp_GetAnalyzerPriority(self.tp)

    tplib.tp_SetAutoFileLookup.argtypes = [c_int, c_int]
    def setAutoFileLookup(self, autoLookup):
        tplib.tp_SetAutoFileLookup(self.tp, autoLookup)

    tplib.tp_GetAutoFileLookup.argtypes = [c_int]
    def getAutoFileLookup(self):
        return tplib.tp_GetAutoFileLookup(self.tp)

    def getNotification(self):
        type = c_int()
        fileId = c_int()
        status = c_int()
        ret = tplib.tp_GetNotification(self.tp, byref(type), byref(fileId), byref(status))
        return (ret, type.value, fileId.value, status.value)

    def getStatus(self):
        buf = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        ret = tplib.tp_GetStatus(self.tp, buf, len)
        return (ret, unicode(buf.value, "UTF-8", 'replace'))

    def getError(self):
        buf = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        tplib.tp_GetError(self.tp, buf, len)
        return unicode(buf.value, "UTF-8", 'replace')

    tplib.tp_SetDebug.argtypes = [c_int, c_int]
    def setDebug(self, debug):
        tplib.tp_SetDebug(self.tp, debug)

    tplib.tp_GetDebug.argtypes = [c_int]
    def getDebug(self):
        return tplib.tp_GetDebug(self.tp)

    tplib.tp_AddFile.argtypes = [c_int, c_char_p, c_int]
    def addFile(self, file, readMetadataNow=0):
        return tplib.tp_AddFile(self.tp, toUTF8(file), readMetadataNow)

    tplib.tp_AddDir.argtypes = [c_int, c_char_p]
    def addDir(self, dir):
        return tplib.tp_AddDir(self.tp, toUTF8(dir))

    tplib.tp_Remove.argtypes = [c_int, c_int]
    def remove(self, file):
        tplib.tp_Remove(self.tp, file)

    tplib.tp_GetNumFiles.argtypes = [c_int]
    def getNumFiles(self):
        return tplib.tp_GetNumFiles(self.tp)

    tplib.tp_GetNumUnsubmitted.argtypes = [c_int]
    def getNumUnsubmitted(self):
        return tplib.tp_GetNumUnsubmitted(self.tp)

    tplib.tp_GetNumUnsavedItems.argtypes = [c_int]
    def getNumUnsavedItems(self):
        return tplib.tp_GetNumUnsavedItems(self.tp)

    maxCounts = 16
    def getTrackCounts(self):
        arrayClass = c_int * self.maxCounts
        counts = arrayClass()
        tplib.tp_GetTrackCounts(self.tp, counts, self.maxCounts)
        ret = []
        for count in counts:
           ret.append(count)
        return ret

    def getFileIds(self):
        num = self.getNumFiles()
        arrayClass = c_int * num
        ids = arrayClass()
        tplib.tp_GetFileIds(self.tp, ids, num)
        ret = []
        for id in ids:
           ret.append(id)
        return ret

    tplib.tp_GetTrack.argtypes = [c_int, c_int]
    def getTrack(self, id):
        tr = track.track(self, tplib.tp_GetTrack(self.tp, id))
        return tr

    tplib.tp_ReleaseTrack.argtypes = [c_int, c_int]
    def releaseTrack(self, tr):
        tplib.tp_ReleaseTrack(self.tp, tr.getTrackObject())

    tplib.tp_Wake.argtypes = [c_int, c_int]
    def wake(self, tr):
        tplib.tp_Wake(self.tp, tr.getTrackObject())

    tplib.tp_SelectResult.argtypes = [c_int, c_int, c_int]
    def selectResult(self, tr, index):
        return tplib.tp_SelectResult(self.tp, tr, index)

    tplib.tp_Misidentified.argtypes = [c_int, c_int]
    def misidentified(self, fileId):
        tplib.tp_Misidentified(self.tp, fileId)

    tplib.tp_IdentifyAgain.argtypes = [c_int, c_int]
    def identifyAgain(self, fileId):
        tplib.tp_IdentifyAgain(self.tp, fileId)

    def writeTags(self, fileIds):
        num = len(fileIds)
        arrayClass = c_int * num
        ids = arrayClass()
        for i in xrange(num):
            ids[i] = fileIds[i] 

        return tplib.tp_WriteTags(self.tp, ids, num)

    tplib.tp_AddTRMSubmission.argtypes = [c_int, c_char_p, c_char_p]
    def addTRMSubmission(self, trackid, trmId):
        tplib.tp_AddTRMSubmission(self.tp, trackId, trmId)

    tplib.tp_SubmitTRMs.argtypes = [c_int]
    def submitTRMs(self, trackId, trmId):
        return tplib.tp_SubmitTRMs(self.tp)

    tplib.tp_SetRenameFiles.argtypes = [c_int, c_int]
    def setRenameFiles(self, rename):
        tplib.tp_SetRenameFiles(self.tp, rename)

    tplib.tp_GetRenameFiles.argtypes = [c_int]
    def getRenameFiles(self):
        return tplib.tp_GetRenameFiles(self.tp)

    tplib.tp_SetMoveFiles.argtypes = [c_int, c_int]
    def setMoveFiles(self, move):
        tplib.tp_SetMoveFiles(self.tp, move)

    tplib.tp_GetMoveFiles.argtypes = [c_int]
    def getMoveFiles(self):
        return tplib.tp_GetMoveFiles(self.tp)

    tplib.tp_SetFileNameEncoding.argtypes = [c_int, c_char_p]
    def setFileNameEncoding(self, encoding):
        tplib.tp_SetFileNameEncoding(self.tp, toUTF8(encoding))

    def getFileNameEncoding(self):
        encoding = c_buffer(self.BUFLEN)
        encodingLen = c_int(self.BUFLEN)
        tplib.tp_GetFileNameEncoding(self.tp, encoding, encodingLen)
        return unicode(encoding.value, "UTF-8", 'replace')

    tplib.tp_SetWriteID3v1.argtypes = [c_int, c_int]
    def setWriteID3v1(self, write):
        tplib.tp_SetWriteID3v1(self.tp, write)

    tplib.tp_GetWriteID3v1.argtypes = [c_int]
    def getWriteID3v1(self):
        return tplib.tp_GetWriteID3v1(self.tp)

    tplib.tp_SetWriteID3v2_3.argtypes = [c_int, c_int]
    def setWriteID3v2_3(self, write):
        tplib.tp_SetWriteID3v2_3(self.tp, write)

    tplib.tp_GetWriteID3v2_3.argtypes = [c_int]
    def getWriteID3v2_3(self):
        return tplib.tp_GetWriteID3v2_3(self.tp)

    tplib.tp_SetID3Encoding.argtypes = [c_int, c_int]
    def setID3Encoding(self, write):
        tplib.tp_SetID3Encoding(self.tp, write)

    tplib.tp_GetID3Encoding.argtypes = [c_int]
    def getID3Encoding(self):
        return tplib.tp_GetID3Encoding(self.tp)

    tplib.tp_SetClearTags.argtypes = [c_int, c_int]
    def setClearTags(self, clear):
        tplib.tp_SetClearTags(self.tp, clear)

    tplib.tp_GetClearTags.argtypes = [c_int]
    def getClearTags(self):
        return tplib.tp_GetClearTags(self.tp)

    tplib.tp_SetFileMask.argtypes = [c_int, c_char_p]
    def setFileMask(self, mask):
        tplib.tp_SetFileMask(self.tp, toUTF8(mask))

    def getFileMask(self):
        mask = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        tplib.tp_GetFileMask(self.tp, mask, len)
        return unicode(mask.value, "UTF-8", 'replace')

    tplib.tp_SetVariousFileMask.argtypes = [c_int, c_char_p]
    def setVariousFileMask(self, mask):
        tplib.tp_SetVariousFileMask(self.tp, toUTF8(mask))

    def getVariousFileMask(self):
        mask = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        tplib.tp_GetVariousFileMask(self.tp, mask, len)
        return unicode(mask.value, "UTF-8", 'replace')

    tplib.tp_SetNonAlbumFileMask.argtypes = [c_int, c_char_p]
    def setNonAlbumFileMask(self, mask):
        tplib.tp_SetNonAlbumFileMask(self.tp, toUTF8(mask))

    def getNonAlbumFileMask(self):
        mask = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        tplib.tp_GetNonAlbumFileMask(self.tp, mask, len)
        return unicode(mask.value, "UTF-8", 'replace')

    tplib.tp_SetAllowedFileCharacters.argtypes = [c_int, c_char_p]
    def setAllowedFileCharacters(self, allowedChars):
        tplib.tp_SetAllowedFileCharacters(self.tp, toUTF8(allowedChars))

    def getAllowedFileCharacters(self):
        chrs = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        tplib.tp_GetAllowedFileCharacters(self.tp, chrs, len)
        return unicode(chrs.value, "UTF-8", 'replace')

    tplib.tp_SetDestDir.argtypes = [c_int, c_char_p]
    def setDestDir(self, destDir):
        tplib.tp_SetDestDir(self.tp, toUTF8(destDir))

    def getDestDir(self):
        dir = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        tplib.tp_GetDestDir(self.tp, dir, len)
        return unicode(dir.value, "UTF-8", 'replace')

    tplib.tp_SetTopSrcDir.argtypes = [c_int, c_char_p]
    def setTopSrcDir(self, topSrcDir):
        tplib.tp_SetTopSrcDir(self.tp, toUTF8(topSrcDir))

    def getTopSrcDir(self):
        dir = c_buffer(self.BUFLEN)
        len = c_int(self.BUFLEN)
        tplib.tp_GetTopSrcDir(self.tp, dir, len)
        return unicode(dir.value, "UTF-8", 'replace')

    tplib.tp_SetTRMCollisionThreshold.argtypes = [c_int, c_int]
    def setTRMCollisionThreshold(self, thres):
        tplib.tp_SetTRMCollisionThreshold(self.tp, thres)

    tplib.tp_GetTRMCollisionThreshold.argtypes = [c_int]
    def getTRMCollisionThreshold(self):
        return tplib.tp_GetTRMCollisionThreshold(self.tp)

    tplib.tp_SetMinTRMThreshold.argtypes = [c_int, c_int]
    def setMinTRMThreshold(self, thres):
        tplib.tp_SetMinTRMThreshold(self.tp, thres)

    tplib.tp_GetMinTRMThreshold.argtypes = [c_int]
    def GetMinTRMThreshold(self):
        return tplib.tp_GetMinTRMThreshold(self.tp)

    tplib.tp_SetAutoSaveThreshold.argtypes = [c_int, c_int]
    def setAutoSaveThreshold(self, thres):
        tplib.tp_SetAutoSaveThreshold(self.tp, thres)

    tplib.tp_GetAutoSaveThreshold.argtypes = [c_int]
    def getAutoSaveThreshold(self):
        return tplib.tp_GetAutoSaveThreshold(self.tp)

    tplib.tp_SetMaxFileNameLen.argtypes = [c_int, c_int]
    def setMaxFileNameLen(self, len):
        tplib.tp_SetMaxFileNameLen(self.tp, len)

    tplib.tp_GetMaxFileNameLen.argtypes = [c_int]
    def getMaxFileNameLen(self):
        return tplib.tp_GetMaxFileNameLen(self.tp)

    tplib.tp_SetAutoRemovedSavedFiles.argtypes = [c_int, c_int]
    def setAutoRemovedSavedFiles(self, autoRemove):
        tplib.tp_SetAutoRemovedSavedFiles(self.tp, autoRemove)

    tplib.tp_GetAutoRemovedSavedFiles.argtypes = [c_int]
    def getAutoRemovedSavedFiles(self):
        return tplib.tp_GetAutoRemovedSavedFiles(self.tp)
