/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.
                  Let a thousand taggers bloom!

   Copyright (C) Robert Kaye 2003

   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: lookup.cpp,v 1.38 2005/02/21 01:10:28 robert Exp $

----------------------------------------------------------------------------*/
#ifdef WIN32
#	if _MSC_VER == 1200
#		pragma warning(disable:4786)
#	endif
#else
#	include <unistd.h>
#endif

#include <stdio.h>
#include "../config.h"
#include "lookup.h"
#include "lookuptools.h"
#include "tunepimp.h"
#include "submit.h"

const int iDataFieldLen = 255;

#define DB printf("%s:%d\n", __FILE__, __LINE__);

//---------------------------------------------------------------------------

LookupThread::LookupThread(TunePimp   *tunePimpArg,
                           FileCache  *cacheArg,
                           SubmitInfo *submitInfoArg) : Thread()
{
    autoFileLookup = true;
    tunePimp = tunePimpArg;
    cache = cacheArg;
    submitInfo = submitInfoArg;
    exitThread = false;
    sem = new Semaphore();
}

//---------------------------------------------------------------------------

LookupThread::~LookupThread(void)
{
    exitThread = true;
    sem->signal();
    join();
    delete sem;
}

//---------------------------------------------------------------------------

void LookupThread::setAutoFileLookup(bool enable)
{
    autoFileLookup = enable;
}

bool LookupThread::getAutoFileLookup(void)
{
    return autoFileLookup;
}

//---------------------------------------------------------------------------

void LookupThread::wake(void)
{
    sem->signal();
}

//---------------------------------------------------------------------------

void LookupThread::threadMain(void)
{
    string  fileName, status, trm, trackId;
    Track  *track;

    for(; !exitThread;)
    {
        track = cache->getNextItem(eTRMLookup);
        if (track == NULL)
        {
           sem->wait();
           continue;
        }

        track->lock();
        track->getFileName(fileName);
        track->unlock();

        if (submitInfo->find(trm, trackId))
        {
            Track *otherTrack;

            otherTrack = cache->getTrackFromTrackId(trackId);
            if (otherTrack)
            {
                Metadata data;

                otherTrack->getServerMetadata(data);
                cache->release(otherTrack);

                track->lock();
                track->setServerMetadata(data);
                track->setStatus(eRecognized);
                track->unlock();
            }
            else
                trackId = string("");
        }
        if (trackId.empty())
            lookup(track);

        tunePimp->wake(track);
        cache->release(track);
    }
}

//---------------------------------------------------------------------------

void LookupThread::lookup(Track *track)
{
    LookupTRM                  lookup(tunePimp);
    LookupStatus               status;
    vector<Metadata>           result;
    vector<Metadata>::iterator i, bestIndex;
    vector<int>                sims;
    vector<TPResult *>         collResults;
    Metadata                   id3;
    MetadataCompare            comp;
    string                     trm;
    int                        sim, bestSim = -1, j;

    track->lock();
    track->getLocalMetadata(id3);
    track->getTRM(trm);
    lookup.setArgs(trm, id3);

    track->unlock();
    status = lookup.lookup();
    track->lock();

    if (track->getStatus() != eTRMLookup)
    {
        track->unlock();
        return;
    }

    if (status == eLookupError)
    {
        string err;

        lookup.getError(err);
        tunePimp->setStatus(string("Lookup error: ") + string(err));
        track->setError(err);
        track->setStatus(eError);
        track->unlock();
        return;
    }

    if (status == eNotFound)
    {
        if (autoFileLookup)
            track->setStatus(eFileLookup);
        else
            track->setStatus(eUnrecognized);
        track->unlock();
        return;
    }

    // Check to see if there is more than one matching track, and if only
    // one track then we're set
    lookup.getResult(result);
    if (result.size() == 1)
    {
        sim = (int)comp.compare(id3, *(result.begin()));
        if (sim < tunePimp->context.getMinTRMThreshold())
            track->setStatus(eUnrecognized);
        else
        {
            if (lookup.isFuzzyMatch())
                submitInfo->add((*result.begin()).trackId, trm);

            track->setServerMetadata(*(result.begin()));
            if (sim >= tunePimp->context.getAutoSaveThreshold() &&
                tunePimp->context.getAutoSaveThreshold() >= 0)
                track->setStatus(eVerified);
            else
                track->setStatus(eRecognized);
        }
        track->unlock();
        return;
    }

    // Crap, more than one track matches. Loop through them and find the
    // best match and return that if it is higher then the collision threshold
    track->unlock();
    for(bestIndex = i = result.begin(); i != result.end(); i++)
    {
        sim = (int)comp.compare(id3, *i);
        if (sim > bestSim)
        {
            bestSim = sim;
            bestIndex = i;
        }
        sims.push_back(sim);
    }
    track->lock();

    if (bestSim >= tunePimp->context.getTRMCollisionThreshold())
    {
        track->setServerMetadata(*bestIndex);

        if (bestSim >= tunePimp->context.getAutoSaveThreshold() &&
            tunePimp->context.getAutoSaveThreshold() >= 0)
            track->setStatus(eVerified);
        else
            track->setStatus(eRecognized);
        track->unlock();
        return;
    }

    // Take the results and wrap them into the same result format you get from
    // a FileLookup query and then toss them back to the user to figure
    // out what to do.
    for(i = result.begin(), j = 0; i != result.end(); i++, j++)
    {
        TPArtistResult      artist;
        TPAlbumResult       album;
        TPAlbumTrackResult *albumTrack;

        artist.setId((*i).artistId);
        artist.setName((*i).artist);
        artist.setSortName((*i).sortName);

        album.setArtist(artist);
        album.setId((*i).albumId);
        album.setName((*i).album);
        album.setNumCDIndexIds(-1);
        album.setNumTracks(-1);
        album.setType((*i).albumType);
        album.setStatus((*i).albumStatus);
        album.setVariousArtists((*i).variousArtist);
        album.setNonAlbum((*i).nonAlbum);

        albumTrack = new TPAlbumTrackResult();
        albumTrack->setId((*i).trackId);
        albumTrack->setArtist(artist);
        albumTrack->setAlbum(album);
        albumTrack->setName((*i).track);
        albumTrack->setDuration((*i).duration);
        albumTrack->setTrackNum((*i).trackNum);
        albumTrack->setNumTRMIds((*i).numTRMIds);
        albumTrack->setRelevance(sims[j]);

        collResults.push_back(albumTrack);
    }

    // Nothing was a close match, so screw it and not guess, let the
    // user figure out the right track.
    track->setResults(eTrackList, collResults);
    track->setStatus(eTRMCollision);

    track->unlock();
    return;
}
