/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: fileio_win32.cpp,v 1.2 2005/07/02 05:56:41 robert Exp $

----------------------------------------------------------------------------*/

#include <assert.h>
#include <errno.h>
#include <string>
#include <windows.h>
#include <io.h>
#include <direct.h>
using namespace std;

#include "../config_win32.h"
#include "fileio.h"

static int uniqueId = 0;

#define DB printf("%s:%d\n", __FILE__, __LINE__);

#ifdef __cplusplus
extern "C"
{
#endif

static const char *dirSep = "\\";
static const char dirSepChar = '\\';

static int copyAndDelete(const char *from, const char *to, const char *encoding);
#define BUF_SIZE 4096

// For some reason MSVC++ can't find the prototype for this function
//int access(const char *, int);
#define F_OK 0

typedef struct _modeMapping
{
    char *code;
    int   flags;
    int   dwCreationDispositionIsALousyName;
} modeMapping;

const int numModeMappings = 6;
const modeMapping modeMappings[numModeMappings] =
{
    { "r",  GENERIC_READ,                 OPEN_EXISTING },
    { "r+", GENERIC_READ | GENERIC_WRITE, OPEN_EXISTING }, 
    { "w",  GENERIC_WRITE,                CREATE_ALWAYS },
    { "w+", GENERIC_READ | GENERIC_WRITE, CREATE_ALWAYS },
    { "a",  GENERIC_WRITE,                OPEN_ALWAYS   },
    { "a+", GENERIC_READ | GENERIC_WRITE, OPEN_ALWAYS   }
};

TFILE   *topen(const char *path, const char *modeArg, const char *encoding)
{
    HANDLE  fh;
    int     i;
    char   *ptr, mode[12];

	strcpy(mode, modeArg);
    ptr = strchr(mode, 'b');
    if (ptr)		
        memmove(ptr, ptr+1, strlen(ptr));

    string fileName = string("\\\\?\\") + string(path);
    LPWSTR wFileName = new WCHAR[fileName.size() + 1];
    MultiByteToWideChar(CP_UTF8, 0, fileName.c_str(), -1, wFileName, fileName.size() + 1);

    for(i = 0; i < numModeMappings; i++)
        if (strcmp(mode, modeMappings[i].code) == 0)
            break;
    assert(i != numModeMappings);

    fh = CreateFileW(wFileName, modeMappings[i].flags, 0, NULL, 
                     modeMappings[i].dwCreationDispositionIsALousyName, 0, NULL);
    delete [] wFileName;

    if (fh == INVALID_HANDLE_VALUE)
        return (TFILE*)NULL;

    if (strchr(mode, 'a'))
        SetFilePointer(fh, 0, NULL, FILE_END);

    return (TFILE *)fh;
}

size_t   tread(void *ptr, size_t size, size_t nmemb, TFILE *stream)
{
    if (nmemb == 0)
       return 0;

	DWORD numRead;
    ReadFile((HANDLE)stream, ptr, (DWORD)size * (DWORD)nmemb, &numRead, NULL);
    return numRead / (DWORD)size;
}

size_t   twrite(const void *ptr, size_t size, size_t nmemb, TFILE *stream)
{
    DWORD numRead;
    if (nmemb == 0)
       return 0;

    WriteFile((HANDLE)stream, ptr, (DWORD)size * (DWORD)nmemb, &numRead, NULL);
    return numRead / (DWORD)size;
}

int      tseek(TFILE *stream, long offset, int whence)
{
    DWORD newPos;
    newPos = SetFilePointer((HANDLE)stream, offset, NULL, whence);
    if (newPos == INVALID_SET_FILE_POINTER)
		return -1;
	return 0;
}

long     ttell(TFILE *stream)
{
    DWORD ret;
    ret = SetFilePointer((HANDLE)stream, 0, NULL, FILE_CURRENT);
    if (ret == INVALID_SET_FILE_POINTER)
		return -1;
	return ret;
}

int      tclose(TFILE *stream)
{
    return (int)CloseHandle((HANDLE)stream) - 1;
}

int      tflush(TFILE *stream)
{
    return (int)FlushFileBuffers((HANDLE)stream) - 1;
}

int      tunlink(const char *pathname, const char *encoding)
{
    int ret;
    string newFileName;
    LPWSTR wFileName;

    newFileName = string("\\\\?\\") + pathname;
    wFileName = new WCHAR[newFileName.size() + 1];
    MultiByteToWideChar(CP_UTF8, 0, newFileName.c_str(), -1, wFileName, newFileName.size() + 1);
    ret = (int)DeleteFileW(wFileName);
    delete [] wFileName;
	return ret - 1;
}

int trename(const char *oldpath, const char *newpath, const char *encoding)
{
    int ret;
    string newFileName, oldFileName;
    LPWSTR wNewFileName, wOldFileName;

    newFileName = string("\\\\?\\") + newpath;
    oldFileName = string("\\\\?\\") + oldpath;

    wNewFileName = new WCHAR[newFileName.size() + 1];
    wOldFileName = new WCHAR[oldFileName.size() + 1];
    MultiByteToWideChar(CP_UTF8, 0, newFileName.c_str(), -1, wNewFileName, newFileName.size() + 1);
    MultiByteToWideChar(CP_UTF8, 0, oldFileName.c_str(), -1, wOldFileName, oldFileName.size() + 1);
	ret = MoveFileW(wOldFileName, wNewFileName);
    delete [] wNewFileName;
    delete [] wOldFileName;

    if (!ret && GetLastError() == ERROR_FILE_EXISTS)
        errno = EEXIST;

    if (!ret && GetLastError() == ERROR_NOT_SAME_DEVICE)
    {
        string err;
        return copyAndDelete(oldpath, newpath, encoding);
    }

    return (int)ret - 1;
}

int tmkdir(const char *pathname, const char *encoding)
{
    string newFileName;
    LPWSTR wFileName;
    int ret;

    newFileName = string("\\\\?\\") + pathname;
    wFileName = new WCHAR[newFileName.size() + 1];
    MultiByteToWideChar(CP_UTF8, 0, newFileName.c_str(), -1, wFileName, newFileName.size() + 1);
    ret = (int)CreateDirectoryW(wFileName, NULL);
	if (ret == 0 && GetLastError() == ERROR_ALREADY_EXISTS)
		ret = 1;

    delete [] wFileName;
    return (int)ret - 1;
}

int trmdir(const char *pathname, const char *encoding)
{
    int ret;
    string newFileName;
    LPWSTR wFileName;

    newFileName = string("\\\\?\\") + pathname;
    wFileName = new WCHAR[newFileName.size() + 1];
    MultiByteToWideChar(CP_UTF8, 0, newFileName.c_str(), -1, wFileName, newFileName.size() + 1);
    ret = (int)RemoveDirectoryW(wFileName);
    delete [] wFileName;
    return (int)ret - 1;
}

int taccess(const char *pathname, int mode, const char *encoding)
{
    assert(mode == F_OK);
    string newFileName;
    LPWSTR wFileName;
    DWORD ret;

    newFileName = string("\\\\?\\") + pathname;
    wFileName = new WCHAR[newFileName.size() + 1];
    MultiByteToWideChar(CP_UTF8, 0, newFileName.c_str(), -1, wFileName, newFileName.size() + 1);
    ret = GetFileAttributesW(wFileName);
    delete [] wFileName;
    return (ret == INVALID_FILE_ATTRIBUTES) ? -1 : 0;
}

void tmktempname(const char *path, char *newPath, int newPathLen)
{
	char *ptr, *temp;

    temp = (char *)malloc(strlen(path) + 32);
    ptr = strrchr(path, dirSepChar);
    if (ptr)
    {
        int len = (int)(ptr - path);
        strncpy(temp, path, len);
        temp[len] = 0;
    }
    else
        strcpy(temp, ".");

    strcat(temp, dirSep);
    sprintf(temp + strlen(temp), "libtp%d%d.temp", (int)GetCurrentProcessId(), uniqueId++);
    strncpy(newPath, temp, newPathLen - 1);
    newPath[newPathLen - 1] = 0;
    free(temp);
}

//---------------------------------------------------------------------------
static int copyAndDelete(const char *from, const char *to, const char *encoding)
{
    TFILE *in, *out;
    int    ret = 0;

    errno = 0;
    in = topen(from, "rb", encoding);
    if (in == NULL)
        return -1;

    out = topen(to, "wb", encoding);
    if (out == NULL)
    {
        tclose(in);
        return -1;
    }

    char *buf = new char[BUF_SIZE];
    for(;;)
    {
        int numRead = tread(buf, sizeof(char), BUF_SIZE, in);
        if (numRead <= 0)
            break;

        int numWritten = twrite(buf, sizeof(char), numRead, out);
        if (numWritten != numRead)
        {
            //err = string("Could not write file to destination directory. Disk full?");
            ret = -1;
            break;
        }
    }
    tclose(in);
    tclose(out);
    delete [] buf;

    if (ret == 0)
    {
        ret = tunlink(from, encoding);
        if (ret < 0)
        {
            //err = string("Could remove old file: '") + from + string("'.");
            tunlink(to, encoding);
        }
    }

    return ret;
}

#ifdef __cplusplus
}
#endif
