/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   Copyright (C) Simon George 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: ape_trm.cpp,v 1.4 2003/10/24 19:10:48 robert Exp $

----------------------------------------------------------------------------*/
#include "../config.h"

#ifdef HAVE_LIBAPE
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#ifdef WIN32
#include <windows.h>
#endif
#pragma hdrstop

#include "tunepimp.h"
#include <musicbrainz/mb_c.h>
#include "ape_trm.h"

#define DLLEXPORT
#include <MACLib.h>		// APE

#define DB Debug_v("%s:%d", __FILE__, __LINE__);


#define min(a,b) ((a) < (b) ? (a) : (b))

#ifdef WIN32
    typedef __int64   mb_int64_t;
#else				            
    typedef long long mb_int64_t;
#endif

/*-------------------------------------------------------------------------*/

const int BUFFER_LEN = 4096;

/*-------------------------------------------------------------------------*/

TRMResult TRMGeneratorApe::generate(const string &fileName, 
                                    string &trmId, 
                                    unsigned long &durationArg)
{
    char		  *copyBuffer;
    unsigned long  waveSize = 0;
    mb_int64_t     fileLen = 0;
    trm_t          trm;
    char           trmidRaw[32];
    char           trmid[37];
    TRMResult      result = eOtherError;
    string         proxyServer;
    short          proxyPort;

	//---	Open APE file

	int				nRetVal = 0;
	IAPEDecompress*	pAPEDecompress = CreateIAPEDecompress(fileName.c_str(), &nRetVal);
	if (pAPEDecompress == NULL) 
	{
       errorString = strdup("File not found");
       return eFileNotFound;
	}

    fileLen = pAPEDecompress->GetInfo(APE_INFO_LENGTH_MS) / 1000;

    trm = trm_New ();

   tunepimp->getProxy(proxyServer, proxyPort);
	if (proxyServer.size() > 0 && proxyPort != 0)
		 trm_SetProxy(trm, (char*)proxyServer.c_str(), proxyPort);

    trm_SetPCMDataInfo(trm, pAPEDecompress->GetInfo(APE_INFO_SAMPLE_RATE),
                            pAPEDecompress->GetInfo(APE_INFO_CHANNELS),
                            pAPEDecompress->GetInfo(APE_INFO_BITS_PER_SAMPLE) );

	if ( durationArg == 0 )
    {
       trm_SetSongLength(trm, pAPEDecompress->GetInfo(APE_INFO_LENGTH_MS) / 1000);
       fprintf(stderr, "Duration: %d\n", fileLen);
    }
    else
    {
       //fprintf(stderr, "Duration: %d\n", *durationArg / 1000);
       trm_SetSongLength(trm, durationArg / 1000);
    }

    copyBuffer = (char*)malloc( pAPEDecompress->GetInfo(APE_INFO_BLOCK_ALIGN) );
    if (copyBuffer == NULL)
    {
       errorString = strdup("Cannot allocate buffer space.");
	   delete pAPEDecompress;
       return eDecodeError;
    }


	int		total_blocks = pAPEDecompress->GetInfo(APE_DECOMPRESS_TOTAL_BLOCKS);

    for( int b=0; b < total_blocks; ++b )
    {
		int nBlocksRetrieved;

		nRetVal = pAPEDecompress->GetData(copyBuffer, 1, &nBlocksRetrieved);

		if ( (nRetVal != ERROR_SUCCESS) || (nBlocksRetrieved == 0) )
		{
			errorString = strdup("Cannot read from file.");
			delete pAPEDecompress;
			return eDecodeError;
		}

        if (trm_GenerateSignature(trm, (char *)copyBuffer, pAPEDecompress->GetInfo(APE_INFO_BLOCK_ALIGN)))
        {
           break;
        }
    }
    free(copyBuffer);

    trmId = string("");

    musicbrainz_t mb = mb_New();
    mb_WSAInit(mb);

    int ret = trm_FinalizeSignature(trm, (char *)trmidRaw, NULL);
    result = (ret == 0) ? eOk : eCannotConnect;

    trm_ConvertSigToASCII(trm, (char *)trmidRaw, (char *)trmid);
    trm = NULL;
    trm_Delete(trm);

    trmId = string((char *)trmid);

	delete pAPEDecompress;

    mb_WSAStop(mb);
    mb_Delete(mb);

    return result;
}
#endif

