/***************************************************************************
                         qgsprocessingmodelresult.h
                         ----------------------
    begin                : April 2024
    copyright            : (C) 2024 by Nyall Dawson
    email                : nyall dot dawson at gmail dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef QGSPROCESSINGMODELRESULT_H
#define QGSPROCESSINGMODELRESULT_H

#include "qgis_core.h"
#include "qgis.h"
#include <QSet>

/**
 * \ingroup core
 * \brief Encapsulates the results of running a child algorithm within a model.
 *
 * \since QGIS 3.38
*/
class CORE_EXPORT QgsProcessingModelChildAlgorithmResult
{
  public:

    QgsProcessingModelChildAlgorithmResult();

    /**
     * Returns the status of executing the child algorithm.
     *
     * \see setExecutionStatus()
     */
    Qgis::ProcessingModelChildAlgorithmExecutionStatus executionStatus() const { return mExecutionStatus; }

    /**
     * Sets the \a status of executing the child algorithm.
     *
     * \see executionStatus()
     */
    void setExecutionStatus( Qgis::ProcessingModelChildAlgorithmExecutionStatus status ) { mExecutionStatus = status; }

    /**
     * Returns the inputs used for the child algorithm.
     *
     * \see setInputs()
     */
    QVariantMap inputs() const { return mInputs; }

    /**
     * Sets the \a inputs used for the child algorithm.
     *
     * \see inputs()
     */
    void setInputs( const QVariantMap &inputs ) { mInputs = inputs; }

    /**
    * Returns the outputs generated by the child algorithm.
    *
    * \see setOutputs()
    */
    QVariantMap outputs() const { return mOutputs; }

    /**
     * Sets the \a outputs generated by child algorithm.
     *
     * \see outputs()
     */
    void setOutputs( const QVariantMap &outputs ) { mOutputs = outputs; }

    /**
     * Returns the HTML formatted contents of logged messages which occurred while running the child.
     *
     * \see setHtmlLog()
     */
    QString htmlLog() const { return mHtmlLog; }

    /**
     * Sets the HTML formatted contents of logged messages which occurred while running the child.
     *
     * \see htmlLog()
     */
    void setHtmlLog( const QString &log ) { mHtmlLog = log; }

    bool operator==( const QgsProcessingModelChildAlgorithmResult &other ) const
    {
      return mExecutionStatus == other.mExecutionStatus
             && mHtmlLog == other.mHtmlLog
             && mInputs == other.mInputs
             && mOutputs == other.mOutputs;
    }
    bool operator!=( const QgsProcessingModelChildAlgorithmResult &other ) const
    {
      return !( *this == other );
    }

  private:

    Qgis::ProcessingModelChildAlgorithmExecutionStatus mExecutionStatus = Qgis::ProcessingModelChildAlgorithmExecutionStatus::NotExecuted;
    QVariantMap mInputs;
    QVariantMap mOutputs;
    QString mHtmlLog;

};

/**
 * \ingroup core
 * \brief Encapsulates the results of running a Processing model.
 *
 * \since QGIS 3.38
*/
class CORE_EXPORT QgsProcessingModelResult
{
  public:

    QgsProcessingModelResult();

    /**
     * Clears any existing results.
     */
    void clear();

    /**
     * Merges this set of results with an \a other set of results.
     *
     * Conflicting results from \a other will replace results in this object.
     */
    void mergeWith( const QgsProcessingModelResult &other );

    /**
     * Returns the map of child algorithm results.
     *
     * Map keys refer to the child algorithm IDs.
     */
    QMap< QString, QgsProcessingModelChildAlgorithmResult > childResults() const { return mChildResults; }

    /**
     * Returns a reference to the map of child algorithm results.
     *
     * Map keys refer to the child algorithm IDs.
     *
     * \note Not available in Python bindings
     */
    QMap< QString, QgsProcessingModelChildAlgorithmResult > &childResults() SIP_SKIP { return mChildResults; }

    /**
     * Returns a reference to the map of raw child algorithm inputs.
     *
     * Map keys refer to the child algorithm IDs. Map values may take any form, including
     * values which are not safe to access from Python.
     *
     * \note Not available in Python bindings
     */
    QVariantMap &rawChildInputs() SIP_SKIP { return mRawChildInputs; }

    /**
     * Returns a reference to the map of raw child algorithm outputs.
     *
     * Map keys refer to the child algorithm IDs. Map values may take any form, including
     * values which are not safe to access from Python.
     *
     * \note Not available in Python bindings
     */
    QVariantMap &rawChildOutputs() SIP_SKIP { return mRawChildOutputs; }

    /**
     * Returns a reference to the set of child algorithm IDs which were executed
     * during the model execution.
     *
     * \note Not available in Python bindings
     */
    QSet< QString > &executedChildIds() SIP_SKIP { return mExecutedChildren; }

    /**
     * Returns the set of child algorithm IDs which were executed during the model execution.
     */
    QSet< QString > executedChildIds() const { return mExecutedChildren; }

  private:

    QMap< QString, QgsProcessingModelChildAlgorithmResult > mChildResults;

    QSet< QString > mExecutedChildren;
    QVariantMap mRawChildInputs;
    QVariantMap mRawChildOutputs;

};


#endif // QGSPROCESSINGMODELRESULT_H




