/*
 * Decompiled with CFR 0.152.
 */
package com.dd.plist;

import com.dd.plist.BinaryLocationInformation;
import com.dd.plist.NSArray;
import com.dd.plist.NSData;
import com.dd.plist.NSDate;
import com.dd.plist.NSDictionary;
import com.dd.plist.NSNumber;
import com.dd.plist.NSObject;
import com.dd.plist.NSSet;
import com.dd.plist.NSString;
import com.dd.plist.ParsedObjectStack;
import com.dd.plist.PropertyListFormatException;
import com.dd.plist.PropertyListParser;
import com.dd.plist.UID;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Objects;
import java.util.function.BiFunction;

public final class BinaryPropertyListParser {
    private static final int SIMPLE_TYPE = 0;
    private static final int INT_TYPE = 1;
    private static final int REAL_TYPE = 2;
    private static final int DATE_TYPE = 3;
    private static final int DATA_TYPE = 4;
    private static final int ASCII_STRING_TYPE = 5;
    private static final int UTF16_STRING_TYPE = 6;
    private static final int UTF8_STRING_TYPE = 7;
    private static final int UID_TYPE = 8;
    private static final int ARRAY_TYPE = 10;
    private static final int ORDERED_SET_TYPE = 11;
    private static final int SET_TYPE = 12;
    private static final int DICTIONARY_TYPE = 13;
    private int majorVersion;
    private int minorVersion;
    private byte[] bytes;
    private int objectRefSize;
    private int offsetSize;
    private int numObjects;
    private int offsetTableOffset;
    private final HashMap<Integer, NSObject> parsedObjects = new HashMap();

    private BinaryPropertyListParser() {
    }

    public static NSObject parse(File f) throws IOException, PropertyListFormatException {
        return BinaryPropertyListParser.parse(f.toPath());
    }

    public static NSObject parse(Path path) throws IOException, PropertyListFormatException {
        try (InputStream fileInputStream = Files.newInputStream(path, new OpenOption[0]);){
            NSObject nSObject = BinaryPropertyListParser.parse(fileInputStream);
            return nSObject;
        }
    }

    public static NSObject parse(InputStream is) throws IOException, PropertyListFormatException {
        return BinaryPropertyListParser.parse(PropertyListParser.readAll(is));
    }

    public static NSObject parse(byte[] data) throws PropertyListFormatException, UnsupportedEncodingException {
        BinaryPropertyListParser parser = new BinaryPropertyListParser();
        return parser.doParse(data);
    }

    public static long parseUnsignedInt(byte[] bytes) {
        return BinaryPropertyListParser.parseUnsignedInt(bytes, 0, bytes.length);
    }

    public static long parseUnsignedInt(byte[] bytes, int startIndex, int endIndex) {
        long l = 0L;
        for (int i = startIndex; i < endIndex; ++i) {
            l <<= 8;
            l |= (long)(bytes[i] & 0xFF);
        }
        return l &= 0xFFFFFFFFL;
    }

    public static long parseLong(byte[] bytes) {
        return BinaryPropertyListParser.parseLong(bytes, 0, bytes.length);
    }

    public static long parseLong(byte[] bytes, int startIndex, int endIndex) {
        long l = 0L;
        for (int i = startIndex; i < endIndex; ++i) {
            l <<= 8;
            l |= (long)(bytes[i] & 0xFF);
        }
        return l;
    }

    public static double parseDouble(byte[] bytes) {
        return BinaryPropertyListParser.parseDouble(bytes, 0, bytes.length);
    }

    public static double parseDouble(byte[] bytes, int startIndex, int endIndex) {
        if (endIndex - startIndex == 8) {
            return Double.longBitsToDouble(BinaryPropertyListParser.parseLong(bytes, startIndex, endIndex));
        }
        if (endIndex - startIndex == 4) {
            return Float.intBitsToFloat((int)BinaryPropertyListParser.parseLong(bytes, startIndex, endIndex));
        }
        throw new IllegalArgumentException("endIndex (" + endIndex + ") - startIndex (" + startIndex + ") != 4 or 8");
    }

    public static byte[] copyOfRange(byte[] src, int startIndex, int endIndex) {
        int length = endIndex - startIndex;
        if (length < 0) {
            throw new IllegalArgumentException("startIndex (" + startIndex + ") > endIndex (" + endIndex + ")");
        }
        byte[] dest = new byte[length];
        System.arraycopy(src, startIndex, dest, 0, length);
        return dest;
    }

    private NSObject doParse(byte[] data) throws PropertyListFormatException {
        Objects.requireNonNull(data);
        if (data.length < 8) {
            throw new PropertyListFormatException("The available binary property list data is too short.");
        }
        this.bytes = data;
        String magic = new String(BinaryPropertyListParser.copyOfRange(this.bytes, 0, 8), StandardCharsets.US_ASCII);
        if (!(magic.startsWith("bplist") && magic.length() >= 8 && Character.isDigit(magic.charAt(6)) && Character.isDigit(magic.charAt(7)))) {
            throw new PropertyListFormatException("The binary property list has an invalid file header: " + magic);
        }
        this.majorVersion = magic.charAt(6) - 48;
        this.minorVersion = magic.charAt(7) - 48;
        if (this.majorVersion > 0) {
            throw new PropertyListFormatException("Unsupported binary property list format: v" + this.majorVersion + "." + this.minorVersion + ". Version 1.0 and later are not yet supported.");
        }
        if (this.bytes.length < 40) {
            throw new PropertyListFormatException("The binary property list does not contain a complete object offset table.");
        }
        byte[] trailer = BinaryPropertyListParser.copyOfRange(this.bytes, this.bytes.length - 32, this.bytes.length);
        this.offsetSize = (int)BinaryPropertyListParser.parseUnsignedInt(trailer, 6, 7);
        this.objectRefSize = (int)BinaryPropertyListParser.parseUnsignedInt(trailer, 7, 8);
        this.numObjects = (int)BinaryPropertyListParser.parseUnsignedInt(trailer, 8, 16);
        int topObject = (int)BinaryPropertyListParser.parseUnsignedInt(trailer, 16, 24);
        this.offsetTableOffset = (int)BinaryPropertyListParser.parseUnsignedInt(trailer, 24, 32);
        if (this.offsetTableOffset + (this.numObjects + 1) * this.offsetSize > this.bytes.length || topObject >= this.bytes.length - 32) {
            throw new PropertyListFormatException("The binary property list contains a corrupted object offset table.");
        }
        return this.parseObject(ParsedObjectStack.empty(), topObject);
    }

    private NSObject parseObject(ParsedObjectStack stack, int obj) throws PropertyListFormatException {
        NSObject result;
        stack = stack.push(obj);
        if (this.parsedObjects.containsKey(obj)) {
            return this.parsedObjects.get(obj);
        }
        int offset = this.getObjectOffset(obj);
        BinaryLocationInformation loc = new BinaryLocationInformation(obj, offset);
        byte type = this.bytes[offset];
        int objType = (type & 0xF0) >> 4;
        int objInfo = type & 0xF;
        try {
            switch (objType) {
                case 0: {
                    result = this.parseSimpleObject(offset, objInfo, obj);
                    break;
                }
                case 1: {
                    result = this.parseNumber(offset, objInfo, 0);
                    break;
                }
                case 2: {
                    result = this.parseNumber(offset, objInfo, 1);
                    break;
                }
                case 3: {
                    result = this.parseDate(offset, objInfo);
                    break;
                }
                case 4: {
                    result = this.parseData(offset, objInfo);
                    break;
                }
                case 5: {
                    result = this.parseString(offset, objInfo, (o, l) -> l, StandardCharsets.US_ASCII.name());
                    break;
                }
                case 6: {
                    result = this.parseString(offset, objInfo, (o, l) -> 2 * l, StandardCharsets.UTF_16BE.name());
                    break;
                }
                case 7: {
                    result = this.parseString(offset, objInfo, this::calculateUtf8StringLength, StandardCharsets.UTF_8.name());
                    break;
                }
                case 8: {
                    result = this.parseUid(obj, offset, objInfo + 1);
                    break;
                }
                case 10: {
                    result = this.parseArray(offset, objInfo, stack);
                    break;
                }
                case 11: {
                    result = this.parseSet(offset, objInfo, true, stack);
                    break;
                }
                case 12: {
                    result = this.parseSet(offset, objInfo, false, stack);
                    break;
                }
                case 13: {
                    result = this.parseDictionary(offset, objInfo, stack);
                    break;
                }
                default: {
                    throw new PropertyListFormatException(this.buildTypeError(offset));
                }
            }
        }
        catch (PropertyListFormatException ex) {
            if (ex.getLocationInformation() == null) {
                ex.setLocationInformation(loc);
            }
            throw ex;
        }
        catch (UnsupportedEncodingException ex) {
            throw new PropertyListFormatException("The encoding of the NSString at offset " + offset + " is not supported.", loc, ex);
        }
        if (result != null) {
            result.setLocationInformation(loc);
        }
        this.parsedObjects.put(obj, result);
        return result;
    }

    private NSDate parseDate(int offset, int objInfo) throws PropertyListFormatException {
        if (objInfo != 3) {
            throw new PropertyListFormatException(this.buildTypeError(offset, "NSDate"));
        }
        if (offset + 9 > this.bytes.length) {
            throw new PropertyListFormatException(BinaryPropertyListParser.buildLengthError(offset, "NSDate"));
        }
        return new NSDate(this.bytes, offset + 1, offset + 9);
    }

    private NSData parseData(int offset, int objInfo) throws PropertyListFormatException {
        int length;
        int[] lengthAndOffset = this.readLengthAndOffset(objInfo, offset);
        int dataOffset = offset + lengthAndOffset[1];
        if (dataOffset + (length = lengthAndOffset[0]) > this.bytes.length) {
            throw new PropertyListFormatException(BinaryPropertyListParser.buildLengthError(offset, "NSData"));
        }
        return new NSData(BinaryPropertyListParser.copyOfRange(this.bytes, dataOffset, dataOffset + length));
    }

    private NSObject parseSimpleObject(int offset, int objInfo, int obj) throws PropertyListFormatException {
        switch (objInfo) {
            case 0: {
                return null;
            }
            case 8: {
                return new NSNumber(false);
            }
            case 9: {
                return new NSNumber(true);
            }
            case 12: 
            case 13: {
                throw new PropertyListFormatException("The NSObject at offset " + offset + " is a URL, which is not supported.");
            }
            case 14: {
                return this.parseUid(obj, offset, 16);
            }
        }
        throw new PropertyListFormatException(this.buildTypeError(offset));
    }

    private UID parseUid(int obj, int offset, int length) throws PropertyListFormatException {
        if (offset + 1 + length >= this.bytes.length) {
            throw new PropertyListFormatException(BinaryPropertyListParser.buildLengthError(offset, "UID"));
        }
        return new UID(String.valueOf(obj), BinaryPropertyListParser.copyOfRange(this.bytes, offset + 1, offset + 1 + length));
    }

    private NSNumber parseNumber(int offset, int objInfo, int integer) throws PropertyListFormatException {
        int length = (int)Math.pow(2.0, objInfo);
        try {
            return new NSNumber(this.bytes, offset + 1, offset + 1 + length, integer);
        }
        catch (IndexOutOfBoundsException ex) {
            throw new PropertyListFormatException(BinaryPropertyListParser.buildLengthError(offset, "NSNumber"), ex);
        }
    }

    private NSString parseString(int offset, int objInfo, BiFunction<Integer, Integer, Integer> stringLengthCalculator, String charsetName) throws PropertyListFormatException, UnsupportedEncodingException {
        int length;
        int[] lengthAndOffset = this.readLengthAndOffset(objInfo, offset);
        int strOffset = offset + lengthAndOffset[1];
        if (strOffset + (length = stringLengthCalculator.apply(strOffset, lengthAndOffset[0]).intValue()) > this.bytes.length) {
            throw new PropertyListFormatException(BinaryPropertyListParser.buildLengthError(offset, "NSString"));
        }
        return new NSString(this.bytes, strOffset, strOffset + length, charsetName);
    }

    private NSArray parseArray(int offset, int objInfo, ParsedObjectStack stack) throws PropertyListFormatException, UnsupportedEncodingException {
        int[] lengthAndOffset = this.readLengthAndOffset(objInfo, offset);
        int length = lengthAndOffset[0];
        int arrayOffset = offset + lengthAndOffset[1];
        NSArray array = new NSArray(length);
        for (int i = 0; i < length; ++i) {
            int objRef = this.parseObjectReferenceFromList(arrayOffset, i);
            array.setValue(i, this.parseObject(stack, objRef));
        }
        return array;
    }

    private NSSet parseSet(int offset, int objInfo, boolean ordered, ParsedObjectStack stack) throws PropertyListFormatException, UnsupportedEncodingException {
        int[] lengthAndOffset = this.readLengthAndOffset(objInfo, offset);
        int length = lengthAndOffset[0];
        int setOffset = offset + lengthAndOffset[1];
        NSSet set = new NSSet(ordered);
        HashSet<Integer> addedObjectReferences = new HashSet<Integer>();
        for (int i = 0; i < length; ++i) {
            int objRef = this.parseObjectReferenceFromList(setOffset, i);
            if (!addedObjectReferences.add(objRef)) continue;
            set.addObject(this.parseObject(stack, objRef));
        }
        return set;
    }

    private NSDictionary parseDictionary(int offset, int objInfo, ParsedObjectStack stack) throws PropertyListFormatException, UnsupportedEncodingException {
        int[] lengthAndOffset = this.readLengthAndOffset(objInfo, offset);
        int length = lengthAndOffset[0];
        int contentOffset = lengthAndOffset[1];
        int keyListOffset = offset + contentOffset;
        int valueListOffset = keyListOffset + length * this.objectRefSize;
        NSDictionary dict = new NSDictionary();
        for (int i = 0; i < length; ++i) {
            int keyRef = this.parseObjectReferenceFromList(keyListOffset, i);
            int valRef = this.parseObjectReferenceFromList(valueListOffset, i);
            NSObject key = this.parseObject(stack, keyRef);
            if (key == null) {
                throw new PropertyListFormatException("The key #" + (i + 1) + " of the NSDictionary at offset " + offset + " is NULL.");
            }
            NSObject val = this.parseObject(stack, valRef);
            dict.put(key.toString(), val);
        }
        return dict;
    }

    private int[] readLengthAndOffset(int objInfo, int offset) throws PropertyListFormatException {
        try {
            int lengthValue = objInfo;
            int offsetValue = 1;
            if (objInfo == 15) {
                byte int_type = this.bytes[offset + 1];
                int intType = (int_type & 0xF0) >> 4;
                if (intType != 1) {
                    System.err.println("BinaryPropertyListParser: Length integer has an unexpected type (" + intType + "). Attempting to parse anyway...");
                }
                int intInfo = int_type & 0xF;
                int intLength = (int)Math.pow(2.0, intInfo);
                offsetValue = 2 + intLength;
                lengthValue = intLength < 3 ? (int)BinaryPropertyListParser.parseUnsignedInt(this.bytes, offset + 2, offset + 2 + intLength) : new BigInteger(BinaryPropertyListParser.copyOfRange(this.bytes, offset + 2, offset + 2 + intLength)).intValue();
            }
            return new int[]{lengthValue, offsetValue};
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException ex) {
            throw new PropertyListFormatException("The length/offset integer at offset " + offset + " is invalid.", ex);
        }
    }

    private int calculateUtf8StringLength(int offset, int numCharacters) {
        int length = 0;
        for (int i = 0; i < numCharacters; ++i) {
            int tempOffset = offset + length;
            if (this.bytes.length <= tempOffset) {
                return numCharacters;
            }
            byte currentByte = this.bytes[tempOffset];
            if ((currentByte & 0x80) != 128) {
                ++length;
                continue;
            }
            int n = 0;
            if ((currentByte & 0xC0) == 128) {
                return numCharacters;
            }
            if ((currentByte & 0xE0) == 192) {
                n = 1;
            } else if ((currentByte & 0xF0) == 224) {
                n = 2;
            } else if ((currentByte & 0xF8) == 240) {
                n = 3;
            }
            if (this.hastUtf8Sequence(tempOffset, n)) {
                length += 2;
                continue;
            }
            return numCharacters;
        }
        return length;
    }

    private boolean hastUtf8Sequence(int offset, int n) {
        for (int i = 1; i <= n; ++i) {
            if (offset + i < this.bytes.length && (this.bytes[offset + i] & 0xC0) == 128) continue;
            return false;
        }
        return true;
    }

    private int parseObjectReferenceFromList(int baseOffset, int objectIndex) throws PropertyListFormatException {
        return this.parseObjectReference(baseOffset + objectIndex * this.objectRefSize);
    }

    private int parseObjectReference(int offset) throws PropertyListFormatException {
        if (offset + this.objectRefSize >= this.bytes.length) {
            throw new PropertyListFormatException("Encountered the end of the file while parsing the object reference at offset " + offset + ".");
        }
        return (int)BinaryPropertyListParser.parseUnsignedInt(this.bytes, offset, offset + this.objectRefSize);
    }

    private int getObjectOffset(int obj) throws PropertyListFormatException {
        if (obj >= this.numObjects) {
            throw new PropertyListFormatException("The given binary property list contains an invalid object identifier (" + obj + ").");
        }
        int startOffset = this.offsetTableOffset + obj * this.offsetSize;
        return (int)BinaryPropertyListParser.parseUnsignedInt(this.bytes, startOffset, startOffset + this.offsetSize);
    }

    private String buildTypeError(int offset) {
        return this.buildTypeError(offset, "NSObject");
    }

    private String buildTypeError(int offset, String objectType) {
        return String.format("The %s at offset %d has an unknown or unsupported type (0x%02x)", objectType, offset, this.bytes[offset]);
    }

    private static String buildLengthError(int offset, String objectType) {
        return String.format("The length of the %s at offset %d is larger than the amount of available data.", objectType, offset);
    }
}

